//
//  WCXFDataController.h
//
//  NOTE
//  -----------------------
//  1. read前外部要準備好比對重複用的compareDuiplicateModel資料
//  2. read後加入db前要檢查是否存在
//  3. read不會讀modifiedTime，因為時間若和目前差異超過十天，同步會撈不到這筆資料。(改為讀取，由上層決定是不是要使用。)
//  4. google的group已在元件內部處理
//  5. 舊版產生的unverify群組一律丟到google下的other contact或是worldcard下的unfiled，視source而定
//


#import <Foundation/Foundation.h>
#import "WCCardModel.h"
#import "WCImageDefine.h"
#import "WCXFRootInfoModel.h"
#import "WCXFDataControllerDefine.h"


////////////////////////////////////////////////////////////////////////////////////////////////////
/// WCXF file extension

extern NSString * const WCXFDataController_FileExt; // @"wcxf"


////////////////////////////////////////////////////////////////////////////////////////////////////
/// Error code of WCXFDataController

typedef NS_OPTIONS(NSInteger, WCXFDataController_Error)
{
    WCXFDataController_Error_Unknown = 0,
    WCXFDataController_Error_NoCardToWrite,
    WCXFDataController_Error_MustHaveDelegateMethod,
    WCXFDataController_Error_FailedToInitObject,
    WCXFDataController_Error_FailedToOpenFileForWriting,
    WCXFDataController_Error_FailedToOpenFileForReading,
    WCXFDataController_Error_FailedToReadRootInfo,
    WCXFDataController_Error_FailedToReadCardInfo,
    WCXFDataController_Error_FailedToWriteRootInfo,
    WCXFDataController_Error_FailedToWriteCardInfo,
    WCXFDataController_Error_CancelWritingCard
};

////////////////////////////////////////////////////////////////////////////////////////////////////
/// WCXFDataControllerDelegate

@class WCXFDataController;
@protocol WCXFDataControllerDelegate <NSObject>
@optional

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Delegate for read WCXF

/**
 * Ask if can read card or not
 *
 * @param wcxfDataController - Data controller instance
 * @param index - Read card index
 * @param totalCount - Total count
 * @return YES to read card, NO to abort reading rest cards.
 */
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
  shouldReadCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount;


/**
 * Send read card field data
 *
 * @param wcxfDataController - Data controller instance
 * @param cardModel - WCCardModel with fieldData
 */
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
            readCardModel:(WCCardModel *)cardModel;

/**
 * Send read card image data
 *
 * @param wcxfDataController - Data controller instance
 * @param imageData - Read image data
 * @param imageType - Image type
 * @param cardID - Image's owner
 */
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
            readImageData:(NSData *)imageData
                imageType:(WC_ImageType)imageType
                   cardID:(NSString *)cardID;

/**
 * Get relative groupID by group name
 *
 * @param wcxfDataController - Data controller instance
 * @param groupName - Group name
 * @param error 回傳錯誤
 * @return Group ID
 */
- (WC_GroupID)wcxfDataContrller:(WCXFDataController *)wcxfDataController
       groupIDWithReadGroupName:(NSString *)groupName
                          error:(NSError **)error;

/**
 * Notify one card is read finish
 *
 * @param wcxfDataController - Data controller instance
 * @param index - Read card index
 * @param totalCount - Total count
 */
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
     didReadCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Delegate for write WCXF

/**
 * Ask if can write card or not
 *
 * @param wcxfDataController - Data controller instance
 * @param index - Write card index
 * @param totalCount - Total count
 * @return YES to read card, NO to abort reading rest cards.
 */
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
 shouldWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount;


/**
 * Request field data to write
 *
 * @param wcxfDataController - Data controller instance
 * @param cardID - Card that need get field data
 * @return WCCardModel with field data of card
 */
- (WCCardModel *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
       cardModelToWriteWithCardID:(NSString *)cardID;

/**
 * Request image data to write
 *
 * @param wcxfDataController - Data controller instance
 * @param cardID - Card that need get image data
 * @param imageType - Needed image type
 * @return Image data of card
 */
- (NSData *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
   imageDataToWriteWithCardID:(NSString *)cardID
                    imageType:(WC_ImageType)imageType;

/**
 * Request group name to write
 *
 * @param wcxfDataController - Data controller instance
 * @param groupID - Group ID that need get name
 * @return Group name
 */
- (NSString *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    groupNameToWriteWithGroupID:(WC_GroupID)groupID;

/**
 * Notify one card is write finish
 *
 * @param wcxfDataController - Data controller instance
 * @param index - Write card index
 * @param totalCount - Total count
 */
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

/// WCXFDataController
@interface WCXFDataController : NSObject

/**
 * Initialize
 *
 * @param delegate - Instance that can handle WCXFDataControllerDelegate
 * @return WCXFDataController instance
 */
- (instancetype)initWithDelegate:(id<WCXFDataControllerDelegate>)delegate;

/**
 * Get WCXF root info
 *
 * @param filePath - WCXF file path
 * @param error - return error
 * @return Root info
 */
- (WCXFRootInfoModel *)rootInfoOfFile:(NSString *)filePath error:(NSError **)error;

/**
 * Read WCXF
 *
 * @param filePath - WCXF file path
 * @param error - return error
 * @return YES if success
 */
- (BOOL)readFileWithPath:(NSString *)filePath error:(NSError **)error;

/**
 * Write WCXF
 *
 * @param filePath - WCXF file path
 * @param cardIDs - Cards that wnat to write
 * @param error - return error
 * @return YES if success
 */
- (BOOL)writeFileWithPath:(NSString *)filePath cardIDs:(NSArray *)cardIDs error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - class methods

/**
 * 設定使用WCXFDataController的產品是哪一種
 *
 * @param productLine WCXFDataController_ProductLine
 */
+ (void)setProductLine:(WCXFDataController_ProductLine)productLine;
@end
