//
//  WCVCFDataController.m
//


#import "WCVCFDataController.h"

#import "WCFieldDefine.h"
#import "WCCardModel.h"
#import "NSError+Custom.h"
#import "PPIndexingController.h"

#if TARGET_OS_IPHONE

#import "WCABDataController.h"

#define ABDataController WCABDataController

#else 

#import "WCMACABDataController.h"

#define ABDataController WCMACABDataController

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const WCVCFDataController_FileExt = @"vcf";

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCVCFDataController ()
@property (nonatomic, retain) ABDataController *abDataController;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCVCFDataController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_abDataController release];
    _abDataController = nil;

    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (void)checkAccessPermissionWithCompletion:(void (^)(BOOL authorized))completion
{
#if TARGET_OS_IPHONE
    [ABDataController checkAccessPermissionWithCompletion:completion];
#elif TARGET_OS_MAC
    
#endif
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (void)generateHiraganaWithCardModel:(WCCardModel *)cardModel
{
    NSMutableArray *nameFields = [cardModel fieldArrayWithType:WC_FT_Name];
    
    for(WCFieldModel *nameField in nameFields)
    {
        if([nameField hasFieldWithSubType2:WC_FST2_Name_Last] == YES &&
           [nameField hasFieldWithSubType2:WC_FST2_Name_LastPhonetic] == NO)
        {
            NSString *source = [nameField valueWithSubType2:WC_FST2_Name_Last];
            NSString *phonetic = [PPIndexingController phoneticOfString:source forStyle:PPIndexingStyle_Hiragana];
            
            if([phonetic length] > 0)
            {
                [nameField setSubType2FieldWithValue:phonetic recogRect:CGRectZero subType2:WC_FST2_Name_LastPhonetic];
            }
        }
        
        if([nameField hasFieldWithSubType2:WC_FST2_Name_First] == YES &&
           [nameField hasFieldWithSubType2:WC_FST2_Name_FirstPhonetic] == NO)
        {
            NSString *source = [nameField valueWithSubType2:WC_FST2_Name_First];
            NSString *phonetic = [PPIndexingController phoneticOfString:source forStyle:PPIndexingStyle_Hiragana];
            
            if([phonetic length] > 0)
            {
                [nameField setSubType2FieldWithValue:phonetic recogRect:CGRectZero subType2:WC_FST2_Name_FirstPhonetic];
            }
        }
    }
    
    //////////////////////////////////////////////////
    
    NSMutableArray *companyFields = [cardModel fieldArrayWithType:WC_FT_Company];
    
    for(WCFieldModel *companyField in companyFields)
    {
        if([companyField hasFieldWithSubType2:WC_FST2_Company_Name] == YES &&
           [companyField hasFieldWithSubType2:WC_FST2_Company_Phonetic] == NO)
        {
            NSString *source = [companyField valueWithSubType2:WC_FST2_Company_Name];
            NSString *phonetic = [PPIndexingController phoneticOfString:source forStyle:PPIndexingStyle_Hiragana];
            
            if([phonetic length] > 0)
            {
                [companyField setSubType2FieldWithValue:phonetic recogRect:CGRectZero subType2:WC_FST2_Company_Phonetic];
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (id)init
{
    NSAssert(self == nil, @"Use -initWithError: instead");
    
    return nil;
}


//================================================================================
//
//================================================================================
- (id)initWithError:(NSError **)error
{
    if(self = [super init])
    {
        self.abDataController = [[[ABDataController alloc] initWithError:error] autorelease];
        
        if(self.abDataController == nil)
        {
            [self dealloc];
            return nil;
        }
    }

    return self;
}


//================================================================================
//
//================================================================================
- (WCCardModel *)cardModelFromVCardString:(NSString *)vCardString
                                 encoding:(NSStringEncoding)encoding
                         generateHiragana:(BOOL)generateHiragana
                                    error:(NSError **)error
{
    WCCardModel *cardModel = nil;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            if([vCardString length] == 0)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_InvalidParameters, @"vCardString is nil", nil);
                break;
            }
            
            
            //////////////////////////////////////////////////
            // decode string
            
            NSData *vCardData = [vCardString dataUsingEncoding:encoding];
            
            if(vCardData == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_FailedToConvert, @"Failed to convert vCard string", nil);
                break;
            }
            
            
            //////////////////////////////////////////////////
            // convert to cardModel
            
            cardModel = [[self.abDataController cardModelWithVCardData:vCardData error:&returnError] retain];

            if(cardModel == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_FailedToConvert, @"cardModelWithVCardData failed", returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(generateHiragana == YES)
            {
                [self generateHiraganaWithCardModel:cardModel];
            }
        }
        while(0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }

    } // end of @autoreleasepool

    //////////////////////////////////////////////////

    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return [cardModel autorelease];
}


//================================================================================
//
//================================================================================
- (NSString *)vCardStringFromCardModel:(WCCardModel *)cardModel
                              encoding:(NSStringEncoding)encoding
                                 error:(NSError **)error
{
    NSError *returnError = nil;
    NSString *vCardString = nil;
    
    @autoreleasepool
    {
        do
        {
            if(cardModel == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_InvalidParameters, @"cardModel is nil", nil);
                break;
            }
            
            //////////////////////////////////////////////////
         
            WCABCardModel *abCardModel = [WCABCardModel abCardModelFromCardModel:cardModel];

            NSData *vCardData = [self.abDataController vCardDataWithCardModel:abCardModel error:&returnError];

            if(vCardData == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_FailedToConvert, @"vCardDataWithCardModel failed", returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            vCardString = [[NSString alloc] initWithData:vCardData encoding:encoding];
            
        }
        while (0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////

    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return [vCardString autorelease];
}


//================================================================================
//
//================================================================================
- (WCCardModel *)readFileWithPath:(NSString *)filePath generateHiragana:(BOOL)generateHiragana error:(NSError **)error
{
    WCCardModel *cardModel = nil;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            if([[NSFileManager defaultManager] fileExistsAtPath:filePath isDirectory:nil] == NO &&
               [[NSURL URLWithString:filePath] checkResourceIsReachableAndReturnError:&returnError]==NO)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_FailedToAccessFile, @"File not found", nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSData *data = [[NSData alloc] initWithContentsOfFile:filePath];

            cardModel = [[self.abDataController cardModelWithVCardData:data error:&returnError] retain];

            [data release];
            
        }
        while(0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return [cardModel autorelease];
}


//================================================================================
//
//================================================================================
- (BOOL)writeFileWithPath:(NSString *)filePath
                cardModel:(WCCardModel *)cardModel
                    error:(NSError **)error
{
    return [self writeFileWithPath:filePath
                         cardModel:cardModel
                           IDPhoto:nil
                             error:error];
}



//==============================================================================
//
//==============================================================================
- (BOOL)writeFileWithPath:(NSString *)filePath
                cardModel:(WCCardModel *)cardModel
                  IDPhoto:(CPImage *)IDPhoto
                    error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            if([filePath length] == 0)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_InvalidParameters, @"filePath is nil", nil);
                break;
            }
            
            if(cardModel == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_InvalidParameters, @"cardModel is nil", nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            WCABCardModel *abCardModel = [WCABCardModel abCardModelFromCardModel:cardModel];
            abCardModel.abPhotoImage = IDPhoto;
            
            NSData *vCardData = [self.abDataController vCardDataWithCardModel:abCardModel error:&returnError];
            
            if(vCardData == nil)
            {
                returnError = PPErrorMake(WCVCFDataController_Error_FailedToConvert, @"vCardDataWithCardModel failed", returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([[NSFileManager defaultManager] fileExistsAtPath:filePath isDirectory:nil] == YES)
            {
                if([[NSFileManager defaultManager] removeItemAtPath:filePath error:&returnError] == NO)
                {
                    returnError = PPErrorMake(WCVCFDataController_Error_FailedToAccessFile, @"Failed to remove file", returnError);
                    break;
                }
            }
            
            if([vCardData writeToFile:filePath
                              options:NSDataWritingAtomic
                                error:&returnError] == NO)
            {
                break;
            }
            
            result = YES;
        }
        while(0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return result;
}

@end
