//
//  WCTSyncFlowController.m
//  Example
//
//  Created by Eddie Hu on 2016/4/11.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import "WCTSyncFlowController.h"
#import "PPNetworkReachabilityController.h"
#import "PPLogController.h"
#import "NSError+Custom.h"
#import "PPBackgroundTaskController.h"
#import "WCToolController.h"
#import "WCTSyncActionController.h"
#import "NSString+Additions.h"
#import "WCTAccountDataController.h"
#import "WCTRestClientController.h"
#import "PPSettingsController.h"
#import "WCTSettingsKey.h"
#import "WCTDataController.h"
#import "WCTAccountRelationModel.h"
#import "NSDate+Format.h"
#import "WCSyncTargetDefine.h"


#if TARGET_OS_IPHONE
#import "UIApplication+Idle.h"
#elif TARGET_OS_MAC
#import "NSApplication+Idle.h"
#endif

#import <ifaddrs.h>
#import <arpa/inet.h>

////////////////////////////////////////////////////////////////////////////////////////////////////
// !! 共用的字串由app的字串檔取得

#define WCSFC_MLS_Ok                                    [@"MLS_OK" localized]
#define WCSFC_MLS_Cancel                                [@"MLS_Cancel" localized]
#define WCSFC_MLS_ConnectInternet                       [@"MLS_FailedToConnectInternet" localized]
#define WCSFC_MLS_FailedToConnectServer                 [@"MLS_FailedToConnectServer" localized]
#define WCSFC_MLS_ServerBusy                            [@"MLS_ServerBusy" localized]
#define WCSFC_MLS_SyncFailed                            [@"MLS_SyncFailed" localized]
#define WCSFC_MLS_AuthenticationFailed                  [@"MLS_AuthenticationFailed" localized]
#define WCSFC_MLS_AccountIsInheriting                   [@"MLS_AccountIsInheriting" localized]
#define WCSFC_MLS_OverPrivateContactCountLimitation     [@"MLS_OverPrivateContactCountLimitation" localized]
#define WCSFC_MLS_OverPublicContactCountLimitation      [@"MLS_OverPublicContactCountLimitation" localized]
#define WCSFC_MLS_OverServerContactCountLimitation      [@"MLS_OverServerContactCountLimitation" localized]
#define WCSFC_MLS_NotEnoughServerSpace                  [@"MLS_NotEnoughServerSpace" localized]
#define WCSFC_MLS_NotEnoughClientSpace                  [@"MLS_NotEnoughClientSpace" localized]
#define WCSFC_MLS_ServerInternalError                   [@"MLS_ServerInternalError" localized]
#define WCSFC_MLS_ServerMaintaining                     [@"MLS_ServerMaintaining" localized]
#define WCSFC_MLS_ServerVersionIsNewer                  [@"MLS_ServerVersionIsNewer" localized]
#define WCSFC_MLS_UnsupportedServerVersion              [@"MLS_UnsupportedServerVersion" localized]
#define WCSFC_MLS_LimitedAccount                        [@"MLS_LimitedAccount" localized]

////////////////////////////////////////////////////////////////////////////////////////////////////

/// 同步機制版本
int const WCTSyncFlowController_SyncActionControllerVersionNumber = 1;

NSString * const WCTSyncFlowController_TaskKey = @"WCTSyncFlowController_TaskKey";
NSString * const WCTSyncFlowController_DataDir = @"SyncData";
NSString * const WCTSyncFlowController_LogDir = @"WCTSyncFlowLog";

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCTSyncFlowController () <PPSyncActionControllerDelegate>
@property (atomic, retain) PPLogController *logController;
@property (atomic, retain) WCTSyncActionController *syncActionController;
@property (atomic, retain) NSError *lastError;
@property (atomic, assign) CGFloat currentProgress;
@property (atomic, assign) WCSyncFlow_Flow currentFlow;
@property (atomic, assign) WCSyncFlow_Network availableNetwork;
@property (atomic, assign) CGFloat stepPercentageBase;
@property (atomic, assign) CGFloat stepPercentageRatio;
@property (atomic, assign) BOOL isAutoMode; /// 記錄是否自動同步模式，僅供外部使用。

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCTSyncFlowController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)init
{
    if(self=[super init])
    {
        //////////////////////////////////////////////////
        // 建立sync log instance
        
        self.logController = [[[PPLogController alloc] init] autorelease];
        
        if(self.logController != nil)
        {
            NSString *logDirPath = [WCTSyncFlowController logDirPath];
            
            if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
            {
                [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                          withIntermediateDirectories:YES
                                                           attributes:nil
                                                                error:nil];
            }
            
            [self.logController setFileName:WCTSyncFlowController_LogDir atPath:logDirPath];
            [self.logController setMask:PPLogControllerMask_Normal];
        }

    }
    
    return self;
}

//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.logController = nil;
    self.syncActionController = nil;
    self.lastError = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//===============================================================================
//
//===============================================================================
- (void)logMessageWithFormat:(NSString *)format, ...
{
    va_list arguments;
    va_start(arguments, format);
    [self.logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
}


//==============================================================================
//
//==============================================================================
- (void)cleanLogFileWithReserveCount:(NSInteger)reserveCount
{
    NSString *logDirPath = [WCTSyncFlowController logDirPath];
    NSFileManager *fileManager = [NSFileManager defaultManager];
    
    
    if([fileManager fileExistsAtPath:logDirPath] == YES)
    {
        NSArray *fileList = [fileManager contentsOfDirectoryAtPath:logDirPath error:nil];
        NSInteger removeCount = [fileList count]-reserveCount;
        
        // !! 檔案已照時間排序，留下最後幾個。
        if(removeCount > 0)
        {
            for(int i=0; i<removeCount; i++)
            {
                NSString *fullPath = [logDirPath stringByAppendingPathComponent:[fileList objectAtIndex:i]];
                [fileManager removeItemAtPath:fullPath error:nil];
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)postSyncFlowNotificationWithUserInfo:(NSDictionary *)userInfo
{
    [[NSNotificationCenter defaultCenter] postNotificationName:WCSyncFlow_Notification
                                                        object:nil
                                                      userInfo:userInfo];
}


//===============================================================================
//
//===============================================================================
- (void)sendNotificationWithKey:(NSString *)key value:(id)value
{
    @autoreleasepool
    {
        if(value == nil)
        {
            value = [NSNull null];
        }
        
        [self performSelectorOnMainThread:@selector(postSyncFlowNotificationWithUserInfo:)
                               withObject:@{key:value}
                            waitUntilDone:NO];
    }
}


//==============================================================================
// 取得所有ip
//==============================================================================
- (NSArray *)ipAddresses
{
    NSMutableArray *ipAddresses = [NSMutableArray array];
    
    struct ifaddrs *interfaces = NULL;
    struct ifaddrs *temp_addr = NULL;
    struct sockaddr_in *s4;
    struct sockaddr_in6 *s6;
    char buf[64];
    int success = 0;
    
    // retrieve the current interfaces - returns 0 on success
    success = getifaddrs(&interfaces);
    if (success == 0)
    {
        // Loop through linked list of interfaces
        temp_addr = interfaces;
        while(temp_addr != NULL)
        {
            if(temp_addr->ifa_addr->sa_family == AF_INET)
            {
                
                s4 = (struct sockaddr_in *)temp_addr->ifa_addr;
                
                if (inet_ntop(temp_addr->ifa_addr->sa_family, (void *)&(s4->sin_addr), buf, sizeof(buf)) == NULL)
                {
                    NSLog(@"%s: inet_ntop failed for v4!\n",temp_addr->ifa_name);
                }
                else
                {
                    NSString *address = [NSString stringWithUTF8String:buf];
                    if (address)
                    {
                        [ipAddresses addObject:address];
                    }
                    
                }
                
            }
            
            else if(temp_addr->ifa_addr->sa_family == AF_INET6)
            {
                s6 = (struct sockaddr_in6 *)(temp_addr->ifa_addr);
                
                if (inet_ntop(temp_addr->ifa_addr->sa_family, (void *)&(s6->sin6_addr), buf, sizeof(buf)) == NULL)
                {
                    NSLog(@"%s: inet_ntop failed for v6!\n",temp_addr->ifa_name);
                }
                else
                {
                    NSString *address = [NSString stringWithUTF8String:buf];
                    if (address)
                    {
                        [ipAddresses addObject:address];
                    }
                }
            }
            
            temp_addr = temp_addr->ifa_next;
        }
    }
    
    // Free memory
    freeifaddrs(interfaces);
    
    return ipAddresses;
}


//================================================================================
//
//================================================================================
- (NSString *)ipAddress
{
    NSString *resultIPString = @"";
    
    NSArray *ipAddresses = [self ipAddresses];
    
    //////////////////////////////////////////////////
    // 過濾ipV6
    // 過濾127.0.0.1
    NSArray *sortedIPAddresses = [ipAddresses sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)];
    
    NSNumberFormatter *numberFormatter = [[[NSNumberFormatter alloc] init] autorelease];
    numberFormatter.allowsFloats = NO;
    
    for (NSString *potentialIPAddress in sortedIPAddresses)
    {
        if ([potentialIPAddress isEqualToString:@"127.0.0.1"])
        {
            continue;
        }
        
        NSArray *ipParts = [potentialIPAddress componentsSeparatedByString:@"."];
        BOOL isMatch = YES;
        
        for (NSString *ipPart in ipParts)
        {
            if (![numberFormatter numberFromString:ipPart])
            {
                isMatch = NO;
                break;
            }
        }
        
        if (isMatch)
        {
            if ([resultIPString length]>0)
            {
                resultIPString = [resultIPString stringByAppendingString:@"    "];
            }
            resultIPString = [resultIPString stringByAppendingString:potentialIPAddress];
            
        }
    }
    
    //////////////////////////////////////////////////
    if ([resultIPString length]>0)
    {
        return resultIPString;
    }
    else
    {
        // No IP found
        return @"?.?.?.?";
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)userName
{
    WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_All];
    WCTAccountRelationModel *accountRelationModel = [dataController accountRelationWithAcountGuid:[PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID]];
    [dataController release];
    
    return accountRelationModel.name;
}


//================================================================================
//
//================================================================================
- (BOOL)isCancelError:(NSError *)error
{
    if(error.code == -999 ||
       ([error.domain isEqualToString:@"WCTSyncActionController"] == YES && error.code == WCSyncFlow_Error_UserCancel))
    {
        return YES;
    }
    else
    {
        id object = [error.userInfo objectForKey:NSErrorCustom_Key_Object];
        
        if([object isKindOfClass:[NSError class]] == YES)
        {
            return [self isCancelError:object];
        }
        else
        {
            return NO;
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)finishSync:(BOOL)result isAutoMode:(NSNumber *)isAutoMode error:(NSError *)error
{
    self.currentFlow = WCSyncFlow_Flow_None;
    
    if(result == YES)
    {
        self.currentProgress = 1.0;
        [self sendNotificationWithKey:WCSyncFlow_Notification_kProgress value:@(self.currentProgress)];
        [self sendNotificationWithKey:WCSyncFlow_Notification_kSuccess value:isAutoMode];
    }
    else
    {
        // !! error domain name統一設定為WCTSyncFlowController，外部才比較方便判斷。

        NSError *networkError = [error findNetworkError];
        
        if(networkError != nil)
        {
            //////////////////////////////////////////////////
            // 網路錯誤
            // 檢查是否有網路連線，有的話表示是server問題，沒有的話就是網路問題
            if ([PPNetworkReachabilityController checkForInternetConnection]==NO)
            {
                [self logMessageWithFormat:@"   %s is network problem", __func__];
                self.lastError = PPErrorMake(WCSyncFlow_Error_NetworkProblem, WCSFC_MLS_ConnectInternet, isAutoMode);
            }
            else
            {
                if (networkError.code==NSURLErrorTimedOut)
                {
                    [self logMessageWithFormat:@"   %s server is busy", __func__];
                    self.lastError = PPErrorMake(WCSyncFlow_Error_ServerBusy, WCSFC_MLS_ServerBusy, isAutoMode);
                }
                else
                {
                    [self logMessageWithFormat:@"   %s can't connect server", __func__];
                    self.lastError = PPErrorMake(WCSyncFlow_Error_FailedToConnectServer, WCSFC_MLS_FailedToConnectServer, isAutoMode);
                }
            }
        }
        else
        {
            if(error == nil)
            {
                self.lastError = PPErrorMake(WCSyncFlow_Error_Unknown, @"Unknown - 有地方沒有回傳error, 要檢查", nil);
            }
            else
            {
                if([self isCancelError:error] == YES)
                {
                    self.lastError = PPErrorMake(WCSyncFlow_Error_UserCancel, nil, isAutoMode);
                }
                else if([error.domain isEqualToString:@"WCTSyncActionController"] == YES &&
                        error.code == WCTSyncActionControllerError_NotEnoughDiskSpace)
                {
                    self.lastError = PPErrorMake(WCSyncFlow_Error_NotEnoughClientSpace, WCSFC_MLS_NotEnoughClientSpace, isAutoMode);
                }
                else if([error.domain isEqualToString:@"WCTSyncFlowController"] == YES &&
                        error.code == WCSyncFlow_Error_FailedToConnectServer)
                {
                    self.lastError = PPErrorMake(WCSyncFlow_Error_FailedToConnectServer, WCSFC_MLS_FailedToConnectServer, isAutoMode);
                }
                else if([error.domain isEqualToString:@"WCTSyncFlowController"] == YES &&
                        error.code == WCSyncFlow_Error_ServerBusy)
                {
                    self.lastError = PPErrorMake(WCSyncFlow_Error_ServerBusy, WCSFC_MLS_ServerBusy, isAutoMode);
                }
                else if([error.domain isEqualToString:@"WCTRestClientController"] == YES)
                {
                    switch (error.code)
                    {
                        case WCTRestClientController_ErrorCode_ServerRestored:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_AuthenticationFailed, WCSFC_MLS_AuthenticationFailed, isAutoMode);
                            break;
                        }
                        case WCTRestClientController_ErrorCode_LoginAfterServerUpdated:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_AuthenticationFailed, WCSFC_MLS_AuthenticationFailed, isAutoMode);
                            break;
                        }
                        case WCTRestClientController_ErrorCode_ServerVersionIsNewer:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_ServerVersionIsNewer, WCSFC_MLS_ServerVersionIsNewer, isAutoMode);
                            break;
                        }
                        case WCTRestClientController_ErrorCode_UnsupportedServerVersion:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_UnsupportedServerVersion, WCSFC_MLS_UnsupportedServerVersion, isAutoMode);
                            break;
                        }
                        default:
                            break;
                    }
                }
                else
                {
                    switch ([WCTRestClientController statusCodeFromAFRKNetworkingError:error])
                    {
                        case NSNotFound: // server沒回傳錯誤碼，代表可能是連線問題，只顯示同步失敗。
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_Undefined, WCSFC_MLS_SyncFailed, isAutoMode);
                            break;
                        }
                            
                        case WCTServer_Common_ErrorCode_AuthenticationFailed:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_AuthenticationFailed, WCSFC_MLS_AuthenticationFailed, isAutoMode);
                            break;
                        }

                        case WCTServer_Common_ErrorCode_AccountIsInheriting:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_AccountIsInheriting, WCSFC_MLS_AccountIsInheriting, isAutoMode);
                            break;
                        }
                            
                        case WCTServer_Common_ErrorCode_ServerMaintenance:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_ServerMaintenance, WCSFC_MLS_ServerBusy, isAutoMode);
                            break;
                        }
                            
                        case WCTServer_Common_ErrorCode_ServerTaskCollision:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_ServerTaskCollision, WCSFC_MLS_ServerBusy, isAutoMode);
                            break;
                        }
                            
                        case WCTServer_Common_ErrorCode_OverPrivateContactCountLimitation:
                        {
                            NSInteger maxContactCount = [PPSettingsController integerValueWithKey:WCTSettingsKey_MaxPrivateContactCount];
                            NSString *message = [NSString stringWithFormat:WCSFC_MLS_OverPrivateContactCountLimitation,[self userName], maxContactCount];
                            self.lastError = PPErrorMake(WCSyncFlow_Error_OverContactCountLimitation, message, isAutoMode);
                            break;
                        }


                        case WCTServer_Common_ErrorCode_OverPublicContactCountLimitation:
                        {
                            NSInteger maxContactCount = [PPSettingsController integerValueWithKey:WCTSettingsKey_MaxPublicContactCount];
                            NSString *message = [NSString stringWithFormat:WCSFC_MLS_OverPublicContactCountLimitation, [self userName], maxContactCount];
                            self.lastError = PPErrorMake(WCSyncFlow_Error_OverPublicContactCountLimitation, message, isAutoMode);
                            break;
                        }

                        case WCTServer_Common_ErrorCode_OverServerContactCountLimitation:
                        {
                            NSInteger maxContactCount = [PPSettingsController integerValueWithKey:WCTSettingsKey_MaxServerContactCount];
                            NSString *message = [NSString stringWithFormat:WCSFC_MLS_OverServerContactCountLimitation, maxContactCount];
                            self.lastError = PPErrorMake(WCSyncFlow_Error_OverServerContactCountLimitation, message, isAutoMode);
                            break;
                        }
                            
                        case WCTServer_Common_ErrorCode_NotEnoughServerSpace:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_NotEnoughServerSpace, WCSFC_MLS_NotEnoughServerSpace, isAutoMode);
                            break;
                        }

                        case WCTServer_Common_ErrorCode_LimitedAccountAccessDenied:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_LimitedAccount, WCSFC_MLS_LimitedAccount, isAutoMode);
                            break;
                        }
                        case WCTServer_Common_ErrorCode_Restoring:
                        case WCTServer_Common_ErrorCode_Backuping:
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_ServerMaintenance, WCSFC_MLS_ServerMaintaining, isAutoMode);
                            break;
                        }
                        default: // 包含500及其他未定義的server錯誤碼
                        {
                            self.lastError = PPErrorMake(WCSyncFlow_Error_ServerInternalError, WCSFC_MLS_ServerInternalError, isAutoMode);
                            break;
                        }
                    }
                }
            }
        }
        
        [self sendNotificationWithKey:WCSyncFlow_Notification_kFailed value:self.lastError];
    }

    //////////////////////////////////////////////////
    
    // 清掉多餘的log檔（同步成功保留100個檔案，失敗保留200個）
    [self cleanLogFileWithReserveCount:(result==YES) ? 100:200];
    
    // 建立新的log檔供下次同步使用（每次同步都獨立一個log紀錄）
    [self.logController renewLogFile];
    
    /// 完成後重設同步進度
    self.currentProgress = 0.0;
}


//===============================================================================
//
//===============================================================================
- (void)bgThreadSync:(NSNumber *)isAutoMode
{
    @autoreleasepool
    {
        //////////////////////////////////////////////////
        // Handle logs
        
        
        // 寫入這次同步的IP
        [self logMessageWithFormat:@"\tDevice IP:%@", [self ipAddress]];
        // log start status
        [self logMessageWithFormat:@""];
        [self logMessageWithFormat:@"\t⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇⬇"];
       
        
        //////////////////////////////////////////////////
        // Handle long-time task start
        
#if TARGET_OS_IPHONE
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [[UIApplication sharedApplication] enableIdle:NO];
        });
        
        [PPBackgroundTaskController addTaskWithKey:WCTSyncFlowController_TaskKey terminate:nil];
        
#elif TARGET_OS_MAC
        
        uint32_t activityID = [NSApplication pauseIdleMechanismWithReason:@"WorldCardTeam Sync"];
        
#endif
        
        
        //////////////////////////////////////////////////
        // sync task begin
        
        NSError *error = nil;
        BOOL syncResult = NO;
        
        do
        {
            self.currentProgress = 0.0;
            [self sendNotificationWithKey:WCSyncFlow_Notification_kProgress value:@(self.currentProgress)];
            
            //////////////////////////////////////////////////
            // 檢查server是否可連線
            WCTRCServerIsAliveResponseResult *serverIsAliveResponseResult = [[WCTRestClientController shareRestClientController] serverIsAliveWithError:&error];
            
            if(serverIsAliveResponseResult==nil ||
               serverIsAliveResponseResult.data==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            // 更新列表，及相關設定
            [WCTAccountDataController updateAllDataFromServerWithCompleteHandler:^(BOOL hasNetwork, NSError *error) {
                
                [WCTAccountDataController updateCategoriesFromServerWithCompleteHandler:nil];
            }];

            self.currentProgress = 0.01;
            [self sendNotificationWithKey:WCSyncFlow_Notification_kProgress value:@(self.currentProgress)];

            //////////////////////////////////////////////////
            // 初始化同步資料
            
            self.syncActionController = [[[WCTSyncActionController alloc] init] autorelease];
            
            if(self.syncActionController == nil)
            {
                error = PPErrorMake(WCSyncFlow_Error_InitObjectFailed, @"WCTSyncActionController init failed", isAutoMode);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.syncActionController.delegate = self;
            syncResult = [self.syncActionController syncWithError:&error quickSyncVaildTimeInterval:(10*24*60*60)];
            
        }
        while (0);

        //////////////////////////////////////////////////
        // Handle long-time task finish
        
#if TARGET_OS_IPHONE
        
        [PPBackgroundTaskController removeTaskWithKey:WCTSyncFlowController_TaskKey];

        dispatch_async(dispatch_get_main_queue(), ^{
            [[UIApplication sharedApplication] enableIdle:YES];
        });

#elif TARGET_OS_MAC
        
        [NSApplication resumeIdleMechanismWithID:activityID];
        
#endif
        
        
        //////////////////////////////////////////////////
        // Handle logs
        
        [self logMessageWithFormat:@"\t⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆⬆"];
        [self logMessageWithFormat:@""];
        [self logMessageWithFormat:@"\t%s syncResult=%d, error=%@", __func__, syncResult, error];
        
        
        //////////////////////////////////////////////////
        // Finish
        
        [self finishSync:syncResult isAutoMode:isAutoMode error:error];
    }
}


//===============================================================================
//
//===============================================================================
- (void)startSyncWithAvailableNetwork:(WCSyncFlow_Network)availableNetwork
                           isAutoMode:(BOOL)isAutoMode
{
    [self logMessageWithFormat:@"▶▶ %s (isAutoMode=%d)", __func__, isAutoMode];
    
    //////////////////////////////////////////////////
    
    NSString *message = nil;
    
    do
    {
        //////////////////////////////////////////////////
        // 檢查網路狀態
        
        PPNetworkReachabilityController *reachability = [PPNetworkReachabilityController networkReachabilityControllerForInternetConnection];
        PPNetworkReachabilityControllerStatus status = [reachability status];
        
        
        if(status == PPNetworkReachabilityControllerStatus_ReachableNone)
        {
            NSError *error = PPErrorMake(WCSyncFlow_Error_NetworkProblem, WCSFC_MLS_ConnectInternet, @(isAutoMode));
            [self sendNotificationWithKey:WCSyncFlow_Notification_kFailed value:error];
            
            message = @"no network";
            break;
        }
        else if(status == PPNetworkReachabilityControllerStatus_ReachableViaWWAN)
        {
            if(availableNetwork == WCSyncFlow_Network_WiFi)
            {
                message = @"can't sync by WWAN";
                break;
            }
        }
        
        
        //////////////////////////////////////////////////
        // 檢查登入狀態
//        if ([[WCMSyncFlowController currentAccount] length] == 0)
//        {
//            message = @"not logged in";
//            break;
//        }
        
        
        //////////////////////////////////////////////////
        // 檢查流程狀態是否可以進行同步
        
        if(self.currentFlow != WCSyncFlow_Flow_None)
        {
            message = [NSString stringWithFormat:@"has running flow:%ld", (long)self.currentFlow];
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 開始同步
        
        message = @"start sync";
        
        // 有mutithread操作的參數在一開始就要設定，避免後續判斷錯誤。
//        self.lastError = nil;
        self.isAutoMode = isAutoMode;
        self.currentFlow = WCSyncFlow_Flow_Syncing;
        self.currentProgress = 0.0;
        self.availableNetwork = availableNetwork;
        [self sendNotificationWithKey:WCSyncFlow_Notification_kStart value:@(WCSyncTarget_TeamServer)];
        
        // start sync in background thread
        [self performSelectorInBackground:@selector(bgThreadSync:) withObject:@(isAutoMode)];
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    [self logMessageWithFormat:@"◀◀ %s (%@)", __func__, message];
}


//===============================================================================
//
//===============================================================================
- (void)cancelSync
{
    [self.syncActionController cancelActions];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPSyncContactsControllerDelegate methods

//===============================================================================
//
//===============================================================================
- (void)ppSyncActionController:(PPSyncActionController *)ppSyncActionController step:(PPSyncActionStep)step;
{
    switch (step)
    {
        case WCTSyncActionControllerStep_GetNomalSyncInfo:
        {
            self.stepPercentageBase = 0.01;
            self.stepPercentageRatio = 0.04;
            break;
        }
        case WCTSyncActionControllerStep_GetLocalContactSyncInfo:
        {
            self.stepPercentageBase = 0.05;
            self.stepPercentageRatio = 0.05;
            break;
        }

        case WCTSyncActionControllerStep_FetchUpdateActions_Stage1:
        {
            self.stepPercentageBase = 0.10;
            self.stepPercentageRatio = 0.10;
            break;
        }

        case WCTSyncActionControllerStep_FetchUpdateActions_Stage2:
        {
            self.stepPercentageBase = 0.20;
            self.stepPercentageRatio = 0.10;
            break;
        }

            
        case WCTSyncActionControllerStep_ApplyContactUpdateActions:
        {
            self.stepPercentageBase = 0.30;
            self.stepPercentageRatio = 0.69;
            break;
        }
        case WCTSyncActionControllerStep_SyncDone:
        {
            self.stepPercentageBase = 0.99;
            self.stepPercentageRatio = 0.01;
            break;
        }
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    [self logMessageWithFormat:@"\t-> step:%d", step];
}


//===============================================================================
//
//===============================================================================
- (void)ppSyncActionController:(PPSyncActionController *)ppSyncActionController progress:(CGFloat)progress;
{
    if(progress >= 0.0 && progress <= 1.0)
    {
        self.currentProgress = self.stepPercentageBase + (self.stepPercentageRatio * progress);
        [self logMessageWithFormat:@"\t-> currentProgress:%.04f (%.04f + (%.04f * %.04f))", self.currentProgress, self.stepPercentageBase, self.stepPercentageRatio, progress];
        
        [self sendNotificationWithKey:WCSyncFlow_Notification_kProgress value:@(self.currentProgress)];
    }
}


//===============================================================================
//
//===============================================================================
- (void)ppSyncActionController:(PPSyncActionController *)ppSyncActionController logMessage:(NSString *)logMessage
{
    if([logMessage length] == 0)
    {
        [self logMessageWithFormat:@""];
    }
    else
    {
        [self logMessageWithFormat:@"\t%@", logMessage];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//===============================================================================
//
//===============================================================================
+ (instancetype)sharedInstance
{
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}


//===============================================================================
//
//===============================================================================
+ (NSString *)logDirPath
{
    return [WCToolController baseStorePathWithDirName:WCTSyncFlowController_LogDir isCreatDirPath:NO];
}


//================================================================================
//
//================================================================================
+ (void)syncWithAvailableNetwork:(WCSyncFlow_Network)availableNetwork
                      isAutoMode:(BOOL)isAutoMode
{
    WCTSyncFlowController *controller = [WCTSyncFlowController sharedInstance];
    
    [controller startSyncWithAvailableNetwork:availableNetwork isAutoMode:isAutoMode];
}


//================================================================================
//
//================================================================================
+ (void)cancelSync
{
    WCTSyncFlowController *controller = [WCTSyncFlowController sharedInstance];
    
    [controller cancelSync];
}


//================================================================================
//
//================================================================================
+ (WCSyncFlow_Flow)currentFlow
{
    return [WCTSyncFlowController sharedInstance].currentFlow;
}


//================================================================================
//
//================================================================================
+ (CGFloat)currentSyncProgress
{
    return [WCTSyncFlowController sharedInstance].currentProgress;
}


//================================================================================
//
//================================================================================
+ (NSDate *)lastSyncActionSuccessDate
{
    return [WCTSyncActionController lastSyncActionSuccessDate];
}


//================================================================================
//
//================================================================================
+ (void)resetToFirstSync
{
    [WCTSyncActionController resetToFirstSync];
}


//==============================================================================
//
//==============================================================================
+ (void)forceRecompareSync
{
    [WCTSyncActionController forceRecompareSync];
}


//================================================================================
//
//================================================================================
+ (BOOL)isAutoMode
{
    WCTSyncFlowController *controller = [WCTSyncFlowController sharedInstance];
    
    return controller.isAutoMode;
}

@end
