//  WCDataController.m
//  WorldCard series
//
//  Created by  Eddie on 12/4/9.
//  Copyright (c) 2012年 __MyCompanyName__. All rights reserved.
//
//
//


#import "WCTDataController.h"

//define
#import "CrossPlatformDefine.h"
#import "PPSectionIndexDefine.h"
#import "WCTSettingsKey.h"
#import "WCTCardDBDefine.h"

// model
#import "WCCardModel.h"
#import "WCCacheModel.h"
#import "WCCardSectionModel.h"
#import "WCCardStreamModel.h"
#import "WCDuplicateCompareModel.h"

// category
#import "WCFieldModel+DisplayName.h"
#import "PPSettingsController+DisplayRule.h"
#import "WCCompareTool.h"
#import "NSDate+Format.h"
#import "NSString+Additions.h"
#import "WCCardModel+FixFieldLength.h"
#import "WCCardModel+CombineCustomFields.h"
#import "WCGroupModel+Tree.h"
#import "NSError+Custom.h"
#import "NSOperation+Additions.h"

//controller
#import "WCMakeVirtualCardController.h"
#import "WCTCardDBController.h"
#import "WCCardImageController.h"
#import "PPIndexingController.h"

#import "PPLogController.h"

#if TARGET_OS_IPHONE
#import "UIImage+Additions.h"
#elif TARGET_OS_MAC
#import "NSImage+Additions.h"
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

#define DefaultGroupCount 2

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCTDataController
{
    NSString					*baseDirPath_;
//    WC_SourceID                 sourceID_;
    WCDC_AccessMode             accessMode_;
    WCTCardDBController			*cardDBController_;
    WCCardImageController		*cardImageController_;
    BOOL                        isImporting_;
    BOOL                        filterDuplicate_;
    WCSC_DisplyRule				importDisplayRule_;
    NSOperationQueue            *operationQueue_;
    WCDuplicateCompareModel     *duplicateCompareModel_;
    NSError                     *lastError_;
    BOOL                        withoutGroup_;
    BOOL                        useSharedDBInstance_;
    BOOL                        isExistingDB_;
}

@synthesize lastError = lastError_;

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//===============================================================================
//
//===============================================================================
- (instancetype)initWithAccessMode:(WCDC_AccessMode)accessMode
                       baseDirPath:(NSString *)baseDirPath
               useSharedDBInstance:(BOOL)useSharedDBInstance
                      isExistingDB:(BOOL)isExistingDB
{
    if(self = [super init])
    {
        cardDBController_ = nil;
        cardImageController_ = nil;
        isImporting_ = NO;
        operationQueue_ = nil;
        duplicateCompareModel_ = nil;
        lastError_ = nil;
        
        accessMode_ = accessMode;
        baseDirPath_ = [baseDirPath retain];
        useSharedDBInstance_ = useSharedDBInstance;
        isExistingDB_ = isExistingDB;
        
        if((accessMode_ & WCDC_AM_Text) && ![self loadCardDBController])
        {
            [self dealloc];
            return nil;
        }
        
        if((accessMode_ & WCDC_AM_Image) && ![self loadCardImageController])
        {
            [self dealloc];
            return nil;
        }
    }
    
    return self;
}


//===============================================================================
// Default initial for normal access
//===============================================================================
- (instancetype)initWithAccessMode:(WCDC_AccessMode)accessMode
{
    NSString *dataDirPath = [WCTDataController dataDirPath];
    
    return [self initWithAccessMode:accessMode
                        baseDirPath:dataDirPath
                useSharedDBInstance:YES
                       isExistingDB:NO];
}


//===============================================================================
// Special initial for parsing backup data
//===============================================================================
- (instancetype)initWithBaseDirPath:(NSString *)baseDirPath
{
    return [self initWithAccessMode:WCDC_AM_Text
                        baseDirPath:baseDirPath
                useSharedDBInstance:NO
                       isExistingDB:YES];
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
    [self freeCardDBController];
    [self freeCardImageController];

    [baseDirPath_ release];
    [duplicateCompareModel_ release];

    if(operationQueue_)
    {
        @synchronized(operationQueue_)
        {
            [operationQueue_ cancelAllOperations];
            [operationQueue_ waitUntilAllOperationsAreFinished];
            [operationQueue_ release];
            operationQueue_ = nil;
        }
    }

    self.lastError = nil;

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//===============================================================================
//
//===============================================================================
- (void)setLastErrorWithCode:(NSInteger)code description:(NSString *)description
{
    self.lastError = nil;

    NSMutableDictionary *userInfo = nil;

    if([description length])
        userInfo = [NSMutableDictionary dictionaryWithObjectsAndKeys:description, NSLocalizedDescriptionKey, nil];

    self.lastError = [NSError errorWithDomain:WCDC_ErrorDomain code:code userInfo:userInfo];
}


//===============================================================================
//
//===============================================================================
- (WCCacheModel *)cacheOfCurrentSource
{
    return [WCCacheModel defaultCache];
}


#pragma mark ##### Life cycle #####


//================================================================================
//
//================================================================================
- (WCTCardDBController *)cardDBController
{
    return cardDBController_;
}


//===============================================================================
//
//===============================================================================
- (BOOL)loadCardDBController
{
    if (cardDBController_) {
        return YES;
    }

    BOOL result = NO;

    if(isExistingDB_ == YES)
    {
        //-----------------------------------------
        // 不是使用主要資料庫時（例如還原），不用檢查，直接建立物件。
        //-----------------------------------------
        cardDBController_ = [[WCTCardDBController alloc] initWithPath:[baseDirPath_ stringByAppendingFormat:@"/%@", WCDC_DBName]];

        result = [cardDBController_ connectDB];
        
        // debug for DB not created
        NSAssert((cardDBController_ != nil), @"WCTCardDBController initWithPath failed when using exist DB");
    }
    else
    {
        if(useSharedDBInstance_ == NO)
        {
            cardDBController_ = [[WCTCardDBController alloc] initWithPath:[baseDirPath_ stringByAppendingFormat:@"/%@", WCDC_DBName]];
        }
        else
        {
            NSString *dbPath = [NSString stringWithFormat:@"%@/%@", baseDirPath_, WCDC_DBName];
            cardDBController_ = [WCTCardDBController sharedWCTCardDBControllerWithDBPath:dbPath] ;
        }
        
        // debug for DB not created
        NSAssert((cardDBController_ != nil), @"WCTCardDBController initWithPath failed when create new DB");

        if(cardDBController_)
        {
            if([cardDBController_ isDBExist])
            {
                result = [cardDBController_ connectDB];
            }
            else
            {
                //-----------------------------------------------------
                // 如果資料庫不存在，重新建立。
                //-----------------------------------------------------
                if([cardDBController_ createDB])
                {
                    if(withoutGroup_ == NO)
                    {
                        NSString *groupAllGuid = [NSString GUID];
                        
                        [cardDBController_ addGroupWithName:@"MLS_All" ID:WC_GID_All guid:groupAllGuid superGroupGuid:@"" helper:@""];
                        [cardDBController_ addGroupWithName:@"MLS_Unfiled" ID:WC_GID_Unfiled guid:[NSString GUID] superGroupGuid:groupAllGuid helper:@""];
                    }

                    result = YES;
                }
                else
                {
                    // debug for DB not created
                    NSAssert(NO, @"cardDBController_ createDB failed");
                }
            }


            if(result == NO)
            {
                [cardDBController_ release];
                cardDBController_ = nil;
            }

#ifdef _DUMP_LOG_
            [g_dumpLog logByFormat:@"%s create new DB - thread:%p retain:%d", __FUNCTION__, [NSThread currentThread], [cardDBController_ retainCount]];
#endif
        }
    }

    return result;
}


//===============================================================================
//
//===============================================================================
- (void)freeCardDBController
{
    if (!cardDBController_) {
        return;
    }

    if (isExistingDB_ == YES || useSharedDBInstance_ == NO)
    {
        [cardDBController_ release];
        cardDBController_ = nil;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)loadCardImageController
{
    if (cardImageController_) {
        return YES;
    }

    NSString *imageDir = [NSString stringWithFormat:@"%@/images", baseDirPath_];
    cardImageController_ = [[WCCardImageController alloc] initWithDirPath:imageDir];

    return (cardImageController_ != nil);
}


//===============================================================================
//
//===============================================================================
- (void)freeCardImageController
{
    [cardImageController_ release];
    cardImageController_ = nil;
}


#pragma mark ##### Send notify #####

//===============================================================================
// PARAMETER: paramArray -> index 0 : notify name
//                                1 : user info
//===============================================================================
- (void)mainThreadSendNotify:(NSArray *)paramArray
{
    NSString *notifyName = [paramArray objectAtIndex:0];
    NSDictionary *userInfo = nil;

    if([paramArray count] > 1)
        userInfo = [paramArray objectAtIndex:1];

    [[NSNotificationCenter defaultCenter] postNotificationName:notifyName object:nil userInfo:userInfo];
}


//===============================================================================
//
//===============================================================================
- (void)notifyDataChanged:(NSString *)notifyName infoKey:(NSString *)infoKey infoValue:(id)infoValue
{
    if([infoKey length] && infoValue)
    {
        NSDictionary *userInfo = [[NSDictionary alloc] initWithObjectsAndKeys:infoValue, infoKey, nil];
        NSArray *paramArray = [[NSArray alloc] initWithObjects:notifyName, userInfo, nil];

        [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];

        [userInfo release];
        [paramArray release];
    }
    else
    {
        NSArray *paramArray = [[NSArray alloc] initWithObjects:notifyName, nil];

        [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];

        [paramArray release];
    }
}


//===============================================================================
//
//===============================================================================
- (void)notifyImageChanged:(WCDC_NotifyChangeType)imageChange imageType:(WC_ImageType)imageType cardID:(NSString *)cardID
{
    NSNumber *numImageChange = [[NSNumber alloc] initWithInt:imageChange];
    NSNumber *numImageType = [[NSNumber alloc] initWithInt:imageType];
    NSDictionary *userInfo = [[NSDictionary alloc] initWithObjectsAndKeys:
                              numImageChange, WCDC_NOTIFY_UserInfo_kChangeType,
                              numImageType, WCDC_NOTIFY_UserInfo_kImageType,
                              cardID, WCDC_NOTIFY_UserInfo_kCardID, nil];
    NSArray *paramArray = [[NSArray alloc] initWithObjects:WCDC_NOTIFY_DisplayImageChanged, userInfo, nil];

    [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];

    [paramArray release];
    [userInfo release];
    [numImageType release];
    [numImageChange release];
}


//===============================================================================
//
//===============================================================================
- (void)notifyLoadCardSectionsResult:(BOOL)result error:(NSError *)error
{
    NSMutableDictionary *userInfo = [[NSMutableDictionary alloc] init];
    
    [userInfo setObject:[NSNumber numberWithBool:result] forKey:WCDC_NOTIFY_UserInfo_kResult];
    
    if(error != nil)
    {
        [userInfo setObject:error forKey:WCDC_NOTIFY_UserInfo_kError];
    }
    
    NSArray *paramArray = [[NSArray alloc] initWithObjects:WCDC_NOTIFY_LoadCardSectionsResult, userInfo, nil];

    [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];

    [paramArray release];
    [userInfo release];
}


//===============================================================================
//
//===============================================================================
- (void)notifyMoveCardProgress:(CGFloat)percentage
{
    NSDictionary *userInfo = [[NSDictionary alloc] initWithObjectsAndKeys:[NSNumber numberWithFloat:percentage], WCDC_NOTIFY_UserInfo_kPercentage, nil];
    NSArray *paramArray = [[NSArray alloc] initWithObjects:WCDC_NOTIFY_MoveCardProgress, userInfo, nil];

    [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];

    [paramArray release];
    [userInfo release];
}


//===============================================================================
//
//===============================================================================
- (void)notifyLoadFavoritesResult:(BOOL)result
{
    NSDictionary *userInfo = @{WCDC_NOTIFY_UserInfo_kResult:@(result)};
    NSArray *paramArray = [[NSArray alloc] initWithObjects:WCDC_NOTIFY_LoadFavoritesResult, userInfo, nil];
    
    [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:paramArray waitUntilDone:YES];
    
    [paramArray release];
}


#pragma mark ##### db info #####


//===============================================================================
//
//===============================================================================
- (NSString *)infoValueWithKey:(NSString *)infoKey
{
    return [cardDBController_ infoValueWithKey:infoKey];
}


//==============================================================================
//
//==============================================================================
- (void)setDBVersionInfo:(NSString *)versionInfo;
{
    if(versionInfo)
    {
        [cardDBController_ updateInfoValue:versionInfo withKey:WCTCDBC_IK_Version];
    }
}



#pragma mark ##### time section title handler #####


//================================================================================
// 使用日期產生sectionTitle
// 規則:
// 1。與今天同一週的，一天一個title
// 2。與今天同一個月，但不同週的，為本月
// 3。與今天不同月份的，以月為單位，一個月一個title
//================================================================================
- (NSString *)sectionTitleForDate:(NSDate *)date
{
    NSString *title = [NSDateFormatter localizedStringFromDate:date
                                                     dateStyle:NSDateFormatterShortStyle
                                                     timeStyle:NSDateFormatterNoStyle];
    
    do
    {
        //取得今天年月日
        NSDateComponents *nowComponents = [[NSCalendar currentCalendar] components:NSCalendarUnitYear|NSCalendarUnitMonth|NSCalendarUnitDay|NSCalendarUnitWeekdayOrdinal
                                                                          fromDate:[NSDate date]];
        if(nowComponents==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSDateComponents *dateComponents = [[NSCalendar currentCalendar] components:NSCalendarUnitYear|NSCalendarUnitMonth|NSCalendarUnitDay|NSCalendarUnitWeekdayOrdinal
                                                                                fromDate:date];
        if(dateComponents==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
        
        if(nowComponents.year==dateComponents.year &&
           nowComponents.month==dateComponents.month &&
           nowComponents.weekdayOrdinal==dateComponents.weekdayOrdinal)
        {
            [dateFormatter setDateFormat:@"yyyy.MM.dd"];
            
            title = [dateFormatter stringFromDate:date];
        }
        else if(nowComponents.year==dateComponents.year &&
                nowComponents.month==dateComponents.month)
        {
            title = [@"MLS_ThisMonth" localized];
        }
        else
        {
            [dateFormatter setDateFormat:@"yyyy.MM"];
            
            title = [dateFormatter stringFromDate:date];
        }
        
        
        [dateFormatter release];
    }
    while (0);
    
    return title;
}


#pragma mark ##### Image #####


//===============================================================================
// 檢查是否有名片圖
//===============================================================================
- (NSInteger)cardImageCountWithCardID:(NSString *)cardID
{
    NSInteger imageCount = 0;
    if ([self hasCardImageWithCardID:cardID type:WC_IT_FrontSide])
    {
        imageCount++;
    }
    if ([self hasCardImageWithCardID:cardID type:WC_IT_BackSide])
    {
        imageCount++;
    }
    if ([self hasCardImageWithCardID:cardID type:WC_IT_IDPhoto])
    {
        imageCount++;
    }
    return imageCount;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCDataControllerProtocol_Image

//===============================================================================
// 檢查是否有名片圖
//===============================================================================
- (BOOL)hasCardImageWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [cardImageController_ hasImageWithCardID:cardID type:type];
}


//================================================================================
//
//================================================================================
- (NSString *)cardImagePathWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    NSString *path = [cardImageController_ imagePathWithCardID:cardID type:type subtype:subtype];
    
    if([[NSFileManager defaultManager] fileExistsAtPath:path] == NO)
    {
        path = nil;
    }
    
    return path;
}


//===============================================================================
// 取得假名片圖
// return : CPImage (release after using)
//===============================================================================
- (CPImage *)copyVirtualCardImageWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    //--------------------------------
    // 只有正面能產生假名片圖
    //--------------------------------
    if (type != WC_IT_FrontSide) {
        return nil;
    }

    //--------------------------------
    // 需要取資料，所以要確認資料庫有開啓。
    //--------------------------------
    if(!cardDBController_)
    {
        if([self loadCardDBController])
            accessMode_ |= WCDC_AM_Text;
        else return nil;
    }


    //--------------------------------
    // 建立假名片圖
    //--------------------------------
    NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
    WCCardModel         *cardModel = [self copyCardFullDataWithCardID:cardID];
    CPImage             *image = nil;

    if(cardModel)
    {
        CPImage *virtualImage = nil;

        //////////////////////////////////////////////////
        // 產生假名片
        WCMakeVirtualCardController      *makeVirtualCardController = [[WCMakeVirtualCardController alloc] init];

        CPImage *IDPhoto = [cardImageController_ copyImageWithCardID:cardModel.ID type:WC_IT_IDPhoto subtype:WC_IST_Thumbnail];
        if (IDPhoto)
        {
            makeVirtualCardController.headImage = IDPhoto;
            [IDPhoto release];
        }
        
        makeVirtualCardController.cardModel = cardModel;
        
        virtualImage = [[makeVirtualCardController virtualCardImage] retain];
        [makeVirtualCardController release];
        
        //////////////////////////////////////////////////
        // !! 產生的假名片圖是720x440，所以如果是WC_IST_Thumbnail還要再縮小
        if(subtype == WC_IST_Thumbnail)
        {
            image = [[cardImageController_ rotateAndResizeImage:virtualImage rotateDegree:0 maxSize:WC_IMS_Thumbnail] retain];
        }
        else
        {
            image = [virtualImage retain];
        }

        [virtualImage release];
        [cardModel release];
    }

    [pool release];

    return image;
}


//===============================================================================
// 取得名片圖 （沒有影像時會直接回傳假名片圖）
// return : CPImage (release after using)
//===============================================================================
- (CPImage *)copyCardImageWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;

    CPImage *cardImage = nil;
    WCCacheModel *cacheModel = [self cacheOfCurrentSource];


    //--------------------------------------------
    // !! 取圖時先檢查是否有暫存影像 (僅正面縮圖, 大頭貼)
    //--------------------------------------------
    if(cacheModel && subtype == WC_IST_Thumbnail)
    {
        if (type == WC_IT_FrontSide)
        {
            cardImage = [cacheModel thumbImageWithCardID:cardID];
            
            if(cardImage)
                return [cardImage retain];
        }
        else if (type == WC_IT_IDPhoto)
        {
            cardImage = [cacheModel idPhotoWithCardID:cardID];
            
            if(cardImage)
                return [cardImage retain];
        }
    }


    //--------------------------------------------
    // copy image from file
    //--------------------------------------------
    cardImage = [cardImageController_ copyImageWithCardID:cardID type:type subtype:subtype];


    //--------------------------------------------
    // !! 讀取完加入暫存。(僅正面縮圖, 大頭貼)
    //--------------------------------------------
    if(cardImage && subtype == WC_IST_Thumbnail)
    {
        if (type == WC_IT_FrontSide)
        {
            [cacheModel addThumbImage:cardImage withCardID:cardID];
        }
        else if (type == WC_IT_IDPhoto)
        {
            [cacheModel addIDPhoto:cardImage withCardID:cardID];
        }
    }


    //--------------------------------------------
    // !! generate virtual card image (只有frontSide)
    //--------------------------------------------
    if(!cardImage)
    {
        cardImage = [self copyVirtualCardImageWithCardID:cardID type:type subtype:subtype];
        
        if(cardImage && type == WC_IT_FrontSide && subtype == WC_IST_Thumbnail)
            [cacheModel addThumbImage:cardImage withCardID:cardID];
    }
    return cardImage;
}


//===============================================================================
// 取得名片圖資料
// return : NSData (release after using)
//===============================================================================
- (NSData *)copyCardImageDataWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    return [cardImageController_ copyImageDataWithCardID:cardID type:type subtype:subtype];
}


//===============================================================================
// 設定名片圖
//===============================================================================
- (BOOL)setCardImage:(CPImage *)image withCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [self setCardImage:image imageSHA1:nil withCardID:cardID type:type];
}


//===============================================================================
// 設定名片圖
//===============================================================================
- (BOOL)setCardImage:(CPImage *)image withCardID:(NSString *)cardID type:(WC_ImageType)type isImport:(BOOL)isImport
{
    return [self setCardImage:image imageSHA1:nil withCardID:cardID type:type isImport:isImport];
}


//===============================================================================
// 設定名片圖資料
//===============================================================================
- (BOOL)setCardImageData:(NSData *)imageData withCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [self setCardImageData:imageData imageSHA1:nil withCardID:cardID type:type];
}


//===============================================================================
// 設定名片圖資料
//===============================================================================
- (BOOL)setCardImageData:(NSData *)imageData withCardID:(NSString *)cardID type:(WC_ImageType)type isImport:(BOOL)isImport
{
    return [self setCardImageData:imageData imageSHA1:nil withCardID:cardID type:type isImport:isImport];
}


//===============================================================================
// 刪除指定的名片圖
//===============================================================================
- (void)removeCardImageWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    if([cardImageController_ hasImageWithCardID:cardID type:type] == YES)
    {
        //cache
        WCCacheModel *cacheModel = [self cacheOfCurrentSource];
        [cacheModel removeThumbImageWithCardID:cardID];
        [cacheModel removeIDPhotoWithCardID:cardID];
        
        [cardImageController_ removeImageWithCardID:cardID type:type];
        
        
        // !! update modified time of card
        if(!cardDBController_)
        {
            if([self loadCardDBController])
                accessMode_ |= WCDC_AM_Text;
        }
        
        [cardDBController_ updateModifiedTimeWithCardID:cardID];
        [cardDBController_ updateCardSyncActionForImageChangeWithCardID:cardID imageType:type modifiedTime:nil];
        
        // !! notify image changed
        [self notifyImageChanged:WCDC_NCT_Delete imageType:type cardID:cardID];    
    }
}


//===============================================================================
// 刪除一筆名片所有關聯的圖
//===============================================================================
- (void)removeAllCardImagesWithCardID:(NSString *)cardID
{
    [cardImageController_ removeImageWithCardID:cardID type:WC_IT_FrontSide];
    [cardImageController_ removeImageWithCardID:cardID type:WC_IT_BackSide];
    [cardImageController_ removeImageWithCardID:cardID type:WC_IT_IDPhoto];
}


//==============================================================================
// card holder list顯示圖片用的
// 會先取大頭貼，再取名片圖
//==============================================================================
- (CPImage *)cachedDisplayCardImageWithCardID:(NSString *)cardID
{
    //!! 先取大頭貼，沒有時才取名片圖
    WCCacheModel *cacheModel = [WCCacheModel defaultCache];
    CPImage *cachedImage = [cacheModel idPhotoWithCardID:cardID];
    if (cachedImage==nil)
    {
        cachedImage = [cacheModel thumbImageWithCardID:cardID];
    }
    return cachedImage;
}


//==============================================================================
// card holder list顯示圖片用的
// 會先取大頭貼，再取名片圖
//==============================================================================
- (CPImage *)displayCardImageWithCardID:(NSString *)cardID
{
    //!! 先取大頭貼，沒有時才取名片圖
    CPImage *thumbImage = [[self copyCardImageWithCardID:cardID
                                                   type:WC_IT_IDPhoto
                                                subtype:WC_IST_Thumbnail] autorelease];
    if (thumbImage==nil)
    {
        thumbImage = [[self copyCardImageWithCardID:cardID
                                              type:WC_IT_FrontSide
                                           subtype:WC_IST_Thumbnail] autorelease];
    }
    return thumbImage;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Favorite methods

//==============================================================================
//
//==============================================================================
- (void)bgThreadLoadFavoritesWithSearchText:(NSString *)searchText
{
    NSMutableArray *favorites = nil;
    
    favorites = [self copyFavoritesWithSearchText:searchText];
    
    WCCacheModel *cacheModel = [WCCacheModel defaultCache];
    if ([searchText length]>0)
    {
        cacheModel.searchFavoriteCardsArray = favorites;
    }
    else
    {
        [cacheModel clearFavoriteCachedData];
        
        for (WCCardModel *cardModel in favorites)
        {
            [cacheModel addFavoriteCardToCache:cardModel];
        }
    }
    
    [favorites release];
    
    [self notifyLoadFavoritesResult:YES];
}


//===============================================================================
// 取得所有favorite
// return : array of cardID (release after using)
//===============================================================================
- (NSMutableArray *)copyAllFavorites
{
    NSMutableArray *favoritesArray = [cardDBController_ copyAllFavorites];
    if (!favoritesArray) {
        self.lastError = cardDBController_.lastError;
    }
    
    return favoritesArray;
}


#pragma mark - WCDataControllerProtocol_Favorites


//==============================================================================
//
//==============================================================================
- (void)loadFavoriteCardWithSearchText:(NSString *)searchText
{
    @synchronized(operationQueue_)
    {
        if(!operationQueue_)
        {
            operationQueue_ = [[NSOperationQueue alloc] init];
            [operationQueue_ setMaxConcurrentOperationCount:1];
        }
//        else if([operationQueue_ operationCount])
//        {
//            [operationQueue_ cancelAllOperations];
//        }
        
        NSInvocationOperation *operation = [[NSInvocationOperation alloc] initWithTarget:self selector:@selector(bgThreadLoadFavoritesWithSearchText:) object:searchText];
        
        [operationQueue_ addOperation:operation];
        [operation release];
    }
}


//===============================================================================
// 搜尋favorite
// return : array of WCCardModel (release after using)
//===============================================================================
- (NSMutableArray *)copyFavoritesWithSearchText:(NSString *)searchText
{
    NSMutableArray *favorites =  [cardDBController_ copyFavoritesWithSearchText:searchText];
    return favorites;
}


//===============================================================================
// 是否設定為favorite
//===============================================================================
- (BOOL)isFavorite:(NSString *)cardID
{
    return ([cardDBController_ favoriteOrderWithCardID:cardID] != -1);
}


//===============================================================================
// 加入favorite
//===============================================================================
- (BOOL)addFavoriteWithCardID:(NSString *)cardID
{
    BOOL result = [cardDBController_ addFavoriteWithCardID:cardID isInTransaction:NO];

    if(result)
    {
        //////////////////////////////////////////////////
        // 調整syncAction紀錄
        
        [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardID modifiedTime:nil];

        
        //////////////////////////////////////////////////
        // 調整cache
        
        WCCardModel *cardModel = nil;
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
#if TARGET_OS_IPHONE
        
        cardModel = [self copyCardDisplayDataWithCardID:cardID];
        [cacheModel addFavoriteCardToCache:cardModel];
        [cardModel release];
        
#elif TARGET_OS_MAC
        
        if(cacheModel.groupID == WC_GID_Favorite)
        {
            cardModel = [self copyCardDisplayDataWithCardID:cardID];
            
            if(cardModel != nil)
            {
                [cacheModel addCardToCache:cardModel];
                [cardModel release];            
            }
        }
        else
        {
            cardModel = [cacheModel cardFromCacheWithCardID:cardID];
            
            if(cardModel != nil)
            {
                cardModel.tagMask |= WC_TagMask_Favorite;
            }
        }
        
#endif
        
        
        //////////////////////////////////////////////////
        // notify UI update
        
        [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kAddFavorite
                      infoValue:[NSArray arrayWithObject:cardID]];
    }
    else
    {
        //接SQL error
        [self setLastErrorWithCode:[cardDBController_ dbErrorCode] description:[cardDBController_ dbErrorMessage] ];
    }

    return result;
}


//===============================================================================
// 移除favorite
//===============================================================================
- (BOOL)removeFavoriteWithCardID:(NSString *)cardID
{
    BOOL result = [cardDBController_ removeFavoriteWithCardID:cardID isInTransaction:NO];

    if(result)
    {
        //////////////////////////////////////////////////
        // 調整syncAction紀錄
        
        [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardID modifiedTime:nil];


        //////////////////////////////////////////////////
        // 調整cache
        
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
#if TARGET_OS_IPHONE

        [cacheModel removeFavoriteCardFromCacheWithCardID:cardID];
        
#elif TARGET_OS_MAC
        
        if(cacheModel.groupID == WC_GID_Favorite)
        {
            [cacheModel removeCardFromCacheWithCardID:cardID];
        }
        else
        {
            WCCardModel *cardModel = [cacheModel cardFromCacheWithCardID:cardID];
            
            if(cardModel != nil)
            {
                cardModel.tagMask &= ~WC_TagMask_Favorite;
            }
        }
        
#endif
        
        
        //////////////////////////////////////////////////
        // notify UI update
        
        [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kRemoveFavorite
                      infoValue:[NSArray arrayWithObject:cardID]];
    }

    return result;

}


//===============================================================================
// 變更favorite順序
//===============================================================================
- (BOOL)moveFavoriteWithCardID:(NSString *)cardID toOrder:(NSInteger)toOrder
{
    return [cardDBController_ moveFavoriteWithCardID:cardID toOrder:toOrder];
}


//==============================================================================
//
//==============================================================================
- (void)setActionType:(NSInteger)actionType withCardID:(NSString *)cardID
{
    [cardDBController_ setActionType:actionType withCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (NSInteger)actionTypeWithCardID:(NSString *)cardID
{
    return [cardDBController_ actionTypeWithCardID:cardID];
}


//================================================================================
//
//================================================================================
- (NSInteger)favoriteCount
{
    return [cardDBController_ favoriteCount];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Unverified methods
#pragma mark - Private

//===============================================================================
// 取得所有Unverified
// return : array of cardID (release after using)
//===============================================================================
- (NSMutableArray *)copyAllUnverified
{
    return [cardDBController_ copyAllUnverified];
}



//===============================================================================
// 加入Unverified
//===============================================================================
- (BOOL)addUnverifiedWithCardID:(NSString *)cardID withNotification:(BOOL)isSendNotification
{
    BOOL result = [cardDBController_ addUnverifiedWithCardID:cardID];
    
    if(result)
    {
        //////////////////////////////////////////////////
        // 調整syncAction紀錄
        
        [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardID modifiedTime:nil];

        
        //////////////////////////////////////////////////
        // 調整cache
        
        WCCardModel *cardModel = nil;
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
#if TARGET_OS_IPHONE
        
        cardModel = [cacheModel cardFromCacheWithCardID:cardID];
        
        if (cardModel != nil)
        {
            // TODO: 這裡為什麼是拿掉tag?
            cardModel.tagMask &= ~WC_TagMask_Unverified;
            [cacheModel updateCardInCache:cardModel];
        }
        
        
        //////////////////////////////////////////////////
        // notify UI update
        
        if(isSendNotification)
        {
            [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }
        
        
        //////////////////////////////////////////////////
        // 我的最愛那邊的名片也要更新cache
        
        WCCardModel *cachedFavoritesCardModel = [cacheModel favoriteCardFromCacheWithCardID:cardID];
        if (cachedFavoritesCardModel!=nil)
        {
            cachedFavoritesCardModel.tagMask &= ~WC_TagMask_Unverified;
            [cacheModel updateFavoriteCardInCache:cachedFavoritesCardModel];
        }
        
        if(isSendNotification)
        {
            // notify UI update
            [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }

#elif TARGET_OS_MAC
        
        if(cacheModel.groupID == WC_GID_Unverified)
        {
            cardModel = [self copyCardDisplayDataWithCardID:cardID];
            
            if(cardModel != nil)
            {
                [cacheModel addCardToCache:cardModel];
                [cardModel release];
            }
        }
        else
        {
            cardModel = [cacheModel cardFromCacheWithCardID:cardID];
            
            if(cardModel)
            {
                cardModel.tagMask |= WC_TagMask_Unverified;
            }
        }
        
        
        //////////////////////////////////////////////////
        // notify UI update
        
        if(isSendNotification)
        {
            [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }
        
#endif

    }
    
    
    return result;
}


//===============================================================================
// 移除Unverified
//===============================================================================
- (BOOL)removeUnverifiedWithCardID:(NSString *)cardID withNotification:(BOOL)isSendNotification updateCardSyncAction:(BOOL)updateCardSyncAction;
{
    BOOL result = [cardDBController_ removeUnverifiedWithCardID:cardID];
    
    if(result)
    {
        [cardDBController_ updateModifiedTimeWithCardID:cardID];
        
        
        //////////////////////////////////////////////////
        // 調整syncAction紀錄
        
        if(updateCardSyncAction==YES)
        {
            [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardID modifiedTime:nil];
        }
        
        //////////////////////////////////////////////////
        // 調整cache
        
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
#if TARGET_OS_IPHONE
        
        WCCardModel *cachedCardModel = [cacheModel cardFromCacheWithCardID:cardID];
        if (cachedCardModel!=nil)
        {
            cachedCardModel.tagMask &= ~WC_TagMask_Unverified;
            [cacheModel updateCardInCache:cachedCardModel];
        }
        
        if(isSendNotification)
        {
            // notify UI update
            [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }
        
        
        //////////////////////////////////////////////////
        // 我的最愛那邊的名片也要更新cache
        
        WCCardModel *cachedFavoritesCardModel = [cacheModel favoriteCardFromCacheWithCardID:cardID];
        if (cachedFavoritesCardModel!=nil)
        {
            cachedFavoritesCardModel.tagMask &= ~WC_TagMask_Unverified;
            [cacheModel updateFavoriteCardInCache:cachedFavoritesCardModel];
        }
        
        // notify UI update
        if(isSendNotification)
        {
            [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }
        
#elif TARGET_OS_MAC
        
        if(cacheModel.groupID == WC_GID_Unverified)
        {
            [cacheModel removeCardFromCacheWithCardID:cardID];
        }
        else
        {
            WCCardModel *cardModel = [cacheModel cardFromCacheWithCardID:cardID];
            
            if(cardModel != nil)
            {
                cardModel.tagMask &= ~WC_TagMask_Unverified;
            }
        }
        
        // notify UI update
        if(isSendNotification)
        {
            [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                          infoValue:[NSArray arrayWithObject:cardID]];
        }
        
#endif
        
        
    }
    
    return result;
}

//===============================================================================
// 移除Unverified
//===============================================================================
- (BOOL)removeUnverifiedWithCardID:(NSString *)cardID withNotification:(BOOL)isSendNotification
{
    return [self removeUnverifiedWithCardID:cardID withNotification:isSendNotification updateCardSyncAction:YES];
}


#pragma mark - WCDataControllerProtocol_Unverified


//===============================================================================
// 取得所有Unverified
// return : array of WCCardSectionModel (release after using)
//===============================================================================
- (NSMutableArray *)copyAllUnverifiedCardSection
{
    NSMutableArray *cardArray = nil;
    NSMutableArray *resultArray = nil;//[[NSMutableArray alloc] init];
    NSMutableDictionary *sectionDict = [NSMutableDictionary dictionary];
    NSMutableArray *allUnverified = [cardDBController_ copyAllUnverified];
    
    WC_SortedByField sortingByField = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
    PPSIC_Mode indexMode = [PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
    
    // 依sectionTitle分敗
    for (WCCardModel * cardModel in allUnverified)
    {
        NSString *currentSectionTitle = [cardModel displaySectionTitleWithSortByField:sortingByField indexingMode:indexMode];
        
        if ([currentSectionTitle length]==0)
        {
            continue;
        }
        
        //////////////////////////////////////////////////
        if((cardArray = [sectionDict objectForKey:currentSectionTitle]))
        {
            // 要包含是否已校正的資訊
            [cardArray addObject:cardModel];
        }
        else
        {
            cardArray = [[NSMutableArray alloc] initWithObjects:cardModel, nil];
            [sectionDict setObject:cardArray forKey:currentSectionTitle];
            [cardArray release];
        }
    }
    
    //////////////////////////////////////////////////
    // 組成cardSectionModel array
    for (NSString *sectionTitle in sectionDict.allKeys)
    {
        if (resultArray==nil)
        {
            resultArray = [[NSMutableArray alloc] init];
        }
        WCCardSectionModel *cardSectionModel = [[WCCardSectionModel alloc] init];
        cardSectionModel.title = sectionTitle;
        cardSectionModel.cardArray = [sectionDict objectForKey:sectionTitle];

        //////////////////////////////////////////////////
        // sort
        [WCCompareTool cardArray:cardSectionModel.cardArray sortByField:sortingByField];
        
        cardSectionModel.needResortCard = NO;
        [resultArray addObject:cardSectionModel];
        [cardSectionModel release];
    }
    [allUnverified release];
    
    return resultArray;
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasUnverified
{
    return ([cardDBController_ unverifiedCount] > 0);
}


//==============================================================================
//
//==============================================================================
- (NSInteger)unverifiedCount
{
    return [cardDBController_ unverifiedCount];
}


//==============================================================================
//
//==============================================================================
- (BOOL)isUnverifiedWithCardID:(NSString *)cardID
{
    return [cardDBController_ isUnverifiedWithCardID:cardID];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Custom field info


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyAllCustomFieldInfos
{
    return [cardDBController_ copyAllCustomFieldInfos];
}


//==============================================================================
//
//==============================================================================
- (WCCustomFieldInfo *)customFieldInfoWithGuid:(NSString *)guid
{
    return [cardDBController_ customFieldInfoWithGuid:guid];
}


//==============================================================================
//
//==============================================================================
- (BOOL)addCustomFieldWithInfo:(WCCustomFieldInfo *)customFieldInfo
{
    return [cardDBController_ addCustomFieldWithInfo:customFieldInfo];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCustomFieldWithInfo:(WCCustomFieldInfo *)customFieldInfo
{
    return [cardDBController_ updateCustomFieldWithInfo:customFieldInfo];
}

//==============================================================================
//
//==============================================================================
- (void)removeCustomFieldWithGuid:(NSString *)guid
{
    [cardDBController_ removeCustomFieldWithGuid:guid];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group methods
#pragma mark - WCDataControllerProtocol_Group

//===============================================================================
// 判斷群組名稱是否已經存在 （groupName須為完整路徑）
//===============================================================================
- (BOOL)isGroupNameExist:(NSString *)groupName
{
    NSString *groupID = [self groupIDWithName:groupName];
    
    return ([groupID integerValue] != WC_GID_None);
}


//===============================================================================
// 新增群組
// return : WCGroupModel (release after using)
//===============================================================================
- (WCGroupModel *)newGroupWithName:(NSString *)groupName
{
    WCTGroupSyncActionModel *groupSyncActionModel = nil;
    if([groupName length]==0)
    {
        return nil;
    }

    groupSyncActionModel = [[WCTGroupSyncActionModel alloc] init];
    groupSyncActionModel.dataGuid = [NSString GUID];
    groupSyncActionModel.actionType = WCTSyncActionType_Add;
    groupSyncActionModel.modifiedTime = [NSDate date];
    
    WCGroupModel *groupModel = [self newGroupWithName:groupName syncActionModel:groupSyncActionModel];
    [groupSyncActionModel release];
    

    return groupModel;
}


//==============================================================================
//
//==============================================================================
- (void)updateOrderWithGroupIDArray:(NSArray *)groupIDArray
{
    [cardDBController_ updateOrderWithGroupIDArray:groupIDArray];
    
    [self setGroupOrderModifiedTime:[NSDate date]];
}


//===============================================================================
// 變更群組名稱
//===============================================================================
- (BOOL)updateGroupName:(NSString *)groupName withID:(NSString *)groupID
{
    if([groupID length]==0)
        return NO;
    
    WC_GroupID realGroupID = [groupID integerValue];
    
    WCTGroupSyncActionModel *groupSyncActionModel = [[[WCTGroupSyncActionModel alloc] init] autorelease];
    
    if(groupSyncActionModel != nil)
    {
        groupSyncActionModel.dataGuid = [cardDBController_ groupGuidWithID:realGroupID];
        groupSyncActionModel.actionType = WCTSyncActionType_Modify;
        groupSyncActionModel.modifiedTime = [NSDate date];
    }
    
    return [self updateGroupName:groupName withID:realGroupID syncActionModel:groupSyncActionModel];
}


//===============================================================================
// 刪除群組
//===============================================================================
- (BOOL)removeGroupWithID:(NSString *)groupID;
{
    if([groupID length]==0)
        return NO;
    
    WC_GroupID realGroupID = [groupID integerValue];

    NSString *groupGuid = [cardDBController_ groupGuidWithID:realGroupID];
    if ([groupGuid length]==0)
    {
        self.lastError = cardDBController_.lastError;
        return NO;
    }
    
    WCTGroupSyncActionModel *groupSyncActionModel = nil;
    groupSyncActionModel = [[[WCTGroupSyncActionModel alloc] init] autorelease];
    groupSyncActionModel.dataGuid = groupGuid;
    groupSyncActionModel.actionType = WCTSyncActionType_Delete;
    groupSyncActionModel.modifiedTime = [NSDate date];
    
    return [self removeGroupWithID:realGroupID syncActionModel:groupSyncActionModel];
}


//===============================================================================
// 取得所有群組資料
// getCardCount : 是否需要取得名片數量
// return : array of WCGroupModel (release after using)
//===============================================================================
- (NSMutableArray *)copyAllGroupsAndGetCardCount:(BOOL)getCardCount
{
    self.lastError = nil;

    NSMutableArray *groupArray = [cardDBController_ copyAllGroups];
    [groupArray sortUsingComparator:^NSComparisonResult(WCGroupModel * obj1, WCGroupModel *obj2) {
        
        if (obj1.order>obj2.order)
        {
            return NSOrderedDescending;
        }
        else if (obj1.order<obj2.order)
        {
            return NSOrderedAscending;
        }
        else
        {
            return NSOrderedSame;
        }
    }];
    
    //////////////////////////////////////////////////
    
    for(WCGroupModel *groupModel in groupArray)
    {
        //----------------------------------------------
        // set editable status and virtual group name
        //----------------------------------------------
        if(groupModel.ID <= WC_GID_Unfiled)
        {
            // !! groupID <= 100都是系統預設group，名稱從字串表取得，不可編輯。
            groupModel.name = [groupModel.name localized];
            groupModel.editable = NO;
        }
        else
        {
            groupModel.editable = YES;
        }


        //----------------------------------------------
        // get card count if need
        //----------------------------------------------
        if(getCardCount == YES)
        {
            groupModel.cardCount = [cardDBController_ cardCountOfGroup:groupModel.ID];
        }
    }

    self.lastError = cardDBController_.lastError;

    return groupArray;
}


//===============================================================================
// 取得群組ID string
// return : groupIDString, nil means group not found.
//===============================================================================
- (NSString *)groupIDWithName:(NSString *)groupName
{
    NSString *groupIDString = nil;
    
    
    //////////////////////////////////////////////////
    // 檢查是否為各語系的系統group
    
    NSMutableDictionary *mappingDict = [NSMutableDictionary dictionary];
    NSInteger defaultID[DefaultGroupCount] = {WC_GID_All, WC_GID_Unfiled};
    NSString *defaultNames[DefaultGroupCount] = {WC_GCGN_All, WC_GCGN_OtherContacts};
    NSArray *defaultNameArray = nil;
    
    for(int i=0; i<DefaultGroupCount; i++)
    {
        defaultNameArray = [defaultNames[i] componentsSeparatedByString:@","];
        
        for (NSString *name in defaultNameArray)
        {
            [mappingDict setObject:[NSString stringWithInteger:defaultID[i]] forKey:name ];
        }
    }
    
    groupIDString = [mappingDict objectForKey:groupName];
    
    if(groupIDString != nil)
    {
        return groupIDString;
    }


    //////////////////////////////////////////////////
    // 逐層檢查路徑名稱
    
    NSMutableArray *allGroups = [[self copyAllGroupsAndGetCardCount:NO] autorelease];
    
    if(allGroups != nil)
    {
        WCGroupModel *rootGroup = [WCGroupModel groupTreeWithGroups:allGroups];
        NSArray *nameLayers = [groupName pathComponents];
        NSArray *groups = rootGroup.subGroups;
        
        for(int i=0; i<[nameLayers count]; i++)
        {
            NSString *name = nameLayers[i];
            BOOL isFound = NO;
            
            for(WCGroupModel *groupModel in groups)
            {
                if([groupModel.name isEqualToString:name] == YES)
                {
                    if(i < [nameLayers count]-1)
                    {
                        // 找到繼續往下一層找
                        groups = groupModel.subGroups;
                    }
                    else
                    {
                        // 在最後一層找到，結束
                        groupIDString = [NSString stringWithInteger:groupModel.ID];
                    }
                    
                    isFound = YES;
                    break;
                }
            }
            
            // 沒找到就結束
            if(isFound == NO)
            {
                break;
            }
        }
    }

    return groupIDString;
}


//===============================================================================
// 取得群組名稱
//===============================================================================
- (NSString *)groupNameWithID:(NSString *)groupID
{
    if([groupID length]==0)
        return nil;
    
    WC_GroupID realGroupID = [groupID integerValue];

    if(realGroupID==-1)
    {
        return [@"MLS_AllCardGroupName" localized];
    }
    
    NSString *groupName = [cardDBController_ groupNameWithID:realGroupID];

    return [groupName localized];
}


//===============================================================================
// 取得多個群組model
// 不包含數量
//===============================================================================
- (NSArray *)groupModelsWithIDArray:(NSArray *)groupIDArray
{
    NSMutableArray *array = [NSMutableArray array];
    NSMutableArray *allgrouparray =[cardDBController_ copyAllGroups];
    for(int i=0;i<[groupIDArray count];i++)
    {
        NSString *groupID = [groupIDArray objectAtIndex:i];
        for(int j=0;j<[allgrouparray count];j++)
        {
            WCGroupModel *groupModel = [allgrouparray objectAtIndex:j];
            if([groupID integerValue] == groupModel.ID)
            {
                [array addObject:groupModel];
                break;
            }
        }
    }
    
    [allgrouparray release];
    return array;
}


//===============================================================================
// 取得多個群組名稱（逗號分隔）
//===============================================================================
- (NSString *)groupNamesWithIDArray:(NSArray *)groupIDArray
{
    NSMutableString *groupNames = [NSMutableString string];

    if(![groupIDArray count])
    {
        [groupNames appendFormat:@"%@", [self groupNameWithID:[NSString stringWithInteger:WC_GID_Unfiled]]];
    }
    else
    {
        for(NSString *oneGroupID in groupIDArray)
        {
            if([groupNames length])
                [groupNames appendString:@", "];

            [groupNames appendFormat:@"%@", [self groupNameWithID:oneGroupID]];
        }
    }

    return (NSString *)groupNames;
}


//==============================================================================
//
//==============================================================================
- (NSDate *)groupOrderModifiedTime
{
    return [cardDBController_ groupOrderModifiedTime];
}


//==============================================================================
//
//==============================================================================
- (void)setGroupOrderModifiedTime:(NSDate *)groupOrderModifiedTime
{
    [cardDBController_ setGroupOrderModifiedTime:groupOrderModifiedTime];
}


//==============================================================================
//
//==============================================================================
- (void)removeGroupOrderModifiedTime
{
    [cardDBController_ removeGroupOrderModifiedTime];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Indivisual Group methods

//================================================================================
//
//================================================================================
- (BOOL)addOrUpdateGroupWithGuid:(NSString *)guid name:(NSString *)name superGroupGuid:(NSString *)superGroupGuid helper:(NSString *)helper
{
    WCGroupModel *groupModel = [cardDBController_ groupWithGuid:guid];
    
    if(groupModel == nil)
    {
        // 新增
        return [cardDBController_ addGroupWithName:name ID:WC_GID_None guid:guid superGroupGuid:superGroupGuid helper:helper];
    }
    else
    {
        // 更新
        if([groupModel.name isEqualToString:name] == NO)
        {
            [cardDBController_ updateGroupName:name withID:groupModel.ID syncActionModel:nil];
        }
        
        if([groupModel.superGroupGuid isEqualToString:superGroupGuid] == NO)
        {
            [cardDBController_ updateSuperGroupGuid:superGroupGuid withID:groupModel.ID];
        }

        if([groupModel.helper isEqualToString:helper] == NO)
        {
            [cardDBController_ updateGroupHelper:helper withID:groupModel.ID];
        }

        return YES;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)updateSuperGroupGuid:(NSString *)superGroupGuid withGroupID:(WC_GroupID)groupID
{
    return [cardDBController_ updateSuperGroupGuid:superGroupGuid withID:groupID];
}

//================================================================================
//
//================================================================================
- (BOOL)clearAllGroupPinnedOrder
{
    return [cardDBController_ clearAllGroupPinnedOrder];
}


//================================================================================
//
//================================================================================
- (BOOL)updatePinnedOrder:(NSInteger)pinnedOrder withGroupGuid:(NSString *)groupGuid
{
    return [cardDBController_ updateGroupPinnedOrder:pinnedOrder withGuid:groupGuid];
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)pinnedGroupsAndGetCardCount:(BOOL)getCardCount;
{
    self.lastError = nil;
    
    NSMutableArray *groupArray = [cardDBController_ pinnedGroups];
    
    for(WCGroupModel *groupModel in groupArray)
    {
        // set editable status and virtual group name
        if(groupModel.ID <= WC_GID_Unfiled)
        {
            // !! groupID <= 100都是系統預設group，名稱從字串表取得，不可編輯。
            groupModel.name = [groupModel.name localized];
            groupModel.editable = NO;
        }
        else
        {
            groupModel.editable = YES;
        }
                
        // get card count if need
        if(getCardCount == YES)
        {
            groupModel.cardCount = [cardDBController_ cardCountOfGroup:groupModel.ID];
        }
    }
    
    self.lastError = cardDBController_.lastError;
    
    return groupArray;
}


//================================================================================
//
//================================================================================
- (NSString *)groupGuidWithID:(WC_GroupID)groupID
{
    return [cardDBController_ groupGuidWithID:groupID];
}


//================================================================================
//
//================================================================================
- (NSString *)groupHelperWithID:(WC_GroupID)groupID
{
    return [cardDBController_ groupHelperWithID:groupID];
}


//================================================================================
//
//================================================================================
- (void)setMustHaveGroupIDAfterRemoveGroup:(WC_GroupID)groupID
{
    [cardDBController_ setMustHaveGroupIDAfterRemoveGroup:groupID];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card-Group methods
#pragma mark - Private

//===============================================================================
// 設定卡片群組
// groupIDArray : groupID "String" array
//===============================================================================
- (BOOL)setCardGroupsWithCardID:(NSString *)cardID groupIDArray:(NSArray *)groupIDArray updateModifiedTime:(BOOL)updateModifiedTime updateCardSyncAction:(BOOL)updateCardSyncAction
{
    BOOL result = [cardDBController_ setCardGroupsWithCardID:cardID groupIDArray:groupIDArray isInTransaction:NO updateModifiedTIme:updateModifiedTime];
    
    if(result == YES)
    {
        // 調整syncAction紀錄
        if(updateCardSyncAction==YES)
        {
            [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardID modifiedTime:nil];
        }
    }
    
    return result;
}

//===============================================================================
// 設定卡片群組
// groupIDArray : groupID "String" array
//===============================================================================
- (BOOL)setCardGroupsWithCardID:(NSString *)cardID groupIDArray:(NSArray *)groupIDArray updateModifiedTime:(BOOL)updateModifiedTime
{
    return [self setCardGroupsWithCardID:cardID groupIDArray:groupIDArray updateModifiedTime:updateModifiedTime updateCardSyncAction:YES];
}


#pragma mark - WCDataControllerProtocol_CardGroup

//===============================================================================
// 設定卡片群組，不改變Contact時間
// groupIDArray : groupID "String" array
//===============================================================================
- (BOOL)setCardGroupsWithCardID:(NSString *)cardID groupIDArray:(NSArray *)groupIDArray
{
    return [self setCardGroupsWithCardID:cardID groupIDArray:groupIDArray updateModifiedTime:NO];
}


//===============================================================================
// 取得名片所屬的所有群組ID
// return : groupID "String" array (release after using)
//===============================================================================
- (NSMutableArray *)copyGroupIDArrayWithCardID:(NSString *)cardID
{
    return [cardDBController_ copyGroupIDArrayWithCardID:cardID];
}


//===============================================================================
// 取得群組中所有名卡ID
// return : cardID "String" array (release after using)
//===============================================================================
- (NSMutableArray *)copyCardIDArrayWithGroupID:(NSString *)groupID
{
    if([groupID length]==0)
        return nil;
    
    WC_GroupID realGroupID = [groupID integerValue];

    self.lastError = nil;

    NSMutableArray *groupIDArray = [cardDBController_ copyCardIDArrayWithGroupID:realGroupID];

    self.lastError = cardDBController_.lastError;

    return groupIDArray;
}


//==============================================================================
//
//==============================================================================
- (NSInteger)cardCountForGoogleGroup
{
    return [cardDBController_ cardCountForGoogleGroup];
}


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card methods
#pragma mark - Private

#pragma mark - WCDataControllerProtocol_Card


//===============================================================================
// 取得名片總數
//===============================================================================
- (NSInteger)totalCardCount
{
    return [cardDBController_ totalCardCount];
}


//===============================================================================
//
//===============================================================================
- (BOOL)isCardIDExist:(NSString *)cardID
{
    return [cardDBController_ isCardIDExist:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)isEmptyCardWithCardID:(NSString *)cardID frontSideImage:(CPImage *)frontSideImage backSideImage:(CPImage *)backSideImage
{
    BOOL isEmpty = NO;
    NSUInteger fieldCount = [cardDBController_ fieldCountWithCardID:cardID];
    BOOL hasIDSidePhoto=[self hasCardImageWithCardID:cardID type:WC_IT_IDPhoto];
    BOOL hasBackSidePhoto = (backSideImage!=nil)?YES:NO;
    BOOL hasFrontSidePhoto = (frontSideImage!=nil)?YES:NO;
    
    BOOL isCardHasImage=hasBackSidePhoto|hasFrontSidePhoto|hasIDSidePhoto;
    
    if (fieldCount==0 && !isCardHasImage)
    {
        isEmpty = YES;
    }
    return isEmpty;
}


//===============================================================================
// 取得lite用來匯出的名片所有資料 (姓名，公司，電話, 各一筆)
// return : WCCardModel (release after using)
//===============================================================================
- (WCCardModel *)copyLiteExportCardDataWithCardID:(NSString *)cardID
{
    self.lastError = nil;
    WCCardModel *cardModel = [cardDBController_ copyCardWithCardID:cardID getAllField:YES];
    
    // 移除多餘的資料
    for (WC_FieldType fieldType=WC_FT_None; fieldType <=WC_FT_UnifiedBusinessNo; fieldType++)
    {
        // 只保留姓名，公司，電話, 各一筆
        if (fieldType==WC_FT_Name ||
            fieldType==WC_FT_Company ||
            fieldType==WC_FT_Phone)
        {
            NSArray *fieldModelArray=[cardModel fieldArrayWithType:fieldType];
            NSInteger fieldArrayCount=[fieldModelArray count];
            
            // 刪除資料欄位
            for(NSInteger i=fieldArrayCount-1;i>0;i--)
            {
                [cardModel removeFieldWithType:fieldType index:i];
            }
        }
        else
        {
            NSArray *fieldModelArray=[cardModel fieldArrayWithType:fieldType];
            NSInteger fieldArrayCount=[fieldModelArray count];

            // 刪除資料欄位
            for(NSInteger i=fieldArrayCount-1;i>=0;i--)
            {
                [cardModel removeFieldWithType:fieldType index:i];
            }
        }
    }
    
    self.lastError = cardDBController_.lastError;
    return cardModel;
}


//===============================================================================
// 取得名片所有資料 (display + field data)
// return : WCCardModel (release after using)
//===============================================================================
- (WCCardModel *)copyCardFullDataWithCardID:(NSString *)cardID
{
    self.lastError = nil;
    
    WCCardModel *cardModel = [cardDBController_ copyCardWithCardID:cardID getAllField:YES];
    
    if(cardModel == nil)
    {
        if([[cardDBController_.lastError description] containsString:@"card not exist"] == YES)
        {
            [self setLastErrorWithCode:WCDC_ErrorCode_GetCardNotExist description:@"card not exist"];
        }
        else
        {
            self.lastError = cardDBController_.lastError;
        }
    }
    
    return cardModel;
}


//===============================================================================
// 取得名片顯示資料
// return : WCCardModel (release after using)
//===============================================================================
- (WCCardModel *)copyCardDisplayDataWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel = nil;

    if([cardID length])
    {
        // check if exist in cache
        WCCacheModel *cacheModel = [self cacheOfCurrentSource];
        cardModel = [cacheModel cardFromCacheWithCardID:cardID];

        // use cache data if exist
        if(cardModel)
        {
            [cardModel retain];
        }
        else // not in cache, load from database.
        {
            cardModel = [cardDBController_ copyCardWithCardID:cardID getAllField:NO];
        }
    }

    return cardModel;
}


//===============================================================================
// 取得指定欄位資料
//===============================================================================
- (NSMutableArray *)copyFieldValueArrayWithCardIDArray:(NSArray *)cardIDArray fieldType:(WC_FieldType)fieldType
{
    return [cardDBController_ copyFieldValueArrayWithCardIDArray:cardIDArray fieldType:fieldType];
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyPhoneArrayWithCardIDArray:(NSArray *)cardIDArray
{
    return [cardDBController_ copyPhoneArrayWithCardIDArray:cardIDArray];
}


//===============================================================================
//
//===============================================================================
- (NSMutableArray *)copyPhoneArrayForSendMessageWithCardIDArray:(NSArray *)cardIDArray
{
    return [cardDBController_ copyPhoneArrayForSendMessageWithCardIDArray:cardIDArray];
}


//================================================================================
//
//================================================================================
- (BOOL)saveCard:(WCCardModel *)cardModel withFrontSideImage:(CPImage *)frontSideImage backSideImage:(CPImage *)backSideImage idPhotoImage:(CPImage *)idPhotoImage isImportMode:(BOOL)isImportMode sendNotification:(BOOL)sendNotification
{
    //////////////////////////////////////////////////
    // !! account相關的在外部自行處理
    
    
    //////////////////////////////////////////////////
    // Image相關處理 (因顯示需求須先記錄狀態，實際image寫入卻需要card處理成功後才進行)
    
    cardModel.containImageType = WC_IT_None;
    
    if(frontSideImage != nil)
    {
        cardModel.containImageType |= WC_IT_FrontSide;
    }
    
    if(backSideImage != nil)
    {
        cardModel.containImageType |= WC_IT_BackSide;
    }
    
    if(idPhotoImage != nil)
    {
        cardModel.containImageType |= WC_IT_IDPhoto;
    }
    
    
    //////////////////////////////////////////////////
    // 時間相關處理 (不允許建立未來時間)
    
    NSDate *now = [NSDate date];
    
    if([cardModel.createdTime compare:now] == NSOrderedDescending)
    {
        cardModel.createdTime = now;
    }
    
    if([cardModel.modifiedTime compare:now] == NSOrderedDescending)
    {
        cardModel.modifiedTime = now;
    }
    
    
    //////////////////////////////////////////////////
    // Add or import
    
    if(isImportMode == YES)
    {
        // 匯入時規則:
        // 1. 看cardID是不是已存在
        // 2. 如果不存在，直接新增
        // 3. 如果已存在，比對modifiedTime, 以較新的資料更新
        // 4. group的資料要以聯集的方式處理
        
        NSString *importedCardID = cardModel.ID;
        NSString *cardID = [self cardIDWithImportedCardID:importedCardID];
        
        if ([cardID length] > 0)
        {
            // 如果已存在，用資料庫中的ID來處理
            cardModel.ID = cardID;
        }
        else
        {
            // 不存在的話，建立新的
            cardModel.ID =  [NSString GUID];
            
            // 紀錄匯入的ID
            [self setImporedCardID:importedCardID withCardID:cardModel.ID];
        }
        
        // 匯入要把電話排序
        [cardModel sortPhoneFields];
        
        if ([cardModel.ID length]>0 && [self isCardIDExist:cardModel.ID]==YES)
        {
            //////////////////////////////////////////////////
            // 取group聯集
            
            WCCardModel *currentModel = [[self copyCardFullDataWithCardID:cardModel.ID] autorelease];
            NSMutableSet *importGroupIDStringSet = [NSMutableSet setWithArray:cardModel.groupIDArray];
            NSSet *existGroupIDStringSet = [NSSet setWithArray:currentModel.groupIDArray];
            
            [importGroupIDStringSet unionSet:existGroupIDStringSet];
            
            // 處理unfiled group
            if([importGroupIDStringSet count] > 1)
            {
                [importGroupIDStringSet removeObject:[NSString stringWithInteger:WC_GID_Unfiled]];
            }
            
            
            //////////////////////////////////////////////////
            //比對修改時間
            
            WCCardModel *cardModelForEdit = cardModel;
            
            if ([currentModel.modifiedTime timeIntervalSinceReferenceDate] >= [cardModel.modifiedTime timeIntervalSinceReferenceDate])
            {
                //  資料庫較新, 以資料庫的名片為主
                cardModelForEdit = currentModel;
            }
            
            
            //////////////////////////////////////////////////
            // 更新資料
            
            if (cardModelForEdit != nil)
            {
                // 設定新的group
                [cardModelForEdit setGroupIDArray:[importGroupIDStringSet allObjects] isInitCard:NO];
                
                // 更新名片資料
                if([self updateCard:cardModelForEdit withSendNotification:sendNotification] == NO)
                {
                    return NO;
                }
            }
        }
        else
        {
            // import add
            
            if ([cardModel.ID length]==0)
            {
                cardModel.ID =  [NSString GUID];
                
//#if defined (PRODUCTLINE_WCMAC)
//                cardModel.ID = [WCToolController generateCardID];
//#elif defined (PRODUCTLINE_WCTMAC)
//                cardModel.ID =  [NSString GUID];
//#endif
            }
            
            
            // 直接新增
            if([self addCard:cardModel withSendNotification:sendNotification] == NO)
            {
                return NO;
            }
        }
        
    }  // end of (isImportMode == YES)
    else
    {
        //////////////////////////////////////////////////
        // add
        
        if ([cardModel.ID length] == 0)
        {
            cardModel.ID =  [NSString GUID];
            
//#if defined (PRODUCTLINE_WCMAC)
//            cardModel.ID = [WCToolController generateCardID];
//#elif defined (PRODUCTLINE_WCTMAC)
//            cardModel.ID =  [NSString GUID];
//#endif
        }
        
        if([self addCard:cardModel withSendNotification:sendNotification] == NO)
        {
            return NO;
        }
        
    } // end of (isImport == NO)
    
    
    //////////////////////////////////////////////////
    
    if(frontSideImage != nil)
    {
        [self setCardImage:frontSideImage withCardID:cardModel.ID type:WC_IT_FrontSide];
    }
    else
    {
        [self removeCardImageWithCardID:cardModel.ID type:WC_IT_FrontSide];
    }
    
    if(backSideImage != nil)
    {
        [self setCardImage:backSideImage withCardID:cardModel.ID type:WC_IT_BackSide];
    }
    else
    {
        [self removeCardImageWithCardID:cardModel.ID type:WC_IT_BackSide];
    }
    
    if(idPhotoImage != nil)
    {
        [self setCardImage:idPhotoImage withCardID:cardModel.ID type:WC_IT_IDPhoto];
    }
    else
    {
        [self removeCardImageWithCardID:cardModel.ID type:WC_IT_IDPhoto];
    }

    //////////////////////////////////////////////////

    return YES;
}



//==============================================================================
// 儲存來自系統聯絡人的名片（包含資料檢查）
//==============================================================================
- (BOOL)saveWCABCard:(WCABCardModel *)abCardModel sendNotification:(BOOL)sendNotification
{
    NSString *cardID = [self cardIDWithABPersonID:abCardModel.abPersonID];
    
    if (abCardModel.abPersonID != nil &&
        [cardID length]>0)
    {
        //////////////////////////////////////////////////
        // 合併，更新
        abCardModel.ID = cardID;
        
        WCCardModel *existedCardModel = [[self copyCardFullDataWithCardID:abCardModel.ID] autorelease];
        NSMutableSet *importGroupIDStringSet = [NSMutableSet setWithArray:abCardModel.groupIDArray];
        
        // 特殊處理group, 要取聯集
        NSSet *existGroupIDStringSet = [NSSet setWithArray:existedCardModel.groupIDArray];
        
        [importGroupIDStringSet unionSet:existGroupIDStringSet];
        
        // 處理unfiled group
        if([importGroupIDStringSet count] > 1)
        {
            [importGroupIDStringSet removeObject:[NSString stringWithInteger:WC_GID_Unfiled]];
        }
        
        //////////////////////////////////////////////////
        // 寫一個新的merge function for new rule
        [existedCardModel combineCard:abCardModel];
        
        // 設定為已校正
        // !!fix mantis #0017016: [iOS 9.0.2_iPhone 4S] 新增一筆聯絡人PPP，此聯絡人【未校正】 ->匯出到系統聯絡人->在系統聯絡人中變更這個聯絡人資料->再匯入PPP聯絡人，這筆聯絡人不應該還顯示【未校正】的狀態
        existedCardModel.tagMask &= ~WC_TagMask_Unverified;
        
        //////////////////////////////////////////////////
        // 設定新的group
        [existedCardModel setGroupIDArray:[importGroupIDStringSet allObjects] isInitCard:NO];
        
        
        //////////////////////////////////////////////////
        // 檢查名片是否還在
        if ([self isCardIDExist:cardID]==NO)
        {
            return NO;
            
        }
        
        //////////////////////////////////////////////////
        // 更新名片
        if([self updateCard:existedCardModel withSendNotification:sendNotification]==NO)
        {
            return NO;
        }
        
        //////////////////////////////////////////////////
        // 更新address book data
        // 更新address book data
        if ([self updateAddressBookDataWithCardModel:abCardModel]==NO)
        {
            return NO;
        }
        
    }
    else
    {
        // 新增
        if ([abCardModel.ID length] == 0)
        {
            abCardModel.ID =  [NSString GUID];
        }
        
        if([self addCard:abCardModel withSendNotification:sendNotification]== NO)
        {
            return NO;
        }
        
        //////////////////////////////////////////////////
        // 更新address book data
        if ([self updateAddressBookDataWithCardModel:abCardModel]==NO)
        {
            return NO;
        }
    }
    
    
    //////////////////////////////////////////////////
    // 儲存或刪除大頭貼
    if (abCardModel.abPhotoImage)
    {
        // !! 因為從系統回來的大頭貼方向可能是不對的，所以要轉正後再存
        CPImage *adjustedImage = [abCardModel.abPhotoImage imageByAdjustOrientationWithMaxLength:WC_IMS_Original];
        if (adjustedImage==nil)
        {
            return NO;
            
        }
        
        if([self setCardImage:adjustedImage withCardID:abCardModel.ID type:WC_IT_IDPhoto]==NO)
        {
            return NO;
        }
    }
    else
    {
        [self removeCardImageWithCardID:abCardModel.ID type:WC_IT_IDPhoto];
    }
    
    return YES;
}


//===============================================================================
// 加入名片
//===============================================================================
- (BOOL)addCard:(WCCardModel *)cardModel withSendNotification:(BOOL)sendNotification
{
    WCTCardSyncActionModel *addCardSyncActionModel = nil;
    if (cardModel!=nil)
    {
        addCardSyncActionModel = [[[WCTCardSyncActionModel alloc] init] autorelease];
        addCardSyncActionModel.dataGuid = cardModel.ID;
        addCardSyncActionModel.actionType = WCTSyncActionType_Add;
        addCardSyncActionModel.modifiedTime = [NSDate date];
    }

    return [self addCard:cardModel syncActionModel:addCardSyncActionModel withSendNotification:sendNotification];
}


//===============================================================================
// 更新名片內容
//===============================================================================
- (BOOL)updateCard:(WCCardModel *)cardModel withSendNotification:(BOOL)sendNotification
{
    if([self updateCard:cardModel withSendNotification:sendNotification syncActionModel:nil mustExist:NO])
    {
        [cardDBController_ updateCardSyncActionForGroupChangeWithCardID:cardModel.ID modifiedTime:nil];
        [cardDBController_ updateCardSyncActionForContentChangeWithCardID:cardModel.ID modifiedTime:nil];
        
        return YES;
    }
    return NO;
}


//===============================================================================
// 刪除名片
//===============================================================================
- (BOOL)removeCards:(NSArray *)cardIDs
{
#ifdef _DUMP_LOG_
    [DumpLog logWithMemSize:YES startTime:0 format:@"!!! removeCards begin"];
#endif

    NSMutableArray  *favoriteCardIDArray = [[NSMutableArray alloc] init];

    //---------------------------------
    // delete cards in DB
    //---------------------------------
    
    // delete card data and image
    for (NSString *cardID in cardIDs)
    {
        @autoreleasepool {
            
            WCTCardSyncActionModel *syncActionModel = nil;
            WCTCardSyncActionModel *oldSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];
            
            // 如果已同步過，或不是新增，才要加入刪除的syncActionModel
            if (oldSyncActionModel.actionType!=WCTSyncActionType_Add ||
                oldSyncActionModel.syncState>WCTSyncState_UnSync)
            {
                syncActionModel = [[WCTCardSyncActionModel alloc] init];
                syncActionModel.dataGuid = cardID;
                syncActionModel.syncState = oldSyncActionModel.syncState;
                syncActionModel.actionType = WCTSyncActionType_Delete;
                syncActionModel.modifiedTime = [NSDate date];
            }
            else
            {
                // 如果是新增還沒同步，直接把syncAction刪掉
                [self removeCardSyncActionWithCardID:cardID];
            }
            [oldSyncActionModel release];

            //////////////////////////////////////////////////
            // delete card data
            if ([cardDBController_ removeCardWithCardID:cardID syncActionModel:syncActionModel])
            {                
                // delete all image of current card
                [self removeAllCardImagesWithCardID:cardID];
                
 
                //////////////////////////////////////////////////
                WCCacheModel *cacheModel = [self cacheOfCurrentSource];

                // 後面的removeUnverifiedWithCardID也會更新時間，所以這邊就不更新了
                if([cardDBController_ removeFavoriteWithCardID:cardID isInTransaction:NO])
                {
                    [favoriteCardIDArray addObject:cardID];
                    [cacheModel removeFavoriteCardFromCacheWithCardID:cardID];
                }

                if([cardDBController_ removeUnverifiedWithCardID:cardID])
                {
                    [cacheModel removeCardFromCacheWithCardID:cardID];
                }
                
                [cardDBController_ updateCardSyncActionForContentChangeWithCardID:cardID modifiedTime:syncActionModel.modifiedTime];
                
            }
            [syncActionModel release];
        } // end of autoreleasepool
    }

    //---------------------------------
    // !! notify UI update
    //---------------------------------
    [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                    infoKey:WCDC_NOTIFY_UserInfo_kRemoveCard
                  infoValue:cardIDs];

    if ([favoriteCardIDArray count]) {
        [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kRemoveFavorite
                      infoValue:favoriteCardIDArray];
    }
    
    [favoriteCardIDArray release];


#ifdef _DUMP_LOG_
    //    [DumpLog logWithMemSize:YES startTime:0 format:@"!!! removeCards end"];
#endif

    return YES;
}


//==============================================================================
//
//==============================================================================
- (void)updateWithCardID:(NSString *)cardID
          fromFieldSource:(WC_FieldSource)fromFieldSource
            toFieldSource:(WC_FieldSource)toFieldSource
          removeRecogRect:(BOOL)removeRecogRect;

{
    // 如果名片不存在，就不處理
    if ([self isCardIDExist:cardID]==NO)
    {
        return ;
    }
    
    // 取得要處理的名片
    WCCardModel *cardModel = [self copyCardFullDataWithCardID:cardID];
    if (cardModel==nil)
    {
        return ;
    }
    
    //////////////////////////////////////////////////
    // 如果是反面變正面，還要處理辨識語系
    if (fromFieldSource==WC_FS_BackRecog &&
        toFieldSource==WC_FS_FrontRecog)
    {
        cardModel.frontRecogLang = cardModel.backRecogLang;
        cardModel.backRecogLang = -1;
    }
    
    //////////////////////////////////////////////////
    //  處理欄位來源，與辨識rect
    NSArray *allFieldArray = [cardModel.fieldArrayDict allValues];
    
    for(NSArray *fieldArray in allFieldArray)
    {
        for(WCFieldModel *fieldModel in fieldArray)
        {
            if(fieldModel.source==fromFieldSource)
            {
                // 變更欄位來源
                fieldModel.source = toFieldSource;
                
                //////////////////////////////////////////////////
                // 清除辨識rect
                if (removeRecogRect)
                {
                    if(!CGRectIsEmpty(fieldModel.recogRect))
                    {
                        fieldModel.recogRect = CGRectZero;
                    }
                    
                    if([fieldModel.value isKindOfClass:[NSDictionary class]])
                    {
                        NSArray *subFieldArray = [(NSDictionary *)fieldModel.value allValues];
                        
                        for(WCFieldModel *subFieldModel in subFieldArray)
                        {
                            if(!CGRectIsEmpty(subFieldModel.recogRect))
                            {
                                subFieldModel.recogRect = CGRectZero;
                            }
                        }
                    }
                }
            }
        }
    }
    
    //////////////////////////////////////////////////
    [self updateCard:cardModel withSendNotification:YES];
    [cardModel release];
}


//==============================================================================
//
//==============================================================================
- (void)updateOrderToFirstWithIndex:(NSInteger)index withCardID:(NSString *)cardID fieldType:(NSInteger)fieldType
{
    [cardDBController_ updateOrderToFirstWithIndex:index withCardID:cardID fieldType:fieldType];
}


//==============================================================================
//
//==============================================================================
- (NSString *)noteWithCardID:(NSString *)cardID
{
    return [cardDBController_ noteWithCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateNote:(NSString *)note withCardID:(NSString *)cardID
{
    BOOL result = [cardDBController_ updateNote:note withCardID:cardID];
    if (result)
    {
        [cardDBController_ updateModifiedTimeWithCardID:cardID];
        [cardDBController_ updateCardSyncActionForContentChangeWithCardID:cardID modifiedTime:nil];

        // !! 這邊是從info call進來的，所以不用變更為已校正
//        // 變更未校正狀態也要更新cache
//        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
//        WCCardModel *cachedCardModel = [cacheModel cardFromCacheWithCardID:cardID];
//        if (cachedCardModel!=nil)
//        {
//            cachedCardModel.tagMask &= ~WC_TagMask_Unverified;
//            [cacheModel updateCardInCache:cachedCardModel];
//        }
//        
//        // notify UI update
//        [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
//                        infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
//                      infoValue:[NSArray arrayWithObject:cardID]];
//        
//        
//        //////////////////////////////////////////////////
//        // 我的最愛那邊的名片也要更新cache
//        WCCardModel *cachedFavoritesCardModel = [cacheModel favoriteCardFromCacheWithCardID:cardID];
//        if (cachedFavoritesCardModel!=nil)
//        {
//            cachedFavoritesCardModel.tagMask &= ~WC_TagMask_Unverified;
//            [cacheModel updateFavoriteCardInCache:cachedFavoritesCardModel];
//        }
//        
//        // notify UI update
//        [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
//                        infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
//                      infoValue:[NSArray arrayWithObject:cardID]];
        
        return YES;
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardModifiedTime:(NSDate *)modifiedTime withCardID:(NSString *)cardID
{
    return [cardDBController_ updateCardModifiedTime:modifiedTime withCardID:cardID];
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCABCardModel methods


//==============================================================================
//
//==============================================================================
- (NSString *)cardIDWithABPersonID:(NSString *)abPersonID
{
    return [cardDBController_ cardIDWithABPersonID:abPersonID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)isCardExistWithABPersonID:(NSString *)abPersonID
{
    NSString *cardID = [cardDBController_ cardIDWithABPersonID:abPersonID];
    return ([cardID length]>0)?YES:NO;
}



//==============================================================================
//
//==============================================================================
- (BOOL)updateAddressBookDataWithCardModel:(WCABCardModel *)abCardModel
{
    return [cardDBController_ updateAddressBookDataWithCardModel:abCardModel];
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeAddressBookDataWithCardID:(NSString *)cardID
{
    return [cardDBController_ removeAddressBookDataWithCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)getAddressBookDataWithCardModel:(WCABCardModel *)abCardModel
{
    return [cardDBController_ getAddressBookDataWithCardModel:abCardModel];
}


//==============================================================================
//
//==============================================================================
- (BOOL)clearAddressBookData
{
    return [cardDBController_ clearAddressBookData];
}


//==============================================================================
//
//==============================================================================
- (NSArray *)allAddressBookData
{
    return [cardDBController_ allAddressBookData];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - AddressBook convert to Contacts framework methods

//==============================================================================
//
//==============================================================================
- (NSArray <NSString *>*)allCardIDFromAddressBookMapping
{
    return [cardDBController_ allCardIDFromAddressBookMapping];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - recent contact methods


//==============================================================================
//
//==============================================================================
- (BOOL)insertRecentAction:(NSInteger)action withContent:(NSString *)content cardID:(NSString *)cardID
{
    return [cardDBController_ insertRecentAction:action withContent:content cardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeRecentActionWithCardID:(NSString *)cardID
{
    return [cardDBController_ removeRecentActionWithCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyAllRecentCardSection
{
    NSMutableDictionary *sectionDict = [NSMutableDictionary dictionary];
    NSMutableArray *cardSections = nil;
    NSMutableArray *recentCardModels = [cardDBController_ copyAllRecentContacts];
    
    for (WCRecentCardModel *recentCardModel in recentCardModels)
    {
        WCCardModel *cardModel = [self copyCardFullDataWithCardID:recentCardModel.ID];
        WCRecentCardModel *tempRecentCard = [WCRecentCardModel recentCardModelFromCardModel:cardModel];
        [cardModel release];
        
        tempRecentCard.actionType = recentCardModel.actionType;
        tempRecentCard.content = recentCardModel.content;
        tempRecentCard.actionTime = recentCardModel.actionTime;
        
        NSString *recentTime = [tempRecentCard.actionTime stringWithFormat:NSDateFormat_Day];
        NSMutableArray *currentSection = [sectionDict objectForKey:recentTime];
        if (currentSection==nil)
        {
            currentSection = [NSMutableArray array];
            [sectionDict setObject:currentSection forKey:recentTime];
        }

        [currentSection addObject:tempRecentCard];
    }
    [recentCardModels release];
    //////////////////////////////////////////////////
    // 轉換為array
    for (NSString *key in [sectionDict allKeys])
    {
        WCCardSectionModel *sectionModel = [[WCCardSectionModel alloc] init];
        sectionModel.title = key;
        sectionModel.cardArray = [sectionDict objectForKey:key];
        
        if (cardSections==nil)
        {
            cardSections = [[NSMutableArray alloc] init];
        }
        
        [cardSections addObject:sectionModel];
        [sectionModel release];
    }
    
    return cardSections;
}


//==============================================================================
//
//==============================================================================
- (BOOL)clearRecentContactData
{
    return [cardDBController_ clearRecentContactData];
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasRecentContacts
{
    return [cardDBController_ hasRecentContacts];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Section methods

#pragma mark - Public

//================================================================================
//
//================================================================================
- (void)cancelLoadCardSections
{
    @synchronized(operationQueue_)
    {
        if([operationQueue_.operations count] > 0)
        {
            // 注意因為外部需求是所有operation都要回傳結果
            // 用operationQueue cancelAllOperations的話未執行的operation是直接被移除無法回傳
            // 所以改用自訂的internalCancel控制
//            [operationQueue_ cancelAllOperations];
            
            for(NSOperation *operation in operationQueue_.operations)
            {
                operation.internalCancel = YES;
            }

            
            WCTCardDBController *cardDBController;
            
#if TARGET_OS_IPHONE
            
            cardDBController = [WCTCardDBController sharedWCTCardDBControllerWithDBPath:[baseDirPath_ stringByAppendingFormat:@"/%@",WCDC_DBName]];
            
#elif TARGET_OS_MAC
            

            cardDBController = cardDBController_;
#endif
            
            // !! 一定要取消不然會等很久
            [cardDBController cancel];
        }
    }
}





#pragma mark - Private

//===============================================================================
// thread load card sections (private)
// !! 因為是在operation thread中處理，不可使用物件中的cardDBController_，要使用自行建立的cardDBController。
// !! 資料先送到cache中儲存，上層收到notification後再去cache取資料
// 資料格式:
// @{
//    @(WC_SBF_Name):
//        @{
//            @(PPSIC_M_English):allCardSectionArray,
//            @(PPSIC_M_Stroke):allCardSectionArray,
//            .
//            .
//            .
//            @(PPSIC_M_Hangul):allCardSectionArray
//        },
//    @(WC_SBF_Company):
//        @{
//            @(PPSIC_M_English):allCardSectionArray,
//            @(PPSIC_M_Stroke):allCardSectionArray,
//            .
//            .
//            .
//            @(PPSIC_M_Hangul):allCardSectionArray
//        },
//    @(WC_SBF_CreateTime):
//        @{
//            @(PPSIC_M_None):allCardSectionArray
//        },
//    @(WC_SBF_ModifiedTime):
//        @{
//            @(PPSIC_M_None):allCardSectionArray
//        }
// }

//===============================================================================
- (NSError *)bgThreadLoadCardSections:(NSArray *)paramArray
{
    NSError *error = nil;

    @autoreleasepool
    {
        WC_GroupID			groupID = [[paramArray objectAtIndex:0] intValue];
        NSString            *searchText = [paramArray count] > 1 ? [paramArray objectAtIndex:1] : nil;
        
        NSMutableDictionary *allCardSectionDict = [NSMutableDictionary dictionary];
        NSMutableArray		*allCardSectionArray = nil;
        WCTCardDBController *cardDBController = nil;
        BOOL				result = NO;
        NSArray             *allCardArray = nil;
        
        
        cardDBController = [WCTCardDBController sharedWCTCardDBControllerWithDBPath:[baseDirPath_ stringByAppendingFormat:@"/%@",WCDC_DBName]];
        
        
        if([cardDBController connectDB])
        {
            //////////////////////////////////////////////////
            // 取得所有名片，改為在dataController處理card section
            
            allCardArray = [cardDBController copyAllCardsWithGroupID:groupID searchText:searchText];
            
            if(allCardArray == nil)
            {
                // 用copy避免cardDBController.lastError被後續動作覆寫
                error = [cardDBController.lastError copy];
                goto _EXIT;
            }
            
            
            //////////////////////////////////////////////////
#ifdef DEBUG
            PPLog_StartLogCostTime(@"bgThreadLoadCardSections");
#endif
            WCSC_DisplyRule displayRule = [PPSettingsController displayRule];
            
            WC_SortedByField currentSortingFields = displayRule.sortingByField;
            PPSIC_Mode currentSectionModes = displayRule.sectionIndexMode;
            
#ifdef DEBUG
            PPLog_StartLogCostTime(@"loadSection");
#endif
            NSMutableDictionary *cardSectionDict = [NSMutableDictionary dictionary];;
            
            for (WCCardModel *cardModel in allCardArray)
            {
                //////////////////////////////////////////////////
                // check if operation is cancelled
                
                @synchronized(operationQueue_)
                {
                    if([operationQueue_.operations count])
                    {
                        NSOperation *operation = [operationQueue_.operations firstObject];
                        
                        if([operation isCancelled])
                        {
                            //                            NSLog(@"%s %@ cancel", __func__, searchText);
                            goto _EXIT;
                        }
                    }
                }
                
                //////////////////////////////////////////////////
                // 依sectionTitle產生cardSectionModel
                NSString *sectionTitle = [cardModel displaySectionTitleWithSortByField:currentSortingFields indexingMode:currentSectionModes];
                
                // !! ios要加到cache的sectionTitle要轉為目前使用的值
                cardModel.sectionTitle = sectionTitle;
                
                // !! 先看有沒有舊的 card seciton
                WCCardSectionModel *cardSectionModel = [cardSectionDict objectForKey:sectionTitle];
                
                // 沒有舊的，產生一個新的 card seciton
                if(cardSectionModel==nil)
                {
                    cardSectionModel = [[[WCCardSectionModel alloc] init] autorelease];
                    cardSectionModel.title = sectionTitle;
                    cardSectionModel.cardArray = [NSMutableArray array];
                    
                    [cardSectionDict setObject:cardSectionModel forKey:sectionTitle];
                }
                
                cardSectionModel.needResortCard = YES;
                
                [cardSectionModel.cardArray addObject:cardModel];
            }
            
#ifdef DEBUG
            NSString *msg = [NSString stringWithFormat:@"currentSortingFields:%ld, currentSectionModes:%ld, handle card section", (long)currentSortingFields, (long)currentSectionModes];
            PPLog_LogCostTime(@"loadSection", PPLogControllerMask_Normal, msg);
#endif
            
            //////////////////////////////////////////////////
            // 轉換為card section array
            allCardSectionArray = [[NSMutableArray alloc] init];
            [allCardSectionArray addObjectsFromArray:[cardSectionDict allValues]];
            
            // sort sections, 不是cacheModel可以直接排
            [WCCompareTool cardSectionArray:allCardSectionArray sortByField:currentSortingFields];
            
            [allCardSectionDict setObject:allCardSectionArray forKey:@(currentSortingFields)];
            
#ifdef DEBUG
            PPLog_StopLogCostTime(@"loadSection");
#endif
            
        }
        else
        {
            // 用copy避免cardDBController.lastError被後續動作覆寫
            error = [cardDBController.lastError copy];
        }
        
        
        if([[NSThread currentThread] isCancelled])
        {
            goto _EXIT;
        }
        
        result = YES;
        
        //////////////////////////////////////////////////
        
_EXIT:
        
        [allCardArray release];
        
        // save result
        if(result)
        {
            WCCacheModel *cacheModel = [WCCacheModel defaultCache];
            
            [cacheModel updateFromAllCardSections:allCardSectionArray];
            
            cacheModel.groupID = groupID;
        }
        
        [allCardSectionArray release];
        
        //test : long load process
        //[NSThread sleepForTimeInterval:2];
    }
    
    return [error autorelease];
}


//===============================================================================
// thread load card display data dict (private) (WCTMAC)
//===============================================================================
- (NSError *)bgThreadLoadCardSectionDict:(NSArray *)paramArray
{
    NSError *error = nil;

    @autoreleasepool
    {
        WC_GroupID			groupID = [[paramArray objectAtIndex:0] intValue];
        NSString            *searchText = [paramArray count] > 1 ? [paramArray objectAtIndex:1] : nil;
        NSMutableDictionary *cardSectionDict = nil;
        BOOL				result = NO;
        NSArray             *allCardArray = nil;
        
        // MARK: 130後，一般搜尋都是找全部資料，不用分群組。
        if([searchText length] > 0)
        {
            groupID = WC_GID_All;
        }

        if([cardDBController_ connectDB])
        {
#ifdef DEBUG
            PPLog_StartLogCostTime(@"bgThreadLoadCardSectionDict");
#endif

            //////////////////////////////////////////////////
            // parse card array to allCardSectionDict (WCTMAC data structure)
            // WCTMAC儲存的sectionTitle結構是 ”姓名的所有模式的索引值;公司名的所有模式的索引值"
            // 要轉換成allCardSectionDict儲存，結構是@{SectionTitle:@[CardArray]}
            
            switch (groupID)
            {
                case WC_GID_Favorite:
                {
                    allCardArray = [[cardDBController_ copyFavoritesWithSearchText:searchText] autorelease];
                    break;
                }

                case WC_GID_Unverified:
                {
                    allCardArray = [[cardDBController_ copyUnverifiedWithSearchText:searchText] autorelease];
                    break;
                }
                    
                default:
                {
                    allCardArray = [[cardDBController_ copyAllCardsWithGroupID:groupID searchText:searchText] autorelease];
                    break;
                }
            }
            
#ifdef DEBUG
            PPLog_LogCostTime(@"bgThreadLoadCardSectionDict", PPLogControllerMask_Normal, @"load DB");
#endif

            if(allCardArray == nil)
            {
                // 用copy避免cardDBController.lastError被後續動作覆寫
                error = [cardDBController_.lastError copy];
                goto _EXIT;
            }
            
            // MARK: 130後，一般搜尋結果不用分index
            if([searchText length] > 0)
            {
                cardSectionDict = [NSMutableDictionary dictionary];
            }
            else
            {
                cardSectionDict = [self cardSectionDictWithCards:allCardArray error:&error];
            }
            
            if(error != nil)
            {
                [error retain];
                goto _EXIT;
            }
            
            // !! special case : mac版有"All"的section
            [cardSectionDict setObject:allCardArray forKey:PPSIC_Title_All];
            
#ifdef DEBUG
            PPLog_LogCostTime(@"bgThreadLoadCardSectionDict", PPLogControllerMask_Normal, @"make section");
            PPLog_StopLogCostTime(@"bgThreadLoadCardSectionDict");
#endif
        }
        else
        {
            // 用copy避免cardDBController.lastError被後續動作覆寫
            error = [cardDBController_.lastError copy];
        }
        
        if([[NSThread currentThread] isCancelled])
        {
            error = [PPErrorOperationCancel(nil) retain];
            goto _EXIT;
        }
        
        result = YES;
        
        //////////////////////////////////////////////////
        
_EXIT:
        
        // save result
        if(result == YES)
        {
            WCCacheModel *cacheModel = [WCCacheModel defaultCache];
            [cacheModel updateFromAllCardSectionDict:cardSectionDict];
            cacheModel.sourceID = self.sourceID;
            cacheModel.groupID = groupID;
        }
    }
    
    return [error autorelease];
}


//================================================================================
// return dictionary @{sectionTitle:@[cardArray]}
//================================================================================
- (NSMutableDictionary *)cardSectionDictWithCards:(NSArray *)cards error:(NSError **)error
{
    NSError *returnError = nil;
    NSMutableDictionary *cardSectionDict = [NSMutableDictionary dictionary];
    
    WCSC_DisplyRule displayRule = [PPSettingsController displayRule];
    WC_SortedByField sortField = displayRule.sortingByField;
    PPSIC_Mode indexMode = displayRule.sectionIndexMode;
    
    for(WCCardModel *cardModel in cards)
    {
        //////////////////////////////////////////////////
        // check if operation is cancelled
        
        @synchronized(operationQueue_)
        {
            if([operationQueue_.operations count])
            {
                NSOperation *operation = [operationQueue_.operations firstObject];
                
                if(operation.internalCancel == YES)
                {
                    returnError = PPErrorOperationCancel(nil);
                    break;
                }
            }
        }
        
        
        //////////////////////////////////////////////////
        // handle card section
        
        NSString *sectionTitle = [cardModel displaySectionTitleWithSortByField:sortField indexingMode:indexMode];
        NSMutableArray *cardArray = [cardSectionDict objectForKey:sectionTitle];
        
        if(cardArray == nil)
        {
            cardArray = [NSMutableArray array];
            [cardSectionDict setObject:cardArray forKey:sectionTitle];
        }
        
        [cardArray addObject:cardModel];
        
        
        //////////////////////////////////////////////////
        // handle card image
        
        cardModel.containImageType = WC_IT_None;
        
        if([self hasCardImageWithCardID:cardModel.ID type:WC_IT_FrontSide] == YES)
        {
            cardModel.containImageType |= WC_IT_FrontSide;
        }
        
        if([self hasCardImageWithCardID:cardModel.ID type:WC_IT_BackSide] == YES)
        {
            cardModel.containImageType |= WC_IT_BackSide;
        }
    }
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return cardSectionDict;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCDataControllerProtocol_Section

//===============================================================================
// 取得section資料 （非同步模式，包含暫存機制）
// return: YES表示進行讀取，NO表示沒讀取。
// NOTE: 1. 使用背景執行及暫存機制
//       2. 完成時透過 WCDC_NOTIFY_LoadCardSectionsResult 通知
//       3. 主要給主畫面的card list使用
//===============================================================================
- (BOOL)loadCardSectionsWithGroupID:(NSString *)groupID searchText:(NSString *)searchText forceReload:(BOOL)forceReload
{
    return [self loadCardSectionsWithGroupID:groupID
                                  searchText:searchText
                                 forceReload:forceReload
                                  completion:nil];
}


//================================================================================
// completion == nil 表示完成時會透過 WCDC_NOTIFY_LoadCardSectionsResult 通知。
//================================================================================
- (BOOL)loadCardSectionsWithGroupID:(NSString *)groupID searchText:(NSString *)searchText forceReload:(BOOL)forceReload completion:(LoadCardSectionCompletion)completion
{
    if ([groupID length] == 0)
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    
    WC_GroupID realGroupID = [groupID integerValue];
    
    if(realGroupID == WC_GID_None)
    {
        return NO;
    }

    //////////////////////////////////////////////////
    
    @synchronized(operationQueue_)
    {
        if(operationQueue_ == nil)
        {
            operationQueue_ = [[NSOperationQueue alloc] init];
            [operationQueue_ setMaxConcurrentOperationCount:1];
        }
        
    }

    //////////////////////////////////////////////////

    WCCacheModel *cacheModel = [WCCacheModel defaultCache];

    if(forceReload == YES || cacheModel.groupID != realGroupID || [searchText length] > 0)
    {
        // !! 有其他operation在搜尋時一定要先cancel，尤其是DB部分。
        [self cancelLoadCardSections];
        
        __block typeof(self) blockSelf = self;
        __block LoadCardSectionCompletion blockCompletion = [completion copy];
        __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{

            //////////////////////////////////////////////////
            // 判斷是否取消
            
            @synchronized(operationQueue_)
            {
                if(blockOperation.internalCancel == YES)
                {
                    if(blockCompletion != nil)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            
                            blockCompletion(PPErrorOperationCancel(nil));
                            [blockCompletion release];
                        });
                    }

                    return;
                }
            }
            
            
            //////////////////////////////////////////////////
            // 指定讀取資料的method，iOS/macOS會不一樣。
            
            NSMutableArray *paramArray = [NSMutableArray arrayWithObjects:@(realGroupID), searchText, nil];
            NSError *error = nil;
            
            // !! iphone , macOS的判斷要用下面的寫法，不然iphone也有define TARGET_OS_MAC, 會造成錯誤                
#if TARGET_OS_IPHONE
            
            error = [blockSelf bgThreadLoadCardSections:paramArray];

#elif TARGET_OS_MAC
                
            // mac需求不同，要記錄的資料結構也不同。
            error = [blockSelf bgThreadLoadCardSectionDict:paramArray];

#endif
            

            //////////////////////////////////////////////////
            // 通知結果

            if(blockCompletion != nil)
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    blockCompletion(error);
                    [blockCompletion release];
                });
            }
            else
            {
                [blockSelf notifyLoadCardSectionsResult:YES error:error];
            }
        }];

                                                    
        //////////////////////////////////////////////////
        // 開始執行

        @synchronized(operationQueue_)
        {
            [operationQueue_ addOperation:blockOperation];
        }
        
        return YES;
    }

    return NO;
}


//===============================================================================
// 取得section資料 （同步模式，不包含暫存機制）
//===============================================================================
- (NSMutableArray *)copyCardSectionsWithGroupID:(NSString *)groupID searchText:(NSString *)searchText
{
    if ([groupID length]==0)
        return nil;

    WC_GroupID realGroupID = [groupID integerValue];
    
    NSArray *allCardArray = [cardDBController_ copyAllCardsWithGroupID:realGroupID searchText:searchText];

    if (allCardArray==nil)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    WC_SortedByField currentSortingFields = (WC_SortedByField)[PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
    PPSIC_Mode currentSectionModes = (PPSIC_Mode)[PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];

    //////////////////////////////////////////////////
    // 要針對不同排序的allCardSectionArray的處理放在下面
    NSMutableDictionary *cardSectionDict = [NSMutableDictionary dictionary];;

    for (WCCardModel *cardModel in allCardArray)
    {
        //////////////////////////////////////////////////
        // check if operation is cancelled
        
        @synchronized(operationQueue_)
        {
            if([operationQueue_.operations count])
            {
                NSOperation *operation = [operationQueue_.operations firstObject];
                
                if([operation isCancelled])
                {
                    //                            NSLog(@"%s %@ cancel", __func__, searchText);
                    break;
                }
            }
        }
        
        //////////////////////////////////////////////////
        // 依sectionTitle產生cardSectionModel
        NSString *sectionTitle = [cardModel displaySectionTitleWithSortByField:currentSortingFields indexingMode:currentSectionModes];
        
        // !! 先看有沒有舊的 card seciton
        WCCardSectionModel *cardSectionModel = [cardSectionDict objectForKey:sectionTitle];
        
        // 沒有舊的，產生一個新的 card seciton
        if(cardSectionModel==nil)
        {
            cardSectionModel = [[[WCCardSectionModel alloc] init] autorelease];
            cardSectionModel.title = sectionTitle;
            cardSectionModel.cardArray = [NSMutableArray array];
            
            [cardSectionDict setObject:cardSectionModel forKey:sectionTitle];
        }
        
        cardSectionModel.needResortCard = YES;
        
        [cardSectionModel.cardArray addObject:cardModel];
    }
    
    
    //////////////////////////////////////////////////
    // 轉換為card section array
    NSMutableArray		*allCardSectionArray = [[NSMutableArray alloc] init];
    [allCardSectionArray addObjectsFromArray:[cardSectionDict allValues]];
    
    // sort sections
    [WCCompareTool cardSectionArray:allCardSectionArray sortByField:currentSortingFields];
    
    [allCardArray release];

    return allCardSectionArray;
}


//===============================================================================
// 將Section array轉換為WCCardStreamModel，供flow/thumb顯示模式使用。
// !! cardSectionArray must be sorted
//===============================================================================
- (WCCardStreamModel *)copyCardStreamFromCardSectionArray:(NSMutableArray *)cardSectionArray
{
    if (![cardSectionArray count]) {
        return nil;
    }
    
#ifdef _DUMP_LOG_
    //	[DumpLog logWithMemSize:YES startTime:0 format:@"%s begin", __FUNCTION__];
#endif
    
    WCCardStreamModel *cardStreamModel = [[WCCardStreamModel alloc] init];
    NSInteger curIndex = 0;
    
    for(WCCardSectionModel *cardSectionModel in cardSectionArray)
    {
        [cardStreamModel.sectionIndexDict setObject:@(curIndex) forKey:cardSectionModel.title];
        [cardStreamModel.cardArray addObjectsFromArray:cardSectionModel.cardArray];
        curIndex += [cardSectionModel.cardArray count];
    }
#ifdef _DUMP_LOG_
    //	[DumpLog logWithMemSize:YES startTime:0 format:@"%s end", __FUNCTION__];
#endif
    
    
    return cardStreamModel;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Change display rule methods



//===============================================================================
// Make all display data for card
// NOTE: display data will be composed here
//===============================================================================
- (void)makeCardsDisplayData:(NSArray *)cardArray displayRule:(WCSC_DisplyRule)displayRule
{
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    NSMutableArray	*fieldArray = nil;
    
    
    if(displayRule.sectionIndexMode == PPSIC_M_None)
        displayRule.sectionIndexMode = [PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
    
    if(displayRule.easternNameOrder == WC_NO_None)
        displayRule.easternNameOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kEasternNameOrder];
    
    if(displayRule.westernNameOrder == WC_NO_None)
        displayRule.westernNameOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kWesternNameOrder];
    
    if(displayRule.sortingByField == WC_FT_None)
        displayRule.sortingByField = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];

    for(WCCardModel *cardModel in cardArray)
    {
        //-------------------------------------
        // set default display values
        //-------------------------------------
        cardModel.sectionTitle = PPIndexStringOfOther;
        cardModel.displayName = @"";
        cardModel.displayCompany = @"";
        cardModel.displayJobTitle = @"";
        cardModel.displayAddress = @"";
        cardModel.displayGPS = @"";
        
        //////////////////////////////////////////////////
        // 如果沒有建立時間或修改時間，要補上
        if (cardModel.createdTime==nil)
        {
            cardModel.createdTime = [NSDate date];
        }
        
        if (cardModel.modifiedTime==nil)
        {
            cardModel.modifiedTime = [NSDate date];
        }
        
        //-------------------------------------
        // get display name & section title
        // add each name display data for search
        //-------------------------------------
        fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
        
        for(int i=0; i<[fieldArray count]; i++)
        {
            // use first name field to generate display data
            WCFieldModel *nameField = [fieldArray objectAtIndex:i];
            
            if(i==0)
            {
                // set card display name
                cardModel.displayName = [nameField stringDisplayName];
                
                // get section title
                if(displayRule.sortingByField == WC_SBF_Name)
                {
                    PPIndexingStyle indexingStyle = PPIndexingStyle_English;
                    switch (displayRule.sectionIndexMode)
                    {
                        case PPSIC_M_Stroke:
                        {
                            indexingStyle = PPIndexingStyle_Stroke;
                            break;
                        }
                        case PPSIC_M_Hanpin:
                        {
                            indexingStyle = PPIndexingStyle_Hanpin;
                            break;
                        }
                        case PPSIC_M_Zuyin:
                        {
                            indexingStyle = PPIndexingStyle_Zhuyin;
                            break;
                        }
                        case PPSIC_M_Hiragana:
                        {
                            indexingStyle = PPIndexingStyle_Hiragana;
                            break;
                        }
                        case PPSIC_M_Hangul:
                        {
                            indexingStyle = PPIndexingStyle_Hangul;
                            break;
                        }
                        case PPSIC_M_English:
                        default:
                            break;
                    }
                    cardModel.sectionTitle = [PPIndexingController indexOfString:cardModel.displayName forStyle:indexingStyle];
                }
            }
            
            // set each field display name (for search)
            WCFieldModel *searchFieldModel = [nameField copy];
            searchFieldModel.subType2 = WC_FST2_SearchOnly;
            searchFieldModel.value = [nameField stringDisplayName];
            [nameField setSubType2Field:searchFieldModel];
            [searchFieldModel release];
        }
        
        
        //-------------------------------------
        // get display company / department / job title
        //-------------------------------------
        fieldArray = [cardModel fieldArrayWithType:WC_FT_Company];
        
        if([fieldArray count])
        {
            WCFieldModel *jobTitleField = nil;
            
            for(WCFieldModel *fieldModel in fieldArray)
            {
                if([fieldModel hasFieldWithSubType2:WC_FST2_Company_JobTitle])
                {
                    jobTitleField = fieldModel;
                    break;
                }
            }
            
            cardModel.displayJobTitle = [jobTitleField valueWithSubType2:WC_FST2_Company_JobTitle];
            
            //////////////////////////////////////////////////
            
            WCFieldModel *companyField = nil;
            
            for(WCFieldModel *fieldModel in fieldArray)
            {
                if([fieldModel hasFieldWithSubType2:WC_FST2_Company_Name])
                {
                    companyField = fieldModel;
                    break;
                }
            }
            
            cardModel.displayCompany = [companyField stringDisplayCompany];
            
            // !! 要把日文的株式會社等公司稱謂濾掉
            NSString *trimCompanyName = [WCToolController trimCompanyPrefix:cardModel.displayCompany];
            
            if(![cardModel.displayCompany isEqualToString:trimCompanyName])
                cardModel.displayCompany = trimCompanyName;
            
            // get section title
            if(displayRule.sortingByField == WC_SBF_Company)
            {
                PPIndexingStyle indexingStyle = PPIndexingStyle_English;
                switch (displayRule.sectionIndexMode)
                {
                    case PPSIC_M_Stroke:
                    {
                        indexingStyle = PPIndexingStyle_Stroke;
                        break;
                    }
                    case PPSIC_M_Hanpin:
                    {
                        indexingStyle = PPIndexingStyle_Hanpin;
                        break;
                    }
                    case PPSIC_M_Zuyin:
                    {
                        indexingStyle = PPIndexingStyle_Zhuyin;
                        break;
                    }
                    case PPSIC_M_Hiragana:
                    {
                        indexingStyle = PPIndexingStyle_Hiragana;
                        break;
                    }
                    case PPSIC_M_Hangul:
                    {
                        indexingStyle = PPIndexingStyle_Hangul;
                        break;
                    }
                    case PPSIC_M_English:
                    default:
                        break;
                }
                cardModel.sectionTitle = [PPIndexingController indexOfString:cardModel.displayCompany forStyle:indexingStyle];
            }
        }
        
        
        //-------------------------------------
        // get display address & GPS
        // add each address display data for search
        //-------------------------------------
        fieldArray = [cardModel fieldArrayWithType:WC_FT_Address];
        
        for(int i=0; i<[fieldArray count]; i++)
        {
            WCFieldModel *addressField = [fieldArray objectAtIndex:i];
            NSString *displayAddress = nil;
            
            // !! set address format if not exist
            if(![addressField hasFieldWithSubType2:WC_FST2_Address_Format])
            {
                NSInteger bcrLanguage = [cardModel recognitionlanguageWithFieldSource:addressField.source];
                NSInteger addrFormat = [addressField addressFormatWithBCRLanguage:bcrLanguage];
                WCFieldModel *subType2FieldModel = [addressField copy];
                
                subType2FieldModel.subType2 = WC_FST2_Address_Format;
                subType2FieldModel.value = [NSString stringWithInteger:addrFormat];
                subType2FieldModel.recogRect = CGRectZero;
                [addressField setSubType2Field:subType2FieldModel];
                [subType2FieldModel release];
            }
            
            
            // get display address for each field
            NSInteger bcrLanguage = [cardModel recognitionlanguageWithFieldSource:addressField.source];
            displayAddress = [addressField stringDisplayAddressWithBCRLanguage:bcrLanguage];
            
            // MARK: WCM7之後要同時取GPS??
            
            if(i==0)
            {
                cardModel.displayAddress = displayAddress;
                cardModel.displayGPS = [addressField stringDisplayGPS];
            }
            
            // set each field display name (for search)
            WCFieldModel *searchFieldModel = [addressField copy];
            searchFieldModel.subType2 = WC_FST2_SearchOnly;
            searchFieldModel.value = displayAddress;
            [addressField setSubType2Field:searchFieldModel];
            [searchFieldModel release];
        }
        
        //-------------------------------------
        // get create time section title
        //-------------------------------------
        if(displayRule.sortingByField == WC_SBF_CreateTime)
        {
            // TODO: 時間的section取法不同
            cardModel.sectionTitle = [self sectionTitleForDate:cardModel.createdTime];
        }
        
        //-------------------------------------
        // get modified time section title
        //-------------------------------------
        if(displayRule.sortingByField == WC_SBF_ModifiedTime)
        {
            cardModel.sectionTitle = [self sectionTitleForDate:cardModel.modifiedTime];
        }
        
    }
    
    [pool release];
}


//===============================================================================
// 變更顯示規則
//===============================================================================
- (BOOL)applyDisplayRuleChangeWithSendNotification:(BOOL)sendNotification
{
    NSMutableArray      *cardArray = nil;
    NSMutableArray      *sectionArray = nil;
    BOOL                result = NO;


    self.lastError = nil;
    
    // get all card's field data that is use to make display data
    cardArray = [cardDBController_ copyAllCardsForChangeDisplayRule];

    if([cardArray count] == 0)
    {
        result = YES;
        goto _EXIT;
    }

    // reset display data by new rule
    [self makeCardsStoreData:cardArray];
//    [self makeCardsDisplayData:cardArray displayRule:displayRule];

    // save to databse
    if([cardDBController_ updateAllCardsForChangeDisplayRule:cardArray] == NO)
    {
        self.lastError = cardDBController_.lastError;
        goto _EXIT;
    }

    if(sendNotification == YES)
    {
        WCCardModel *cardModel = [cardArray firstObject];
        
        [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                      infoValue:@[cardModel.ID]];
    }

    result = YES;

    ///////////////////////////////////////////////////////////////////////////////////////////

_EXIT:

    [cardArray release];
    [sectionArray release];

    return result;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card store methods

//===============================================================================
// macOS : 產生儲存用資料，和iOS的差異在sectionTitle儲存的是所有模式會用到的索引。
//===============================================================================
- (void)makeCardsStoreData:(NSArray *)cardArray
{
    WCDisplayNameOrder eastOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kEasternNameOrder];
    WCDisplayNameOrder westOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kWesternNameOrder];
    
    
    //////////////////////////////////////////////////
    // handle cards
    
    for(WCCardModel *cardModel in cardArray)
    {
        @autoreleasepool
        {
            
            NSString *displayNamePhonetic = nil;
            NSString *displayCompanyPhonetic = nil;
            
            //
            cardModel.displayName = @"";
            cardModel.displayCompany = @"";
            cardModel.displayJobTitle = @"";
            cardModel.displayAddress = @"";
            cardModel.displayGPS = @"";
            
            //////////////////////////////////////////////////
            // 如果沒有建立時間或修改時間，要補上
            
            if (cardModel.createdTime==nil)
            {
                cardModel.createdTime = [NSDate date];
            }
            
            if (cardModel.modifiedTime==nil)
            {
                cardModel.modifiedTime = [NSDate date];
            }
            
            
            //////////////////////////////////////////////////
            // handle name data
            
            NSArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
            
            for(int i=0; i<[fieldArray count]; i++)
            {
                // use first name field to generate display data
                WCFieldModel *nameField = [fieldArray objectAtIndex:i];
                
                if(i==0)
                {
                    // set card display name
                    cardModel.displayName = [nameField stringDisplayNameWithEastOrder:eastOrder westOrder:westOrder];
                    displayNamePhonetic = [nameField stringDisplayNamePhoneticWithEastOrder:eastOrder westOrder:westOrder];
                }
                
                // set each field display name (for search)
                WCFieldModel *searchFieldModel = [nameField copy];
                searchFieldModel.subType2 = WC_FST2_SearchOnly;
                searchFieldModel.value = [nameField stringDisplayNameWithEastOrder:eastOrder westOrder:westOrder];
                [nameField setSubType2Field:searchFieldModel];
                [searchFieldModel release];
            }
            
            
            //////////////////////////////////////////////////
            // handle company & department & jobTitle data
            
            fieldArray = [cardModel fieldArrayWithType:WC_FT_Company];
            
            if([fieldArray count])
            {
                WCFieldModel *jobTitleField = [fieldArray firstObject];
            
                cardModel.displayJobTitle = [jobTitleField valueWithSubType2:WC_FST2_Company_JobTitle];
                
                //////////////////////////////////////////////////
                
                WCFieldModel *companyField = [fieldArray firstObject];
                
                cardModel.displayCompany = [companyField stringDisplayCompany];
                displayCompanyPhonetic = [companyField stringDisplayCompanyPhonetic];
                
                // !! 要把日文的株式會社等公司稱謂濾掉
                NSString *trimCompanyName = [WCToolController trimCompanyPrefix:cardModel.displayCompany];
                
                if(![cardModel.displayCompany isEqualToString:trimCompanyName])
                {
                    cardModel.displayCompany = trimCompanyName;
                }
            }
            
            
            //////////////////////////////////////////////////
            // handle address & GPS data
            
            fieldArray = [cardModel fieldArrayWithType:WC_FT_Address];
            
            for(int i=0; i<[fieldArray count]; i++)
            {
                WCFieldModel *addressField = [fieldArray objectAtIndex:i];
                NSString *displayAddress = nil;
                
                // !! set address format if not exist
                if(![addressField hasFieldWithSubType2:WC_FST2_Address_Format])
                {
                    NSInteger bcrLanguage = [cardModel recognitionlanguageWithFieldSource:addressField.source];
                    NSInteger addrFormat = [addressField addressFormatWithBCRLanguage:bcrLanguage];
                    WCFieldModel *subType2FieldModel = [addressField copy];
                    
                    subType2FieldModel.subType2 = WC_FST2_Address_Format;
                    subType2FieldModel.value = [NSString stringWithInteger:addrFormat];
                    subType2FieldModel.recogRect = CGRectZero;
                    [addressField setSubType2Field:subType2FieldModel];
                    [subType2FieldModel release];
                }
                
                // get display address for each field
                NSInteger bcrLanguage = [cardModel recognitionlanguageWithFieldSource:addressField.source];
                displayAddress = [addressField stringDisplayAddressWithBCRLanguage:bcrLanguage];
                
                // MARK: WCM7之後要同時取GPS??
                if(i==0)
                {
                    cardModel.displayAddress = displayAddress;
                    cardModel.displayGPS = [addressField stringDisplayGPS];
                }
                
                // set each field display name (for search)
                WCFieldModel *searchFieldModel = [addressField copy];
                searchFieldModel.subType2 = WC_FST2_SearchOnly;
                searchFieldModel.value = displayAddress;
                [addressField setSubType2Field:searchFieldModel];
                [searchFieldModel release];
            }
            
            
            //////////////////////////////////////////////////
            // handle section title
            
            NSMutableArray *nameSectionTitles = [NSMutableArray array];
            NSMutableArray *companySectionTitles = [NSMutableArray array];
            
            NSArray *indexModes = @[@(PPSIC_M_English),
                                    @(PPSIC_M_Stroke),
                                    @(PPSIC_M_Hanpin),
                                    @(PPSIC_M_Zuyin),
                                    @(PPSIC_M_Hiragana),
                                    @(PPSIC_M_Hangul),
                                    @(PPSIC_M_Thai),
                                    @(PPSIC_M_Swedish)];
            
            for(NSNumber *modeNumer in indexModes)
            {
                @autoreleasepool
                {
                    // !! ios的section有些要加入英文
#if TARGET_OS_IPHONE
                    BOOL isNeedEnglish = NO;
#endif
                    // 除日文外，其他都不用讀音產生
                    NSString *targetName = cardModel.displayName;
                    NSString *targetCompany = cardModel.displayCompany;
                    
                    PPIndexingStyle indexingStyle = PPIndexingStyle_English;
                    switch ([modeNumer integerValue])
                    {
                        case PPSIC_M_Stroke:
                        {
                            indexingStyle = PPIndexingStyle_Stroke;
#if TARGET_OS_IPHONE
                            isNeedEnglish = YES;
#endif
                            break;
                        }
                        case PPSIC_M_Hanpin:
                        {
                            indexingStyle = PPIndexingStyle_Hanpin;
                            break;
                        }
                        case PPSIC_M_Zuyin:
                        {
                            indexingStyle = PPIndexingStyle_Zhuyin;
#if TARGET_OS_IPHONE
                            isNeedEnglish = YES;
#endif
                            break;
                        }
                        case PPSIC_M_Hiragana:
                        {
                            indexingStyle = PPIndexingStyle_Hiragana;
                            // 如果讀音欄位有值，以讀音做indexing,
                            // 如果沒有，以姓名或公司名做indexing
                            targetName = [displayNamePhonetic length]?displayNamePhonetic:cardModel.displayName;
                            targetCompany = [displayCompanyPhonetic length]?displayCompanyPhonetic:cardModel.displayCompany;
#if TARGET_OS_IPHONE
                            isNeedEnglish = YES;
#endif
                            break;
                        }
                        case PPSIC_M_Hangul:
                        {
                            indexingStyle = PPIndexingStyle_Hangul;
#if TARGET_OS_IPHONE
                            isNeedEnglish = YES;
#endif
                            break;
                        }
                        case PPSIC_M_Thai:
                        {
                            indexingStyle = PPIndexingStyle_Thai;
                            
#if TARGET_OS_IPHONE
                            isNeedEnglish = YES;
#endif
                            break;
                        }
                        case PPSIC_M_Swedish:
                        {
                            indexingStyle = PPIndexingStyle_Swedish;
                            break;
                        }
                        case PPSIC_M_English:
                        default:
                            break;
                    }
                    
                    NSString *nameSectionTitle = [PPIndexingController indexOfString:targetName forStyle:indexingStyle];
                    NSString *companySectionTitle = [PPIndexingController indexOfString:targetCompany forStyle:indexingStyle];
                    
                    // !! ios的section有些要加入英文
#if TARGET_OS_IPHONE
                    if (isNeedEnglish==YES)
                    {
                        if ([nameSectionTitle isEqualToString:PPIndexStringOfOther])
                        {
                            nameSectionTitle = [PPIndexingController indexOfString:targetName forStyle:PPIndexingStyle_English];
                        }
                        
                        if ([companySectionTitle isEqualToString:PPIndexStringOfOther])
                        {
                            companySectionTitle = [PPIndexingController indexOfString:targetCompany forStyle:PPIndexingStyle_English];
                        }
                    }
#endif
                    
                    [nameSectionTitles addObject:nameSectionTitle];
                    [companySectionTitles addObject:companySectionTitle];
                } // end of autoreleasepool
            } // end of for loop
            
            // combine sectionTitle string to store in database
            NSString *nameTitles = [nameSectionTitles componentsJoinedByString:@"|"];
            NSString *companyTitles = [companySectionTitles componentsJoinedByString:@"|"];
            cardModel.sectionTitle = [@[nameTitles, companyTitles] componentsJoinedByString:@";"];
        } // end of autoreleasepool
    } // end of for loop
}


//===============================================================================
// macOS : 依照目前的設定值更新sectionTitle
//===============================================================================
- (void)updateDisplaySectionTitleWithCards:(NSArray *)cardArray
{
    WC_SortedByField sortingByField = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
    PPSIC_Mode sectionIndexMode = [PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
    
    
    for(WCCardModel *cardModel in cardArray)
    {
        @autoreleasepool
        {
            NSArray *indexsStrings = [cardModel.sectionTitle componentsSeparatedByString:@";"];
            NSArray *sectionTitles = nil;
            
            switch (sortingByField)
            {
                    case WC_SBF_Name:
                {
                    sectionTitles = [indexsStrings[0] componentsSeparatedByString:@"|"];
                    cardModel.sectionTitle = sectionTitles[sectionIndexMode-1];
                    break;
                }
                    
                    case WC_SBF_Company:
                {
                    sectionTitles = [indexsStrings[1] componentsSeparatedByString:@"|"];
                    cardModel.sectionTitle = sectionTitles[sectionIndexMode-1];
                    break;
                }
                    
#if TARGET_OS_IPHONE
                    
                    case WC_SBF_CreateTime:
                {
                    // !!如果是時間，動態產生sectionTitle
                    cardModel.sectionTitle = [self sectionTitleForDate:cardModel.createdTime];
                    
                    break;
                }
                    case WC_SBF_ModifiedTime:
                {
                    // !!如果是時間，動態產生sectionTitle
                    cardModel.sectionTitle = [self sectionTitleForDate:cardModel.modifiedTime];
                    break;
                }
#endif
                    
                default:
                {
                    NSAssert(NO, @"!!! Invalid sorting field !!!");
                    break;
                }
            }
            
        } // end of autoreleasepool
        
    } // end of for loop
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Duplicate check methods


//==============================================================================
//
//==============================================================================
- (NSString *)noteStringWithCustomField:(WCFieldModel *)customField
{
    if (customField.type!=WC_FT_UserDefine)
    {
        return @"";
    }
    
    //////////////////////////////////////////////////
    WCCustomFieldInfo *fieldInfo = [self customFieldInfoWithGuid:customField.customFieldInfoGuid];
    
    NSString *value = nil;
    
    if (fieldInfo.contentType==WCCustomFieldContentType_Picklist)
    {
        for (WCCustomFieldListItem *item in fieldInfo.picklistItems)
        {
            if ([item.guid isEqualToString:customField.value])
            {
                value = item.itemText;
                break;
            }
        }
    }
    else
    {
        value = customField.value;
    }
    
    return [NSString stringWithFormat:@"%@:%@", fieldInfo.name, value];
}


//==============================================================================
// cardIDs: 要合併的名片ID，要用來合併的base card要放在第一筆
// manualMode: 多選合併時使用，主要不同是都只有正面時，會把一個當正面一筆當反面
//==============================================================================
- (void)mergeWithCardIDs:(NSArray *)cardIDs manualMode:(BOOL)manualMode
{
    WCCardModel *baseCardModel = nil;
    NSString *baseCardID = nil;
    NSInteger cardIndex = 0;
    
    // !! 三個圖的時間要分開記，合併時，每一張都獨立比對modifiedTime，保留最新的圖片
    NSTimeInterval frontCardLastTimeInterval = 0;
    NSTimeInterval backCardLastTimeInterval = 0;
    NSTimeInterval idPhotoLastTimeInterval = 0;
    
    // 特殊處理group, 要取聯集
    NSMutableArray *groupIDArray = [NSMutableArray array];

    
    for (NSString *cardID in cardIDs)
    {
        // 如果存在才要取
        if ([self isCardIDExist:cardID])
        {
            // 以第一張名片當base
            if (cardIndex==0)
            {
                baseCardModel = [self copyCardFullDataWithCardID:cardID];
                baseCardID = cardID;
                
                // 記錄base名片的群組
                for (NSString *groupIDString in baseCardModel.groupIDArray)
                {
                    // 不是未分類才要加, 而且不在array中才要加
                    if ([groupIDString integerValue]!=WC_GID_Unfiled &&
                        [groupIDArray containsObject:groupIDString]==NO)
                    {
                        [groupIDArray addObject:groupIDString];
                    }
                }
                
                //////////////////////////////////////////////////
                // 有圖才要記時間，沒圖就是0
                NSTimeInterval currentTimeInterval = [baseCardModel.modifiedTime timeIntervalSince1970];
                if ([self hasCardImageWithCardID:cardID type:WC_IT_FrontSide])
                {
                    frontCardLastTimeInterval = currentTimeInterval;
                }
                
                if ([self hasCardImageWithCardID:cardID type:WC_IT_BackSide])
                {
                    backCardLastTimeInterval = currentTimeInterval;
                }
                
                if ([self hasCardImageWithCardID:cardID type:WC_IT_IDPhoto])
                {
                    idPhotoLastTimeInterval = currentTimeInterval;
                }
            }
            else
            {
                WCCardModel *currentCardModel = [self copyCardFullDataWithCardID:cardID];
                NSTimeInterval currentTimeInterval = [currentCardModel.modifiedTime timeIntervalSince1970];
                
                //////////////////////////////////////////////////
                // 記錄目前群組
                for (NSString *groupIDString in currentCardModel.groupIDArray)
                {
                    // 不是未分類才要加, 而且不在array中才要加
                    if ([groupIDString integerValue]!=WC_GID_Unfiled &&
                        [groupIDArray containsObject:groupIDString]==NO)
                    {
                        [groupIDArray addObject:groupIDString];
                    }
                }
            
                //////////////////////////////////////////////////
                [baseCardModel combineCard:currentCardModel];
                
                //////////////////////////////////////////////////
                [baseCardModel combineCustomFieldsWithCardModel:currentCardModel
                                          withRedandanceHandler:^NSString *(WCFieldModel *redandanceField) {
                                              
                                              return [self noteStringWithCustomField:redandanceField];
                                          }];
                
                //////////////////////////////////////////////////
                // 合併影像資料，以時間最新的影像儲存

                // !!如果是手動，且都只有正面，把第二張當反面
                if (manualMode &&
                    (frontCardLastTimeInterval!=0 &&
                     backCardLastTimeInterval==0 &&
                     [self hasCardImageWithCardID:cardID type:WC_IT_FrontSide]==YES &&
                     [self hasCardImageWithCardID:cardID type:WC_IT_BackSide]==NO))
                {
                    NSData *currentImageData = [self copyCardImageDataWithCardID:cardID type:WC_IT_FrontSide subtype:WC_IST_Original];
                    [self setCardImageData:currentImageData withCardID:baseCardID type:WC_IT_BackSide];
                    [currentImageData release];
                }
                else
                {
                    if ([self hasCardImageWithCardID:cardID type:WC_IT_FrontSide])
                    {
                        // 如果目前的圖比較新的話，要存下目前的圖
                        if (currentTimeInterval>frontCardLastTimeInterval)
                        {
                            NSData *currentImageData = [self copyCardImageDataWithCardID:cardID type:WC_IT_FrontSide subtype:WC_IST_Original];
                            [self setCardImageData:currentImageData withCardID:baseCardID type:WC_IT_FrontSide];
                            [currentImageData release];
                            
                            frontCardLastTimeInterval = currentTimeInterval;
                        }
                    }
                    
                    if ([self hasCardImageWithCardID:cardID type:WC_IT_BackSide])
                    {
                        // 如果目前的圖比較新的話，要存下目前的圖
                        if (currentTimeInterval>backCardLastTimeInterval)
                        {
                            NSData *currentImageData = [self copyCardImageDataWithCardID:cardID type:WC_IT_BackSide subtype:WC_IST_Original];
                            [self setCardImageData:currentImageData withCardID:baseCardID type:WC_IT_BackSide];
                            [currentImageData release];
                            
                            backCardLastTimeInterval = currentTimeInterval;
                        }
                    }
                }
                
                if ([self hasCardImageWithCardID:cardID type:WC_IT_IDPhoto])
                {
                    // 如果目前的圖比較新的話，要存下目前的圖
                    if (currentTimeInterval>idPhotoLastTimeInterval)
                    {
                        NSData *currentImageData = [self copyCardImageDataWithCardID:cardID type:WC_IT_IDPhoto subtype:WC_IST_Thumbnail];
                        [self setCardImageData:currentImageData withCardID:baseCardID type:WC_IT_IDPhoto];
                        [currentImageData release];
                        
                        idPhotoLastTimeInterval = currentTimeInterval;
                    }
                }
                
                [currentCardModel release];
                
                //////////////////////////////////////////////////
                // 合併後刪除此筆資料
                [self removeCards:@[cardID]];
            }
            cardIndex ++;
        }
    }
    
    if (baseCardModel)
    {
        //////////////////////////////////////////////////
        // 設定新的group
        [baseCardModel setGroupIDArray:groupIDArray isInitCard:NO];

        [self updateCard:baseCardModel withSendNotification:YES];
    }
    
    [baseCardModel release];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Unrecognized methods

//===============================================================================
// 加入未辨識影像
//===============================================================================
- (BOOL)addUnrecognizedImage:(CPImage *)image
{
    self.lastError = nil;
    
    NSError *error = nil;
    NSString *unrecognizedID = [NSString stringWithFormat:@"%u", (uint)([NSDate timeIntervalSinceReferenceDate]*10)];
    BOOL result = [cardImageController_ setImage:image withCardID:unrecognizedID type:WC_IT_FrontSide error:&error];

    self.lastError = error;
    
    return result;
}


//===============================================================================
// 刪除未辨識影像
//===============================================================================
- (BOOL)removeUnrecognizedImageWithID:(NSString *)unrecognizedID
{
    BOOL result = [cardImageController_ removeImageWithCardID:unrecognizedID type:WC_IT_FrontSide];

    return result;
}


//===============================================================================
// 取得所有未辨識影像ID
//===============================================================================
- (NSArray *)unrecognizedIDArray
{
    NSMutableSet *unrecognizedIDSet = [[NSMutableSet alloc] init];
    NSString *unrecognizedDirPath = [NSString stringWithFormat:@"%@/images", baseDirPath_];

    NSFileManager *fileManager = [NSFileManager defaultManager];

    NSDirectoryEnumerator *enumerator = [fileManager enumeratorAtPath:unrecognizedDirPath];
    NSString *file;
    NSString *unrecognizedID;


    while (file = [enumerator nextObject])
    {
        unrecognizedID = [[file componentsSeparatedByString:@"_"] objectAtIndex:0];
        [unrecognizedIDSet addObject:unrecognizedID];
    }

    NSArray *resultArray = [[unrecognizedIDSet allObjects] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];

    [unrecognizedIDSet release];

    return resultArray;
}


//===============================================================================
// 取得未辨識影像
//===============================================================================
- (CPImage *)copyUnrecognizedImageWithID:(NSString *)unrecognizedID imageSubType:(WC_ImageSubType)imageSubType
{
    CPImage *image = [cardImageController_ copyImageWithCardID:unrecognizedID type:WC_IT_FrontSide subtype:imageSubType];

    return image;
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Utiliey methods


//===============================================================================
//
//===============================================================================
- (void)compactData
{
    [cardDBController_ compactDB];
}


//===============================================================================
//
//===============================================================================
- (void)reconect
{
    [cardDBController_ disconnectDB];
    [cardDBController_ connectDB];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - 以下為WCT新增或調整函式
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTDataControllerProtocol_Group



//===============================================================================
// 新增群組
// return : WCGroupModel (release after using)
//===============================================================================
- (WCGroupModel *)newGroupWithName:(NSString *)groupName syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    NSAssert(NO, @"v1.3.0 群組需OnLine更新後不會使用newGroupWithName:syncActionModel:，請檢查");

    self.lastError = nil;
    WCGroupModel	*groupModel = nil;
    NSString *groupIDString = [self groupIDWithName:groupName];
    NSInteger groupID = [groupIDString integerValue];
    
    // 如果沒有群組ID，建立新的
    if(groupID== WC_GID_None)
    {
        groupID = [cardDBController_ addGroupWithName:groupName syncActionModel:syncActionModel];
        
        // 新增時要修改GroupOrderModifyedTime
        [self setGroupOrderModifiedTime:syncActionModel.modifiedTime?:[NSDate date]];
    }
    
    if(groupID > 0)
    {
        groupModel = [[WCGroupModel alloc] init];
        groupModel.ID = groupID;
        groupModel.name = groupName;
        groupModel.cardCount = 0;
        groupModel.editable = YES;
        
        // !! 通知群組新增
        [self notifyDataChanged:WCDC_NOTIFY_GroupAdd
                        infoKey:WCDC_NOTIFY_UserInfo_kGroupID
                      infoValue:@(groupID)];
        
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return groupModel;
}



//===============================================================================
// 新增群組
// return : WCGroupModel (release after using)
//===============================================================================
- (WCGroupModel *)newGroupWithNameForSync:(NSString *)groupName
{
    WCTGroupSyncActionModel *groupSyncActionModel = nil;
    if([groupName length]==0)
    {
        return nil;
    }
    
    groupSyncActionModel = [[WCTGroupSyncActionModel alloc] init];
    groupSyncActionModel.dataGuid = [NSString GUID];
    groupSyncActionModel.actionType = WCTSyncActionType_Add;
    groupSyncActionModel.modifiedTime = [NSDate date];
    
    WCGroupModel *groupModel = [self newGroupWithNameForSync:groupName syncActionModel:groupSyncActionModel];
    [groupSyncActionModel release];
    
    return groupModel;
}

//===============================================================================
// 新增群組
// return : WCGroupModel (release after using)
//===============================================================================
- (WCGroupModel *)newGroupWithNameForSync:(NSString *)groupName syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    self.lastError = nil;
    WCGroupModel	*groupModel = nil;
    NSInteger		groupID = WC_GID_None;
    
    
    if(groupID == WC_GID_None)
        groupID = [cardDBController_ addGroupWithName:groupName syncActionModel:syncActionModel];
    
    if(groupID > 0)
    {
        groupModel = [[WCGroupModel alloc] init];
        groupModel.ID = groupID;
        groupModel.name = groupName;
        groupModel.cardCount = 0;
        
        // !! 通知群組新增
        [self notifyDataChanged:WCDC_NOTIFY_GroupAdd
                        infoKey:WCDC_NOTIFY_UserInfo_kGroupID
                      infoValue:@(groupID)];
        
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return groupModel;
}


//===============================================================================
// 刪除群組
//===============================================================================
- (BOOL)removeGroupWithID:(WC_GroupID)groupID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    self.lastError = nil;
    BOOL result = [cardDBController_ removeGroupWithID:groupID syncActionModel:syncActionModel];
    
    if(result)
    {
        // 更新類別順序修改時間
        [self setGroupOrderModifiedTime:syncActionModel.modifiedTime?:[NSDate date]];

        [self notifyDataChanged:WCDC_NOTIFY_GroupRemoved
                        infoKey:WCDC_NOTIFY_UserInfo_kGroupID
                      infoValue:@(groupID)];
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return result;
}



//===============================================================================
// 變更群組名稱
//===============================================================================
- (BOOL)updateGroupName:(NSString *)groupName withID:(NSInteger)groupID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    self.lastError = nil;
    BOOL result = [cardDBController_ updateGroupName:groupName withID:groupID syncActionModel:syncActionModel];
    
    if(result)
    {
        [self notifyDataChanged:WCDC_NOTIFY_GroupNameChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kGroupID
                      infoValue:@(groupID)];
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return result;
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateGroupGuid:(NSString *)guid withID:(NSInteger)groupID;
{
    return [cardDBController_ updateGroupGuid:guid withID:groupID];
}

//===============================================================================
//
//===============================================================================
- (WC_GroupID)groupIDWithGuid:(NSString *)groupGuid
{
    WC_GroupID groupID = WC_GID_None;
    NSInteger result = [cardDBController_ groupIDWithGuid:groupGuid];
    
    if(result != -1)
    {
        groupID = result;
    }
    
    return groupID;
}


//===============================================================================
// 取得群組名稱
//===============================================================================
- (NSString *)groupNameWithGuid:(NSString *)groupGuid
{
    NSString *groupName = [cardDBController_ groupNameWithGuid:groupGuid];
    
    return [groupName localized];
}


//================================================================================
//
//================================================================================
- (WCGroupModel *)groupWithGuid:(NSString *)groupGuid
{
    self.lastError = nil;
    
    WCGroupModel *groupModel = [cardDBController_ groupWithGuid:groupGuid];
    
    if(groupModel == nil)
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return groupModel;
}


//================================================================================
//
//================================================================================
- (NSDictionary *)groupIDAndGuidMappingDict
{
    return [cardDBController_ groupIDAndGuidMappingDict];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTDataControllerProtocol_Card

//===============================================================================
// 加入名片
//===============================================================================
- (BOOL)addCard:(WCCardModel *)cardModel syncActionModel:(WCTCardSyncActionModel*)syncActionModel withSendNotification:(BOOL)sendNotification
{
    self.lastError = nil;
    
    if(!cardModel)
    {
        [self setLastErrorWithCode:WCDC_ErrorCode_Normal description:@"No cardModel"];
        return NO;
    }
    
    if(cardModel.sourceID == WC_SID_None)
    {
        cardModel.sourceID = WC_SID_WorldCard;
    }
    
    
    //////////////////////////////////////////////////
    // fix field length
    [cardModel fixFieldLength];
    
    //////////////////////////////////////////////////
    
    // 所有模式的sectionTitle都儲存起來
    [self makeCardsStoreData:@[cardModel]];
    
    //////////////////////////////////////////////////
    // !! WCT沒有google 預設群組，但匯入WCXF時可能會有，所以要另外處理，把google 預設群組新增為一般群組
    NSMutableArray *convertedGroupIDArray = [NSMutableArray array];
    
    for (NSString *groupIDString in cardModel.groupIDArray)
    {
        NSString *tempGroupIDString = groupIDString;
        WC_GroupID groupID = [groupIDString integerValue];
        if (groupID==WC_GID_Google_MyContacts ||
            groupID==WC_GID_Google_Friends ||
            groupID==WC_GID_Google_Family ||
            groupID==WC_GID_Google_Coworkers)
        {
            NSString *groupName = nil;
            switch (groupID)
            {
                case WC_GID_Google_MyContacts:
                {
                    groupName = [@"MLS_MyContacts" localized];
                    break;
                }
                    
                case WC_GID_Google_Friends:
                {
                    groupName = [@"MLS_Friends" localized];
                    break;
                }
                    
                case WC_GID_Google_Family:
                {
                    groupName = [@"MLS_Family" localized];
                    break;
                }
                    
                case WC_GID_Google_Coworkers:
                {
                    groupName = [@"MLS_Coworkers" localized];
                    break;
                }
                default:
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            tempGroupIDString = [self groupIDWithName:groupName];
            if([tempGroupIDString integerValue]==WC_GID_None)
            {
                // 群組不存在的話要建立新的
                WCGroupModel *groupModel = [self newGroupWithName:groupName];
                tempGroupIDString = [NSString stringWithInteger:groupModel.ID];
                [groupModel release];
            }
         }
        
        // 紀錄轉換後的groupIDArray
        [convertedGroupIDArray addObject:tempGroupIDString];
    }
    
    [cardModel setGroupIDArray:convertedGroupIDArray isInitCard:NO] ;
    //////////////////////////////////////////////////
    
    if([cardDBController_ insertCard:cardModel syncActionModel:syncActionModel isImportMode:YES isInTransaction:NO] == YES)
    {
        // !! 注意insertCard裡已經有設定group資料了，這裡再設定一次是多餘的，而且會影響到同步設定。
        //        [self setCardGroupsWithCardID:cardModel.ID groupIDArray:cardModel.groupIDArray isImportMode:YES];
        
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
#if TARGET_OS_IPHONE
        
        // !! ios寫到cache中的CardModel的sectionTitle要以目前設定為主
        
        WC_SortedByField sortingByFields = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
        PPSIC_Mode sectionIndexMode = [PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
        
        cardModel.sectionTitle = [cardModel displaySectionTitleWithSortByField:sortingByFields indexingMode:sectionIndexMode];
        
        if((cacheModel && [cardModel isBelongToGroupID:cacheModel.groupID]))
        {
            // !! must resort sections in cache if display data changed
            [cacheModel addCardToCache:cardModel];
            
            // 排序cacheModel中的section array 要用WCCacheModel的methods
            [WCCacheModel sectionArray:cacheModel.allCardSectionArray sortByField:sortingByFields];
            
            if(sendNotification)
            {
                // notify UI update
                [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                                infoKey:WCDC_NOTIFY_UserInfo_kAddCard
                              infoValue:@[cardModel.ID]];
            }
        }

        //////////////////////////////////////////////////
        // 同時處理我的最愛cache
        
        if ((cardModel.tagMask & WC_TagMask_Favorite)==WC_TagMask_Favorite)
        {
            [cacheModel addFavoriteCardToCache:cardModel];
        }
        else
        {
            [cacheModel removeFavoriteCardFromCacheWithCardID:cardModel.ID];
        }
        
        if(sendNotification)
        {
            // notify UI update
            [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kAddCard
                          infoValue:@[cardModel.ID]];
        }

        
#elif TARGET_OS_MAC

        // mac版要判斷sourceID，Unverified/Favorite要分開判斷。
        if(cacheModel != nil && cacheModel.sourceID == cardModel.sourceID &&
           ((cacheModel.groupID == WC_GID_Unverified && cardModel.tagMask & WC_TagMask_Unverified) ||
            (cacheModel.groupID == WC_GID_Favorite && cardModel.tagMask & WC_TagMask_Favorite) ||
            [cardModel isBelongToGroupID:cacheModel.groupID]))
        {
            // !! must resort sections in cache if display data changed
            [cacheModel addCardToCache:cardModel];
        }
        
        // macOS即使不是目前選取的group也要通知
        if(sendNotification)
        {
            // notify UI update
            [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                            infoKey:WCDC_NOTIFY_UserInfo_kAddCard
                          infoValue:@[cardModel.ID]];
        }

#endif
        
        return YES;
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return NO;
}


//===============================================================================
// 更新名片內容
//===============================================================================
- (BOOL)updateCard:(WCCardModel *)cardModel withSendNotification:(BOOL)sendNotification syncActionModel:(WCTCardSyncActionModel *)syncActionModel mustExist:(BOOL)mustExist
{
    if (cardModel==nil)
    {
        [self setLastErrorWithCode:WCDC_ErrorCode_Normal description:@"CardModle不能為空"];
        return NO;
    }
    
    self.lastError = nil;
    BOOL result = NO;

    //////////////////////////////////////////////////
    // fix field length
    [cardModel fixFieldLength];
    
    // 所有模式的sectionTitle都儲存起來
    [self makeCardsStoreData:@[cardModel]];
    
    BOOL updateResult = [cardDBController_ updateCard:cardModel syncActionModel:syncActionModel mustExist:mustExist];
    
    // !! 如果mustExist==NO, 但updateResult==NO, 不算錯誤
    if(mustExist==NO&&updateResult==NO)
    {
        return YES;
    }
    
    //////////////////////////////////////////////////
    if(updateResult)
    {
        //////////////////////////////////////////////////
        // 處理cache資料
        // !! must resort sections in cache if display data changed
        
        WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
        if(cacheModel)
        {
            
            // 檢查是否還在目前cache的group裡
            BOOL isInCachedGroup = NO;
            
            for(NSString *groupID in cardModel.groupIDArray)
            {
                // !! 移除與source有關的判斷
                // 目前判斷在目前chacheModel的規則加下
                // 1。如果目前cacheModel是WC_GID_All
                // 2。groupID與CardModel的groupID一致
                if(cacheModel.groupID == WC_GID_All)
                {
                    isInCachedGroup = YES;
                    break;
                }
                else if (cacheModel.groupID == [groupID intValue])
                {
                    isInCachedGroup = YES;
                    break;
                }
            }
            
#if TARGET_OS_IPHONE
            
            //////////////////////////////////////////////////
            // !! iOS寫到cache中的CardModel的sectionTitle要以目前設定為主
            
            WC_SortedByField sortingByFields = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
            PPSIC_Mode sectionIndexMode = [PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
            
            cardModel.sectionTitle = [cardModel displaySectionTitleWithSortByField:sortingByFields indexingMode:sectionIndexMode];
            
#elif TARGET_OS_MAC
            
            //////////////////////////////////////////////////
            // macOS要判斷favorite和unverified
            
            if(isInCachedGroup == NO &&
               (((cardModel.tagMask & WC_TagMask_Favorite) && (cacheModel.groupID == WC_GID_Favorite)) ||
                ((cardModel.tagMask & WC_TagMask_Unverified) && (cacheModel.groupID == WC_GID_Unverified))))
            {
                isInCachedGroup = YES;
            }
#endif
            
            
            //////////////////////////////////////////////////
            
            if(isInCachedGroup)
            [cacheModel updateCardInCache:cardModel];
            else [cacheModel removeCardFromCacheWithCardID:cardModel.ID];
            
            // 排序, 排序cacheModel中的section array 要用WCCacheModel的methods
#if TARGET_OS_IPHONE
            [WCCacheModel sectionArray:cacheModel.allCardSectionArray sortByField:sortingByFields];
#endif
            
            //////////////////////////////////////////////////
            // 同時處理我的最愛cache
            
            if ((cardModel.tagMask & WC_TagMask_Favorite)==WC_TagMask_Favorite)
            {
                WCCardModel *favoriteCardModel = [cacheModel favoriteCardFromCacheWithCardID:cardModel.ID];
                if (favoriteCardModel==nil)
                {
                    [cacheModel addFavoriteCardToCache:cardModel];
                }
                else
                {
                    [cacheModel updateFavoriteCardInCache:cardModel];
                }
            }
            else
            {
                [cacheModel removeFavoriteCardFromCacheWithCardID:cardModel.ID];
            }
        } // end of if(cacheModel)
        
        //////////////////////////////////////////////////
        // notify UI update
        if(sendNotification)
        {
            // 如果是虛擬名片圖，要先清掉cache的圖，這樣名片圖才會更新
            if(![self hasCardImageWithCardID:cardModel.ID type:WC_IT_FrontSide])
            {
                WCCacheModel *cacheModel = [self cacheOfCurrentSource];
                [cacheModel removeThumbImageWithCardID:cardModel.ID];
                [cacheModel removeIDPhotoWithCardID:cardModel.ID];
            }
            
            if(sendNotification)
            {
                //////////////////////////////////////////////////
                // 通知名片變動
                [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                                infoKey:WCDC_NOTIFY_UserInfo_kUpdateCard
                              infoValue:[NSArray arrayWithObject:cardModel.ID]];
                
                //////////////////////////////////////////////////
                // 我的最愛也要通知
                [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                                infoKey:WCDC_NOTIFY_UserInfo_kAddCard
                              infoValue:@[cardModel.ID]];
            }
            
            // 更新圖片有更新
            if(![self hasCardImageWithCardID:cardModel.ID type:WC_IT_FrontSide])
            [self notifyImageChanged:WCDC_NCT_Update imageType:WC_IT_FrontSide cardID:cardModel.ID];
        }
        
        result = YES;
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }

    return result;
}

//===============================================================================
// 更新時的刪除名片動作 (for sync use)
//===============================================================================
- (BOOL)removeCardDataForSyncUpdate:(NSString *)cardID
{
    self.lastError = nil;
    
    //---------------------------------
    // delete cards in DB
    //---------------------------------
    if([cardDBController_ removeCardWithCardID:cardID syncActionModel:nil forUpdate:NO isInTransaction:NO] == YES)
    {
        // remove images
        [self removeAllCardImagesWithCardID:cardID];

        //////////////////////////////////////////////////
        WCCacheModel *cacheModel = [self cacheOfCurrentSource];
        
        // 後面的removeUnverifiedWithCardID也會更新時間，所以這邊就不更新了
        if([cardDBController_ removeFavoriteWithCardID:cardID isInTransaction:NO])
        {
            // 我的最愛的cache也要同時清除
            [cacheModel removeFavoriteCardFromCacheWithCardID:cardID];
        }
        
        if([cardDBController_ removeUnverifiedWithCardID:cardID])
        {
            [cacheModel removeCardFromCacheWithCardID:cardID];
        }
        
        
        // 通知介面更新
        [self notifyDataChanged:WCDC_NOTIFY_DisplayDataChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kRemoveCard
                      infoValue:@[cardID]];
        
        // 我的最愛也要通知
        [self notifyDataChanged:WCDC_NOTIFY_FavoriteChanged
                        infoKey:WCDC_NOTIFY_UserInfo_kRemoveCard
                      infoValue:@[cardID]];
        
        return YES;
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    return NO;
}


//==============================================================================
//
//==============================================================================
- (NSDate *)cardModifiedTimeWithCardID:(NSString *)cardID
{
    return [cardDBController_ cardModifiedTimeWithCardID:cardID];
}


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTDataControllerProtocol_Duplicate
//#define DumpDuplicateData

//===============================================================================
// 取得重覆資料, 以
// return : array of WCCardSectionModel, sorted by modified time
//
//     returnArray -> cardArray -> cardModel
//                                 cardModel
//                                 ...
//                    cardArray -> cardModel
//                                 cardModel
//                                 ...
//===============================================================================
- (NSMutableArray *)copyDuplicateCardSectionModelsWithOwnerID:(NSString *)ownerID compareNameOnly:(BOOL)compareNameOnly
{
    NSMutableArray *cardSectionModelArray = nil;
    NSMutableArray *groupedCardArray =  [cardDBController_ copyDuplicateCardsWithSameNameOnly:compareNameOnly];
    
    //    ownerID = nil;
    
#ifdef DumpDuplicateData
    NSInteger groupIndex = 0;
    
    PPLogController *resultLog = [[PPLogController alloc] init];
    [resultLog setFileName:@"result_Sorted" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
    resultLog.toFile = YES;
    resultLog.toConsole = NO;
    
#endif
    
    for (NSMutableArray *duplicateCardArray in groupedCardArray)
    {

        
        // 重複聯絡人用的比對規則 (team)
        // 1。擁有者排最前面
        // 2。沒有擁有者就照時間排
        [duplicateCardArray sortUsingComparator:^NSComparisonResult(WCCardModel *model1, WCCardModel *model2) {
            
            // 判斷擁有者
            if ([ownerID length]>0)
            {
                if ([model1.creator isEqualToString:model2.creator]==NO)
                {
                    if ([model1.creator isEqualToString:ownerID]==YES)
                    {
                        return NSOrderedAscending;
                    }
                    else if ([model2.creator isEqualToString:ownerID]==YES)
                    {
                        return NSOrderedDescending;
                    }
                }
            }
            
            return [model2.modifiedTime compare:model1.modifiedTime];;
        }];
        
        
#ifdef DumpDuplicateData

        NSInteger cardIDIndex = 0;
        for (WCCardModel *cardModel in duplicateCardArray)
        {
            if (cardIDIndex==0)
            {
                [resultLog logWithMask:PPLogControllerMask_Normal format:@"Group %td", groupIndex];
            }
            
            [resultLog logWithMask:PPLogControllerMask_Normal format:@"\t%@", cardModel.ID];
            cardIDIndex ++;
        }
#endif
        
        if (cardSectionModelArray==nil)
        {
            cardSectionModelArray = [[NSMutableArray alloc] init];
        }
        
        WCCardSectionModel *duplicateCardSectionModel = [[WCCardSectionModel alloc] init];
        if(duplicateCardSectionModel)
        {
            WCCardModel *firstCard = [duplicateCardArray firstObject];
            duplicateCardSectionModel.title = [firstCard displayName];
            duplicateCardSectionModel.cardArray = duplicateCardArray;
            
            [cardSectionModelArray addObject:duplicateCardSectionModel];
            [duplicateCardSectionModel release];
        }
        
#ifdef DumpDuplicateData
        groupIndex++;
#endif
    }
    [groupedCardArray release];
    
#ifdef DumpDuplicateData
    [resultLog release];
#endif

    return cardSectionModelArray;
}


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTDataControllerProtocol_SystemGuid


//================================================================================
//
//================================================================================
- (BOOL)setDefaultGroupGuid:(NSString *)groupGuid withGroupID:(WC_GroupID)groupID
{
    if (groupID!=WC_GID_All &&
        groupID!=WC_GID_Unfiled)
    {
        return NO;
    }
    
    return [cardDBController_ updateGroupGuid:groupGuid withID:groupID];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Account list methods

//================================================================================
//
//================================================================================
- (NSMutableArray *)copyAllAccountList
{
    NSMutableArray *resultArray = [cardDBController_ copyAllAccountListWithResign:NO];
    [resultArray sortUsingComparator:^NSComparisonResult(WCTAccountRelationModel *obj1, WCTAccountRelationModel *obj2) {
        
        return [obj1.name compare:obj2.name options:NSCaseInsensitiveSearch|NSDiacriticInsensitiveSearch|NSNumericSearch|NSForcedOrderingSearch];
    }];
    return resultArray;
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)copyAllAccountListWithResign:(BOOL)resign
{
    NSMutableArray *resultArray = [cardDBController_ copyAllAccountListWithResign:resign];
    [resultArray sortUsingComparator:^NSComparisonResult(WCTAccountRelationModel *obj1, WCTAccountRelationModel *obj2) {
        
        return [obj1.name compare:obj2.name options:NSCaseInsensitiveSearch|NSDiacriticInsensitiveSearch|NSNumericSearch|NSForcedOrderingSearch];
    }];
    return resultArray;
}


//================================================================================
//
//================================================================================
- (BOOL)updateAllAccountWithList:(NSArray *)list
{
    return [cardDBController_ updateAllAccountWithList:list];
}


//==============================================================================
//
//==============================================================================
- (WCTAccountRelationModel *)accountRelationWithAcountGuid:(NSString *)accountGuid
{
    return [cardDBController_ accountRelationWithAcountGuid:accountGuid];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Shared account methods

//================================================================================
//
//================================================================================
- (NSMutableArray *)copySharedAccountWithCardID:(NSString *)cardID
{
    return [cardDBController_ copySharedAccountWithCardID:cardID];
}


//================================================================================
//
//================================================================================
- (BOOL)setCardSharedAccountGuids:(NSArray *)sharedAccountGuids withCardID:(NSString *)cardID
{
    return [cardDBController_ setCardSharedAccountGuids:sharedAccountGuids withCardID:cardID];
}


//================================================================================
//
//================================================================================
- (BOOL)setCardSharedAccountGuids:(NSArray *)sharedAccountGuids withCardIDs:(NSArray *)cardIDs
{
    return [cardDBController_ setCardSharedAccountGuids:sharedAccountGuids withCardIDs:cardIDs];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group sync action methods


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyGroupSyncActions
{
    return [cardDBController_ copyGroupSyncActions];
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyGroupSyncActionsAfterDate:(NSDate *)date
{
    return [cardDBController_ copyGroupSyncActionsAfterDate:date];
}


//==============================================================================
//
//==============================================================================
- (WCTGroupSyncActionModel *)copyGroupSyncActionModelWithGuid:(NSString *)groupGuid
{
    return [cardDBController_ copyGroupSyncActionModelWithGuid:groupGuid];
}


//================================================================================
//
//================================================================================
- (BOOL)removeGroupSyncActionWithGuid:(NSString *)groupGuid
{
    return [cardDBController_ removeGroupSyncActionWithGuid:groupGuid];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateGroupModifiedTime:(NSDate *)modifiedTime withGuid:(NSString *)guid
{
    return [cardDBController_ updateGroupModifiedTime:modifiedTime withGuid:guid];
}


//==============================================================================
//
//==============================================================================
- (BOOL)resetGroupSyncActionToNoneWithGuid:(NSString *)guid
{
    return [cardDBController_ resetGroupSyncActionToNoneWithGuid:guid];
}


//==============================================================================
// 同步用，不會更新groupOrderModifiedTime
//==============================================================================
- (BOOL)removeGroupWithIDForSync:(NSString *)groupID
{
    if([groupID length]==0)
        return NO;
    
    WC_GroupID realGroupID = [groupID integerValue];
    
    NSString *groupGuid = [cardDBController_ groupGuidWithID:realGroupID];
    if ([groupGuid length]==0)
    {
        self.lastError = cardDBController_.lastError;
        return NO;
    }
    
    WCTGroupSyncActionModel *groupSyncActionModel = nil;
    groupSyncActionModel = [[[WCTGroupSyncActionModel alloc] init] autorelease];
    groupSyncActionModel.dataGuid = groupGuid;
    groupSyncActionModel.actionType = WCTSyncActionType_Delete;
    groupSyncActionModel.modifiedTime = [NSDate date];
    
    return [self removeGroupWithIDForSync:realGroupID syncActionModel:groupSyncActionModel];
}


//==============================================================================
// 同步用，不會更新groupOrderModifiedTime
//==============================================================================
- (BOOL)removeGroupWithIDForSync:(WC_GroupID)groupID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    self.lastError = nil;
    BOOL result = [cardDBController_ removeGroupWithID:groupID syncActionModel:syncActionModel];
    
    if(result)
    {
        [self notifyDataChanged:WCDC_NOTIFY_GroupRemoved
                        infoKey:WCDC_NOTIFY_UserInfo_kGroupID
                      infoValue:@(groupID)];
    }
    else
    {
        self.lastError = cardDBController_.lastError;
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - imported card id mapping


//==============================================================================
//
//==============================================================================
- (NSString *)cardIDWithImportedCardID:(NSString *)importedCardID
{
    if([importedCardID length] > 0)
    {
        return [cardDBController_ cardIDWithImportedCardID:importedCardID];
    }
    else
    {
        return nil;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)setImporedCardID:(NSString *)importedCardID withCardID:(NSString *)cardID
{
    if([importedCardID length] > 0 && [cardID length] > 0)
    {
        return [cardDBController_ setImporedCardID:importedCardID withCardID:cardID];
    }
    else
    {
        return NO;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeImportedCardIDWithCardID:(NSString *)cardID
{
    return [cardDBController_ removeImportedCardIDWithCardID:cardID];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card sync action methods


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyCardSyncActions
{
    return [cardDBController_ copyCardSyncActions];
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyCardSyncActionsAfterDate:(NSDate *)date
{
    return [cardDBController_ copyCardSyncActionsAfterDate:date];
}


//==============================================================================
//
//==============================================================================
- (WCTCardSyncActionModel *)copyCardSyncActionModelWithCardID:(NSString *)cardID;
{
    return [cardDBController_ copyCardSyncActionModelWithCardID:cardID];
}

//================================================================================
//
//================================================================================
- (BOOL)removeCardSyncActionWithCardID:(NSString *)cardID
{
    return [cardDBController_ removeCardSyncActionWithCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncAction:(WCTCardSyncActionModel *)syncActionModel withCheckModifyTime:(NSDate *)checkModifyTime
{
    return [cardDBController_ updateCardSyncAction:syncActionModel withCheckModifyTime:checkModifyTime isInTransaction:NO];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncActionModifiedTime:(NSDate *)syncAcctinoModifiedTime withCardID:(NSString *)cardID
{
    return [cardDBController_ updateCardSyncActionModifiedTime:syncAcctinoModifiedTime withCardID:cardID];
}


//==============================================================================
//
//==============================================================================
- (BOOL)resetCardSyncActionToNoneWithCardID:(NSString *)cardID checkModifyTime:(NSDate *)checkModifyTime
{
    return [cardDBController_ resetCardSyncActionToNoneWithCardID:cardID checkModifyTime:checkModifyTime];
}


//==============================================================================
//
//==============================================================================
- (BOOL)replaceGuidFromOld:(NSString *)oldGuid toNew:(NSString *)newGuid
{
    return [cardDBController_ replaceGuidFromOld:oldGuid toNew:newGuid];
}


//==============================================================================
//
//==============================================================================
- (BOOL)isCardSyncedWithGuids:(NSArray *)guids
{
    for (NSString *guid in guids)
    {
        if ([cardDBController_ isCardSyncedWithGuid:guid]==NO)
        {
            return NO;
        }
    }
    return YES;
}


//==============================================================================
//
//==============================================================================
- (void)fixInvalidRecord
{
    [cardDBController_ fixInvalidRecord];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card sync error methods


//==============================================================================
//
//==============================================================================
- (NSArray *)copyAllSyncErrorCardWithDisplayData
{
    return [cardDBController_ copyAllSyncErrorCardWithDisplayData];
}



//==============================================================================
//
//==============================================================================
- (NSArray *)copySyncErrorCardIDArray
{
    return [cardDBController_ copySyncErrorCardIDArray];
}



//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncErrorInfoWithCardID:(NSString *)cardID errorCode:(NSInteger)errorCode startSyncTime:(NSDate *)startSyncTime;
{
    return [cardDBController_ updateCardSyncErrorInfoWithCardID:cardID errorCode:errorCode startSyncTime:startSyncTime];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card image SHA1 methods


//===============================================================================
// 設定名片圖
//===============================================================================
- (BOOL)setCardImage:(CPImage *)image imageSHA1:(NSString *)imageSHA1 withCardID:(NSString *)cardID type:(WC_ImageType)type isImport:(BOOL)isImport
{
    self.lastError = nil;
    
    //////////////////////////////////////////////////
    
    BOOL result = NO;
    BOOL isExist = ([cardImageController_ hasImageWithCardID:cardID type:type] == YES);
    NSError *imageError = nil;
    
    if(image != nil)
    {
        result = [cardImageController_ setImage:image withCardID:cardID type:type error:&imageError];
    }
    else
    {
        result = [cardImageController_ removeImageWithCardID:cardID type:type];
    }
    
    if(result)
    {
        //cache
        // !! 更新影像時，需刪除暫存縮圖影像，下次使用時才會重新載入。
        WCCacheModel *cacheModel = [self cacheOfCurrentSource];
        [cacheModel removeThumbImageWithCardID:cardID];
        [cacheModel removeIDPhotoWithCardID:cardID];
        
        // !! update modified time of card
        if(!cardDBController_)
        {
            if([self loadCardDBController])
                accessMode_ |= WCDC_AM_Text;
        }
        
        if(imageSHA1 != nil)
        {
            // 同步時更新SHA1紀錄
            [cardDBController_ setImageSHA1:imageSHA1 withImageType:type cardID:cardID];
        }
        else
        {
            // 一般操作時影像變更
            [cardDBController_ updateCardSyncActionForImageChangeWithCardID:cardID imageType:type modifiedTime:nil];
        }
        
        // !! 匯入時不需要變更ModifiedTime，也不需要發送notify
        if(!isImport)
        {
            [cardDBController_ updateModifiedTimeWithCardID:cardID];
            
            WCDC_NotifyChangeType changeType;
            
            if(image != nil)
            {
                changeType = isExist ? WCDC_NCT_Update : WCDC_NCT_Add;
            }
            else
            {
                changeType = WCDC_NCT_Delete;
            }
            
            [self notifyImageChanged:changeType imageType:type cardID:cardID];
        }
    }
    else
    {
        self.lastError = imageError;
    }
    
    return result;
}



//===============================================================================
// 設定名片圖資料
//===============================================================================
- (BOOL)setCardImageData:(NSData *)imageData imageSHA1:(NSString *)imageSHA1 withCardID:(NSString *)cardID type:(WC_ImageType)type isImport:(BOOL)isImport
{
    self.lastError = nil;
    
    //////////////////////////////////////////////////
    
    BOOL result = NO;
    BOOL isExist = ([cardImageController_ hasImageWithCardID:cardID type:type] == YES);
    NSError *imageError = nil;
    
    if(imageData != nil)
    {
        result = [cardImageController_ setImageData:imageData withCardID:cardID type:type error:&imageError];
    }
    else
    {
        result = [cardImageController_ removeImageWithCardID:cardID type:type];
    }
    
    if(result)
    {
        //cache
        // !! 更新影像時，需刪除暫存縮圖影像，下次使用時才會重新載入。
        WCCacheModel *cacheModel = [self cacheOfCurrentSource];
        [cacheModel removeThumbImageWithCardID:cardID];
        [cacheModel removeIDPhotoWithCardID:cardID];
        
        
        // !! update modified time of card
        if(!cardDBController_)
        {
            if([self loadCardDBController])
                accessMode_ |= WCDC_AM_Text;
        }
        
        if(imageSHA1 != nil)
        {
            // 同步時更新SHA1紀錄
            [cardDBController_ setImageSHA1:imageSHA1 withImageType:type cardID:cardID];
        }
        else
        {
            // 一般操作時影像變更
            [cardDBController_ updateCardSyncActionForImageChangeWithCardID:cardID imageType:type modifiedTime:nil];
        }
        
        // !! 匯入時不需要變更ModifiedTime，也不需要發送notify
        if(!isImport)
        {
            [cardDBController_ updateModifiedTimeWithCardID:cardID];
            
            WCDC_NotifyChangeType changeType;
            
            if(imageData != nil)
            {
                changeType = isExist ? WCDC_NCT_Update : WCDC_NCT_Add;
            }
            else
            {
                changeType = WCDC_NCT_Delete;
            }
            
            [self notifyImageChanged:changeType imageType:type cardID:cardID];
        }
    }
    else
    {
        self.lastError = imageError;
    }
    
    return result;
}


//===============================================================================
// 設定名片圖
//===============================================================================
- (BOOL)setCardImage:(CPImage *)image imageSHA1:(NSString *)imageSHA1 withCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [self setCardImage:image imageSHA1:imageSHA1 withCardID:cardID type:type isImport:NO];
}


//===============================================================================
// 設定名片圖資料
//===============================================================================
- (BOOL)setCardImageData:(NSData *)imageData imageSHA1:(NSString *)imageSHA1 withCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [self setCardImageData:imageData imageSHA1:imageSHA1 withCardID:cardID type:type isImport:NO];
}


//==============================================================================
//
//==============================================================================
- (BOOL)isImageSha1:(NSString *)imageSha1 equalWithCardID:(NSString *)cardID  type:(WC_ImageType)type
{
    NSString *localImageSha1 = [cardDBController_ imageSha1WithCardID:cardID type:type];
 
    if ([localImageSha1 length]==0)
    {
        localImageSha1 = @"empty";
    }
    
    if ([imageSha1 compare:localImageSha1 options:NSCaseInsensitiveSearch]==NSOrderedSame)
    {
        return YES;
    }
    else
    {
        return NO;
    }
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Advance serach

//================================================================================
//
//================================================================================
- (NSMutableArray *)copyFavoritesCardSectionModelsWithAdvancedSearchModel:(WCTAdvancedSearchModel *)advanceSearchModel;
{
    NSMutableArray *allCardArray = [[cardDBController_ copyCardWithAdvancedSearchModel:advanceSearchModel inFavorites:YES getAllField:NO] autorelease];
    
    if ([allCardArray count]==0)
    {
        
        return nil;
    }
    
    
    // 我的最愛，不分section
    WCCardSectionModel *favoriteSectionModel = [[[WCCardSectionModel alloc] init] autorelease];
    favoriteSectionModel.cardArray = allCardArray;
    
    return [[NSMutableArray alloc] initWithObjects:favoriteSectionModel, nil];
    
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyCardSectionModelsWithAdvancedSearchModel:(WCTAdvancedSearchModel *)advanceSearchModel;
{
    NSMutableArray *allCardArray = [cardDBController_ copyCardWithAdvancedSearchModel:advanceSearchModel inFavorites:NO getAllField:NO];
    
    if (allCardArray==nil)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    WC_SortedByField currentSortingFields = (WC_SortedByField)[PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
    PPSIC_Mode currentSectionModes = (PPSIC_Mode)[PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode];
    
    //////////////////////////////////////////////////
    // 要針對不同排序的allCardSectionArray的處理放在下面
    NSMutableDictionary *cardSectionDict = [NSMutableDictionary dictionary];;
    
    for (WCCardModel *cardModel in allCardArray)
    {
        //////////////////////////////////////////////////
        // check if operation is cancelled
        
        @synchronized(operationQueue_)
        {
            if([operationQueue_.operations count])
            {
                NSOperation *operation = [operationQueue_.operations firstObject];
                
                if([operation isCancelled])
                {
                    //                            NSLog(@"%s %@ cancel", __func__, searchText);
                    break;
                }
            }
        }
        
        //////////////////////////////////////////////////
        // 依sectionTitle產生cardSectionModel
        NSString *sectionTitle = [cardModel displaySectionTitleWithSortByField:currentSortingFields indexingMode:currentSectionModes];
        
        // !! 先看有沒有舊的 card seciton
        WCCardSectionModel *cardSectionModel = [cardSectionDict objectForKey:sectionTitle];
        
        // 沒有舊的，產生一個新的 card seciton
        if(cardSectionModel==nil)
        {
            cardSectionModel = [[[WCCardSectionModel alloc] init] autorelease];
            cardSectionModel.title = sectionTitle;
            cardSectionModel.cardArray = [NSMutableArray array];
            
            [cardSectionDict setObject:cardSectionModel forKey:sectionTitle];
        }
        
        cardSectionModel.needResortCard = YES;
        
        [cardSectionModel.cardArray addObject:cardModel];
    }
    
    
    //////////////////////////////////////////////////
    // 轉換為card section array
    NSMutableArray        *allCardSectionArray = [[NSMutableArray alloc] init];
    [allCardSectionArray addObjectsFromArray:[cardSectionDict allValues]];
    
    // sort sections
    [WCCompareTool cardSectionArray:allCardSectionArray sortByField:currentSortingFields];
    
    [allCardArray release];
    
    return allCardSectionArray;
}


//================================================================================
// for WCTMAC
//================================================================================
- (BOOL)loadCardSectionsWithAdvancedSearchModel:(WCTAdvancedSearchModel *)advancedSearchModel
{
    self.lastError = nil;
    
    NSMutableArray *allCardArray = [[cardDBController_ copyCardWithAdvancedSearchModel:advancedSearchModel inFavorites:NO getAllField:NO] autorelease];
    
    if (cardDBController_.lastError != nil)
    {
        self.lastError = cardDBController_.lastError;
        return NO;
    }
    
    //////////////////////////////////////////////////
    
    NSMutableDictionary *cardSectionDict = [NSMutableDictionary dictionary];
    
    if(allCardArray != nil)
    {
        [cardSectionDict setObject:allCardArray forKey:PPSIC_Title_All];
    }
    
    //////////////////////////////////////////////////

    WCCacheModel *cacheModel = [WCCacheModel defaultCache];
        
    [cacheModel updateFromAllCardSectionDict:cardSectionDict];
    cacheModel.sourceID = self.sourceID;
    cacheModel.groupID = WC_GID_None;
    
    return YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (NSString *)dataDirPath
{
    NSString *dataDir = @"";
    
    //////////////////////////////////////////////////
    
#if TARGET_OS_IPHONE
    
    dataDir = @"WorldCard";
    
#elif TARGET_OS_MAC
    
    // !! macOS每個帳號要有獨立的資料，所以要根據accountGuid建立路徑。
    NSString *accountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
    
    // !! 沒有accountGuid就不用繼續了，檢查流程。
    NSAssert(([accountGuid length] > 0), @"只有accountGuid存在時可以建立資料庫！");
    
    dataDir = [NSString stringWithFormat:@"WorldCard/%@/Personal", accountGuid];
    
#endif

    return [WCToolController baseStorePathWithDirName:dataDir];
}

@end
