//
//  WCCardDBController.m
//  WorldCardMobile
//
//  Created by  Eddie on 2012/3/22.
//  Copyright 2012 Penpower. All rights reserved.
//

#import "PPMutableDictionary+NSInteger.h"

// define
#import "WCFieldDefine.h"
#import "PPSectionIndexDefine.h"
#import "WCTFieldLengthDefine.h"

// model
#import "WCGroupModel.h"
#import "WCCardModel.h"
#import "WCTGroupSyncActionModel.h"
#import "WCTCardSyncActionModel.h"
#import "WCTAccountRelationModel.h"
#import "WCSyncErrorCardModel.h"
#import "WCTSearchOptionModel.h"

// category
#import "WCFieldModel+DisplayName.h"
#import "NSString+Additions.h"
#import "NSDate+Format.h"

// controller
#import "WCTCardDBController.h"
#import "WCToolController.h"
#import "WCTCardDBCompareFunctions.h"
#import "PPLogController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#if TARGET_OS_IPHONE

#elif TARGET_OS_MAC
static NSString *const WCTCardDBController_DefaultSeperator = @"<!SEP!>";
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Basic table definition


//////////////////////////////////////////////////
// !! 注意事項
// !! 1. GroupID/FieldID 因為是循序正整數，所以需用 WCInfo/WCDeletedGroupID/WCDeletedFieldID table管理。
//       取得新ID的邏輯為:
//          a. 先檢查WCDeletedRecord，看看是否已經有不再使用的ID
//          b. 若沒有可用的ID，從WCInfo取得最後一筆ID的值，以此ID加上1，當作新ID。
//
// !! 2. 所有時間欄位都是以GMT+0時間



//////////////////////////////////////////////////
// WCInfo table definition
// (InfoKey definitions are in header file)
#define CreateTable_WCInfo	@"CREATE TABLE WCInfo \
(InfoKey TEXT UNIQUE PRIMARY KEY, \
InfoValue TEXT);"


//////////////////////////////////////////////////
// WCFavorite table definition
#define CreateIndex_WCFavorite @"CREATE INDEX idx_WCFavorite ON WCFavorite(CardID)"
#define CreateTable_WCFavorite @"CREATE TABLE WCFavorite \
(CardID TEXT UNIQUE PRIMARY KEY,\
FavoriteOrder INTEGER, \
ActionType INTEGER DEFAULT (0));"


//////////////////////////////////////////////////
// WCUnverified table definition
#define CreateTable_WCUnverified @"CREATE TABLE WCUnverified \
(CardID TEXT UNIQUE PRIMARY KEY);"


//////////////////////////////////////////////////
// WCGroup table definition
// 紀錄group名稱與ID的對應
// !! GroupID 1~99保留給系統預設群組
// !! GroupID 100代表未分類
// !! GroupID 100以後才是給使用者自訂群組使用
//
// WCT 1.3.0
//     - 加入SuperGroupGuid 記錄父群組guid
//     - 加入PinnedOrder 紀錄釘選順序（0代表未釘選）
//     - 加入Helper 記錄server建立的幫助掃描者guid（空字串代表一般類別）
//     - 尚未加入Index，效率太差時再加入。
//
#define CreateTable_WCGroup @"CREATE TABLE WCGroup \
(GroupID INTEGER UNIQUE PRIMARY KEY , \
GroupGuid TEXT, \
GroupName TEXT, \
ModifiedTime FLOAT, \
GroupOrder INTEGER, \
SuperGroupGuid TEXT, \
PinnedOrder INTEGER, \
Helper TEXT);"


//////////////////////////////////////////////////
// WCCardGroup table definition
// 紀錄CardID所屬的類別(多類別)
// !! 1. 為了方便計算數量，當名片是未分類時，寫入資料庫時會把WC_GID_Unfiled寫入WCCardGroup。
// !! 2. 從資料庫讀出來時，當名片是未分類時，要把WC_GID_Unfiled加到groupIDArray，不是groupIDArray設為nil。
//
#define CreateIndex_WCCardGroup	@"CREATE INDEX idx_WCCardGroup ON WCCardGroup(CardID,GroupID)"
#define CreateTable_WCCardGroup @"CREATE TABLE WCCardGroup (CardID TEXT, GroupID INTEGER)"


//////////////////////////////////////////////////
// WCCard table definition
// 紀錄顯示或排序需要的資料，讓app可以快速取得顯示用資料。
#define CreateIndex_WCCard	@"CREATE INDEX idx_WCCard ON WCCard(CardID)"
#define CreateTable_WCCard	@"CREATE TABLE WCCard \
(CardID TEXT UNIQUE PRIMARY KEY, \
DisplayName TEXT, \
DisplayCompany TEXT, \
DisplayJobTitle TEXT, \
DisplayAddress TEXT, \
DisplayGPS TEXT, \
Creator TEXT, \
Owner TEXT, \
Editor TEXT, \
SectionTitle TEXT, \
CreatedTime FLOAT, \
ModifiedTime FLOAT, \
FrontRecogLang INTEGER, \
BackRecogLang INTEGER);"


//////////////////////////////////////////////////
// WCCardField table definition
// 紀錄卡片欄位詳細資料
// !! 1. FiledID為程式產生的循序正整數 (>0)
// !! 2. 可搜尋的資料都放置在此
// !! 3. 讀取時依照FieldType或ValueCombinedType對應的資料形態進行轉換。
// !! 4. 欄位資料如果是dictionary，要加入搜尋/比對重複資料需使用的資料。（ValueCombinedType = 0)
//
#define CreateIndex_WCCardField	@"CREATE INDEX idx_WCCardField ON WCCardField(CardID,FieldID)"
#define CreateTable_WCCardField @"CREATE TABLE WCCardField \
(CardID TEXT, \
FieldID INTEGER, \
FieldOrder INTEGER, \
FieldSource INTEGER, \
FieldType INTEGER, \
FieldSubType1 INTEGER, \
FieldSubType2 INTEGER, \
ValueString TEXT, \
RecogRect TEXT);"


//////////////////////////////////////////////////
// WCDeletedGroupID table definition (for reuse)
#define CreateTable_WCDeletedGroupID @"CREATE TABLE WCDeletedGroupID \
(RecordID INTEGER PRIMARY KEY);"


//////////////////////////////////////////////////
// WCDeletedFieldID table definition (for reuse)
#define CreateTable_WCDeletedFieldID @"CREATE TABLE WCDeletedFieldID \
(RecordID INTEGER PRIMARY KEY);"


//////////////////////////////////////////////////
// 最近聯繫聯絡人資料
#define CreateIndex_WCRecentContacts @"CREATE INDEX idx_WCRecentContacts ON WCRecentContacts(CardID)"
#define CreateTable_RecentContacts @"CREATE TABLE WCRecentContacts \
(CardID TEXT PRIMARY KEY, \
ActionType INTEGER, \
ActionContent TEXT, \
ActionTime FLOAT);"


//////////////////////////////////////////////////
// 記錄cardID與address book recordID的mapping
#define CreateIndex_WCAddressBookMapping @"CREATE INDEX idx_WCAddressBookMapping ON WCAddressBookMapping(CardID, ABPersonID)"
#define CreateTable_AddressBookMapping @"CREATE TABLE WCAddressBookMapping \
(CardID TEXT UNIQUE PRIMARY KEY, \
ABPersonID TEXT UNIQUE,\
ABSourceID TEXT,\
ABGroupID TEXT);"





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCT additional table definition



//////////////////////////////////////////////////
/// 記錄所有帳號列表，及各帳號與自已帳號的關係
#define CreateIndex_WCTAccountRelationship @"CREATE INDEX idx_WCTAccountRelationship ON WCTAccountRelationship(AccountGUID)"
#define CreateTable_WCTAccountRelationship @"CREATE TABLE WCTAccountRelationship \
(AccountGUID TEXT UNIQUE PRIMARY KEY,\
AccountName TEXT, \
EMail TEXT, \
Relationship INTEGER, \
IsOutgoingEmployee INTEGER, \
IsLimitedAccount INTEGER);"


//////////////////////////////////////////////////
/// 記錄可檢視用戶
#define CreateIndex_WCTCardSharedAccount @"CREATE INDEX idx_WCTCardSharedAccount ON WCTCardSharedAccount(CardID)"
#define CreateTable_WCTCardSharedAccount @"CREATE TABLE WCTCardSharedAccount \
(CardID TEXT UNIQUE PRIMARY KEY,\
AccountGUIDs TEXT);"


//////////////////////////////////////////////////
/// 記錄匯入的舊式cardID, 用來比對有沒有匯入過，
#define CreateIndex_WCTImportedCardIDMapping @"CREATE INDEX idx_WCTImportedCardIDMapping ON WCTImportedCardIDMapping(CardID,ImportedCardID)"
#define CreateTable_WCTImportedCardIDMapping @"CREATE TABLE WCTImportedCardIDMapping \
(CardID TEXT UNIQUE PRIMARY KEY,\
ImportedCardID TEXT);"




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Sync table definition

//////////////////////////////////////////////////
// group動作紀錄
#define CreateIndex_WCTGroupSyncAction @"CREATE INDEX idx_WCTGroupSyncAction ON WCTGroupSyncAction(GroupGuid, ModifiedTime)"
#define CreateTable_WCTGroupSyncAction @"CREATE TABLE WCTGroupSyncAction \
(GroupGuid TEXT UNIQUE, \
ModifiedTime FLOAT, \
SyncAction INTEGER);"


//////////////////////////////////////////////////
// card動作紀錄
#define CreateIndex_WCTCardSyncAction @"CREATE INDEX idx_WCTCardSyncAction ON WCTCardSyncAction(CardID, ModifiedTime)"
#define CreateTable_WCTCardSyncAction @"CREATE TABLE WCTCardSyncAction \
(CardID TEXT UNIQUE, \
ModifiedTime FLOAT, \
SyncAction INTEGER, \
SyncState INTEGER, \
SharedAccountSha1, \
GroupSHA1 TEXT, \
ContentSHA1 TEXT, \
FrontSideSHA1 TEXT, \
BackSideSHA1 TEXT, \
IDPhotoSHA1 TEXT);"


//////////////////////////////////////////////////
// 同步失敗記錄
#define CreateIndex_WCTCardSyncErrorInfo @"CREATE INDEX idx_WCTCardSyncErrorInfo ON WCTCardSyncErrorInfo(CardID, StartSyncTime)"
#define CreateTable_WCTCardSyncErrorInfo @"CREATE TABLE WCTCardSyncErrorInfo \
(CardID TEXT UNIQUE, \
ErrorCode INTEGER DEFAULT (0), \
StartSyncTime FLOAT);"





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Custom field table

//////////////////////////////////////////////////
/// 記錄自訂欄位資料與屬性
#define CreateIndex_WCCustomFieldInfo @"CREATE INDEX idx_WCCustomFieldInfo ON WCCustomFieldInfo(CustomFieldInfoGuid)"
#define CreateTable_WCCustomFieldInfo @"CREATE TABLE WCCustomFieldInfo \
(CustomFieldInfoGuid TEXT UNIQUE PRIMARY KEY,\
Name TEXT,\
CustomFieldCategory TEXT,\
CustomFieldContentType TEXT);"


//////////////////////////////////////////////////
/// 記錄選單的guid與值
#define CreateIndex_WCCustomFieldListItem @"CREATE INDEX idx_WCCustomFieldListItem ON WCCustomFieldListItem(ItemGuid)"
#define CreateTable_WCCustomFieldListItem @"CREATE TABLE WCCustomFieldListItem \
(ItemGuid TEXT UNIQUE PRIMARY KEY,\
CustomFieldInfoGuid TEXT,\
ItemText TEXT);"


//////////////////////////////////////////////////
/// 記錄自訂欄位ID與Guid,CustomFieldInfoGuid的mappnig
#define CreateIndex_WCCustomFieldDataMapping @"CREATE INDEX idx_WCCustomFieldDataMapping ON WCCustomFieldDataMapping(FieldID,FieldGuid)"
#define CreateTable_WCCustomFieldDataMapping @"CREATE TABLE WCCustomFieldDataMapping \
(FieldID INTEGER PRIMARY KEY,\
FieldGuid TEXT,\
CustomFieldInfoGuid TEXT);"


////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTCardDBController private interface

@interface WCTCardDBController ()
@property (atomic, assign) BOOL isCancelDBAction;
@property (atomic, retain) NSDateFormatter *dateFormatter;
@property (atomic, retain) PPLogController *logController;
@property (atomic, assign) WC_GroupID mustHaveGroupID;
@end


#pragma mark -

@implementation WCTCardDBController
@synthesize isCancelDBAction = isCancelDBAction_;

////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//===============================================================================
//
//===============================================================================
+ (WCTCardDBController *)sharedWCTCardDBControllerWithDBPath:(NSString *)dbPath
{
    static WCTCardDBController *sharedWCTCardDBControllerInstance;
    
    // !! dbPath == nil代表只是要取得目前的sharedInstance，直接回傳不要干擾後面流程。
    if([dbPath length] == 0)
    {
        return sharedWCTCardDBControllerInstance;
    }
    
    //////////////////////////////////////////////////
    
    static dispatch_once_t once;
    
    dispatch_once(&once, ^ {
        
        if([NSThread isMainThread])
        {
            sharedWCTCardDBControllerInstance = [[WCTCardDBController alloc] initWithPath:dbPath];
        }
        else
        {
            dispatch_sync(dispatch_get_main_queue(), ^{
                sharedWCTCardDBControllerInstance = [[WCTCardDBController alloc] initWithPath:dbPath];
            });
        }
    });
    
    // macOS切換使用者時資料庫路徑會變更
    if([dbPath isEqualToString:sharedWCTCardDBControllerInstance.dbPath] == NO)
    {
        [sharedWCTCardDBControllerInstance resetDBPath:dbPath];
    }
    
    return sharedWCTCardDBControllerInstance;
}


//===============================================================================
//
//===============================================================================
- (id)initWithPath:(NSString *)dbPath
{
    self = [super initWithPath:dbPath];
    
    if(self)
    {
        self.mustHaveGroupID = WC_GID_Unfiled;
        
        //////////////////////////////////////////////////
        
        dirPath_ = [[dbPath stringByDeletingLastPathComponent] retain];
        
        _dateFormatter = [[NSDateFormatter alloc] init];
        [_dateFormatter setFormatterBehavior:NSDateFormatterBehavior10_4];
        [_dateFormatter setDateFormat:WCTCDBC_DateTimeFormat];
        [_dateFormatter setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
        
        //////////////////////////////////////////////////
        // 建立card log instance
        
#ifdef ENABLE_LOG
        
        self.logController = [[[PPLogController alloc] init] autorelease];
        
        if(self.logController != nil)
        {
            NSString *logDirPath = [WCTCardDBController logPath];
                        
            [self.logController setFileName:@"WCTCardDBLog" atPath:logDirPath];
            [self.logController setMask:PPLogControllerMask_Normal];
        }
#endif
        
    }
    
    return self;
}


//===============================================================================
// PARAMETERS: <in> dirPath => base store path
//             <in> dbName => database name
//===============================================================================
- (id)initWithDirPath:(NSString *)dirPath dbName:(NSString *)dbName
{
    NSString *dbPath = [NSString stringWithFormat:@"%@/%@", dirPath, dbName];
    self = [[WCTCardDBController sharedWCTCardDBControllerWithDBPath:dbPath] retain];
    
    if(self)
    {
        dirPath_ = [dirPath retain];
        isCancelDBAction_ = NO;
    }
    
    return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
#ifdef ENABLE_LOG
    self.logController = nil;
#endif
    
    self.dateFormatter = nil;
    [self disconnectDB];
    [dirPath_ release];
    
    [super dealloc];
}


//===============================================================================
// compact database
// 移除database/table fragmentation，保持資料庫效能。
// !! 一定要找機會壓縮資料庫，不然會越長越大。
//===============================================================================
- (void)compactDB
{
    @synchronized(self)
    {
        [super compactDB];
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//===============================================================================
//
//===============================================================================
- (void)setLastErrorWithErrorStep:(NSString *)errorStep
{
    self.lastError = nil;
    
    if(errorStep == nil)
    {
        errorStep = @"";
    }
    
    NSDictionary *userInfo = @{NSLocalizedFailureReasonErrorKey:errorStep,
                               WCTCDBC_UserInfoKey_DBErrorMessage:[super dbErrorMessage]};
    
    self.lastError = [NSError errorWithDomain:NSStringFromClass([self class])
                                         code:[super dbErrorCode]
                                     userInfo:userInfo];
}





#pragma mark ##### handle recordID #####


//===============================================================================
//
//===============================================================================
- (NSString *)deletedRecordTableNameWithRecordType:(WCTCDBC_RecordType)recordType
{
    switch (recordType)
    {
        case WCTCDBC_RT_Group: return @"WCDeletedGroupID";
        case WCTCDBC_RT_Field: return @"WCDeletedFieldID";
        default: break;
    }
    
    return nil;
}


//===============================================================================
//
//===============================================================================
- (NSInteger)minDeletedRecordIDWithRecordType:(WCTCDBC_RecordType)recordType
{
    @synchronized(self)
    {
        NSInteger recordID = WC_InvalidRecordID;
        NSString *tableName = [self deletedRecordTableNameWithRecordType:recordType];
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT MIN(RecordID) from %@", tableName];
        
        recordID = [super intResultWithSelectCommand:sqlCommand];
        
        return recordID;
    }
}


//===============================================================================
//
//===============================================================================
- (NSMutableArray *)copyDeletedRecordIDArrayWithRecordType:(WCTCDBC_RecordType)recordType maxNeededCount:(NSInteger)maxNeededCount
{
    @synchronized(self)
    {
        sqlite3_stmt *stmt;
        NSInteger recordID = WC_InvalidRecordID;
        NSString *recordIDString = nil;
        NSString *tableName = [self deletedRecordTableNameWithRecordType:recordType];
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT RecordID from %@ ORDER BY RecordID LIMIT %ld", tableName, (long)maxNeededCount];
        NSMutableArray *recordIDArray = [[NSMutableArray alloc] init];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                recordID = [super integerFromStmt:stmt column:0];
                recordIDString = [[NSString alloc] initWithFormat:@"%ld", (long)recordID];
                [recordIDArray addObject:recordIDString];
                
                [recordIDString release];
                recordIDString = nil;
            }
            
            sqlite3_finalize(stmt);
        }
        
        return recordIDArray;
    }
}



//===============================================================================
// get last record ID with specified record type
// RETURNED: record ID, return WC_InvalidRecordID if failed
//===============================================================================
- (NSInteger)maxUsedRecordIDWithRecordType:(WCTCDBC_RecordType)recordType
{
    @synchronized(self)
    {
        NSInteger lastRecordID = WC_InvalidRecordID;
        
        switch (recordType)
        {
            case WCTCDBC_RT_Group:
            {
                lastRecordID = [super intResultWithSelectCommand:@"SELECT max(GroupID) FROM WCGroup"];
                break;
            }
                
            case WCTCDBC_RT_Field:
            {
                lastRecordID = [super intResultWithSelectCommand:@"SELECT max(FieldID) FROM WCCardField"];
                break;
            }
            default:
                break;
        }
        
        return lastRecordID;
    }
}


//===============================================================================
// remove deleted record ID with specified record type
//===============================================================================
- (void)addDeletedRecordID:(NSInteger)recordID withRecordType:(WCTCDBC_RecordType)recordType
{
    @synchronized(self)
    {
        NSString *tableName = [self deletedRecordTableNameWithRecordType:recordType];
        NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO %@ VALUES(%ld)", tableName, (long)recordID];
        
        [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
// remove deleted record ID with specified record type
//===============================================================================
- (void)removeDeletedRecordID:(NSInteger)recordID withRecordType:(WCTCDBC_RecordType)recordType
{
    @synchronized(self)
    {
        NSString *tableName = [self deletedRecordTableNameWithRecordType:recordType];
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM %@ WHERE RecordID=%ld", tableName, (long)recordID];
        
        [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
// !! 重新產生DeletedRecordID記錄
//===============================================================================
- (BOOL)resetDeletedRecordIDWithRecordType:(WCTCDBC_RecordType)recordType
{
    @synchronized(self)
    {
        NSLog(@"!! resetDeletedRecordIDWithRecordType : shouldn't happen, check why !!");
        
        sqlite3_stmt    *stmt;
        NSString        *sqlCommand;
        NSString        *recordIDString;
        NSMutableArray  *existRecordIDArray = nil;
        NSInteger       existRecordID;
        NSInteger       lastRecordID;
        BOOL            result = YES;
        NSRange         range;
        NSInteger       index;
        NSInteger       startIndex;
        NSString        *tableName = nil;
        
        
        switch (recordType)
        {
            case WCTCDBC_RT_Group:
                [super runSqlCommand:@"DELETE FROM WCDeletedGroupID"];
                
                lastRecordID = WC_GID_Unfiled;
                startIndex = WC_GID_Unfiled+1;
                sqlCommand = [NSString stringWithFormat:@"SELECT GroupID From WCGroup WHERE GroupID>%ld ORDER BY GroupID", (long)WC_GID_Unfiled];
                tableName = @"WCDeletedGroupID";
                break;
                
            case WCTCDBC_RT_Field:
                [super runSqlCommand:@"DELETE FROM WCDeletedFieldID"];
                
                lastRecordID = 0;
                startIndex = 1;
                sqlCommand = @"SELECT DISTINCT FieldID FROM WCCardField ORDER BY FieldID";
                tableName = @"WCDeletedFieldID";
                break;
                
            default:
                return NO;
        }
        
        // 取得目前所有recordID
        existRecordIDArray = [[NSMutableArray alloc] init];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                existRecordID = [super integerFromStmt:stmt column:0];
                recordIDString = [[NSString alloc] initWithFormat:@"%ld", (long)existRecordID];
                [existRecordIDArray addObject:recordIDString];
                [recordIDString release];
                recordIDString = nil;
                
                lastRecordID = existRecordID;
            }
            
            sqlite3_finalize(stmt);
            
            
            
            // 重新取得沒有使用的recordID
            range = NSMakeRange(0, [existRecordIDArray count]);
            [super beginTransaction];
            
            for(NSInteger i=startIndex; i<lastRecordID; i++)
            {
                recordIDString = [[NSString alloc] initWithFormat:@"%ld", (long)i];
                
                index = [existRecordIDArray indexOfObject:recordIDString
                                            inSortedRange:range
                                                  options:NSBinarySearchingFirstEqual
                                          usingComparator:^(NSString *model1, NSString *model2)
                         {
                             if([model1 intValue] > [model2 intValue])
                                 return NSOrderedDescending;
                             else if([model1 intValue] < [model2 intValue])
                                 return NSOrderedAscending;
                             else return NSOrderedSame;
                         }];
                
                [recordIDString release];
                recordIDString = nil;
                
                if(index >= range.length)
                {
                    sqlCommand = [[NSString alloc] initWithFormat:@"INSERT INTO %@ VALUES(%ld)", tableName, (long)i];
                    [super runSqlCommand:sqlCommand];
                    [sqlCommand release];
                }
            }
            
            result = [super commitTransaction];
            
            if(!result)
                [self setLastErrorWithErrorStep:@"Failed to save deleted recordID"];
        }
        else
        {
            result = NO;
            [self setLastErrorWithErrorStep:@"Failed to get all groupID"];
        }
        
        [existRecordIDArray release];
        
        return result;
    }
}






#pragma mark ##### convert #####

//===============================================================================
//
//===============================================================================
- (BOOL)isCalledByMainThread
{
    return ([NSThread isMainThread]);
}


//===============================================================================
//
//===============================================================================
- (NSString *)stringFromValue:(id)value
{
    if(!value)
        return @"";
    
    if([value isKindOfClass:[NSString class]])
    {
        return (NSString *)value;
    }
    else if([value isKindOfClass:[NSNumber class]])
    {
        return [NSString stringWithFormat:@"%d", [(NSNumber *)value intValue]];
    }
    else if([value isKindOfClass:[NSDate class]])
    {
        // !!統一使用標準時間
        NSDate *GMTDate = value;
        NSString *dateString = [self.dateFormatter stringFromDate:GMTDate];
        
        if(![dateString length])
        {
            // !! add Crashlytics log to see what datetime string can't be converted
            NSAssert(NO, @"datetime string can't be converted !");
//            CLS_LOG(@"isCalledByMainThread : %d", [self isCalledByMainThread]);
//            CLS_LOG(@"GMTDate : %@", GMTDate);
//            CLS_LOG(@"---------------------");
        }
        
        return dateString;
    }
    else return @"";
}


//===============================================================================
// 各FieldType的資料形態
// NOTE: 把非string的列出來，其他直接回傳string
//===============================================================================
- (WC_ValueType)valueTypeOfField:(WCFieldModel *)fieldModel
{
    switch (fieldModel.type)
    {
        case WC_FT_Name:
        case WC_FT_Company:
        case WC_FT_Address:
        {
            if(fieldModel.subType1 != WC_FST1_None && fieldModel.subType2 == WC_FST2_None)
                return WC_VT_Dictionary;
            else break;
        }
            
        case WC_FT_Date:
            return WC_VT_DateTime;
            
        default: break;
    }
    
    // 其他都是string
    return WC_VT_String;
}


//===============================================================================
//
//===============================================================================
- (id)valueFromString:(NSString *)valueString valueType:(WC_ValueType)valueType
{
    switch (valueType)
    {
        case WC_VT_String:
            return valueString;
            
        case WC_VT_Integer:
            return [NSNumber numberWithInt:[valueString intValue]];
            
        case WC_VT_DateTime:
        {
            // !!統一使用標準時間
            NSDate *GMTDate = [self.dateFormatter dateFromString:valueString];
            
            if(GMTDate)
            {
                return GMTDate;
            }
            else
            {
                // !! 如果轉換失敗，就用目前時間代替
                return [NSDate date];
            }
            
            break;
        }
            
        case WC_VT_Dictionary:
        {
            NSMutableDictionary *dict = [[[NSMutableDictionary alloc] init] autorelease];
            return dict;
        }
            
        default:
            break;
    }
    
    return nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - search condition methods


//==============================================================================
//
//==============================================================================
- (NSString *)searchConditionWithSearchText:(NSString *)searchText searchField:(NSString *)searchField
{
    if ([searchField length]==0)
    {
        NSAssert([searchField length]==0, @"searchField應該要有值");
        return nil;
    }
    if ([searchText length]==0)
    {
        NSAssert([searchText length]==0, @"searchText應該要有值");
        return nil;
    }
    
    //////////////////////////////////////////////////
    NSMutableString *condition = [NSMutableString string];
    // 查詢原文
    [condition appendFormat:@" %@ LIKE '%%%@%%'",searchField, searchText];
    
    // !! 以日文優先，只要searchText有日文，就套用日文規則
    if ([searchText hasJapanese])
    {
        //////////////////////////////////////////////////
        // !!WCM 7.2新增，要同時搜尋平假名與片假名 規則
        // case 1. 原文與平假名與片假名結果相同，用原文找
        // case 2. 原文與平假名與片假名的其中一個不同，一個相同，用原文與不同的那一個找
        // case 3. 原文與平假名與片假名結果都不同，用原文找
        //////////////////////////////////////////////////
        
        NSString *hiragana = [searchText hiraganaString];
        NSString *katakana = [searchText katakanaString];
        
        BOOL hiraganaEqual = [hiragana isEqualToString:searchText];
        BOOL katakanaEqual = [katakana isEqualToString:searchText];
        
        if ((hiraganaEqual==NO && katakanaEqual==YES) ||
            (hiraganaEqual==YES && katakanaEqual==NO))
        {
            // 平假名與原文不同
            if (hiraganaEqual==NO)
            {
                [condition appendFormat:@" OR %@ LIKE '%%%@%%'",searchField, hiragana];
            }
            
            // 片假名與原文不同
            if (katakanaEqual==NO)
            {
                [condition appendFormat:@" OR %@ LIKE '%%%@%%'",searchField, katakana];
            }
        }
    }
    else
    {
        
        //////////////////////////////////////////////////
        // !!WCM 7.1新增，要同時搜尋繁體與簡體 規則
        // case 1. 原文與繁，簡結果相同，用原文找
        // case 2. 原文與繁，簡的其中一個不同，一個相同，用原文與不同的那一個找
        // case 3. 原文與繁，簡結果都不同，用原文找
        //////////////////////////////////////////////////
        
        NSString *chtString = [searchText traditionalChineseString];
        NSString *chsString = [searchText simplifiedChineseString];
        
        BOOL chtEqual = [chtString isEqualToString:searchText];
        BOOL chsEqual = [chsString isEqualToString:searchText];
        
        if ((chtEqual==NO && chsEqual==YES) ||
            (chtEqual==YES && chsEqual==NO))
        {
            // 繁中與原文不同
            if (chtEqual==NO)
            {
                [condition appendFormat:@" OR %@ LIKE '%%%@%%'",searchField, chtString];
            }
            
            // 簡中與原文不同
            if (chsEqual==NO)
            {
                [condition appendFormat:@" OR %@ LIKE '%%%@%%'",searchField, chsString];
            }
        }
    }
    
    
    if ([condition length]>0)
    {
        return condition;
    }
    else
    {
        return nil;
    }
}






#pragma mark - Database methods

//===============================================================================
//
//===============================================================================
- (BOOL)createDB
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSFileManager *fileManager = [NSFileManager defaultManager];
        
        
        // Check if base db exist. (create new one if not exist)
        if(![super isDBExist])
        {
            //------------------------
            // create new database
            //------------------------
            if([WCToolController createDirPath:dirPath_] && [super connectDB])
            {
                //------------------------
                // create tables
                //------------------------
                if(![super runSqlCommand:CreateTable_WCInfo])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCGroup])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCardGroup])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCard])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCardField])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCFavorite])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCUnverified])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCDeletedGroupID])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCDeletedFieldID])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_RecentContacts])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_AddressBookMapping])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTGroupSyncAction])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTCardSyncAction])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTCardSyncErrorInfo])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTAccountRelationship])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTCardSharedAccount])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCTImportedCardIDMapping])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCustomFieldInfo])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCustomFieldListItem])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateTable_WCCustomFieldDataMapping])
                    goto _EXIT;
                
                //------------------------
                // create table index
                //------------------------
                if(![super runSqlCommand:CreateIndex_WCCardGroup])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCCard])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCCardField])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCFavorite])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCRecentContacts])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCAddressBookMapping])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTGroupSyncAction])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTCardSyncAction])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTCardSyncErrorInfo])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTAccountRelationship])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTCardSharedAccount])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCTImportedCardIDMapping])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCCustomFieldInfo])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCCustomFieldListItem])
                    goto _EXIT;
                
                if(![super runSqlCommand:CreateIndex_WCCustomFieldDataMapping])
                    goto _EXIT;
                
                //------------------------
                // add initial info value
                //------------------------
                if(![self addInfoValue:WCTCDBC_CurrentVersion withKey:WCTCDBC_IK_Version])
                    goto _EXIT;
            }
        }
        
        result = YES;
        
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        if(!result)
        {
            // disconnect database
            [super disconnectDB];
            [fileManager removeItemAtPath:self.dbPath error:nil];
        }
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)connectDB
{
    @synchronized(self)
    {
        if([super connectDB])
        {
            return YES;
        }
        else
        {
            [self setLastErrorWithErrorStep:@"Faield to connect DB"];
            return NO;
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)disconnectDB
{
    @synchronized(self)
    {
        [super disconnectDB];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - check db methods

//===============================================================================
//
//===============================================================================
- (BOOL)isTableExist:(NSString *)tableName withSynchronized:(BOOL)synchronized
{
    if(synchronized)
    {
        @synchronized(self)
        {
            return [super isTableExist:tableName];
        }
    }
    else
    {
        return [super isTableExist:tableName];
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)isIndexExist:(NSString *)indexName withSynchronized:(BOOL)synchronized
{
    if(synchronized)
    {
        @synchronized(self)
        {
            return [super isIndexExist:indexName];
        }
    }
    else
    {
        return [super isIndexExist:indexName];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Info methods

//===============================================================================
// get info value with specified key
// return nil if failed
//===============================================================================
- (NSString *)infoValueWithKey:(NSString *)infoKey
{
    @synchronized(self)
    {
        sqlite3_stmt *stmt;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT InfoValue FROM WCInfo WHERE InfoKey='%@'", [super sqlString:infoKey]];
        NSString *infoValue = nil;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if (sqlite3_step(stmt) == SQLITE_ROW)
                infoValue = [super stringFromStmt:stmt column:0];
            
            sqlite3_finalize(stmt);
        }
        
        return infoValue;
    }
}


//===============================================================================
// set info value with specified key
// PARAMETERS: bTryUpdate => try update first, try insert if update failed
// RETURNED: return NO if failed
//===============================================================================
- (BOOL)addInfoValue:(NSString *)infoValue withKey:(NSString *)infoKey
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCInfo VALUES('%@','%@')", [super sqlString:infoKey], [super sqlString:infoValue]];
        BOOL result = [super runSqlCommand:sqlCommand];
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateInfoValue:(NSString *)infoValue withKey:(NSString *)infoKey
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCInfo SET InfoValue='%@' WHERE InfoKey='%@'", [super sqlString:infoValue], [super sqlString:infoKey]];
        BOOL result = [super runSqlCommand:sqlCommand];
        
        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Favorite methods



//===============================================================================
// RETURNED: mutable array of cardModel (with necessary display data only)
//           must release after using.
//===============================================================================
- (NSMutableArray *)copyAllFavorites
{
    return [self copyFavoritesWithSearchText:nil];
}


//===============================================================================
// PARAM: searchText = nil means get all
// RETURNED: WCCardModel array, must release after using
// NOTE: macOS need section title; iOS needn't
//===============================================================================
- (NSMutableArray *)copyFavoritesWithSearchText:(NSString *)searchText
{
    @synchronized(self)
    {
        NSMutableSet        *cardIDSet = [[NSMutableSet alloc] init];
        NSMutableArray		*cardArray = [[NSMutableArray alloc] init];
        WCCardModel			*cardModel;
        NSString            *cardID;
        NSMutableString		*sqlCommand = [NSMutableString string];
        sqlite3_stmt		*stmt;
        
        
        //-------------------------------------------------
        // prepare sql command
        //-------------------------------------------------
        [sqlCommand appendString:@"SELECT c.CardID,u.CardID,c.DisplayName,c.DisplayCompany,c.DisplayJobTitle,c.Creator,c.Owner,c.Editor,c.SectionTitle,c.CreatedTime,c.ModifiedTime FROM WCFavorite f"];
        [sqlCommand appendString:@" LEFT JOIN WCCard c ON c.CardID = f.CardID"];
        [sqlCommand appendString:@" LEFT JOIN WCUnverified u ON u.CardID=f.CardID"];

        
        if([searchText length])
        {
            [sqlCommand appendString:@" LEFT JOIN WCCardField cf ON cf.CardID = c.CardID"];

            [sqlCommand appendFormat:@" WHERE "];
            
            // !! 要排除自訂欄位
            [sqlCommand appendFormat:@" cf.FieldType!=%ld AND", (long)WC_FT_UserDefine];
            
            NSString *searchCondition = [self searchConditionWithSearchText:searchText searchField:@"cf.ValueString"];
            if ([searchCondition length])
            {
                [sqlCommand appendString:searchCondition];
            }
        }

        [sqlCommand appendString:@" ORDER BY f.FavoriteOrder"];

        //////////////////////////////////////////////////
        // clear cancel flag
        [self clearCancelFlag];
        
        
        //-------------------------------------------------
        // start search
        //-------------------------------------------------
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                ////////////////////////////////////
                // check if interrupt
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    [cardArray release];
                    cardArray = nil;
                    break;
                }
                
                int column = 0;
                ////////////////////////////////////
                // get result
                cardID = [super stringFromStmt:stmt column:column++];
                
                // !! 需檢察是否需要做重複判斷
                if(cardID && ![cardIDSet containsObject:cardID])
                    //if(cardID)    //andy fixbug
                {
                    
                    // add to result array
                    cardModel = [[WCCardModel alloc] init];
                    cardModel.ID = cardID;
                    
                    // !! 如果有值就是未校正
                    NSString *unverified = [super stringFromStmt:stmt column:column++];
                    
                    if([unverified length]>0)
                    {
                        cardModel.tagMask |= WC_TagMask_Unverified;
                    }
                    
                    cardModel.displayName = [super stringFromStmt:stmt column:column++];
                    cardModel.displayCompany = [super stringFromStmt:stmt column:column++];
                    cardModel.displayJobTitle = [super stringFromStmt:stmt column:column++];

                    cardModel.creator = [super stringFromStmt:stmt column:column++];
                    cardModel.owner = [super stringFromStmt:stmt column:column++];
                    cardModel.editor = [super stringFromStmt:stmt column:column++];
                    cardModel.sectionTitle = [super stringFromStmt:stmt column:column++];
                    cardModel.createdTime = [super dateFromStmt:stmt column:column++];
                    cardModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    
                    // 處理我的最愛
                    // 這邊都是我的最愛
                    cardModel.tagMask |= WC_TagMask_Favorite;
                    
                    [cardArray addObject:cardModel];
                    [cardModel release];
                    //andy fixbug
                    [cardIDSet addObject:cardID];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        
        [cardIDSet release];
        
        
        return cardArray;
    }
}


//===============================================================================
// RETURNED: order, return -1 if not found.
//===============================================================================
- (NSInteger)favoriteOrderWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT FavoriteOrder FROM WCFavorite WHERE CardID='%@'", [super sqlString:cardID]];
        NSInteger order = [super intResultWithSelectCommand:sqlCommand];
        
        return order;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)addFavoriteWithCardID:(NSString *)cardID isInTransaction:(BOOL)isInTransaction
{
    @synchronized(self)
    {
        // !! 如果cardID不存在就不新增，避免空資料
        if([self isCardIDExist:cardID withSynchronized:NO]==NO)
        {
            return NO;
        }
        
        if ([self favoriteOrderWithCardID:cardID]!=-1)
        {
            return YES;
        }
        
        BOOL result = NO;
        NSInteger lastOrder = [super recordCountInTable:@"WCFavorite"];
        
        if (!isInTransaction)
        {
            [self beginTransaction];
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCFavorite(FavoriteOrder, CardID) VALUES(%ld,'%@')",
                                (long)lastOrder,
                                [super sqlString:cardID]];
        
        if ( [super runSqlCommand:sqlCommand])
        {
            result = YES;
        }
        
        if (result)
        {
            if (!isInTransaction)
            {
                [super commitTransaction];
            }
        }
        else
        {
            if (!isInTransaction)
            {
                [super rollbackTransaction];
            }
        }
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)insertFavoriteWithCardID:(NSString *)cardID toOrder:(NSInteger)toOrder
{
    @synchronized(self)
    {
        NSString *sqlCommand = nil;
        BOOL result = NO;
        
        
        [super beginTransaction];
        
        // update current order
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCFavorite SET FavoriteOrder=FavoriteOrder+1 WHERE FavoriteOrder>=%ld", (long)toOrder];
        
        if(![super runSqlCommand:sqlCommand])
            goto _EXIT;
        
        sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCFavorite(CardID, FavoriteOrder) VALUES('%@', %ld)",
                      [super sqlString:cardID],
                      (long)toOrder];
        
        if(![super runSqlCommand:sqlCommand])
            goto _EXIT;
 
        result = YES;
        
        
    _EXIT:
        if(result)
        {
            result = [super commitTransaction];
        }
        else
        {
            [super rollbackTransaction];
        }
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)removeFavoriteWithCardID:(NSString *)cardID isInTransaction:(BOOL)isInTransaction
{
    @synchronized(self)
    {
        NSString *sqlCommand = nil;
        BOOL result = NO;
        NSInteger order = [self favoriteOrderWithCardID:cardID];
        
        
        if(order < 0)
            return NO;
        
        if (!isInTransaction)
        {
            [super beginTransaction];
        }
        
        // delete current
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCFavorite WHERE CardID='%@'", [super sqlString:cardID]];
        
        if(![super runSqlCommand:sqlCommand])
            goto _EXIT;
        
        // reset order
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCFavorite SET FavoriteOrder=FavoriteOrder-1 WHERE FavoriteOrder>%ld", (long)order];
        
        if(![super runSqlCommand:sqlCommand])
            goto _EXIT;

        result = YES;
        
        
    _EXIT:
        
        
        if(result)
        {
            if (!isInTransaction)
            {
                result = [super commitTransaction];
            }
        }
        else
        {
            if (!isInTransaction)
            {
                [super rollbackTransaction];
            }
        }
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)moveFavoriteWithCardID:(NSString *)cardID toOrder:(NSInteger)toOrder
{
    // !! 因為會刪除我的最愛再新增，所以要先把目前的actionType紀錄下來，再設定回去，不然會出錯
    NSInteger actionType = [self actionTypeWithCardID:cardID];
    
    if(![self removeFavoriteWithCardID:cardID isInTransaction:NO])
        return NO;
    
    if([self insertFavoriteWithCardID:cardID toOrder:toOrder])
    {
        [self setActionType:actionType withCardID:cardID];
        return YES;
    }
    else
    {
        return NO;
    }
}


//===============================================================================
//
//===============================================================================
- (void)setActionType:(NSInteger)actionType withCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCFavorite SET ActionType = %ld WHERE CardID ='%@'",
                                (long)actionType, [super sqlString:cardID]];
        [self runSqlCommand:sqlCommand];
    }
}



//===============================================================================
//
//===============================================================================
- (NSInteger)actionTypeWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT ActionType FROM WCFavorite WHERE CardID ='%@'",
                                 [super sqlString:cardID]];
        return [super intResultWithSelectCommand:sqlCommand];
    }
}


//================================================================================
//
//================================================================================
- (NSInteger)favoriteCount
{
    @synchronized(self)
    {
        NSString *sqlCommand = @"SELECT COUNT(*) FROM WCFavorite";
        NSInteger count = [super intResultWithSelectCommand:sqlCommand];
        
        return count;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Unverified methods

//===============================================================================
//
//===============================================================================
- (NSMutableArray *)copyAllUnverified
{
    return [self copyUnverifiedWithSearchText:nil];
}


//===============================================================================
// PARAM: searchText = nil means get all
// RETURNED: WCCardModel array, must release after using
// NOTE: macOS need section title; iOS needn't
//===============================================================================
- (NSMutableArray *)copyUnverifiedWithSearchText:(NSString *)searchText
{
    @synchronized(self)
    {
        NSMutableSet        *cardIDSet = [[NSMutableSet alloc] init];
        NSMutableArray		*cardArray = [[NSMutableArray alloc] init];
        WCCardModel			*cardModel;
        NSString            *cardID;
        NSMutableString		*sqlCommand = [NSMutableString string];
        sqlite3_stmt		*stmt;
        
        
        //-------------------------------------------------
        // prepare sql command
        //-------------------------------------------------
        [sqlCommand appendString:@"SELECT c.CardID,fav.FavoriteOrder,c.DisplayName,c.DisplayCompany,c.DisplayJobTitle,c.Creator,c.Owner,c.Editor,c.SectionTitle,c.CreatedTime,c.ModifiedTime FROM WCUnverified u"];
        [sqlCommand appendString:@" LEFT JOIN WCCard c ON c.CardID = u.CardID"];
        [sqlCommand appendString:@" LEFT JOIN WCFavorite fav ON fav.CardID=u.CardID"];

        if([searchText length] > 0)
        {
            [sqlCommand appendString:@" LEFT JOIN WCCardField cf ON cf.CardID = c.CardID"];
           
            [sqlCommand appendFormat:@" WHERE "];
            
            // !! 要排除自訂欄位
            [sqlCommand appendFormat:@" cf.FieldType!=%ld AND", (long)WC_FT_UserDefine];
            
            NSString *searchCondition = [self searchConditionWithSearchText:searchText searchField:@"cf.ValueString"];
            if ([searchCondition length])
            {
                [sqlCommand appendString:searchCondition];
            }
        }
        
        //////////////////////////////////////////////////
        // clear cancel flag
        [self clearCancelFlag];
        
        
        //-------------------------------------------------
        // start search
        //-------------------------------------------------
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                ////////////////////////////////////
                // check if interrupt
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    [cardArray release];
                    cardArray = nil;
                    break;
                }
                
                int column = 0;
                ////////////////////////////////////
                // get result
                cardID = [super stringFromStmt:stmt column:column++];
                
                // !! 需檢察是否需要做重複判斷
                if(cardID && ![cardIDSet containsObject:cardID])
                    //if(cardID)    //andy fixbug
                {
                    
                    // add to result array
                    cardModel = [[WCCardModel alloc] init];
                    cardModel.ID = cardID;
                    
                    
                    // !! 如果有值就是我的最愛
                    NSString *favorite = [super stringFromStmt:stmt column:column++];
                    
                    if ([favorite length]>0)
                    {
                        cardModel.tagMask |= WC_TagMask_Favorite;
                    }
                    
                    cardModel.displayName = [super stringFromStmt:stmt column:column++];
                    cardModel.displayCompany = [super stringFromStmt:stmt column:column++];
                    cardModel.displayJobTitle = [super stringFromStmt:stmt column:column++];
                    
                    cardModel.creator = [super stringFromStmt:stmt column:column++];
                    cardModel.owner = [super stringFromStmt:stmt column:column++];
                    cardModel.editor = [super stringFromStmt:stmt column:column++];
                    cardModel.sectionTitle = [super stringFromStmt:stmt column:column++];
                    cardModel.createdTime = [super dateFromStmt:stmt column:column++];
                    cardModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    
                    //////////////////////////////////////////////////
                    // 處理未校正
                    // 這邊抓到的都是未校正
                    cardModel.tagMask |= WC_TagMask_Unverified;
                    
                    
                    [cardArray addObject:cardModel];
                    [cardModel release];
                    //andy fixbug
                    [cardIDSet addObject:cardID];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        
        [cardIDSet release];
        
        
        return cardArray;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)addUnverifiedWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        // !! 如果cardID不存在就不新增，避免空資料
        if([self isCardIDExist:cardID withSynchronized:NO]==NO)
        {
            return NO;
        }
        
        if ([self isUnverifiedWithCardID:cardID]==YES)
        {
            return YES;
        }
        
        BOOL result = NO;
        NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCUnverified VALUES('%@')", [super sqlString:cardID]];
        
        if ([super runSqlCommand:sqlCommand])
        {
            result = YES;
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeUnverifiedWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCUnverified WHERE CardID='%@'", [super sqlString:cardID]];
        
        if ([super runSqlCommand:sqlCommand])
        {
            result = YES;
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)isUnverifiedWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt		*stmt;
        BOOL result = NO;
        NSString *resultCardID = nil;
        
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCUnverified WHERE CardID='%@'", [super sqlString:cardID]];
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                resultCardID = [super stringFromStmt:stmt column:0];
            }
            
            sqlite3_finalize(stmt);
        }
        if([resultCardID length]>0)
            result = YES;
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (NSInteger)unverifiedCount
{
    @synchronized(self)
    {
        NSString *sqlCommand = @"SELECT COUNT(*) FROM WCUnverified";
        NSInteger count = [super intResultWithSelectCommand:sqlCommand];
        
        return count;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group methods

//===============================================================================
// copy all groups
// RETURNED: mutable array of WCGroupModel, must release after using.
//===============================================================================
- (NSMutableArray *)copyAllGroups
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString		*utf8Command;
        WCGroupModel	*groupModel;
        NSMutableArray	*groupArray = [[NSMutableArray alloc] init];
        
        utf8Command = [NSString stringWithFormat:@"SELECT GroupID,GroupGuid,GroupName,ModifiedTime,GroupOrder,SuperGroupGuid,PinnedOrder,Helper FROM WCGroup ORDER BY GroupID"];
        
        self.lastError = nil;
        
        if(groupArray == nil)
        {
            [self setLastErrorWithErrorStep:@"copyAllGroups : Faield to alloc group array"];
            return nil;
        }
        
        if(sqlite3_prepare_v2(self.dbHandle, [utf8Command UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            // get group data
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                
                if((groupModel = [[WCGroupModel alloc] init]))
                {
                    groupModel.ID = [super integerFromStmt:stmt column:column++];
                    groupModel.guid = [super stringFromStmt:stmt column:column++];
                    groupModel.name = [super stringFromStmt:stmt column:column++];
                    groupModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    groupModel.order = [super integerFromStmt:stmt column:column++];
                    groupModel.superGroupGuid = [super stringFromStmt:stmt column:column++];
                    groupModel.pinnedOrder = [super integerFromStmt:stmt column:column++];
                    groupModel.helper = [super stringFromStmt:stmt column:column++];
                    
                    [groupArray addObject:groupModel];
                    [groupModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyAllGroups : Failed to select group in DB"];
            [groupArray release];
            return nil;
        }
        
        if([groupArray count] == 0)
        {
            [groupArray release];
            groupArray = nil;
        }
        
        return groupArray;
    }
}


//===============================================================================
// add group with specified name and ID
//===============================================================================
- (BOOL)addGroupWithName:(NSString *)groupName ID:(NSInteger)groupID guid:(NSString *)guid superGroupGuid:(NSString *)superGroupGuid helper:(NSString *)helper
{
    @synchronized(self)
    {
        BOOL result = NO;
        do
        {
            BOOL isUseDeletedRecordID = YES;
            
            if([groupName length]==0)
            {
                [self setLastErrorWithErrorStep:@"addGroupWithName : no groupName"];
                break;
            }

            if([guid length]==0)
            {
                [self setLastErrorWithErrorStep:@"addGroupWithName : no guid"];
                break;
            }

            if(groupID == WC_GID_None)
            {
                // get new groupID
                if((groupID = (WC_GroupID)[self minDeletedRecordIDWithRecordType:WCTCDBC_RT_Group]) > WC_GID_Unfiled)
                {
                    isUseDeletedRecordID = YES;
                }
                else
                {
                    groupID = (WC_GroupID)[self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Group]+1;
                    isUseDeletedRecordID = NO;
                }
            }
            
            // 沒有值時會回傳-1，+1後剛好從0開始
            NSInteger maxOrder = [self intResultWithSelectCommand:@"SELECT max(GroupOrder) FROM WCGroup"];
            
            NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCGroup VALUES(%td,'%@','%@',%f,%td,'%@',0,'%@')", groupID, guid, [super sqlString:groupName], [[NSDate date] timeIntervalSince1970], (maxOrder+1), [super sqlString:superGroupGuid], [super sqlString:helper]];
            
            result = [super runSqlCommand:sqlCommand];
            
            if (result)
            {
                // new groupID is from deleted recordID
                if(isUseDeletedRecordID)
                {
                    [self removeDeletedRecordID:groupID withRecordType:WCTCDBC_RT_Group];
                }
            }
            
        }
        while(0);
        
        return result;
    }
}


//===============================================================================
// add custom group with specified name
// RETURNED: new groupID, return WC_GID_None if failed.
// !! group紀錄的modifiedTime要和syncAction紀錄的一致
//===============================================================================
- (WC_GroupID)addGroupWithName:(NSString *)groupName syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    @synchronized(self)
    {
        BOOL		isUseDeletedRecordID = YES;
        WC_GroupID	groupID = WC_GID_None;
        NSString	*sqlCommand = nil;
        BOOL        hasRetry = NO;
        BOOL        result = NO;
        
        
        self.lastError = nil;
        
        if([groupName length]==0)
        {
            [self setLastErrorWithErrorStep:@"addGroupWithName : no groupName"];
            return WC_GID_None;
        }

        if(syncActionModel==nil)
        {
            [self setLastErrorWithErrorStep:@"addGroupWithName : no syncActionModel"];
            return WC_GID_None;
        }
        
        //////////////////////////////////////////////////

        while (true)
        {
            // get new groupID
            if((groupID = (WC_GroupID)[self minDeletedRecordIDWithRecordType:WCTCDBC_RT_Group]) > WC_GID_Unfiled)
            {
                isUseDeletedRecordID = YES;
            }
            else
            {
                groupID = (WC_GroupID)[self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Group]+1;
                isUseDeletedRecordID = NO;
            }

            //////////////////////////////////////////////////
            
            //test : 模擬已經groupID已經存在的狀況
            //        if(!hasRetry)
            //            groupID = 101;

            //////////////////////////////////////////////////

            // create group
            if(groupID > WC_GID_Unfiled)
            {
                // 沒有值時會回傳-1，+1後剛好從0開始
                NSInteger maxOrder = [self intResultWithSelectCommand:@"SELECT max(GroupOrder) FROM WCGroup"];
                
                //////////////////////////////////////////////////
                
                [super beginTransaction];
                
                if ([groupName length]>WCT_FML_Group)
                {
                    groupName = [groupName substringWithRange:NSMakeRange(0, WCT_FML_Group)];
                }
                
                sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCGroup VALUES(%td,'%@','%@',%f,%td,'',0,'')",
                              groupID,
                              [super sqlString:syncActionModel.dataGuid],
                              [super sqlString:groupName],
                              [syncActionModel.modifiedTime timeIntervalSince1970],
                              (maxOrder+1)];
                
                [super runSqlCommand:sqlCommand];
                [self updateGroupSyncAction:syncActionModel];
                result = [self commitTransaction];
            }
            
            //////////////////////////////////////////////////

            if(result)
            {
                // new groupID is from deleted recordID
                if(isUseDeletedRecordID)
                {
                    [self removeDeletedRecordID:groupID withRecordType:WCTCDBC_RT_Group];
                }
                
                break;
            }
            else // 錯誤處理
            {
                // !! 已經重試過就跳出
                if(!hasRetry && [super dbErrorCode] == SQLITE_CONSTRAINT)
                {
                    // !! 檢查是否groupID已經存在
                    NSString *existGroupName = [self groupNameWithID:groupID];
                    
                    // !! groupID已存在，重新設定deleted groupID
                    if([existGroupName length])
                    {
                        [self resetDeletedRecordIDWithRecordType:WCTCDBC_RT_Group];
                    }
                    
                    hasRetry = YES;
                }
                else
                {
                    [self setLastErrorWithErrorStep:@"addGroupWithName : insert failed"];
                    break;
                }
            }
        }
        
        return groupID;
    }
}


//===============================================================================
// delete group with specified ID
//===============================================================================
- (BOOL)removeGroupWithID:(WC_GroupID)groupID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    @synchronized(self)
    {
        BOOL            result = NO;
        NSString        *sqlCommand;
        sqlite3_stmt	*stmt;
        NSMutableSet    *cardIDSet = nil;
        NSString        *cardID = nil;
        NSArray         *cardIDArray = nil;
        

        self.lastError = nil;
        
        if(groupID <= WC_GID_Unfiled)
        {
            [self setLastErrorWithErrorStep:@"removeGroupWithID : can't remove system group"];
            return NO;
        }
        
        
        //////////////////////////////////////////////////
        // 取出要刪除的group中的所有cardID
        
        sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCCardGroup WHERE GroupID=%ld", (long)groupID];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                cardID = [super stringFromStmt:stmt column:0];
                if([cardID length])
                {
                    if(!cardIDSet)
                    {
                        cardIDSet = [[NSMutableSet alloc] init];
                    }
                    
                    [cardIDSet addObject:cardID];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"removeGroupWithID : failed to get cardIDs in group"];
            return NO;
        }
        
        
        //////////////////////////////////////////////////
        // 刪除包含其他group的cardID (WC_GID_Unverified除外)
        // 剩下的就是最後要加到未分類的cardID
        
        if([cardIDSet count])
        {
            NSInteger  curGroupID;
            
            cardIDArray = [[NSArray alloc] initWithArray:[cardIDSet allObjects]];
            
            for(cardID in cardIDArray)
            {
                sqlCommand = [NSString stringWithFormat:@"SELECT GroupID FROM WCCardGroup WHERE CardID='%@'", [super sqlString:cardID]];
                
                if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
                {
                    while (sqlite3_step(stmt) == SQLITE_ROW)
                    {
                        curGroupID = [super integerFromStmt:stmt column:0];
                        
                        if(curGroupID != groupID && curGroupID != WC_GID_Unverified)
                        {
                            [cardIDSet removeObject:cardID];
                            break;
                        }
                    }
                    
                    sqlite3_finalize(stmt);
                }
                else
                {
                    [self setLastErrorWithErrorStep:@"removeGroupWithID : failed to get GroupID in WCCardGroup"];
                }
            }
            
            [cardIDArray release];
        }
        
        
        //////////////////////////////////////////////////
        // delete from WCGroup table
        
        [self beginTransaction];
        
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCGroup WHERE GroupID=%ld", (long)groupID];
        
        [super runSqlCommand:sqlCommand];

        // delete from WCCardGroup table
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCardGroup WHERE GroupID=%ld", (long)groupID];
        
        [super runSqlCommand:sqlCommand];
            
        // add card to unfiled
        if([cardIDSet count])
        {
            cardIDArray = [cardIDSet allObjects];
            
            for(cardID in cardIDArray)
            {
                sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCardGroup(CardID,GroupID) VALUES('%@',%td)",
                              [super sqlString:cardID],
                              self.mustHaveGroupID];
                [super runSqlCommand:sqlCommand];
            }
        }
        
        // keep deleted groupID
        [self addDeletedRecordID:groupID withRecordType:WCTCDBC_RT_Group];
        
        //////////////////////////////////////////////////
        
        [self updateGroupSyncAction:syncActionModel];
        result = [super commitTransaction];

        
        //////////////////////////////////////////////////
        // finish
        
        if(result == YES)
        {
            // !! 把多餘的deleted GroupID清除
            NSString *tableName = [self deletedRecordTableNameWithRecordType:WCTCDBC_RT_Group];
            NSInteger maxGroupID = [self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Group];
            sqlCommand = [NSString stringWithFormat:@"DELETE FROM %@ WHERE RecordID>=%ld", tableName, (long)maxGroupID];
            [super runSqlCommand:sqlCommand];
        }
        else
        {
            [self setLastErrorWithErrorStep:@"removeGroupWithID : failed to commit commands"];
        }
        
        [cardIDSet release];
        
        return result;
    }
}


//===============================================================================
// update group name with specified ID
//===============================================================================
- (BOOL)updateGroupName:(NSString *)groupName withID:(NSInteger)groupID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    @synchronized(self)
    {
        if(groupID <= WC_GID_Unfiled)
        {
            return NO;
        }

        // 130以後group不用同步
//        if(syncActionModel == nil)
//        {
//            [self setLastErrorWithErrorStep:@"updateGroupName : no syncActionModel"];
//            return WC_GID_None;
//        }

        //////////////////////////////////////////////////
        
        [super beginTransaction];

        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET GroupName='%@',ModifiedTime=%f WHERE GroupID=%ld",
                                [super sqlString:groupName],
                                [syncActionModel.modifiedTime timeIntervalSince1970],
                                (long)groupID];
        
        [super runSqlCommand:sqlCommand];
        
        //////////////////////////////////////////////////
        
        [self updateGroupSyncAction:syncActionModel];

        //////////////////////////////////////////////////

        return [super commitTransaction];
    }
}


//===============================================================================
//
//===============================================================================
- (NSString *)groupNameWithID:(NSInteger)groupID
{
    @synchronized(self)
    {
        NSString *groupName = nil;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupName FROM WCGroup WHERE GroupID=%ld", (long)groupID];
        sqlite3_stmt *stmt;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                groupName = [super stringFromStmt:stmt column:0];
                break;
            }
            
            sqlite3_finalize(stmt);
        }
        
        return groupName;
    }
}


//===============================================================================
//
//===============================================================================
- (NSDate *)groupModifiedTimeWithID:(NSInteger)groupID
{
    @synchronized(self)
    {
        NSString		*sqlCommand = [NSString stringWithFormat:@"SELECT ModifiedTime FROM WCGroup WHERE GroupID=%ld", (long)groupID];
        sqlite3_stmt	*stmt;
        NSDate          *modifiedTime = nil;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                modifiedTime = [self dateFromStmt:stmt column:0];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return modifiedTime;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateOrderWithGroupIDArray:(NSArray<NSString *> *)groupIDArray
{
    @synchronized(self)
    {
        BOOL result = NO;
        
        if ([groupIDArray count]==0)
        {
            [self setLastErrorWithErrorStep:@"updateOrderWithGroupIDArray : groupIDArray is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////

        NSInteger groupOrder = 1;   //order要從1開始
        NSDate *modifiedTime = [NSDate date];
        
        for (NSString *groupIDString in groupIDArray)
        {
            result = NO;
            
            WC_GroupID groupID = [groupIDString integerValue];
            
            [super beginTransaction];
            
            NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET GroupOrder=%ld,ModifiedTime=%f WHERE GroupID=%ld",
                                    (long)groupOrder,
                                    [modifiedTime timeIntervalSince1970],
                                    (long)groupID];
            [super runSqlCommand:sqlCommand];
            
            //////////////////////////////////////////////////
            
            result = [super commitTransaction];
            
            if(result == NO)
            {
                break;
            }
            else
            {
                groupOrder++;
            }
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)groupGuidWithID:(WC_GroupID)groupID
{
    @synchronized(self)
    {
        if (groupID==WC_GID_None)
        {
            return nil;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupGuid FROM WCGroup WHERE GroupID=%td", groupID];
        return [self stringResultWithSelectCommand:sqlCommand];
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)groupHelperWithID:(WC_GroupID)groupID
{
    @synchronized(self)
    {
        if (groupID==WC_GID_None)
        {
            return nil;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT Helper FROM WCGroup WHERE GroupID=%td", groupID];
        return [self stringResultWithSelectCommand:sqlCommand];
    }
}


//================================================================================
//
//================================================================================
- (void)setMustHaveGroupIDAfterRemoveGroup:(WC_GroupID)groupID
{
    if(groupID == WC_GID_None)
    {
        self.mustHaveGroupID = WC_GID_Unfiled;
    }
    else
    {
        self.mustHaveGroupID = groupID;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card-Group methods

//===============================================================================
// private
//===============================================================================
- (BOOL)removeCardGroupIDsWithCardID:(NSString *)cardID
{
    return [self removeCardGroupIDsWithCardID:cardID syncActionModel:nil];
}


//===============================================================================
// private
//===============================================================================
- (BOOL)removeCardGroupIDsWithCardID:(NSString *)cardID syncActionModel:(WCTGroupSyncActionModel *)syncActionModel
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCardGroup WHERE CardID='%@'", [super sqlString:cardID]];
        
        BOOL result = [super runSqlCommand:sqlCommand];
        
        if(result)
        {
            /// 更新WCTGroupSyncActionModel
            [self updateGroupSyncAction:syncActionModel];
        }
        return result;
    }
}


//===============================================================================
// PARAMETER: groupIDArray不可為nil，若是未分類請加入WC_GID_Unfiled。
//===============================================================================
- (BOOL)setCardGroupsWithCardID:(NSString *)cardID groupIDArray:(NSArray *)groupIDArray isInTransaction:(BOOL)isInTransaction updateModifiedTIme:(BOOL)updateModifiedTIme
{
    @synchronized(self)
    {
        BOOL            result = NO;
        NSString        *sqlCommand;
        NSString        *errorStep = nil;
        NSMutableArray  *saveGroupIDArray = [[NSMutableArray alloc] init];
        
        
        self.lastError = nil;
        
        if(!isInTransaction)
            [super beginTransaction];
        
        //--------------------------------
        // check if group exist
        //--------------------------------
        for (NSString *groupID in groupIDArray)
        {
            // 如果group存在，才要加入
            if ([self groupNameWithID:[groupID integerValue]])
            {
                [saveGroupIDArray addObject:groupID];
            }
        }
        
        //--------------------------------
        // delete original groups
        //--------------------------------
        if(![self removeCardGroupIDsWithCardID:cardID])
        {
            errorStep = @"setCardGroupsWithCardID : delete from WCCardGroup failed";
            goto _EXIT;
        }
        
        
        //--------------------------------
        // set new groups
        // !! 完全無group資料時，設定為未分類。
        //--------------------------------
        if([saveGroupIDArray count]==0)
        {
            [saveGroupIDArray addObject:[NSString stringWithInteger:WC_GID_Unfiled]];
        }
        
        //////////////////////////////////////////////////
        for(NSString *groupIDString in saveGroupIDArray)
        {
            NSInteger groupID = [groupIDString intValue];
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCardGroup(CardID,GroupID) VALUES('%@',%ld)",
                          [super sqlString:cardID],
                          (long)groupID];
            
            if(![super runSqlCommand:sqlCommand])
            {
                errorStep = @"setCardGroupsWithCardID : insert into WCCardGroup failed";
                goto _EXIT;
            }
        }
        
        
        //--------------------------------
        // update last modified time
        // !! import時不需要更新modified time
        //--------------------------------
        if(updateModifiedTIme)
            [self updateModifiedTimeWithCardID:cardID withSynchronized:NO];
        
        
        //--------------------------------
        // finish with success
        //--------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        [saveGroupIDArray release];
        
        if(!isInTransaction)
        {
            if(result)
            {
                if(!(result = [super commitTransaction]))
                {
                    errorStep = @"setCardGroupsWithCardID : setCardGroups commit failed";
                }
            }
            else
            {
                [super rollbackTransaction];
            }
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (NSInteger)cardCountOfGroup:(NSInteger)groupID
{
    @synchronized(self)
    {
        NSString *sqlCommand = nil;
        NSInteger cardCount = 0;
        
        if(groupID == WC_GID_All)
            sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard"];
        else if(groupID == WC_GID_Unverified)
            sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCardGroup WHERE GroupID=%ld", (long)groupID];
        else sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCardGroup WHERE GroupID=%ld", (long)groupID];
        
        cardCount = [super intResultWithSelectCommand:sqlCommand];
        
        return (cardCount == -1) ? 0 : cardCount;
    }
}


//===============================================================================
// get all groupID of one card
// NOTE: 當名片是未分類時，要把WC_GID_Unfiled加到groupIDArray，不是groupIDArray設為nil。
//===============================================================================
- (NSMutableArray *)copyGroupIDArrayWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString        *sqlCommand = [NSString stringWithFormat:@"SELECT GroupID FROM WCCardGroup WHERE CardID='%@'", [super sqlString:cardID]];
        NSInteger       groupID;
        NSMutableArray  *groupIDArray = [[NSMutableArray alloc] init];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                // !! 當名片是未分類時，要把WC_GID_Unfiled加到groupIDArray，不是groupIDArray設為nil。
                groupID = [super integerFromStmt:stmt column:0];
                [groupIDArray addObject:[NSString stringWithInteger:groupID]];
            }
            
            sqlite3_finalize(stmt);
        }
        
        if(![groupIDArray count])
        {
            [groupIDArray release];
            groupIDArray = nil;
        }
        
        return groupIDArray;
    }
}


//===============================================================================
// copy all cardID of one group
//===============================================================================
- (NSMutableArray *)copyCardIDArrayWithGroupID:(NSInteger)groupID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString		*sqlCommand = nil;
        NSString        *cardID;
        NSMutableArray	*cardIDArray = [[NSMutableArray alloc] init];
        
        
        self.lastError = nil;
        
        if(groupID == WC_GID_All)
            sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCCard"];
        else sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCCardGroup WHERE GroupID=%ld", (long)groupID];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                
                ////////////////////////////////////
                // check if interrupt
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    [cardIDArray release];
                    cardIDArray = nil;
                    sqlite3_finalize(stmt);
                    return nil;
                }
                
                if((cardID = [super stringFromStmt:stmt column:0]))
                    [cardIDArray addObject:cardID];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyCardIDArrayWithGroupID : failed to select cardID in WCCard"];
            [cardIDArray release];
            return nil;
        }
        
        if(![cardIDArray count])
        {
            [cardIDArray release];
            cardIDArray = nil;
        }
        
        return cardIDArray;
    }
}


//==============================================================================
//
//==============================================================================
- (NSInteger)cardCountForGoogleGroup
{
    @synchronized(self)
    {
        NSInteger totalCardCount = 0;
        for (WC_GroupID groupID=WC_GID_Google_MyContacts; groupID<=WC_GID_Google_Coworkers; groupID++)
        {
            NSString *sqlCommand = [NSString stringWithFormat:@"SELECT count(*) FROM WCCardGroup WHERE GroupID = %ld", (long)groupID];
            NSInteger cardCount = [self intResultWithSelectCommand:sqlCommand];
            if (cardCount!=-1)
            {
                totalCardCount += cardCount;
            }
        }
        
        return totalCardCount;
    }
}



#pragma mark - Field methods (private)
/*
 //===============================================================================
 //
 //===============================================================================
 - (NSMutableArray *)copyCombinedValueStringArrayFromDictValue:(NSMutableDictionary *)dictValue fieldType:(WC_FieldType)fieldType
 {
 NSMutableArray *combinedValueStringArray = [[NSMutableArray alloc] init];
 NSInteger combinedType;
 id combinedValue;
 
 for(int i=0; i<DB_MaxCombineCount; i++)
 {
 combinedType = i+1;
 combinedValue = [dictValue objectForKey:[NSString stringWithFormat:@"%d", combinedType]];
 [combinedValueStringArray addObject:[self stringFromValue:combinedValue]];
 }
 
 return combinedValueStringArray;
 }
 
 
 //===============================================================================
 //
 //===============================================================================
 - (NSMutableDictionary *)copyDictValueFromCombinedValueStringArray:(NSMutableArray *)combinedValueStringArray fieldType:(WC_FieldType)fieldType
 {
 NSMutableDictionary *dictValue = [[NSMutableDictionary alloc] init];
 NSString *combinedValueString;
 NSInteger combinedType;
 WC_ValueType valueType;
 
 for(int i=0; i<DB_MaxCombineCount; i++)
 {
 combinedType = i+1;
 valueType = [self valueTypeOfFieldType:fieldType combinedType:combinedType];
 combinedValueString = [combinedValueStringArray objectAtIndex:i];
 
 if([combinedValueString length])
 {
 [dictValue setObject:[self valueFromString:combinedValueString valueType:valueType]
 forKey:[NSString stringWithFormat:@"%d", combinedType]];
 }
 }
 
 return dictValue;
 }
 */

//===============================================================================
//
//===============================================================================
- (BOOL)insertFieldWithCardID:(NSString *)cardID fieldModel:(WCFieldModel *)fieldModel
{
    @synchronized(self)
    {
        BOOL result;
        NSString *sqlCommand;
        NSString *valueString = [self stringFromValue:fieldModel.value];
        NSString *rectString;
        
        
        //--------------------------------
        // get recog rect string
        //--------------------------------
        if(CGRectIsEmpty(fieldModel.recogRect))
            rectString = @"";
        else rectString = CPStringFromRect(fieldModel.recogRect);
        
        
        //--------------------------------
        // run command
        //--------------------------------
        sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCardField(CardID,FieldID,FieldOrder,FieldSource,FieldType,FieldSubType1,FieldSubType2,ValueString,RecogRect) VALUES('%@',%ld,%ld,%ld,%ld,%ld,%ld,'%@','%@')",
                      [super sqlString:cardID],
                      (long)fieldModel.ID,
                      (long)fieldModel.order,
                      (long)fieldModel.source,
                      (long)fieldModel.type,
                      (long)fieldModel.subType1,
                      (long)fieldModel.subType2,
                      [super sqlString:valueString],
                      [super sqlString:rectString]];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(result)
        {
            if([fieldModel.guid length]!=0 &&
               [fieldModel.customFieldInfoGuid length]!=0)
            {
                //--------------------------------
                // 處理自訂欄位資料
                //--------------------------------
                sqlCommand = [NSString stringWithFormat:@"SELECT count(*) FROM WCCustomFieldDataMapping WHERE FieldID=%ld",
                              (long)fieldModel.ID];
                NSInteger count = [self intResultWithSelectCommand:sqlCommand];
                
                if(count>0)
                {
                    sqlCommand = [NSString stringWithFormat:@"UPDATE WCCustomFieldDataMapping SET FieldGuid= '%@', CustomFieldInfoGuid='%@' WHERE FieldID=%ld",
                                  [super sqlString:fieldModel.guid],
                                  [super sqlString:fieldModel.customFieldInfoGuid],
                                  (long)fieldModel.ID];
                }
                else
                {
                    sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCustomFieldDataMapping(FieldID,FieldGuid,CustomFieldInfoGuid) VALUES(%ld,'%@','%@')",
                                  (long)fieldModel.ID,
                                  [super sqlString:fieldModel.guid],
                                  [super sqlString:fieldModel.customFieldInfoGuid]];
                }
                
                result = [super runSqlCommand:sqlCommand];
            }
        }
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateFieldWithCardID:(NSString *)cardID fieldModel:(WCFieldModel *)fieldModel
{
    @synchronized(self)
    {
        BOOL result;
        NSString *sqlCommand;
        NSString *valueString = [self stringFromValue:fieldModel.value];
        NSString *rectString;
        
        if(CGRectIsEmpty(fieldModel.recogRect))
            rectString = @"";
        else rectString = CPStringFromRect(fieldModel.recogRect);
        
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCCardField SET FieldOrder=%ld,FieldSource=%ld,FieldType=%ld,FieldSubtype1=%ld,ValueString='%@',RecogRect='%@' WHERE FieldID=%ld AND FieldSubtype2=%ld",
                      (long)fieldModel.order,
                      (long)fieldModel.source,
                      (long)fieldModel.type,
                      (long)fieldModel.subType1,
                      [super sqlString:valueString],
                      [super sqlString:rectString],
                      (long)fieldModel.ID,
                      (long)fieldModel.subType2];
        
        result = [super runSqlCommand:sqlCommand];
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)removeFieldWithFieldID:(NSInteger)fieldID
{
    @synchronized(self)
    {
        BOOL result;
        NSString *sqlCommand;
        
        // delete from WCCardField
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCardField WHERE FieldID=%ld", (long)fieldID];
        result = [super runSqlCommand:sqlCommand];
        
        // release record ID
        if(result)
            [self addDeletedRecordID:fieldID withRecordType:WCTCDBC_RT_Field];
        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)getFieldsToCard:(WCCardModel *)cardModel
{
    @synchronized(self)
    {
        BOOL                result = YES;
        NSString            *sqlCommand = nil;
        sqlite3_stmt        *stmt;
        WCFieldModel        *fieldModel = nil;
        WC_ValueType        valueType;
        NSString            *valueString;
        int                 column = 0;
        
        
        sqlCommand = [NSString stringWithFormat:@"SELECT CF.FieldID,CF.FieldOrder,CF.FieldSource,CF.FieldType,\
                      CF.FieldSubType1,CF.FieldSubType2,CF.ValueString,CF.RecogRect,\
                      CDM.FieldGuid,CDM.CustomFieldInfoGuid FROM WCCardField as CF \
                      LEFT JOIN WCCustomFieldDataMapping as CDM ON CF.FieldID = CDM.FieldID WHERE CardID='%@'\
                      ORDER BY CF.FieldOrder,CF.FieldID,CF.FieldSubType2", [super sqlString:cardModel.ID]];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                column = 0;
                
                fieldModel = [[WCFieldModel alloc] init];
                
                fieldModel.ID = [super integerFromStmt:stmt column:column++];
                fieldModel.order = [super integerFromStmt:stmt column:column++];
                fieldModel.source = (WC_FieldSource)[super integerFromStmt:stmt column:column++];
                fieldModel.type = (WC_FieldType)[super integerFromStmt:stmt column:column++];
                fieldModel.subType1 = (WC_FieldSubType1)[super integerFromStmt:stmt column:column++];
                fieldModel.subType2 = (WC_FieldSubType2)[super integerFromStmt:stmt column:column++];
                
                if((valueString = [super stringFromStmt:stmt column:column++]))
                {
                    // set field value according to value type
                    valueType = [self valueTypeOfField:fieldModel];
                    fieldModel.value = [self valueFromString:valueString valueType:valueType];
                }
                
                if((valueString = [super stringFromStmt:stmt column:column++]))
                {
                    fieldModel.recogRect = CPRectFromString(valueString);
                }
                
                fieldModel.guid = [super stringFromStmt:stmt column:column++];
                fieldModel.customFieldInfoGuid = [super stringFromStmt:stmt column:column++];
                
                // !! 如果有 customFieldInfoGuid，先取得這個自訂欄位的內容格式
                if([fieldModel.customFieldInfoGuid length]>0)
                {
                    NSArray *customFieldInfos = [self copyAllCustomFieldInfos];

                    for (WCCustomFieldInfo *customFieldInfo in customFieldInfos)
                    {
                        if ([customFieldInfo.guid isEqualToString:fieldModel.customFieldInfoGuid])
                        {
                            fieldModel.customFieldContentType = customFieldInfo.contentType;
                            break;
                        }
                    }
                    
                    [customFieldInfos release];
                }
                if(fieldModel.subType2 != WC_FST2_None)
                {
                    // !! subType2有值代表是第2層資料，先檢查第1層的fieldModel是否存在
                    WCFieldModel *subType1FieldModel = [cardModel fieldWithType:fieldModel.type ID:fieldModel.ID];
                    
                    if(!subType1FieldModel)
                    {
                        subType1FieldModel = [fieldModel copy];
                        subType1FieldModel.subType2 = WC_FST2_None;
                        subType1FieldModel.value = nil;
                        [cardModel addField:subType1FieldModel];
                        [subType1FieldModel release];
                    }
                    
                    [subType1FieldModel setSubType2Field:fieldModel];
                }
                else
                {
                    // !! subType2沒有值代表是第1層資料，直接加入card model
                    [cardModel addField:fieldModel];
                }
                
                [fieldModel release];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            result = NO;
        }

        
        return result;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateFieldOrderWithID:(NSInteger)fieldID order:(NSInteger)fieldOrder
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCCardField SET FieldOrder=%ld WHERE FieldID=%ld", (long)fieldOrder, (long)fieldID];
        return [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
// check if field has value with specified type
//===============================================================================
- (BOOL)hasValueWithCardID:(NSString *)cardID fieldType:(WC_FieldType)fieldType
{
    @synchronized(self)
    {
        NSString    *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCardField WHERE CardID='%@' AND FieldType=%ld", [super sqlString:cardID], (long)fieldType];
        NSInteger   count = [super intResultWithSelectCommand:sqlCommand];
        
        return (count != 0);
    }
}


//===============================================================================
//
//===============================================================================
- (NSInteger)fieldCountWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString    *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(DISTINCT FieldID) FROM WCCardField WHERE CardID='%@'", [super sqlString:cardID]];
        NSInteger   count = [super intResultWithSelectCommand:sqlCommand];
        
        return (count != 0);
    }
    
}


//==============================================================================
//
//==============================================================================
- (NSString *)noteWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString *note = nil;
        sqlite3_stmt	*stmt;
        NSString    *sqlCommand = [NSString stringWithFormat:@"SELECT ValueString FROM WCCardField \
                                   WHERE CardID='%@' AND FieldType = %ld",
                                   [super sqlString:cardID], (long)WC_FT_Note];
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                note = [super stringFromStmt:stmt column:0];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return note;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateNote:(NSString *)note withCardID:(NSString *)cardID
{
    NSInteger maxID = [self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Field];

    @synchronized(self)
    {
        BOOL result = NO;
        if (note==nil )
        {
            [self setLastErrorWithErrorStep:@"updateNote:withCardID: note==nil"];
            return result;
        }

        if([cardID length]==0)
        {
            [self setLastErrorWithErrorStep:@"updateNote:withCardID: cardID==nil"];
            return result;
        }
        
        // !! 如果cardID不存在就不新增，避免空資料
        if([self isCardIDExist:cardID withSynchronized:NO]==NO)
        {
            return NO;
        }
        
        // 先檢查有沒有note
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT count(*) FROM WCCardField WHERE CardID='%@' AND FieldType = %ld",
                                [super sqlString:cardID], (long)WC_FT_Note];

        NSInteger noteCount = [super intResultWithSelectCommand:sqlCommand];
        // 有note就更新，沒note就新增
        if (noteCount<=0)
        {
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCardField(CardID, FieldID, FieldType, ValueString) VALUES('%@', %ld, %ld, '%@')",
                          [super sqlString:cardID],
                          (long)maxID+1,
                          (long)WC_FT_Note,
                          [super sqlString:note]];
        }
        else
        {
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCCardField SET ValueString='%@' WHERE CardID='%@' AND FieldType = %ld",
                          [super sqlString:note],
                          [super sqlString:cardID],
                          (long)WC_FT_Note];
        }
        
        result = [super runSqlCommand:sqlCommand];

        if (!result)
        {
            [self setLastErrorWithErrorStep:@"updateNote:withCardID: update failed"];
        }
        return result;
    }
}


//===============================================================================
// check!!
//===============================================================================
- (NSMutableArray *)copyFieldIDArrayWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString        *sqlCommand = [NSString stringWithFormat:@"SELECT DISTINCT FieldID FROM WCCardField WHERE CardID='%@'", [super sqlString:cardID]];
        NSInteger       fieldID;
        NSString        *fieldIDString = nil;
        NSMutableArray  *fieldIDArray = [[NSMutableArray alloc] init];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                fieldID = [super integerFromStmt:stmt column:0];
                fieldIDString = [[NSString alloc] initWithFormat:@"%ld", (long)fieldID];
                [fieldIDArray addObject:fieldIDString];
                [fieldIDString release];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return fieldIDArray;
    }
}


//===============================================================================
//
//===============================================================================
- (NSMutableArray *)copyFieldValueArrayWithCardIDArray:(NSArray *)cardIDArray fieldType:(WC_FieldType)fieldType
{
    @synchronized(self)
    {
        NSMutableArray  *valueArray = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand = nil;
        sqlite3_stmt    *stmt;
        
        
        for(NSString *cardID in cardIDArray)
        {
            sqlCommand = [NSMutableString string];
            [sqlCommand appendFormat:@"SELECT ValueString FROM WCCardField WHERE CardID='%@' AND FieldType=%ld ORDER BY FieldOrder", cardID, (long)fieldType];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    NSString *valueString = [super stringFromStmt:stmt column:0];
                    
                    if([valueString length])
                        [valueArray addObject:valueString];
                }
                
                sqlite3_finalize(stmt);
            }
            
        }
        
        if(![valueArray count])
        {
            [valueArray release];
            valueArray = nil;
        }
        
        return valueArray;
    }
}


//===============================================================================
// 把屬於cardID的所有欄位串成一個字串
//===============================================================================
- (NSString *)fieldValueStringWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSMutableString *fieldValueString = [NSMutableString string];
        NSMutableString *sqlCommand = nil;
        sqlite3_stmt    *stmt;
        
        
        sqlCommand = [NSMutableString string];
        [sqlCommand appendFormat:@"SELECT FieldType, FieldSubType1, FieldSubType2, ValueString FROM WCCardField WHERE CardID='%@' ORDER BY ValueString", cardID];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                WC_FieldType fieldType = [super integerFromStmt:stmt column:column++];
                WC_FieldSubType1 subtype1 = [super integerFromStmt:stmt column:column++];
                WC_FieldSubType2 subtype2 = [super integerFromStmt:stmt column:column++];
                NSString *valueString = [super stringFromStmt:stmt column:column++];
                
                if ([fieldValueString length]>0)
                {
                    [fieldValueString appendString:@","];
                }
                
                if([valueString length])
                {
                    [fieldValueString appendFormat:@"%ld-%ld-%ld-%@", (long)fieldType, (long)subtype1, (long)subtype2, valueString];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        return fieldValueString;
    }
}


//===============================================================================
// 過濾傳真，傳呼機的號碼
//===============================================================================
- (NSMutableArray *)copyPhoneArrayWithCardIDArray:(NSArray *)cardIDArray
{
    @synchronized(self)
    {
        NSMutableArray  *valueArray = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand = nil;
        sqlite3_stmt    *stmt;
        
        for(NSString *cardID in cardIDArray)
        {
            sqlCommand = [NSMutableString string];
            [sqlCommand appendFormat:@"SELECT ValueString,FieldSubType1 FROM WCCardField WHERE CardID='%@' AND FieldType=%ld ORDER BY FieldOrder", cardID, (long)WC_FT_Phone];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                // 取得目前聯絡人的所有電話
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    int columnIndex = 0;
                    NSString *valueString = [super stringFromStmt:stmt column:columnIndex++];
                    WC_FieldSubType1 subType1 = (WC_FieldSubType1)[super integerFromStmt:stmt column:columnIndex++];
                    
                    // 過濾傳真，傳呼機
                    if([valueString length] &&
                       subType1 != WC_FST1_Phone_WorkFax &&
                       subType1 != WC_FST1_Phone_HomeFax &&
                       subType1 != WC_FST1_Phone_Pager)
                    {
                        [valueArray addObject:valueString];
                    }
                }
                
                sqlite3_finalize(stmt);
            }
            
        }
        
        if(![valueArray count])
        {
            [valueArray release];
            valueArray = nil;
        }
        
        return valueArray;
    }
}


//===============================================================================
// 7.0 規則
// 1。過濾傳真，傳呼機，有分機的號碼
// 2。一筆資料只傳送一個號碼
// 3。順序為: 行動電話，iphone, 商務電話，住家電話，公司總機，gooel Voice
// WC_FST1_Phone_Mobile, WC_FST1_Phone_iPhone, WC_FST1_Phone_Work,WC_FST1_Phone_Home,WC_FST1_Phone_Main,WC_FST1_Phone_GoogleVoice,WC_FST1_Phone_Other
//===============================================================================
- (NSMutableArray *)copyPhoneArrayForSendMessageWithCardIDArray:(NSArray *)cardIDArray
{
    @synchronized(self)
    {
        // 優先順序
        WC_FieldSubType1 orderArray[7] = {WC_FST1_Phone_Mobile, WC_FST1_Phone_iPhone, WC_FST1_Phone_Work,WC_FST1_Phone_Home,WC_FST1_Phone_Main,WC_FST1_Phone_GoogleVoice, WC_FST1_Phone_Other};

        // 分機過濾條件
        NSCharacterSet  *extSymbolSet = [NSCharacterSet characterSetWithCharactersInString:@"#,p"];

        NSMutableArray  *valueArray = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand = nil;
        sqlite3_stmt    *stmt;
        
        for(NSString *cardID in cardIDArray)
        {
            sqlCommand = [NSMutableString string];
            [sqlCommand appendFormat:@"SELECT ValueString,FieldSubType1 FROM WCCardField WHERE CardID='%@' AND FieldType=%ld ORDER BY FieldOrder", cardID, (long)WC_FT_Phone];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                // 取得目前聯絡人的所有電話
                NSMutableDictionary *phonesInOneCard = [[NSMutableDictionary alloc] init];
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    int columnIndex = 0;
                    NSString *valueString = [super stringFromStmt:stmt column:columnIndex++];
                    WC_FieldSubType1 subType1 = (WC_FieldSubType1)[super integerFromStmt:stmt column:columnIndex++];
                    
                    // 過濾傳真，傳呼機
                    if([valueString length] &&
                       subType1 != WC_FST1_Phone_WorkFax &&
                       subType1 != WC_FST1_Phone_HomeFax &&
                       subType1 != WC_FST1_Phone_Pager)
                    {
                        // 濾掉有分機符號的電話
                        NSRange range = [valueString rangeOfCharacterFromSet:extSymbolSet];
                        
                        if(range.length == 0)
                        {
                            // 如果同類型的已經有了就不用紀錄
                            if ([phonesInOneCard objectForKey:@(subType1)]==nil)
                            {
                                [phonesInOneCard setObject:valueString forKey:@(subType1)];
                            }
                        }
                    }
                }
                
                //////////////////////////////////////////////////
                // 依序決定要取哪一個電話
                if ([phonesInOneCard count]<=1) //只有一個電話，不用看順序
                {
                    NSString *value = [[phonesInOneCard allValues] firstObject];
                    
                    if ([value length]>0)
                    {
                        [valueArray addObject:value];
                    }
                }
                else
                {
                    for (NSInteger orderIndex = 0; orderIndex < 7; orderIndex++)
                    {
                        NSString *value = [phonesInOneCard objectForKey:@(orderArray[orderIndex])];
                        if ([value length]>0)
                        {
                            [valueArray addObject:value];
                            break;
                        }
                    }
                }
                [phonesInOneCard release];
                sqlite3_finalize(stmt);
            }
            
        }
        
        if(![valueArray count])
        {
            [valueArray release];
            valueArray = nil;
        }
        
        return valueArray;
    }
}





#pragma mark - Card methods

//===============================================================================
//
//===============================================================================
- (NSInteger)totalCardCount
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard"];
        
        return [super intResultWithSelectCommand:sqlCommand];
    }
}


//===============================================================================
//
//===============================================================================
- (NSInteger)totalTwoSideCardCountWithSourceID:(WC_SourceID)sourceID
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard WHERE BackRecogLang!= -1 AND SourceID=%ld",(long)sourceID];
        
        return [super intResultWithSelectCommand:sqlCommand];
    }
}


//===============================================================================
// NOTE: Group資料不會在此處理，統一由Card-Group相關函式處理
// 包含unverified, favorite
//===============================================================================
- (BOOL)insertCard:(WCCardModel *)cardModel isImportMode:(BOOL)isImportMode
{
    return [self insertCard:cardModel syncActionModel:nil isImportMode:isImportMode isInTransaction:NO];
}


//===============================================================================
// NOTE: Group資料不會在此處理，統一由Card-Group相關函式處理
//===============================================================================
- (BOOL)removeCardWithCardID:(NSString *)cardID
{
    return [self removeCardWithCardID:cardID syncActionModel:nil];
}

//===============================================================================
// get card data
// PARAMS: getAllField [in] - load field data or not
// NOTE: Group資料不會在此處理，統一由Card-Group相關函式處理
//===============================================================================
- (WCCardModel *)copyCardWithCardID:(NSString *)cardID getAllField:(BOOL)getAllField
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if(![cardID length])
        {
            [self setLastErrorWithErrorStep:@"copyCardWithCardID : no cardID"];
            return nil;
        }
        
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        BOOL                result = NO;
        NSString            *sqlCommand = nil;
        sqlite3_stmt        *stmt;
        int                 column = 0;
        NSString            *errorStep = nil;
        NSArray             *groupIDArray = nil;
        WCCardModel         *cardModel = [[WCCardModel alloc] init];
        
        cardModel.ID = cardID;
        self.lastError = nil;
        
        //--------------------------------
        // get main data
        //--------------------------------
        sqlCommand = [NSString stringWithFormat:@"SELECT DisplayName, DisplayCompany,\
                      DisplayJobTitle, DisplayAddress, DisplayGPS, \
                      Creator, Owner, Editor, SectionTitle, CreatedTime, ModifiedTime, \
                      FrontRecogLang, BackRecogLang FROM WCCard WHERE CardID='%@'", [super sqlString:cardID]];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if(sqlite3_step(stmt) == SQLITE_ROW)
            {
                cardModel.displayName = [super stringFromStmt:stmt column:column++];
                cardModel.displayCompany = [super stringFromStmt:stmt column:column++];
                cardModel.displayJobTitle = [super stringFromStmt:stmt column:column++];
                cardModel.displayAddress = [super stringFromStmt:stmt column:column++];
                cardModel.displayGPS = [super stringFromStmt:stmt column:column++];
                cardModel.creator = [super stringFromStmt:stmt column:column++];
                cardModel.owner = [super stringFromStmt:stmt column:column++];
                cardModel.editor = [super stringFromStmt:stmt column:column++];
                cardModel.sectionTitle = [super stringFromStmt:stmt column:column++];
                cardModel.createdTime = [self dateFromStmt:stmt column:column++];
                cardModel.modifiedTime = [self dateFromStmt:stmt column:column++];
                
                // TODO: 如果是依時間排序，要修正sectionTitle
                
                cardModel.frontRecogLang = [self integerFromStmt:stmt column:column++];
                cardModel.backRecogLang = [self integerFromStmt:stmt column:column++];
                
                sqlite3_finalize(stmt);
            }
            else
            {
                sqlite3_finalize(stmt);
                errorStep = @"copyCardWithCardID : card not exist";
                goto _EXIT;
            }
        }
        else
        {
            errorStep = @"copyCardWithCardID : failed to get main data";
            goto _EXIT;
        }
        
        //////////////////////////////////////////////////
        // 處理未校正
        if([self isUnverifiedWithCardID:cardModel.ID])
        {
            cardModel.tagMask |= WC_TagMask_Unverified;
        }
        
        // 處理我的最愛
        if([self favoriteOrderWithCardID:cardModel.ID]!=-1)
        {
            cardModel.tagMask |= WC_TagMask_Favorite;
        }
        
        
        //--------------------------------
        // get group data
        //--------------------------------
        groupIDArray = [self copyGroupIDArrayWithCardID:cardModel.ID];
        [cardModel setGroupIDArray:groupIDArray isInitCard:YES];
        [groupIDArray release];
        
        //--------------------------------
        // get shared account data
        //--------------------------------
        NSArray *sharedAccountGuids = [self copySharedAccountWithCardID:cardModel.ID];
        cardModel.sharedAccountGUIDArray = sharedAccountGuids;
        [sharedAccountGuids release];
        
        if(!getAllField)
        {
            result = YES;
            goto _EXIT;
        }
        
        //--------------------------------
        // get field data
        //--------------------------------
        if(![self getFieldsToCard:cardModel])
        {
            errorStep = @"copyCardWithCardID : failed to getFieldsToCard";
            goto _EXIT;
        }
        
        //--------------------------------
        // finish with success
        //--------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        if(!result)
        {
            [self setLastErrorWithErrorStep:errorStep];
            [cardModel release];
            cardModel = nil;
        }
        
        [pool release];
        
        return cardModel;
    }
}

//===============================================================================
//
//===============================================================================
- (BOOL)isCardExist:(WCCardModel *)cardModel
{
    @synchronized(self)
    {
        NSInteger cardCount = 0;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard WHERE DisplayName='%@' AND DisplayCompany='%@' AND DisplayJobTitle='%@' AND DisplayAddress='%@' AND DisplayGPS='%@' AND SectionTitle='%@' AND CreatedTime=%f AND ModifiedTime=%f AND FrontRecogLang=%ld AND BackRecogLang=%ld",
                                [super sqlString:cardModel.displayName],
                                [super sqlString:cardModel.displayCompany],
                                [super sqlString:cardModel.displayJobTitle],
                                [super sqlString:cardModel.displayAddress],
                                [super sqlString:cardModel.displayGPS],
                                [super sqlString:cardModel.sectionTitle],
                                [cardModel.createdTime timeIntervalSince1970],
                                [cardModel.modifiedTime timeIntervalSince1970],
                                (long)cardModel.frontRecogLang,
                                (long)cardModel.backRecogLang];
        
        cardCount = [super intResultWithSelectCommand:sqlCommand];
        
        return (cardCount > 0);
        
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)isCardIDExist:(NSString *)cardID withSynchronized:(BOOL)synchronized
{
    if (synchronized)
    {
        @synchronized(self)
        {
            NSInteger cardCount = 0;
            NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard WHERE CardID='%@'"
                                    , [super sqlString:cardID]];
            cardCount = [super intResultWithSelectCommand:sqlCommand];
            return (cardCount > 0);
        }
    }
    else
    {
        NSInteger cardCount = 0;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCCard WHERE CardID='%@'"
                                , [super sqlString:cardID]];
        cardCount = [super intResultWithSelectCommand:sqlCommand];
        return (cardCount > 0);
    }
    
}


//==============================================================================
//
//==============================================================================
- (BOOL)isCardIDExist:(NSString *)cardID
{
    return [self isCardIDExist:cardID withSynchronized:YES];
}


//===============================================================================
// RETURNED: mutable array of cardID (with necessary display data only)
//           must release after using.
//===============================================================================
- (NSArray *)copyEmptyCardID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSMutableArray	*cardArray = [[NSMutableArray alloc] init];
        NSString        *stringValue;
        const char		*utf8Command="SELECT CardID FROM WCCard Where WCCard.CardID not in (SELECT CardID FROM WCCardField)";
        
        
        if(sqlite3_prepare_v2(self.dbHandle, utf8Command, -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                stringValue = [super stringFromStmt:stmt column:0];
                
                if(!stringValue)
                    continue;
                
                [cardArray addObject:stringValue];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyEmptyCardID : Failed to select CardID in DB"];
            [cardArray release];
            cardArray = nil;
        }
        
        return cardArray;
    }
}


//==============================================================================
//
//==============================================================================
- (void)updateOrderToFirstWithIndex:(NSInteger)index withCardID:(NSString *)cardID fieldType:(NSInteger)fieldType
{
    WCCardModel *cardModel = [self copyCardWithCardID:cardID getAllField:YES];
    if (cardModel==nil)
    {
        return ;
    }
    
    NSMutableArray *fields =nil;
    @synchronized(self)
    {
        //////////////////////////////////////////////////
        // 取得此名片所以相同fieldType的fieldID與Order
        fields = [cardModel fieldArrayWithType:fieldType];
        
        NSInteger fieldIndex = 0;
        for (WCFieldModel * fieldModel in fields)
        {
            if (fieldIndex == index)
            {
                fieldModel.order = 0;
            }
            else if (fieldIndex < index)
            {
                fieldModel.order ++;
            }
            
            fieldIndex ++;
        }
        
        //////////////////////////////////////////////////
        // 依順序重排fieldOrder
        for (WCFieldModel * fieldModel in fields)
        {
            [self updateFieldOrderWithID:fieldModel.ID order:fieldModel.order];
            //        [self updateFieldWithCardID:cardModel.ID fieldModel:fieldModel];
        }
        
        
        // 更新修改時間
        [self updateModifiedTimeWithCardID:cardID withSynchronized:NO];
    }
    
    [cardModel release];
}


//===============================================================================
// copy all cardModel of one group (only display data)
// !! 列表及搜尋功能使用
//===============================================================================
- (NSMutableArray *)copyAllCardsWithGroupID:(NSInteger)groupID searchText:(NSString *)searchText
{
    @synchronized(self)
    {
        NSMutableArray      *allCardArray = [[NSMutableArray alloc] init];
        WCCardModel         *cardModel;
        NSMutableString     *sqlCommand = [NSMutableString string];
        NSString            *valueString;
        sqlite3_stmt        *stmt;
        BOOL                hasSearchText = ([searchText length] > 0);
        BOOL                hasGroupID = (groupID != WC_GID_All && groupID != WC_GID_AllCardInDb);
        
        
        self.lastError = nil;
        
        //--------------------------------
        // get main data
        //--------------------------------
        
        // !! 因為取資料的順序是一致的，這裡不管是不是search，select敘述都要一致。
        [sqlCommand appendString:@"SELECT c.CardID,fav.FavoriteOrder,u.CardID,c.DisplayName,c.DisplayCompany,c.DisplayJobTitle,c.DisplayAddress,c.DisplayGPS,c.SectionTitle,c.Creator,c.Owner,c.Editor,c.CreatedTime,c.ModifiedTime FROM WCCard c"];
        
        // !! search時才需要合併WCCardField
        if(hasSearchText == YES)
        {
            [sqlCommand appendString:@" LEFT JOIN WCCardField f ON f.CardID=c.CardID"];
        }
        
        // !! 未校正與我的最愛資訊也一起抓，因為一筆都只有一個未校正與我的最愛狀態，所以應該不會太慢
        [sqlCommand appendString:@" LEFT JOIN WCFavorite fav ON fav.CardID=c.CardID"];
        [sqlCommand appendString:@" LEFT JOIN WCUnverified u ON u.CardID=c.CardID"];
        
        if(hasGroupID == YES)
        {
            [sqlCommand appendString:@" LEFT JOIN WCCardGroup g ON g.CardID=c.CardID"];
        }
        
        if(hasGroupID == YES || hasSearchText == YES)
        {
            BOOL needAppendAnd = NO;
            
            [sqlCommand appendString:@" WHERE"];
            
            if(hasGroupID == YES)
            {
                [sqlCommand appendFormat:@" g.GroupID=%td", groupID];
                needAppendAnd = YES;
            }
            
            if(hasSearchText == YES)
            {
                if(needAppendAnd == YES)
                {
                    [sqlCommand appendString:@" AND"];
                }
                
                // !! 要排除自訂欄位
                [sqlCommand appendFormat:@" f.FieldType!=%td", WC_FT_UserDefine];
                needAppendAnd = YES;
                
                NSString *searchCondition = [self searchConditionWithSearchText:searchText searchField:@"f.ValueString"];
                if ([searchCondition length] > 0)
                {
                    if(needAppendAnd == YES)
                    {
                        [sqlCommand appendString:@" AND"];
                    }
                    
                    [sqlCommand appendString:searchCondition];
                }
                
                [sqlCommand appendString:@" GROUP BY f.CardID"];
            }
        }
        
        
        // clear cancel flag
        [self clearCancelFlag];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //-------------------------------------------------
                // check if interrupt
                //-------------------------------------------------
                if([self isCanceled])
                {
                    //                    NSLog(@"%s cancel search:%@", __func__, searchText);
                    
                    sqlite3_interrupt(self.dbHandle);
                    sqlite3_finalize(stmt);
                    [allCardArray release];
                    return nil;
                }
                
                
                //-------------------------------------------------
                // handel result
                //-------------------------------------------------
                int column = 0;
                valueString = [super stringFromStmt:stmt column:column++];
                
                if(![valueString length])
                    continue;
                
                cardModel = [[WCCardModel alloc] init];
                cardModel.ID = valueString;
                
                // !! 如果有值就是我的最愛
                NSString *favorite = [super stringFromStmt:stmt column:column++];
                
                if ([favorite length]>0)
                {
                    cardModel.tagMask |= WC_TagMask_Favorite;
                }
                
                // !! 如果有值就是未校正
                NSString *unverified = [super stringFromStmt:stmt column:column++];
                
                if([unverified length]>0)
                {
                    cardModel.tagMask |= WC_TagMask_Unverified;
                }
                
                
                cardModel.displayName = [super stringFromStmt:stmt column:column++];
                cardModel.displayCompany = [super stringFromStmt:stmt column:column++];
                cardModel.displayJobTitle = [super stringFromStmt:stmt column:column++];
                cardModel.displayAddress = [super stringFromStmt:stmt column:column++];
                cardModel.displayGPS = [super stringFromStmt:stmt column:column++];
                cardModel.sectionTitle = [super stringFromStmt:stmt column:column++];
                
                // WCT新增
                cardModel.creator = [super stringFromStmt:stmt column:column++];
                cardModel.owner = [super stringFromStmt:stmt column:column++];
                cardModel.editor = [super stringFromStmt:stmt column:column++];
                cardModel.createdTime = [super dateFromStmt:stmt column:column++];
                cardModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                
                //////////////////////////////////////////////////
                
                [allCardArray addObject:cardModel];
                [cardModel release];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"Failed to get card"];
        }
        
        return allCardArray;
    }
}


//===============================================================================
//
//===============================================================================
- (NSDate *)cardModifiedTimeWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString		*sqlCommand = [NSString stringWithFormat:@"SELECT ModifiedTime FROM WCCard WHERE CardID='%@'", [super sqlString:cardID]];
        sqlite3_stmt	*stmt;
        NSDate          *modifiedTime = nil;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                modifiedTime = [self dateFromStmt:stmt column:0];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return modifiedTime;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardModifiedTime:(NSDate *)modifiedTime withCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        if (modifiedTime==nil)
        {
            return NO;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCCard SET ModifiedTime=%f WHERE CardID='%@'", [modifiedTime timeIntervalSince1970], [super sqlString:cardID]];
        
        if(![self runSqlCommand:sqlCommand])
        {
            [self setLastErrorWithErrorStep:@"updateCardModifiedTime failed"];

            return NO;
        }
        
        return YES;
    }
}





#pragma mark - Change display rule methods

//===============================================================================
// 取得所有用來產生顯示資料的欄位資料 (name, company, 電話，地址，建立時間，修改時間)
// RETURNED: WCCardModel array
// NOTE: !! 變更顯示設定時使用
// 因為建立時間，修改時間全部都有，所以修件改為全部都抓
//===============================================================================
- (NSMutableArray *)copyAllCardsForChangeDisplayRule
{
    @synchronized(self)
    {
        NSMutableArray			*cardArray = [[NSMutableArray alloc] init];
        NSMutableDictionary		*cardDict = [[NSMutableDictionary alloc] init];
        WCCardModel				*cardModel = nil;
        WCFieldModel			*fieldModel = nil;
        sqlite3_stmt			*stmt;
        NSString				*sqlCommand = nil;
        NSString				*cardID = nil;
        NSString                *temp;
        int                     column = 0;
        WC_ValueType            valueType;
        
        
        // !! 需要欄位 : first name/first name phonetic/last name/last name phonetic/company name/company name phonetic/ create time/ modified time
        sqlCommand = [NSString stringWithFormat:
                      @"SELECT c.CardID,c.CreatedTime,c.ModifiedTime,f.FieldID,f.FieldOrder,f.FieldSource,f.FieldType,f.FieldSubType1,f.FieldSubType2,f.ValueString FROM WCCard c LEFT JOIN WCCardField f ON c.CardID = f.CardID"];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                column = 0;
                
                if((cardID = [super stringFromStmt:stmt column:column++]))
                {
                    // 不管要不要用都要把值取走，不然後面會取錯值
                    NSDate *createdTime = [self dateFromStmt:stmt column:column++];
                    NSDate *modifiedTime = [self dateFromStmt:stmt column:column++];
                    
                    if(!(cardModel = [cardDict objectForKey:cardID]))
                    {
                        cardModel = [[WCCardModel alloc] init];
                        cardModel.ID = cardID;
                        
                        //////////////////////////////////////////////////
                        // 加入建立，修改時間
                        if(createdTime!=nil)
                            cardModel.createdTime = createdTime;
                        
                        if(modifiedTime!=nil)
                            cardModel.modifiedTime = modifiedTime;
                        
                        
                        [cardDict setObject:cardModel forKey:cardID];
                        [cardArray addObject:cardModel];
                        [cardModel release];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if((fieldModel = [[WCFieldModel alloc] init]))
                    {
                        fieldModel.ID = [super integerFromStmt:stmt column:column++];
                        fieldModel.order = [super integerFromStmt:stmt column:column++];
                        fieldModel.source = (WC_FieldSource)[super integerFromStmt:stmt column:column++];
                        fieldModel.type = (WC_FieldType)[super integerFromStmt:stmt column:column++];
                        fieldModel.subType1 = (WC_FieldSubType1)[super integerFromStmt:stmt column:column++];
                        fieldModel.subType2 = (WC_FieldSubType2)[super integerFromStmt:stmt column:column++];
                        
                        if((temp = [super stringFromStmt:stmt column:column++]))
                        {
                            valueType = [self valueTypeOfField:fieldModel];
                            fieldModel.value = [self valueFromString:temp valueType:valueType];
                        }
                        
                        // 這邊因為會用到電話，所以也要分兩個方式取得
                        if(fieldModel.subType2 != WC_FST2_None)
                        {
                            // !! subType2有值代表是第2層資料，先檢查第1層的fieldModel是否存在
                            WCFieldModel *subType1FieldModel = [cardModel fieldWithType:fieldModel.type ID:fieldModel.ID];
                            
                            if(!subType1FieldModel)
                            {
                                subType1FieldModel = [fieldModel copy];
                                subType1FieldModel.subType2 = WC_FST2_None;
                                subType1FieldModel.value = nil;
                                [cardModel addField:subType1FieldModel];
                                [subType1FieldModel release];
                            }
                            
                            [subType1FieldModel setSubType2Field:fieldModel];
                        }
                        else
                        {
                            // !! subType2沒有值代表是第1層資料，直接加入card model
                            [cardModel addField:fieldModel];
                        }

                        [fieldModel release];
                    }
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else [self setLastErrorWithErrorStep:@"copyAllCardsForChangeDisplayRule : failed to get cards"];
        
        [cardDict release];
        
        return cardArray;
    }
}


//===============================================================================
// 更新所有顯示資料的相關欄位 (name, company)
// NOTE: !! 變更顯示設定時使用
//===============================================================================
- (BOOL)updateAllCardsForChangeDisplayRule:(NSArray *)cardArray
{
    @synchronized(self)
    {
        BOOL        result = NO;
        NSString    *sqlCommand;
        NSArray     *checkNameArray;
        NSString    *displayName;
        NSString    *errorStep = nil;
        
        

        [super beginTransaction];

        for(WCCardModel *cardModel in cardArray)
        {
            @autoreleasepool
            {
                //--------------------------------
                // update WCCard table
                //--------------------------------
                sqlCommand = [NSString stringWithFormat:@"UPDATE WCCard SET DisplayName='%@',DisplayCompany='%@',DisplayJobTitle='%@',DisplayAddress='%@',DisplayGPS='%@',SectionTitle='%@' WHERE CardID='%@'",
                             [super sqlString:cardModel.displayName],
                             [super sqlString:cardModel.displayCompany],
                             [super sqlString:cardModel.displayJobTitle],
                             [super sqlString:cardModel.displayAddress],
                             [super sqlString:cardModel.displayGPS],
                             [super sqlString:cardModel.sectionTitle],
                             [super sqlString:cardModel.ID]];
                
                
                //--------------------------------
                // update search data in WCCardField table
                //--------------------------------
                checkNameArray = [cardModel fieldArrayWithType:WC_FT_Name];
                
                for(WCFieldModel *nameField in checkNameArray)
                {
                    displayName = [nameField stringDisplayName];
                    NSString *temp = [NSString stringWithFormat:@"UPDATE WCCardField SET ValueString='%@' WHERE FieldID=%ld AND FieldType=%ld AND FieldSubType2=%ld",
                                      [super sqlString:displayName], (long)nameField.ID, (long)WC_FT_Name, (long)WC_FST2_SearchOnly];
                    
                    if ([sqlCommand length]>0)
                    {
                        sqlCommand = [sqlCommand stringByAppendingString:@";"];
                    }
                    sqlCommand = [sqlCommand stringByAppendingString:temp];
                }
                
                
                if([sqlCommand length]>0)
                {
                    if(![super runSqlCommand:sqlCommand])
                    {
                        errorStep = @"updateAllCardsForChangeDisplayRule: update WCCardField failed";
                        goto _EXIT;
                    }
                }
            } // end of autoreleasepool
        }
        
        if(![super commitTransaction])
        {
            errorStep = @"commitTransaction failed";
            goto _EXIT;
        }
        
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        if(!result)
        {
            [super rollbackTransaction];
            [self setLastErrorWithErrorStep:errorStep];
        }
     
        return result;
    }
}






#pragma mark - Duplicate check methods
// 用來dump 比對用的資料
//#define DumpDuplicateData

// 用來dump執行時間
//#define DumpDuplicateTime

//===============================================================================
// 取得所有比對用的資料
// NOTE: duplicate判斷條件
//       1. FirstName + LastName 相同 (main key)
//       2. Company or Email 有一個是相同的
//===============================================================================
- (WCDuplicateCompareModel *)copyDuplicateCompareModelFromDBWithNameOnly:(BOOL)nameOnly
{
    @synchronized(self)
    {
        WCDuplicateCompareModel *checkModel = [[WCDuplicateCompareModel alloc] init];
        WCDuplicateCheckNameModel *nameModel = nil;
        
        sqlite3_stmt		*stmt;
        NSString			*cardID = nil;
        NSInteger           fieldType = 0;
        NSInteger           fieldSubType2 = 0;
        NSInteger           fieldID = 0;
        NSString			*fieldValue;
        int                 column = 0;
        NSMutableDictionary *fieldValueSetDict;
        NSMutableSet        *fieldValueSet;
        BOOL                result = NO;
        NSString            *sqlCommand = nil;
        
#ifdef DumpDuplicateTime
        NSString *logKey = @"copyDuplicateCompareModelFromDB";
        PPLog_StartLogCostTime(logKey);
#endif
        // 取得有重姓名的CardID
        sqlCommand = [NSString stringWithFormat:
                      @"SELECT CardID, ValueString from WCCardField WHERE ValueString COLLATE NOCASE in \
                      (SELECT ValueString from WCCardField WHERE FieldType=%ld AND FieldSubType2=%ld AND length(ValueString)>0\
                      GROUP BY ValueString COLLATE NOCASE having count(ValueString)>1) \
                      AND FieldType=%ld AND FieldSubType2=%ld \
                      ORDER BY ValueString COLLATE NOCASE, CardID COLLATE NOCASE;",
                      (long)WC_FT_Name, (long)WC_FST2_SearchOnly,(long)WC_FT_Name, (long)WC_FST2_SearchOnly
                      ];
#ifdef DumpDuplicateData
        PPLogController *nameLog = [[PPLogController alloc] init];
        [nameLog setFileName:@"name" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
        nameLog.toFile = YES;
        nameLog.toConsole = NO;
#endif
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                column = 0;
                cardID = [super stringFromStmt:stmt column:column++];
                fieldValue = [super stringFromStmt:stmt column:column++];
#ifdef DumpDuplicateData
                [nameLog logWithMask:PPLogControllerMask_Normal format:@"%@",cardID];
#endif
                nameModel = [WCDuplicateCheckNameModel copyNameModelWithCardID:cardID name:fieldValue];
                [checkModel.nameArray addObject:nameModel];
                [nameModel release];
            }
            
            sqlite3_finalize(stmt);
        }
        
#ifdef DumpDuplicateData
        [nameLog release];
#endif
        //////////////////////////////////////////////////
        // !!只取姓名時，不取空白名片
        if(nameOnly==NO)
        {
#ifdef DumpDuplicateTime
            PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得有重複姓名的CardID (%ld) end", [checkModel.nameArray count]);
#endif
            //        NSLog(@"取得有沒有姓名的CardID");
            //-----------------------------------------
            // !! Dr. tsay要求，沒有姓名也要比對。
            //-----------------------------------------
            sqlCommand = [NSString stringWithFormat:
                          @"SELECT CardID FROM WCCard WHERE DisplayName='' ORDER BY CardID COLLATE NOCASE"];
            
#ifdef DumpDuplicateData
            PPLogController *noNameLog = [[PPLogController alloc] init];
            [noNameLog setFileName:@"no name" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
            noNameLog.toFile = YES;
            noNameLog.toConsole = NO;
#endif
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    cardID = [super stringFromStmt:stmt column:0];
#ifdef DumpDuplicateData
                    [noNameLog logWithMask:PPLogControllerMask_Normal format:@"%@",cardID];
#endif
                    
                    nameModel = [WCDuplicateCheckNameModel copyNameModelWithCardID:cardID name:@""];
                    [checkModel.nameArray addObject:nameModel];
                    [nameModel release];
                }
                
                sqlite3_finalize(stmt);
            }
            
#ifdef DumpDuplicateData
            [noNameLog release];
#endif
            
#ifdef DumpDuplicateTime
            PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得沒有姓名的CardID (%ld) End", [checkModel.nameArray count]);
#endif
        }
        
        //-----------------------------------------
        // 以產生CardID列表
        //-----------------------------------------
        NSMutableString *cardIDList = [NSMutableString string];
        for (WCDuplicateCheckNameModel *checkNameModel in checkModel.nameArray)
        {
            if ([cardIDList length]>0)
            {
                [cardIDList appendFormat:@","];
            }
            [cardIDList appendFormat:@"'%@'", checkNameModel.cardID];
        }
        
        if ([cardIDList length]==0)
        {
#ifdef DumpDuplicateTime
            PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"End 產生CardID列表 - cardIDList==0");
#endif
            goto _EXIT;
        }
        
        //-----------------------------------------
        // 以CardID, 取得所有姓名，全部轉為小寫
        // 串成 WC_FT_Name-WC_FST2_Name_First-(FirstName)-WC_FST2_Name_Last-(LastName)的格式
        //-----------------------------------------
        NSMutableDictionary *fullNameDict = [NSMutableDictionary dictionary];
        
        sqlCommand = [NSString stringWithFormat:
                      @"SELECT CardID, FieldID, FieldType, FieldSubType2, LOWER(ValueString) FROM WCCardField WHERE CardID in (%@) \
                      AND FieldType=%ld AND (FieldSubType2=%ld OR FieldSubType2=%ld) ORDER BY CardID, FieldSubType2",
                      cardIDList,
                      (long)WC_FT_Name, (long)WC_FST2_Name_First, (long)WC_FST2_Name_Last];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //-------------------------------------------------
                // check if interrupt
                //-------------------------------------------------
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    sqlite3_finalize(stmt);
#ifdef DumpDuplicateTime
                    PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"End 取得所有姓名 - cancel");
#endif
                    goto _EXIT;
                }
                
                
                //-------------------------------------------------
                // get result
                //-------------------------------------------------
                column = 0;
                cardID = [super stringFromStmt:stmt column:column++];
                fieldID = [super integerFromStmt:stmt column:column++];
                fieldType = [super integerFromStmt:stmt column:column++];
                fieldSubType2 = [super integerFromStmt:stmt column:column++];
                fieldValue = [super stringFromStmt:stmt column:column++];
                
                NSString *fieldString = [NSString stringWithFormat:@"%td-%@", fieldSubType2, [super sqlString:fieldValue]];
                
                // 如果本來就有，要串在一起
                NSMutableDictionary *currentNameDict = [fullNameDict objectForKey:cardID];
                
                if (currentNameDict==nil)
                {
                    currentNameDict = [NSMutableDictionary dictionary];
                    [fullNameDict setObject:currentNameDict forKey:cardID];
                }
                
                NSString *currentName = [currentNameDict objectForKey:@(fieldID)];
                if([currentName length]>0)
                {
                    fieldString = [currentName stringByAppendingFormat:@"-%@", fieldString];
                }
                
                [currentNameDict setObject:fieldString forKey:@(fieldID)];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
#ifdef DumpDuplicateTime
            PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得所有姓名 - failed");
#endif
            goto _EXIT;
        }

#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得所有姓名 End");
#endif
        
        [fullNameDict enumerateKeysAndObjectsUsingBlock:^(NSString *  _Nonnull cardID, NSMutableDictionary *currentNameDict, BOOL * _Nonnull stop) {

            NSMutableDictionary *fieldValueSetDict = [checkModel.restFieldDict objectForKey:cardID];
            
            if(!fieldValueSetDict)
            {
                fieldValueSetDict = [[NSMutableDictionary alloc] init];
                [checkModel.restFieldDict setObject:fieldValueSetDict forKey:cardID];
            }
            else [fieldValueSetDict retain];

            [currentNameDict enumerateKeysAndObjectsUsingBlock:^(NSNumber *fieldID, NSString *nameString, BOOL * _Nonnull stop) {
                
                NSMutableSet *fieldValueSet = [fieldValueSetDict objectForIntKey:WC_FT_Name];
                
                if(!fieldValueSet)
                {
                    fieldValueSet = [[NSMutableSet alloc] init];
                    [fieldValueSetDict setObject:fieldValueSet forIntKey:WC_FT_Name];
                }
                else [fieldValueSet retain];
                
                [fieldValueSet addObject:nameString];
                [fieldValueSet release];
            }];

            [fieldValueSetDict release];
        }];
    
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得所有姓名-加到checkModels End");
#endif
        
        if (nameOnly==NO)
        {
            //-----------------------------------------
            // 以CardID, 取得公司, 電話與郵件資料，全部轉為小寫
            //-----------------------------------------
            sqlCommand = [NSString stringWithFormat:
                          @"SELECT CardID, FieldType, LOWER(ValueString) FROM WCCardField WHERE CardID in (%@) \
                          AND ((FieldType=%ld AND FieldSubType2=%ld) OR FieldType=%ld OR FieldType=%ld)",
                          cardIDList,
                          (long)WC_FT_Company, (long)WC_FST2_Company_Name,
                          (long)WC_FT_Phone,
                          (long)WC_FT_Email];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    //-------------------------------------------------
                    // check if interrupt
                    //-------------------------------------------------
                    if([self isCanceled])
                    {
                        sqlite3_interrupt(self.dbHandle);
                        sqlite3_finalize(stmt);
#ifdef DumpDuplicateTime
                        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"End 取得所有名片的company, email, phone - cancel");
#endif
                        goto _EXIT;
                    }
                    
                    
                    //-------------------------------------------------
                    // get result
                    //-------------------------------------------------
                    column = 0;
                    cardID = [super stringFromStmt:stmt column:column++];
                    fieldType = [super integerFromStmt:stmt column:column++];
                    fieldValue = [super stringFromStmt:stmt column:column++];
                    
                    if([fieldValue length])
                    {
                        switch (fieldType)
                        {
                            default:
                            {
                                fieldValueSetDict = [checkModel.restFieldDict objectForKey:cardID];
                                
                                if(!fieldValueSetDict)
                                {
                                    fieldValueSetDict = [[NSMutableDictionary alloc] init];
                                    [checkModel.restFieldDict setObject:fieldValueSetDict forKey:cardID];
                                }
                                else [fieldValueSetDict retain];
                                
                                fieldValueSet = [fieldValueSetDict objectForIntKey:fieldType];
                                
                                if(!fieldValueSet)
                                {
                                    fieldValueSet = [[NSMutableSet alloc] init];
                                    [fieldValueSetDict setObject:fieldValueSet forIntKey:fieldType];
                                }
                                else [fieldValueSet retain];
                                
                                [fieldValueSet addObject:fieldValue];
                                [fieldValueSet release];
                                [fieldValueSetDict release];
                                break;
                            }
                        }
                    }
                }
                
                sqlite3_finalize(stmt);
            }
            else
            {
#ifdef DumpDuplicateTime
                PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得所有名片的company, email, phone - failed");
#endif
                goto _EXIT;
            }
            
#ifdef DumpDuplicateTime
            PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得所有名片的company, email, phone end");
#endif
        }

        result = YES;
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
#ifdef DumpDuplicateTime
        [PPLogController stopLogCostTime:logKey];
#endif
        if(!result)
        {
            [checkModel release];
            checkModel = nil;
        }
        
        
        return checkModel;
    }
}


//===============================================================================
// compare all card and get duplicate data
//
// RETURNED: array of cardID
//
// groupedCardArray -> cardArray -> cardID
//                                  cardID
//                                  ...
//                     cardArray -> cardID
//                                  cardID
//                                  ...
//
// NOTE: duplicate判斷條件
//       1. FirstName + LastName 相同 (main key)
//       2. Company or Email or Phone 有一個是相同的
//===============================================================================
- (NSMutableArray *)copySameNameCardIDs
{
    @synchronized (self)
    {
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        NSMutableArray      *duplicateCardIDArray = nil;
        NSMutableArray      *groupedCardIDArray = [[NSMutableArray alloc] init];
        
        NSString            *errorStep = nil;
        BOOL				result = NO;
        
        WCDuplicateCompareModel *compareModel = nil;
        WCDuplicateCheckNameModel *nameModel = nil;
        
        // clear cancel flag
        [self clearCancelFlag];
        
#ifdef DumpDuplicateTime
        NSString *logKey = @"copyDuplicateCardIDs";
        PPLog_StartLogCostTime(logKey);
#endif
        
        //-----------------------------------------------
        // 1. 取得主要比對資料
        //-----------------------------------------------
        compareModel = [self copyDuplicateCompareModelFromDBWithNameOnly:YES];
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得比對資料 end");
#endif
        if(!compareModel)
        {
            if([self isCanceled])
                errorStep = @"copyCheckModelFromDB canceled";
            else errorStep = @"copyCheckModelFromDB failed";
            
            goto _EXIT;
        }
        
        // 有2筆以上資料才需要執行比對
        if([compareModel.nameArray count] < 2)
        {
            result = YES;
            goto _EXIT;
        }
        
        //////////////////////////////////////////////////
        // !! 1。 姓名不同，依姓名排
        // !! 2。 姓名相同，依姓名個數排
        // !! 3。 姓名個數一樣，依cardID排
        
        [compareModel.nameArray sortUsingComparator:^NSComparisonResult(WCDuplicateCheckNameModel *obj1, WCDuplicateCheckNameModel *obj2) {
            
            if ([obj1.name compare:obj2.name options:NSCaseInsensitiveSearch]==NSOrderedSame)
            {
                // 姓名相同，依姓名個數排
                NSInteger obj1NameCount = [[[compareModel.restFieldDict objectForKey:obj1.cardID] objectForIntKey:WC_FT_Name] count];
                NSInteger obj2NameCount = [[[compareModel.restFieldDict objectForKey:obj2.cardID] objectForIntKey:WC_FT_Name] count];
                if (obj1NameCount==obj2NameCount)
                {
                    // 姓名個數一樣，依cardID排
                    return [obj1.cardID unicodeCompare:obj2.cardID];
                }
                else
                {
                    return (obj1NameCount>obj2NameCount)?NSOrderedAscending:NSOrderedDescending;
                }
            }
            else
            {
                // 姓名不同，依姓名排
                if([obj1.name length]==0)
                {
                    return NSOrderedAscending;
                }
                
                if([obj2.name length]==0)
                {
                    return NSOrderedAscending;
                }
                
                return [obj1.name unicodeCompare:obj2.name];
            }
        }];
        
#ifdef DumpDuplicateData
        PPLogController *nameCountLog = [[PPLogController alloc] init];
        [nameCountLog setFileName:@"name count" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
        nameCountLog.toFile = YES;
        nameCountLog.toConsole = NO;
        for (WCDuplicateCheckNameModel *nameModel in compareModel.nameArray)
        {
            NSInteger obj1NameCount = [[[compareModel.restFieldDict objectForKey:nameModel.cardID] objectForIntKey:WC_FT_Name] count];
            [nameCountLog logWithMask:PPLogControllerMask_Normal format:@"%@,%d,%@", nameModel.cardID, obj1NameCount, nameModel.name];
        }
        [nameCountLog release];
#endif

        //-----------------------------------------------
        // 2. 取得相同資料的cardID
        //-----------------------------------------------
#ifdef DumpDuplicateData
        PPLog(PPLogControllerMask_Normal, @"normal compare (%ld):", [compareModel.nameArray count]);
#endif
        while ([compareModel.nameArray count] > 0)
        {
            // 取得第一筆來比對
            nameModel = [[compareModel.nameArray objectAtIndex:0] retain];
            [compareModel.nameArray removeObjectAtIndex:0];
            
            // 比對並回傳重複的cardID array
            duplicateCardIDArray = [compareModel copySameNameCardIDArrayWithCheckNameModel:nameModel
                                                                    checkFieldValueSetDict:nil
                                                                            findStartIndex:NO];
            
            if([duplicateCardIDArray count])
            {
                // !! 有重複資料
                [groupedCardIDArray addObject:duplicateCardIDArray];
                [compareModel removeDataWithCardID:nameModel.cardID];
            }
            
            
            [duplicateCardIDArray release];
            duplicateCardIDArray = nil;
            
            [nameModel release];
            nameModel = nil;
        }
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"compare end");
#endif

        //---------------------------------------
        // finish with success
        //---------------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
#ifdef DumpDuplicateTime
        PPLog_StopLogCostTime(logKey);
#endif
        // delete buffer
        [compareModel release];
        
        if(!result || ![groupedCardIDArray count])
        {
            [groupedCardIDArray release];
            groupedCardIDArray = nil;
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        [pool release];
        
        return groupedCardIDArray;
    }
}

//===============================================================================
// compare all card and get duplicate data
//
// RETURNED: array of cardID
//
// groupedCardArray -> cardArray -> cardID
//                                  cardID
//                                  ...
//                     cardArray -> cardID
//                                  cardID
//                                  ...
//
// NOTE: duplicate判斷條件
//       1. FirstName + LastName 相同 (main key)
//       2. Company or Email or Phone 有一個是相同的
//===============================================================================
- (NSMutableArray *)copyDuplicateCardIDs
{
    @synchronized (self)
    {
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        NSMutableArray      *duplicateCardIDArray = nil;
        NSMutableArray      *groupedCardIDArray = [[NSMutableArray alloc] init];
        NSMutableArray      *fullEmptyCardModels = [[NSMutableArray alloc] init];
        
        NSString            *errorStep = nil;
        BOOL				result = NO;
        
        WCDuplicateCompareModel *compareModel = nil;
        WCDuplicateCheckNameModel *nameModel = nil;
        
        // clear cancel flag
        [self clearCancelFlag];
        
#ifdef DumpDuplicateTime
        NSString *logKey = @"copyDuplicateCardIDs";
        PPLog_StartLogCostTime(logKey);
#endif
        
        //-----------------------------------------------
        // 1. 取得主要比對資料
        //-----------------------------------------------
        compareModel = [self copyDuplicateCompareModelFromDBWithNameOnly:NO];
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"取得比對資料 end");
#endif
        if(!compareModel)
        {
            if([self isCanceled])
                errorStep = @"copyCheckModelFromDB canceled";
            else errorStep = @"copyCheckModelFromDB failed";
            
            goto _EXIT;
        }
        
        // 有2筆以上資料才需要執行比對
        if([compareModel.nameArray count] < 2)
        {
            result = YES;
            goto _EXIT;
        }
        
        //////////////////////////////////////////////////
        // !! 1。 姓名不同，依姓名排
        // !! 2。 姓名相同，依姓名個數排
        // !! 3。 姓名個數一樣，依cardID排
        // !! 4。 沒有姓名排在最後
        
        [compareModel.nameArray sortUsingComparator:^NSComparisonResult(WCDuplicateCheckNameModel *obj1, WCDuplicateCheckNameModel *obj2) {
            
            if ([obj1.name compare:obj2.name options:NSCaseInsensitiveSearch]==NSOrderedSame)
            {
                // 姓名相同，依姓名個數排
                NSInteger obj1NameCount = [[[compareModel.restFieldDict objectForKey:obj1.cardID] objectForIntKey:WC_FT_Name] count];
                NSInteger obj2NameCount = [[[compareModel.restFieldDict objectForKey:obj2.cardID] objectForIntKey:WC_FT_Name] count];
                if (obj1NameCount==obj2NameCount)
                {
                    // 姓名個數一樣，依cardID排
                    return [obj1.cardID unicodeCompare:obj2.cardID];
                }
                else
                {
                    return (obj1NameCount>obj2NameCount)?NSOrderedAscending:NSOrderedDescending;
                }
            }
            else
            {
                // 姓名不同，依姓名排
                if([obj1.name length]==0)
                {
                    return NSOrderedAscending;
                }
                
                if([obj2.name length]==0)
                {
                    return NSOrderedAscending;
                }
                
                return [obj1.name unicodeCompare:obj2.name];
            }
        }];
        
#ifdef DumpDuplicateData
        PPLogController *nameCountLog = [[PPLogController alloc] init];
        [nameCountLog setFileName:@"name count" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
        nameCountLog.toFile = YES;
        nameCountLog.toConsole = NO;
        for (WCDuplicateCheckNameModel *nameModel in compareModel.nameArray)
        {
            NSInteger obj1NameCount = [[[compareModel.restFieldDict objectForKey:nameModel.cardID] objectForIntKey:WC_FT_Name] count];
            [nameCountLog logWithMask:PPLogControllerMask_Normal format:@"%@,%d,%@", nameModel.cardID, obj1NameCount, nameModel.name];
        }
        [nameCountLog release];
#endif

        //-----------------------------------------------
        // 2. 取得相同資料的cardID
        //-----------------------------------------------
#ifdef DumpDuplicateData
        PPLog(PPLogControllerMask_Normal, @"normal compare (%ld):", [compareModel.nameArray count]);
#endif
        while ([compareModel.nameArray count] > 0)
        {
            // 取得第一筆來比對
            nameModel = [[compareModel.nameArray objectAtIndex:0] retain];
            [compareModel.nameArray removeObjectAtIndex:0];
            
            
            if([nameModel.name length]==0 && [compareModel isRestFieldEmpty:nameModel.cardID])
            {
                // !! 如果姓名，email, phone ,company都是空的
                // 取得所有欄位串起來的字串，後面再比對
                nameModel.allFieldValueString = [self fieldValueStringWithCardID:nameModel.cardID];
                [fullEmptyCardModels addObject:nameModel];
            }
            else
            {
                // 比對並回傳重複的cardID array
                duplicateCardIDArray = [compareModel copyDuplicateCardIDArrayWithCheckNameModel:nameModel
                                                                         checkFieldValueSetDict:nil
                                                                                 findStartIndex:NO];
                
                if([duplicateCardIDArray count])
                {
                    // !! 有重複資料
                    [groupedCardIDArray addObject:duplicateCardIDArray];
                    [compareModel removeDataWithCardID:nameModel.cardID];
                }
                
                
                [duplicateCardIDArray release];
                duplicateCardIDArray = nil;
            }
            
            [nameModel release];
            nameModel = nil;
        }
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"compare end");
#endif
        
#ifdef DumpDuplicateData
        PPLog(PPLogControllerMask_Normal, @"比對姓名，email, phone ,company都是空的資料 (%ld):", [fullEmptyCardModels count]);
#endif
        
        //-----------------------------------------------
        // 3. 比對姓名，email, phone ,company都是空的資料
        //-----------------------------------------------
        
        
#ifdef DumpDuplicateData
        for(NSUInteger i=0; i<[fullEmptyCardModels count]; i++)
        {
            WCDuplicateCheckNameModel *curNameModel = [fullEmptyCardModels objectAtIndex:i];
            
            PPLog(PPLogControllerMask_Normal, @"\t%@", curNameModel.cardID);
        }
#endif
        
        // 先排序, allFieldValueString空的排後面
        [fullEmptyCardModels sortUsingComparator:^NSComparisonResult(WCDuplicateCheckNameModel *obj1, WCDuplicateCheckNameModel *obj2) {
            
            if ([obj1.allFieldValueString length]==0)
            {
                return NSOrderedDescending;
            }
            if ([obj2.allFieldValueString length]==0)
            {
                return NSOrderedAscending;
            }
            
            return [obj1.allFieldValueString unicodeCompare:obj2.allFieldValueString];
        }];
        
        
        // 比對是否有重複
        while ([fullEmptyCardModels count] > 1)
        {
            nameModel = [[fullEmptyCardModels objectAtIndex:0] retain];
            [fullEmptyCardModels removeObjectAtIndex:0];
            
            
            //////////////////////////////////////////////////
            // !! 因為已經排序過，如果不同就不用再往後面比
            BOOL duplicate = NO;
            for(NSUInteger i=0; i<[fullEmptyCardModels count]; i++)
            {
                WCDuplicateCheckNameModel *curNameModel = [fullEmptyCardModels objectAtIndex:i];
                
                if(([nameModel.allFieldValueString length]>0 &&
                    [nameModel.allFieldValueString isEqualToString:curNameModel.allFieldValueString])||
                   ([nameModel.allFieldValueString length]==0 &&
                    [curNameModel.allFieldValueString length]==0))
                {
                    if (duplicateCardIDArray==nil)
                    {
                        duplicateCardIDArray = [[NSMutableArray alloc] init];
                    }
                    
                    //  如果allFieldValueString一樣，才算重複，記錄下來
                    // 把判斷完的nameModel從fullEmptyCardModels移除
                    if([duplicateCardIDArray containsObject:nameModel.cardID]==NO)
                    {
                        [duplicateCardIDArray addObject:nameModel.cardID];
                    }
                    
                    [duplicateCardIDArray addObject:curNameModel.cardID];
                    [fullEmptyCardModels removeObjectAtIndex:i];
                    i--;
                    
                    duplicate = YES;
                }
                else
                {
                    break;
                }
            }
            
            if(duplicate)
            {
                // !! 有重複資料
                [duplicateCardIDArray sortUsingComparator:^NSComparisonResult(NSString *obj1, NSString *obj2) {
                    
                    return [obj1 unicodeCompare:obj2];
                }];
                
                [groupedCardIDArray addObject:duplicateCardIDArray];
            }
            
            [duplicateCardIDArray release];
            duplicateCardIDArray = nil;
            
            [nameModel release];
            nameModel = nil;
        }
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"比對姓名，email, phone ,company都是空的資料 end");
#endif
        //---------------------------------------
        // finish with success
        //---------------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
#ifdef DumpDuplicateTime
        PPLog_StopLogCostTime(logKey);
#endif
        // delete buffer
        [fullEmptyCardModels release];
        [compareModel release];
        
        if(!result || ![groupedCardIDArray count])
        {
            [groupedCardIDArray release];
            groupedCardIDArray = nil;
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        [pool release];
        
        return groupedCardIDArray;
    }
}


//===============================================================================
// compare all card and get duplicate data
//
// RETURNED: array of cardModel array, sorted by modified time
//
// groupedCardArray -> cardArray -> cardModel
//                                  cardModel
//                                  ...
//                     cardArray -> cardModel
//                                  cardModel
//                                  ...
//
// NOTE: duplicate判斷條件
//       1. FirstName + LastName 相同 (main key)
//       2. Company or Email or Phone 有一個是相同的
//===============================================================================
- (NSMutableArray *)copyDuplicateCardsWithSameNameOnly:(BOOL)sameNameOnly
{
    @synchronized (self)
    {
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        NSMutableArray      *groupedCardIDArray = nil;
        NSMutableArray      *groupedCardArray = [[NSMutableArray alloc] init];
        NSMutableArray      *cardArray = nil;
        
        WCCardModel         *cardModel = nil;
        NSString            *errorStep = nil;
        BOOL				result = NO;
        
        
        // clear cancel flag
        [self clearCancelFlag];
        
#ifdef DumpDuplicateTime
#ifdef PPLogMacro
        NSString *logKey = @"copyDuplicateCards";
#endif
        PPLog_StartLogCostTime(logKey);
#endif
        if(sameNameOnly)
        {
            groupedCardIDArray = [self copySameNameCardIDs];
        }
        else
        {
            groupedCardIDArray = [self copyDuplicateCardIDs];
        }
        
#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"重複聯絡人比對完成時間");
#endif
       
        //-----------------------------------------------
        // 1. 取得所有資料並排序
        //-----------------------------------------------
#ifdef DumpDuplicateData
        NSInteger groupID = 0;

        PPLogController *resultLog = [[PPLogController alloc] init];
        [resultLog setFileName:@"resultWithoutSort" atPath:[WCToolController baseStorePathWithDirName:@"DuplicateData" isCreatDirPath:YES]];
        resultLog.toFile = YES;
        resultLog.toConsole = NO;
#endif

#ifdef DumpDuplicateData
#endif
        for(NSArray *duplicateCardIDArray in groupedCardIDArray)
        {
#ifdef DumpDuplicateData
            NSInteger cardIDIndex = 0;
#endif

            cardArray = [[NSMutableArray alloc] init];
            [groupedCardArray addObject:cardArray];
            [cardArray release];
            
            for(NSString *cardID in duplicateCardIDArray)
            {

                if([self isCanceled])
                {
                    errorStep = @"cancel copyCardWithCardID loop";
                    goto _EXIT;
                }
                
                if((cardModel = [self copyCardWithCardID:cardID getAllField:NO]))
                {
#ifdef DumpDuplicateData
                    if(cardIDIndex==0)
                    {
//                        [resultLog logWithMask:PPLogControllerMask_Normal format:@"Group %ld: - %@", groupID++, [cardModel.displayName length]>0?cardModel.displayName:@"沒有姓名"];
                        [resultLog logWithMask:PPLogControllerMask_Normal format:@"Group %ld", groupID++];
                    }

                    [resultLog logWithMask:PPLogControllerMask_Normal format:@"\t%@", cardID];

#endif
                    [cardArray addObject:cardModel];
                    [cardModel release];
                }
                else
                {
                    errorStep = @"copyCardWithCardID failed";
                    goto _EXIT;
                }
                
#ifdef DumpDuplicateData
                cardIDIndex++;
#endif
            }
            
            // sort by modified time
            [cardArray sortUsingFunction:WCTCompareModifiedTime context:nil];
        }
        
#ifdef DumpDuplicateData
        [resultLog release];
#endif

#ifdef DumpDuplicateTime
        PPLog_LogCostTime(logKey, PPLogControllerMask_Normal, @"依時間排序結果 end");
#endif
        //---------------------------------------
        // finish with success
        //---------------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
#ifdef DumpDuplicateTime
        PPLog_StopLogCostTime(logKey);
#endif
        // delete buffer
        [groupedCardIDArray release];
        
        if(!result || ![groupedCardArray count])
        {
            [groupedCardArray release];
            groupedCardArray = nil;
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        [pool release];
        
        return groupedCardArray;
    }
}






#pragma mark - Cancel methods

//===============================================================================
// private
//===============================================================================
- (void)clearCancelFlag
{
    self.isCancelDBAction = NO;
    //    NSMutableDictionary *threadDict = [[NSThread mainThread] threadDictionary];
    //    [threadDict removeObjectForKey:WCTCDBC_ThreadDict_kInterrupt];
}


//===============================================================================
// private
//===============================================================================
- (BOOL)isCanceled
{
    //test
    //    NSLog(@"%s : %p, %d", __func__, [NSThread currentThread], self.isCancelDBAction);
    
    return self.isCancelDBAction;
    
    //    @synchronized(self)
    //    {
    //        NSMutableDictionary *threadDict = [[NSThread mainThread] threadDictionary];
    //        BOOL result = ([threadDict objectForKey:WCTCDBC_ThreadDict_kInterrupt] != nil);
    //
    //        //test
    //        NSLog(@"%s : %p, %d", __func__, [NSThread currentThread], result);
    //
    //        return result;
    //    }
}


//===============================================================================
//
//===============================================================================
- (void)cancel
{
    
    //test
    //    NSLog(@"%s : %p, %d", __func__, [NSThread currentThread], self.isCancelDBAction);
    
    self.isCancelDBAction = YES;
    
    
    //    @synchronized(self)
    //    {
    //        NSMutableDictionary *threadDict = [[NSThread mainThread] threadDictionary];
    //        [threadDict setObject:[NSNull null] forKey:WCTCDBC_ThreadDict_kInterrupt];
    //    }
}





#pragma mark - Other methods

//===============================================================================
//
//===============================================================================
- (BOOL)updateModifiedTimeWithCardID:(NSString *)cardID withSynchronized:(BOOL)synchronized
{
    if (synchronized)
    {
        @synchronized(self)
        {
            NSString *sqlCommand;
            BOOL result = NO;
            
            sqlCommand =[NSString stringWithFormat:@"UPDATE WCCard SET ModifiedTime=%f WHERE CardID='%@'",
                         [[NSDate date] timeIntervalSince1970],
                         [super sqlString:cardID]];
            
            result = [super runSqlCommand:sqlCommand];
            
            return result;
        }
    }
    else
    {
        NSString *sqlCommand;
        BOOL result = NO;
        
        sqlCommand =[NSString stringWithFormat:@"UPDATE WCCard SET ModifiedTime=%f WHERE CardID='%@'",
                     [[NSDate date] timeIntervalSince1970],
                     [super sqlString:cardID]];
        
        result = [super runSqlCommand:sqlCommand];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateModifiedTimeWithCardID:(NSString *)cardID
{
    return [self updateModifiedTimeWithCardID:cardID withSynchronized:YES];
}





#pragma mark - Check and recover methods


//===============================================================================
// result NSNumber array
//===============================================================================
- (NSMutableArray *)copyAllInvalidPhoneticFieldIDWithFieldSubType2:(NSInteger)fieldSubType2
{
    NSMutableArray  *fieldIDArray = nil;
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString        *sqlCommand = [NSString stringWithFormat:@"SELECT FieldID, ValueString FROM WCCardField WHERE FieldSubType2='%@'", @(fieldSubType2)];
        NSInteger       fieldID;
        fieldIDArray = [[NSMutableArray alloc] init];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                fieldID = [super integerFromStmt:stmt column:0];
                NSString *valueString = [super stringFromStmt:stmt column:1];
                
                // 檢查內容是否有不合法字元
                if ([valueString hasInvalidCharacter])
                {
                    [fieldIDArray addObject:@(fieldID)];
                }
            }
            
            sqlite3_finalize(stmt);
        }
    }
    
    return fieldIDArray;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - address book methods


//==============================================================================
//
//==============================================================================
- (NSString *)cardIDWithABPersonID:(NSString *)abPersonID
{
    @synchronized(self)
    {
        NSString *personID = nil;
        
        personID = abPersonID;
        
        if([personID length] == 0)
        {
            return nil;
        }
        
        //////////////////////////////////////////////////
        
        NSString *cardID = nil;
        sqlite3_stmt *stmt = NULL;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCAddressBookMapping WHERE ABPersonID = '%@'", personID];
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //////////////////////////////////////////////////
                // check if interrupt
                
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    break;
                }
                
                
                //////////////////////////////////////////////////

                // get result
                cardID = [super stringFromStmt:stmt column:0];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return cardID;
    }
}


//==============================================================================
// insertCard用的，不用@synchronized包
//==============================================================================
- (BOOL)updateAddressBookDataWithCardModel:(WCABCardModel *)cardModel
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *cardID = [super sqlString:cardModel.ID];
        
        // !! 如果cardID不存在就不新增，避免空資料
        if([self isCardIDExist:cardID withSynchronized:NO]==NO)
        {
            return NO;
        }
        
        NSString *sqlCommand = nil;
        sqlCommand = [NSString stringWithFormat:@"SELECT count(*) FROM WCAddressBookMapping WHERE CardID = '%@'", [super sqlString:cardID]];
        NSInteger count = [self intResultWithSelectCommand:sqlCommand];
        if (count==-1)
        {
            [self setLastErrorWithErrorStep:@"updateAddressBookDataWithCardModel: get address book mapping count failed"];
            return result;
        }
        
        NSString *personIDString = cardModel.abPersonID;
        NSString *sourceIDString = cardModel.abSourceID;
        
#if TARGET_OS_IPHONE

        NSString *groupIDString = cardModel.abGroupID;
        
#elif TARGET_OS_MAC

        NSString *groupIDString = [cardModel.parentGroupIDs componentsJoinedByString:WCTCardDBController_DefaultSeperator];
#endif

        if (count>0)
        {
            // update
            //--------------------------------
            // run command
            //--------------------------------
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCAddressBookMapping SET ABPersonID='%@', ABSourceID='%@', ABGroupID='%@' WHERE CardID='%@'",
                          [super sqlString:personIDString],
                          [super sqlString:sourceIDString],
                          [super sqlString:groupIDString],
                          [super sqlString:cardID]];
            
        }
        else
        {
            //--------------------------------
            // run command
            //--------------------------------
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCAddressBookMapping(CardID,ABPersonID,ABSourceID,ABGroupID) VALUES('%@','%@','%@','%@')",
                          [super sqlString:cardID],
                          [super sqlString:personIDString],
                          [super sqlString:sourceIDString],
                          [super sqlString:groupIDString]
                          ];
        }
        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"updateAddressBookDataWithCardModel: update mapping data failed"];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeAddressBookDataWithCardID:(NSString*)cardID
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *sqlCommand = nil;
        
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCAddressBookMapping WHERE CardID='%@'", [super sqlString:cardID]];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"removeAddressBookDataWithCardID: delete failed"];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)getAddressBookDataWithCardModel:(WCABCardModel *)abCardModel
{
    @synchronized(self)
    {
        if (abCardModel==nil)
        {
            [self setLastErrorWithErrorStep:@"getABCardModel with nil abCardModel"];
            return NO;
        }
        
        BOOL result = NO;
        NSString *sqlCommand = nil;
        sqlite3_stmt *stmt;
        
        if([abCardModel.ID length] > 0 && [abCardModel.abPersonID length] == 0)
        {
            //////////////////////////////////////////////////
            // 有AddressBook的personID，找出cardID。
            
            sqlCommand = [NSString stringWithFormat:@"SELECT ABPersonID, ABSourceID, ABGroupID FROM WCAddressBookMapping WHERE CardID = '%@'", [super sqlString:abCardModel.ID]];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    if([self isCanceled])
                    {
                        sqlite3_interrupt(self.dbHandle);
                        break;
                    }
                    
                    int column = 0;
                    NSString *personID = [super stringFromStmt:stmt column:column++];
                    NSString *sourceID = [super stringFromStmt:stmt column:column++];
                    NSString *groupID = [super stringFromStmt:stmt column:column++];
                    
                    abCardModel.abPersonID = personID;
                    abCardModel.abSourceID = sourceID;
                    
#if TARGET_OS_IPHONE
                    abCardModel.abGroupID = groupID;
#elif TARGET_OS_MAC
                    abCardModel.parentGroupIDs = [groupID componentsSeparatedByString:WCTCardDBController_DefaultSeperator];
#endif
                }
                
                sqlite3_finalize(stmt);
                result = YES;
            }
        }
        else if([abCardModel.ID length] == 0 && [abCardModel.abPersonID length] > 0)
        {
            //////////////////////////////////////////////////
            // 有cardID，找出AddressBook的personID。
            
            sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCAddressBookMapping WHERE ABPersonID = '%@'", [super sqlString:abCardModel.abPersonID]];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    if([self isCanceled])
                    {
                        sqlite3_interrupt(self.dbHandle);
                        break;
                    }
                    
                    abCardModel.ID = [super stringFromStmt:stmt column:0];
                }
                
                sqlite3_finalize(stmt);
                
                result = ([abCardModel.ID length]>0) ? YES : NO;
            }
        }
        else if([abCardModel.ID length] > 0 && [abCardModel.abPersonID length] > 0)
        {
            //////////////////////////////////////////////////
            // 兩個都有就不找
            
            result = YES;
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)clearAddressBookData
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *sqlCommand = nil;
        
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCAddressBookMapping"];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"clearAddressBookData: delete failed"];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (NSArray *)allAddressBookData
{
    @synchronized(self)
    {
        NSMutableArray *abCards = [NSMutableArray array];
        sqlite3_stmt *stmt;
        NSString *sqlCommand = @"SELECT CardID,ABPersonID,ABSourceID,ABGroupID FROM WCAddressBookMapping";
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    break;
                }
                
                //////////////////////////////////////////////////
                
                WCABCardModel *abCardModel = [[WCABCardModel alloc] init];
                
                if(abCardModel != nil)
                {
                    int column = 0;
                    abCardModel.ID = [super stringFromStmt:stmt column:column++];
                    abCardModel.abPersonID = [super stringFromStmt:stmt column:column++];
                    abCardModel.abSourceID = [super stringFromStmt:stmt column:column++];
                    
                    NSString *groupID = [super stringFromStmt:stmt column:column++];
                    
#if TARGET_OS_IPHONE
                    abCardModel.abGroupID = groupID;
#elif TARGET_OS_MAC
                    abCardModel.parentGroupIDs = [groupID componentsSeparatedByString:WCTCardDBController_DefaultSeperator];
#endif
                    
                    [abCards addObject:abCardModel];
                    [abCardModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        
        return abCards;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - AddressBook convert to Contacts framework methods

//==============================================================================
//
//==============================================================================
- (NSArray <NSString *>*)allCardIDFromAddressBookMapping
{
    @synchronized(self)
    {
        NSMutableArray *result = nil;
        sqlite3_stmt *stmt = NULL;
        NSString *sqlCommand = @"SELECT CardID FROM WCAddressBookMapping";
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //////////////////////////////////////////////////
                // check if interrupt
                
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    break;
                }
                
                
                //////////////////////////////////////////////////
                
                // get result
                NSString *cardID = [super stringFromStmt:stmt column:0];
                
                if (cardID)
                {
                    if (result==nil)
                    {
                        result = [NSMutableArray array];
                    }
                    
                    [result addObject:cardID];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        
        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - recent contact methods

//==============================================================================
//
//==============================================================================
- (BOOL)insertRecentAction:(NSInteger)action withContent:(NSString *)content cardID:(NSString *)cardID
{
    @synchronized(self)
    {
        BOOL result = NO;
        
        if ([cardID length]==0)
        {
            [self setLastErrorWithErrorStep:@"insertRecentAction: [cardID length]==0"];
            return result;
        }
        
        if ([content length]==0)
        {
            [self setLastErrorWithErrorStep:@"insertRecentAction: [content length]==0"];
            return result;
        }
        
        // !! 如果cardID不存在就不新增，避免空資料
        if([self isCardIDExist:cardID withSynchronized:NO]==NO)
        {
            return NO;
        }
        
        // 同一個聯絡人只紀錄一個動作
        //////////////////////////////////////////////////
        NSString *sqlCommand = nil;
        //先移除舊的
        sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM WCRecentContacts WHERE CardID = '%@'", [super sqlString:cardID]];
        
        NSInteger count  = [self intResultWithSelectCommand:sqlCommand];
        
        //--------------------------------
        // run command
        //--------------------------------
        if(count>0)
        {
            // update
            
            sqlCommand =
            [NSString stringWithFormat:@"UPDATE WCRecentContacts SET ActionType=%ld,ActionContent='%@',ActionTime=%f WHERE CardID = '%@'",
             (long)action,
             [super sqlString:content],
             [[NSDate date] timeIntervalSince1970],
             [super sqlString:cardID]];
        }
        else
        {
            // insert
            sqlCommand =
            [NSString stringWithFormat:@"INSERT INTO WCRecentContacts(CardID,ActionType,ActionContent,ActionTime) VALUES('%@',%ld,'%@',%f)",
             [super sqlString:cardID],
             (long)action,
             [super sqlString:[NSString stringWithString:content]],
             [[NSDate date] timeIntervalSince1970]];
        }

        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"insertRecentAction: insert failed"];
        
        return result;
    }
}

- (BOOL)removeRecentActionWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *sqlCommand = nil;
        
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCRecentContacts WHERE CardID = '%@'", [super sqlString:cardID]];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"removeRecentActionWithCardID: delete failed"];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyAllRecentContacts
{
    @synchronized(self)
    {
        NSMutableArray *result = nil;
        NSString *sqlCommand = nil;
        sqlite3_stmt		*stmt;
        
        //--------------------------------
        // run command
        //--------------------------------
        sqlCommand = [NSString stringWithFormat:@"SELECT * FROM WCRecentContacts ORDER BY ActionTime"];
        
        
        //-------------------------------------------------
        // start search
        //-------------------------------------------------
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                ////////////////////////////////////
                // check if interrupt
                if([self isCanceled])
                {
                    sqlite3_interrupt(self.dbHandle);
                    break;
                }
                
                
                ////////////////////////////////////
                // get result
                if (result==nil)
                {
                    result = [[NSMutableArray alloc] init];
                }
                
                int column = 0;
                WCRecentCardModel *recentCardModel = [[WCRecentCardModel alloc] init];
                recentCardModel.ID = [super stringFromStmt:stmt column:column++];
                recentCardModel.actionType = [super integerFromStmt:stmt column:column++];
                recentCardModel.content = [super stringFromStmt:stmt column:column++];
                recentCardModel.actionTime = [self dateFromStmt:stmt column:column++];

                //////////////////////////////////////////////////
                
                [result addObject:recentCardModel];
                [recentCardModel release];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)clearRecentContactData
{
    @synchronized(self)
    {
        BOOL result = NO;
        NSString *sqlCommand = nil;
        
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCRecentContacts"];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(!result)
            [self setLastErrorWithErrorStep:@"clearRecentContactData: delete failed"];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasRecentContacts
{
    @synchronized(self)
    {
        NSString *sqlCommand = @"SELECT COUNT(*) FROM WCRecentContacts";
        NSInteger count = [super intResultWithSelectCommand:sqlCommand];
        
        return (count>0)?YES:NO;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - 以下為WCT新增或調整函式
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Account list methods

//==============================================================================
// 用的地方有synchronized，所以這邊不用
//==============================================================================
- (BOOL)removeAllAccount
{
    NSString *sqlCommand = @"DELETE FROM WCTAccountRelationship";

    return [self runSqlCommand:sqlCommand];
}



//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyAllAccountList
{
    return [self copyAllAccountListWithResign:NO];
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyAllAccountListWithResign:(BOOL)resign
{
    @synchronized(self)
    {
        sqlite3_stmt *stmt = NULL;
        NSMutableArray *accountList = [[NSMutableArray alloc] init];
        NSString *sqlCommand = @"SELECT * FROM WCTAccountRelationship";
        
        if(resign==NO)
        {
            sqlCommand = [sqlCommand stringByAppendingString:@" WHERE IsOutgoingEmployee=0"];
        }
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                WCTAccountRelationModel *accountRelationModel = [[WCTAccountRelationModel alloc] init];
                
                if(accountRelationModel != nil)
                {
                    accountRelationModel.guid = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.name = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.email = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.relation = [super integerFromStmt:stmt column:column++];
                    accountRelationModel.isOutgoingEmployee = (BOOL)[super integerFromStmt:stmt column:column++];
                    accountRelationModel.isLimitedAccount = (BOOL)[super integerFromStmt:stmt column:column++];

                    [accountList addObject:accountRelationModel];
                    [accountRelationModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyAllAccountList : Failed to run sql command"];
            [accountList release];
            return nil;
        }
        
        return accountList;
    }
}



//==============================================================================
//
//==============================================================================
- (BOOL)updateAllAccountWithList:(NSArray *)list
{
    @synchronized (self)
    {
        // 先清除全部
        [self removeAllAccount];

        //////////////////////////////////////////////////
        // 更新
        BOOL result = YES;
        
        [self beginTransaction];
        
        for (WCTAccountRelationModel *accountRelationModel in list)
        {
            NSString *sqlCommand =
            [NSString stringWithFormat:@"INSERT INTO WCTAccountRelationship(AccountGUID,AccountName,EMail,Relationship,IsOutgoingEmployee,IsLimitedAccount)\
             VALUES('%@', '%@','%@',%ld,%ld,%ld)",
             [super sqlString:accountRelationModel.guid],
             [super sqlString:accountRelationModel.name],
             [super sqlString:accountRelationModel.email],
             (long)accountRelationModel.relation,
             (long)accountRelationModel.isOutgoingEmployee,
             (long)accountRelationModel.isLimitedAccount];
            
            result = [self runSqlCommand:sqlCommand];
            
            if (result==NO)
            {
                break;
            }
        }
        
        if(result)
        {
            result = [super commitTransaction];
        }
        else
        {
            [super rollbackTransaction];
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (WCTAccountRelationModel *)accountRelationWithAcountGuid:(NSString *)accountGuid
{
    @synchronized(self)
    {
        if ([accountGuid length]==0)
        {
            [self setLastErrorWithErrorStep:@"accountRelationWithAcountGuid : empty accountGuid"];
            return nil;
        }
        
        //////////////////////////////////////////////////
        
        sqlite3_stmt *stmt = NULL;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT * FROM WCTAccountRelationship WHERE AccountGUID='%@'",
                                [super sqlString:accountGuid]];
        
        WCTAccountRelationModel *accountRelationModel = nil;
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                accountRelationModel = [[[WCTAccountRelationModel alloc] init] autorelease];
                
                if(accountRelationModel != nil)
                {
                    accountRelationModel.guid = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.name = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.email = [super stringFromStmt:stmt column:column++];
                    accountRelationModel.relation = [super integerFromStmt:stmt column:column++];
                    accountRelationModel.isOutgoingEmployee = (BOOL)[super integerFromStmt:stmt column:column++];
                    accountRelationModel.isLimitedAccount = (BOOL)[super integerFromStmt:stmt column:column++];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"accountRelationWithAcountGuid : Failed to run sql command"];
            return nil;
        }
        
        return accountRelationModel;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Shared account methods


//==============================================================================
//
//==============================================================================
- (NSString *)sharedAccountStringWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString        *sqlCommand = [NSString stringWithFormat:@"SELECT AccountGUIDs FROM WCTCardSharedAccount WHERE CardID='%@'", [super sqlString:cardID]];
        NSString *sharedAccountString = nil;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                sharedAccountString = [self stringFromStmt:stmt column:0];
            }
            sqlite3_finalize(stmt);
        }

        return sharedAccountString;
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)ownerIDWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt	*stmt;
        NSString        *sqlCommand = [NSString stringWithFormat:@"SELECT Owner FROM WCCard WHERE CardID='%@'", [super sqlString:cardID]];
        NSString *sharedAccountString = nil;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                sharedAccountString = [self stringFromStmt:stmt column:0];
            }
            sqlite3_finalize(stmt);
        }
        
        return sharedAccountString;
    }
}

//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copySharedAccountWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSMutableArray *sharedAccountGuidArray = nil;
        NSString *sharedAccountString = [self sharedAccountStringWithCardID:cardID];
        NSArray *separatedArray = [sharedAccountString componentsSeparatedByString:@","];

        if([separatedArray count]>0)
        {
            sharedAccountGuidArray = [[NSMutableArray alloc] initWithArray:separatedArray];
        }
        
        return sharedAccountGuidArray;
    }
}


//===============================================================================
// private
//===============================================================================
- (BOOL)removeCardSharedAccountGuidsWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCTCardSharedAccount WHERE CardID='%@'", [super sqlString:cardID]];
        
        BOOL result = [super runSqlCommand:sqlCommand];
        
        //        if(result)
        //        {
        //            /// 更新WCTGroupSyncActionModel
        //            [self updateGroupSyncAction:syncActionModel isInTransaction:YES];
        //        }
        return result;
    }
}


//===============================================================================
// PARAMETER: sharedAccountGuids==nil時相當於要把可檢視清除，正常來說應該會至少有自已
//===============================================================================
- (BOOL)setCardSharedAccountGuids:(NSArray *)sharedAccountGuids withCardID:(NSString *)cardID isInTransaction:(BOOL)isInTransaction updateModifiedTime:(BOOL)updateModifiedTime
{
    @synchronized(self)
    {
        // sharedAccountGuids 為空時就不處理，當成功
        if([sharedAccountGuids count]==0)
        {
            return YES;
        }
        
        
        BOOL result = NO;
        NSString *sqlCommand = nil;
        NSString *errorStep = nil;
        
        
        self.lastError = nil;
        
        if(!isInTransaction)
            [super beginTransaction];
        

        NSString *ownerID = [self ownerIDWithCardID:cardID];
        
        //--------------------------------
        // get original shared account guid
        //--------------------------------
        NSString *sharedAccountString = [self sharedAccountStringWithCardID:cardID];
        
        if ([sharedAccountString length]>0)
        {
            // 原本就有share account
            NSString *appendingShareAccount = @"";
            for(NSString *sharedAccountGuid in sharedAccountGuids)
            {
                // !! 不能重複，也不能是owner
                if ([sharedAccountString containsString:sharedAccountGuid]==NO &&
                    [sharedAccountGuid compare:ownerID options:NSCaseInsensitiveSearch]!=NSOrderedSame)
                {
                    appendingShareAccount = [appendingShareAccount stringByAppendingFormat:@",%@",sharedAccountGuid];
                }
            }
            
            if([appendingShareAccount length]>0)
            {
                sharedAccountString = [sharedAccountString stringByAppendingString:appendingShareAccount];

                sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSharedAccount SET AccountGUIDs='%@' WHERE CardID='%@'",
                              [super sqlString:sharedAccountString],
                              [super sqlString:cardID]];
            }
        }
        else
        {
            // 原本沒有share account
            NSString *appendingShareAccount = @"";
           for(NSString *sharedAccountGuid in sharedAccountGuids)
            {
                if ([sharedAccountGuid compare:ownerID options:NSCaseInsensitiveSearch]!=NSOrderedSame)
                {
                    if([appendingShareAccount length]>0)
                    {
                        appendingShareAccount = [appendingShareAccount stringByAppendingString:@","];
                    }
                    appendingShareAccount = [appendingShareAccount stringByAppendingString:sharedAccountGuid];
                }
            }
     
            if([appendingShareAccount length]>0)
            {
                
                sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCTCardSharedAccount(CardID,AccountGUIDs) VALUES('%@','%@')",
                              [super sqlString:cardID],
                              [super sqlString:appendingShareAccount]];
            }
        }

        if ([sqlCommand length]==0)
        {
            result = YES;
            goto _EXIT;
        }
        
        if(![super runSqlCommand:sqlCommand])
        {
            errorStep = @"setCardSharedAccountGuids : insert into WCTCardSharedAccount failed";
            goto _EXIT;
        }
        
        //--------------------------------
        // update last modified time
        //--------------------------------
        if(updateModifiedTime)
            [self updateModifiedTimeWithCardID:cardID withSynchronized:NO];
        
        
        //--------------------------------
        // finish with success
        //--------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        if(!isInTransaction)
        {
            if(result)
            {
                if(!(result = [super commitTransaction]))
                {
                    errorStep = @"setCardSharedAccountGuids : commit failed";
                }
            }
            else
            {
                [super rollbackTransaction];
            }
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)setCardSharedAccountGuids:(NSArray *)sharedAccountGuids withCardID:(NSString *)cardID
{
    @synchronized (self)
    {
        [self beginTransaction];
        BOOL result = [self setCardSharedAccountGuids:sharedAccountGuids withCardID:cardID isInTransaction:YES updateModifiedTime:NO];
        
        if (result==YES)
        {
            result = [self updateCardSyncActionForSharedAccountChangeWithCardID:cardID modifiedTime:[NSDate date]];
            
            if(result)
            {
                result = [self commitTransaction];
            }
            else
            {
                [self rollbackTransaction];
            }
            return result;
        }
        
        return NO;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)setCardSharedAccountGuids:(NSArray *)sharedAccountGuids withCardIDs:(NSArray *)cardIDs
{
    @synchronized (self)
    {
        BOOL result = NO;
        NSString        *errorStep = nil;
        
        [self beginTransaction];
        
        for (NSString *cardID in cardIDs)
        {
            @autoreleasepool
            {
                // 不存在就不加
                if ([self isCardIDExist:cardID]==NO)
                {
                    continue;
                }
                result = [self setCardSharedAccountGuids:sharedAccountGuids withCardID:cardID isInTransaction:YES updateModifiedTime:NO];
                
                if (result==YES)
                {
                    result = [self updateCardSyncActionForSharedAccountChangeWithCardID:cardID modifiedTime:[NSDate date]];
                }
                
                if (result==NO)
                {
                    errorStep = @"setCardSharedAccountGuids:withCardIDs: set sharedAccountGuids failed";
                    break;
                }
            }
        }
        
        if(result)
        {
            if(!(result = [super commitTransaction]))
            {
                errorStep = @"setCardSharedAccountGuids:withCardIDs:: commit failed";
            }
        }
        else
        {
            [self rollbackTransaction];
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];

        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Custom field

//===============================================================================
/// 取得所有自訂欄位資料
//===============================================================================
- (NSMutableArray *)copyAllCustomFieldInfos
{
    @synchronized(self)
    {
        NSMutableArray *customFieldInfos = nil;
        
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT * FROM WCCustomFieldInfo"];
        sqlite3_stmt *stmt;
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if(customFieldInfos==nil)
            {
                customFieldInfos = [[NSMutableArray alloc] init];
            }
            
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int colunm = 0;
                
                WCCustomFieldInfo *customFieldInfo =[[WCCustomFieldInfo alloc] init];
                
                customFieldInfo.guid = [super stringFromStmt:stmt column:colunm++];
                customFieldInfo.name = [super stringFromStmt:stmt column:colunm++];
                customFieldInfo.category = [super integerFromStmt:stmt column:colunm++];
                customFieldInfo.contentType = [super integerFromStmt:stmt column:colunm++];
                
                ///////////////////////////////////////////////////
                // 如果是picklist 另外取得listItem
                if (customFieldInfo.contentType==WCCustomFieldContentType_Picklist)
                {
                    customFieldInfo.picklistItems = [self customFieldListItemsWithGuid:customFieldInfo.guid];
                }
                
                [customFieldInfos addObject:customFieldInfo];
                [customFieldInfo release];
            }
            
            sqlite3_finalize(stmt);
        }
        
        return customFieldInfos;
    }
}

//===============================================================================
/// 依自訂欄位guid, 取得自訂欄位資料
//===============================================================================
- (WCCustomFieldInfo *)customFieldInfoWithGuid:(NSString *)guid
{
    @synchronized(self)
    {
        WCCustomFieldInfo *customFieldInfo = nil;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT * FROM WCCustomFieldInfo WHERE CustomFieldInfoGuid='%@'", [super sqlString:guid]];
        sqlite3_stmt *stmt;
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if(sqlite3_step(stmt) == SQLITE_ROW)
            {
                int colunm = 0;
                
                customFieldInfo = [[WCCustomFieldInfo alloc] init];

                customFieldInfo.guid = [super stringFromStmt:stmt column:colunm++];
                customFieldInfo.name = [super stringFromStmt:stmt column:colunm++];
                customFieldInfo.category = [super integerFromStmt:stmt column:colunm++];
                customFieldInfo.contentType = [super integerFromStmt:stmt column:colunm++];
                
                ///////////////////////////////////////////////////
                // 如果是picklist 另外取得listItem
                if (customFieldInfo.contentType==WCCustomFieldContentType_Picklist)
                {
                    customFieldInfo.picklistItems = [self customFieldListItemsWithGuid:guid];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        

        
        return [customFieldInfo autorelease];
    }
}

//===============================================================================
/// 新增一個自訂欄位資料
//===============================================================================
- (BOOL)addCustomFieldWithInfo:(WCCustomFieldInfo *)customFieldInfo
{
    @synchronized(self)
    {
        if (customFieldInfo==nil)
        {
            NSLog(@"customFieldInfo不能為空");
            return NO;
        }
        
        BOOL result = NO;
        NSString *errorStep = nil;
        
        do {
            NSString *sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCustomFieldInfo(CustomFieldInfoGuid,Name,CustomFieldCategory,CustomFieldContentType)\
                                    VALUES('%@','%@',%ld,%ld)",
                                    [super sqlString:customFieldInfo.guid],
                                    [super sqlString:customFieldInfo.name],
                                    (long)customFieldInfo.category,
                                    (long)customFieldInfo.contentType];
            
            if ([super runSqlCommand:sqlCommand]==NO)
            {
                errorStep = @"updateCustomFieldWithInfo : INSERT or UPDATE WCCustomFieldInfo failed";
                break;
            }
            
            //////////////////////////////////////////////////
            // 寫入picklist item
            if (customFieldInfo.picklistItems)
            {
                // !!以完全取代方式處理，所以先清除舊的，再將目前的加入
                sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCustomFieldListItem WHERE CustomFieldInfoGuid = '%@'",
                              [super sqlString:customFieldInfo.guid]];
                
                if ([super runSqlCommand:sqlCommand]==NO)
                {
                    errorStep = @"updateCustomFieldWithInfo : DELETE WCCustomFieldListItem failed";
                    break;
                }
                
                for (WCCustomFieldListItem *listItem in customFieldInfo.picklistItems)
                {
                    sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCustomFieldListItem(ItemGuid, CustomFieldInfoGuid, ItemText) VALUES('%@','%@','%@')",
                                  [super sqlString:listItem.guid],
                                  [super sqlString:customFieldInfo.guid],
                                  [super sqlString:listItem.itemText]];
                    
                    if ([super runSqlCommand:sqlCommand]==NO)
                    {
                        errorStep = [NSString stringWithFormat:@"updateCustomFieldWithInfo : INSERT INTO WCCustomFieldListItem failed ([%@] %@)", [super sqlString:listItem.guid], [super sqlString:listItem.itemText]];
                        break;
                    }
                }
            }
        } while (0);

        if(!result)
            [self setLastErrorWithErrorStep:errorStep];

        return result;
    }
}


//==============================================================================
// 如果已存在，就更新，如果不存在就新增
//==============================================================================
- (BOOL)updateCustomFieldWithInfo:(WCCustomFieldInfo *)customFieldInfo
{
    @synchronized(self)
    {
        if (customFieldInfo==nil)
        {
            NSLog(@"customFieldInfo不能為空");
            return NO;
        }

        NSString *errorStep = nil;
        [self removeCustomFieldWithGuid:customFieldInfo.guid];
        
        BOOL result = NO;

        do {
            [super beginTransaction];

            //////////////////////////////////////////////////
            // 先檢查此custom fiel是否存在
            
            NSString *sqlCommand = [NSString stringWithFormat:@"SELECT count(*) FROM WCCustomFieldInfo WHERE CustomFieldInfoGuid = '%@'", [super sqlString:customFieldInfo.guid]];
            
            BOOL isExist = [self intResultWithSelectCommand:sqlCommand];
            
            if (isExist)
            {
                sqlCommand = [NSString stringWithFormat:@"UPDATE WCCustomFieldInfo SET \
                              Name='%@',\
                              CustomFieldCategory=%ld,\
                              CustomFieldContentType=%ld \
                              WHERE CustomFieldInfoGuid = '%@'",
                              [super sqlString:customFieldInfo.name],
                              (long)customFieldInfo.category,
                              (long)customFieldInfo.contentType,
                              [super sqlString:customFieldInfo.guid]];
            }
            else
            {
                sqlCommand = [NSString stringWithFormat:@"INSERT INTO \
                              WCCustomFieldInfo(CustomFieldInfoGuid,Name,CustomFieldCategory,CustomFieldContentType)\
                              VALUES('%@','%@',%ld,%ld)",
                              [super sqlString:customFieldInfo.guid],
                              [super sqlString:customFieldInfo.name],
                              (long)customFieldInfo.category,
                              (long)customFieldInfo.contentType];
                
            }
            
            if ([super runSqlCommand:sqlCommand]==NO)
            {
                errorStep = @"updateCustomFieldWithInfo : INSERT or UPDATE WCCustomFieldInfo failed";
                break;
            }
            
            //////////////////////////////////////////////////
            // 寫入picklist item
            if (customFieldInfo.picklistItems)
            {
                // !!以完全取代方式處理，所以先清除舊的，再將目前的加入
                sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCustomFieldListItem WHERE CustomFieldInfoGuid = '%@'",
                              [super sqlString:customFieldInfo.guid]];
                
                if ([super runSqlCommand:sqlCommand]==NO)
                {
                    errorStep = @"updateCustomFieldWithInfo : DELETE WCCustomFieldListItem failed";
                    break;
                }
                
                for (WCCustomFieldListItem *listItem in customFieldInfo.picklistItems)
                {
                    sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCustomFieldListItem(ItemGuid, CustomFieldInfoGuid, ItemText) VALUES('%@','%@','%@')",
                                  [super sqlString:listItem.guid],
                                  [super sqlString:customFieldInfo.guid],
                                  [super sqlString:listItem.itemText]];
                    
                    if ([super runSqlCommand:sqlCommand]==NO)
                    {
                        errorStep = [NSString stringWithFormat:@"updateCustomFieldWithInfo : INSERT INTO WCCustomFieldListItem failed ([%@] %@)", [super sqlString:listItem.guid], [super sqlString:listItem.itemText]];
                        break;
                    }
                }
            }

            result = YES;
        } while (0);
        
        if(result)
        {
            if(!(result = [super commitTransaction]))
            {
                errorStep = @"updateCustomFieldWithInfo : commitTransaction failed";
            }
        }
        else
        {
            [super rollbackTransaction];
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        return result;
    }
}


//===============================================================================
/// 移除一個自訂欄位
//===============================================================================
- (void)removeCustomFieldWithGuid:(NSString *)guid
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCustomFieldInfo WHERE CustomFieldInfoGuid='%@'", [super sqlString:guid]];
        [super runSqlCommand:sqlCommand];
    }
}


//==============================================================================
//
//==============================================================================
- (NSArray *)customFieldListItemsWithGuid:(NSString *)guid
{
    sqlite3_stmt *stmt = NULL;
    NSString *sqlCommand = nil;
    NSMutableArray *listItems = nil;
    
    //////////////////////////////////////////////////
    sqlCommand = [NSString stringWithFormat:@"SELECT ItemGuid, ItemText FROM WCCustomFieldListItem WHERE CustomFieldInfoGuid='%@'", [super sqlString:guid]];
    
    if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
    {
        while (sqlite3_step(stmt) == SQLITE_ROW)
        {
            if (listItems==nil)
            {
                listItems = [NSMutableArray array];
            }
            
            int colunm = 0;
            
            WCCustomFieldListItem *customFieldListItem = [[WCCustomFieldListItem alloc] init];
            
            customFieldListItem.guid = [super stringFromStmt:stmt column:colunm++];
            customFieldListItem.itemText = [super stringFromStmt:stmt column:colunm++];
            
            [listItems addObject:customFieldListItem];
            
            [customFieldListItem release];
        }
        sqlite3_finalize(stmt);
    }
    
    return listItems;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group methods

//===============================================================================
//
//===============================================================================
- (NSInteger)groupIDWithGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupID FROM WCGroup WHERE GroupGuid='%@'", [super sqlString:groupGuid]];
        return [super intResultWithSelectCommand:sqlCommand];
    }
}


//===============================================================================
//
//===============================================================================
- (NSString *)groupNameWithGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        NSString *groupName = nil;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupName FROM WCGroup WHERE GroupGuid='%@'", [super sqlString:groupGuid]];
        sqlite3_stmt *stmt;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                groupName = [super stringFromStmt:stmt column:0];
                break;
            }
            
            sqlite3_finalize(stmt);
        }
        
        return groupName;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateGroupGuid:(NSString *)guid withID:(NSInteger)groupID
{
    @synchronized(self)
    {
        if(groupID == WC_GID_None || [guid length] == 0)
        {
            return NO;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET GroupGuid='%@' WHERE GroupID=%td", [super sqlString:guid], groupID];
        return [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateSuperGroupGuid:(NSString *)superGroupGuid withID:(NSInteger)groupID
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if(groupID == WC_GID_None)
        {
            [self setLastErrorWithErrorStep:@"updateSuperGroupGuid : groupID is empty"];
            return WC_GID_None;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET SuperGroupGuid='%@' WHERE GroupID=%td",
                                [super sqlString:superGroupGuid],
                                groupID];
        
        
        return [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)clearAllGroupPinnedOrder
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET PinnedOrder=0"];        
        
        return [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)updateGroupPinnedOrder:(NSInteger)pinnedOrder withGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET PinnedOrder=%td WHERE GroupGuid='%@'",
                                pinnedOrder,
                                [super sqlString:groupGuid]];
        
        
        return [super runSqlCommand:sqlCommand];
    }
}


//===============================================================================
// 
//===============================================================================
- (NSMutableArray *)pinnedGroups
{
    @synchronized(self)
    {
        sqlite3_stmt    *stmt;
        NSString        *utf8Command;
        WCGroupModel    *groupModel;
        NSMutableArray    *groupArray = [[NSMutableArray alloc] init];
        
        utf8Command = [NSString stringWithFormat:@"SELECT GroupID,GroupGuid,GroupName,ModifiedTime,GroupOrder,SuperGroupGuid,PinnedOrder,Helper FROM WCGroup WHERE PinnedOrder>0 ORDER BY PinnedOrder"];
        
        self.lastError = nil;
        
        if(groupArray == nil)
        {
            [self setLastErrorWithErrorStep:@"copyAllGroups : Faield to alloc group array"];
            return nil;
        }
        
        if(sqlite3_prepare_v2(self.dbHandle, [utf8Command UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            // get group data
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                
                if((groupModel = [[WCGroupModel alloc] init]))
                {
                    groupModel.ID = [super integerFromStmt:stmt column:column++];
                    groupModel.guid = [super stringFromStmt:stmt column:column++];
                    groupModel.name = [super stringFromStmt:stmt column:column++];
                    groupModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    groupModel.order = [super integerFromStmt:stmt column:column++];
                    groupModel.superGroupGuid = [super stringFromStmt:stmt column:column++];
                    groupModel.pinnedOrder = [super integerFromStmt:stmt column:column++];
                    groupModel.helper = [super stringFromStmt:stmt column:column++];

                    [groupArray addObject:groupModel];
                    [groupModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyAllGroups : Failed to select group in DB"];
            [groupArray release];
            return nil;
        }
        
        if([groupArray count] == 0)
        {
            [groupArray release];
            groupArray = nil;
        }
        
        return [groupArray autorelease];
    }
}


//================================================================================
//
//================================================================================
- (WCGroupModel *)groupWithGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        WCGroupModel    *groupModel = nil;
        NSString        *sqlCommand = nil;
        sqlite3_stmt    *stmt;
        int             column = 0;
        NSString        *dbVersion = [self infoValueWithKey:WCTCDBC_IK_Version];
        
        
        // 不同版本group資料有差異
        if([dbVersion isEqualToString:WCTCDBC_FormatVersion_V100] == YES)
        {
            // 100版
            sqlCommand = [NSString stringWithFormat:@"SELECT GroupID,GroupName,ModifiedTime,GroupOrder FROM WCGroup WHERE GroupGuid='%@'", [super sqlString:groupGuid]];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    groupModel = [[WCGroupModel alloc] init];
                    
                    if(groupModel != nil)
                    {
                        groupModel.guid = groupGuid;
                        groupModel.ID = [super integerFromStmt:stmt column:column++];
                        groupModel.name = [super stringFromStmt:stmt column:column++];
                        groupModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                        groupModel.order = [super integerFromStmt:stmt column:column++];
                    }
                    
                    break;
                }
                
                sqlite3_finalize(stmt);
            }
            else
            {
                [self setLastErrorWithErrorStep:@"groupWithGuid"];
            }
        }
        else
        {
            // 130版 （目前版本）
            sqlCommand = [NSString stringWithFormat:@"SELECT GroupID,GroupName,ModifiedTime,GroupOrder,SuperGroupGuid,PinnedOrder,Helper FROM WCGroup WHERE GroupGuid='%@'", [super sqlString:groupGuid]];
            
            if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
            {
                while (sqlite3_step(stmt) == SQLITE_ROW)
                {
                    groupModel = [[WCGroupModel alloc] init];
                    
                    if(groupModel != nil)
                    {
                        groupModel.guid = groupGuid;
                        groupModel.ID = [super integerFromStmt:stmt column:column++];
                        groupModel.name = [super stringFromStmt:stmt column:column++];
                        groupModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                        groupModel.order = [super integerFromStmt:stmt column:column++];
                        groupModel.superGroupGuid = [super stringFromStmt:stmt column:column++];
                        groupModel.pinnedOrder = [super integerFromStmt:stmt column:column++];
                        groupModel.helper = [super stringFromStmt:stmt column:column++];
                    }
                    
                    break;
                }
                
                sqlite3_finalize(stmt);
            }
            else
            {
                [self setLastErrorWithErrorStep:@"groupWithGuid"];
            }
        }
        
        return [groupModel autorelease];
    }
}


//================================================================================
//
//================================================================================
- (NSDictionary *)groupIDAndGuidMappingDict
{
    @synchronized(self)
    {
        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        NSString            *sqlCommand = @"SELECT GroupID,GroupGuid FROM WCGroup";
        sqlite3_stmt        *stmt;
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                WC_GroupID groupID = [super integerFromStmt:stmt column:0];
                NSString *groupGuid = [super stringFromStmt:stmt column:1];
                
                [dict setObject:groupGuid forKey:@(groupID)];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"groupWithGuid"];
        }
        
        return dict;
    }
}


//==============================================================================
//
//==============================================================================
- (NSArray *)copyGroupGuidArrayWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        sqlite3_stmt*stmt;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupGuid FROM WCCardGroup WHERE CardID='%@'", [super sqlString:cardID]];
        NSMutableArray *groupGuidArray = [[NSMutableArray alloc] init];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                // !! 當名片是未分類時，要把WC_GID_Unfiled加到groupIDArray，不是groupIDArray設為nil。
                NSString *groupGuid = [super stringFromStmt:stmt column:0];
                [groupGuidArray addObject:groupGuid];
            }
            
            sqlite3_finalize(stmt);
        }
        
        if(![groupGuidArray count])
        {
            [groupGuidArray release];
            groupGuidArray = nil;
        }
        
        return groupGuidArray;
    }
}


//==============================================================================
//
//==============================================================================
- (NSDate *)groupOrderModifiedTime
{
    NSString *timeIntervalString = [self infoValueWithKey:WCTCDBC_UserInfoKey_CategoryOrderModifiedTime];
    double timeInterval = [timeIntervalString doubleValue];
    
    return [NSDate dateWithTimeIntervalSince1970:timeInterval];
}


//==============================================================================
//
//==============================================================================
- (void)setGroupOrderModifiedTime:(NSDate *)groupOrderModifiedTime
{
    NSTimeInterval timeInterval = [groupOrderModifiedTime timeIntervalSince1970];
    
    NSString *timeIntervalString = [NSString stringWithFormat:@"%lf", timeInterval] ;
    
    NSString *oldTimeIntervalString = [self infoValueWithKey:WCTCDBC_UserInfoKey_CategoryOrderModifiedTime];
    
    if ([oldTimeIntervalString length]==0)
    {
        [self addInfoValue:timeIntervalString withKey:WCTCDBC_UserInfoKey_CategoryOrderModifiedTime];
    }
    else
    {
        [self updateInfoValue:timeIntervalString withKey:WCTCDBC_UserInfoKey_CategoryOrderModifiedTime];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeGroupOrderModifiedTime
{
    //設為0代表清除
    [self updateInfoValue:@"0" withKey:WCTCDBC_UserInfoKey_CategoryOrderModifiedTime];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateGroupHelper:(NSString *)helper withID:(NSInteger)groupID
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if(groupID == WC_GID_None)
        {
            [self setLastErrorWithErrorStep:@"updateGroupHelper : groupID is empty"];
            return WC_GID_None;
        }
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET Helper='%@' WHERE GroupID=%td",
                                [super sqlString:helper],
                                groupID];
        
        
        return [super runSqlCommand:sqlCommand];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group sync action methods

//================================================================================
// 取得SyncAction不是None的紀錄
//================================================================================
- (NSMutableArray *)copyGroupSyncActions
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *actions = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand = [NSMutableString stringWithString:@"SELECT GroupGUID,ModifiedTime,SyncAction FROM WCTGroupSyncAction WHERE SyncAction!=0"];
        sqlite3_stmt *stmt;
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                WCTGroupSyncActionModel *actionModel = [[WCTGroupSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:0];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:1];
                    actionModel.actionType = [super integerFromStmt:stmt column:2];
                    
                    [actions addObject:actionModel];
                    [actionModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            [actions release];
            return nil;
        }
        
        return actions;
    }
}


//================================================================================
// date如果是nil表示要全部資料
//================================================================================
- (NSMutableArray *)copyGroupSyncActionsAfterDate:(NSDate *)date
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *actions = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand = [NSMutableString stringWithString:@"SELECT GroupGUID,ModifiedTime,SyncAction FROM WCTGroupSyncAction"];
        sqlite3_stmt *stmt;
        
        if(date != nil)
        {
            [sqlCommand appendFormat:@" WHERE ModifiedTime >= %.3f", [date timeIntervalSince1970]];
        }
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                WCTGroupSyncActionModel *actionModel = [[WCTGroupSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:0];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:1];
                    actionModel.actionType = [super integerFromStmt:stmt column:2];
                    
                    [actions addObject:actionModel];
                    [actionModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            [actions release];
            return nil;
        }
        
        return actions;
    }
}


//==============================================================================
//
//==============================================================================
- (WCTGroupSyncActionModel *)copyGroupSyncActionModelWithGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        self.lastError = nil;
        if ([groupGuid length]==0)
        {
            [self setLastErrorWithErrorStep:@"groupSyncActionModelWithGuid : guid should not be nil"];
            return nil;
        }
        
        WCTGroupSyncActionModel *actionModel = nil;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT GroupGUID,ModifiedTime,SyncAction FROM WCTGroupSyncAction WHERE GroupGUID='%@'", [super sqlString:groupGuid]];
        sqlite3_stmt *stmt;
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if (sqlite3_step(stmt) == SQLITE_ROW)
            {
                actionModel = [[WCTGroupSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:0];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:1];
                    actionModel.actionType = [super integerFromStmt:stmt column:2];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            return nil;
        }
        
        return actionModel;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)removeGroupSyncActionWithGuid:(NSString *)groupGuid
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if([groupGuid length] == 0)
        {
            [self setLastErrorWithErrorStep:@"removeGroupBySyncWithGuid : group guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCTGroupSyncAction WHERE GroupGuid='%@'", [super sqlString:groupGuid]];
        
        return [super runSqlCommand:sqlCommand];
    }
}


//==============================================================================
// 用這個API的地方都有用 synchronized保護，所以這邊先不用
//==============================================================================
- (BOOL)updateGroupSyncAction:(WCTGroupSyncActionModel *)groupSyncAction
{
    if (groupSyncAction==nil)
    {
        [self setLastErrorWithErrorStep:@"updateGroupSyncAction : no groupSyncAction"];
        return NO;
    }
    
    NSString *sqlCommand = nil;
    
    // 先檢查GroupGuid是否存在
    NSInteger count = [self intResultWithSelectCommand:[NSString stringWithFormat:@"SELECT COUNT(*) FROM WCTGroupSyncAction WHERE GroupGuid='%@'", [super sqlString:groupSyncAction.dataGuid]]];
    
    if (count == WC_InvalidRecordID)
    {
        [self setLastErrorWithErrorStep:@"updateGroupSyncAction : group note exist"];
        return NO;
    }
    
    if (groupSyncAction.modifiedTime == nil)
    {
        [self setLastErrorWithErrorStep:@"updateGroupSyncAction : must have modifiedTime"];
        return NO;
    }
    
    //////////////////////////////////////////////////
    
    if (count<=0)
    {
        // 如果不存在，新增record
        sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCTGroupSyncAction(GroupGuid,ModifiedTime,SyncAction) VALUES('%@', %f, %ld)",
                      [super sqlString:groupSyncAction.dataGuid],
                      [groupSyncAction.modifiedTime timeIntervalSince1970],
                      (long)groupSyncAction.actionType];
    }
    else
    {
        // 如果存在，更新原本的record
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCTGroupSyncAction SET ModifiedTime=%f, SyncAction=%ld WHERE GroupGuid = '%@'",
                      [groupSyncAction.modifiedTime timeIntervalSince1970],
                      (long)groupSyncAction.actionType,
                      [super sqlString:groupSyncAction.dataGuid]];
    }
    
    return [self runSqlCommand:sqlCommand];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateGroupModifiedTime:(NSDate *)modifiedTime withGuid:(NSString *)guid
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if(modifiedTime==nil)
        {
            [self setLastErrorWithErrorStep:@"updateGroupModifiedTime : modifiedTime is empty"];
            return NO;
        }
        
        if([guid length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateGroupModifiedTime : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = nil;
        BOOL result = NO;
        
        [super beginTransaction];
        
        do {
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCTGroupSyncAction SET ModifiedTime=%f WHERE GroupGuid='%@'",
                          [modifiedTime timeIntervalSince1970],
                          [super sqlString:guid]];
            
            result = [super runSqlCommand:sqlCommand];
            
            if(result==NO)
            {
                break;
            }
            
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCGroup SET ModifiedTime=%f WHERE GroupGuid='%@'",
                          [modifiedTime timeIntervalSince1970],
                          [super sqlString:guid]];
            
            result = [super runSqlCommand:sqlCommand];

            if(result==NO)
            {
                break;
            }
            
        } while (0);

        
        if(result)
        {
            if(!(result = [super commitTransaction]))
            {
                [self setLastErrorWithErrorStep:@"updateGroupModifiedTime : commitTransaction failed"];
            }
        }
        else
        {
            [super rollbackTransaction];
        }
        return result;
    }
}


//==============================================================================
// 把同步動作設為NONE
//==============================================================================
- (BOOL)resetGroupSyncActionToNoneWithGuid:(NSString *)guid
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if([guid length] == 0)
        {
            [self setLastErrorWithErrorStep:@"resetGroupSyncActionToNoneWithGuid : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand;
        
        [super beginTransaction];
        
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCTGroupSyncAction SET SyncAction=%ld WHERE GroupGuid='%@'",
                      (long)WCTSyncActionType_None,
                      [super sqlString:guid]];
        
        BOOL result = [super runSqlCommand:sqlCommand];
        
        if(result)
        {
            if(!(result = [super commitTransaction]))
            {
                [self setLastErrorWithErrorStep:@"updateGroupModifiedTime : commitTransaction failed"];
            }
        }
        else
        {
            [super rollbackTransaction];
        }
        return result;
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card methods

//===============================================================================
// NOTE: Group資料不會在此處理，統一由Card-Group相關函式處理
// 包含unverified, favorite
//===============================================================================
- (BOOL)insertCard:(WCCardModel *)cardModel syncActionModel:(WCTCardSyncActionModel *)syncActionModel isImportMode:(BOOL)isImportMode isInTransaction:(BOOL)isInTransaction
{
    @synchronized(self)
    {
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        const int           maxRetry = 10;
        NSString            *sqlCommand = nil;
        NSString            *errorStep = nil;
        NSInteger           maxRecordID;
        NSInteger           totalFieldCount;
        NSMutableArray      *deletedRecordIDArray = nil;
        NSArray             *allFieldArray = nil;
        BOOL                isUseDeletedRecordID;
        BOOL                result = NO;
        

        self.lastError = nil;
        
        
        //--------------------------------
        // initial check
        //--------------------------------
        if(!cardModel || ![cardModel.ID length])
        {
            errorStep = @"insertCard : card model or ID not exist";
            goto _EXIT;
        }

        //--------------------------------
        // prepare recordID data for add field
        //--------------------------------
        maxRecordID = [self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Field];
        totalFieldCount = [cardModel totalFieldCount];
        deletedRecordIDArray = [self copyDeletedRecordIDArrayWithRecordType:WCTCDBC_RT_Field
                                                             maxNeededCount:totalFieldCount];
        
        
        // 把多的欄位放到note
        [cardModel fixFieldCount];
        
        // 依目前的field array順序，重設field order的值
        [cardModel resetAllFieldOrder];
        
        //--------------------------------
        // add main data
        //--------------------------------
        if(!isInTransaction)
        {
            [super beginTransaction];
        }
        
        if (cardModel.createdTime==nil)
        {
            cardModel.createdTime = [NSDate date];
        }

        if (cardModel.modifiedTime==nil)
        {
            cardModel.modifiedTime = [NSDate date];
        }
// convert time to GMT+0 time
        NSTimeInterval currentTimeInterval = [[NSDate date] timeIntervalSince1970];
        NSTimeInterval createdTimeInterval = cardModel.createdTime ? [cardModel.createdTime timeIntervalSince1970] : currentTimeInterval;
        NSTimeInterval modifiedTimeInterval = cardModel.modifiedTime ? [cardModel.modifiedTime timeIntervalSince1970] : currentTimeInterval;
        
        
        // !! cardID 重複時，重新產生cardID並重試
        for(int i=0; i<maxRetry; i++)
        {
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCCard \
                          (CardID,DisplayName,DisplayCompany,\
                          DisplayJobTitle,DisplayAddress,DisplayGPS,Creator,Owner,Editor,\
                          SectionTitle,CreatedTime,ModifiedTime,FrontRecogLang,BackRecogLang) \
                          VALUES('%@','%@','%@','%@','%@','%@','%@','%@','%@','%@',%f,%f,%ld,%ld)",
                          [super sqlString:cardModel.ID],
                          [super sqlString:cardModel.displayName],
                          [super sqlString:cardModel.displayCompany],
                          [super sqlString:cardModel.displayJobTitle],
                          [super sqlString:cardModel.displayAddress],
                          [super sqlString:cardModel.displayGPS],
                          [super sqlString:cardModel.creator],
                          [super sqlString:cardModel.owner],
                          [super sqlString:cardModel.editor],
                          [super sqlString:cardModel.sectionTitle],
                          createdTimeInterval,
                          modifiedTimeInterval,
                          (long)cardModel.frontRecogLang,
                          (long)cardModel.backRecogLang];
            
            if(![super runSqlCommand:sqlCommand])
            {
                // !! WCCard 只有cardID重複才會出現SQLITE_CONSTRAINT錯誤
                if([super dbErrorCode] == SQLITE_CONSTRAINT && i < maxRetry-1)
                {
                    // generate other cardID and retry
                    cardModel.ID = [NSString GUID];
                    [NSThread sleepForTimeInterval:0.01];
                }
                else
                {
                    errorStep = @"insertCard : failed to insert display data";
                    goto _EXIT;
                }
            }
            else break;
        }
        
        
        //--------------------------------
        // add field data
        //--------------------------------
        allFieldArray = [cardModel.fieldArrayDict allValues];
        
        for(NSMutableArray *fieldArray in allFieldArray)
        {
            for(WCFieldModel *fieldModel in fieldArray)
            {
                /////////////////////////////////////////
                // get new field ID
                //
                if([deletedRecordIDArray count])
                {
                    fieldModel.ID = [[deletedRecordIDArray objectAtIndex:0] intValue];
                    // !! 這邊要更新maxRecordID
                    // fix 0032617: 聯絡人->手動建立姓名->之後再執行"重新辨識"，某些流程操作下，顯示出來的姓名會有錯亂的情形，參考圖片
                    maxRecordID = MAX(maxRecordID, fieldModel.ID);
                    isUseDeletedRecordID = YES;
                }
                else
                {
                    fieldModel.ID = maxRecordID + 1;
                    isUseDeletedRecordID = NO;
                }
                
                /////////////////////////////////////////
                // Insert field
                //
                if(fieldModel.ID > 0)
                {
                    if([fieldModel.value isKindOfClass:[NSMutableDictionary class]])
                    {
                        NSMutableDictionary *dictValue = (NSMutableDictionary *)fieldModel.value;
                        NSArray *subFieldArray = [dictValue allValues];
                        
                        for(WCFieldModel *subFieldModel in subFieldArray)
                        {
                            subFieldModel.ID = fieldModel.ID;
                            
                            if(![self insertFieldWithCardID:cardModel.ID fieldModel:subFieldModel])
                            {
                                errorStep = @"insertCard : failed to insert dict field";
                                goto _EXIT;
                            }
                        }
                    }
                    else
                    {
                        if(![self insertFieldWithCardID:cardModel.ID fieldModel:fieldModel])
                        {
                            errorStep = @"insertCard : failed to insert normal field";
                            goto _EXIT;
                        }
                    }
                    
                    /////////////////////////////////////////
                    // update recordID data
                    [self removeDeletedRecordID:fieldModel.ID withRecordType:WCTCDBC_RT_Field];
                    
                    if(isUseDeletedRecordID)
                        [deletedRecordIDArray removeObjectAtIndex:0];
                    else maxRecordID++;
                }
                else
                {
                    errorStep = @"insertCard : can't get valid fieldID";
                    goto _EXIT;
                }
            }
        }
        
        
        //////////////////////////////////////////////////
        // 處理未校正
        
        if ((cardModel.tagMask & WC_TagMask_Unverified) == WC_TagMask_Unverified)
        {
            [self addUnverifiedWithCardID:cardModel.ID];
        }
        else
        {
            [self removeUnverifiedWithCardID:cardModel.ID];
        }
        
        
        //////////////////////////////////////////////////
        // 處理我的最愛
        if ((cardModel.tagMask & WC_TagMask_Favorite) == WC_TagMask_Favorite)
        {
            [self addFavoriteWithCardID:cardModel.ID isInTransaction:YES];
        }
        else
        {
            [self removeFavoriteWithCardID:cardModel.ID isInTransaction:YES];
        }
        
        //--------------------------------
        // add group data
        //--------------------------------
        if(![self setCardGroupsWithCardID:cardModel.ID groupIDArray:cardModel.groupIDArray isInTransaction:YES updateModifiedTIme:NO])
        {
            errorStep = @"insertCard : failed to set group data";
            goto _EXIT;
        }
        
        //--------------------------------
        // add shared account data
        //--------------------------------
        
        // 先檢查是否包含自已，與主管，如果沒有要補上去
        // 因為每次都做太多餘了，所以還是改外部塞
//        NSMutableArray *adjustAccountGUIDArray = [NSMutableArray arrayWithArray:cardModel.sharedAccountGUIDArray];
//        
//        NSArray *accountList = [self copyAllAccountList];
//        for (WCTAccountRelationModel *accountRelationModel in accountList)
//        {
//            if(accountRelationModel.relation==WCTAccountRelationModel_Relation_Account)
//            {
//                if ([adjustAccountGUIDArray containsObject:accountRelationModel.guid]==NO)
//                {
//                    [adjustAccountGUIDArray addObject:accountRelationModel.guid];
//                }
//            }
//            else if(accountRelationModel.relation==WCTAccountRelationModel_Relation_Boss)
//            {
//                if ([adjustAccountGUIDArray containsObject:accountRelationModel.guid]==NO)
//                {
//                    [adjustAccountGUIDArray addObject:accountRelationModel.guid];
//                }
//            }
//        }
//        
//        cardModel.sharedAccountGUIDArray = [NSArray arrayWithArray:adjustAccountGUIDArray];
        
        if(![self setCardSharedAccountGuids:cardModel.sharedAccountGUIDArray withCardID:cardModel.ID isInTransaction:YES updateModifiedTime:NO])
        {
            errorStep = @"insertCard : failed to set shared account data";
            goto _EXIT;
        }
        
        //--------------------------------
        // Remove Deleted CardID
        //--------------------------------
        // 動作要記錄到WCTCardSyncAction
        if (syncActionModel)
        {
            if([self updateCardSyncAction:syncActionModel isInTransaction:YES]==NO)
            {
                errorStep = @"insertCard : failed to update cardSyncAction";
                goto _EXIT;
            }
        }
        
        
        //--------------------------------
        // finish with success
        //--------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        [deletedRecordIDArray release];
        
        if(result)
        {
            if (!isInTransaction)
            {
                if(!(result = [super commitTransaction]))
                {
                    errorStep = @"insertCard : commitTransaction failed";
                }
            }
        }
        else
        {
            if (!isInTransaction)
            {
                [super rollbackTransaction];
            }
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        
        [pool release];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCard:(WCCardModel *)cardModel syncActionModel:(WCTCardSyncActionModel *)syncActionModel mustExist:(BOOL)mustExist
{
    @synchronized(self)
    {
        if (cardModel==nil)
        {
            [self setLastErrorWithErrorStep:@"CardModle不能為空"];
            return NO;
        }
        
        self.lastError = nil;
        BOOL result = NO;
        
        [self beginTransaction];
        
        //////////////////////////////////////////////////
        // 檢查名片是否存在
        
        BOOL isCardExist = [self isCardIDExist:cardModel.ID];
        BOOL deleteSuccess = NO;
        
        if(isCardExist == YES)
        {
            deleteSuccess = [self removeCardWithCardID:cardModel.ID syncActionModel:nil forUpdate:YES isInTransaction:YES];
        }
        else
        {
            if(mustExist == YES)
            {
                [self setLastErrorWithErrorStep:@"mustExist == YES && isCardExist == NO"];
                return NO;
            }
            else
            {
                deleteSuccess = YES;
            }
        }

        
        if(deleteSuccess)
        {
            // 再加回名片
            if([self insertCard:cardModel syncActionModel:nil isImportMode:NO isInTransaction:YES])
            {
                //////////////////////////////////////////////////
                [self setCardGroupsWithCardID:cardModel.ID groupIDArray:cardModel.groupIDArray isInTransaction:YES updateModifiedTIme:NO];
                result = YES;
            }
        }
        
        //////////////////////////////////////////////////
        NSString *errorStep = nil;
        if(result)
        {
            // 調整syncAction紀錄
            if(syncActionModel)
            {
                [self updateCardSyncAction:syncActionModel isInTransaction:YES];
            }
            
            if(!(result = [super commitTransaction]))
            {
                errorStep = @"insertCard : commitTransaction failed";
            }
        }
        else
        {
            [super rollbackTransaction];
        }
        
        if(!result)
        {
            [self setLastErrorWithErrorStep:errorStep];
        }
        
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeCardWithCardID:(NSString *)cardID syncActionModel:(WCTCardSyncActionModel*)syncActionModel
{
    return [self removeCardWithCardID:cardID syncActionModel:syncActionModel forUpdate:NO isInTransaction:NO];
}


//===============================================================================
// NOTE: Group資料不會在此處理，統一由Card-Group相關函式處理
// forUpdate: 如果是updateCard要刪除時要這為YES，這樣不在cardModel中的資料才不會被清除
//===============================================================================
- (BOOL)removeCardWithCardID:(NSString *)cardID syncActionModel:(WCTCardSyncActionModel*)syncActionModel forUpdate:(BOOL)forUpdate isInTransaction:(BOOL)isInTransaction
{
    @synchronized(self)
    {
        NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
        BOOL                result = NO;
        NSString            *sqlCommand;
        NSString            *errorStep = nil;
        NSMutableArray      *fieldIDArray = nil;
        
        
        self.lastError = nil;
        
        //--------------------------------
        // get all field ID that will be removed
        //--------------------------------
        fieldIDArray = [self copyFieldIDArrayWithCardID:cardID];
        
        
        if(!isInTransaction)
        {
            [super beginTransaction];
        }
        
        
        //--------------------------------
        // delete group data
        //--------------------------------
        if(![self removeCardGroupIDsWithCardID:cardID])
        {
            errorStep = @"removeCardWithCardID : removeCardGroupIDswithCardID failed";
            goto _EXIT;
        }
        
        
        //--------------------------------
        // delete card share account data
        //--------------------------------
        if(![self removeCardSharedAccountGuidsWithCardID:cardID])
        {
            errorStep = @"removeCardWithCardID : removeCardSharedAccountGuidsWithCardID failed";
            goto _EXIT;
        }
        
        //--------------------------------
        // delete field data
        //--------------------------------
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCardField WHERE CardID='%@'", [super sqlString:cardID]];
        
        if(![super runSqlCommand:sqlCommand])
        {
            errorStep = @"removeCardWithCardID : delete from WCCardField failed";
            goto _EXIT;
        }
        
        //--------------------------------
        // delete main data
        //--------------------------------
        sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCCard WHERE CardID='%@'", [super sqlString:cardID]];
        
        if(![super runSqlCommand:sqlCommand])
        {
            errorStep = @"removeCardWithCardID : Delete from WCCard failed";
            goto _EXIT;
        }
        
        if (!forUpdate)
        {
            //--------------------------------
            // delete address book data
            //--------------------------------
            if ([self removeAddressBookDataWithCardID:cardID]==NO)
            {
                errorStep = @"removeCardWithCardID : Delete from WCAddressBookMapping failed";
                goto _EXIT;
            }
            
            //--------------------------------
            // delete recent record
            //--------------------------------
            [self removeRecentActionWithCardID:cardID];
            
            //--------------------------------
            // delete imported card mapping
            //--------------------------------
            [self removeImportedCardIDWithCardID:cardID];
            
            //--------------------------------
            // delete card sync error
            //--------------------------------
            [self removeCardSyncErrorInfoWithCardID:cardID];
        }
        
        //--------------------------------
        // update sync action
        //--------------------------------
        if(syncActionModel != nil)
        {
            if([self updateCardSyncAction:syncActionModel isInTransaction:YES] == NO)
            {
                errorStep = @"removeCardWithCardID : updateCardSyncAction failed";
                goto _EXIT;
            }
        }
        else
        {
            // !!加果不是ForUpdate, 且syncActionModel==nil會同時刪除syncAction
            if(!forUpdate)
            {
                [self removeCardSyncActionWithCardID:cardID isInTransaction:YES];
            }
        }
        
        //--------------------------------
        // keep deleted fieldID
        //--------------------------------
        for(NSString *fieidIDString in fieldIDArray)
            [self addDeletedRecordID:[fieidIDString intValue] withRecordType:WCTCDBC_RT_Field];
        
        //--------------------------------
        // finish with success
        //--------------------------------
        result = YES;
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        
    _EXIT:
        
        if(!isInTransaction)
        {
            if(result)
            {
                if((result = [super commitTransaction]))
                {
                    // !! 把多餘的deleted FieldID清除
                    NSString *tableName = [self deletedRecordTableNameWithRecordType:WCTCDBC_RT_Field];
                    NSInteger maxFieldID = [self maxUsedRecordIDWithRecordType:WCTCDBC_RT_Field];
                    sqlCommand = [NSString stringWithFormat:@"DELETE FROM %@ WHERE RecordID>=%ld", tableName, (long)maxFieldID];
                    [super runSqlCommand:sqlCommand];
                }
                else
                {
                    errorStep = @"removeCardWithCardID : commitTransaction failed";
                }
            }
            else
            {
                [super rollbackTransaction];
            }
        }
        
        if(!result)
            [self setLastErrorWithErrorStep:errorStep];
        
        [fieldIDArray release];
        fieldIDArray = nil;
        
        [pool release];
        
        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - imported card id mapping


//==============================================================================
//
//==============================================================================
- (NSString *)cardIDWithImportedCardID:(NSString *)importedCardID
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        NSString *cardID = nil;
        NSString *sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCTImportedCardIDMapping WHERE ImportedCardID='%@'", [super sqlString:importedCardID]];
        sqlite3_stmt *stmt;
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if(sqlite3_step(stmt) == SQLITE_ROW)
            {
                cardID = [super stringFromStmt:stmt column:0];
            }
        }
        
        return cardID;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)setImporedCardID:(NSString *)importedCardID withCardID:(NSString *)cardID
{
    @synchronized (self)
    {
        NSString *sqlCommand = nil;
        // 先檢查cardID是否存在
        NSInteger count = [self intResultWithSelectCommand:[NSString stringWithFormat:@"SELECT COUNT(*) FROM WCTImportedCardIDMapping WHERE CardID='%@'", [super sqlString:cardID]]];
        
        if (count==WC_InvalidRecordID)
        {
            return NO;
        }
        
        if (count<=0)
        {
            // 如果不存在，新增record
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCTImportedCardIDMapping (CardID,ImportedCardID) VALUES('%@', '%@')",
                          [super sqlString:cardID],
                          [super sqlString:importedCardID]];
        }
        else
        {
            // 如果存在，更新原本的record
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCTImportedCardIDMapping SET ImportedCardID='%@' WHERE CardID='%@'",
                          [super sqlString:importedCardID],
                          [super sqlString:cardID]];
        }
        
        //////////////////////////////////////////////////
        
        BOOL result = NO;
        
        result = [self runSqlCommand:sqlCommand];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeImportedCardIDWithCardID:(NSString *)cardID
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"removeOldCardIDWithCardID : cardID guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCTImportedCardIDMapping WHERE CardID='%@'", [super sqlString:cardID]];
        
        BOOL result =  [super runSqlCommand:sqlCommand];
        
        
        return result;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card sync action methods

//==============================================================================
//
//==============================================================================
- (BOOL)removeCardSyncActionWithCardID:(NSString *)cardID isInTransaction:(BOOL)isInTransaction
{
   self.lastError = nil;
    
    if([cardID length] == 0)
    {
        [self setLastErrorWithErrorStep:@"removeCardSyncActionWithCardID : cardID guid is empty"];
        return NO;
    }
    
    //////////////////////////////////////////////////
    if (isInTransaction==NO)
    {
        [self beginTransaction];
    }
    
    NSString *sqlCommand = [NSString stringWithFormat:@"DELETE FROM WCTCardSyncAction WHERE CardID='%@'", [super sqlString:cardID]];
    
    BOOL result =  [super runSqlCommand:sqlCommand];

    
    if (isInTransaction==NO)
    {
        if (result)
        {
            [self commitTransaction];
        }
        else
        {
            [self rollbackTransaction];
        }
    }
    
    return result;
}


//================================================================================
// 取得SyncAction不是None的紀錄
//================================================================================
- (NSMutableArray *)copyCardSyncActions
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *actions = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand =
        [NSMutableString stringWithString:@"SELECT CardID,ModifiedTime,SyncAction,SyncState,GroupSHA1,\
         ContentSHA1,SharedAccountSHA1,FrontSideSHA1,BackSideSHA1,IDPhotoSHA1\
         FROM WCTCardSyncAction WHERE CardID NOT IN (SELECT CardID From WCTCardSyncErrorInfo) and SyncAction!=0"];
        sqlite3_stmt *stmt;
        
        //////////////////////////////////////////////////
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                WCTCardSyncActionModel *actionModel = [[WCTCardSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:column++];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    actionModel.actionType = [super integerFromStmt:stmt column:column++];
                    actionModel.syncState = [super integerFromStmt:stmt column:column++];
                    actionModel.groupSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.contentSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.sharedAccountSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.frontSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.backSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.IDPhotoSHA1 = [super stringFromStmt:stmt column:column++];

                    [actions addObject:actionModel];
                    [actionModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            [actions release];
            return nil;
        }
        
        return actions;
    }
}


//================================================================================
// date如果是nil表示要全部資料
//================================================================================
- (NSMutableArray *)copyCardSyncActionsAfterDate:(NSDate *)date
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *actions = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand =
        [NSMutableString stringWithString:@"SELECT CardID,ModifiedTime,SyncAction,SyncState,GroupSHA1, \
         ContentSHA1,SharedAccountSHA1,FrontSideSHA1,BackSideSHA1,IDPhotoSHA1 \
         FROM WCTCardSyncAction WHERE CardID not IN (SELECT CardID From WCTCardSyncErrorInfo)"];
        sqlite3_stmt *stmt;
        
        if(date != nil)
        {
            [sqlCommand appendFormat:@" AND ModifiedTime >= %.3f", [date timeIntervalSince1970]];
        }
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column = 0;
                WCTCardSyncActionModel *actionModel = [[WCTCardSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:column++];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    actionModel.actionType = [super integerFromStmt:stmt column:column++];
                    actionModel.syncState = [super integerFromStmt:stmt column:column++];
                    actionModel.groupSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.contentSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.sharedAccountSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.frontSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.backSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.IDPhotoSHA1 = [super stringFromStmt:stmt column:column++];
                   
                    [actions addObject:actionModel];
                    [actionModel release];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            [actions release];
            return nil;
        }
        
        return actions;
    }
}


//==============================================================================
//
//==============================================================================
- (WCTCardSyncActionModel *)copyCardSyncActionModelWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        self.lastError = nil;
        if ([cardID length]==0)
        {
            [self setLastErrorWithErrorStep:@"copyCardSyncActionModelWithGuid : guid should not be nil"];
            return nil;
        }
        
        //////////////////////////////////////////////////
        sqlite3_stmt *stmt;
        WCTCardSyncActionModel *actionModel = nil;
        NSString *sqlCommand =
        [NSString stringWithFormat:@"SELECT CardID,ModifiedTime,SyncAction,SyncState,GroupSHA1,\
         ContentSHA1,SharedAccountSHA1,FrontSideSHA1,BackSideSHA1,IDPhotoSHA1 \
         FROM WCTCardSyncAction WHERE CardID='%@'", [super sqlString:cardID]];
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            if (sqlite3_step(stmt) == SQLITE_ROW)
            {
                int column =0;
                actionModel = [[WCTCardSyncActionModel alloc] init];
                
                if(actionModel != nil)
                {
                    actionModel.dataGuid = [super stringFromStmt:stmt column:column++];
                    actionModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                    actionModel.actionType = [super integerFromStmt:stmt column:column++];
                    actionModel.syncState = [super integerFromStmt:stmt column:column++];
                    actionModel.groupSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.contentSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.sharedAccountSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.frontSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.backSideSHA1 = [super stringFromStmt:stmt column:column++];
                    actionModel.IDPhotoSHA1 = [super stringFromStmt:stmt column:column++];
                }
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyGroupSyncActionsAfterDate : Failed to run sql command"];
            return nil;
        }
        
        return actionModel;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeCardSyncActionWithCardID:(NSString *)cardID
{
    @synchronized(self)
    {
        return [self removeCardSyncActionWithCardID:cardID isInTransaction:NO];
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)resetCardSyncActionToNoneWithCardID:(NSString *)cardID checkModifyTime:(NSDate *)checkModifyTime
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : guid is empty"];
            return NO;
        }
        
        // 先檢查cardID是否存在
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];
        
        //!! 如果有checkModifyTime, 要檢查是否與目前的一致，不一樣的話就不寫入
        if (checkModifyTime!=nil)
        {
            if ([currentSyncActionModel.modifiedTime isEqual:checkModifyTime]==NO)
            {
                [currentSyncActionModel release];
                // 這時要算成功，不然同步會停止
                return YES;
            }
        }
        [currentSyncActionModel release];
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand =
        [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET SyncAction=%ld WHERE CardID='%@'",
         (long)WCTSyncActionType_None,
         [super sqlString:cardID]];
        
        return [super runSqlCommand:sqlCommand];
    }
}



//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncAction:(WCTCardSyncActionModel *)cardSyncAction withCheckModifyTime:(NSDate *)checkModifyTime isInTransaction:(BOOL)isInTransaction
{
    @synchronized (self)
    {
        if (cardSyncAction==nil)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncAction:withCheckModifyTime:isInTransaction: cardSyncAction==nil"];
            return NO;
        }
        
        // 先檢查cardID是否存在
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardSyncAction.dataGuid];

        //!! 如果有checkModifyTime, 要檢查是否與目前的一致，不一樣的話就不寫入
        if (checkModifyTime!=nil)
        {
            if ([currentSyncActionModel.modifiedTime isEqual:checkModifyTime]==NO)
            {
                [currentSyncActionModel release];
                // 這時要算成功，不然同步會停止
                return YES;
            }
        }
        
        NSString *sqlCommand = nil;
        
        //////////////////////////////////////////////////
        
        NSDate *modifiedTime = cardSyncAction.modifiedTime ? cardSyncAction.modifiedTime : [NSDate date];
        
        if (currentSyncActionModel==nil)
        {
            // 如果不存在，新增record
            sqlCommand = [NSString stringWithFormat:@"INSERT INTO WCTCardSyncAction\
                          (CardID,ModifiedTime,SyncAction,SyncState,GroupSHA1,ContentSHA1,\
                          SharedAccountSHA1,FrontSideSHA1,BackSideSHA1,IDPhotoSHA1) \
                          VALUES('%@', %f, %ld, %ld, '%@', '%@', '%@', '%@', '%@', '%@')",
                          [super sqlString:cardSyncAction.dataGuid],
                          [modifiedTime timeIntervalSince1970],
                          (long)cardSyncAction.actionType,
                          (long)cardSyncAction.syncState,
                          cardSyncAction.groupSHA1?:@"",
                          cardSyncAction.contentSHA1?:@"",
                          cardSyncAction.sharedAccountSHA1?:@"",
                          cardSyncAction.frontSideSHA1?:@"",
                          cardSyncAction.backSideSHA1?:@"",
                          cardSyncAction.IDPhotoSHA1?:@""];
        }
        else
        {
            // !! 如果actionType不是none, 不能改為modify
            if (cardSyncAction.actionType==WCTSyncActionType_Modify &&
                currentSyncActionModel.actionType!=WCTSyncActionType_None)
            {
                cardSyncAction.actionType = currentSyncActionModel.actionType;
            }
            
            // 如果存在，更新原本的record
            sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET \
                          ModifiedTime=%f, SyncAction=%ld, SyncState=%ld, GroupSHA1='%@', ContentSHA1='%@', SharedAccountSHA1='%@',\
                          FrontSideSHA1='%@', BackSideSHA1='%@', IDPhotoSHA1='%@' WHERE CardID = '%@'",
                          [modifiedTime timeIntervalSince1970],
                          (long)cardSyncAction.actionType,
                          (long)cardSyncAction.syncState,
                          cardSyncAction.groupSHA1?:@"",
                          cardSyncAction.contentSHA1?:@"",
                          cardSyncAction.sharedAccountSHA1?:@"",
                          cardSyncAction.frontSideSHA1?:@"",
                          cardSyncAction.backSideSHA1?:@"",
                          cardSyncAction.IDPhotoSHA1?:@"",
                          [super sqlString:cardSyncAction.dataGuid]];
        }
        
        [currentSyncActionModel release];
        
        //////////////////////////////////////////////////
        
        BOOL result = NO;
        
        if (isInTransaction == NO)
        {
            @synchronized (self)
            {
                result = [self runSqlCommand:sqlCommand];
            }
        }
        else
        {
            result = [self runSqlCommand:sqlCommand];
        }
        
        // for debug
        if(result == NO)
        {
            NSString *step = [NSString stringWithFormat:@"failed to runSqlCommand : %@", sqlCommand];
            [self setLastErrorWithErrorStep:step];
        }
        
        return result;
    }
}



//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncAction:(WCTCardSyncActionModel *)cardSyncAction isInTransaction:(BOOL)isInTransaction
{
    return [self updateCardSyncAction:cardSyncAction withCheckModifyTime:nil isInTransaction:isInTransaction];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncActionModifiedTime:(NSDate *)modifiedTime withCardID:(NSString *)cardID;
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if(modifiedTime==nil)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : syncAcctinoModifiedTime is empty"];
            return NO;
        }
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand =
        [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET ModifiedTime=%f WHERE CardID='%@'",
         [modifiedTime timeIntervalSince1970],
         [super sqlString:cardID]];
        
        BOOL result =  [super runSqlCommand:sqlCommand];

        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncActionForGroupChangeWithCardID:(NSString *)cardID modifiedTime:(NSDate *)modifiedTime
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        //////////////////////////////////////////////////
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionForGroupChangeWithCardID : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        if(modifiedTime == nil)
        {
            modifiedTime = [NSDate date];
        }
        
        //////////////////////////////////////////////////
        // !! 如果actionType不是none, 不能改為modify
        WCTSyncActionType actionType = WCTSyncActionType_Modify;
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];
        
        if (currentSyncActionModel.actionType!=WCTSyncActionType_None)
        {
            actionType = currentSyncActionModel.actionType;
        }
        
        [currentSyncActionModel release];
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET ModifiedTime=%f,GroupSHA1='',SyncAction=%ld WHERE CardID='%@'",
                                [modifiedTime timeIntervalSince1970],
                                (long)actionType,
                                [super sqlString:cardID]];
        
        BOOL result = [self runSqlCommand:sqlCommand];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncActionForSharedAccountChangeWithCardID:(NSString *)cardID modifiedTime:(NSDate *)modifiedTime
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        //////////////////////////////////////////////////
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionForSharedAccountChangeWithCardID : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        if(modifiedTime == nil)
        {
            modifiedTime = [NSDate date];
        }
        
        //////////////////////////////////////////////////
        // !! 如果actionType不是none, 不能改為modify
        WCTSyncActionType actionType = WCTSyncActionType_Modify;
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];
        
        if (currentSyncActionModel.actionType!=WCTSyncActionType_None)
        {
            actionType = currentSyncActionModel.actionType;
        }
        [currentSyncActionModel release];
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET ModifiedTime=%f,SharedAccountSHA1='',SyncAction=%ld WHERE CardID='%@'",
                                [modifiedTime timeIntervalSince1970],
                                (long)actionType,
                                [super sqlString:cardID]];
        
        BOOL result = [self runSqlCommand:sqlCommand];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncActionForContentChangeWithCardID:(NSString *)cardID modifiedTime:(NSDate *)modifiedTime
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        //////////////////////////////////////////////////
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionForContentChangeWithCardID : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        if(modifiedTime == nil)
        {
            modifiedTime = [NSDate date];
        }
        
        //////////////////////////////////////////////////
        // !! 如果actionType不是none, 不能改為modify
        // !! 修改內容，就把syncErrorCode設為0
        WCTSyncActionType actionType = WCTSyncActionType_Modify;
        WCTSyncState syncState = WCTSyncState_UnSync;
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];
        
        if (currentSyncActionModel.actionType!=WCTSyncActionType_None)
        {
            actionType = currentSyncActionModel.actionType;
        }
        
        syncState = currentSyncActionModel.syncState;
        
        if(currentSyncActionModel.syncState==WCTSyncState_Synced)
        {
            syncState = WCTSyncState_Synced_Modify;
        }
        [currentSyncActionModel release];
        
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET ModifiedTime=%f,ContentSHA1='',SyncAction=%ld, SyncState=%ld WHERE CardID='%@'",
                                [modifiedTime timeIntervalSince1970],
                                (long)actionType,
                                (long)syncState,
                                [super sqlString:cardID]];
        
        BOOL result = [self runSqlCommand:sqlCommand];
        
        // !! 內容更新就移除錯誤記錄
        [self removeCardSyncErrorInfoWithCardID:cardID];
        
        return result;
    }
}


//==============================================================================
///
//==============================================================================
- (BOOL)updateCardSyncActionForImageChangeWithCardID:(NSString *)cardID imageType:(WC_ImageType)imageType modifiedTime:(NSDate *)modifiedTime
{
    @synchronized (self)
    {
        self.lastError = nil;
        
        //////////////////////////////////////////////////
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionForImageChangeWithCardID : guid is empty"];
            return NO;
        }
        
        //////////////////////////////////////////////////
        
        NSString *colunmName = nil;
        
        switch (imageType)
        {
            case WC_IT_FrontSide:
            {
                colunmName = @"FrontSideSHA1";
                break;
            }
            case WC_IT_BackSide:
            {
                colunmName = @"BackSideSHA1";
                break;
            }
            case WC_IT_IDPhoto:
            {
                colunmName = @"IDPhotoSHA1";
                break;
            }
            default:
            {
                // 不會發生這個case
                break;
            }
        }
        
        if(modifiedTime == nil)
        {
            modifiedTime = [NSDate date];
        }
        
        //////////////////////////////////////////////////
        // !! 如果actionType不是none, 不能改為modify
        WCTSyncActionType actionType = WCTSyncActionType_Modify;
        WCTCardSyncActionModel *currentSyncActionModel = [self copyCardSyncActionModelWithCardID:cardID];

        if (currentSyncActionModel.actionType!=WCTSyncActionType_None)
        {
            actionType = currentSyncActionModel.actionType;
        }
        [currentSyncActionModel release];
        //////////////////////////////////////////////////
        
        NSString *sqlCommand = [NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET ModifiedTime=%f,%@='',SyncAction=%ld WHERE CardID='%@'",
                                [modifiedTime timeIntervalSince1970],
                                colunmName,
                                (long)actionType,
                                [super sqlString:cardID]];
        
        BOOL result = [self runSqlCommand:sqlCommand];
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)replaceGuidFromOld:(NSString *)oldGuid toNew:(NSString *)newGuid
{
    if ([oldGuid length]==0 ||
        [newGuid length]==0)
    {
        [self setLastErrorWithErrorStep:@"replaceGuidFromOld : toNew:"];
        return NO;
    }
    
    BOOL result = NO;
    
    // TODO: 更換guid ,  其他值都不能動
    @synchronized (self)
    {
        self.lastError = nil;

        [self beginTransaction];
        
        do {
            NSString *sqlCommand = nil;
            
            NSArray *tableNameArray = @[@"WCFavorite",
                                        @"WCUnverified",
                                        @"WCCardGroup",
                                        @"WCCard",
                                        @"WCCardField",
                                        @"WCRecentContacts",
                                        @"WCAddressBookMapping",
                                        @"WCTCardSharedAccount",
                                        @"WCTImportedCardIDMapping",
                                        @"WCTCardSyncAction"];
            //////////////////////////////////////////////////

            for (NSString * tableName in tableNameArray)
            {
                sqlCommand = [NSString stringWithFormat:@"UPDATE %@ SET CardID='%@' WHERE CardID='%@'",
                              [super sqlString:tableName],
                              [super sqlString:newGuid],
                              [super sqlString:oldGuid]];
                
                NSLog(@"sqlCommand:%@", sqlCommand);
                
                if ((result = [self runSqlCommand:sqlCommand])==NO)
                {
                    break;
                }
            }

            if (result==NO)
            {
                break;
            }

        } while (0);
        
        if (result)
        {
            [self commitTransaction];
        }
        else
        {
            [self rollbackTransaction];
        }
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)isCardSyncedWithGuid:(NSString *)guid
{
    @synchronized (self)
    {
        if ([guid length]==0)
        {
            return NO;
        }
        
        NSString *sqlCommand =[NSString stringWithFormat:@"SELECT SyncState FROM WCTCardSyncAction WHERE CardID='%@'", [super sqlString:guid]];
        
        WCTSyncState isSync = [self intResultWithSelectCommand:sqlCommand];
        if (isSync==WCTSyncState_Synced)
        {
            return YES;
        }
        else
        {
            return NO;
        }
    }
}


//==============================================================================
// 修正錯誤的同步紀錄，以免同步後不一致
//==============================================================================
- (void)fixInvalidRecord
{
    @synchronized (self)
    {
        // 刪除在WCCard但不在WCTCardSyncAction中的記錄
        NSString *sqlCommand =[NSString stringWithFormat:@"DELETE FROM WCTCardSyncAction WHERE CardID not in (SELECT CardID FROM WCCARD) AND SyncAction!=%td", WCTSyncActionType_Delete];
        
        if ([self runSqlCommand:sqlCommand]==NO)
        {
            [self setLastErrorWithErrorStep:@"remove CardID not in WCCard failed"];
        }
        
        //////////////////////////////////////////////////
        // 在WCTCardSyncAction但不在WCCard中的名片, 要補上syncAction
        sqlite3_stmt *stmt = NULL;
        NSMutableArray *invalidCardIDs = [NSMutableArray array];
        sqlCommand =[NSString stringWithFormat:@"SELECT CardID FROM WCCard WHERE CardID not in (SELECT CardID FROM WCTCardSyncAction)"];
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                NSString *cardID = [super stringFromStmt:stmt column:0];
                
                if(cardID)
                {
                    [invalidCardIDs addObject:cardID];
                }
            }
            
            sqlite3_finalize(stmt);
        }

        //////////////////////////////////////////////////
        
        for (NSString *cardID in invalidCardIDs)
        {
            WCTCardSyncActionModel *cardSyncAction = [[WCTCardSyncActionModel alloc] init];
            cardSyncAction.actionType = WCTSyncActionType_Add;
            cardSyncAction.syncState = WCTSyncState_UnSync;
            cardSyncAction.dataGuid = cardID;
            cardSyncAction.modifiedTime = [NSDate date];
            
            [self updateCardSyncAction:cardSyncAction isInTransaction:NO];
            [cardSyncAction release];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Sync Card error  methods


//==============================================================================
//
//==============================================================================
- (NSArray *)copyAllSyncErrorCardWithDisplayData
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *allCardArray = [[NSMutableArray alloc] init];
        NSString *sqlCommand =@"SELECT c.CardID,c.DisplayName,c.DisplayCompany,c.DisplayJobTitle,c.DisplayAddress,\
         c.DisplayGPS,c.SectionTitle,c.Creator,c.Owner,c.Editor,c.CreatedTime,c.ModifiedTime,f.errorCode \
         FROM WCTCardSyncErrorInfo f LEFT JOIN WCCard c ON f.CardID=c.CardID ORDER BY f.StartSyncTime DESC";
        
        sqlite3_stmt *stmt;
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //-------------------------------------------------
                // handel result
                //-------------------------------------------------
                int column = 0;
                NSString *valueString = [super stringFromStmt:stmt column:column++];
                
                if(![valueString length])
                    continue;
                
                WCSyncErrorCardModel *cardModel = [[WCSyncErrorCardModel alloc] init];
                cardModel.ID = valueString;
                cardModel.displayName = [super stringFromStmt:stmt column:column++];
                cardModel.displayCompany = [super stringFromStmt:stmt column:column++];
                cardModel.displayJobTitle = [super stringFromStmt:stmt column:column++];
                cardModel.displayAddress = [super stringFromStmt:stmt column:column++];
                cardModel.displayGPS = [super stringFromStmt:stmt column:column++];
                cardModel.sectionTitle = [super stringFromStmt:stmt column:column++];
                
                // WCT新增
                cardModel.creator = [super stringFromStmt:stmt column:column++];
                cardModel.owner = [super stringFromStmt:stmt column:column++];
                cardModel.editor = [super stringFromStmt:stmt column:column++];
                cardModel.createdTime = [super dateFromStmt:stmt column:column++];
                cardModel.modifiedTime = [super dateFromStmt:stmt column:column++];
                
                // 記錄error code
                cardModel.syncErrorCode = [super integerFromStmt:stmt column:column++];
                //////////////////////////////////////////////////
                
                [allCardArray addObject:cardModel];
                [cardModel release];
                
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copyAllSyncErrorCardWithDisplayData : Failed to run sql command"];
            [allCardArray release];
            return nil;
        }
        
        return allCardArray;
    }
}



//==============================================================================
//
//==============================================================================
- (NSArray *)copySyncErrorCardIDArray
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *cardIDArray = [[NSMutableArray alloc] init];
        NSMutableString *sqlCommand =
        [NSMutableString stringWithString:@"SELECT CardID FROM WCTCardSyncErrorInfo ORDER BY StartSyncTime"];
        sqlite3_stmt *stmt;
        
        //////////////////////////////////////////////////
        
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                NSString *cardID = [super stringFromStmt:stmt column:0];

                [cardIDArray addObject:cardID];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"copySyncErrorCardIDArray : Failed to run sql command"];
            [cardIDArray release];
            return nil;
        }
        
        return cardIDArray;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)addCardSyncErrorInfoWithCardID:(NSString *)cardID errorCode:(NSInteger)errorCode startSyncTime:(NSDate *)startSyncTime
{
    if ([cardID length]==0)
    {
        [self setLastErrorWithErrorStep:@"addCardSyncErrorInfoWithCardID:errorCode:startSyncTime : [cardID length]==0"];
        
        return NO;
    }
    
    self.lastError = nil;
    
    NSMutableString *sqlCommand =
    [NSMutableString stringWithFormat:@"INSERT INTO WCTCardSyncErrorInfo(CardID,ErrorCode,StartSyncTime)\
     VALUES ('%@', %ld, %lf)",
     [super sqlString:cardID],
     (long)errorCode,
     [startSyncTime timeIntervalSince1970]];
    
    return [self runSqlCommand:sqlCommand];
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeCardSyncErrorInfoWithCardID:(NSString *)cardID
{
    NSMutableString *sqlCommand =
    [NSMutableString stringWithFormat:@"DELETE FROM WCTCardSyncErrorInfo WHERE CardID='%@'",
     [super sqlString:cardID]];
    
    return [self runSqlCommand:sqlCommand];
}


//==============================================================================
//
//==============================================================================
- (BOOL)updateCardSyncErrorInfoWithCardID:(NSString *)cardID errorCode:(NSInteger)errorCode startSyncTime:(NSDate *)startSyncTime
{
    @synchronized (self)
    {
        [super beginTransaction];
        
        BOOL result = FALSE;
        
        result = [self removeCardSyncErrorInfoWithCardID:cardID];
        
        if(result)
        {
            result = [self addCardSyncErrorInfoWithCardID:cardID errorCode:errorCode startSyncTime:startSyncTime];
        }
        
        if (result)
        {
            [self commitTransaction];
        }
        else
        {
            [self rollbackTransaction];
        }
        
        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card image SHA1 methods

//==============================================================================
//
//==============================================================================
- (BOOL)setImageSHA1:(NSString*)imageSHA1 withImageType:(WC_ImageType)imageType cardID:(NSString *)cardID
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : guid is empty"];
            return NO;
        }
        
        if (imageType==WC_IT_None)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : guid is empty"];
            return NO;
        }
        
        // 如果沒有imageSHA1, 塞空字串
        if (imageSHA1==nil)
        {
            imageSHA1 = @"";
        }
        
        //////////////////////////////////////////////////
        
        NSString *colunmName = nil;
        
        switch (imageType)
        {
            case WC_IT_FrontSide:
            {
                colunmName = @"FrontSideSHA1";
                break;
            }
            case WC_IT_BackSide:
            {
                colunmName = @"BackSideSHA1";
                break;
            }
            case WC_IT_IDPhoto:
            {
                colunmName = @"IDPhotoSHA1";
                break;
            }
            default:
            {
                // 不會發生這個case
                break;
            }
        }
        
        NSString *sqlCommand = nil;
        
        // 先檢查cardID是否存在
        NSInteger count = [self intResultWithSelectCommand:[NSString stringWithFormat:@"SELECT COUNT(*) FROM WCTCardSyncAction WHERE CardID='%@'", [super sqlString:cardID]]];
        
        if (count==WC_InvalidRecordID)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : get count failed"];
            return NO;
        }
        
        if (count<=0)
        {
            // 新增
            sqlCommand =[NSString stringWithFormat:@"INSERT INTO WCTCardSyncAction(CardID, %@) VALUES('%@', '%@')",
                         colunmName,
                         [super sqlString:cardID],
                         [super sqlString:imageSHA1]];
        }
        else
        {
            // 更新
            sqlCommand =[NSString stringWithFormat:@"UPDATE WCTCardSyncAction SET %@='%@' WHERE CardID='%@'",
                         colunmName,
                         [super sqlString:imageSHA1],
                         [super sqlString:cardID]];
        }
        
        
        return [super runSqlCommand:sqlCommand];
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)imageSha1WithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        if([cardID length] == 0)
        {
            [self setLastErrorWithErrorStep:@"imageSha1WithCardID:type: : cardID is empty"];
            return nil;
        }
        
        if (type==WC_IT_None)
        {
            [self setLastErrorWithErrorStep:@"updateCardSyncActionModifiedTime:withGuid: : type==WC_IT_None"];
            return nil;
        }
        
        //////////////////////////////////////////////////
        
        NSString *colunmName = nil;
        
        switch (type)
        {
            case WC_IT_FrontSide:
            {
                colunmName = @"FrontSideSHA1";
                break;
            }
            case WC_IT_BackSide:
            {
                colunmName = @"BackSideSHA1";
                break;
            }
            case WC_IT_IDPhoto:
            {
                colunmName = @"IDPhotoSHA1";
                break;
            }
            default:
            {
                // 不會發生這個case
                break;
            }
        }
        
        NSString *command = [NSString stringWithFormat:@"SELECT %@ FROM WCTCardSyncAction WHERE CardID='%@'",
                             colunmName,
                             [super sqlString:cardID]];
        
        return [super stringResultWithSelectCommand:command];
    } // end of @synchronized
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Convert methods

//===============================================================================
//
//===============================================================================
//- (BOOL)convertFromXXXToXXX
//{
//
//}


//===============================================================================
//
//===============================================================================
- (BOOL)convertFrom100To130
{
    BOOL result = NO;

    self.lastError = nil;
    
    do
    {
        //////////////////////////////////////////////////
        // 130 WCGroup加入SuperGroupGuid/PinnedOrder/Helper欄位

        if([super isColumn:@"SuperGroupGuid" inTable:@"WCGroup"]==NO)
        {
            if([super runSqlCommand:@"ALTER TABLE WCGroup ADD COLUMN SuperGroupGuid TEXT DEFAULT '';"] == NO)
            {
                [self setLastErrorWithErrorStep:@"Faield to add column 'SuperGroupGuid'"];
                break;
            }
        }
        
        if([super isColumn:@"PinnedOrder" inTable:@"WCGroup"]==NO)
        {
            if([super runSqlCommand:@"ALTER TABLE WCGroup ADD COLUMN PinnedOrder INTEGER DEFAULT 0;"] == NO)
            {
                [self setLastErrorWithErrorStep:@"Faield to add column 'PinnedOrder'"];
                break;
            }
        }
        
        if([super isColumn:@"Helper" inTable:@"WCGroup"]==NO)
        {
            if([super runSqlCommand:@"ALTER TABLE WCGroup ADD COLUMN Helper TEXT DEFAULT '';"] == NO)
            {
                [self setLastErrorWithErrorStep:@"Faield to add column 'Helper'"];
                break;
            }
        }

        
        //////////////////////////////////////////////////
        // 130後改為多層多類別，轉換時要設定父類別。

        NSString *groupAllGuid = [self groupGuidWithID:WC_GID_All];
        NSString *command = [NSString stringWithFormat:@"UPDATE WCGroup SET SuperGroupGuid='%@' WHERE GroupID>=%td", groupAllGuid, WC_GID_Unfiled];

        if([super runSqlCommand:command] == NO)
        {
            [self setLastErrorWithErrorStep:@"Faield to update super group guid"];
            break;
        }

        
        //////////////////////////////////////////////////
        // 更新資料庫版本號

        [self updateInfoValue:WCTCDBC_FormatVersion_V130 withKey:WCTCDBC_IK_Version];
        result = YES;
    }
    while(0);
    
    return result;
}


//===============================================================================
//
//===============================================================================
- (BOOL)convertFrom130To140
{
    BOOL result = NO;
    
    self.lastError = nil;
    
    do
    {
        // !! 轉換到140主要是重新產生sectionTitle的資料
        // 動作會在DataController中做，這邊只要更新db table,與版號
        
        // !! 加入帳號是否為暫停使用狀態的值
        if([super isColumn:@"IsLimitedAccount" inTable:@"WCTAccountRelationship"]==NO)
        {
            if([super runSqlCommand:@"ALTER TABLE WCTAccountRelationship ADD COLUMN IsLimitedAccount INTEGER DEFAULT 0;"] == NO)
            {
                [self setLastErrorWithErrorStep:@"Faield to add column 'WCTAccountRelationship'"];
                break;
            }
        }
        
        //////////////////////////////////////////////////
        // 更新資料庫版本號
        
        [self updateInfoValue:WCTCDBC_FormatVersion_V140 withKey:WCTCDBC_IK_Version];
        result = YES;
    }
    while(0);
    
    return result;
}


//===============================================================================
//
//===============================================================================
- (BOOL)convertFrom140To141
{
    BOOL result = NO;
    
    self.lastError = nil;
    
    do
    {
        
        // !! 轉換到421是要清除包含地址包含科學園區的GPS
        [self beginTransaction];
        NSString *sqlCommand = nil;
        
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCCard SET DisplayGPS='' WHERE CardID in (SELECT CardID FROM WCCard WHERE DisplayAddress LIKE '%%%@%%')", [self sqlString:@"科學園區"]];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(result==NO)
        {
            break;
        }
        
        sqlCommand = [NSString stringWithFormat:@"UPDATE WCCardField SET ValueString='' WHERE FieldType=3 AND FieldSubType2=357 AND CardID in (SELECT CardID FROM WCCard WHERE DisplayAddress LIKE '%%%@%%')", [self sqlString:@"科學園區"]];
        
        result = [super runSqlCommand:sqlCommand];
        
        if(result==NO)
        {
            break;
        }
        
        result = [self updateInfoValue:WCTCDBC_FormatVersion_V141 withKey:WCTCDBC_IK_Version];
        
    }
    while(0);
    
    if(result==YES)
    {
        [self commitTransaction];
    }
    else
    {
        [self rollbackTransaction];
        [self setLastErrorWithErrorStep:@"convertFrom140To141: Failed to update version to 141"];
    }
    return result;
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Advance Search methods


//==============================================================================
//
//==============================================================================
- (NSString *)wccardFieldNameFromSearchType:(WCTSearchType)searchType
{
    if (searchType==WCTSearchType_Owner)
    {
        return @"Owner";
    }
    else if (searchType==WCTSearchType_Creator)
    {
        return @"Creator";
    }
    else if (searchType==WCTSearchType_Editor)
    {
        return @"Editor";
    }
    else if (searchType==WCTSearchType_CreateTime)
    {
        return @"CreatedTime";
    }
    else if (searchType==WCTSearchType_ModifiedTime)
    {
         return @"ModifiedTime";
    }
    return nil;
}


//==============================================================================
//
//==============================================================================
- (NSString *)conditionStringFromSearchOptionMode:(WCTSearchOptionModel*)searchOptionModel tableAlias:(NSString *)tableAlias
{
    return [self conditionStringFromSearchOptionMode:searchOptionModel tableAlias:tableAlias customFieldAlisa:nil];
}


//==============================================================================
// @param searchOptionModel 用來產生條件的searchOptionModel
// @param tableAlias 條件式中用的table別名
// @param customFieldAlisa 條件式中用的WCCustomFieldDataMapping別名
//==============================================================================
- (NSString *)conditionStringFromSearchOptionMode:(WCTSearchOptionModel*)searchOptionModel tableAlias:(NSString *)tableAlias customFieldAlisa:(NSString *)customFieldAlisa
{
    NSMutableString *sqlCommand = [NSMutableString string];
    if (searchOptionModel.searchType<WCTSearchType_UserDefine)
    {
        if ([searchOptionModel.text length]==0)
        {
            return nil;
        }
        
        [sqlCommand appendFormat:@"%@.FieldType=%td",tableAlias, searchOptionModel.searchType];
        
        // !! 如果是搜尋姓名，還要包含暱稱
        if (searchOptionModel.searchType==WCTSearchType_Name)
        {
            [sqlCommand appendFormat:@" OR %@.FieldType=%td",tableAlias, WC_FT_Nickname];
            
            // 前後加上小括號
            [sqlCommand insertString:@"(" atIndex:0];
            [sqlCommand appendString:@")"];
        }
        
        [sqlCommand appendFormat:@" AND %@.ValueString LIKE '%%%@%%'",tableAlias, [self sqlString:searchOptionModel.text]];
    }
    else if (searchOptionModel.searchType==WCTSearchType_UserDefine)
    {
        [sqlCommand appendFormat:@"%@.CustomFieldInfoGuid='%@'",customFieldAlisa, searchOptionModel.customFieldID];

        switch (searchOptionModel.customFieldType)
        {
            case WCCustomFieldContentType_Text:
            case WCCustomFieldContentType_URL:
            case WCCustomFieldContentType_Email:
            {
                if ([searchOptionModel.customFieldValue length]==0)
                {
                    // 清空
                    [sqlCommand setString:@""];
                    break;
                }
                
                // 文字類，用like
                [sqlCommand appendFormat:@" AND %@.ValueString LIKE '%%%@%%'",tableAlias, [self sqlString:searchOptionModel.customFieldValue]];
                break;
            }
            case WCCustomFieldContentType_Float:
            case WCCustomFieldContentType_Integer:
            {
                // 至少有要一個有值
                if ([searchOptionModel.customFieldValueFrom length]==0&&
                    [searchOptionModel.customFieldValueTo length]==0)
                {
                    // 清空
                    [sqlCommand setString:@""];
                    break;
                }
                
                // 數值類，用from, to
                if ([searchOptionModel.customFieldValueFrom length]>0)
                {
                    [sqlCommand appendFormat:@" AND cast(%@.ValueString AS INTEGER)>=%@",tableAlias, [self sqlString:searchOptionModel.customFieldValueFrom ]];
                }
                
                if ([searchOptionModel.customFieldValueTo length]>0)
                {
                    [sqlCommand appendFormat:@" AND cast(%@.ValueString AS INTEGER)<=%@",tableAlias, [self sqlString:searchOptionModel.customFieldValueTo]];
                }
                break;
            }
            case WCCustomFieldContentType_Date:
            {
                // 至少有要一個有值
                if (searchOptionModel.fromDate==nil &&
                    searchOptionModel.toDate==nil)
                {
                    // 清空
                    [sqlCommand setString:@""];
                    break;
                }
                
                // 日期類
                if (searchOptionModel.fromDate)
                {
                    [sqlCommand appendFormat:@" AND %@.ValueString>='%@'",tableAlias, [searchOptionModel.fromDate stringWithFormat:NSDateFormat_Day]];
                }
                
                if (searchOptionModel.toDate)
                {
                    [sqlCommand appendFormat:@" AND %@.ValueString<='%@'",tableAlias, [searchOptionModel.toDate stringWithFormat:NSDateFormat_Day]];
                }
                break;
            }
            case WCCustomFieldContentType_DateTime:
            {
                // 至少有要一個有值
                if (searchOptionModel.fromDate==nil &&
                    searchOptionModel.toDate==nil)
                {
                    // 清空
                    [sqlCommand setString:@""];
                    break;
                }
                
                // 時間類
                if (searchOptionModel.fromDate)
                {
                    [sqlCommand appendFormat:@" AND %@.ValueString>='%@ 00:00'",tableAlias, [searchOptionModel.fromDate stringWithFormat:NSDateFormat_Day]];
                }
                
                if (searchOptionModel.toDate)
                {
                    [sqlCommand appendFormat:@" AND %@.ValueString<='%@ 23:59'",tableAlias, [searchOptionModel.toDate stringWithFormat:NSDateFormat_Day]];
                }
                break;
            }
            case WCCustomFieldContentType_Picklist:
            {
                if([searchOptionModel.customFieldListItemIDs count]==0)
                {
                    // 清空
                    [sqlCommand setString:@""];
                    break;
                }
                
                // 選單類
                NSMutableString *listGuidString = [NSMutableString string];
                NSInteger index = 0;
                
                for (NSString *listItemGuid in searchOptionModel.customFieldListItemIDs)
                {
                    if (index>0)
                    {
                        [listGuidString appendString:@","];
                    }
                    
                    [listGuidString appendFormat:@"'%@'", [self sqlString:listItemGuid]];
                    index++;
                }
                
                if([listGuidString length]>0)
                {
                    [sqlCommand appendFormat:@" AND %@.ValueString IN (%@)",tableAlias, listGuidString];
                }

                break;
            }
            default:
                break;
        }
    }
    else if (searchOptionModel.searchType>=WCTSearchType_Company &&
             searchOptionModel.searchType<=WCTSearchType_JobTitle)
    {
        if ([searchOptionModel.text length]==0)
        {
            return nil;
        }
        
        [sqlCommand appendFormat:@"%@.FieldSubType2=%td",tableAlias, searchOptionModel.searchType];
        
        // !! 如果是搜尋公司名，還要包含公司讀音
        if (searchOptionModel.searchType==WCTSearchType_Company)
        {
            [sqlCommand appendFormat:@" OR %@.FieldSubType2=%td",tableAlias, WC_FST2_Company_Phonetic];
            
            // 前後加上小括號
            [sqlCommand insertString:@"(" atIndex:0];
            [sqlCommand appendString:@")"];
        }
        
        [sqlCommand appendFormat:@" AND %@.ValueString LIKE '%%%@%%'",tableAlias, [self sqlString:searchOptionModel.text]];
    }
    else if (searchOptionModel.searchType==WCTSearchType_Birthday ||
             searchOptionModel.searchType<=WCTSearchType_Anniversary)
    {
        // MARK: 生日紀念日，的搜尋目前不支援
//        [sqlCommand appendFormat:@"%@.FieldSubType1=%td",tableAlias, searchOptionModel.searchType];

    }
    else if (searchOptionModel.searchType==WCTSearchType_CreateTime ||
             searchOptionModel.searchType==WCTSearchType_ModifiedTime)
    {
        // 至少有要一個有值
        if (searchOptionModel.fromDate==nil &&
            searchOptionModel.toDate==nil)
        {
            return nil;
        }
        
        NSString *fieldName = [self wccardFieldNameFromSearchType:(WCTSearchType)searchOptionModel.searchType];

        if (searchOptionModel.fromDate)
        {
            NSString *fromDay = [searchOptionModel.fromDate stringWithFormat:NSDateFormat_Day];
            if ([fromDay length]>0)
            {
                NSDate *theDate = [NSDate dateFromString:[fromDay stringByAppendingString:@" 00:00:00"] format:NSDateFormat_Second];
                if (theDate)
                {
                    [sqlCommand appendFormat:@"%@.%@>=%f", tableAlias, fieldName, [theDate timeIntervalSince1970]];
                }
            }
        }
        
        if (searchOptionModel.toDate)
        {
            if ([sqlCommand length]>0)
            {
                [sqlCommand appendString:@" AND "];
            }
            
            NSString *toDay = [searchOptionModel.toDate stringWithFormat:NSDateFormat_Day];
            if ([toDay length]>0)
            {
                NSDate *theDate = [NSDate dateFromString:[toDay stringByAppendingString:@" 23:59:59"] format:NSDateFormat_Second];
                if (theDate)
                {
                    [sqlCommand appendFormat:@"%@.%@<=%f", tableAlias, fieldName, [theDate timeIntervalSince1970]];
                }
            }
        }
    }
    else if (searchOptionModel.searchType>=WCTSearchType_Owner)
    {
        if([searchOptionModel.accountGuids count]==0)
        {
            return nil;
        }
        
        NSString *fieldName = [self wccardFieldNameFromSearchType:(WCTSearchType)searchOptionModel.searchType];
        if ([fieldName length]==0)
        {
            return nil;
        }
        
        NSMutableString *accountGuidString = [NSMutableString string];
        NSInteger index = 0;
        
        for (NSString *accountGuid in searchOptionModel.accountGuids)
        {
            if (index>0)
            {
                [accountGuidString appendString:@","];
            }
            
            [accountGuidString appendFormat:@"'%@'", [self sqlString:accountGuid]];
            index++;
        }
        
        if([accountGuidString length]>0)
        {
            [sqlCommand appendFormat:@"%@.%@ IN (%@)", tableAlias, fieldName, accountGuidString];
        }
    }
    else if (searchOptionModel.searchType==WCTSearchType_Group)
    {
        if([searchOptionModel.groupGuids count]==0)
        {
            return nil;
        }
        
        NSMutableString *groupGuidString = [NSMutableString string];
        NSInteger index = 0;
        
        for (NSString *groupGuid in searchOptionModel.groupGuids)
        {
            if (index>0)
            {
                [groupGuidString appendString:@","];
            }
            
            [groupGuidString appendFormat:@"'%@'", [self sqlString:groupGuid]];
            index++;
        }
        
        if([groupGuidString length]>0)
        {
            [sqlCommand appendFormat:@"%@.GroupGuid IN (%@)",tableAlias, groupGuidString];
        }
    }

    return [sqlCommand length]?[NSString stringWithFormat:@" (%@)", sqlCommand]:nil;
}


//==============================================================================
// MARK: searchOptionModel轉為sqlCommand
//==============================================================================
- (NSString *)sqlCommandFromSearchOptionModel:(WCTSearchOptionModel*)searchOptionModel inFavorites:(BOOL)inFavorites
{
    NSMutableString *sqlCommand = [NSMutableString string];

    // 基礎語法
    if (inFavorites)
    {
        // !!要用WCFavorite來合併WCCard，因為WCFavorite一定比WCCard少
        [sqlCommand appendString:@"SELECT distinct f.CardID FROM WCFavorite as f"];
        [sqlCommand appendString:@" LEFT JOIN WCCard as c ON c.CardID = f.CardID"];
    }
    else
    {
        [sqlCommand appendString:@"SELECT distinct c.CardID FROM WCCard as c"];
    }

    //////////////////////////////////////////////////
    // 條件式 處理
    NSString *conditionString = nil;

    if(searchOptionModel.searchType==WCTSearchType_UserDefine)
    {
        [sqlCommand appendString:@" LEFT JOIN WCCardField as cf ON cf.CardID = c.CardID"];
        [sqlCommand appendString:@" LEFT JOIN WCCustomFieldDataMapping as cfdm ON cfdm.FieldID=cf.FieldID"];
        [sqlCommand appendString:@" LEFT JOIN WCCustomFieldInfo as cfi ON cfi.CustomFieldInfoGuid = cfdm.customFieldInfoGuid"];

        // 自訂欄位條件
        conditionString = [self conditionStringFromSearchOptionMode:searchOptionModel tableAlias:@"cf" customFieldAlisa:@"cfdm"];
    }
    else if(searchOptionModel.searchType<WCTSearchType_CreateTime)
    {
        // !! search一般欄位時才需要合併WCCardField
        [sqlCommand appendString:@" LEFT JOIN WCCardField as cf ON cf.CardID = c.CardID"];
        
        conditionString = [self conditionStringFromSearchOptionMode:searchOptionModel tableAlias:@"cf"];
    }
    else
    {
        switch (searchOptionModel.searchType)
        {
            case WCTSearchType_Group:
            {
                // !! 搜尋group才合併WCCardGroup
                // 要同時合併WCGroup，才能以GroupGuid search
                [sqlCommand appendString:@" LEFT JOIN WCCardGroup as cg ON cg.CardID = c.CardID"];
                [sqlCommand appendString:@" LEFT JOIN WCGroup as g ON g.GroupID = cg.GroupID "];
                
                conditionString = [self conditionStringFromSearchOptionMode:searchOptionModel tableAlias:@"g"];

                break;
            }
            case WCTSearchType_Owner:
            case WCTSearchType_Creator:
            case WCTSearchType_Editor:
            case WCTSearchType_CreateTime:
            case WCTSearchType_ModifiedTime:
            {
                // 建立，修改時間來自WCCard
                conditionString = [self conditionStringFromSearchOptionMode:searchOptionModel tableAlias:@"c"];
                break;
            }
            {
                break;
            }
            default:
                break;
        }
    }
    
    //////////////////////////////////////////////////
    if ([conditionString length]>0)
    {
        [sqlCommand appendString:@" WHERE"];
        [sqlCommand appendString:conditionString];
    }
    else
    {
        // 如果沒有condition表示這個search option 不合法，所以不處理
        return nil;
    }
    
    // !! 我的最愛依favorites order排序
    if (inFavorites)
    {
        [sqlCommand appendString:@" ORDER BY f.FavoriteOrder"];
    }
    return sqlCommand;
}


//==============================================================================
// sql必需只select cardID才可以使用,
// !!因為用NSSet, 所以指令中的order會無效，需要外部另外排序
//==============================================================================
- (NSMutableSet *)cardIDSetWithSQLCommand:(NSString *)sqlCommand
{
    NSMutableArray *cardIDArray = [self cardIDArrayWithSQLCommand:sqlCommand];
    return [NSMutableSet setWithArray:cardIDArray];
}



//==============================================================================
// sql必需只select cardID才可以使用,
//==============================================================================
- (NSMutableArray *)cardIDArrayWithSQLCommand:(NSString *)sqlCommand
{
    @synchronized(self)
    {
        self.lastError = nil;
        
        NSMutableArray *cardIDArray = [[NSMutableArray alloc] init];
        sqlite3_stmt *stmt = NULL;
        
        //////////////////////////////////////////////////
        
        if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
        {
            while (sqlite3_step(stmt) == SQLITE_ROW)
            {
                //-------------------------------------------------
                // handel result
                //-------------------------------------------------
                NSString *cardID = [super stringFromStmt:stmt column:0];
                
                if(![cardID length])
                    continue;
                
                [cardIDArray addObject:cardID];
            }
            
            sqlite3_finalize(stmt);
        }
        else
        {
            [self setLastErrorWithErrorStep:@"cardIDSetWithSQLCommand : Failed to run sql command"];
            [cardIDArray release];
            return nil;
        }
        
        return [cardIDArray autorelease];
    }
}


//==============================================================================
// 依我的最愛排序回傳
//==============================================================================
- (NSArray *)arraySortedByFavoriteOrderWithFavoriteCardIDs:(NSArray *)favoriteCardIDs
{
    if ([favoriteCardIDs count]<1)
    {
        return favoriteCardIDs;
    }
    
    NSMutableString *cardIDString = [NSMutableString string];
    for(NSString *cardID in favoriteCardIDs)
    {
        if ([cardIDString length]>0)
        {
            [cardIDString appendString:@","];
        }
        
        [cardIDString appendFormat:@"'%@'", [self sqlString:cardID]];
    }
    
    NSString *sqlCommand = [NSString stringWithFormat:@"SELECT CardID FROM WCFavorite WHERE CardID IN (%@) ORDER BY FavoriteOrder", cardIDString];
    
    NSMutableArray *result = [self cardIDArrayWithSQLCommand:sqlCommand];
    
    return result?[NSArray arrayWithArray:result]:nil;
}


//==============================================================================
// 依輸入的Array取得結果
//==============================================================================
- (NSMutableSet *)resultSetFromConditions:(NSMutableArray <WCTSearchOptionModel *>*)conditions inFavorites:(BOOL)inFavorites
{
    NSMutableSet *resultSet = nil;

    for (WCTSearchOptionModel *searchOptionModel in conditions)
    {
        NSString *sqlCommand = [self sqlCommandFromSearchOptionModel:searchOptionModel inFavorites:inFavorites];
        
        // !! 如果沒有合法指命就跳過
        if ([sqlCommand length]==0)
        {
            continue ;
        }
        
        
        NSMutableSet *tempSet = [self cardIDSetWithSQLCommand:sqlCommand];
        
        //////////////////////////////////////////////////
        // !! 第一次把結果直接assign, 之後都依operator做AND/OR
        if (resultSet==nil)
        {
            resultSet = tempSet;
        }
        else
        {
            if (searchOptionModel.searchOperator==WCTSearchOperator_AND)
            {
                [resultSet intersectSet:tempSet];
            }
            else
            {
                [resultSet unionSet:tempSet];
            }
        }
    }
    
    return resultSet;
}


//==============================================================================
// MARK: 依進階搜尋條件，取得結果
// 這邊的做法，每個array中的條件先做完，之後再把不同array的結果AND起來
//==============================================================================
- (NSMutableArray *)copyCardWithAdvancedSearchModel:(WCTAdvancedSearchModel *)advanceSearchModel inFavorites:(BOOL)inFavorites getAllField:(BOOL)getAllField;
{
    NSMutableSet *finalResultSet = nil;
    NSMutableSet *contactResultSet = nil;
    NSMutableSet *categoryResultSet = nil;
    NSMutableSet *dateResultSet = nil;
    NSMutableSet *accountResultSet = nil;
    NSMutableSet *customFieldResultSet = nil;

    //////////////////////////////////////////////////
    // 取得各section的結果
    contactResultSet = [self resultSetFromConditions:advanceSearchModel.contactInfoConditions inFavorites:inFavorites];
    
    // !! 因為各set間是用交集，所以如果其中一個set是空的就不用再往下做了
    if (contactResultSet!=nil &&[contactResultSet count]==0)
    {
        return nil;
    }
    
    categoryResultSet = [self resultSetFromConditions:advanceSearchModel.categoryConditions inFavorites:inFavorites];

    // !! 因為各set間是用交集，所以如果其中一個set是空的就不用再往下做了
    if (categoryResultSet!=nil &&[categoryResultSet count]==0)
    {
        return nil;
    }
    
    dateResultSet = [self resultSetFromConditions:advanceSearchModel.dateConditions inFavorites:inFavorites];

    // !! 因為各set間是用交集，所以如果其中一個set是空的就不用再往下做了
    if (dateResultSet!=nil &&[dateResultSet count]==0)
    {
        return nil;
    }
    
    accountResultSet = [self resultSetFromConditions:advanceSearchModel.accountConditions inFavorites:inFavorites];
    
    // !! 因為各set間是用交集，所以如果其中一個set是空的就不用再往下做了
    if (accountResultSet!=nil &&[accountResultSet count]==0)
    {
        return nil;
    }
    
    customFieldResultSet = [self resultSetFromConditions:advanceSearchModel.customFieldConditions inFavorites:inFavorites];

    // !! 因為各set間是用交集，所以如果其中一個set是空的就不用再往下做了
    if (customFieldResultSet!=nil &&[customFieldResultSet count]==0)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    // 合併所有結果
    if (contactResultSet!=nil)
    {
        finalResultSet = contactResultSet;
    }
    else if (categoryResultSet!=nil)
    {
        finalResultSet = categoryResultSet;
    }
    else if (dateResultSet!=nil)
    {
        finalResultSet = dateResultSet;
    }
    else if (accountResultSet!=nil)
    {
        finalResultSet = accountResultSet;
    }
    else if (customFieldResultSet!=nil)
    {
        finalResultSet = customFieldResultSet;
    }


    if (finalResultSet)
    {
        if(categoryResultSet!=nil)
        {
            [finalResultSet intersectSet:categoryResultSet];
        }
        if(dateResultSet!=nil)
        {
            [finalResultSet intersectSet:dateResultSet];
        }
        if(accountResultSet!=nil)
        {
            [finalResultSet intersectSet:accountResultSet];
        }
        if(customFieldResultSet!=nil)
        {
            [finalResultSet intersectSet:customFieldResultSet];
        }
    }

    //////////////////////////////////////////////////
    // 取得簡易資料
    NSMutableArray *result = [[NSMutableArray alloc] init];
    
    NSArray *cardIDArray = [finalResultSet allObjects];
    
    // !! 我的最愛要另外排序
    if (inFavorites)
    {
        cardIDArray = [self arraySortedByFavoriteOrderWithFavoriteCardIDs:cardIDArray];
    }
    
    for (NSString *cardID in cardIDArray)
    {
        WCCardModel *cardModel = [self copyCardWithCardID:cardID getAllField:getAllField];

        [result addObject:cardModel];
        [cardModel release];
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Log methods


//==============================================================================
//
//==============================================================================
+ (NSString *)logPath
{
    NSString *logDir = @"WCTCardDB";
    return [WCToolController baseStorePathWithDirName:logDir isCreatDirPath:YES];
}



@end
