//
//  WCTAccountDataController.m
//  WorldCardTeam
//
//  Created by sanhue on 2016/4/25.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import "WCTAccountDataController.h"

// defines
#import "WCSettingsKey.h"
#import "WCTSettingsKey.h"
#import "WCTAccountDataController+ResourceDefine.h"
#import "WCTRestClientController+ErrorCodeDefine.h"
#import "WCTRestClientController+Retry.h"

//
#import "WCCustomFieldInfo+WCTRCCustomFieldInfo.h"
#import "WCGroupModel+WCTRCCategoryInfoMultiLayer.h"

// controllers
#import "PPNetworkReachabilityController.h"
#import "PPSettingsController.h"
#import "WCDisplayNameController.h"
#import "WCTRestClientController.h"
#import "WCTRestClientController+Version.h"


#import "WCTAccountRelationModel.h"
#import "WCTGroupSyncActionModel.h"
#import "NSString+Additions.h"


NSString * const WCTAccountDataController_SettingsKey_SettingsModifiedTime = @"WCTAccountDataController_SettingsKey_SettingsModifiedTime";
// 這邊retry次數少一點
NSInteger const WCTADC_SyncMaxRetryCount = 3;

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation WCTAccountDataController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (private) class methods


//==============================================================================
//
//==============================================================================
+(void)asyncCheckNetworkWithActionBlock:(void(^)(BOOL hasNetwork))actionBlock completeHandler:(void(^)(BOOL hasNetwork))completeHandler
{
    BOOL hasNetwork = [PPNetworkReachabilityController checkForInternetConnection];
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        // 有網路時要執行的動作
        if(actionBlock)
        {
            actionBlock(hasNetwork);
        }
        //////////////////////////////////////////////////
        dispatch_async(dispatch_get_main_queue(), ^{
            
            // 完成後的通知
            if(completeHandler)
            {
                completeHandler(hasNetwork);
            }
        });
    });
}


//==============================================================================
//
//==============================================================================
+(void)syncCheckNetworkWithActionBlock:(void(^)(BOOL hasNetwork))actionBlock completeHandler:(void(^)(BOOL hasNetwork))completeHandler
{
    BOOL hasNetwork = [PPNetworkReachabilityController checkForInternetConnection];
    // 有網路時要執行的動作
    if(actionBlock)
    {
        actionBlock(hasNetwork);
    }
    
    // 完成後的通知
    if(completeHandler)
    {
        completeHandler(hasNetwork);
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Account list methods

//================================================================================
//
//================================================================================
+ (NSArray *)allAccountListWithError:(NSError **)error
{
    NSError *returnError = nil;
    NSMutableArray *allAccounts = [NSMutableArray array];
    
    do
    {
        if (allAccounts==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        //取得使用者帳號資訊
        WCTRCAccountInfo *userAccountInfo          = nil;
        WCTRCAccountInfoResponseResult *userResult = [[WCTRestClientController shareRestClientController] accountInfoWithError:&returnError];
     
        // !! 只要有錯誤就停止
        if(returnError!=nil)
        {
            break;
        }
        
        if (userResult!=nil)
        {
            userAccountInfo = [userResult.data firstObject];
            
            //////////////////////////////////////////////////
            //!! 更新資訊至設定內
            [PPSettingsController setIntegerValue:[userAccountInfo.accountType isEqualToString:WCTRC_AccountType_AD]?YES:NO
                                          withKey:WCTSettingsKey_IsADAccount];
            
            [PPSettingsController setIntegerValue:userAccountInfo.exportAbility
                                          withKey:WCTSettingsKey_ExportAbility];
            [PPSettingsController setIntegerValue:userAccountInfo.secretary
                                          withKey:WCTSettingsKey_HelpScanningAbility];
            [PPSettingsController setIntegerValue:userAccountInfo.printAbility
                                          withKey:WCTSettingsKey_PrintAbility];
            
            [PPSettingsController setStringValue:userAccountInfo.role
                                         withKey:WCTSettingsKey_AccountRole];
            
            //////////////////////////////////////////////////
            //添加使用者帳號資訊
            BOOL isOutgoingEmployee = [userAccountInfo.status isEqualToString:WCTRC_Status_Resigned];
            BOOL isLimitedAccount = [userAccountInfo.accountSubscriptionStatus isEqualToString:WCTRC_AccountSubscriptionStatus_TemplateInvalidate];
            
            WCTAccountRelationModel *userModel = [WCTAccountRelationModel accountRelationModelWithGuid:userAccountInfo.guid
                                                                                                  name:userAccountInfo.name
                                                                                                 email:userAccountInfo.email
                                                                                     shareContactCount:0
                                                                                              relation:WCTAccountRelationModel_Relation_Account
                                                                                    isOutgoingEmployee:isOutgoingEmployee
                                                                                      isLimitedAccount:isLimitedAccount];
            
            if (userModel!=nil)
            {
                [allAccounts insertObject:userModel atIndex:0];
            }
        }

        //////////////////////////////////////////////////
        //取得主管帳號資訊，Boss僅有一筆
        WCTRCAccountInfo *bossAccountInfo          = nil;
        WCTRCAccountInfoResponseResult *bossResult = [[WCTRestClientController shareRestClientController] accountBossWithError:&returnError];
        
        if(returnError!=nil)
        {
            NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
            if(statusCode==WCTServer_Common_ErrorCode_DataNotFound)
            {
                // !! 找不到主管不算錯誤, 不用中斷
                returnError = nil;
            }
            else
            {
                // !! 其他錯誤要停止
                break;
            }
        }
        
        if (bossResult!=nil)
        {
            bossAccountInfo = [bossResult.data firstObject];
            // 順便寫入設定值
            [PPSettingsController setStringValue:bossAccountInfo.guid withKey:WCTSettingsKey_BossAccountGUID];
        }
        else
        {
            [PPSettingsController setStringValue:@"" withKey:WCTSettingsKey_BossAccountGUID];
        }
        
        //////////////////////////////////////////////////
        //取得下屬帳號資訊
        NSMutableArray *subAccountInfoArray = nil;
        WCTRCAccountInfoResponseResult *subResult = [[WCTRestClientController shareRestClientController] accountSubordinatesWithError:error];

        if(returnError!=nil)
        {
            NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
            if(statusCode==WCTServer_Common_ErrorCode_DataNotFound)
            {
                // !! 找不到下屬不算錯誤, 不用中斷
                returnError = nil;
            }
            else
            {
                // !! 其他錯誤要停止
                break;
            }
        }
        
        
        if (subResult!=nil)
        {
            //!! 為了避免重複比對已確認為下屬的帳號
            subAccountInfoArray = [NSMutableArray arrayWithArray:subResult.data];
        }
        
        //////////////////////////////////////////////////
        //取得所有帳號清單
        WCTRCOtherAccountListResponseResult *listResult = [[WCTRestClientController shareRestClientController] otherAccountsInfoWithError:error];

        if(returnError!=nil)
        {
            NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
            if(statusCode==WCTServer_Common_ErrorCode_DataNotFound)
            {
                // !! 找不到其他帳號不算錯誤, 不用中斷
                returnError = nil;
            }
            else
            {
                // !! 其他錯誤要停止
                break;
            }
        }
        
        
        if (listResult!=nil)
        {
            for (NSInteger index=0; index<listResult.data.count; index++)
            {
                @autoreleasepool
                {
                    WCTRCAccountLessInfo *accountInfo   = [listResult.data objectAtIndex:index];
                    BOOL isOutgoingEmployee = [accountInfo.status isEqualToString:WCTRC_Status_Resigned];
                    BOOL isLimitedAccount = [accountInfo.accountSubscriptionStatus isEqualToString:WCTRC_AccountSubscriptionStatus_TemplateInvalidate];
                    
                    if (bossAccountInfo!=nil && [bossAccountInfo.guid isEqualToString:accountInfo.guid]==YES)
                    {
                        
                        //!! 確定為上司帳號，設定為上司並加入清單內
                        WCTAccountRelationModel *bossModel = [WCTAccountRelationModel accountRelationModelWithGuid:bossAccountInfo.guid
                                                                                                              name:bossAccountInfo.name
                                                                                                             email:bossAccountInfo.email
                                                                                                 shareContactCount:0
                                                                                                          relation:WCTAccountRelationModel_Relation_Boss
                                                                                                isOutgoingEmployee:isOutgoingEmployee
                                                                                                  isLimitedAccount:isLimitedAccount];
                        if (bossModel!=nil)
                        {
                            [allAccounts insertObject:bossModel atIndex:0];
                        }
                    }
                    else
                    {
                        BOOL isSubAccount = NO;
                        
                        //////////////////////////////////////////////////
                        // 判斷是否為下屬
                        if (subAccountInfoArray!=nil && subAccountInfoArray.count>0)
                        {
                            for (NSInteger subIndex=0; subIndex<subAccountInfoArray.count; subIndex++)
                            {
                                WCTRCAccountInfo *subAccountInfo = [subAccountInfoArray objectAtIndex:subIndex];
                                if ([subAccountInfo.guid isEqualToString:accountInfo.guid]==YES)
                                {
                                    isSubAccount = YES;
                                    [subAccountInfoArray removeObject:subAccountInfo];
                                }
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        //添加進清單內
                        if (isSubAccount==YES)
                        {
                            WCTAccountRelationModel *subModel = [WCTAccountRelationModel accountRelationModelWithGuid:accountInfo.guid
                                                                                                                 name:accountInfo.name
                                                                                                                email:accountInfo.email
                                                                                                    shareContactCount:0
                                                                                                             relation:WCTAccountRelationModel_Relation_Subordinate
                                                                                                   isOutgoingEmployee:isOutgoingEmployee
                                                                                                     isLimitedAccount:isLimitedAccount];
                            if (subModel!=nil)
                            {
                                [allAccounts addObject:subModel];
                            }
                        }
                        else
                        {
                            WCTAccountRelationModel *relationModel = [WCTAccountRelationModel accountRelationModelWithGuid:accountInfo.guid
                                                                                                                      name:accountInfo.name
                                                                                                                     email:accountInfo.email
                                                                                                         shareContactCount:0
                                                                                                                  relation:WCTAccountRelationModel_Relation_None
                                                                                                        isOutgoingEmployee:isOutgoingEmployee
                                                                                                          isLimitedAccount:isLimitedAccount];
                            if (relationModel!=nil)
                            {
                                [allAccounts addObject:relationModel];
                            }
                        }
                    }
                }
            }
        }
    } while (0);
    
    if (returnError!=nil)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
    }
    return allAccounts;
}


//==============================================================================
//
//==============================================================================
+ (NSArray *)updateAccountListWithError:(NSError **)error
{
    NSError *returnError = nil;
    NSArray *otherAccounts = [self allAccountListWithError:&returnError];
    
    //////////////////////////////////////////////////
    //!! 更新資料庫內的帳號關係清單
    if (returnError!=nil)
    {
        if(error != nil)
        {
            *error = returnError;
        }
        
        return nil;
    }
    
    if([otherAccounts count]!=0)
    {
        WCTDataController *saveAccountList = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_All];
        if (saveAccountList!=nil)
        {
            [saveAccountList updateAllAccountWithList:otherAccounts];
            [saveAccountList release];
        }
    }
    
    return otherAccounts;
}

//==============================================================================
//
//==============================================================================
+ (BOOL)updateAccountListFromServerWithCompleteHandler:(void(^)(NSArray *allAccountList, NSError *error))completeHandler
{
    __block NSError *error = nil;
    NSArray *allAccounts = [self updateAccountListWithError:&error];
    
    //////////////////////////////////////////////////
    
    if (completeHandler)
    {
        completeHandler(allAccounts, error);
    }
    
    return (error==nil)?YES:NO;
}


//==============================================================================
//
//==============================================================================
+ (NSInteger)numberOfAccountWithError:(NSError **)error
{
    NSInteger numberOfAccount = 0;
    NSError *returnError = nil;
    //////////////////////////////////////////////////
    //取得所有帳號清單
    WCTRCOtherAccountListResponseResult *listResult = [[WCTRestClientController shareRestClientController] otherAccountsInfoWithError:&returnError];
    
    if(returnError!=nil)
    {
        NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
        if(statusCode==WCTServer_Common_ErrorCode_DataNotFound)
        {
            // !! 找不到其他帳號不算錯誤
            returnError = nil;
            numberOfAccount = 1;
        }
    }
    else
    {
        NSMutableArray *availibleAccounts = [NSMutableArray array];
        for (WCTRCAccountInfo *accountInfo in listResult.data)
        {
            BOOL isLimitedAccount = [accountInfo.accountSubscriptionStatus isEqualToString:WCTRC_AccountSubscriptionStatus_TemplateInvalidate];
            if(isLimitedAccount)
            {
                continue;
            }
            
            //////////////////////////////////////////////////
            if([accountInfo.status isEqualToString:WCTRC_Status_Resigned]||
               [accountInfo.status isEqualToString:WCTRC_Status_Deleted])
            {
                continue;
            }

            [availibleAccounts addObject:accountInfo];
        }
        
        // other Accunt不包含自已，所以要把自已加進去
        numberOfAccount = [availibleAccounts count]+1;
    }
    
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if(error!=nil)
        {
            *error = returnError;
        }
    }
    
    return numberOfAccount;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Account info methods


//==============================================================================
//
//==============================================================================
+ (BOOL)updateAccountInfoWithError:(NSError **)error
{
    WCTRCAccountInfoResponseResult *infoResult = [[WCTRestClientController shareRestClientController] accountInfoWithError:error];
    if (infoResult!=nil && *error==nil)
    {
        WCTRCAccountInfo *accountInfoStored = [infoResult.data firstObject];
        
        //////////////////////////////////////////////////
        //!! 更新資訊至設定內
        [PPSettingsController setIntegerValue:[accountInfoStored.accountType isEqualToString:WCTRC_AccountType_AD]?YES:NO
                                      withKey:WCTSettingsKey_IsADAccount];
        
        [PPSettingsController setIntegerValue:accountInfoStored.exportAbility
                                      withKey:WCTSettingsKey_ExportAbility];
        [PPSettingsController setIntegerValue:accountInfoStored.secretary
                                      withKey:WCTSettingsKey_HelpScanningAbility];
        [PPSettingsController setIntegerValue:accountInfoStored.printAbility
                                      withKey:WCTSettingsKey_PrintAbility];
        
        [PPSettingsController setStringValue:accountInfoStored.role
                                     withKey:WCTSettingsKey_AccountRole];
        
        return YES;
    }
    
    return NO;
}


//================================================================================
//
//================================================================================
+ (void)updateAccountInfoFromServerWithCompleteHandler:(CompleteHandler)completeHandler
{
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *error;
            [self updateAccountInfoWithError:&error];
        }
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(hasNetwork);
        }
    }];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)isADAccount
{
    return [PPSettingsController integerValueWithKey:WCTSettingsKey_IsADAccount];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - display settings methods


//==============================================================================
//
//==============================================================================
+ (void)fetchGlobalShareRuleWithCompleteHandler:(void(^)(BOOL shareToCompany))completeHandler
{
    [self asyncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (hasNetwork)
        {
            NSError *error = nil;
            WCTRCShareRoleResponseResult *shareRule = [[WCTRestClientController shareRestClientController] globalShareRoleWithError:&error];
            if(shareRule)
            {
                BOOL shareToCompany = NO;
                if ([shareRule.data isEqualToString:WCTRC_GlobalShareRole_All])
                {
                    shareToCompany = YES;
                }
                [PPSettingsController setIntegerValue:shareToCompany withKey:WCTSettingsKey_GlobalShareToCompany];
            }
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        
        if (completeHandler)
        {
            BOOL shareToCompany = [PPSettingsController integerValueWithKey:WCTSettingsKey_GlobalShareToCompany];
            completeHandler(shareToCompany);
        }
    }];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)updateSettingsWithError:(NSError **)error
{
    BOOL result = NO;
    do {
        float minSupportedVersion = [[[WCTRestClientController shareRestClientController] minSupportedVersion] floatValue];
        
        // 取得所有設定
        WCTRCAllUserSettingsResponseResult *responseResult = [[WCTRestClientController shareRestClientController] allUserSettingsResponseWithError:error];
        if (responseResult!=nil)
        {
            WCTRCAccountPrivateSettings *settings = (WCTRCAccountPrivateSettings*)responseResult.data;
            
            // MARK: 要記錄設定修改時間，如果local較新，要更新到server上，如果server較新，要更新到手機
            NSNumber *localModifiedTime = [PPSettingsController numberValueWithKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];
            NSTimeInterval serverModifiedTimeInterval = [settings.modifyTime timeIntervalSince1970];
            __block NSError *returnError = nil;
     
            if(serverModifiedTimeInterval-[localModifiedTime doubleValue]>0.001)
            {
                // server比較新, 寫到local
                
                //////////////////////////////////////////////////
                // 東方姓名排序
                WCDisplayNameOrder oldEasternOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kEasternNameOrder];
                if([settings.asianSurnameOrder isEqualToString:WCTRC_NameOrder_FirstNameLastName])
                {
                    [PPSettingsController setIntegerValue:WCDisplayNameOrder_FirstLast withKey:WCSC_IV_kEasternNameOrder];
                }
                else
                {
                    [PPSettingsController setIntegerValue:WCDisplayNameOrder_LastFirst withKey:WCSC_IV_kEasternNameOrder];
                }

                WCDisplayNameOrder newEasternOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kEasternNameOrder];

                if(newEasternOrder!=oldEasternOrder)
                {
                    // 觸發變更顯示規則
                    [PPSettingsController setIntegerValue:0 withKey:WCSC_IV_kIsDBUpdatedDisplayRuleData];
                }

                //////////////////////////////////////////////////
                // 西方姓名排序
                WCDisplayNameOrder oldWesternOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kWesternNameOrder];
                if([settings.westernSurnameOrder isEqualToString:WCTRC_NameOrder_FirstNameLastName])
                {
                    [PPSettingsController setIntegerValue:WCDisplayNameOrder_FirstLast withKey:WCSC_IV_kWesternNameOrder];
                }
                else
                {
                    [PPSettingsController setIntegerValue:WCDisplayNameOrder_LastFirst withKey:WCSC_IV_kWesternNameOrder];
                }
                
                WCDisplayNameOrder newWesternOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kWesternNameOrder];
                
                if(newWesternOrder!=oldWesternOrder)
                {
                    // 觸發變更顯示規則
                    [PPSettingsController setIntegerValue:0 withKey:WCSC_IV_kIsDBUpdatedDisplayRuleData];
                }
                
                //////////////////////////////////////////////////
                // 中文辨識結果
                if([settings.identificationResult isEqualToString:WCTRC_ChineseRecognizeOutput_CHS])
                {
                    [PPSettingsController setIntegerValue:1 withKey:WCSC_IV_kRecogChineseOutputLangSet];
                }
                else if([settings.identificationResult isEqualToString:WCTRC_ChineseRecognizeOutput_CHT])
                {
                    [PPSettingsController setIntegerValue:2 withKey:WCSC_IV_kRecogChineseOutputLangSet];
                }
                else
                {
                    [PPSettingsController setIntegerValue:0 withKey:WCSC_IV_kRecogChineseOutputLangSet];
                }
                
                //////////////////////////////////////////////////
                // 地圖顯示
                if ([settings.mapEngine isEqualToString:WCTRC_MapEngine_Google])
                {
                    [PPSettingsController setIntegerValue:1 withKey:WCSC_IV_UserMapEngine];
                }
                else if ([settings.mapEngine isEqualToString:WCTRC_MapEngine_Baidu])
                {
                    [PPSettingsController setIntegerValue:2 withKey:WCSC_IV_UserMapEngine];
                }
                else
                {
                    [PPSettingsController setIntegerValue:0 withKey:WCSC_IV_UserMapEngine];
                }
                
                //////////////////////////////////////////////////
                // 依欄位排序
                if([settings.sortingOrder isEqualToString:WCTRC_SortByField_Name])
                {
                    [PPSettingsController setIntegerValue:WC_SBF_Name withKey:WCSC_IV_kSortingByField];
                }
                else if([settings.sortingOrder isEqualToString:WCTRC_SortByField_Company])
                {
                    [PPSettingsController setIntegerValue:WC_SBF_Company withKey:WCSC_IV_kSortingByField];
                }
                else if([settings.sortingOrder isEqualToString:WCTRC_SortByField_ModifiedTime])
                {
                    [PPSettingsController setIntegerValue:WC_SBF_ModifiedTime withKey:WCSC_IV_kSortingByField];
                }
                else
                {
                    [PPSettingsController setIntegerValue:WC_SBF_CreateTime withKey:WCSC_IV_kSortingByField];
                }

                //////////////////////////////////////////////////
                // 辨識後行為設定
                // API version 7.0之後才支援
                if(minSupportedVersion>=7.0)
                {
                    [PPSettingsController setIntegerValue:settings.editAfterRecog withKey:WCSC_IV_kShowEditAfterRecognition];
                    [PPSettingsController setIntegerValue:settings.addNoteInfoAfterRecog withKey:WCSC_IV_kShowNoteAfterRecognition];
                }
                //////////////////////////////////////////////////
                // 記錄修改時間
                [PPSettingsController setNumberValue:@(serverModifiedTimeInterval) withKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];
            }
            else if(serverModifiedTimeInterval-[localModifiedTime doubleValue]<-0.001)
            {
                // local比較新，寫回server
                //////////////////////////////////////////////////
                // 東方姓名排序
                WCDisplayNameOrder eastNameOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kEasternNameOrder];
                [[self class] setEasternNameOrder:eastNameOrder withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                    if(error)
                    {
                        returnError = error;
                    }
                }];
                
                //////////////////////////////////////////////////
                // 西方姓名排序
                WCDisplayNameOrder westNameOrder = [PPSettingsController integerValueWithKey:WCSC_IV_kWesternNameOrder];
                [[self class] setWesternNameOrder:westNameOrder withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                    if(error)
                    {
                        returnError = error;
                    }
                }];
                
                //////////////////////////////////////////////////
                // 中文辨識結果
                NSInteger chineseCharOutput = [PPSettingsController integerValueWithKey:WCSC_IV_kRecogChineseOutputLangSet];
                [[self class] setChineseRecognizeOutput:chineseCharOutput withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                    if(error)
                    {
                        returnError = error;
                    }
                }];
                //////////////////////////////////////////////////
                // 地圖顯示
                NSInteger mapEngine = [PPSettingsController integerValueWithKey:WCSC_IV_UserMapEngine];
                [[self class] setUserMapEngine:mapEngine withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                    if(error)
                    {
                        returnError = error;
                    }
                }];

                //////////////////////////////////////////////////
                // 依欄位排序
                WC_SortedByField sortByField = [PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
                [[self class] setSortByField:sortByField withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                    if(error)
                    {
                        returnError = error;
                    }
                }];
                
                //////////////////////////////////////////////////
                // 辨識後行為設定
                // API version 7.0之後才支援
                if(minSupportedVersion>=7.0)
                {
                    NSInteger editAfterRecogValue = [PPSettingsController integerValueWithKey:WCSC_IV_kShowEditAfterRecognition];
                    if(editAfterRecogValue==PPSettingsController_UnsetIntValue)
                    {
                        editAfterRecogValue = NO;
                    }
                    
                    [[self class] setEditAfterRecog:editAfterRecogValue withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                        if(error)
                        {
                            returnError = error;
                        }
                    }];
                    
                    NSInteger addNoteAfterRecogValue = [PPSettingsController integerValueWithKey:WCSC_IV_kShowNoteAfterRecognition];
                    if(addNoteAfterRecogValue==PPSettingsController_UnsetIntValue)
                    {
                        addNoteAfterRecogValue = YES;
                    }
                    
                    [[self class] setAddNoteAfterRecog:addNoteAfterRecogValue withCompleteHandler:^(NSError *error, BOOL hasNetwork) {
                        if(error)
                        {
                            returnError = error;
                        }
                    }];
                }
                //////////////////////////////////////////////////
                // 自動分享設定
                /*
                BOOL autoShare = [PPSettingsController integerValueWithKey:WCTSettingsKey_EnableBusinessCardAutoSharing];
                [[self class] setAutoShare:autoShare withCompleteHandler:nil];
                
                //////////////////////////////////////////////////
                // 自動分享列表
                NSArray *shareContactToAccountGuids = [PPSettingsController arrayValueWithKey:WCTSettingsKey_BusinessCardSharingAccount];
                
                // 如果沒有就不用再設定給server
                if([shareContactToAccountGuids count]>0)
                {
                    [[self class] setShareContatToAccountGuids:shareContactToAccountGuids  withCompleteHandler:nil];
                }
                */
                //////////////////////////////////////////////////
                // 取得新的設定修改時間，且記錄下來
                if(returnError==nil)
                {
                    WCTRCAllUserSettingsResponseResult *newResponseResult = [[WCTRestClientController shareRestClientController] allUserSettingsResponseWithError:error];
                    
                    if(newResponseResult)
                    {
                        WCTRCAccountPrivateSettings *newSettings = (WCTRCAccountPrivateSettings*)newResponseResult.data;
                        NSTimeInterval serverModifiedTimeInterval_New = [newSettings.modifyTime timeIntervalSince1970];
                        
                        [PPSettingsController setNumberValue:@(serverModifiedTimeInterval_New) withKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];
                        
//                        NSLog(@"\t ****serverModifiedTimeInterval:%lf", serverModifiedTimeInterval);
//                        NSLog(@"\t ****serverModifiedTimeInterval:%lf", serverModifiedTimeInterval_New);
                    }
                }
            }
            else
            {
                // !!相同就不用更新
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // MARK: !! 以下不管時間如何都是以server 為主
            
            //////////////////////////////////////////////////
            // 自動分享設定
            [PPSettingsController setIntegerValue:settings.autoShare withKey:WCTSettingsKey_EnableBusinessCardAutoSharing];
            
            //////////////////////////////////////////////////
            // 自動分享列表
            NSMutableArray *visibleAccountGuids = [NSMutableArray array];
            
            for (WCTRCAccountSharedTarget *acocuntSharedTarget in settings.contactShareTargets)
            {
                [visibleAccountGuids addObject:acocuntSharedTarget.accountGuid];
            }
            
            [PPSettingsController setArrayValue:visibleAccountGuids withKey:WCTSettingsKey_BusinessCardSharingAccount];
            
            //////////////////////////////////////////////////
            // 全公司分享
            if([settings.globalShareRule isEqualToString:WCTRC_GlobalShareRole_All])
            {
                [PPSettingsController setIntegerValue:YES withKey:WCTSettingsKey_GlobalShareToCompany];
            }
            else
            {
                [PPSettingsController setIntegerValue:NO withKey:WCTSettingsKey_GlobalShareToCompany];
            }
            
            //////////////////////////////////////////////////
            // CRM setting
            if ([settings.crmUrl length]!=0)
            {
                [PPSettingsController setStringValue:settings.crmUrl withKey:WCTSettingsKey_CRMLoginURL];
            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_CRMLoginURL];
            }
            
            if ([settings.userCRMAccount length]!=0)
            {
                [PPSettingsController setStringValue:settings.userCRMAccount withKey:WCTSettingsKey_CRMAccount];
            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_CRMAccount];
            }
            
            if ([settings.userCRMExportMode length]!=0)
            {
                [PPSettingsController setStringValue:settings.userCRMExportMode withKey:WCTSettingsKey_CRMExportMode];

            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_CRMExportMode];
            }
            
            if ([settings.crmCompanyAssignmentOption length]!=0)
            {
                [PPSettingsController setStringValue:settings.crmCompanyAssignmentOption withKey:WCTSettingsKey_CRMCompanyAssignmentOption];
                
            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_CRMCompanyAssignmentOption];
            }
            
            //////////////////////////////////////////////////
            // Exchange setting
            if ([settings.userStoredContactServerType length]==0 ||
                [settings.userStoredContactServerType isEqualToString:WCTRC_ExportType_None]==YES)
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_ExchangeType];
            }
            else
            {
                [PPSettingsController setStringValue:settings.userStoredContactServerType withKey:WCTSettingsKey_ExchangeType];
            }
            
            if ([settings.userContactServerAccount length]!=0)
            {
                [PPSettingsController setStringValue:settings.userContactServerAccount withKey:WCTSettingsKey_ExchangeAccount];
            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_ExchangeAccount];
            }
            
            if ([settings.userContactServerExportMode length]!=0)
            {
                [PPSettingsController setStringValue:settings.userContactServerExportMode withKey:WCTSettingsKey_ExchangeExportMode];
                
            }
            else
            {
                [PPSettingsController removeValueWithKey:WCTSettingsKey_ExchangeExportMode];
            }
            
            //////////////////////////////////////////////////
            // Max Contact count
            NSInteger maxPrivateContactCount = [[WCTRestClientController shareRestClientController] maxPrivateContactsCountWithError:error];

            if(*error==nil)
            {
                [PPSettingsController setIntegerValue:maxPrivateContactCount withKey:WCTSettingsKey_MaxPrivateContactCount];
            }
            
            NSInteger maxPublicContactCount = [[WCTRestClientController shareRestClientController] maxPublicContactsCountWithError:error];
            
            if(*error==nil)
            {
                [PPSettingsController setIntegerValue:maxPublicContactCount withKey:WCTSettingsKey_MaxPublicContactCount];
            }

            NSInteger maxServerContactCount = [[WCTRestClientController shareRestClientController] maxServerContactsCountWithError:error];
            
            if(*error==nil)
            {
                [PPSettingsController setIntegerValue:maxServerContactCount withKey:WCTSettingsKey_MaxServerContactCount];
            }
            
            if (returnError!=nil && error!=nil)
            {
                *error = returnError;
            }
        }
        
        if (*error!=nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        result = YES;
        
    } while (0);
    
    return result;
}

//==============================================================================
//
//==============================================================================
+ (void)updateSettingsFromServerWithWillStartHandler:(CompleteHandler)willStartHandler
                                     CompleteHandler:(CompleteHandler)completeHandler
{
    [self asyncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (willStartHandler)
        {
            willStartHandler(hasNetwork);
        }
        
        if (hasNetwork)
        {
            NSError *error = nil;
            
            [self updateSettingsWithError:&error];
        }
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(hasNetwork);
        }
    }];
}


//==============================================================================
//
//==============================================================================
+ (void)updateSettingsFromServerWithCompleteHandler:(CompleteHandler)completeHandler
{
    [self updateSettingsFromServerWithWillStartHandler:nil CompleteHandler:completeHandler];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setEasternNameOrder:(WCDisplayNameOrder)easternNameOrder withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (hasNetwork)
        {
            NSError *returnError = nil;
            if (easternNameOrder==WCDisplayNameOrder_FirstLast)
            {
                [[WCTRestClientController shareRestClientController] setAsianSurnameOrderWithNameOrder:WCTRC_NameOrder_FirstNameLastName error:&returnError];
            }
            else
            {
                [[WCTRestClientController shareRestClientController] setAsianSurnameOrderWithNameOrder:WCTRC_NameOrder_LastNameFirstName error:&returnError];
            }
            
            if (returnError)
            {
                error = returnError;
            }
        }
        else
        {
            
        }
        
        //////////////////////////////////////////////////
        // 記錄修改時間
        [PPSettingsController setNumberValue:@([[NSDate date] timeIntervalSince1970]) withKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];

    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setWesternNameOrder:(WCDisplayNameOrder)westernNameOrder withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *returnError = nil;
            if (westernNameOrder==WCDisplayNameOrder_FirstLast)
            {
                [[WCTRestClientController shareRestClientController] setWesternSurnameOrderWithNameOrder:WCTRC_NameOrder_FirstNameLastName error:&returnError];
            }
            else
            {
                [[WCTRestClientController shareRestClientController] setWesternSurnameOrderWithNameOrder:WCTRC_NameOrder_LastNameFirstName error:&returnError];
            }
            
            
            if (returnError)
            {
                error = returnError;
            }
        }
        else
        {
            
        }
        
        //////////////////////////////////////////////////
        // 記錄修改時間
        [PPSettingsController setNumberValue:@([[NSDate date] timeIntervalSince1970]) withKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];

    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setAutoShare:(BOOL)autoShare withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *returnError = nil;
            [[WCTRestClientController shareRestClientController] setAutoShareSettingWithEnable:autoShare error:&returnError];
            
            if (returnError)
            {
                error = returnError;
            }
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setShareContatToAccountGuids:(NSArray *)shareContatToAccountGuids withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    return [self setShareAccountGuids:shareContatToAccountGuids withItem:WCTRC_ShareItem_Contact completeHandler:completeHandler];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setShareAccountGuids:(NSArray *)shareContatToAccountGuids withItem:(NSString *)item completeHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    if (shareContatToAccountGuids==nil ||
        item==nil)
    {
        return NO;
    }
    
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *returnError = nil;

            WCTRCAccountSharedTargetListRequest * request = [[WCTRCAccountSharedTargetListRequest alloc] init];
            request.accountGuids = shareContatToAccountGuids;
            request.shareItem = item;
            [[WCTRestClientController shareRestClientController] updateAccountSharedTargetListWithRequest:request error:&returnError];
            [request release];
            
            if (returnError)
            {
                error = returnError;
            }

        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setChineseRecognizeOutput:(NSInteger)chineseRecognizeOutput withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *error = nil;
            
            NSString *chineseRecognizeOutputString = WCTRC_ChineseRecognizeOutput_ContactContent;
            switch (chineseRecognizeOutput)
            {
                case 1:
                {
                    chineseRecognizeOutputString = WCTRC_ChineseRecognizeOutput_CHS;
                    break;
                }
                case 2:
                {
                    chineseRecognizeOutputString = WCTRC_ChineseRecognizeOutput_CHT;
                    break;
                }
                default:
                    break;
            }

            NSError *returnError = nil;
            
            [[WCTRestClientController shareRestClientController] setChineseRecognizeWithOutput:chineseRecognizeOutputString error:&returnError];
            
            if (returnError)
            {
                error = returnError;
            }

        }
        else
        {
            
        }

    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setUserMapEngine:(NSInteger)userMapEngine withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *error = nil;
            
            NSString *userMapEngineString = WCTRC_MapEngine_Auto;
            switch (userMapEngine)
            {
                case 1:
                {
                    userMapEngineString = WCTRC_MapEngine_Google;
                    break;
                }
                case 2:
                {
                    userMapEngineString = WCTRC_MapEngine_Baidu;
                    break;
                }
                default:
                    break;
            }
            
            NSError *returnError = nil;

            [[WCTRestClientController shareRestClientController] setMapEngineWithMapEngine:userMapEngineString error:&returnError];
            
            
            if (returnError)
            {
                error = returnError;
            }

        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setSortByField:(WC_SortedByField)sortByField withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *error = nil;
            
            NSString *sortByFieldString = WCTRC_SortByField_Name;
            switch (sortByField)
            {
                case WC_SBF_Company:
                {
                    sortByFieldString = WCTRC_SortByField_Company;
                    break;
                }
                case WC_SBF_CreateTime:
                {
                    sortByFieldString = WCTRC_SortByField_CreateTime;
                    break;
                }
                case WC_SBF_ModifiedTime:
                {
                    sortByFieldString = WCTRC_SortByField_ModifiedTime;
                    break;
                }
                default:
                    break;
            }
            [[WCTRestClientController shareRestClientController] setSortByFieldWithField:sortByFieldString error:&error];

        }
        else
        {
            
        }
        
        //////////////////////////////////////////////////
        // 記錄修改時間
        [PPSettingsController setNumberValue:@([[NSDate date] timeIntervalSince1970]) withKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setEditAfterRecog:(BOOL)editAfterRecog withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *returnError = nil;
            [[WCTRestClientController shareRestClientController] setEditAfterRecogSettingWithEnable:editAfterRecog error:&returnError];
            
            if (returnError)
            {
                error = returnError;
            }
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}



//==============================================================================
//
//==============================================================================
+ (BOOL)setAddNoteAfterRecog:(BOOL)addNoteAfterRecog withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            NSError *returnError = nil;
            [[WCTRestClientController shareRestClientController] setAddNoteInfoAfterRecogWithEnable:addNoteAfterRecog error:&returnError];
            
            if (returnError)
            {
                error = returnError;
            }
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}

//==============================================================================
//
//==============================================================================
+ (BOOL)setCRMExportMode:(NSString *)exportMode
     companyAssignmentOption:(NSString *)companyAssignmentOption
     withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            [[WCTRestClientController shareRestClientController] setCRMExportModeWithCRMType:WCTRC_CRMType_Salesforce
                                                                               CTMExportMode:exportMode
                                               CRMCompanyAssignmentOption:companyAssignmentOption        error:&error];
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setExchangeExportMode:(NSString *)exportMode withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            [[WCTRestClientController shareRestClientController] setExchangeServerExportModeWithMode:exportMode
                                                                                               error:&error];
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)setQcontactzExportMode:(NSString *)exportMode withCompleteHandler:(void(^)(NSError *error, BOOL hasNetwork))completeHandler
{
    __block NSError *error = nil;
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        if (hasNetwork)
        {
            [[WCTRestClientController shareRestClientController] setQContactzExportMode:exportMode error:&error];
        }
        else
        {
            
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        if (completeHandler)
        {
            completeHandler(error, hasNetwork);
        }
    }];
    
    
    return NO;
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - get system category methods


//==============================================================================
//
//==============================================================================
+ (BOOL)fetchSystemCategoryGuidsWithError:(NSError **)error
{
    WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text];
    NSDictionary *groupGuidMapping = [dataController groupIDAndGuidMappingDict];
    
    // 還沒有guid才要做
    if ([[groupGuidMapping objectForKey:@(WC_GID_All)] length]==0)
    {
        // 取得目前列表
        NSString *currentAccountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
        
        WCTRCMLCategoryInfoListResponseResult *result = [[WCTRestClientController shareRestClientController] multiLayerCategoryListWithAccountGuid:currentAccountGuid error:error];
        
        if (*error==nil)
        {
            [dataController release];
            return NO;
        }
        
        // 將All, other的guid填入資料庫
        for (WCTRCCategoryInfoMultiLayer *categoryInfo in result.data)
        {
            if([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_All])
            {
                [dataController setDefaultGroupGuid:categoryInfo.guid withGroupID:WC_GID_All];
            }
            else if ([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_Other])
            {
                [dataController setDefaultGroupGuid:categoryInfo.guid withGroupID:WC_GID_Unfiled];
            }
        }
        
    }
    
    [dataController release];
    return YES;    
}


//==============================================================================
//
//==============================================================================
+ (BOOL)fetchSystemCategoryGuidsWithCompleteHandler:(void(^)(BOOL success, NSString *categoryGuid_All, NSString *categoryGuid_Other, NSString *categoryGuid_Favorites))completeHandler
{
    __block BOOL success = NO;
    
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (hasNetwork)
        {
            NSError *error = nil;
            
            success = [self fetchSystemCategoryGuidsWithError:&error];
        }
    } completeHandler:^(BOOL hasNetwork) {
        
        if (completeHandler)
        {
            WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text];
            NSDictionary *groupGuidMapping = [dataController groupIDAndGuidMappingDict];
            
            NSString *allGuid = [groupGuidMapping objectForKey:[NSString stringWithInteger:WC_GID_All]];
            NSString *otherGuid = [groupGuidMapping objectForKey:[NSString stringWithInteger:WC_GID_Unfiled]];;
            NSString *favoritesGuid = nil;

            [dataController release];

            completeHandler(success, allGuid, otherGuid, favoritesGuid);
        }
    }];
    
    
    return NO;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - custom field methods


//==============================================================================
//
//==============================================================================
+ (BOOL)fetchSystemCustomFieldInfosWithCategory:(NSString *)category error:(NSError **)error
{
    WCTRCCustomFieldListResponseResult *result = [[WCTRestClientController shareRestClientController] customFieldsWithCategory:category error:error];
    
    if (*error==nil)
    {
        WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text];
        NSMutableArray *localCustomFields = [dataController copyAllCustomFieldInfos];

        // 資料寫入資料庫
        // 不在localCustomFields，就要新增
        // 已經在的就更新
        // 不存在的要刪除
        for (WCTRCCustomFieldInfo *remoteCustomFieldInfo in result.data)
        {
            // 先檢查是否存在local
            BOOL isExist = NO;
            for (WCCustomFieldInfo *localCustomFieldInfo in localCustomFields)
            {
                if ([localCustomFieldInfo.guid isEqualToString:remoteCustomFieldInfo.guid]==YES)
                {
                    isExist = YES;
                    
                    // 已存在，update
                    WCCustomFieldInfo *convertedCustomFieldInfo =
                    [WCCustomFieldInfo customFieldInfoWithWCTRCCustomFieldInfo:remoteCustomFieldInfo];
                    
                    [dataController updateCustomFieldWithInfo:convertedCustomFieldInfo];
                    
                    // 移除已處理
                    [localCustomFields removeObject:localCustomFieldInfo];
                    break;
                }
            }
            
            if(isExist==NO)
            {
                // 新增
                WCCustomFieldInfo *convertedCustomFieldInfo =
                [WCCustomFieldInfo customFieldInfoWithWCTRCCustomFieldInfo:remoteCustomFieldInfo];
                
                [dataController addCustomFieldWithInfo:convertedCustomFieldInfo];
            }
        }
        
        // local剩下來的是要刪除的
        for (WCCustomFieldInfo *localCustomFieldInfo in localCustomFields)
        {
            [dataController removeCustomFieldWithGuid:localCustomFieldInfo.guid];
        }
        
        [localCustomFields release];
        [dataController release];
        
        return YES;
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)fetchSystemCustomFieldInfosWithCompleteHandler:(void(^)(BOOL success, BOOL hasNetwork))completeHandler
{
    __block BOOL success = NO;
    
    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (hasNetwork)
        {
            NSError *error = nil;
            
            success = [self fetchSystemCategoryGuidsWithError:&error];
        }
        
    } completeHandler:^(BOOL hasNetwork) {
        
        if (completeHandler)
        {
            completeHandler(success, hasNetwork);
        }
    }];
    
    
    return NO;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - 


//==============================================================================
//
//==============================================================================
+ (void)updateAllDataFromServerWithCompleteHandler:(void(^)(BOOL hasNetwork, NSError *error))completeHandler
{
    __block NSError *returnError = nil;

    [self syncCheckNetworkWithActionBlock:^(BOOL hasNetwork) {
        
        if (hasNetwork)
        {
            // 取得系統群組的guid
            [self fetchSystemCategoryGuidsWithError:&returnError];

            // !!如果是網路錯誤就不再往下做
            if ([returnError findNetworkError]!=nil)
            {
                return ;
            }
            else if ([returnError code]==NSErrorCustom_Code_ParameterInvalidity)
            {
                // 如果是參數錯誤也停止，因為是沒有token造成
                return;
            }

            //////////////////////////////////////////////////
            // 取得帳號列表
            [self updateAccountListWithError:&returnError];

            // !!如果是網路錯誤就不再往下做
            if ([returnError findNetworkError]!=nil)
            {
                return ;
            }
            else if ([returnError code]==NSErrorCustom_Code_ParameterInvalidity)
            {
                // 如果是參數錯誤也停止，因為是沒有token造成
                return;
            }
            
            //////////////////////////////////////////////////
            // 取得自訂欄位資訊
            [self fetchSystemCustomFieldInfosWithCategory:WCTRC_CustomField_Category_Contact error:&returnError];
            
            // !!如果是網路錯誤就不再往下做
            if ([returnError findNetworkError]!=nil)
            {
                return ;
            }
            else if ([returnError code]==NSErrorCustom_Code_ParameterInvalidity)
            {
                // 如果是參數錯誤也停止，因為是沒有token造成
                return;
            }
            
            //////////////////////////////////////////////////
            // 取得在server上的設定值
            [self updateSettingsWithError:&returnError];
        }
    } completeHandler:^(BOOL hasNetwork) {
       
        if (completeHandler!=NULL)
        {
            completeHandler (hasNetwork, returnError);
        }
    }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - display string


//==============================================================================
//
//==============================================================================
+ (NSString *)displayStringForVisibleAccountGuids:(NSArray *)visibleAccountGuids ownerGuid:(NSString *)ownerGuid
{
    NSMutableString *result = [NSMutableString string];
    
    WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text];
    
    //////////////////////////////////////////////////
    // 加入owner
    
    WCTAccountRelationModel *accountRelationModel = [dataController accountRelationWithAcountGuid:ownerGuid];
    if (accountRelationModel)
    {
        [result appendString:accountRelationModel.name?:@"Unknown"];
        [result appendFormat:@"(%@)",WCTADCString_Owner];
    }
    
    
    //////////////////////////////////////////////////
    // 加入可檢視用戶
    
    for (NSString *accontGuid in visibleAccountGuids)
    {
        // !! 如果擁有者在可檢視用戶中要跳過(前面加過了)
        if ([ownerGuid length]>0 &&
            [visibleAccountGuids containsObject:ownerGuid])
        {
            continue;
        }
        
        if ([result length]>0)
        {
            [result appendString:@"; "];
        }
        
        WCTAccountRelationModel *accountRelationModel = [dataController accountRelationWithAcountGuid:accontGuid];
        if (accountRelationModel && [accountRelationModel.name length]>0)
        {
            [result appendString:accountRelationModel.name];
        }
    }
    
    [dataController release];
    
    
    //////////////////////////////////////////////////
    // 加入後綴字串
    
    if ([result length]>0)
    {
        [result appendString:@"; "];
    }
    
    [result appendString:WCTADCString_StringAfterVisibleAccount];
    
    return [NSString stringWithString:result];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - reset methods


//==============================================================================
//
//==============================================================================
+ (void)resetSetttingsModifiedTime
{
    [PPSettingsController removeValueWithKey:WCTAccountDataController_SettingsKey_SettingsModifiedTime];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - get personal category methods

//================================================================================
//
//================================================================================
+ (void)updateCategoriesFromServerWithAccountGuid:(NSString *)accountGuid
                                   dataController:(WCTDataController *)dataController
                               isHelpScanningMode:(BOOL)isHelpScanningMode
                                  completeHandler:(void(^)(NSError *error, BOOL hasRemovedGroup, BOOL hasRemovedNonEmptyCardsGroup))completeHandler
{
    NSError *error = nil;
    BOOL hasRemovedGroup = NO;
    BOOL hasRemovedNonEmptyCardsGroup = NO;
    
    do
    {
        if([accountGuid length] == 0)
        {
            error = PPErrorMake(NSErrorCustom_Code_ParameterInvalidity, @"Invalid accountGuid", nil);
            break;
        }
        
        if(dataController == nil)
        {
            error = PPErrorMake(NSErrorCustom_Code_ParameterInvalidity, @"Invalid dataController", nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // prepare data

        WCTRCMLCategoryInfoListResponseResult *response1 = [[WCTRestClientController shareRestClientController] multiLayerCategoryListWithAccountGuid:accountGuid error:&error];
        
        if(error != nil)
        {
            break;
        }
        
        // get all categories from server
        NSArray *allCategories = response1.data;

        // get groups from database
        NSMutableArray *groups = [[dataController copyAllGroupsAndGetCardCount:YES] autorelease];
        
        
        //////////////////////////////////////////////////
        // update database
        
        // 先取'All'的guid
        NSString *groupAllGuid = nil;
        
        for(WCTRCCategoryInfoMultiLayer *categoryInfo in allCategories)
        {
            if([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_All] == YES)
            {
                groupAllGuid = categoryInfo.guid;
                break;
            }
        }

        
        // 更新DB (不要全部清除再加入，資料庫空間會一直長大。)
        for(WCTRCCategoryInfoMultiLayer *categoryInfo in allCategories)
        {
            if([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_All] == YES)
            {
                for(WCGroupModel *group in groups)
                {
                    if(group.ID == WC_GID_All)
                    {
                        if([group.guid isEqualToString:groupAllGuid] == NO &&
                           [dataController updateGroupGuid:groupAllGuid withID:group.ID] == NO)
                        {
                            error = PPErrorMake(NSErrorCustom_Code_OperationFailed, @"Failed to updadte groupGuid of 'All'", nil);
                            break;
                        }
                        
                        // 移除已處理的group
                        [groups removeObject:group];
                        break;
                    }
                }
            }
            else if([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_Other] == YES)
            {
                for(WCGroupModel *group in groups)
                {
                    if(group.ID == WC_GID_Unfiled)
                    {
                        if([group.guid isEqualToString:categoryInfo.guid] == NO &&
                           [dataController updateGroupGuid:categoryInfo.guid withID:group.ID] == NO)
                        {
                            error = PPErrorMake(NSErrorCustom_Code_OperationFailed, @"Failed to updadte groupGuid of 'Unfiled'", nil);
                            break;
                        }

                        if([group.superGroupGuid isEqualToString:groupAllGuid] == NO &&
                           [dataController updateSuperGroupGuid:groupAllGuid withGroupID:group.ID] == NO)
                        {
                            error = PPErrorMake(NSErrorCustom_Code_OperationFailed, @"Failed to updadte superGroupGuid of 'Unfiled'", nil);
                            break;
                        }

                        // 移除已處理的group
                        [groups removeObject:group];
                        break;
                    }
                }
            }
            else if([categoryInfo.categoryType isEqualToString:WCTRC_CategoryType_Favorite] == YES)
            {
                // database中沒有favorite group存在，另存到設定中。
                [PPSettingsController setStringValue:categoryInfo.guid withKey:WCTSettingsKey_FavoriteCategoryGUID];
                continue;
            }
            else // WCTRC_CategoryType_Normal
            {
                // MARK:幫助掃描模式，不用讀取FromXxx群組
                if(isHelpScanningMode == YES && [categoryInfo.secretaryAccountGuid length] > 0)
                {
                    continue;
                }

                //////////////////////////////////////////////////

                BOOL isExist = NO;
                
                for(WCGroupModel *group in groups)
                {
                    if([group.guid isEqualToString:categoryInfo.guid] == YES)
                    {
                        isExist = YES;
                        
                        // name異動
                        if([group.name isEqualToString:categoryInfo.name] == NO &&
                           [dataController updateGroupName:categoryInfo.name withID:[NSString stringWithInteger:group.ID]] == NO)
                        {
                            error = PPErrorMake(NSErrorCustom_Code_OperationFailed, @"Failed to updadte groupName", nil);
                            break;
                        }
                        
                        // superGroup異動
                        if([group.superGroupGuid isEqualToString:categoryInfo.parentCategoryGuid] == NO &&
                           [dataController updateSuperGroupGuid:categoryInfo.parentCategoryGuid withGroupID:group.ID] == NO)
                        {
                            error = PPErrorMake(NSErrorCustom_Code_OperationFailed, @"Failed to updadte superGroupGuid", nil);
                            break;
                        }

                        // 移除已處理的group
                        [groups removeObject:group];
                        break;
                    }
                }
                
                // 新增
                if(isExist == NO)
                {
                    if([dataController addOrUpdateGroupWithGuid:categoryInfo.guid
                                                           name:categoryInfo.name
                                                 superGroupGuid:categoryInfo.parentCategoryGuid
                                                         helper:categoryInfo.secretaryAccountGuid] == NO)
                    {
                        NSString *reason = [NSString stringWithFormat:@"Failed to add group '%@'", categoryInfo.name];
                        error = PPErrorMake(NSErrorCustom_Code_OperationFailed, reason, nil);
                        break;
                    }
                }
            }
            
            if(error != nil)
            {
                break;
            }
        }
        
        
        //////////////////////////////////////////////////
        // 刪除剩下的local group
        
        if(isHelpScanningMode == YES)
        {
            // MARK: 幫助掃描模式，剩下的名片要歸類到FromXxx
            WC_GroupID groupFromXxxID = WC_GID_None;
            NSString *selfGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];

            for(WCGroupModel *group in groups)
            {
                if(group.helper != nil && [group.helper isEqualToString:selfGuid] == YES)
                {
                    groupFromXxxID = group.ID;
                    [groups removeObject:group];
                    break;
                }
            }

            [dataController setMustHaveGroupIDAfterRemoveGroup:groupFromXxxID];
            
            for(WCGroupModel *group in groups)
            {
                if(hasRemovedNonEmptyCardsGroup==NO)
                {
                    NSMutableArray *cardIDArray = [dataController copyCardIDArrayWithGroupID:[NSString stringWithInteger:group.ID]];
                    
                    if(cardIDArray.count>0)
                    {
                        hasRemovedNonEmptyCardsGroup = YES;
                    }
                    
                    [cardIDArray release];
                }
                
                //////////////////////////////////////////////////

                hasRemovedGroup = YES;
                [dataController removeGroupWithID:[NSString stringWithInteger:group.ID]];
            }

            [dataController setMustHaveGroupIDAfterRemoveGroup:WC_GID_Unfiled];
        }
        else
        {
            // 非幫助掃描模式，直接移除剩下的group
            for(WCGroupModel *group in groups)
            {
                [dataController removeGroupWithID:[NSString stringWithInteger:group.ID]];
            }
        }

        
        //////////////////////////////////////////////////
        // MARK: 更新釘選順序
        
        WCTRCMLCategoryInfoListResponseResult *response3 = [[WCTRestClientController shareRestClientController] toggledMultiLayerCategoryListWithError:&error];
        
        if(error != nil)
        {
            break;
        }
        
        // get toggled categories from server
        NSArray *toggledCategories = response3.data;

        [dataController clearAllGroupPinnedOrder];
        
        for(int i=0; i <[toggledCategories count]; i++)
        {
            WCTRCCategoryInfoMultiLayer *categoryInfo = [toggledCategories objectAtIndex:i];
            
            [dataController updatePinnedOrder:i+1 withGroupGuid:categoryInfo.guid];
        }        
    }
    while(0);

    if(completeHandler != nil)
    {
        completeHandler(error, hasRemovedGroup, hasRemovedNonEmptyCardsGroup);
    }
}


//==============================================================================
//
//==============================================================================
+ (void)updateCategoriesFromServerWithCompleteHandler:(void(^)(NSError *error))completeHandler;
{
    NSString *currentAccountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
    WCTDataController *dataController = [[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text];
    
    [WCTAccountDataController updateCategoriesFromServerWithAccountGuid:currentAccountGuid
                                                         dataController:dataController
                                                     isHelpScanningMode:NO
                                                        completeHandler:^(NSError *error, BOOL hasRemovedGroup, BOOL hasRemovedNonEmptyCardsGroup)
    {
                                                            
        [dataController release];
        
        if (completeHandler)
        {
            completeHandler(error);
        }
    }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - handle unsync categories methods (private)


//==============================================================================
//
//==============================================================================
+ (WCTRCMLCategoryPathCreationResponseResult *)createCategoryWithCategoryRequest:(WCTRCMLCategoryPathCreationRequest *)categoryRequest
                                                                       error:(NSError **)error
{
    __block WCTRCMLCategoryPathCreationResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[[WCTRestClientController shareRestClientController] createMultiLayerCategoryPathWithCreationRequest:categoryRequest error:&error] retain];
            [error retain];
//            
//            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncCreateCategoryWithCategoryRequest (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        // 這邊先不cancel
//        if(cancel_ == YES)
//        {
//            // 在外部檢查是否cancel再產生error code
//            return NO;
//        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTADC_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
+ (WCTRCMLCategoryUpdateResponseResult *)udateCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                  name:(NSString *)name
                                                                 error:(NSError **)error

{
    __block WCTRCMLCategoryUpdateResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[[WCTRestClientController shareRestClientController] updateMultiLayerCategoryWithGuid:categoryGuid
                                                                                                       categoryName:name
                                                                                                              error:&error] retain];
            [error retain];
            
//            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncUpdateCategoryWithCategoryRequest (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        // 這邊先不cancel
//        if(cancel_ == YES)
//        {
//            // 在外部檢查是否cancel再產生error code
//            return NO;
//        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTADC_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
+ (WCTRCMLCategoryDeleteResponseResult *)deleteCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                        error:(NSError **)error

{
    __block WCTRCMLCategoryDeleteResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[[WCTRestClientController shareRestClientController] deleteMultiLayerCategoryWithCategoryGuid:categoryGuid error:&error] retain];
            [error retain];
            
//            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncDeleteCategoryWithCategoryGuid (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        // 這邊先不cancel

//        if(cancel_ == YES)
//        {
//            // 在外部檢查是否cancel再產生error code
//            return NO;
//        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTADC_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - handle unsync categories methods

//==============================================================================
//
//==============================================================================
+ (void)handleUnsyncCategoriesToServer
{
    WCTDataController *dataController = [[[WCTDataController alloc] initWithAccessMode:WCDC_AM_Text] autorelease];
    NSArray *syncActions = [[dataController copyGroupSyncActions] autorelease];

    if ([syncActions count]==0)
    {
        return ;
    }
    
    //////////////////////////////////////////////////
    
    for (WCTGroupSyncActionModel *groupSyncActionModel in syncActions)
    {
        switch (groupSyncActionModel.actionType)
        {
            case WCTSyncActionType_Add:
            {
                // 新增到server
                NSError *error = nil;
                WCTRCMLCategoryPathCreationRequest *categoryInfo = [[[WCTRCMLCategoryPathCreationRequest alloc] init] autorelease];
                WCGroupModel *localGroupModel = [dataController groupWithGuid:groupSyncActionModel.dataGuid];
                
                //////////////////////////////////////////////////
                // send request
                
                categoryInfo.categoryPath = localGroupModel.name; // 舊版只有單層
                categoryInfo.ownerAccountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
                
                WCTRCMLCategoryPathCreationResponseResult *response = nil;
                response = [[self class] createCategoryWithCategoryRequest:categoryInfo error:&error];
                
                // !! 如果是名稱已存在，要取目前server上的類別來用
                if(error.code==WCTServer_Category_ErrorCode_NameAlreadyExists)
                {
                    
                }
                
                if(error == nil)
                {
                    // !! server端建立完要更新local guid
                    [dataController updateGroupGuid:response.data withID:localGroupModel.ID];
                }
                
                break;
            }
            case WCTSyncActionType_Modify:
            {
                NSError *error = nil;
                WCGroupModel *localGroupModel = [dataController groupWithGuid:groupSyncActionModel.dataGuid];
                
                //////////////////////////////////////////////////
                // send request
                
                [[self class] udateCategoryWithCategoryGuid:groupSyncActionModel.dataGuid name:localGroupModel.name error:&error];
                break;
            }
            case WCTSyncActionType_Delete:
            {
                NSError *error = nil;
                
                [[self class] deleteCategoryWithCategoryGuid:groupSyncActionModel.dataGuid error:&error];
                break;
            }
            default:
                break;
        }
        
        // !! 目前不管成功或失敗都先清除同步紀錄
        [dataController removeGroupSyncActionWithGuid:groupSyncActionModel.dataGuid];
    }
}


@end
