//
//  WCShareFlowController.m
//  Example
//
//  Created by Andrew on 2015/8/3.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "WCShareFlowController.h"
#import <Contacts/Contacts.h>
// Define
#import "WCCloudFileDefine.h"

// Model
#import "WCFieldModel.h"
#import "NSError+Custom.h"

// View
#import "PPBusyView.h"

// Controller
#import "SFCShareViewController.h"
#import "PPNavigationController.h"
#import "WCDisplayNameController.h"
#import "PPZipController.h"
#import "WCToolController.h"
#import "WCToastController.h"
#import "PPWorldcardAddressController.h"
#import "PPSelectController.h"
#import "PPAlertController.h"

// DataController
#import "WCCSVDataController.h"
#import "WCXFDataController.h"
#import "WCVCFDataController.h"
#import "UIDocumentInteractionController+CanOpenIn.h"
#import "WCXLSXDataController.h"
#import "PPMailComposeViewController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static NSUInteger const WCShareFlowController_MaxContactFileCount    = 5;
static double const WCShareFlowController_MinimumSystemMemoryRequestBytes = 1024*1024;

static NSString *const WCShareFlowController_ContactShareRootFolder  = @"ContactShareFolder";
static NSString *const WCShareFlowController_ContactFileFolder_VCF   = @"FolderVCF";
static NSString *const WCShareFlowController_ContactFileFolder_Image = @"FolderImage";

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, WCShareFlowController_Error)
{
    WCShareFlowController_Error_Unknown = 0,
    WCShareFlowController_Error_DelegateError,      // 需要檢查delegate是否可用
    WCShareFlowController_Error_NoCardToWriteSimple,
    WCShareFlowController_Error_NoContactInfoToWriteForText,
    WCShareFlowController_Error_NoCardToWriteForImage,
    WCShareFlowController_Error_NoAddressBookPermission
};

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCShareFlowController Extension

@interface WCShareFlowController () <PPSelectControllerDelegate, SFCShareViewControllerDelegate, WCCSVDataControllerDelegate, WCXFDataControllerDelegate, WCXLSXDataControllerDelegate, UIDocumentInteractionControllerDelegate>

@property (nonatomic, assign) WCDisplayNameOrder      displayEastNameOrder;
@property (nonatomic, assign) WCDisplayNameOrder      displayWestNameOrder;

@property (nonatomic, retain) PPBusyView             *busyView;

@property (nonatomic, retain) UIDocumentInteractionController *openInController;
@property (nonatomic, retain) UIActivityViewController        *activityViewController;
@property (nonatomic, retain) SFCShareViewController *shareViewController;
@property (nonatomic, retain) PPSelectController     *selectController;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCShareFlowController Implementation

@implementation WCShareFlowController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if ((self=[super init]))
    {
        _displayEastNameOrder = WCDisplayNameOrder_None;
        _displayWestNameOrder = WCDisplayNameOrder_None;
        
        //////////////////////////////////////////////////
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvUIApplicationDidEnterBackgroundNotification:)
                                                     name:UIApplicationDidEnterBackgroundNotification
                                                   object:nil];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    _delegate = nil;
    
    [_contactIdentifierArray release];
    _contactIdentifierArray = nil;
    
    _openInController.delegate = nil;
    [_openInController release];
    _openInController = nil;
    
    [_activityViewController release];
    _activityViewController = nil;
    
    _shareViewController.delegate = nil;
    [_shareViewController release];
    _shareViewController = nil;
    
    _selectController.delegate = nil;
    [_selectController release];
    _selectController = nil;
    
    [_busyView removeFromSuperview];
    [_busyView release];
    _busyView = nil;
    
    //////////////////////////////////////////////////
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIApplicationWillEnterForegroundNotification object:nil];
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification

//================================================================================
//
//================================================================================
- (void)recvUIApplicationDidEnterBackgroundNotification:(NSNotification *)notify
{
    if (self.openInController!=nil)
    {
        [self.openInController dismissPreviewAnimated:YES];
        [self.openInController dismissMenuAnimated:YES];
        self.openInController = nil;
    }
    else if (self.activityViewController!=nil)
    {
        [self.activityViewController dismissViewControllerAnimated:YES completion:nil];
        self.activityViewController = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (void)showBusyView:(BOOL)show
{
    if(show==YES)
    {
        if(self.busyView==nil)
        {
            _busyView = [[PPBusyView alloc] initWithSuperView:[UIApplication sharedApplication].keyWindow];
        }
    }
    else
    {
        [self.busyView removeFromSuperview];
        self.busyView = nil;
    }
}


//================================================================================
//
//================================================================================
- (NSString *)singleFileCompleteNameWithFormatString:(NSString *)formatString
{
    NSString *deviceName = [[UIDevice currentDevice] name];
    NSString *dateTime   = nil;
    
    //!! 壓縮檔名中不能有空白
    deviceName = [deviceName stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    if(dateFormatter!=nil)
    {
        [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
        
        dateTime = [dateFormatter stringFromDate:[NSDate date]];
        [dateFormatter release];
    }
    
    return [NSString stringWithFormat:WCCloudFileDefine_MultiContactsFileNameFormat, deviceName, dateTime, formatString];
}


//================================================================================
// 因為CSV有兩種樣式，需於時間後面添加_Gmail或_Outlook
//================================================================================
- (NSString *)singleFileCompleteNameForFileTypeCSV:(WCFileTypeSelectView_ContactFileType)fileType
{
    NSString *deviceName = [[UIDevice currentDevice] name];
    NSString *dateTime   = nil;
    
    //!! 壓縮檔名中不能有空白
    deviceName = [deviceName stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    if(dateFormatter!=nil)
    {
        [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
        
        dateTime = [dateFormatter stringFromDate:[NSDate date]];
        [dateFormatter release];
    }
    
    //!! 添加額外的分類字串
    switch (fileType)
    {
        case WCFileTypeSelectView_ContactFileType_CSV_iWorks:
        {
            dateTime = [dateTime stringByAppendingString:@"_iWorks"];
            break;
        }
        case WCFileTypeSelectView_ContactFileType_CSV_Gmail:
        {
            dateTime = [dateTime stringByAppendingString:@"_Gmail"];
            break;
        }
        case WCFileTypeSelectView_ContactFileType_CSV_Outlook:
        {
            dateTime = [dateTime stringByAppendingString:@"_Outlook"];
            break;
        }
        default:
            break;
    }
    
    return [NSString stringWithFormat:WCCloudFileDefine_MultiContactsFileNameFormat, deviceName, dateTime, @"csv"];
}


//================================================================================
//
//================================================================================
- (NSString *)zipFileCompleteNameWithFormatString:(NSString *)formatString
{
    NSString *deviceName = [[UIDevice currentDevice] name];
    NSString *dateTime   = nil;
    
    //!! 壓縮檔名中不能有空白
    deviceName = [deviceName stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    if(dateFormatter!=nil)
    {
        [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
        
        dateTime = [dateFormatter stringFromDate:[NSDate date]];
        [dateFormatter release];
    }
    
    return [NSString stringWithFormat:WCCloudFileDefine_ZipContactFileNameFormat, formatString, deviceName, dateTime];
}


//==============================================================================
//
//==============================================================================
- (void)showActivityFromShareViewControllerWithItems:(NSArray *)activityItems deleteFolderPath:(NSString *)deleteFolderPath
{
    __block typeof(self) blockSelf = self;
    
    _activityViewController = [[UIActivityViewController alloc] initWithActivityItems:activityItems applicationActivities:nil];
    
    [self.activityViewController setCompletionWithItemsHandler:^(NSString *activityType, BOOL completed, NSArray *returnedItems, NSError *activityError){
        
        for (id object in activityItems)
        {
            // item如果是file url才要remove
            if([object isKindOfClass:[NSURL class]])
            {
                [[NSFileManager defaultManager] removeItemAtURL:(NSURL *)object error:nil];
            }
        }
        
        if (deleteFolderPath!=nil)
        {
            [[NSFileManager defaultManager] removeItemAtPath:deleteFolderPath error:nil];
        }
        
        //////////////////////////////////////////////////
        
        if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:didFinishWithError:)]==YES)
        {
            NSError *fileError = nil;
            if (completed==NO)
            {
                if([activityType isEqualToString:UIActivityTypeMail] &&
                   [PPMailComposeViewController canSendMail]==NO)
                {
                    fileError = [NSError errorWithDomain:NSStringFromClass([blockSelf class])
                                                    code:WCShareFlow_Error_NoSettingEMail
                                              lineNumber:(__LINE__)
                                  localizedFailureReason:@"No mail in Settings"
                                                  object:nil];
                }
                else
                {
                    fileError = [NSError errorWithDomain:NSStringFromClass([blockSelf class])
                                                    code:WCShareFlow_Error_CancelActivityAction
                                              lineNumber:(__LINE__)
                                  localizedFailureReason:@"Cancel Activity"
                                                  object:nil];
                }
                
            }
            
            [blockSelf.delegate shareFlowController:blockSelf didFinishWithError:fileError];
        }
    }];
    
    if (UI_USER_INTERFACE_IDIOM() == UIUserInterfaceIdiomPhone)
    {
        //if iPhone
        [self.shareViewController presentViewController:self.activityViewController animated:YES completion:nil];
    }
    else
    {
        UIBarButtonItem *shareBarButton = [[[UIBarButtonItem alloc] initWithCustomView:self.shareViewController.ppButtonShare] autorelease];
        if (shareBarButton!=nil)
        {
            if ([[[UIDevice currentDevice] systemVersion] floatValue]<8.0)
            {
                //if iPad
                UIPopoverController *popover = [[UIPopoverController alloc] initWithContentViewController:self.activityViewController];
                [popover presentPopoverFromBarButtonItem:shareBarButton permittedArrowDirections:UIPopoverArrowDirectionAny animated:YES];
                [popover release];
            }
            else
            {
                if ( [self.activityViewController respondsToSelector:@selector(popoverPresentationController)]==YES)
                {
                    self.activityViewController.popoverPresentationController.barButtonItem = shareBarButton;
                    [self.shareViewController presentViewController:self.activityViewController animated:YES completion:nil];
                }
            }
            
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (void)showFromSuperViewController:(UIViewController *)superViewController fileTypeOrderSource:(NSArray *)fileTypeOrderSource
{
    if (self.shareViewController==nil)
    {
        _shareViewController = [[SFCShareViewController alloc] init];
    }
    
    self.shareViewController.delegate = self;
    self.shareViewController.fileTypeOrderSource = fileTypeOrderSource;    
    
    PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:self.shareViewController];
    if(navigationController!=nil)
    {
        [superViewController presentViewController:navigationController animated:YES completion:^{
            
        }];
        
        [navigationController release];
    }
}


//==============================================================================
//
//==============================================================================
- (void)dismissShareViewControllerWithCompletion:(void(^)(void))completion
{
    [self.shareViewController dismissAnimated:YES completion:completion];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSelectControllerDelegate

//================================================================================
//
//================================================================================
- (void)ppSelectController:(PPSelectController *)selectController itemSelectedIndex:(NSInteger)itemIndex withString:(NSString *)itemString
{
    if ([itemString isEqualToString:WCSFC_MLS_OpenInStatement]==YES)
    {
        if (self.openInController!=nil)
        {
            self.openInController = nil;
        }
        
        //////////////////////////////////////////////////
        
        _openInController = [[UIDocumentInteractionController interactionControllerWithURL:selectController.userInfo] retain];
        self.openInController.delegate = self;
        
        UIBarButtonItem *shareBarButton = [[UIBarButtonItem alloc] initWithCustomView:self.shareViewController.ppButtonShare];
        if (shareBarButton!=nil)
        {
            //!! 必須使用presentOpenInMenuFromBarButtonItem，系統會自動判斷iPhone使用ActionSheet，iPad使用popover
            if ([self.openInController presentOpenInMenuFromBarButtonItem:shareBarButton animated:YES]==NO)
            {
                [WCToastController showMessageToastFromSuperView:self.shareViewController.view withMessage:WCSFC_MLS_OpenInFailed position:PPToastPositionBottom];
            }
            
            [shareBarButton release];
        }
    }
    else if ([itemString isEqualToString:WCSFC_MLS_ActivityStatement]==YES)
    {
        NSMutableArray *activityItems = [NSMutableArray array];
        [activityItems addObject:selectController.userInfo];
        
        //!! fix mantis 0016153: 需額外添加內容字串，才會跑Email的選單
//        [activityItems addObject:@"\n"];
        
        //////////////////////////////////////////////////
        
        if (self.activityViewController!=nil)
        {
            self.activityViewController = nil;
        }
        
        [self showActivityFromShareViewControllerWithItems:activityItems deleteFolderPath:nil];
    }
    else
    {
        return;
    }
}


//================================================================================
//
//================================================================================
- (void)ppSelectControllerDidDismiss:(PPSelectController *)selectController
{
    self.selectController = nil;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - SFCShareViewControllerDelegate

//================================================================================
// 單選故使用if..else...製作檔案格式對應的檔案放置於路徑中
//================================================================================
- (void)shareViewController:(SFCShareViewController *)shareViewController
   didSelectContactFileType:(WCFileTypeSelectView_ContactFileType)contactFileType
   fileTypeTextInfoSelected:(WCFileTypeSelectView_FileTypeTextInfo)fileTypeTextInfoSelected
{
    [self showBusyView:YES];
    
    //////////////////////////////////////////////////
    
    __block WCShareFlowController *blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        BOOL isExistFile              = YES;
        NSString *tempFolderPath      = [[WCToolController tempFolderPath] stringByAppendingPathComponent:WCShareFlowController_ContactShareRootFolder];
        
        __block NSString *deleteFolderPath = nil;
        __block NSError *fileError = nil;
        
        //////////////////////////////////////////////////
        
        do
        {
            if ([[NSFileManager defaultManager] fileExistsAtPath:tempFolderPath]==YES)
            {
                [[NSFileManager defaultManager] removeItemAtPath:tempFolderPath error:nil];
            }
            
            [[NSFileManager defaultManager] createDirectoryAtPath:tempFolderPath
                                      withIntermediateDirectories:YES
                                                       attributes:nil
                                                            error:&fileError];
            
            //////////////////////////////////////////////////
            // 判斷手機記憶體容量

            NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
            NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:nil];
            
            double freeSystemSize = [[dictionary objectForKey:NSFileSystemFreeSize] doubleValue];
            if (freeSystemSize<=WCShareFlowController_MinimumSystemMemoryRequestBytes)
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf showBusyView:NO];
                    
                    [PPAlertController showWithAlertControllerStyle:UIAlertControllerStyleAlert
                                                              title:@""
                                                            message:WCSFC_MLS_ShareDeviceStorageNotEnough
                                                   alertActionStyle:UIAlertActionStyleDefault
                                             showFromViewController:shareViewController animated:YES];
                });
                
                break;
            }
            
            //////////////////////////////////////////////////
            // 判斷檔案類型，製作對應檔案
            
            NSMutableArray *activityItems = [NSMutableArray array];
            
            // MARK: share WCXF
            if ((contactFileType&WCFileTypeSelectView_ContactFileType_WCXF)==WCFileTypeSelectView_ContactFileType_WCXF)
            {
                WCXFDataController *wcxfDataController = [[WCXFDataController alloc] initWithDelegate:blockSelf];
                if (wcxfDataController!=nil)
                {
                    NSString *writePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameWithFormatString:@"wcxf"]];
                    
                    [wcxfDataController writeFileWithPath:writePath
                                                  cardIDs:blockSelf.contactIdentifierArray
                                                    error:&fileError];
                    
                    [fileError retain];
                    
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [blockSelf showBusyView:NO];
                            
                        if (fileError==nil)
                        {
                            NSURL *fileURL = [NSURL fileURLWithPath:writePath];
                            
                            //////////////////////////////////////////////////
                            
                            if ([UIDocumentInteractionController canUseOpenInWithFileExtension:@"wcxf"]==YES)
                            {
                                if (blockSelf.selectController==nil)
                                {
                                    _selectController = [[PPSelectController alloc] init];
                                }
                                
                                //////////////////////////////////////////////////
                                
                                if(blockSelf.selectController!=nil)
                                {
                                    blockSelf.selectController.delegate = blockSelf;
                                    blockSelf.selectController.tableViewCellSelectionStyle = UITableViewCellSelectionStyleNone;
                                     
                                    //////////////////////////////////////////////////
                                    
                                    blockSelf.selectController.sourceItemStringArray = @[WCSFC_MLS_OpenInStatement, WCSFC_MLS_ActivityStatement];
                                    blockSelf.selectController.selectedIndex = -1;
                                    blockSelf.selectController.userInfo = fileURL;
                                    [blockSelf.selectController showFromViewController:blockSelf animated:YES];
                                }
                            }
                            else
                            {
                                [activityItems addObject:fileURL];
                                
                                //////////////////////////////////////////////////
                                
                                [self showActivityFromShareViewControllerWithItems:activityItems deleteFolderPath:deleteFolderPath];
                            }
                        }
                        else
                        {
                            if (fileError.code==WCXFDataController_Error_NoCardToWrite)
                            {
                                [WCToastController showMessageToastFromSuperView:self.shareViewController.view withMessage:WCSFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                            }
                            else if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:didFinishWithError:)]==YES)
                            {
                                [blockSelf.delegate shareFlowController:blockSelf didFinishWithError:fileError];
                            }
                        }
                        
                        [fileError release];
                    });

                    [wcxfDataController release];
                }
            }
            else
            {
                // MARK: share CSV
                if ((contactFileType&WCFileTypeSelectView_ContactFileType_CSV_iWorks)==WCFileTypeSelectView_ContactFileType_CSV_iWorks ||
                    (contactFileType&WCFileTypeSelectView_ContactFileType_CSV_Gmail)==WCFileTypeSelectView_ContactFileType_CSV_Gmail ||
                    (contactFileType&WCFileTypeSelectView_ContactFileType_CSV_Outlook)==WCFileTypeSelectView_ContactFileType_CSV_Outlook)
                {
                    WCCSVDataController *csvDataController = [[WCCSVDataController alloc] initWithDelegate:blockSelf];
                    if (csvDataController!=nil)
                    {
                        NSString *writePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameForFileTypeCSV:contactFileType]];
                        WCCSVDataController_Format format = WCCSVDataController_Format_None;
                        switch (contactFileType)
                        {
                            case WCFileTypeSelectView_ContactFileType_CSV_iWorks:
                            {
                                format = WCCSVDataController_Format_iWorks;
                                break;
                            }
                            case WCFileTypeSelectView_ContactFileType_CSV_Gmail:
                            {
                                format = WCCSVDataController_Format_Gmail;
                                break;
                            }
                            case WCFileTypeSelectView_ContactFileType_CSV_Outlook:
                            {
                                format = WCCSVDataController_Format_Outlook;
                                break;
                            }
                            default:
                                break;
                        }
                        
                        [csvDataController writeFileWithPath:writePath
                                                     cardIDs:blockSelf.contactIdentifierArray
                                                      format:format
                                             willSendByEmail:YES
                                                       error:&fileError];
                        if (fileError==nil)
                        {
                            NSURL *fileURL = [NSURL fileURLWithPath:writePath];
                            [activityItems addObject:fileURL];
                        }
                        else
                        {
                            if (fileError.code==WCCSVDataController_Error_NoCardToWrite)
                            {
                                fileError = PPErrorMake(WCShareFlowController_Error_NoCardToWriteSimple, nil, nil);
                            }
                        }
                        
                        [csvDataController release];
                    }
                }
                // MARK: share XLSX
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_XLSX)==WCFileTypeSelectView_ContactFileType_XLSX)
                {
                    WCXLSXDataController *xlsxDataController = [[WCXLSXDataController alloc] initWithDelegate:blockSelf];
                    if (xlsxDataController!=nil)
                    {
                        NSString *writePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameWithFormatString:@"xlsx"]];
                        
                        [xlsxDataController writeFileWithPath:writePath
                                                     cardIDs:blockSelf.contactIdentifierArray
                                                       error:&fileError];
                        if (fileError==nil)
                        {
                            NSURL *fileURL = [NSURL fileURLWithPath:writePath];
                            [activityItems addObject:fileURL];
                        }
                        else
                        {
                            switch (fileError.code)
                            {
                                case WCXLSXDataController_Error_NoCardToWrite:
                                {
                                    fileError = PPErrorMake(WCShareFlowController_Error_NoCardToWriteSimple, @"沒有可匯出的資料", nil);
                                    break;
                                }
                                case WCXLSXDataController_Error_MustHaveDelegateMethod:
                                {
                                    fileError = PPErrorMake(WCShareFlowController_Error_DelegateError, @"delegate沒有實作", nil);
                                    break;
                                }
                                default:
                                    break;
                            }
                            
                        }
                        
                        [xlsxDataController release];
                    }
                }
                // MARK: share VCF
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_VCF)==WCFileTypeSelectView_ContactFileType_VCF)
                {
                    //////////////////////////////////////////////////
                    // 建立存放VCF檔案的資料夾
                    
                    NSMutableArray *vcfFileArray = [NSMutableArray array];
                    NSString *baseStorePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, WCShareFlowController_ContactFileFolder_VCF];
                    
                    //////////////////////////////////////////////////
                    //!! 一定要先確認權限才能繼續進行操作
                    
                    [WCVCFDataController checkAccessPermissionWithCompletion:^(BOOL authorized) {
                        
                        if (authorized==YES)
                        {
                            do
                            {
                                [[NSFileManager defaultManager] createDirectoryAtPath:baseStorePath
                                                          withIntermediateDirectories:YES
                                                                           attributes:nil
                                                                                error:&fileError];
                                if (fileError!=nil)
                                {
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                // 創建VCFDataController
                                
                                WCVCFDataController *vcfDataController = [[[WCVCFDataController alloc] initWithError:&fileError] autorelease];
                                if (vcfDataController==nil)
                                {
                                    break;
                                }
                                
                                if (fileError!=nil)
                                {
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                // 依序創建每個聯絡人各自的VCF檔
                                
                                for (NSInteger index=0; index<[blockSelf.contactIdentifierArray count]; index++)
                                {
                                    NSString *cardID = [blockSelf.contactIdentifierArray objectAtIndex:index];
                                    
                                    if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:cardModelToWriteWithCardID:)]==YES)
                                    {
                                        //////////////////////////////////////////////////
                                        // 取得檔案名稱(聯絡人前後名稱)
                                        // 104/12/11移除使用聯絡人名稱對檔案命名(英文其他語系會產生亂碼)
                                        
                                        WCCardModel *cardModel = [blockSelf.delegate shareFlowController:blockSelf cardModelToWriteWithCardID:cardID];

//                                        NSString *fullName         = WCShareFlowController_ContactFileDefaultName;
//                                        NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
//                                        if ([fieldArray count]>0 && fieldArray!=nil)
//                                        {
//                                            WCFieldModel *nameField = [fieldArray objectAtIndex:0];
//                                            NSString *firstName     = [nameField valueWithSubType2:WC_FST2_Name_First];
//                                            NSString *lastName      = [nameField valueWithSubType2:WC_FST2_Name_Last];
//                                            
//                                            //////////////////////////////////////////////////
//                                            // 取得名字排序 & 完全姓名
//                                            
//                                            if (blockSelf.displayEastNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameEastOrderInShareFlowController:)]==YES))
//                                            {
//                                                blockSelf.displayEastNameOrder = [blockSelf.delegate contactNameEastOrderInShareFlowController:blockSelf];
//                                            }
//                                            
//                                            if (blockSelf.displayWestNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameWestOrderInShareFlowController:)]==YES))
//                                            {
//                                                blockSelf.displayWestNameOrder = [blockSelf.delegate contactNameWestOrderInShareFlowController:blockSelf];
//                                            }
//                                            
//                                            fullName = [WCDisplayNameController displayNameWithFirstName:firstName
//                                                                                                lastName:lastName
//                                                                                               eastOrder:blockSelf.displayEastNameOrder
//                                                                                               westOrder:blockSelf.displayWestNameOrder];
//                                        }

                                        //////////////////////////////////////////////////
                                        // 創建VCF檔(內部會協助判斷CardModel是否為空)
                                            
                                        NSString *writePath = [NSString stringWithFormat:@"%@/", baseStorePath];
                                        NSString *dateTime  = @"";
                                        
                                        NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
                                        if(dateFormatter!=nil)
                                        {
                                            [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
                                            
                                            dateTime = [dateFormatter stringFromDate:[NSDate date]];
                                            [dateFormatter release];
                                        }
                                        
                                        dateTime  = [[dateTime stringByAppendingString:@"_"] stringByAppendingString:[NSString stringWithFormat:WCCloudFileDefine_SingleContactFileNameFormat, (short)index+1, @"", @"vcf"]];
                                        writePath = [writePath stringByAppendingString:dateTime];
                                            
                                        [vcfDataController writeFileWithPath:writePath cardModel:cardModel error:&fileError];
                                        if (fileError==nil)
                                        {
                                            [vcfFileArray addObject:writePath];
                                        }
                                    }
                                }
                                
                                //////////////////////////////////////////////////
                                // 判斷是否前面所有檔案建置成功
                                
                                if ([vcfFileArray count]>0)
                                {
                                    fileError = nil;
                                    
                                    //////////////////////////////////////////////////
                                    // 判斷檔案數量是否進行壓縮
                                    
                                    if ([vcfFileArray count]>WCShareFlowController_MaxContactFileCount)
                                    {
                                        NSString *zipFile = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf zipFileCompleteNameWithFormatString:WCCloudFileDefine_ZipVcfPrefix]];
                                        
                                        [PPZipController zipWithSrcDir:baseStorePath
                                                    excludeDirPatterns:nil
                                                               dstFile:zipFile
                                                              password:nil];
                                        
                                        //////////////////////////////////////////////////
                                        
                                        NSURL *fileURL = [NSURL fileURLWithPath:zipFile];
                                        [activityItems addObject:fileURL];
                                    }
                                    else
                                    {
                                        for (NSString *writePath in vcfFileArray)
                                        {
                                            NSURL *fileURL = [NSURL fileURLWithPath:writePath];
                                            [activityItems addObject:fileURL];
                                        }
                                    }
                                    
                                    deleteFolderPath = [baseStorePath retain];
                                }
                                else
                                {
                                    fileError = PPErrorMake(WCShareFlowController_Error_NoCardToWriteSimple, nil, nil);
                                }
                                
                            } while (0);
                        }
                    }];
                    
                    // 如果是VCF，要等聯絡人權限開通後才能繼續處理
                    while ([CNContactStore authorizationStatusForEntityType:CNEntityTypeContacts]==CNAuthorizationStatusNotDetermined)
                    {
                        [NSThread sleepForTimeInterval:0.01];
                    }
                    
                    // 如果是不允許，要在這邊處理
                    if([CNContactStore authorizationStatusForEntityType:CNEntityTypeContacts]==CNAuthorizationStatusDenied)
                    {
                        fileError = PPErrorMake(WCShareFlowController_Error_NoAddressBookPermission, nil, nil);
                    }
                }
                // MARK: share Image
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_Image)==WCFileTypeSelectView_ContactFileType_Image)
                {
                    //////////////////////////////////////////////////
                    // 建立存放Image檔案的資料夾
                    
                    NSMutableArray *imageFileArray = [NSMutableArray array];
                    NSString *baseStorePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, WCShareFlowController_ContactFileFolder_Image];
                    
                    do
                    {
                        [[NSFileManager defaultManager] createDirectoryAtPath:baseStorePath
                                                  withIntermediateDirectories:YES
                                                                   attributes:nil
                                                                        error:&fileError];
                        if (fileError!=nil)
                        {
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        // 依序創建每個聯絡人各自的Image檔
                        
                        for (NSInteger index=0; index<[blockSelf.contactIdentifierArray count]; index++)
                        {
                            NSString *cardID = [blockSelf.contactIdentifierArray objectAtIndex:index];
                            
                            //////////////////////////////////////////////////
                            // 取得檔案名稱(聯絡人前後名稱)
                            // 104/12/11移除使用聯絡人名稱對檔案命名(英文其他語系會產生亂碼)
                                
//                            NSString *fullName         = WCShareFlowController_ContactFileDefaultName;
//                            NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
//                            if ([fieldArray count]>0 && fieldArray!=nil)
//                            {
//                                WCFieldModel *nameField = [fieldArray objectAtIndex:0];
//                                NSString *firstName     = [nameField valueWithSubType2:WC_FST2_Name_First];
//                                NSString *lastName      = [nameField valueWithSubType2:WC_FST2_Name_Last];
//                                
//                                //////////////////////////////////////////////////
//                                // 取得名字排序 & 完全姓名
//                                
//                                if (blockSelf.displayEastNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameEastOrderInShareFlowController:)]==YES))
//                                {
//                                    blockSelf.displayEastNameOrder = [blockSelf.delegate contactNameEastOrderInShareFlowController:blockSelf];
//                                }
//                                
//                                if (blockSelf.displayWestNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameWestOrderInShareFlowController:)]==YES))
//                                {
//                                    blockSelf.displayWestNameOrder = [blockSelf.delegate contactNameWestOrderInShareFlowController:blockSelf];
//                                }
//                                
//                                fullName = [WCDisplayNameController displayNameWithFirstName:firstName
//                                                                                    lastName:lastName
//                                                                                   eastOrder:blockSelf.displayEastNameOrder
//                                                                                   westOrder:blockSelf.displayWestNameOrder];
//                            }
                            
                            
                            //////////////////////////////////////////////////
                            // 取得影像檔案(前後名片)
                                
                            if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:imageDataToWriteWithCardID:imageType:)]==YES)
                            {
                                NSData *backSideImageData  = [blockSelf.delegate shareFlowController:blockSelf imageDataToWriteWithCardID:cardID imageType:WC_IT_BackSide];
                                if (backSideImageData!=nil)
                                {
                                    NSString *writePath = [NSString stringWithFormat:@"%@/", baseStorePath];
                                    writePath = [writePath stringByAppendingString:[NSString stringWithFormat:WCCloudFileDefine_SingleContactFileNameFormat, (short)index+1, @"_Back", @"jpg"]];
                                    
                                    [backSideImageData writeToFile:writePath atomically:YES];
                                    [imageFileArray addObject:writePath];
                                }
                                
                                NSData *frontSideImageData = [blockSelf.delegate shareFlowController:blockSelf imageDataToWriteWithCardID:cardID imageType:WC_IT_FrontSide];
                                if (frontSideImageData!=nil)
                                {
                                    NSString *writePath = [NSString stringWithFormat:@"%@/", baseStorePath];
                                    writePath = [writePath stringByAppendingString:[NSString stringWithFormat:WCCloudFileDefine_SingleContactFileNameFormat, (short)index+1, @"_Front", @"jpg"]];
                                    
                                    [frontSideImageData writeToFile:writePath atomically:YES];
                                    [imageFileArray addObject:writePath];
                                }
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        // 判斷檔案數量是否進行壓縮
                        
                        if ([imageFileArray count]>0)
                        {
                            if ([imageFileArray count]>WCShareFlowController_MaxContactFileCount)
                            {
                                NSString *zipFile = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf zipFileCompleteNameWithFormatString:WCCloudFileDefine_ZipImagePrefix]];
                                
                                [PPZipController zipWithSrcDir:baseStorePath
                                            excludeDirPatterns:nil
                                                       dstFile:zipFile
                                                      password:nil];
                                
                                //////////////////////////////////////////////////
                                
                                NSURL *fileURL = [NSURL fileURLWithPath:zipFile];
                                [activityItems addObject:fileURL];
                            }
                            else
                            {
                                for (NSString *writePath in imageFileArray)
                                {
                                    NSURL *fileURL = [NSURL fileURLWithPath:writePath];
                                    [activityItems addObject:fileURL];
                                }
                            }
                        }
                        else
                        {
                            fileError = PPErrorMake(WCShareFlowController_Error_NoCardToWriteForImage, nil, nil);
                        }
                        
                        deleteFolderPath = [baseStorePath retain];
                    } while (0);
                }
                // MARK: share Text
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_Text)==WCFileTypeSelectView_ContactFileType_Text)
                {
                    //////////////////////////////////////////////////
                    // 依序創建每個聯絡人各自的textInfo
                    
                    NSString *contactTextInfo  = @"";
                    
                    for (NSString *cardID in blockSelf.contactIdentifierArray)
                    {
                        if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:cardModelToWriteWithCardID:)]==YES)
                        {
                            WCCardModel *cardModel = [blockSelf.delegate shareFlowController:blockSelf cardModelToWriteWithCardID:cardID];
                            if (cardModel!=nil)
                            {
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_Name)==WCFileTypeSelectView_FileTypeTextInfo_Name)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldName in fieldArray)
                                        {
                                            NSString *firstName = [fieldName valueWithSubType2:WC_FST2_Name_First];
                                            NSString *lastName  = [fieldName valueWithSubType2:WC_FST2_Name_Last];
                                            
                                            //////////////////////////////////////////////////
                                            // 取得名字排序 & 完全姓名
                                            
                                            if (blockSelf.displayEastNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameEastOrderInShareFlowController:)]==YES))
                                            {
                                                blockSelf.displayEastNameOrder = [blockSelf.delegate contactNameEastOrderInShareFlowController:blockSelf];
                                            }
                                            
                                            if (blockSelf.displayWestNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameWestOrderInShareFlowController:)]==YES))
                                            {
                                                blockSelf.displayWestNameOrder = [blockSelf.delegate contactNameWestOrderInShareFlowController:blockSelf];
                                            }
                                            
                                            NSString *fullName = [WCDisplayNameController displayNameWithFirstName:firstName
                                                                                                          lastName:lastName
                                                                                                         eastOrder:blockSelf.displayEastNameOrder
                                                                                                         westOrder:blockSelf.displayWestNameOrder];
                                            if (fullName!=nil)
                                            {
                                                NSString *userName = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Name, fullName];
                                                contactTextInfo = [contactTextInfo stringByAppendingString:userName];
                                            }
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_CompanyInfo)==WCFileTypeSelectView_FileTypeTextInfo_CompanyInfo)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Company];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldCompany in fieldArray)
                                        {
                                            NSString *companyName = [fieldCompany valueWithSubType2:WC_FST2_Company_Name];
                                            if (companyName!=nil)
                                            {
                                                NSString *nameInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Company, companyName];
                                                contactTextInfo = [contactTextInfo stringByAppendingString:nameInfo];
                                            }
                                            
                                            NSString *companyDepartment = [fieldCompany valueWithSubType2:WC_FST2_Company_Department];
                                            if (companyDepartment!=nil)
                                            {
                                                NSString *departmentInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Dept, companyDepartment];
                                                contactTextInfo = [contactTextInfo stringByAppendingString:departmentInfo];
                                            }
                                            
                                            NSString *companyJobTitle = [fieldCompany valueWithSubType2:WC_FST2_Company_JobTitle];
                                            if (companyJobTitle!=nil)
                                            {
                                                NSString *jobTitleInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_JobTitle, companyJobTitle];
                                                contactTextInfo = [contactTextInfo stringByAppendingString:jobTitleInfo];
                                            }
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_Phone)==WCFileTypeSelectView_FileTypeTextInfo_Phone)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Phone];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldPhone in fieldArray)
                                        {
                                            NSString *phoneInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Phone, fieldPhone.value];
                                            contactTextInfo = [contactTextInfo stringByAppendingString:phoneInfo];
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_Address)==WCFileTypeSelectView_FileTypeTextInfo_Address)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Address];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldAddress in fieldArray)
                                        {
                                            NSInteger bcrLanguage = [cardModel recognitionlanguageWithFieldSource:fieldAddress.source];
                                            NSString *addressDisplay = [fieldAddress stringDisplayAddressWithBCRLanguage:bcrLanguage];
                                            if (addressDisplay!=nil)
                                            {
                                                NSString *addressInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Address, addressDisplay];
                                                contactTextInfo = [contactTextInfo stringByAppendingString:addressInfo];
                                            }
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_Email)==WCFileTypeSelectView_FileTypeTextInfo_Email)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Email];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldEmail in fieldArray)
                                        {
                                            NSString *emailInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_Email, fieldEmail.value];
                                            contactTextInfo = [contactTextInfo stringByAppendingString:emailInfo];
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_URL)==WCFileTypeSelectView_FileTypeTextInfo_URL)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_URL];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldURL in fieldArray)
                                        {
                                            NSString *urlInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_WebSite, fieldURL.value];
                                            contactTextInfo = [contactTextInfo stringByAppendingString:urlInfo];
                                        }
                                    }
                                }
                                if ((fileTypeTextInfoSelected&WCFileTypeSelectView_FileTypeTextInfo_IM)==WCFileTypeSelectView_FileTypeTextInfo_IM)
                                {
                                    NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_InstantMessage];
                                    if ([fieldArray count]>0)
                                    {
                                        for (WCFieldModel *fieldIM in fieldArray)
                                        {
                                            NSString *instanceMessageInfo = [NSString stringWithFormat:@"%@: %@\n", WCSFC_MLS_InstantMessenger, fieldIM.value];
                                            contactTextInfo = [contactTextInfo stringByAppendingString:instanceMessageInfo];
                                        }
                                    }
                                }
                            }
                            else
                            {
                                fileError = PPErrorMake(WCShareFlowController_Error_NoCardToWriteSimple, nil, nil);
                            }
                        }
                        
//                        contactTextInfo = [contactTextInfo stringByAppendingString:@"===================================\n"];
                    }
                    
                    //////////////////////////////////////////////////
                    // 完成所有聯絡人字串串接 & 無創建檔案
                    
                    if (fileError==nil)
                    {
                        if ([contactTextInfo isEqualToString:@""]==NO)
                        {
                            [activityItems addObject:contactTextInfo];
                        }
                        else
                        {
                            fileError = PPErrorMake(WCShareFlowController_Error_NoContactInfoToWriteForText, nil, nil);
                        }
                    }
                    
                    isExistFile = NO;
                }
                else
                {
                    NSLog(@"Error ContactFileType");
                    break;
                }
                
                //////////////////////////////////////////////////
                // 進行文件分享
                
                [fileError retain];
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf showBusyView:NO];
                    
                    if (fileError==nil)
                    {
                        [self showActivityFromShareViewControllerWithItems:activityItems deleteFolderPath:deleteFolderPath];
                    }
                    else
                    {
                        if (fileError.code==WCShareFlowController_Error_NoCardToWriteSimple)
                        {
                            [WCToastController showMessageToastFromSuperView:self.shareViewController.view withMessage:WCSFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                        }
                        else if (fileError.code==WCShareFlowController_Error_NoContactInfoToWriteForText)
                        {
                            [WCToastController showMessageToastFromSuperView:blockSelf.shareViewController.view withMessage:WCSFC_MLS_ShareTextError position:PPToastPositionBottom];
                        }
                        else if (fileError.code==WCShareFlowController_Error_NoCardToWriteForImage)
                        {
                            [WCToastController showMessageToastFromSuperView:blockSelf.shareViewController.view withMessage:WCSFC_MLS_ShareImageError position:PPToastPositionBottom];
                        }
                        else if (fileError.code==WCShareFlowController_Error_NoAddressBookPermission)
                        {
                            [PPAlertController showWithAlertControllerStyle:UIAlertControllerStyleAlert
                                                                      title:@""
                                                                    message:WCSFC_MLS_NoAddressBookPermission
                                                           alertActionStyle:UIAlertActionStyleDefault
                                                     showFromViewController:shareViewController animated:YES];
                        }
                        else if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(shareFlowController:didFinishWithError:)]==YES)
                        {
                            [blockSelf.delegate shareFlowController:blockSelf didFinishWithError:fileError];
                        }
                    }
                    
                    [fileError release];
                });
            }
            
        } while (0);
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCSVDataControllerDelegate

//================================================================================
//
//================================================================================
- (WCCardModel *)csvDataContrller:(WCCSVDataController *)csvDataController
       cardModelToWriteWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate shareFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//================================================================================
//
//================================================================================
- (NSString *)csvDataContrller:(WCCSVDataController *)csvDataController
   groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate shareFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}


//================================================================================
//
//================================================================================
- (BOOL)csvDataContrller:(WCCSVDataController *)csvDataController shouldWriteCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    return YES;
}


//================================================================================
//
//================================================================================
- (void)csvDataContrller:(WCCSVDataController *)csvDataController
   didWriteCardWithIndex:(NSInteger)index
              totalCount:(NSInteger)totalCount
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate shareFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCXFDataControllerDelegate


//==============================================================================
//
//==============================================================================
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
 shouldWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    return YES;
}


//================================================================================
//
//================================================================================
- (WCCardModel *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
        cardModelToWriteWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate shareFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//================================================================================
//
//================================================================================
- (NSString *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate shareFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}


//================================================================================
//
//================================================================================
- (NSData *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
   imageDataToWriteWithCardID:(NSString *)cardID
                    imageType:(WC_ImageType)imageType
{
    NSData *imageData = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:imageDataToWriteWithCardID:imageType:)]==YES)
    {
        imageData = [self.delegate shareFlowController:self imageDataToWriteWithCardID:cardID imageType:imageType];
    }
    
    return imageData;
}


//================================================================================
//
//================================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate shareFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCWLSXDataControllerDelegate


//==============================================================================
//
//==============================================================================
- (WCCardModel *)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
        cardModelToWriteWithCardID:(NSString *)cardID;
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate shareFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//==============================================================================
//
//==============================================================================
- (NSString *)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
    groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate shareFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}


//================================================================================
//
//================================================================================
- (BOOL)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller shouldWriteCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    return YES;
}


//==============================================================================
//
//==============================================================================
- (void)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(shareFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate shareFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}


@end
