//
//  WCSearchViewController.m
//  
//
//  Created by sanhue on 2015/7/9.
//  Copyright (c) 2015年 penpower. All rights reserved.
//

#import "WCSearchViewController.h"

// Define
#import "WCAppearanceDefine.h"
#import "WCSearchViewController+ResourceDefine.h"
#import "UIColor+HexString.h"
#import "WCDataControllerProtocol.h"

// Categories
#import "NSString+Additions.h"
#import "PPButton+Factory.h"
//#import "UISearchBar+Additions.h"
#import "NSObject+PPBusyView.h"
#import "NSTimer+Additions.h"

// Views
#import "PPNavigationBarView.h"
#import "PPSearchBar.h"

// Model
#import "WCCardModel.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSInteger, WCSearchViewControllerButtonTag)
{
    WCSearchViewControllerButtonTag_Cancel = 0,
    WCSearchViewControllerButtonTag_AdvanceSearch,
    WCSearchViewControllerButtonTag_MultiSelect,
};

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface WCSearchViewController ()
<
PPButtonDelegate,
UISearchBarDelegate
>
@property (nonatomic, retain, readwrite) PPNavigationBarView *ppNavigationBarView;
@property (nonatomic, retain, readwrite) PPButton *ppButtonAdvanceSearch;
@property (nonatomic, retain, readwrite) PPButton *ppButtonMultiSelect;
@property (nonatomic, retain, readwrite) PPButton *ppButtonCancel;
@property (nonatomic, retain, readwrite) PPSearchBar *searchBar;
@property (nonatomic, retain) NSString *lastSearchText;
@property (nonatomic, retain) NSTimer *timerForLaunchedSearch;


// 是否要focus在search bar上，預設YES
@property (nonatomic, assign) BOOL focusOnSearchBar;

@property (nonatomic, retain) NSOperationQueue *operationQueue;
@property (nonatomic, retain) NSString *selectedCardID;
@property (nonatomic, retain) NSString *searchingGroupName;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCSearchViewController


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - init/dealloc methods

//==============================================================================
//
//==============================================================================
+ (void)initialize
{
}


//==============================================================================
//
//==============================================================================
- (instancetype)initWithMode:(PPSIC_Mode)mode
{
    self = [super initWithMode:mode];
    if (self)
    {
        self.focusOnSearchBar = YES;
        self.operationQueue = [[[NSOperationQueue alloc] init] autorelease];
        if (self.operationQueue)
        {
            [self.operationQueue setMaxConcurrentOperationCount:1];
        }

        NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
        [nc addObserver:self selector:@selector(recvDisplayDataChanged:) name:WCDC_NOTIFY_DisplayDataChanged object:nil];
        [nc addObserver:self selector:@selector(recvDisplayDataChanged:) name:WCDC_NOTIFY_FavoriteChanged object:nil];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self.timerForLaunchedSearch invalidate];
    self.timerForLaunchedSearch = nil;
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:WCDC_NOTIFY_DisplayDataChanged object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:WCDC_NOTIFY_FavoriteChanged object:nil];

    self.delegate = nil;
        
    [self removeMainUI];
    
    [self.operationQueue cancelAllOperations];
    
    self.operationQueue = nil;
    self.lastSearchText = nil;
    self.selectedCardID = nil;
    self.searchingGroupName = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - view controller life cycle

//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    // 取得目前要搜尋的類別名稱
    if ([self.delegate respondsToSelector:@selector(groupNameForSearchViewController:)])
    {
        self.searchingGroupName = [self.delegate groupNameForSearchViewController:self];
    }
    
    [self prepareMainUI];
    
    // 發出一個空的active訊息，用來清空detail，要在 becomeFirstResponder 之前做，不然可能會有當機問題
    [self adjustCurrentSelectedIndexPath];
    [self activeCurrentSelectCell];
    
    //////////////////////////////////////////////////

    BOOL needReloadData = YES;
    
    if([self.delegate respondsToSelector:@selector(reloadTableViewDataWhenViewWillAppear)]==YES)
    {
        needReloadData = [self.delegate reloadTableViewDataWhenViewWillAppear];
    }
    
    
    if (needReloadData==YES &&
        [self.lastSearchText length]>0)
    {
        [self searchText:self.lastSearchText];
    }
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    // 設定focus
    if(self.focusOnSearchBar)
    {
        [self.searchBar becomeFirstResponder];
    }
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [self.searchBar resignFirstResponder];
    
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [self removeMainUI];
    
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidLayoutSubviews
{
    [super viewDidLayoutSubviews];
    
    //////////////////////////////////////////////////
    CGRect searchBarFrame = [self searchBarFrame];
    [self.searchBar setInternalFrame:searchBarFrame];
    [self.searchBar setFrame:searchBarFrame];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - view controller orientation 


//==============================================================================
//
//==============================================================================
- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    [super willRotateToInterfaceOrientation:toInterfaceOrientation duration:duration];
    
    //////////////////////////////////////////////////
    
    [self.searchBar resignFirstResponder];
}



//==============================================================================
//
//==============================================================================
- (void)willAnimateRotationToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    [super willAnimateRotationToInterfaceOrientation:toInterfaceOrientation duration:duration];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - prepare ui


//==============================================================================
//
//==============================================================================
- (void)prepareMainUI
{
    //////////////////////////////////////////////////
    if(self.ppButtonCancel==nil)
    {
        self.ppButtonCancel = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                              tag:WCSearchViewControllerButtonTag_Cancel
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
    }
    //////////////////////////////////////////////////
    if(self.ppButtonMultiSelect==nil)
    {
        self.ppButtonMultiSelect = [PPButton ppButtonWithIconImageName:ImageNameSearchViewControllerButton_MultiSelect
                                                                   tag:WCSearchViewControllerButtonTag_MultiSelect
                                                              delegate:self
                                                 normalBackgroundColor:nil
                                            highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                       imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
    }
    //////////////////////////////////////////////////
    if(self.showAdvanceSearch)
    {
        if(self.ppButtonAdvanceSearch==nil)
        {
            self.ppButtonAdvanceSearch = [PPButton ppButtonWithIconImageName:ImageNameSearchViewControllerButton_AdvanceSearch
                                                                         tag:WCSearchViewControllerButtonTag_AdvanceSearch
                                                                    delegate:self
                                                       normalBackgroundColor:SearchViewController_AdvanceSearchButtonColor_Normal
                                                  highlightedBackgroundColor:SearchViewController_AdvanceSearchButtonColor_Highlighted
                                                             imageEdgeInsets:SearchViewController_AdvanceSearchButtonImageEdgeInsets];
        }
        self.ppButtonAdvanceSearch.frame = SearchViewController_AdvanceSearchButtonBounds;
    }
    //////////////////////////////////////////////////
    
    CGRect searchBarFrame = [self searchBarFrame];
    if(self.searchBar==nil)
    {
        self.searchBar = [[[PPSearchBar alloc] initWithFrame:searchBarFrame] autorelease];
    }
    
    if (self.searchBar)
    {
        [self.searchBar setDelegate:self];
        [self.searchBar setSearchBarStyle:UISearchBarStyleDefault];
        if([self.searchBar respondsToSelector:@selector(setReturnKeyType:)])
        {
            [self.searchBar setReturnKeyType:UIReturnKeyDone];
        }
        
        //設定背景
        UIImage *searchFieldBGImage = [UIImage imageWithColor:[UIColor whiteColor] size:CGSizeMake(searchBarFrame.size.width, 30)];
        [self.searchBar setSearchFieldBackgroundImage:searchFieldBGImage forState:UIControlStateNormal];
        
        //////////////////////////////////////////////////
        // 設定search bar icon
        UIImage *searchIcon = [UIImage imageWithName:ImageNameSearchViewControllerButton_SearchIcon];
        [self.searchBar setImage:searchIcon forSearchBarIcon:UISearchBarIconSearch state:UIControlStateNormal];
        
        UIImage *clearIcon = [UIImage imageWithName:ImageNameSearchViewControllerButton_SearchClear_Normal];
        [self.searchBar setImage:clearIcon forSearchBarIcon:UISearchBarIconClear state:UIControlStateNormal];
 
        //////////////////////////////////////////////////
        
        [self.searchBar setBackgroundColor:[UIColor whiteColor]];

        if(@available(iOS 13, *))
        {
            // 設定 游標顏色
            [[self.searchBar searchTextField] setTintColor:SearchViewController_SearchFieldTextColor];
            
            // 設定place holder
            if ([self.searchingGroupName length]>0)
            {
                NSString *placeholder = [NSString stringWithFormat:SearchViewController_PlaceHolderFormat, self.searchingGroupName];
                [[self.searchBar searchTextField] setPlaceholder:placeholder];
                
                NSAttributedString *attributedPlaceholder =
                [[NSAttributedString alloc] initWithString:placeholder
                                                attributes:@{NSStrokeColorAttributeName:SearchViewController_SearchFieldTextColor}];
                [[self.searchBar searchTextField] setAttributedPlaceholder:attributedPlaceholder];
                [attributedPlaceholder release];
            }
            
            // 設定text field字型
            [[self.searchBar searchTextField] setTextColor:SearchViewController_SearchFieldTextColor];
            [[self.searchBar searchTextField] setFont:[UIFont systemFontOfSize:18]];
        }
        if ([self.lastSearchText length]>0)
        {
            [self.searchBar setText:self.lastSearchText];
        }
    }

    //////////////////////////////////////////////////
    
    if(_ppNavigationBarView==nil)
    {
        _ppNavigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
    }
    
    if (self.ppNavigationBarView)
    {
        [self.ppNavigationBarView setStyle:PPBarViewStyle_CenteredAbsolute];
        [self.ppNavigationBarView.titleLabel setEnabled:NO];
        [self.ppNavigationBarView.titleLabel setHidden:YES];
        [self.ppNavigationBarView setIgnoreEdgeInset:NO];
        //////////////////////////////////////////////////
        NSMutableArray *rightButtons = [NSMutableArray array];
        
        if (self.ppButtonMultiSelect)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.ppButtonMultiSelect];
            itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
            itemModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            

            [rightButtons addObject:itemModel];
        }
        
        if([rightButtons count]>0)
        {
            PPBarViewBlockModel *blockModel = [[PPBarViewBlockModel alloc] init];
            blockModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
            blockModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            [blockModel.itemModels addObjectsFromArray:rightButtons];
            
            [self.ppNavigationBarView setBlockModel:blockModel forType:PPBarViewBlockType_Right];
            [blockModel release];
        }

        //////////////////////////////////////////////////
        NSMutableArray *centerViews = [NSMutableArray array];
        
        if (self.searchBar)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.searchBar];
            itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
            itemModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            
            [centerViews addObject:itemModel];
        }

        if (self.ppButtonAdvanceSearch)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.ppButtonAdvanceSearch];
            itemModel.edgeInsetsForNormalBar = UIEdgeInsetsMake(0, 0, 0, 5);
            itemModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            [centerViews addObject:itemModel];
        }

        if([centerViews count]>0)
        {
            PPBarViewBlockModel *blockModel = [[PPBarViewBlockModel alloc] init];
            blockModel.edgeInsetsForNormalBar = SearchViewController_SearchAreaEdgeInsets;
            blockModel.edgeInsetsForMiniBar = SearchViewController_SearchAreaEdgeInsets;
            [blockModel.itemModels addObjectsFromArray:centerViews];
            
            [self.ppNavigationBarView setBlockModel:blockModel forType:PPBarViewBlockType_Center];
            [blockModel release];
//            [self.ppNavigationBarView setViews:centerViews forBlockType:PPBarViewBlockType_Center];
        }


        //////////////////////////////////////////////////
        if (self.ppButtonCancel)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.ppButtonCancel];
            itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
            itemModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            
            PPBarViewBlockModel *blockModel = [[PPBarViewBlockModel alloc] init];
            blockModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
            blockModel.edgeInsetsForMiniBar = UIEdgeInsetsZero;
            [blockModel.itemModels addObject:itemModel];
            
            [self.ppNavigationBarView setBlockModel:blockModel forType:PPBarViewBlockType_Left];
            [blockModel release];
        }
        

        //////////////////////////////////////////////////
        if (self.navigationController!=nil)
        {
            self.navigationItem.leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.ppNavigationBarView] autorelease];
        }

    }

}


//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    self.navigationItem.leftBarButtonItem = nil;
    
    self.searchBar = nil;
    self.ppButtonAdvanceSearch = nil;
    self.ppButtonMultiSelect = nil;
    self.ppButtonCancel = nil;
    
    self.ppNavigationBarView = nil;
}


//==============================================================================
// 需在ui都建立後才能使用
//==============================================================================
- (CGRect)searchBarFrame
{
    NSInteger searchBarLeftMargin = self.ppButtonCancel.bounds.size.width+SearchViewController_SearchAreaEdgeInsets.left;
    NSInteger searchBarRightMargin = self.ppButtonMultiSelect.bounds.size.width+SearchViewController_SearchAreaEdgeInsets.right;

    if (self.showAdvanceSearch)
    {
        searchBarRightMargin += self.ppButtonAdvanceSearch.bounds.size.width;
    }
    
    CGRect viewFrame = self.navigationController.navigationBar.bounds;
    CGFloat searchBarHeight = viewFrame.size.height-(SearchViewController_SearchAreaEdgeInsets.top+SearchViewController_SearchAreaEdgeInsets.bottom);
    
    return CGRectMake(searchBarLeftMargin,
                      (viewFrame.size.height-searchBarHeight)/2,
                      viewFrame.size.width-(searchBarLeftMargin+searchBarRightMargin),
                      searchBarHeight);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private override busy control


//==============================================================================
//
//==============================================================================
- (void)setBusy:(NSNumber *)busy
{
    [super setBusy:busy];
    //////////////////////////////////////////////////
    
    if ([busy boolValue])
    {
        [self.busyView setStyle:PPBusyViewStyle_Border];
        self.busyView.centerY = (CGRectGetMaxY(self.tableView.frame)+CGRectGetMaxY(self.navigationController.navigationBar.frame))/2;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Search Text Method


//==============================================================================
//
//==============================================================================
- (void)searchText:(NSString*)text
{
    self.focusOnSearchBar = YES;

    __block BOOL isSearching = NO;
    __block typeof(self) blockSelf = self;
    [self.operationQueue cancelAllOperations];
    [self setBusy:@(NO)];

    
    __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{

        if ([blockSelf.delegate respondsToSelector:@selector(copyCardsWithSearchText:)])
        {
            NSMutableArray *searchedResult = nil;
            
            if ([blockOperation isCancelled]==NO)
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    isSearching = YES;
                    [blockSelf setBusy:@(YES)];
                });
                
                if ([text length]>0)
                {
                    searchedResult = [[blockSelf.delegate copyCardsWithSearchText:text] autorelease];
                }
                else
                {
                    blockSelf.selectedCardID = nil;
                }
            }
            
            //////////////////////////////////////////////////
            // !!這邊要用同步的方式，不然非同步更新會造成當機

            if ([blockOperation isCancelled]==NO)
            {
                dispatch_sync(dispatch_get_main_queue(), ^{

                    @synchronized(blockSelf)
                    {
                        if ([blockOperation isCancelled]==NO)
                        {
                            blockSelf.allCardSectionArray = searchedResult;
                        }
                        
                        if ([blockOperation isCancelled]==NO)
                        {
                            [blockSelf.tableView reloadData];
                        }
                    }

                    //////////////////////////////////////////////////
                    if ([blockOperation isCancelled]==NO)
                    {
                        [blockSelf adjustCurrentSelectedIndexPath];
                        [blockSelf selectCurrentCard];
                        [blockSelf activeCurrentSelectCell];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if (isSearching)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [blockSelf setBusy:@(NO)];
                            isSearching = NO;
                        });
                    }
                });
            }
            else
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf setBusy:@(NO)];
                    isSearching = NO;
                });
            }
        }
    }];
    
    [self.operationQueue addOperation:blockOperation];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public


//==============================================================================
//
//==============================================================================
- (void)searchWithOptionModels:(NSArray *)optionModels
{
    // !! 清掉最後搜尋的字串
    self.lastSearchText = nil;
    
    // 進階搜尋過不用focus在search bar上
    self.focusOnSearchBar = NO;
    
    __block BOOL isSearching = NO;
    __block typeof(self) blockSelf = self;
    [self.operationQueue cancelAllOperations];
    [self setBusy:@(NO)];
        
    __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{
        
        if ([blockSelf.delegate respondsToSelector:@selector(copyCardsWithAdvanceSearchOptionModels:)])
        {
            NSMutableArray *searchedResult = nil;
            
            if ([blockOperation isCancelled]==NO)
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    isSearching = YES;
                    [blockSelf setBusy:@(YES)];
                });

                searchedResult = [[blockSelf.delegate copyCardsWithAdvanceSearchOptionModels:optionModels] autorelease];
            }
            
            //////////////////////////////////////////////////
            // !!這邊要用同步的方式，不然非同步更新會造成當機
            
            if ([blockOperation isCancelled]==NO)
            {
                dispatch_sync(dispatch_get_main_queue(), ^{
                    
                    @synchronized(blockSelf)
                    {
                        if ([blockOperation isCancelled]==NO)
                        {
                            blockSelf.allCardSectionArray = searchedResult;
                        }
                        
                        if ([blockOperation isCancelled]==NO)
                        {
                            [blockSelf.tableView reloadData];
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    if ([blockOperation isCancelled]==NO)
                    {
                        [blockSelf adjustCurrentSelectedIndexPath];
                        [blockSelf selectCurrentCard];
                        [blockSelf activeCurrentSelectCell];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if (isSearching)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [blockSelf setBusy:@(NO)];
                            isSearching = NO;
                        });
                    }
                });
            }
            else
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf setBusy:@(NO)];
                    isSearching = NO;
                });
            }
        }
    }];
    
    [self.operationQueue addOperation:blockOperation];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - (private) focus control

//==============================================================================
//
//==============================================================================
- (NSString *)defualtSelectedCardID
{
    NSString *defaultCardID = nil;
    
    for (WCCardSectionModel *sectionModel in self.allCardSectionArray)
    {
        for (WCCardModel *cardModel in sectionModel.cardArray)
        {
            defaultCardID = cardModel.ID;
            if ([defaultCardID length]>0)
            {
                return defaultCardID;
            }
        }
    }
    
    return defaultCardID;
}


//===============================================================================
//
//===============================================================================
- (NSIndexPath *)indexPathWithCardID:(NSString *)cardID
{
    if(cardID == nil)
        return nil;
    
    NSInteger section = -1;
    NSInteger row = -1;
    BOOL isFind = NO;
    
    for (WCCardSectionModel *sectionModel in self.allCardSectionArray)
    {
        section ++;
        for (WCCardModel *cardModel in sectionModel.cardArray)
        {
            row ++;
            if ([cardID isEqualToString:cardModel.ID])
            {
                isFind = YES;
                break;
            }
        }
        
        // 如果有找到的跳出去，如果沒有，重設row之後，找下一個section
        if (isFind)
        {
            break;
        }
        else
        {
            row = -1;
        }
    }
    
    if (section!=-1 && row!=-1)
    {
        return [NSIndexPath indexPathForRow:row inSection:section];
    }
    
    return nil;
}


//==============================================================================
//
//==============================================================================
- (NSIndexPath *)defaultSelectedIndexPath
{
    NSInteger sectionIndex = 0;
    for (WCCardSectionModel *cardSectionModel in self.allCardSectionArray)
    {
        NSInteger rowIndex = 0;
        for (WCCardModel *cardModel in cardSectionModel.cardArray)
        {
            // !! ipad未校正也可以focus
            BOOL isPad = (UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPad);
            if (isPad ||
                !((cardModel.tagMask&WC_TagMask_Unverified)==WC_TagMask_Unverified))
            {
                return [NSIndexPath indexPathForRow:rowIndex inSection:sectionIndex];
            }
            rowIndex++;
        }
        
        sectionIndex++;
    }
    
    return nil;
}


//==============================================================================
//
//==============================================================================
- (void)adjustCurrentSelectedIndexPath
{
    // 若紀錄點擊 indexPath 為未校正名片，則直接清空
    if(self.currentSelectedIndexPath!=nil)
    {
        NSInteger sectionIndex = [self.currentSelectedIndexPath section];
        NSInteger rowIndex = [self.currentSelectedIndexPath row];
        
        // !!  如果之前選取的已經不存在，也要重設
        if (sectionIndex>=[self.allCardSectionArray count])
        {
            self.currentSelectedIndexPath = nil;
            self.selectedCardID = nil;
        }
        else
        {
            WCCardSectionModel *cardSectionModel = [self.allCardSectionArray objectAtIndex:sectionIndex];
            // !!  如果之前選取的row不存在，重設為section的最後一個item
            if (rowIndex>=[[cardSectionModel cardArray] count])
            {
                NSInteger newRowIndex = [[cardSectionModel cardArray] count]-1;
                if (newRowIndex>=0)
                {
                    self.currentSelectedIndexPath = [NSIndexPath indexPathForRow:newRowIndex inSection:sectionIndex];
                    WCCardModel *cardModel = [[cardSectionModel cardArray] lastObject];
                    self.selectedCardID = cardModel.ID;
                }
                else
                {
                    self.currentSelectedIndexPath = nil;
                    self.selectedCardID = nil;
                }
            }
        }
    }
    
    //////////////////////////////////////////////////
    // 如果目前沒有選取且有名片，自動選取第一個
    if(![self.allCardSectionArray count]) // 如果沒有名片，設為沒有選取
    {
        self.currentSelectedIndexPath = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Button action

//==============================================================================
//
//==============================================================================
- (void)onClickMultiSelect
{
    if ([self.delegate respondsToSelector:@selector(didClickMultiSelectWithSearchViewController:)])
    {
        [self.delegate didClickMultiSelectWithSearchViewController:self];
    }
}

//==============================================================================
//
//==============================================================================
- (void)onClickAdvanceSearch
{
    if ([self.delegate respondsToSelector:@selector(didClickAdvanceSearchWithSearchViewController:)])
    {
        [self.delegate didClickAdvanceSearchWithSearchViewController:self];
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate

//==============================================================================
//
//==============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    // 全部都dispatch出去再處理
    __block typeof(self) blockSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        switch (ppButton.tag)
        {
            case WCSearchViewControllerButtonTag_MultiSelect:
            {
                [blockSelf onClickMultiSelect];
                break;
            }
            case WCSearchViewControllerButtonTag_AdvanceSearch:
            {
                [blockSelf onClickAdvanceSearch];
                break;
            }
            case WCSearchViewControllerButtonTag_Cancel:
            {
                [blockSelf goBackAnimated:YES];
                break;
            }
            default:
                break;
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UISearchBarDelegate


//==============================================================================
//
//==============================================================================
- (void)searchBarTextDidBeginEditing:(UISearchBar *)searchBar
{
    self.focusOnSearchBar = YES;
}


//==============================================================================
//
//==============================================================================
- (void)searchBarTextDidEndEditing:(UISearchBar *)searchBar
{
    
}


//==============================================================================
//
//==============================================================================
- (void)searchBar:(UISearchBar *)searchBar textDidChange:(NSString *)searchText
{
    
//    NSLog(@"%s - %@", __PRETTY_FUNCTION__, searchText);
    // 改為不過濾前後空白字元
//    if ([searchText length]>0)
//    {
//        // 去掉空白字元
//        searchText = [searchText stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceAndNewlineCharacterSet]];
//    }
    
    self.lastSearchText = searchText;
    
    self.currentSelectedIndexPath = nil;
    
    [self.timerForLaunchedSearch invalidate];
    NSTimeInterval timeInterval = 0.0;
    
    if ([searchText length]==0)
    {
        timeInterval = 0.0;
    }
    else if ([searchText length]==1)
    {
        timeInterval = 1.0;
    }
    else
    {
        timeInterval = 0.5;
    }
    
    self.timerForLaunchedSearch = [NSTimer pp_scheduledTimerWithTimeInterval:timeInterval repeats:NO block:^(NSTimer * _Nonnull timer) {
        
        [self searchText:searchText];
    }];

}


//==============================================================================
//
//==============================================================================
- (void)searchBarSearchButtonClicked:(UISearchBar *)searchBar
{  
    [searchBar resignFirstResponder];
    
    //////////////////////////////////////////////////

    if([self.delegate respondsToSelector:@selector(searchViewController:didEndSearchWithKeyword:)]==YES)
    {
        // 改為不過濾前後空白字元
        NSString *searchText = self.searchBar.text;//[self.searchBar.text stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceAndNewlineCharacterSet]];
        
        [self.delegate searchViewController:self
                    didEndSearchWithKeyword:searchText];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableView Delegate Methods

//===============================================================================
//
//===============================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (indexPath!=nil)
    {
        NSUInteger          section = [indexPath section];
        NSUInteger          row = [indexPath row];
        NSMutableArray      *cardSectionArray = self.allCardSectionArray;
        
        if (section<[cardSectionArray count])
        {
            WCCardSectionModel  *cardSectionModel = [cardSectionArray objectAtIndex:section];
            
            if (row<[cardSectionModel.cardArray count])
            {
                WCCardModel *cardModel = [cardSectionModel.cardArray objectAtIndex:row];
                
                self.selectedCardID = cardModel.ID;
                
                [self.searchBar resignFirstResponder];
                
            }
            else
            {
                self.selectedCardID = nil;
            }
        }
        else
        {
            self.selectedCardID = nil;
        }
    }
    else
    {
        self.selectedCardID = nil;
    }
    
    // 紀錄目前選取的indexPath
    self.currentSelectedIndexPath = [self indexPathWithCardID:self.selectedCardID];
        
    __block typeof(self) blockSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        if ([blockSelf.delegate respondsToSelector:@selector(searchViewController:didSelectCardID:)])
        {
            [blockSelf.delegate searchViewController:blockSelf didSelectCardID:self.selectedCardID];
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCTableViewControllerProtocol

//==============================================================================
//
//==============================================================================
- (UIImage *)cachedThumbnailImageWithCardID:(NSString *)cardID
{
    if ([self.delegate respondsToSelector:@selector(searchViewController:cachedThumbnailImageWithCardID:)])
    {
        return [self.delegate searchViewController:self cachedThumbnailImageWithCardID:cardID];
    }
    
    return nil;}


//==============================================================================
//
//==============================================================================
- (UIImage *)thumbnailImageWithCardID:(NSString *)cardID
{
    if ([self.delegate respondsToSelector:@selector(searchViewController:thumbnailImageWithCardID:)])
    {
        return [self.delegate searchViewController:self thumbnailImageWithCardID:cardID];
    }
    
    return nil;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notification receiver


//===============================================================================
//
//===============================================================================
- (void)recvDisplayDataChanged:(NSNotification *)notification
{
    BOOL needUpdate = NO;
    NSDictionary *userInfo = [notification userInfo];
    
    if([userInfo[WCDC_NOTIFY_UserInfo_kRemoveCard] count]>0)
    {
        // 如果刪除的名片在目前的結果中，才要更新
        for(NSString *removedCardID in userInfo[WCDC_NOTIFY_UserInfo_kRemoveCard])
        {
            if([self indexPathWithCardID:removedCardID]!=nil)
            {
                needUpdate = YES;
                break;
            }
        }
    }
    
    if([userInfo[WCDC_NOTIFY_UserInfo_kRemoveFavorite] count]>0)
    {
        // 如果刪除的名片在目前的結果中，才要更新
        for(NSString *removedCardID in userInfo[WCDC_NOTIFY_UserInfo_kRemoveFavorite])
        {
            if([self indexPathWithCardID:removedCardID]!=nil)
            {
                needUpdate = YES;
                break;
            }
        }
    }

    
    if(needUpdate==YES)
    {
        if ([self.lastSearchText length]>0)
        {
            [self searchText:self.lastSearchText];
        }
    }
}
@end
