//
//  CardRecognitionController.m
//
//
//  Created by Howard.Lin on 13/12/4.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "CardRecognitionController.h"

// controller
#import "PPLogController.h"

// view

// model

// other
#import "namecard_function.h"
#import "RecognitionOperation.h"

// extension
#import "UIImage+Additions.h"
#import "PHAsset+Image.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface CardRecognitionController()

@property (nonatomic,retain)   NSError                    *lastRecognitionError;
@property (nonatomic,retain)   NSMutableDictionary        *recognitionCountDictionary;
@property (nonatomic,readonly) NSMutableArray             *normalRecogSourceArray;
@property (nonatomic,readonly) NSMutableArray             *silentRecogSourceArray;
@property (nonatomic,readonly) NSOperationQueue           *recognitionQueue;
@property (nonatomic,retain)   WCCardModel                *lastRecognitionCardModel;
@property (nonatomic,assign)   RecognitionOperationMode   lastRecognitionOperationMode;
@property (nonatomic,retain)   WCRecogSourceModel         *lastWCRecogSourceModel;
@property (nonatomic,retain)   UIImage                    *lastRecognitionImage;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation CardRecognitionController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
 delegate                     = delegate_,
 lastRecognitionError         = lastRecognitionError_,
 recognitionCountDictionary   = recognitionCountDictionary_,
 normalRecogSourceArray       = normalRecogSourceArray_,
 silentRecogSourceArray       = silentRecogSourceArray_,
 recognitionQueue             = recognitionQueue_,
 totalSource                  = totalSource_,
 lastRecognitionCardModel     = lastRecognitionCardModel_,
 lastRecognitionOperationMode = lastRecognitionOperationModel_,
 lastWCRecogSourceModel       = lastWCRecogSourceModel_,
 lastRecognitionImage         = lastRecognitionImage_;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        // init other variables
        recognitionQueue_  = [[NSOperationQueue alloc] init];
        
        [recognitionQueue_ setMaxConcurrentOperationCount:1];
        
        //////////////////////////////////////////////////
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(receiveRecognitionOperationEndNotification:)
                                                     name:WCRO_NOTIFY_OperationEnd
                                                   object:nil];
        
        //////////////////////////////////////////////////

        // load BCR kernel
        Connect();
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    delegate_ = nil;
    
    //////////////////////////////////////////////////

    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    //////////////////////////////////////////////////

    [lastRecognitionError_ release];
    lastRecognitionError_ = nil;
    
    [recognitionCountDictionary_ removeAllObjects];
    [recognitionCountDictionary_ release];
    recognitionCountDictionary_ = nil;
    
    [normalRecogSourceArray_ removeAllObjects];
    [normalRecogSourceArray_ release];
    normalRecogSourceArray_ = nil;
    
    [silentRecogSourceArray_ removeAllObjects];
    [silentRecogSourceArray_ release];
    silentRecogSourceArray_ = nil;
    
    [recognitionQueue_ release];
    recognitionQueue_ = nil;

    [lastRecognitionCardModel_ release];
    lastRecognitionCardModel_  = nil;
    
    [lastWCRecogSourceModel_ release];
    lastWCRecogSourceModel_ = nil;
    
    [lastRecognitionImage_ release];
    lastRecognitionImage_ = nil;

    // release BCR kernel
    Disconnect();
    
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - private methods

//===============================================================================
//
//===============================================================================
- (void)mainThreadDelegateDidStartAllRecognition
{
    if([self.delegate respondsToSelector:@selector(cardRecognitionControllerStartAllRecognition:)]==YES)
    {
        [self.delegate cardRecognitionControllerStartAllRecognition:self];
    }
}


//===============================================================================
//
//===============================================================================
- (void)mainThreadDelegateDidFinishAllRecog:(NSNumber *)recogMode
{
    if([self.delegate respondsToSelector:@selector(cardRecognitionController:didFinishAllRecogWithMode:)])
    {
        [self.delegate cardRecognitionController:self didFinishAllRecogWithMode:[recogMode integerValue]];
    }
}


//===============================================================================
//
//===============================================================================
- (void)sendRecogOneSourceWithRecogSourceModel:(WCRecogSourceModel *)recogSourceModel
                                    recogImage:(UIImage *)recogImage
                    ppRecognitionOperationMode:(RecognitionOperationMode)recogMode
                                     cardModel:(WCCardModel *)cardModel
                                     restCount:(NSInteger)restCount
                                         error:(NSError *)error
{
    
    do
    {
        PPLogFunction(@" %s",__func__);

//        for(WCRecogSourceModel *sourceModel in  self.silentRecogSourceArray)
//        {
//            NSLog(@" sourceModel.imageType:%td, sourceModel.cardId:%@",sourceModel.imageType,sourceModel.cardID);
//        }
        
        //////////////////////////////////////////////////

        if(error==nil)
        {
            // MARK: 辨識成功
            
            // index  0: WCRecogSourceModel, 1: image, 2:RecognitionOperationMode 3: WCRecognitionCardModel 4:restCount
        }
        else
        {
            // MARK: 辨識失敗
            
            //不處理繼續往下做
            
             PPLogFunction(@" %s, error:%@",__func__,error);
        }

        //////////////////////////////////////////////////

        //至少還有一張圖片待辨識
        if(self.silentRecogSourceArray.count>1)
        {
            WCRecogSourceModel *nextRecognitionSourceModel = [self.silentRecogSourceArray objectAtIndex:1];
            
            //此id有反面,保留資訊
            if(recogSourceModel.cardID!=nil &&
               [recogSourceModel.cardID compare:nextRecognitionSourceModel.cardID]==NSOrderedSame)
            {
                self.lastRecognitionCardModel     = cardModel;
                self.lastRecognitionError         = nil;
                self.lastRecognitionImage         = recogImage;
                self.lastWCRecogSourceModel       = recogSourceModel;
                self.lastRecognitionOperationMode = recogMode;
                
                break;
            }
        }
        
        //////////////////////////////////////////////////

        if(recogMode==RecognitionOperationMode_Silent)
        {
            if([self.delegate respondsToSelector:@selector(cardRecognitionController:recogFrontImage:recogBackImage:resultFrontCardModel:resultBackCardModel:ppRecognitionOperationMode:frontCardRecognitionError:backCardRecognitionError:)]==YES)
            {
                //此張名片只有正面圖
                if(recogSourceModel.imageType==WC_IT_FrontSide)
                {
                    [self.delegate cardRecognitionController:self
                                             recogFrontImage:recogImage
                                              recogBackImage:nil
                                        resultFrontCardModel:cardModel
                                         resultBackCardModel:nil
                                  ppRecognitionOperationMode:recogMode
                                   frontCardRecognitionError:error
                                    backCardRecognitionError:nil];
                }
                else
                {
                    [self.delegate cardRecognitionController:self
                                             recogFrontImage:self.lastRecognitionImage
                                              recogBackImage:recogImage
                                        resultFrontCardModel:self.lastRecognitionCardModel
                                         resultBackCardModel:cardModel
                                  ppRecognitionOperationMode:recogMode
                                   frontCardRecognitionError:self.lastRecognitionError
                                    backCardRecognitionError:error];
                    
                    //////////////////////////////////////////////////

                    self.lastRecognitionCardModel     = nil;
                    self.lastRecognitionError         = nil;
                    self.lastRecognitionImage         = nil;
                    self.lastWCRecogSourceModel       = nil;
                }
            }
        }
        else
        {
            if([self.delegate respondsToSelector:@selector(cardRecognitionControllerDidRecogOneSource:recogImage:ppRecognitionOperationMode:resultCardModel:error:)]==YES)
            {
                [self.delegate cardRecognitionControllerDidRecogOneSource:recogSourceModel
                                                               recogImage:recogImage
                                               ppRecognitionOperationMode:recogMode
                                                          resultCardModel:cardModel
                                                                    error:nil];
            }
        }
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification methods

//================================================================================
//
//================================================================================
- (void)receiveRecognitionOperationEndNotification:(NSNotification *)notification
{
    [self performSelectorOnMainThread:@selector(mainThreadDelegateDidFinishAllRecog:)
                           withObject:@(RecognitionOperationMode_Silent)
                        waitUntilDone:YES];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - RecognitionOperation methods

//===============================================================================
//
//===============================================================================
- (UIImage *)wcroResetImageBeforeRecognizing:(UIImage *)image recogSource:(WCRecogSourceModel *)recogSource
{
    UIImage *resetImage = nil;
    
    do
    {
        if(self.delegate==nil ||
           [self.delegate respondsToSelector:@selector(resetImageBeforeRecognizing:withCardRecognitionController:recogSource:)]==NO)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////
        
        resetImage = [self.delegate resetImageBeforeRecognizing:image
                                  withCardRecognitionController:self
                                                    recogSource:recogSource];
    } while (0);


    return resetImage;
}


//===============================================================================
//
//===============================================================================
- (UIImage *)wcroResetImageBeforeSaving:(UIImage *)image recogSource:(WCRecogSourceModel *)recogSource
{
    UIImage *resetImage = nil;
    
    do
    {
        if(self.delegate==nil ||
           [self.delegate respondsToSelector:@selector(resetImageBeforeSaving:withCardRecognitionController:recogSource:)]==NO)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////
        
        resetImage = [self.delegate resetImageBeforeSaving:image
                             withCardRecognitionController:self
                                               recogSource:recogSource];
    } while (0);
    
    
    return resetImage;
}


//===============================================================================
//
//===============================================================================
- (void)wcroWillStartRecogWithMode:(RecognitionOperationMode)recogMode totalSource:(NSInteger)totalSource
{
    self.totalSource = totalSource;
    
    [self performSelectorOnMainThread:@selector(mainThreadDelegateDidStartAllRecognition) withObject:nil waitUntilDone:NO];
}


//===============================================================================
//
//===============================================================================
- (void)wcroDidFinishRecogWithMode:(RecognitionOperationMode)recogMode
{
    if(recogMode==RecognitionOperationMode_Silent)
    {
        [self performSelectorOnMainThread:@selector(mainThreadNotifyRecogWaitingCount:) withObject:[NSNumber numberWithInt:0] waitUntilDone:YES];
    }
}

//================================================================================
//
//================================================================================
- (WCCardModel *)cardModelWithRecogImage:(UIImage *)recogImage
                             recogSource:(WCRecogSourceModel *)recogSource
                                   error:(NSError **)error
{
    WCCardModel *cardModel = nil;
    
    if([self.delegate respondsToSelector:@selector(cardModelWithCardRecognitionController:recogImage:recogSource:error:)]==YES)
    {
        cardModel = [self.delegate cardModelWithCardRecognitionController:self
                                                               recogImage:recogImage
                                                              recogSource:recogSource
                                                                    error:error];
    }
    
    return cardModel;
}



//================================================================================
//
//================================================================================
- (void)wcroRecogOneSource:(WCRecogSourceModel *)recogSource
                recogImage:(UIImage *)recogImage
ppRecognitionOperationMode:(RecognitionOperationMode)recogMode
                 cardModel:(WCCardModel *)cardModel
                 restCount:(NSInteger)restCount
                     error:(NSError *)error
{
    // MARK: Howard, 辨識用非亮化影像，顯示用亮化
    UIImage *lightImage = recogImage;
    
    do
    {
        if(recogSource.dockMode==YES ||
           recogSource.imageEnhance==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        lightImage = [CardRecognitionController imageFromRecogSource:recogSource lightImage:YES];
        
        if(lightImage==nil)
        {
            lightImage = recogImage;
        }
        
    } while (0);
    
    //////////////////////////////////////////////////

    // 移除 local image
    if([recogSource.imageSource isKindOfClass:[NSString class]]==YES &&
       [[NSFileManager defaultManager] fileExistsAtPath:recogSource.imageSource]==YES)
    {
       [[NSFileManager defaultManager] removeItemAtPath:recogSource.imageSource error:nil];
    }
    
    if([recogSource.lightImageSource isKindOfClass:[NSString class]]==YES &&
       [[NSFileManager defaultManager] fileExistsAtPath:recogSource.lightImageSource]==YES)
    {
        [[NSFileManager defaultManager] removeItemAtPath:recogSource.lightImageSource error:nil];
    }
    
    //////////////////////////////////////////////////

    [self sendRecogOneSourceWithRecogSourceModel:recogSource
                                      recogImage:lightImage
                      ppRecognitionOperationMode:recogMode
                                       cardModel:cardModel
                                       restCount:restCount
                                           error:error];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance methods

//===============================================================================
//
//===============================================================================
- (void)mainThreadNotifyRecogWaitingCount:(NSNumber *)waitingCount
{
    PPLogFunctionIn();
    
    // notify update waiting counter in toolbar
	NSDictionary *userInfo = [[NSDictionary alloc] initWithObjectsAndKeys:waitingCount,CardRecognitionController_SilentRecogRestCount, nil];
    
	[[NSNotificationCenter defaultCenter] postNotificationName:CardRecognitionController_SilentRecogRestCount object:nil userInfo:userInfo];
    
    [userInfo release];
    
    PPLogFunctionOut();
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)startRecogWithMode:(RecognitionOperationMode)recogMode sources:(NSMutableArray *)recogSourceArray
{
    PPLogFunctionIn();

    self.recognitionCountDictionary = [[[NSMutableDictionary alloc] init] autorelease];

    //////////////////////////////////////////////////
    
    switch (recogMode)
    {
        case RecognitionOperationMode_Normal:
        {
            if(self.normalRecogSourceArray==nil)
            {
                normalRecogSourceArray_ = [[NSMutableArray alloc]init];
            }

            if([recogSourceArray count])
            {
                [self.normalRecogSourceArray addObjectsFromArray:recogSourceArray];
                
                //////////////////////////////////////////////////
                
                WCRecogSourceModel *sourceModel = [self.normalRecogSourceArray objectAtIndex:0];
                
                [self.recognitionCountDictionary setObject:[NSNumber numberWithUnsignedInteger:1] forKey:sourceModel.cardID];
            }
            
            //////////////////////////////////////////////////
            
            // !! 辨識的動畫由外部自行處理
            RecognitionOperation *operation = [[RecognitionOperation alloc] initWithDelegate:self recogSources:self.normalRecogSourceArray];
            
            operation.recognitionOpertaionMode = RecognitionOperationMode_Normal;
            [self.recognitionQueue addOperation:operation];
            
            [operation release];

            break;
        }
            
        case RecognitionOperationMode_Silent:
        {
            if(self.silentRecogSourceArray==nil)
            {
                silentRecogSourceArray_ = [[NSMutableArray alloc] init];
            }
            
            if([recogSourceArray count])
            {
                [self.silentRecogSourceArray addObjectsFromArray:recogSourceArray];
            }
            
            //////////////////////////////////////////////////
            
            //編號
            for (WCRecogSourceModel *sourceModel in self.silentRecogSourceArray)
            {
                NSNumber *numberOfCard = [self.recognitionCountDictionary objectForKey:sourceModel.cardID];
                
                if(numberOfCard==nil)
                {
                    [self.recognitionCountDictionary setObject:[NSNumber numberWithUnsignedInteger:1] forKey:sourceModel.cardID];
                }
                else
                {
                    [self.recognitionCountDictionary setObject:[NSNumber numberWithUnsignedInteger:2] forKey:sourceModel.cardID];
                }
            }
            
            //////////////////////////////////////////////////

            RecognitionOperation *operation = [[RecognitionOperation alloc] initWithDelegate:self
                                                                                recogSources:self.silentRecogSourceArray];
            
            if(operation!=nil)
            {
                operation.recognitionOpertaionMode = RecognitionOperationMode_Silent;
                
                @synchronized(self.recognitionQueue)
                {
                    if([self.recognitionQueue operationCount]>0)
                    {
                        [self.recognitionQueue cancelAllOperations];
                    }
                    
                    [self.recognitionQueue addOperation:operation];
                }
            }
            
            [operation release];

            break;
        }
            
        default:
        {
            break;
        }
    }
    
    //////////////////////////////////////////////////
    
    [self suspendSilentRecognitionProcess:NO];
    
    //////////////////////////////////////////////////

    PPLogFunctionOut();
}


//================================================================================
//
//================================================================================
- (void)suspendSilentRecognitionProcess:(BOOL)suspend
{
    PPLogFunctionIn();

    //////////////////////////////////////////////////

    if([self.recognitionQueue operationCount])
    {
        NSArray *operationArray = [self.recognitionQueue operations];
        
        for(RecognitionOperation *operation in operationArray)
        {
            if(operation.recognitionOpertaionMode == RecognitionOperationMode_Silent)
            {
                operation.suspended = suspend;
            }
        }
    }
    
    PPLogFunctionOut();
}


//================================================================================
//
//================================================================================
- (void)cancelSilentRecognitionProcess
{
    PPLogFunctionIn();
    
    [self.recognitionQueue cancelAllOperations];

    PPLogFunctionOut();
}


//===============================================================================
//
//===============================================================================
- (NSInteger)silentRecognitionCount
{
    NSInteger restCount = [RecognitionOperation cardCountInRecogSourceArray:self.silentRecogSourceArray withStartIndex:0];
    return restCount;
}


//===============================================================================
//
//===============================================================================
- (NSUInteger)silentRecognitionCountForUserCount
{
    //只要有 cardID 存在，即加1(正反面名片只算1)
    return self.recognitionCountDictionary.allKeys.count;
}


//================================================================================
//
//================================================================================
- (BOOL)operationWorking
{
    if([[self recognitionQueue] operationCount]>0)
    {
        return YES;
    }
    else
    {
        return NO;
    }
}

//===============================================================================
//
//===============================================================================
- (void)minusRecognitionCountForCardID:(NSString *)cardID
{
    NSNumber *numberOfCard = [self.recognitionCountDictionary objectForKey:cardID];
    
    if(numberOfCard!=nil)
    {
        if([numberOfCard unsignedIntegerValue]==2)
        {
            [self.recognitionCountDictionary setObject:[NSNumber numberWithUnsignedInteger:1] forKey:cardID];
        }
        else if([numberOfCard unsignedIntegerValue]<=1)
        {
            [self.recognitionCountDictionary removeObjectForKey:cardID];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)removeAllLocalRecognitionAssets
{
    for(WCRecogSourceModel *recogSourceModel in self.silentRecogSourceArray)
    {
        // 移除 local image
        if([recogSourceModel.imageSource isKindOfClass:[NSString class]]==YES &&
           [[NSFileManager defaultManager] fileExistsAtPath:recogSourceModel.imageSource]==YES)
        {
            [[NSFileManager defaultManager] removeItemAtPath:recogSourceModel.imageSource error:nil];
        }
        
        if([recogSourceModel.lightImageSource isKindOfClass:[NSString class]]==YES &&
           [[NSFileManager defaultManager] fileExistsAtPath:recogSourceModel.lightImageSource]==YES)
        {
            [[NSFileManager defaultManager] removeItemAtPath:recogSourceModel.lightImageSource error:nil];
        }
    }
    
    //////////////////////////////////////////////////

    [self.silentRecogSourceArray removeAllObjects];
}


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (instancetype)shareRecognitionInstance
{
    static CardRecognitionController *shareRecognitionInstance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^ {
        shareRecognitionInstance = [[self alloc] init];
    });
    return shareRecognitionInstance;
}


//===============================================================================
//
//===============================================================================
+ (UIImage *)imageFromRecogSource:(WCRecogSourceModel *)recogSource lightImage:(BOOL)lightImage
{
    __block UIImage *resultImage = nil;
    id imageSource = lightImage?recogSource.lightImageSource:recogSource.imageSource;
    //-------------------------------
    // get image object
    //-------------------------------
    if([imageSource isKindOfClass:[UIImage class]])
    {
        resultImage = imageSource;
    }
    else if([recogSource.imageSource isKindOfClass:[PHAsset class]])
    {
        PHAsset *asset = (PHAsset *)recogSource.imageSource;
        resultImage = [asset imageWithTagetSize:CGSizeMake(asset.pixelWidth, asset.pixelHeight)];
    }
    else if([imageSource isKindOfClass:[NSString class]])
    {
        NSString *filePath = (NSString *)imageSource;
        
        NSData *img = [NSData dataWithContentsOfFile:filePath];
        resultImage = [UIImage imageWithData:img];
    }
    
    
    UIImage *tempImage = nil;
    
    //-------------------------------
    // rotate/resize image
    //-------------------------------
    if(recogSource.rotateDegree != 255) // 255代表核心自動旋轉
    {
        CGFloat maxLength = WC_IMS_Original;
        
        if(maxLength>MAX(resultImage.size.width,resultImage.size.height))
        {
            maxLength = MAX(resultImage.size.width,resultImage.size.height);
        }
        
        tempImage = [resultImage imageRotatedByDegrees:recogSource.rotateDegree scalingMaxLength:maxLength];
        resultImage = tempImage;
        
        recogSource.rotateDegree = 0;
    }
    
    return resultImage;
}
@end
