//
//  WCGroupSelectorViewController.m
//  Pods
//
//  Created by Andrew on 2015/7/9.
//
//

#import "WCSingleLayerGroupViewController.h"

// define
#import "UIColor+HexString.h"
#import "WCGroupDefine.h"
#import "WCAppearanceDefine.h"
#import "WCSingleLayerGroupViewController+ResourceDefine.h"

// categories
#import "PPButton+Factory.h"
#import "UIViewController+ShareApplication.h"
#import "UIApplication+Idle.h"

// View
#import "PPAlertView.h"
#import "PPButton.h"
#import "PPNavigationBarView.h"
#import "PPTableView.h"
#import "PPBusyView.h"

// Controller
#import "PPLogController.h"
#import "PPGuideController.h"
#import "PPSettingsController.h"
#import "WCAddOrEditGroupViewController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, WCSingleLayerGroupViewControllerButtonTag)
{
    WCSingleLayerGroupViewControllerButtonTag_Back = 0,
    WCSingleLayerGroupViewControllerButtonTag_Add,
    WCSingleLayerGroupViewControllerButtonTag_Sort,
    WCSingleLayerGroupViewControllerButtonTag_Done
};

typedef NS_ENUM(NSUInteger, WCSingleLayerGroupViewControllerAlertTag)
{
    WCSingleLayerGroupViewControllerAlertTag_Add = 0,
    WCSingleLayerGroupViewControllerAlertTag_Edit,
    WCSingleLayerGroupViewControllerAlertTag_Sort,
};

typedef NS_ENUM(NSUInteger, WCSingleLayerGroupViewControllerGuideIndex)
{
    WCSingleLayerGroupViewControllerGuideIndex_Add = 1000,
    WCSingleLayerGroupViewControllerGuideIndex_Reorder
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCSingleLayerGroupViewController () <PPButtonDelegate, UIAlertViewDelegate, UITableViewDataSource, UITableViewDelegate, PPGuideControllerDelegate>

@property (nonatomic, retain) NSMutableArray      *sourceGroupArray;
@property (nonatomic, retain) NSMutableArray      *selectedGroupArray;

@property (nonatomic, retain) PPButton            *backButton;
@property (nonatomic, retain) PPButton            *addButton;
@property (nonatomic, retain) PPButton            *doneButton;
@property (nonatomic, retain) PPButton            *sortButton;
@property (nonatomic, retain) PPNavigationBarView *navigationBarView;
@property (nonatomic, retain) PPTableView         *groupSelectorView;
@property (nonatomic, retain) PPBusyView          *busyView;
@property (nonatomic, retain) UILabel             *noResultLabel;
@property (nonatomic, retain) NSIndexPath        *selectedIndexPath;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCSingleLayerGroupViewController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        [PPSettingsController setDefaultIntegerValue:0 withKey:WCGroupViewControllerSettingsKey_kIsUseAdjustGroupFunction];
        [PPSettingsController setDefaultIntegerValue:0 withKey:WCGroupViewControllerTipsKey_ShowGuide];
        
        self.clickCellToEdit = YES;
        self.showSortButton = YES;
    }
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;

    self.sourceGroupArray = nil;
    self.selectedGroupArray = nil;
 
    self.selectedIndexPath = nil;
    
    self.userInfo = nil;    
    self.titleString = nil;
    self.noResultString = nil;
    
    [self removeMainUI];

    [self.busyView removeFromSuperview];
    self.busyView = nil;
    //////////////////////////////////////////////////
    
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////

    [self prepareMainUI];
    //////////////////////////////////////////////////
    
    [self loadGroupData];
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [self removeMainUI];
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Method

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    self.groupSelectorView.frame = [self layoutFrame];
}


//==============================================================================
//
//==============================================================================
- (void)prepareMainUI
{
    if(_navigationBarView==nil)
    {
        _navigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
    }
    
    if (self.navigationBarView!=nil)
    {
        self.navigationBarView.style = PPBarViewStyle_CenteredAbsolute;
        self.navigationBarView.titleLabel.text = self.titleString;
        
        if(self.backButton==nil)
        {
            self.backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                              tag:WCSingleLayerGroupViewControllerButtonTag_Back
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        }
        
        if(self.backButton!=nil)
        {
            //////////////////////////////////////////////////
            
            [self.navigationBarView setViews:@[self.backButton] forBlockType:PPBarViewBlockType_Left];
        }
        
        NSMutableArray *rightButtons = [NSMutableArray array];

        //////////////////////////////////////////////////
        //!! 編輯group才要sort
        if (self.style==WCSingleLayerGroupViewControllerStyle_Edit &&
            self.showSortButton)
        {
            if(self.sortButton==nil)
            {
                self.sortButton = [PPButton ppButtonWithIconImageName:WCSingleLayerGroupViewControllerButtonName_Sort
                                                                  tag:WCSingleLayerGroupViewControllerButtonTag_Sort
                                                             delegate:self
                                                normalBackgroundColor:nil
                                           highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                      imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
            }
            [rightButtons addObject:self.sortButton];
                
        }
        
        //////////////////////////////////////////////////
        
        if(self.style!=WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
        {
            if(self.addButton==nil)
            {
                self.addButton = [PPButton ppButtonWithIconImageName:WCSingleLayerGroupViewControllerButtonName_Add
                                                                 tag:WCSingleLayerGroupViewControllerButtonTag_Add
                                                            delegate:self
                                               normalBackgroundColor:nil
                                          highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                     imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
            }
            self.addButton.ppGuideIndex = WCSingleLayerGroupViewControllerGuideIndex_Add;
            [rightButtons addObject:self.addButton];
        }
        
        //////////////////////////////////////////////////
        //!!change group才要done
        if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
            self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
        {
            if(self.doneButton==nil)
            {
                self.doneButton = [PPButton ppButtonWithIconImageName:WCSingleLayerGroupViewControllerButtonName_Done
                                                                  tag:WCSingleLayerGroupViewControllerButtonTag_Done
                                                             delegate:self
                                                normalBackgroundColor:nil
                                           highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                      imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
            }
            [rightButtons addObject:self.doneButton];
        }
        
        
        
        if([rightButtons count]>0)
        {
            [self.navigationBarView setViews:rightButtons forBlockType:PPBarViewBlockType_Right];
        }
        
        self.navigationItem.leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.navigationBarView] autorelease];
    }
    
    //////////////////////////////////////////////////
    if(_groupSelectorView==nil)
    {
        _groupSelectorView = [[PPTableView alloc] initWithFrame:self.view.bounds];
    }

    if (self.groupSelectorView!=nil)
    {
        self.groupSelectorView.dataSource = self;
        self.groupSelectorView.delegate   = self;
        
        //////////////////////////////////////////////////
        [self.view addSubview:self.groupSelectorView];
        
        //////////////////////////////////////////////////
        
        [self.groupSelectorView setSeparatorColor:WCAppearanceDefine_SeparatorColor];
        [self.groupSelectorView setAllowsSelectionDuringEditing:YES];
        [self.groupSelectorView setEditing:YES];
    }
    
    //////////////////////////////////////////////////
    if(_noResultLabel==nil)
    {
        _noResultLabel = [[UILabel alloc] initWithFrame:CGRectMake(0, 0, self.view.bounds.size.width, 44)];
    }
    
    if (self.noResultLabel)
    {
        self.noResultLabel.font = [UIFont boldSystemFontOfSize:18];
        self.noResultLabel.textColor = [UIColor grayColor];
        self.noResultLabel.backgroundColor = [UIColor clearColor];
        self.noResultLabel.textAlignment = NSTextAlignmentCenter;
        self.noResultLabel.hidden = YES;
        self.noResultLabel.text = self.noResultString;
        [self.view addSubview:self.noResultLabel];
    }
    
}


//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    self.sortButton.delegate = nil;
    self.sortButton = nil;
    
    self.addButton.delegate = nil;
    self.addButton = nil;
    
    self.backButton.delegate = nil;
    self.backButton = nil;
    
    self.doneButton.delegate = nil;
    self.doneButton = nil;
    
    self.navigationBarView = nil;
    
    [self.groupSelectorView removeFromSuperview];
    self.groupSelectorView.dataSource = nil;
    self.groupSelectorView.delegate = nil;
    self.groupSelectorView = nil;
    
    [self.noResultLabel removeFromSuperview];
    self.noResultLabel = nil;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - busyview

//================================================================================
//
//================================================================================
- (void)setBusy:(NSNumber*)busy
{
    @synchronized(self)
    {
        if([busy boolValue])
        {
            if (self.busyView)
            {
                [self.busyView removeFromSuperview];
                self.busyView = nil;
            }
            
            self.busyView = [[[PPBusyView alloc] initWithSuperView:self.navigationController.view] autorelease];
        }
        else // hide busy view
        {
            [self.busyView removeFromSuperview];
            self.busyView = nil;
        }
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (private) load data


//==============================================================================
//
//==============================================================================
- (void)loadGroupData
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        dispatch_sync(dispatch_get_main_queue(), ^{
            [blockSelf setBusy:@(YES)];
        });
        
        //////////////////////////////////////////////////
        // Source Group Array
        // 如果已存在blockSelf.sourceGroupArray，就不要重load
        if (blockSelf.sourceGroupArray==nil &&
            blockSelf.delegate!=nil &&
            [blockSelf.delegate respondsToSelector:@selector(sourceGroupArrayForGroupViewController:)]==YES)
        {
            NSMutableArray *groupArray = [blockSelf.delegate sourceGroupArrayForGroupViewController:blockSelf];
            if ([groupArray count]>0)
            {
                blockSelf.sourceGroupArray = [NSMutableArray arrayWithArray:groupArray];
                
                // 重load之後要重新取得
                self.selectedGroupArray = nil;
            }
        }
        
        //////////////////////////////////////////////////
        
        dispatch_sync(dispatch_get_main_queue(), ^{
            [self.groupSelectorView reloadData];
        });

        //////////////////////////////////////////////////
        // 需先將view添加至頁面上，才能進行選取動作，否則無反應
        [blockSelf loadSelectedGroupArray];
        
        //////////////////////////////////////////////////
        dispatch_sync(dispatch_get_main_queue(), ^{
            [blockSelf setBusy:@(NO)];
            [blockSelf showGuideIfNeeded];
        });
    });
}


//==============================================================================
// run on bg thread
//==============================================================================
- (void)loadSelectedGroupArray
{

    //////////////////////////////////////////////////
    // Selected GroupID Array
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(selectedGroupArrayForGroupViewController:)]==YES)
    {
        NSArray *groupSelectedArray = nil;

        // !!self.selectedGroupArray不存在，才要重load
        if (self.selectedGroupArray==nil)
        {
            groupSelectedArray = [self.delegate selectedGroupArrayForGroupViewController:self];
            self.selectedGroupArray = [NSMutableArray arrayWithArray:groupSelectedArray];
        }
        if ([self.selectedGroupArray count]==0 || [self.sourceGroupArray count]==1)
        {
            // !! 沒有選取群組時，預設為unfiled
            for(NSInteger sourceIndex=0; sourceIndex<self.sourceGroupArray.count; sourceIndex++)
            {
                if(((WCGroupModel *)[self.sourceGroupArray objectAtIndex:sourceIndex]).ID==WC_GID_Unfiled)
                {
                    dispatch_sync(dispatch_get_main_queue(), ^{
                        
                        [self.groupSelectorView selectRowAtIndexPath:[NSIndexPath indexPathForRow:sourceIndex inSection:0] animated:NO scrollPosition:UITableViewScrollPositionNone];
                    });
                }
            }
        }
        else
        {
            // 設定已選擇的Group
            for(NSInteger selectIndex=0; selectIndex<self.selectedGroupArray.count;selectIndex++)
            {
                for(NSInteger sourceIndex=0; sourceIndex<self.sourceGroupArray.count; sourceIndex++)
                {
                    if(((WCGroupModel *)[self.selectedGroupArray objectAtIndex:selectIndex]).ID == ((WCGroupModel *)[self.sourceGroupArray objectAtIndex:sourceIndex]).ID)
                    {
                        dispatch_sync(dispatch_get_main_queue(), ^{
                            
                            [self.groupSelectorView selectRowAtIndexPath:[NSIndexPath indexPathForRow:sourceIndex inSection:0] animated:NO scrollPosition:UITableViewScrollPositionNone];
                        });
                        
                    }
                }
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (private) 記錄是否使用過群組排序


//==============================================================================
//
//==============================================================================
- (void)logGroupOrderUsage
{
    
    BOOL isUserdAdjustOrder = (BOOL)[PPSettingsController integerValueWithKey:WCGroupViewControllerSettingsKey_kIsUseAdjustGroupFunction];
    // 已有用過，就不用更新
    if (isUserdAdjustOrder==NO)
    {
        [PPSettingsController setIntegerValue:1 withKey:WCGroupViewControllerSettingsKey_kIsUseAdjustGroupFunction];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Guide methods


//==============================================================================
//
//==============================================================================
- (void)showGuideIfNeeded
{
    UITableViewCell *cell = [self.groupSelectorView cellForRowAtIndexPath:[NSIndexPath indexPathForRow:0 inSection:0]];
    if (cell)
    {
        NSString *reorderClass = [NSString stringWithFormat:@"%@%@", @"UITableViewCell", @"ReorderControl"];
        for (UIView *subview in [cell subviews])
        {
            if ([subview isKindOfClass:NSClassFromString(reorderClass)])
            {
                subview.ppGuideIndex = WCSingleLayerGroupViewControllerGuideIndex_Reorder;
                break;
            }
        }
    }
    
    if ([PPSettingsController integerValueWithKey:WCGroupViewControllerTipsKey_ShowGuide]==NO)
    {        
        [PPGuideController setDelegate:self];
        [PPGuideController clearAllGuideScripts];
        [PPGuideController addGuideScriptWithName:@"WCroupViewGuide" guideDataModels:[WCSingleLayerGroupViewController guideDataModels]];
        [PPGuideController manuallyShowGuideWithViewController:self];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark PPGuideControllerDelegate


//==============================================================================
//
//==============================================================================
- (void)guideControllerDidFinishScript:(NSString *)guideScriptName
{
    [PPSettingsController setIntegerValue:1 withKey:WCGroupViewControllerTipsKey_ShowGuide];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Private) modify group name


//==============================================================================
// @param name 有值表示是修改，沒有值表示是新增
//==============================================================================
- (void)showGroupEditorWithName:(NSString *)name
{
    __block typeof(self) blockSelf = self;

    [WCAddOrEditGroupViewController showAddGroupFromViewController:self
                                                     withMaxLength:self.maxGroupLength
                                                      currentGroup:name
                                                groupNameValidator:^BOOL(NSString *groupName)
     {
         
         if ([blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:isExistGroupNamed:)])
         {
             return ![blockSelf.delegate singleLayerGroupViewController:self isExistGroupNamed:groupName];
         }
         else
         {
             return YES;
         }
     }
                                                   completeHandler:^void (NSString *groupName)
     {
         
         __block typeof(self) blockSelf = self;
         
         dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
             
             if([name length]==0)
             {
                 BOOL isGroupExist = NO;
                 
                 // 再確認一次沒有重名
                 if ([blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:isExistGroupNamed:)])
                 {
                     isGroupExist = [blockSelf.delegate singleLayerGroupViewController:blockSelf isExistGroupNamed:groupName];
                 }
                 
                 // 不存在才能新增
                 if (!isGroupExist && [blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:groupModelByNewGroupName:)])
                 {
                     NSMutableArray *tempGroupArray = [blockSelf.sourceGroupArray mutableCopy];
                     WCGroupModel * newGroupModel = [blockSelf.delegate singleLayerGroupViewController:blockSelf groupModelByNewGroupName:groupName];
                     // 加到group array中
                     [tempGroupArray addObject:newGroupModel];
                     
                     //////////////////////////////////////////////////
                     
                     // 加到最後一個
                     if ([tempGroupArray count]>0)
                     {
                         dispatch_async(dispatch_get_main_queue(), ^{
                             
                             blockSelf.sourceGroupArray = tempGroupArray;
                             
                             if (blockSelf.groupSelectorView)
                             {
                                 // 加到介面
                                 NSIndexPath *indexPath = [NSIndexPath indexPathForRow:[blockSelf.sourceGroupArray count]-1 inSection:0];
                                 [blockSelf.groupSelectorView insertRowsAtIndexPaths:@[indexPath] withRowAnimation:UITableViewRowAnimationAutomatic];
                                 [blockSelf.groupSelectorView scrollToRowAtIndexPath:indexPath atScrollPosition:UITableViewScrollPositionMiddle animated:YES];
                             }
                             [tempGroupArray release];
                         });
                         return ;
                     }
                     [tempGroupArray release];
                 }
                 
             }
             else
             {
                 //  編輯模式要更新到原本的model上
                 if ([blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:updateGroupModel:)])
                 {
                     WCGroupModel *groupModel = [blockSelf.sourceGroupArray objectAtIndex:[blockSelf.selectedIndexPath row]];
                     
                     // !!編輯的話要確認群組還在才能編輯
                     BOOL isGroupExist = NO;
                     
                     if ([self.delegate respondsToSelector:@selector(singleLayerGroupViewController:isExistGroupNamed:)])
                     {
                         isGroupExist = [self.delegate singleLayerGroupViewController:self isExistGroupNamed:groupModel.name];
                     }
                     
                     // 群組存在才能編輯
                     if (isGroupExist==NO)
                     {
                         return;
                     }
                     
                     if (groupModel!=nil)
                     {
                         [groupModel setName:groupName];
                     }
                     
                     [blockSelf.delegate singleLayerGroupViewController:blockSelf updateGroupModel:groupModel];
                     
                     //////////////////////////////////////////////////
                     // 加到介面
                     if (self.groupSelectorView)
                     {
                         // 加到最後一個
                         if ([blockSelf.sourceGroupArray count]>0)
                         {
                             dispatch_async(dispatch_get_main_queue(), ^{
                                 [blockSelf.groupSelectorView reloadRowsAtIndexPaths:@[blockSelf.selectedIndexPath] withRowAnimation:UITableViewRowAnimationAutomatic];
                             });
                             
                         }
                     }
                 }
             }
         });
         
     }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Public) 顯示新增群組介面


//==============================================================================
//
//==============================================================================
- (void)showNewGroupUI
{
    [self showGroupEditorWithName:nil];
}


//==============================================================================
//
//==============================================================================
- (void)clearAndLoadGroupData
{
    self.sourceGroupArray = nil;
    [self loadGroupData];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UIAlertViewDelegate


//==============================================================================
//
//==============================================================================
- (void)alertView:(UIAlertView *)alertView clickedButtonAtIndex:(NSInteger)buttonIndex
{
    __block typeof(self) blockSelf = self;
    NSString *buttonTitle = [alertView buttonTitleAtIndex:buttonIndex];

    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        if ([buttonTitle isEqualToString:WCSingleLayerGroupViewControllerString_Sort])
        {
            // 排序目前群組
            if ([self.delegate respondsToSelector:@selector(singleLayerGroupViewController:sortedGroupWithCurrentGroups:)])
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    [blockSelf setBusy:@(YES)];
                });
                
                
                self.sourceGroupArray = [self.delegate singleLayerGroupViewController:self sortedGroupWithCurrentGroups:self.sourceGroupArray];
                
                // 更新介面
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf.groupSelectorView reloadData];
                    [blockSelf setBusy:@(NO)];
                });
            }
        }
        
    });
    
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        switch (ppButton.tag)
        {
            case WCSingleLayerGroupViewControllerButtonTag_Back:
            {
                if (self.style==WCSingleLayerGroupViewControllerStyle_Edit)
                {
                    if ([blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:didEditFinishWithGroupArray:)])
                    {
                        [blockSelf setBusy:@(YES)];
                        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                            [blockSelf.delegate singleLayerGroupViewController:blockSelf didEditFinishWithGroupArray:blockSelf.sourceGroupArray];
                            
                            //////////////////////////////////////////////////
                            dispatch_async(dispatch_get_main_queue(), ^{
                                [blockSelf setBusy:@(NO)];
                                [blockSelf goBackAnimated:YES];
                            });
                        });
                    }
                    else
                    {
                        [blockSelf goBackAnimated:YES];
                    }
                }
                else
                {
                    [blockSelf goBackAnimated:YES];
                }
                break;
            }
            case WCSingleLayerGroupViewControllerButtonTag_Add:
            {
                if ([self.delegate respondsToSelector:@selector(didClickNewGroupWithViewController:)])
                {
                    [self.delegate didClickNewGroupWithViewController:self];
                }
                break;
            }
            case WCSingleLayerGroupViewControllerButtonTag_Sort:
            {
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCSingleLayerGroupViewControllerString_SortGroupAlert
                                  delegate:self
                                       tag:WCSingleLayerGroupViewControllerButtonTag_Sort
                         cancelButtonTitle:WCSingleLayerGroupViewControllerString_Cancel
                         otherButtonTitles:WCSingleLayerGroupViewControllerString_Sort, nil];
                break;
            }
            case WCSingleLayerGroupViewControllerButtonTag_Done:
            {
                if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
                    self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
                {
                    if ([blockSelf.delegate respondsToSelector:@selector(singleLayerGroupViewController:groupsDidSelected:)]==YES)
                    {
                        NSArray *selectArray=[blockSelf.groupSelectorView indexPathsForSelectedRows];
                        
                        NSMutableArray *selectGroupArray = [NSMutableArray array];
                        
                        for (NSInteger selectIndex=0; selectIndex<selectArray.count; selectIndex++)
                        {
                            NSIndexPath *selectIndexPath = [selectArray objectAtIndex:selectIndex];
                            WCGroupModel *selectGroup    = ((WCGroupModel*)[blockSelf.sourceGroupArray objectAtIndex:selectIndexPath.row]);
                            
                            [selectGroupArray addObject:selectGroup];
                        }
                        
                        [blockSelf.delegate singleLayerGroupViewController:blockSelf groupsDidSelected:selectGroupArray];
                    }
                }
            
                //////////////////////////////////////////////////
                // !!沒有處理的話，就直接回上一層，有處理delegate的話，就看delegate中的動作
                if ([blockSelf.delegate respondsToSelector:@selector(didFinishSingleGroupGroupViewController:)])
                {
                    [blockSelf.delegate didFinishSingleGroupGroupViewController:blockSelf];
                }
                else
                {
                    [blockSelf goBackAnimated:YES];
                }
                
                break;
            }
            default:
            {
                PPLogFunction(@"Error ppButton Tag");
                break;
            }
        }
    });
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource

//================================================================================
//
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    NSInteger sourceGroupCount = [self.sourceGroupArray count];

    if (sourceGroupCount==0)
    {
        self.noResultLabel.hidden = NO;
    }
    else
    {
        self.noResultLabel.hidden = YES;
    }
    return sourceGroupCount;
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    static NSString *CellIdentifier = @"GroupCell";
    UITableViewCell *cell = (UITableViewCell *)[tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    if (cell == nil)
    {
        cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier] autorelease];
    }
    
    //////////////////////////////////////////////////
    //!! 設定成None無法顯示左邊藍色勾勾符號
    //!! 當滾動時，若該Cell有被選取，會自動呼叫Cell的setSelected，所以只需於setSelected覆寫要呈現的樣式即可
    
    cell.selectionStyle = UITableViewCellSelectionStyleBlue;
    cell.selectedBackgroundView = [[[UIView alloc] init] autorelease];
    cell.selectedBackgroundView.backgroundColor = WCAppearanceDefine_CellHighlightedBGColor;
    cell.shouldIndentWhileEditing = YES;
    //////////////////////////////////////////////////
    
    WCGroupModel *groupModel = [self.sourceGroupArray objectAtIndex:indexPath.row];
    cell.textLabel.text = groupModel.name;
    [cell.textLabel setHighlightedTextColor:WCAppearanceDefine_CellTitleColor];
    
    //////////////////////////////////////////////////
    
    if(groupModel.editable==NO)
    {
        cell.textLabel.textColor = [UIColor lightGrayColor];
    }
    else
    {
        cell.textLabel.textColor = WCAppearanceDefine_CellTitleColor;
    }

    return cell;
}


//===============================================================================
//
//===============================================================================
- (BOOL)tableView:(UITableView *)tableView canMoveRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
        self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
    {
        return NO;
    }
    return YES;
}


//===============================================================================
//
//===============================================================================
- (void)tableView:(UITableView *)tableView moveRowAtIndexPath:(NSIndexPath *)sourceIndexPath toIndexPath:(NSIndexPath *)destinationIndexPath
{
    // 如果沒有移動就不用動作
    if ([sourceIndexPath isEqual:destinationIndexPath])
    {
        return;
    }
    
    
    // 記錄有做過群組排序
    [self logGroupOrderUsage];
    
    //////////////////////////////////////////////////
    NSUInteger          sourceSection = [sourceIndexPath section];
    NSUInteger          sourceRow = [sourceIndexPath row];
    
    NSUInteger          destSection = [destinationIndexPath section];
    NSUInteger          destRow = [destinationIndexPath row];
    
    // 目前只支援同section的移動
    if (sourceSection==destSection)
    {
        // 紀錄舊的順序，如果變更失敗要還原回去
        NSMutableArray *oldGroupArray = [self.sourceGroupArray mutableCopy];
        
        WCGroupModel  *targetGroupModel = [[self.sourceGroupArray objectAtIndex:sourceRow] retain];
        
        [self.sourceGroupArray removeObjectAtIndex:sourceRow];
        [self.sourceGroupArray insertObject:targetGroupModel atIndex:destRow];
        
        [targetGroupModel release];
        
        //////////////////////////////////////////////////
        // 如果有實作singleLayerGroupViewController:didOrderChangedWithNewGroupArray:
        if([self.delegate respondsToSelector:@selector(singleLayerGroupViewController:didOrderChangedWithNewGroupArray:)])
        {
            __block typeof(self) blockSelf = self;
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf setBusy:@(YES)];
                });
                
                BOOL success = [blockSelf.delegate singleLayerGroupViewController:blockSelf didOrderChangedWithNewGroupArray:blockSelf.sourceGroupArray];
                
                if(success==NO)
                {
                    self.sourceGroupArray = [[oldGroupArray mutableCopy] autorelease];
                }
                //////////////////////////////////////////////////
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockSelf.groupSelectorView reloadData];
                    [blockSelf setBusy:@(NO)];
                    [oldGroupArray release];
                });
            });
        }
        else
        {
            [oldGroupArray release];
        }
    }
    else
    {
        
    }
}

//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (editingStyle==UITableViewCellEditingStyleDelete)
    {
        if ([self.delegate respondsToSelector:@selector(singleLayerGroupViewController:deleteGroupModel:)])
        {
            __block typeof(self) blockSelf = self;
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    [blockSelf setBusy:@(YES)];
                    [[UIApplication sharedApplication] enableIdle:NO];
                });

                WCGroupModel *groupModel = [blockSelf.sourceGroupArray objectAtIndex:indexPath.row];

                // WCT用的，因為只是pinned cancel所以不用秀這個
                if (self.forWCTMode==NO)
                {
                    if (groupModel.cardCount >0)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                                 title:@""
                                               message:WCSingleLayerGroupViewControllerString_ContactsWillMoveToUnfiled
                                     cancelButtonTitle:WCSingleLayerGroupViewControllerString_Ok
                                     otherButtonTitles:nil];
                        });
                    }
                }
                
                [blockSelf.delegate singleLayerGroupViewController:self deleteGroupModel:groupModel];
                
                //////////////////////////////////////////////////
                // delete group data
                NSMutableArray *tempGroupArray = [[NSMutableArray arrayWithArray:blockSelf.sourceGroupArray] retain];
                [tempGroupArray removeObject:groupModel];
                
                //////////////////////////////////////////////////
                // 更新介面，目前先全部更新，不然會有bug
                // mantis #0015968: [iOS 9.0_iPhone5] 類別，將自定類別變更順序至上方(圖20151001-14-001)，刪除此自定類別，再去點任意類別，會看到某類別字串跑去左邊(20151001-14-002)
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    blockSelf.sourceGroupArray = tempGroupArray;
                    [tempGroupArray release];
                    // !!  刪除時還是要執行deleteRowsAtIndexPaths:不然就算有reload, 狀態還是會錯誤，造成要多按一下，才能再按刪除的問題
                    [blockSelf.groupSelectorView deleteRowsAtIndexPaths:@[indexPath] withRowAnimation:UITableViewRowAnimationNone];
                    
                    [blockSelf.groupSelectorView reloadData];
                    [blockSelf setBusy:@(NO)];
                    [[UIApplication sharedApplication] enableIdle:YES];
                });
            });

        }
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDelegate


//================================================================================
//
//================================================================================
- (UITableViewCellEditingStyle)tableView:(UITableView *)tableView editingStyleForRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
        self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
    {
        // 呈現空白圈圈(Add+Delete)
        return UITableViewCellEditingStyleDelete+UITableViewCellEditingStyleInsert;
    }
    else
    {
        WCGroupModel *groupModel = [self.sourceGroupArray objectAtIndex:indexPath.row];

        if (groupModel.editable)
        {
            // 可編輯才能出現刪除，不然就都沒有符號
            return UITableViewCellEditingStyleDelete;
        }
        return 0;
    }
}


//================================================================================
// 
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    PPLogFunction(@"didSelectRowAtIndexPath");
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        WCGroupModel *groupModel = [self.sourceGroupArray objectAtIndex:indexPath.row];
        
        //////////////////////////////////////////////////
        if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
            self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
        {
            //!! 因為indexPathsForSelectedRows是判斷tableView內該indexPath是否為選取狀態(純粹設定Cell的Selected無法抓取準確數量)
            //!! 需透過selectRowAtIndexPath和deselectRowAtIndexPath來進行選擇
            //!! 使用selectRowAtIndexPath和deselectRowAtIndexPath來進行選擇時，會反應cell的setSelected:animated:
            //!! 並且將selected設定為tableView內該indexPath的選取狀態
            
            //////////////////////////////////////////////////
            [self.selectedGroupArray addObject:groupModel];
            
            if(groupModel.ID==WC_GID_Unfiled)
            {
                [self.selectedGroupArray removeAllObjects];
                NSArray *selectArray=[tableView indexPathsForSelectedRows];
                
                for(NSInteger selectIndex=0; selectIndex<selectArray.count; selectIndex++)
                {
                    NSIndexPath *selectIndexPath=[selectArray objectAtIndex:selectIndex];
                    
                    if(selectIndexPath.section==indexPath.section && selectIndexPath.row!=indexPath.row)
                    {
                        [tableView deselectRowAtIndexPath:selectIndexPath animated:NO];
                    }
                }
            }
            else
            {
                NSArray *selectArray=[tableView indexPathsForSelectedRows];
                
                for(NSInteger selectIndex=0; selectIndex<selectArray.count; selectIndex++)
                {
                    NSIndexPath *selectIndexPath=[selectArray objectAtIndex:selectIndex];
                    
                    WCGroupModel *selectedGroupModel = [self.sourceGroupArray objectAtIndex:selectIndexPath.row];
                    
                    if(selectedGroupModel.ID==WC_GID_Unfiled)
                    {
                        [tableView deselectRowAtIndexPath:selectIndexPath animated:NO];
                        break;
                    }
                }
            }
        }
        else
        {
            [tableView deselectRowAtIndexPath:indexPath animated:YES];
            
            // 編輯模式，點選為改名稱
            if (self.clickCellToEdit&&
                groupModel.editable)
            {
                blockSelf.selectedIndexPath = indexPath;
                [blockSelf showGroupEditorWithName:groupModel.name];
            }
        }
        
    });
}


//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView didDeselectRowAtIndexPath:(NSIndexPath *)indexPath
{
    PPLogFunction(@"didDeselectRowAtIndexPath");
    
    //////////////////////////////////////////////////
    if (self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroup ||
        self.style==WCSingleLayerGroupViewControllerStyle_ChangeGroupWithoutAdd)
    {
        WCGroupModel *groupModel = [self.sourceGroupArray objectAtIndex:indexPath.row];

        [self.selectedGroupArray removeObject:groupModel];

        NSArray *selectArray=[tableView indexPathsForSelectedRows];
        
        if (selectArray.count==0)
        {
            [tableView selectRowAtIndexPath:[NSIndexPath indexPathForRow:0 inSection:0] animated:NO scrollPosition:UITableViewScrollPositionNone];
        }
    }
}


////================================================================================
////
////================================================================================
//- (NSIndexPath *)tableView:(UITableView *)tableView willSelectRowAtIndexPath:(NSIndexPath *)indexPath
//{
//    PPLogFunction(@"willSelectRowAtIndexPath");
//    
//    return indexPath;
//}
//

//================================================================================
//!! 因為在didDeselectRowAtIndexPath判斷會來不及，當下Cell的Selected狀態已經變更！
//================================================================================
//- (NSIndexPath *)tableView:(UITableView *)tableView willDeselectRowAtIndexPath:(NSIndexPath *)indexPath
//{
//    PPLogFunction(@"willDeselectRowAtIndexPath");
//    
//    //////////////////////////////////////////////////
//    if (self.style == WCGroupSelectorViewControllerStyle_ChangeGroup)
//    {
//        if(((WCGroupModel*)[self.sourceGroupArray objectAtIndex:indexPath.row]).ID==WC_GID_Unfiled)
//        {
//            return nil;
//        }
//    }
//    return indexPath;
//}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare Guide model


//================================================================================
//
//================================================================================
+ (NSArray *)guideDataModels
{
    NSArray *guideDataModels = @[
//                                 [PPGuideDataModel modelWithTargetClassName:NSStringFromClass([self class])
//                                                                 guideIndex:WCSingleLayerGroupViewControllerGuideIndex_Add
//                                                                      title:WCSingleLayerGroupViewControllerString_Tips_Title_AddGroup
//                                                                    message:WCSingleLayerGroupViewControllerString_Tips_AddGroup
//                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Vert],
                                 [PPGuideDataModel  modelWithTargetClassName:NSStringFromClass([self class])
                                                                 guideIndex:WCSingleLayerGroupViewControllerGuideIndex_Reorder
                                                                      title:WCSingleLayerGroupViewControllerString_Tips_Title_Reorder
                                                                    message:WCSingleLayerGroupViewControllerString_Tips_Reorder
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Vert]
                                 ];
    
    return guideDataModels;
}

@end
