//
//  NoteViewController.m
//  
//
//  Created by Howard on 2015/7/18.
//
//

#import "NoteViewController.h"

// Define
#import "WCAppearanceDefine.h"
#import "ContactApperanceDefine.h"
#import "WCContactViewController+ResourceDefine.h"

// View
#import "PPNavigationBarView.h"
#import "PPTextView.h"

// Category
#import "NSString+Additions.h"
#import "UIColor+HexString.h"
#import "PPButton+Factory.h"
#import "UIViewController+ShareApplication.h"

// Controller
#import "WCToastController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Parameter Setting

static UIEdgeInsets const NoteViewControllerNoteViewEdgeInset = {0,10,0,10};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface NoteViewController () <PPButtonDelegate, UITextViewDelegate>

@property (nonatomic,retain) PPButton                   *backButton;
@property (nonatomic,retain) PPButton                   *doneButton;
@property (nonatomic,retain) PPButton                   *dateButton;
@property (nonatomic,retain) PPTextView                 *noteTextView;
@property (nonatomic,retain) PPNavigationBarView        *navigationBarView;
@property (nonatomic,retain) UIView                     *toastSuperView;
@property (nonatomic,assign) CGFloat                    keyboardHeight;
@end

@implementation NoteViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        self.view.backgroundColor = [UIColor whiteColor];
        
        //////////////////////////////////////////////////

        _noteTextString = @"";
        _maxLength = NSUIntegerMax;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.noteViewControllerDelegate = nil;
    self.noteCompleteHandler = nil;
    
    [_backButton setDelegate:nil];
    [_backButton removeControlEvents:UIControlEventTouchUpInside];
    [_backButton removeFromSuperview];
    [_backButton release];
    _backButton = nil;
    
    [_doneButton setDelegate:nil];
    [_doneButton removeControlEvents:UIControlEventTouchUpInside];
    [_doneButton removeFromSuperview];
    [_doneButton release];
    _doneButton = nil;
    
    [_navigationBarView removeFromSuperview];
    _navigationBarView = nil;
    
    [_noteTextString release];
    _noteTextString = nil;
    
    [_titleText release];
    _titleText = nil;
    
    [_dateButton release];
    _dateButton = nil;
    
    [_noteTextView release];
    _noteTextView = nil;
    
    [_toastSuperView release];
    _toastSuperView = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View's Event

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    [self.navigationController.view setBackgroundColor:WCAppearanceDefine_NavigationBarColor];
    //////////////////////////////////////////////////

    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(keyboardWillChangeFrame:)
                                                 name:UIKeyboardWillChangeFrameNotification
                                               object:nil];
    
    //////////////////////////////////////////////////

    PPNavigationBarView *navigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
    
    if(navigationBarView!=nil)
    {
        navigationBarView.style = PPBarViewStyle_CenteredAbsolute;
        
        //////////////////////////////////////////////////

        self.navigationBarView = navigationBarView;
        self.navigationBarView.backgroundColor = NoteViewControllerNavigationBarBackgroundColor;
        
        self.navigationBarView.titleLabel.text = self.titleText;
        
        //////////////////////////////////////////////////
        
        self.backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                          tag:NoteViewControllerButtonTag_Back
                                                     delegate:self
                                        normalBackgroundColor:nil
                                   highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                              imageEdgeInsets:WCContactViewControllerNavigationBarButtonImageEdgeInsets];
        
        if(self.backButton!=nil)
        {
            [self.backButton addControlEvents:UIControlEventTouchUpInside];
            
            //////////////////////////////////////////////////
            
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.backButton];
            
            if(itemModel!=nil)
            {
                itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                
                [self.navigationBarView setItemModels:@[itemModel] forBlockType:PPBarViewBlockType_Left];
            }
        }

        //////////////////////////////////////////////////
        // 可編輯時才要有這兩個按鈕
        if (self.noteEditting==YES)
        {
            self.doneButton = [PPButton ppButtonWithIconImageName:ImageNameWCContactViewControllerButtonDone
                                                              tag:NoteViewControllerButtonTag_Done
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCContactViewControllerNavigationBarButtonImageEdgeInsets];
            
            if(self.doneButton!=nil)
            {
                [self.doneButton addControlEvents:UIControlEventTouchUpInside];
            }
            
            
            //////////////////////////////////////////////////
            
            self.dateButton = [PPButton ppButtonWithIconImageName:ImageNameNoteViewControllerButtonDate
                                                              tag:NoteViewControllerButtonTag_InsertDate
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCContactViewControllerNavigationBarButtonImageEdgeInsets];
            
            if(self.dateButton!=nil)
            {
                [self.dateButton addControlEvents:UIControlEventTouchUpInside];
            }
        }

        //////////////////////////////////////////////////

        if(self.doneButton!=nil &&
           self.dateButton!=nil)
        {
            PPBarViewItemModel *dateItemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.dateButton];
            
            if(dateItemModel!=nil)
            {
                dateItemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                
            }
            
            //////////////////////////////////////////////////

            PPBarViewItemModel *doneItemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.doneButton];
            
            if(doneItemModel!=nil)
            {
                doneItemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                
            }
            
            //////////////////////////////////////////////////

            if(dateItemModel!=nil &&
               doneItemModel!=nil)
            {
                [self.navigationBarView setItemModels:@[dateItemModel,doneItemModel] forBlockType:PPBarViewBlockType_Right];
            }
        }
        
        //////////////////////////////////////////////////
        
        UIBarButtonItem *leftBarButtonItem = [[UIBarButtonItem alloc] initWithCustomView:navigationBarView];
        
        if(leftBarButtonItem!=nil)
        {
            self.navigationItem.leftBarButtonItem = leftBarButtonItem;
        }
        
        [leftBarButtonItem release];
    }
    
    [navigationBarView release];
    
    //////////////////////////////////////////////////

    PPTextView *noteTextView = [[PPTextView alloc] init];
    
    if(noteTextView!=nil)
    {
        noteTextView.userInteractionEnabled = self.noteEditting;
        
        //////////////////////////////////////////////////

        [noteTextView setFont:[UIFont systemFontOfSize:NoteViewControllerTextFontSize]];
        [noteTextView setTextColor:NoteViewControllerTextViewTextColor];
        noteTextView.delegate = self;
        
        //////////////////////////////////////////////////

        [self.view addSubview:noteTextView];
        
        //////////////////////////////////////////////////

        self.noteTextView = noteTextView;
        
        //////////////////////////////////////////////////

        [self addNewLineTokenIfNeed];
        
        //////////////////////////////////////////////////

        if(self.noteTextString!=nil)
        {
            [noteTextView insertText:self.noteTextString];
        }
    }
    
    [noteTextView release];
    
    //////////////////////////////////////////////////

    self.toastSuperView = [[[UIView alloc] init] autorelease];
    
    if(self.toastSuperView!=nil)
    {
        self.toastSuperView.userInteractionEnabled = NO;
        self.toastSuperView.backgroundColor = [UIColor clearColor];
        
        [self.view addSubview:self.toastSuperView];
    }
}


//================================================================================
//
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////

    if(self.noteEditting==YES)
    {
        [self.noteTextView becomeFirstResponder];
        
        __block typeof(self) blockself = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [blockself.noteTextView setSelectedRange:NSMakeRange(0, 0)];
        });
    }
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [super viewWillDisappear:animated];
    
    //////////////////////////////////////////////////

    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardDidChangeFrameNotification object:nil];
    
    //////////////////////////////////////////////////
    
    [self.noteTextView removeFromSuperview];
    self.noteTextView = nil;
    
    [self.doneButton removeFromSuperview];
    self.doneButton = nil;
    
    [self.backButton removeFromSuperview];
    self.backButton = nil;
    
    [self.toastSuperView removeFromSuperview];
    self.toastSuperView = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - layout of subview

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////

    CGRect layoutFrame = [self layoutFrame];
    
    self.noteTextView.frame = CGRectMake(layoutFrame.origin.x+NoteViewControllerNoteViewEdgeInset.left,
                                         layoutFrame.origin.y+NoteViewControllerNoteViewEdgeInset.top,
                                         layoutFrame.size.width-NoteViewControllerNoteViewEdgeInset.left-NoteViewControllerNoteViewEdgeInset.right,
                                         layoutFrame.size.height-NoteViewControllerNoteViewEdgeInset.top-NoteViewControllerNoteViewEdgeInset.bottom-self.keyboardHeight);

    //////////////////////////////////////////////////

    self.toastSuperView.frame = self.noteTextView.frame;
    
    //////////////////////////////////////////////////

    self.noteTextView.inputAccessoryView.frame = CGRectMake(0, 0, self.view.bounds.size.width, self.navigationBarView.bounds.size.height);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification Method

//================================================================================
//
//================================================================================
- (void)keyboardWillChangeFrame:(NSNotification *)notification
{
    NSDictionary* keyboardInfo    = [notification userInfo];
    NSValue *keyboardFrameEnd   = [keyboardInfo valueForKey:UIKeyboardFrameEndUserInfoKey];
    
    CGRect keyboardFrame = [keyboardFrameEnd CGRectValue];
    // !!如果位置在畫面外，就是沒有鍵盤
    if(keyboardFrame.origin.y>=[[UIScreen mainScreen] bounds].size.height)
    {
        self.keyboardHeight = 0;
    }
    else
    {
        self.keyboardHeight = MIN(keyboardFrame.size.width,keyboardFrame.size.height);
    }
    
    //////////////////////////////////////////////////
 
    CGRect layoutFrame = [self layoutFrame];
    
    layoutFrame.size.height-=(NoteViewControllerNoteViewEdgeInset.top+NoteViewControllerNoteViewEdgeInset.bottom+self.keyboardHeight);
    layoutFrame.origin.x += NoteViewControllerNoteViewEdgeInset.left;
    layoutFrame.origin.y += NoteViewControllerNoteViewEdgeInset.right;
    layoutFrame.size.width -=(NoteViewControllerNoteViewEdgeInset.left+NoteViewControllerNoteViewEdgeInset.right);
    
    self.noteTextView.frame = layoutFrame;
    
    self.toastSuperView.frame = layoutFrame;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (void)addNewLineTokenIfNeed
{
    if(self.noteTextString.length>0 && [self.noteTextString hasPrefix:@"\n"]==NO)
    {
        [self.noteTextView insertText:@"\n"];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)shouldChangeTextInRange:(NSRange)range replacementText:(NSString *)text
{
    BOOL shouldChange = YES;
    
    if(self.noteTextView.text.length+text.length>self.maxLength)
    {
        // 新增文字，或取代文字
        if(range.length<=text.length &&
           [text length]>0)
        {
            shouldChange = NO;
        }
    }
    
    return shouldChange;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButton Delegate Method

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    do
    {
        //////////////////////////////////////////////////

        switch (ppButton.tag)
        {
            case NoteViewControllerButtonTag_Back:
            {
                if(self.noteCompleteHandler)
                {
                    self.noteCompleteHandler(self, ppButton, controlEvent);
                }
                else if([self.noteViewControllerDelegate respondsToSelector:@selector(noteViewController:ppButton:controlEvent:)]==YES)
                {
                    [self.noteViewControllerDelegate noteViewController:self ppButton:ppButton controlEvent:controlEvent];
                }
                

                break;
            }
            case NoteViewControllerButtonTag_Done:
            {
                self.noteTextString = self.noteTextView.text;
                
                //////////////////////////////////////////////////

                while([self.noteTextString hasPrefix:@"\n"]==YES)
                {
                    self.noteTextString = [self.noteTextString substringFromIndex:1];
                };
          
                //////////////////////////////////////////////////
                if(self.noteCompleteHandler)
                {
                    self.noteCompleteHandler(self, ppButton, controlEvent);
                }
                else if([self.noteViewControllerDelegate respondsToSelector:@selector(noteViewController:ppButton:controlEvent:)]==YES)
                {
                    [self.noteViewControllerDelegate noteViewController:self ppButton:ppButton controlEvent:controlEvent];
                }
                
                
                break;
            }
            case NoteViewControllerButtonTag_InsertDate:
            {
                NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
                [dateFormatter setDateFormat:@"yyyy/MM/dd"];
                
                NSString *string = [dateFormatter stringFromDate:[NSDate date]];
                [dateFormatter release];

                //////////////////////////////////////////////////

                NSInteger location = self.noteTextView.text.length-1;
                
                if(location<0)
                {
                    location = 0;
                }
                
                //////////////////////////////////////////////////

                if([self shouldChangeTextInRange:NSMakeRange(location,0)
                                 replacementText:string]==YES)
                {
                    [self.noteTextView insertText:string];
                }
                else
                {
                    [WCToastController showMessageToastFromSuperView:self.toastSuperView
                                                         withMessage:[NSString stringWithFormat:WCCV_MLS_ArriveLengthLimit,self.maxLength]
                                                            position:PPToastPositionCenter];
                }
                
                break;
            }
            default:
            {
                break;
            }
        }
    }
    while (0);
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITextViewDelegate

//==============================================================================
//
//==============================================================================
- (BOOL)textView:(UITextView *)textView shouldChangeTextInRange:(NSRange)range replacementText:(NSString *)text
{
    BOOL result = [self shouldChangeTextInRange:range replacementText:text];
    
    if(result==NO)
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            [WCToastController showMessageToastFromSuperView:self.toastSuperView
                                                 withMessage:[NSString stringWithFormat:WCCV_MLS_ArriveLengthLimit,self.maxLength]
                                                    position:PPToastPositionCenter];
        });
    }
    
    return result;
}


@end
