//
//  ContactInfoScrollViewController.m
//  WCMContactInfoViewController
//
//  Created by Andrew on 2015/6/1.
//  Copyright (c) 2015年 Andrew.Lai. All rights reserved.
//

// Model
#import "UIImage+Additions.h"
#import "UIColor+HexString.h"
#import "WCContactViewController+PrivateParameterDefine.h"
#import "WCContactInfoScrollViewController+SettingKey.h"
#import "WCAppearanceDefine.h"

// Controller
#import "WCContactInfoScrollViewController.h"
#import "PPNavigationController.h"
#import "PPLogController.h"
#import "PPSettingsController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define ContactFieldModel_MaxLengthForGenernal 256
#define ContactFieldModel_MaxLengthForNote     NSUIntegerMax
#define DefaultDetailViewBGSize                CGRectMake(0, 0, 340, 340)

static NSUInteger const WCContactInfoScrollViewController_MaxNumberOfContentViewController = 3;

static NSString * const GuideScriptName_ContactInfoNormal        = @"ContactInfoNormalGuide";
static NSString * const GuideScriptName_ContactInfoSwitch        = @"ContactInfoSwitchGuide";
static NSString * const GuideScriptName_ContactInfoLongPressCell = @"ContactInfoLongPressCellGuide";

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCContactInfoScrollViewController Extension

@interface WCContactInfoScrollViewController () <PPGuideControllerDelegate, ContactDataExchangeProtocol, ContactInfoViewControllerDataSource, ContactInfoViewControllerDelegate, UIScrollViewDelegate>
@property (nonatomic, assign) ContactInfoViewControllerInterfaceStyle interfaceStyle;
@property (nonatomic, assign) WCDisplayNameOrder                      displayEastNameOrder;
@property (nonatomic, assign) WCDisplayNameOrder                      displayWestNameOrder;
@property (nonatomic, assign) BOOL                                    isShowDefaultView;
@property (nonatomic, assign) BOOL                                    recoveryContactViewLayout;
@property (nonatomic, assign) NSUInteger                              currentContentContactPageIndex; //紀錄當前顯示聯絡人在content裡的聯絡人索引鍵，判斷是否可繼續左右滑動(左(0),中(1),右(2))
@property (nonatomic, assign) NSInteger                               contactArrayIndex; //紀錄當前顯示聯絡人在contentViewControllerArray裡的索引鍵，提供頁面返回時，回復navigationFrame的位置
@property (nonatomic, assign) UIInterfaceOrientation                  historyInfoViewOrientation; //紀錄旋轉前和切頁面前聯絡人頁面方向，因為於layoutSubView系統已自動變更頁面大小去fit，主要用來重設Info的origin

@property (nonatomic, retain) NSArray                                *contactIdentifierArray;
@property (nonatomic, assign) NSInteger                               currentContactIndex; //紀錄目前是所有聯絡人裡的第幾筆聯絡人

@property (nonatomic, assign) NSInteger                               contentArrayCapacity; //目前畫面上最多暫存聯絡人數量(若人數小於最大顯示數量，則自動改為聯絡人數量)
@property (nonatomic, retain) NSMutableArray                         *contentViewControllerArray;
@property (nonatomic, retain) NSMutableArray                         *contactMainTypeArray; //每一個infoView的MainType應該一致，直接紀錄無需再次要求資料
@property (nonatomic, retain) UIScrollView                           *contactSwitchScrollView;

@property (nonatomic, assign) BOOL                                    firstLayout;

//!! Keyboard相關參數
@property (nonatomic, assign) CGFloat keyboardHeightShowed;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCContactInfoScrollViewController Implementation

@implementation WCContactInfoScrollViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)initWithCurrentContactIndex:(NSInteger)currentContactIndex contactIdentifierArray:(NSArray *)identifierArray interfaceStyle:(ContactInfoViewControllerInterfaceStyle)interfaceStyle
{
    if (self = [super init])
    {
        //////////////////////////////////////////////////
        // Guide所需參數設定
        
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCContactInfoScrollViewController_SettingsKey_GuideNormal];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress];
        
        //////////////////////////////////////////////////
        
        do
        {
            NSUInteger capacitySize = WCContactInfoScrollViewController_MaxNumberOfContentViewController;
            _isShowDefaultView      = NO;
            _keyboardHeightShowed   = 0;
            
            if (identifierArray==nil)
            {
                capacitySize            = 1;
                _isShowDefaultView      = YES;
                
                //////////////////////////////////////////////////
                
                _currentContactIndex    = 0;
                _contactIdentifierArray = nil;
            }
            else
            {
                if (currentContactIndex>=identifierArray.count)
                {
                    PPLogFunction(@"Error Of ContactIndex Or Content Count!");
                    
                    self = nil;
                    break;
                }
                
                //////////////////////////////////////////////////
                
                if (identifierArray.count<capacitySize)
                {
                    capacitySize = identifierArray.count;
                }
                
                //////////////////////////////////////////////////
                
                 _currentContactIndex   = currentContactIndex;
                _contactIdentifierArray = [identifierArray retain];
            }
            
            _contentViewControllerArray = [[NSMutableArray alloc] initWithCapacity:capacitySize];
            if (self.contentViewControllerArray==nil)
            {
                PPLogFunction(@"Error In Initial ContentArray!");
                
                self = nil;
                break;
            }
            
            _displayEastNameOrder       = WCDisplayNameOrder_None;
            _displayWestNameOrder       = WCDisplayNameOrder_None;
            _interfaceStyle             = interfaceStyle;
            _recoveryContactViewLayout  = NO;
            _isBackButtonShowed         = YES;
            _firstLayout                = YES;
            _interfaceIdom              = ContactInfoViewControllerUserInterfaceIdom_Phone;
           
            _contentArrayCapacity       = capacitySize;
            _historyInfoViewOrientation = self.interfaceOrientation;
            
            //////////////////////////////////////////////////
            // 創建滾動頁面裡的內容
            
            for (NSInteger index=0; index<self.contentArrayCapacity; index++)
            {
                ContactInfoViewController *infoViewController = [[ContactInfoViewController alloc] init];
                if (infoViewController!=nil)
                {
                    infoViewController.contactDataExchangeProtocol    = self;
                    infoViewController.dataSource                     = self;
                    infoViewController.delegate                       = self;
                    infoViewController.interfaceStyle                 = self.interfaceStyle;
                    infoViewController.isShowDefaultView              = self.isShowDefaultView;
                    
                    [self.contentViewControllerArray addObject:infoViewController];
                    [infoViewController release];
                }
            }
            
        } while (0);
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.dataExchangeProtocol = nil;
    self.dataSource = nil;
    self.delegate = nil;
    
    self.contactIdentifierArray = nil;
    self.contactMainTypeArray   = nil;
    
    // !! 要先把contactInfoViewController的delegate都清掉，再remove,
    // 不然有機會當在ContactInfoViewController的blockOperation中
    for (ContactInfoViewController *contactInfoViewController in self.contentViewControllerArray)
    {
        contactInfoViewController.contactDataExchangeProtocol = nil;
        contactInfoViewController.contactImageProtocol = nil;
        contactInfoViewController.dataSource = nil;
        contactInfoViewController.delegate = nil;
    }
    
    [self.contentViewControllerArray removeAllObjects];
    self.contentViewControllerArray = nil;

    //////////////////////////////////////////////////
    
    [self unRegisterForKeyboardNotifications];
    
    //////////////////////////////////////////////////
    
    self.contactSwitchScrollView = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Rotate Event

//================================================================================
// 用來紀錄旋轉前的寬度
//================================================================================
- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    [super willRotateToInterfaceOrientation:toInterfaceOrientation duration:duration];
    
    //////////////////////////////////////////////////
    //!! 必須記錄畫面方向，避免外部旋轉後切換回來Layout錯誤
    
    self.historyInfoViewOrientation = self.interfaceOrientation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Keyboard Show/Hide

//================================================================================
//
//================================================================================
- (void)registerForKeyboardNotifications
{
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvUIKeyboardWillShowNotification:)
                                                 name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvUIKeyboardWillHideNotification:)
                                                 name:UIKeyboardWillHideNotification object:nil];
}


//================================================================================
//
//================================================================================
- (void)unRegisterForKeyboardNotifications
{
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillHideNotification object:nil];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - recv Keyboard notifivation

//==============================================================================
//
//==============================================================================
- (void)recvUIKeyboardWillShowNotification:(NSNotification*)notification
{
    NSValue * rectValue = [notification.userInfo objectForKey:UIKeyboardFrameEndUserInfoKey];
    
    if(rectValue!=nil)
    {
        CGRect keyboardRect           = [rectValue CGRectValue];
        CGFloat currentKeyboardHeight = MIN(keyboardRect.size.width,keyboardRect.size.height);
        if (self.keyboardHeightShowed!=currentKeyboardHeight)
        {
            [self recoveryInfoViewsFrame];
            [self resetInfoViewsFrameWithKeyboardHeight:currentKeyboardHeight];
            
            self.keyboardHeightShowed = currentKeyboardHeight;
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvUIKeyboardWillHideNotification:(NSNotification*)notification
{
    [self recoveryInfoViewsFrame];
        
    self.keyboardHeightShowed = 0;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];

    //////////////////////////////////////////////////
    // 設定空資料時的底圖
    
    [self.backgroundImageView setBackgroundColor:WCAppearanceDefine_SectionHeaderBGColor];
    
    if ([[UIApplication sharedApplication] isStatusBarHidden]==NO)
    {
        CGRect viewFrame = self.view.frame;
        CGFloat statusBarHeight = MIN ([[UIApplication sharedApplication] statusBarFrame].size.width,
                                       [[UIApplication sharedApplication] statusBarFrame].size.height);
        self.backgroundImageView.frame = CGRectMake(0, statusBarHeight, viewFrame.size.width, viewFrame.size.height-statusBarHeight);
    }
    
    //////////////////////////////////////////////////
    // 添加滾動頁面
    
    _contactSwitchScrollView = [[UIScrollView alloc] initWithFrame:self.view.bounds];
    if (self.contactSwitchScrollView!=nil)
    {
        self.contactSwitchScrollView.delegate      = self;
        self.contactSwitchScrollView.pagingEnabled = YES;
        self.contactSwitchScrollView.scrollsToTop  = NO;
        self.contactSwitchScrollView.showsHorizontalScrollIndicator = NO;
        self.contactSwitchScrollView.showsVerticalScrollIndicator   = NO;
        
        [self loadInfoViewController];
        [self.view addSubview:self.contactSwitchScrollView];
        
        if (self.isShowDefaultView==YES)
        {
            self.contactSwitchScrollView.alpha = 0.0;
        }
    }
    
    //////////////////////////////////////////////////
    PPLogFunction(@"Frame: %@", NSStringFromCGRect(self.view.frame));
    PPLogFunction(@"Bounds: %@", NSStringFromCGRect(self.view.bounds));
    
    //////////////////////////////////////////////////
//    [self layoutSubviewsToOrientation:self.interfaceOrientation];
}


//================================================================================
//
//================================================================================
- (void)viewWillLayoutSubviews
{
    PPLogFunction(@"%s",__func__);
    
    [super viewWillLayoutSubviews];
    
    
    PPLogFunction(@" ids:%@",self.contactIdentifierArray);
    
    //////////////////////////////////////////////////

    [self layoutSubviewsToOrientation:self.interfaceOrientation];
}

//================================================================================
//
//================================================================================
- (void)layoutSubviewsToOrientation:(UIInterfaceOrientation)orientation
{
    PPLogFunction();
    
    //////////////////////////////////////////////////

    self.contactSwitchScrollView.frame       = self.view.bounds;
    self.contactSwitchScrollView.contentSize = CGSizeMake(self.contentArrayCapacity*self.view.bounds.size.width, self.view.bounds.size.height);
    
    //////////////////////////////////////////////////
    // 根據最大資料數和當前聯絡人索引，計算適當的擺放位置
    
    NSInteger  beginContactIndex;
    NSInteger  endContactIndex;
    NSUInteger initContactPageIndex;
    
    if (self.currentContactIndex==0)
    {
        beginContactIndex = 0;
        endContactIndex   = self.contentArrayCapacity-1;
        initContactPageIndex = 0;
    }
    else if (self.currentContactIndex==self.contactIdentifierArray.count-1)
    {
        beginContactIndex = self.currentContactIndex-self.contentArrayCapacity+1;
        endContactIndex   = self.currentContactIndex;
        initContactPageIndex = (self.contentArrayCapacity-1);
    }
    else
    {
        beginContactIndex = self.currentContactIndex-1;
        endContactIndex   = self.currentContactIndex+1;
        initContactPageIndex = ((self.contentArrayCapacity-1)/2);
    }
    
    self.currentContentContactPageIndex = initContactPageIndex;
    self.contactSwitchScrollView.contentOffset = CGPointMake(initContactPageIndex*self.view.bounds.size.width, 0);
   
    PPLogFunction(@"%s, beginContactIndex:%td, endContactIndex:%td, initContactPageIndex:%td, contactSwitchScrollView.contentOffset:%@",__func__,beginContactIndex,endContactIndex,initContactPageIndex,NSStringFromCGPoint(self.contactSwitchScrollView.contentOffset));
    
    //////////////////////////////////////////////////
    // 要求目前要呈現於頁面上的聯絡人資料
    
    NSString *currentContactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex];
    
    for (NSInteger index=beginContactIndex; index<=endContactIndex; index++)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:index-beginContactIndex];

        //////////////////////////////////////////////////
        
        // 利用userInfoObject記錄contactIdentifier
        //!! 需判斷是否為空值才可設定，避免覆蓋紀錄
         NSString *contactIdentifier = [self.contactIdentifierArray objectAtIndex:index];
      
        if(infoViewController.userInfoObject==nil)
        {
            infoViewController.userInfoObject = contactIdentifier;
        }
        
        //////////////////////////////////////////////////

        PPNavigationController *navigationController  = (PPNavigationController *)infoViewController.navigationController;
        
        //////////////////////////////////////////////////
        
        //!! 旋轉時必須額外重新設定navigationView的Frame Origin Point，因為系統已自動Fit Size(初始0.0，第一次免設定)
        
        // HOWARD:這邊要防止資料重新Load，所以要針對對應的連絡人檢視頁面，給予適當的 origin
        CGRect navigationFrame = navigationController.view.frame;
        
        if(self.contactIdentifierArray.count<=1)
        {
            navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x;
        }
        else if(self.contactIdentifierArray.count<=2)
        {
            if([infoViewController.userInfoObject compare:currentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x;
            }
            else
            {
                navigationFrame.origin.x = (self.contactSwitchScrollView.contentOffset.x>0)?0:self.view.bounds.size.width;
            }
        }
        //點到第一張
        else if(beginContactIndex==self.currentContactIndex)
        {
            NSString *firstCurrentContactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex+1];
                       
            //////////////////////////////////////////////////

            if([infoViewController.userInfoObject compare:currentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x;
            }
            else if([infoViewController.userInfoObject compare:firstCurrentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x + self.view.bounds.size.width;
            }
            else
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x + 2*self.view.bounds.size.width;
            }
        }
        //點到最後一張
        else if(endContactIndex==self.currentContactIndex)
        {
            NSString *firstCurrentContactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex-2];
            
            //////////////////////////////////////////////////

            if([infoViewController.userInfoObject compare:currentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x;
            }
            else if([infoViewController.userInfoObject compare:firstCurrentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x - 2*self.view.bounds.size.width;
            }
            else
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x - self.view.bounds.size.width;
            }
        }
        else
        {
            NSString *beforeCurrentContactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex-1];
            
            if([infoViewController.userInfoObject compare:currentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x;
            }
            else if([infoViewController.userInfoObject compare:beforeCurrentContactIdentifier]==NSOrderedSame)
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x - self.view.bounds.size.width;
            }
            else
            {
                navigationFrame.origin.x = self.contactSwitchScrollView.contentOffset.x + self.view.bounds.size.width;
            }
        }
        
        //////////////////////////////////////////////////
        
        if (navigationFrame.origin.x==self.currentContentContactPageIndex*self.view.bounds.size.width)
        {
            //!! 已於viewWillAppear時，回覆navigationController.view的frame.orignX
            //!! 第一次或回復退出頁面時，指向顯示於畫面上的contacrInfoViewController
            self.contactArrayIndex = index-beginContactIndex;
        }
        
        navigationFrame.size.width = self.view.bounds.size.width;
        navigationFrame.size.height = self.view.bounds.size.height;
        navigationController.view.frame = navigationFrame;
        
        PPLogFunction(@"%s, index:%td, navigationFrame:%@, userInfoObject:%@",__func__,index,NSStringFromCGRect(navigationFrame),infoViewController.userInfoObject);
        
        //////////////////////////////////////////////////
        
        if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
            self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Browse)
        {
            [self showGuideInCurrentPage];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    [self unloadInfoViewControllerWillDisappear];
    
    //////////////////////////////////////////////////

    [super viewWillDisappear:animated];
}


//================================================================================
// 跳轉頁面時，必須removeFromSuperView和釋放內部頁面
//!! 必須記錄畫面方向，避免外部旋轉後切換回來Layout錯誤
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    self.historyInfoViewOrientation = self.interfaceOrientation;
    
    //////////////////////////////////////////////////
    
    [self unloadInfoViewControllerDidDisappear];
    
    [self.contactSwitchScrollView removeFromSuperview];
    self.contactSwitchScrollView.delegate = nil;
    self.contactSwitchScrollView = nil;
    
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}


//================================================================================
//
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    
    if ([PPGuideController isGuideDisplaying]==NO && self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Browse)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
        if (infoViewController!=nil)
        {
            if ([PPGuideController isGuideDisplaying]==NO)
            {
                if ([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch]==1 &&
                    infoViewController.isContactCardSwitchShowed==YES)
                {
                    [self addSwitchGuidesInViewController:infoViewController];
                }
                else if ([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress]==1 &&
                         infoViewController.haveLongPressCell==YES)
                {
                    [self addLongPressGuidesInViewController:infoViewController];
                }
            }
        }
    }
    
    //////////////////////////////////////////////////
    
#ifdef PPLogMacro
    for(ContactInfoViewController *info in self.contentViewControllerArray)
    {
        PPLogFunction(@"%s, id:%@, frame:%@",__func__,info.userInfoObject,NSStringFromCGRect(info.navigationController.view.frame));
    }
#endif
    
}


//================================================================================
//
//================================================================================
- (void)resetInfoViewsFrameWithKeyboardHeight:(CGFloat)keyboardHeight
{
    CGRect defaultViewFrame = self.view.frame;
    defaultViewFrame.size.height-=keyboardHeight;
    self.view.frame = defaultViewFrame;
    
    //////////////////////////////////////////////////
    
    for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
    {
        [infoViewController updateInfoLayoutWithKeyboardHeight:-keyboardHeight];
    }
}


//================================================================================
//
//================================================================================
- (void)recoveryInfoViewsFrame
{
    if (self.keyboardHeightShowed!=0)
    {
        CGRect defaultViewFrame = self.view.frame;
        defaultViewFrame.size.height+=self.keyboardHeightShowed;
        self.view.frame = defaultViewFrame;
        
        //////////////////////////////////////////////////
        
        for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
        {
            [infoViewController updateInfoLayoutWithKeyboardHeight:self.keyboardHeightShowed];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Method

//================================================================================
//
//================================================================================
- (void)setIsBackButtonShowed:(BOOL)isBackButtonShowed
{
    _isBackButtonShowed = isBackButtonShowed;
    
    //////////////////////////////////////////////////
    
    for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
    {
        infoViewController.isBackButtonShowed = isBackButtonShowed;
    }
}


//================================================================================
//
//================================================================================
- (void)setInterfaceIdom:(ContactInfoViewControllerUserInterfaceIdom)interfaceIdom
{
    _interfaceIdom = interfaceIdom;
    
    //////////////////////////////////////////////////
    
    if (interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Pad)
    {
        [self registerForKeyboardNotifications];
    }
    
    //////////////////////////////////////////////////
    
    for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
    {
        infoViewController.interfaceIdom = interfaceIdom;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PrivateMethod

//================================================================================
//
//================================================================================
- (void)addNormalGuidesInViewController:(UIViewController *)currentInfoViewController
{
    [PPGuideController setDelegate:self];
    
    //////////////////////////////////////////////////
    // prepare guide data
    
    __block typeof(self) blockSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        PPGuideFocusModel *iconFocusModel = [PPGuideFocusModel modelWithImageName:ImageNameWCContactInfoViewControllerSwipeGuide
                                                                      infoAreaGap:100.0];
        //////////////////////////////////////////////////
        
        NSMutableArray *guideDataModels = [NSMutableArray array];

        // 7.3.2 之後移除我的最愛，備註tip
//        [guideDataModels addObject:[PPGuideDataModel modelWithTargetClassName:@"ContactInfoViewController"
//                                                                   guideIndex:PPGuideDataModel_GuideIndex_InfoFavoriteButton
//                                                                        title:[@"MLS_FavoriteButtonTitle" localizedFromTable:WCCV]
//                                                                      message:[@"MLS_FavoriteButtonMessage" localizedFromTable:WCCV]
//                                                              layoutDirection:PPGuideDataModel_LayoutDirection_Vert]];
//        [guideDataModels addObject:[PPGuideDataModel modelWithTargetClassName:@"ContactInfoViewController"
//                                                                   guideIndex:PPGuideDataModel_GuideIndex_InfoNoteButton
//                                                                        title:[@"MLS_NoteButtonTitle" localizedFromTable:WCCV]
//                                                                      message:[@"MLS_NoteButtonMessage" localizedFromTable:WCCV]
//                                                              layoutDirection:PPGuideDataModel_LayoutDirection_Vert]];
        [guideDataModels addObject:[PPGuideDataModel modelWithTargetClassName:@"ContactInfoViewController"
                                                                   guideIndex:PPGuideDataModel_GuideIndex_InfoContactCardButton
                                                                        title:[@"MLS_ContactCardTitle" localizedFromTable:WCCV]
                                                                      message:[@"MLS_ContactCardMessage" localizedFromTable:WCCV]
                                                              layoutDirection:PPGuideDataModel_LayoutDirection_Vert]];
        
        if (blockSelf.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone)
        {
            [guideDataModels addObject:[PPGuideDataModel modelWithGuideFocusModel:iconFocusModel
                                                                  targetClassName:@"ContactInfoViewController"
                                                                       guideIndex:PPGuideDataModel_ReservedIndex_NoTarget
                                                                            title:[@"MLS_SwapContactTitle" localizedFromTable:WCCV]
                                                                          message:[@"MLS_SwapContactMessage" localizedFromTable:WCCV]
                                                                  layoutDirection:PPGuideDataModel_LayoutDirection_Vert
                                                            specialStyle:PPGuideDataModel_SpecialStyle_GuideInfoInTargetView]];
        }
    
        [PPGuideController clearAllGuideScripts];
        [PPGuideController addGuideScriptWithName:GuideScriptName_ContactInfoNormal guideDataModels:guideDataModels];
        [PPGuideController manuallyShowGuideWithViewController:currentInfoViewController];
    });
}


//================================================================================
//
//================================================================================
- (void)addSwitchGuidesInViewController:(UIViewController *)currentInfoViewController
{
    [PPGuideController setDelegate:self];
    
    //////////////////////////////////////////////////
    
    NSString *className = NSStringFromClass(currentInfoViewController.class);
    NSArray *guideDataModels = @[
                                 [PPGuideDataModel modelWithTargetClassName:className
                                                                 guideIndex:PPGuideDataModel_GuideIndex_InfoContactCardSwitch
                                                                      title:[@"MLS_SwitchCardTitle" localizedFromTable:WCCV]
                                                                    message:[@"MLS_SwitchCardMessage" localizedFromTable:WCCV]
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Horz]
                                 ];
    
    //////////////////////////////////////////////////
    // manually show guide in viewController
    
    [PPGuideController clearAllGuideScripts];
    [PPGuideController addGuideScriptWithName:GuideScriptName_ContactInfoSwitch guideDataModels:guideDataModels];
    [PPGuideController manuallyShowGuideWithViewController:currentInfoViewController];
}


//================================================================================
//
//================================================================================
- (void)addLongPressGuidesInViewController:(UIViewController *)currentInfoViewController
{
    [PPGuideController setDelegate:self];
    
    //////////////////////////////////////////////////
    
    PPGuideFocusModel *rectFocusModel = [PPGuideFocusModel modelWithGuideFocusShape:PPGuideFocusShape_Rectangle
                                                                    canTouchThrough:NO
                                                                    targetAreaInset:CGSizeZero
                                                                        infoAreaGap:WCAppearanceDefine_RectFocusModeInfoAreaGap];
    
    NSString *className = NSStringFromClass(currentInfoViewController.class);
    NSArray *guideDataModels = @[
                                 [PPGuideDataModel modelWithGuideFocusModel:rectFocusModel
                                                            targetClassName:className
                                                                 guideIndex:PPGuideDataModel_GuideIndex_InfoLongPressCell
                                                                      title:[@"MLS_LongPressTitle" localizedFromTable:WCCV]
                                                                    message:[@"MLS_LongPressMessage" localizedFromTable:WCCV]
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Vert]
                                 ];
    
    //////////////////////////////////////////////////
    // manually show guide in viewController
    
    [PPGuideController clearAllGuideScripts];
    [PPGuideController addGuideScriptWithName:GuideScriptName_ContactInfoLongPressCell guideDataModels:guideDataModels];
    [PPGuideController manuallyShowGuideWithViewController:currentInfoViewController];
}


//================================================================================
//
//================================================================================
- (void)updateContactViewLocationWithHorizontalOffset:(CGFloat)horizontalOffset
{
    CGPoint contentOffset  = self.contactSwitchScrollView.contentOffset;
    CGSize  scrollViewSize = self.contactSwitchScrollView.bounds.size;
    
    for (NSInteger index=0; index<self.contentViewControllerArray.count; index++)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:index];
        PPNavigationController *navigationController  = (PPNavigationController *)infoViewController.navigationController;
        if (navigationController!=nil)
        {
            CGRect frame   = navigationController.view.frame;
            frame.origin.x+= horizontalOffset;
            
            //////////////////////////////////////////////////
            
            if (frame.origin.x<0)
            {
                PPLogFunction(@"右移更新資料");
                
                frame.origin.x = (self.contentViewControllerArray.count-1)*scrollViewSize.width;
                
                //////////////////////////////////////////////////
                // 要求外部提供新聯絡人資訊(利用userInfoObject記錄contactIdentifier)
                
                NSString *contactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex+1];
                infoViewController.showFrontCard  = YES;
                infoViewController.userInfoObject = contactIdentifier;
                
                //!! 將更換聯絡人資訊，需清除所有資料
                [infoViewController clearImageForAllPPContactMainType];
                [infoViewController clearFieldDictionaryForAllAllPPContactMainType];
                
                //!! 更新時抓取新的影像資料(contactDataExchangeProtocol)
                [infoViewController prepareContactFieldDictionary];
                [infoViewController fetchContactFieldDictionaryWithReload:YES loadDictionaryCompleted:nil];
                [infoViewController fetchContactHeadImagesData];
                [infoViewController fetchContactCardImagesData];
            }
            else if (frame.origin.x>(self.contentViewControllerArray.count-1)*scrollViewSize.width)
            {
                PPLogFunction(@"左移更新資料");
                
                frame.origin.x = 0;
                
                //////////////////////////////////////////////////
                // 要求外部提供新聯絡人 userInfoObject
                
                NSString *contactIdentifier = [self.contactIdentifierArray objectAtIndex:self.currentContactIndex-1];
                infoViewController.showFrontCard  = YES;
                infoViewController.userInfoObject = contactIdentifier;
                
                //!! 將更換聯絡人資訊，需清除所有影像資料
                [infoViewController clearImageForAllPPContactMainType];
                [infoViewController clearFieldDictionaryForAllAllPPContactMainType];
                
                //!! 更新時抓取新的影像資料(contactDataExchangeProtocol)
                [infoViewController prepareContactFieldDictionary];
                [infoViewController fetchContactFieldDictionaryWithReload:YES loadDictionaryCompleted:nil];
                [infoViewController fetchContactHeadImagesData];
                [infoViewController fetchContactCardImagesData];
            }
            
            navigationController.view.frame = frame;
        }
    }
    
    //////////////////////////////////////////////////
    
    if (horizontalOffset!=0)
    {
        if (horizontalOffset>0)
        {
            self.currentContentContactPageIndex = self.currentContentContactPageIndex+1;
        }
        else
        {
            self.currentContentContactPageIndex = self.currentContentContactPageIndex-1;
        }
    }

    self.contactSwitchScrollView.contentOffset = CGPointMake(contentOffset.x+horizontalOffset, contentOffset.y);
}


//==============================================================================
//
//==============================================================================
- (void)loadInfoViewController
{
    for (NSInteger index=0; index<self.contentArrayCapacity; index++)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:index];
        PPNavigationController *navigationController  = [[PPNavigationController alloc] initWithRootViewController:infoViewController];
    
        if (navigationController!=nil)
        {
            if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
                UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
            {
                [navigationController.navigationBar setTranslucent:NO];
            }
            else
            {
                [navigationController.navigationBar setTranslucent:YES];
            }
            
            
            CGRect navigationFrame = CGRectZero;
            if (self.firstLayout==YES)
            {
                navigationFrame = CGRectMake(self.view.bounds.size.width*index, 0, self.view.bounds.size.width, self.view.bounds.size.height);
            }
            else
            {
                //!! 還原離開時，當下所有的navigationFrame的位置．
                //!! 公式如下：畫面的起始點Ｘ(initPointX) ＝ 當前聯絡人畫面的索引(currentContentContactPageIndex) * 畫面寬度(width)
                //!! 陣列內每個navigationFrame起始點Ｘ(originPointX) ＝ 畫面的起始點Ｘ(initPointX) ＋ {陣列索引(index) - 當前畫面於InfoViewArray的索引紀錄(contactArrayIndex)} * 畫面寬度(width)
                //!! 異常例外處理：(1) if(originPointX<0) -> 總內容寬度(contentSize) + originPointX
                //!!             (2) if(originPointX>=contentSize) -> originPointX - 總內容寬度(contentSize)
                CGFloat contentSize  = self.contentArrayCapacity*self.view.bounds.size.width;
                CGFloat initPointX = self.view.bounds.size.width*self.currentContentContactPageIndex;
                
                CGFloat originPointX = initPointX+((index-self.contactArrayIndex)*self.view.bounds.size.width);
                if (originPointX<0)
                {
                    originPointX = contentSize+originPointX;
                }
                else if (originPointX>=contentSize)
                {
                    originPointX = originPointX-contentSize;
                }
                
                navigationFrame = CGRectMake(originPointX, 0, self.view.bounds.size.width, self.view.bounds.size.height);
            }
            
            navigationController.view.frame = navigationFrame;
            
            [navigationController willMoveToParentViewController:self];
            [self addChildViewController:navigationController];
            [self.contactSwitchScrollView addSubview:navigationController.view];
            
            [navigationController didMoveToParentViewController:self];
        }
        
        [navigationController release];
    }
    
    self.firstLayout = NO;
}



//==============================================================================
//
//==============================================================================
- (void)unloadInfoViewControllerWillDisappear
{
    //////////////////////////////////////////////////
    //!! 避免無自動呼叫viewDidDisappear，需使用beginAppearanceTransition:animated:和endAppearanceTransition包起來
    
    for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
    {
        [infoViewController.navigationController beginAppearanceTransition:NO animated:YES];
        
        //!! must call the willMoveToParentViewController: method of the child view controller before calling
        //   the removeFromParentViewController method, passing in a parent value of nil.
        [infoViewController.navigationController willMoveToParentViewController:nil];
    }
}


//================================================================================
//
//================================================================================
- (void)unloadInfoViewControllerDidDisappear
{
    //!! 避免無自動呼叫viewDidDisappear，需使用beginAppearanceTransition:animated:和endAppearanceTransition包起來
    
    for (ContactInfoViewController *infoViewController in self.contentViewControllerArray)
    {
        [infoViewController.navigationController.view removeFromSuperview];
        [infoViewController.navigationController removeFromParentViewController];
        [infoViewController.navigationController didMoveToParentViewController:nil];
        
        [infoViewController.navigationController endAppearanceTransition];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (BOOL)isShowFrontCardInCurrentContactPage
{
    BOOL isShowFront = NO;
    
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    if (infoViewController!=nil)
    {
        isShowFront = infoViewController.showFrontCard;
    }
    
    return isShowFront;
}


//================================================================================
//
//================================================================================
- (void)reloadContactInformationInCurrentPageWithCardID:(NSString *)cardID Completed:(void (^)(void))completed
{
    @synchronized(self)
    {
        if (cardID==nil)
        {
            self.isShowDefaultView = YES;
            self.contactSwitchScrollView.alpha = 0.0;
        }
        else
        {
            self.isShowDefaultView = NO;
            self.contactSwitchScrollView.alpha = 1.0;
        }
        
        //////////////////////////////////////////////////
        
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
        if (infoViewController!=nil)
        {
            infoViewController.userInfoObject = cardID;
            
            [infoViewController recoveryInterfaceInitialSetting];
            
            //!! 將更換聯絡人資訊，需清除所有影像資料
            [infoViewController clearImageForAllPPContactMainType];
            [infoViewController clearFieldDictionaryForAllAllPPContactMainType];
            
            [infoViewController prepareContactFieldDictionary];
            [infoViewController fetchContactFieldDictionaryWithReload:YES loadDictionaryCompleted:^{
                
                if([PPGuideController isGuideDisplaying]==YES)
                {
                    // 強制關閉所以不收delegate
                    [PPGuideController setDelegate:nil];
                    [PPGuideController finishCurrentGuideScript];
                }
                [self showGuideInCurrentPage];
                if (completed!=nil)
                {
                    completed();
                }
            }];
            
            [infoViewController fetchContactHeadImagesData];
            [infoViewController fetchContactCardImagesData];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)showGuideInCurrentPage
{
#if !defined (PRODUCTLINE_WCSF)
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    if (infoViewController!=nil && [PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideNormal]==1)
    {
        if ([PPGuideController isGuideDisplaying]==NO)
        {
            [self addNormalGuidesInViewController:infoViewController];
        }
    }
#endif
}


//================================================================================
//
//================================================================================
- (void)clearContactInformationInCurrentPage
{
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    if (infoViewController!=nil)
    {
        //!! 將更換聯絡人資訊，需清除所有影像資料
        [infoViewController clearImageForAllPPContactMainType];
        [infoViewController clearFieldDictionaryForAllAllPPContactMainType];
    }
}



//================================================================================
//
//================================================================================
- (NSString *)contactIDInCurrentPage
{
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    return (NSString *)infoViewController.userInfoObject;
}

//================================================================================
//
//================================================================================
- (UIImage *)contactBackCardImageInCurrentPage
{
    UIImage *backCardImage = nil;
    
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    if (infoViewController!=nil)
    {
        backCardImage = [infoViewController imageForPPContactMainType:PPContactMainType_BackCardImage ppContactSubType:PPContactSubType_BackCardImage_Thumbnail];
    }
    
    return backCardImage;
}


//================================================================================
//
//================================================================================
- (UIImage *)contactFrontCardImageInCurrentPage
{
    UIImage *frontCardImage = nil;
    
    ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    if (infoViewController!=nil)
    {
        frontCardImage = [infoViewController imageForPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail];
    }
    
    return frontCardImage;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIScrollViewDelegate

//================================================================================
//
//================================================================================
- (void)scrollViewDidScroll:(UIScrollView *)scrollView
{
    PPLogFunction(@"Offset: %@", NSStringFromCGPoint(scrollView.contentOffset));
    
    //////////////////////////////////////////////////
    // 是否隱藏NavigationBar上的按鈕
    
    NSInteger scrollViewWidth   = scrollView.bounds.size.width;
    NSInteger scrollViewOffsetX = scrollView.contentOffset.x;
    CGFloat singleOffsetX = scrollViewOffsetX%scrollViewWidth;
    
    for (NSInteger index=0; index<self.contentViewControllerArray.count; index++)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:index];
        [infoViewController contactInfoNavigationBarButtonsAlphaWithOffset:singleOffsetX];
        
        if (singleOffsetX==0 && self.recoveryContactViewLayout==YES)
        {
            [infoViewController recoveryInterfaceInitialSetting];
        }
    }
    
    //////////////////////////////////////////////////
    //!! fix mantis 0016975: 先往左滑到第2張名片->然後再往左滑一段距離，但是不放開-> 一秒鐘之後再放開，
    // 會看到名片跳過下一張。此時再往左滑不放開，會看到右邊沒有顯示聯絡人，但實際上後面還有很多聯絡人，參考圖片
    // 原本在手指放開時進行設定，會導致和系統的打架，故必須在此更新與設定
    
    if (self.recoveryContactViewLayout==YES)
    {
        if (self.currentContentContactPageIndex==0 && self.currentContactIndex!=0)
        {
            [self updateContactViewLocationWithHorizontalOffset:scrollView.bounds.size.width];
        }
        else if (self.currentContentContactPageIndex==self.contentArrayCapacity-1 && self.currentContactIndex!=self.contactIdentifierArray.count-1)
        {
            [self updateContactViewLocationWithHorizontalOffset:-scrollView.bounds.size.width];
        }
        
        //////////////////////////////////////////////////
        
        if (self.contactArrayIndex<0)
        {
            self.contactArrayIndex = self.contentArrayCapacity-1;
        }
        else if (self.contactArrayIndex>self.contentArrayCapacity-1)
        {
            self.contactArrayIndex = 0;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewWillEndDragging:(UIScrollView *)scrollView withVelocity:(CGPoint)velocity targetContentOffset:(inout CGPoint *)targetContentOffset
{
    PPLogFunction(@"scrollViewContentOffset: %@", NSStringFromCGPoint(scrollView.contentOffset));
    
    CGPoint finalContentOffset = *targetContentOffset;
    CGFloat currentContentOffsetX = self.currentContentContactPageIndex*scrollView.bounds.size.width;
    
    if (finalContentOffset.x-currentContentOffsetX<=-scrollView.bounds.size.width)
    {
        self.recoveryContactViewLayout      = YES;
        self.currentContactIndex            = self.currentContactIndex-1;
        self.currentContentContactPageIndex = self.currentContentContactPageIndex-1;
        self.contactArrayIndex = self.contactArrayIndex-1;
    }
    else if (finalContentOffset.x-currentContentOffsetX>=scrollView.bounds.size.width)
    {
        self.recoveryContactViewLayout = YES;
        self.currentContactIndex   = self.currentContactIndex+1;
        self.currentContentContactPageIndex = self.currentContentContactPageIndex+1;
        self.contactArrayIndex = self.contactArrayIndex+1;
    }
    else
    {
        self.recoveryContactViewLayout = NO;
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDecelerating:(UIScrollView *)scrollView
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    //!! 為避免每次滾動皆需掃迴圈，故先判斷設定內的紀錄
    
    if (([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch]==1 ||
        [PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress]==1) &&
        self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Browse)
    {
        ContactInfoViewController *infoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
        if (infoViewController!=nil)
        {
            if ([PPGuideController isGuideDisplaying]==NO)
            {
                if (infoViewController.isContactCardSwitchShowed==YES &&
                    [PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch]==1)
                {
                    [self addSwitchGuidesInViewController:infoViewController];
                }
                else if (infoViewController.haveLongPressCell==YES &&
                         [PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress]==1)
                {
                    [self addLongPressGuidesInViewController:infoViewController];
                }
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactDataExchangeProtocol

//================================================================================
// 檢視無需實作儲存影像Protocol
//================================================================================
- (BOOL)sender:(id)sender setImage:(UIImage *)image forPPContactMainType:(PPContactMainType)contactMainType error:(NSError **)error
{
    return NO;
}


//================================================================================
//
//================================================================================
- (UIImage *)sender:(id)sender imageForPPContactMainType:(PPContactMainType)contactMainType ppContactSubType:(PPContactSubType)contactSubType
{
    UIImage *image = nil;
    
    do
    {
        if(sender==nil || [sender isKindOfClass:[WCContactViewController class]]==NO)
        {
            break;
        }
        
        WCContactViewController *contactViewController = (WCContactViewController *)sender;
        
        //////////////////////////////////////////////////
        
        if(contactViewController.userInfoObject==nil || [contactViewController.userInfoObject isKindOfClass:[NSString class]]==NO)
        {
            break;
        }
        
        NSString *contactIdentifier = (NSString *)contactViewController.userInfoObject;
        
        if (self.dataExchangeProtocol!=nil && [self.dataExchangeProtocol respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:imageForPPContactMainType:ppContactSubType:)]==YES)
        {
            image = [self.dataExchangeProtocol contactInfoScrollViewController:self contactIdentifier:contactIdentifier imageForPPContactMainType:contactMainType ppContactSubType:contactSubType];
        }
    } while (0);
    
    return image;
}


//================================================================================
//
//================================================================================
- (NSDictionary *)sender:(id)object fieldDictionaryforPPContactMainTypeArray:(NSArray *)contactMainTypeArray
{
    NSDictionary *fieldDictionary = nil;
    
    do
    {
        if (object==nil || [object isKindOfClass:[WCContactViewController class]]==NO)
        {
            break;
        }
        
        WCContactViewController *contactViewController = (WCContactViewController *)object;
        
        //////////////////////////////////////////////////
        
        if (contactViewController.userInfoObject==nil || [contactViewController.userInfoObject isKindOfClass:[NSString class]]==NO)
        {
            break;
        }
        
        NSString *contactIdentifier = (NSString *)contactViewController.userInfoObject;
        
        if (self.dataExchangeProtocol!=nil && [self.dataExchangeProtocol respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:fieldDictionaryforPPContactMainTypeArray:)]==YES)
        {
            fieldDictionary = [self.dataExchangeProtocol contactInfoScrollViewController:self contactIdentifier:contactIdentifier fieldDictionaryforPPContactMainTypeArray:contactMainTypeArray];
        }
    }
    while (0);
    
    return fieldDictionary;
}


//================================================================================
// 檢視無需實作儲存資料Protocol
//================================================================================
- (BOOL)sender:(id)object setFieldsDictionary:(NSDictionary *)fieldDictionary error:(NSError **)error
{
    return NO;
}


//================================================================================
//
//================================================================================
- (BOOL)favoriteWithSender:(id)object
{
    BOOL isFavorite = NO;
    
    do
    {
        if (object==nil || [object isKindOfClass:[WCContactViewController class]]==NO)
        {
            break;
        }
        
        WCContactViewController *contactViewController = (WCContactViewController *)object;
        
        //////////////////////////////////////////////////
        
        if (contactViewController.userInfoObject==nil || [contactViewController.userInfoObject isKindOfClass:[NSString class]]==NO)
        {
            break;
        }
        
        NSString *contactIdentifier = (NSString *)contactViewController.userInfoObject;
        
        if (self.dataExchangeProtocol!=nil && [self.dataExchangeProtocol respondsToSelector:@selector(contactInfoScrollViewController:favoriteWithcontactIdentifier:)]==YES)
        {
            isFavorite = [self.dataExchangeProtocol contactInfoScrollViewController:self favoriteWithcontactIdentifier:contactIdentifier];
        }
    }
    while (0);
    
    return isFavorite;
}


//================================================================================
//
//================================================================================
- (BOOL)sender:(id)sender setFavorite:(BOOL)favorite error:(NSError **)error
{
    return NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoViewControllerDataSource

//================================================================================
//
//================================================================================
- (BOOL)shouldShowPhoneticOnContactInfoViewController:(ContactInfoViewController *)contactInfoViewController
{
    BOOL showPhonetic = NO;
    
    if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(shouldShowPhoneticOnContactInfoScrollViewController:)]==YES)
    {
        showPhonetic = [self.dataSource shouldShowPhoneticOnContactInfoScrollViewController:self];
    }
    
    return showPhonetic;
}


//================================================================================
//
//================================================================================
- (WCDisplayNameOrder)contactNameEastOrderInContactInfoViewController:(ContactInfoViewController *)contactInfoViewController
{
    if (self.displayEastNameOrder==WCDisplayNameOrder_None)
    {
        if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(contactNameEastOrderInContactInfoScrollViewController:)]==YES)
        {
            self.displayEastNameOrder = [self.dataSource contactNameEastOrderInContactInfoScrollViewController:self];
        }
    }
    
    return self.displayEastNameOrder;
}


//================================================================================
//
//================================================================================
- (WCDisplayNameOrder)contactNameWestOrderInContactInfoViewController:(ContactInfoViewController *)contactInfoViewController
{
    if (self.displayWestNameOrder==WCDisplayNameOrder_None)
    {
        if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(contactNameWestOrderInContactInfoScrollViewController:)]==YES)
        {
            self.displayWestNameOrder = [self.dataSource contactNameWestOrderInContactInfoScrollViewController:self];
        }
    }
    
    return self.displayWestNameOrder;
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)ppContactMainTypeArrayFetchingDataForContactInfoViewController:(ContactInfoViewController *)contactInfoViewController
{
    // !! salesforce每一筆的main type可能不一樣
#if !defined (PRODUCTLINE_WCSF)
    if (self.contactMainTypeArray==nil)
#endif
    {
        if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(ppContactMainTypeArrayFetchingDataForContactInfoScrollViewController:contactIdentifier:)]==YES)
        {
            NSString *contactIdentifier = (NSString *)contactInfoViewController.userInfoObject;
            
            self.contactMainTypeArray = [self.dataSource ppContactMainTypeArrayFetchingDataForContactInfoScrollViewController:self contactIdentifier:contactIdentifier];
        }
    }

    return self.contactMainTypeArray;
}


//================================================================================
//
//================================================================================
- (NSUInteger)contactInfoTableViewCellTextMaxLengthInContactInfoViewController:(ContactInfoViewController *)contactInfoViewController ppContactMainType:(PPContactMainType)contactMainType
{
    NSUInteger maxLength = ContactFieldModel_MaxLengthForGenernal;
    
    //////////////////////////////////////////////////
    
    switch(contactMainType)
    {
        case PPContactMainType_PublicNote:
        case PPContactMainType_Note:
        {
            maxLength = ContactFieldModel_MaxLengthForNote;
            
            break;
        }
        case PPContactMainType_Name:
        case PPContactMainType_Company:
        case PPContactMainType_Address:
        case PPContactMainType_Phone:
        case PPContactMainType_Email:
        case PPContactMainType_URL:
        case PPContactMainType_InstantMessage:
        case PPContactMainType_Date:
        case PPContactMainType_UserDefine:
        case PPContactMainType_BirthDay:
        case PPContactMainType_Group:
        case PPContactMainType_Other:
        default:
        {
            break;
        }
    }
    
    //////////////////////////////////////////////////
    
    return maxLength;
}


//================================================================================
//
//================================================================================
- (NSArray *)translateLanguageArrayInContactInfoViewController:(ContactInfoViewController *)contactInfoViewController translateLocation:(ContactInfoViewControllerTranslateLocation)translateLocation
{
    NSArray *translateLanguageArray = nil;
    
    if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(translateLanguageArrayInContactInfoScrollViewController:translateLocation:)]==YES)
    {
        translateLanguageArray = [self.dataSource translateLanguageArrayInContactInfoScrollViewController:self translateLocation:translateLocation];
    }
    
    return translateLanguageArray;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoViewControllerDelegate

//================================================================================
//
//================================================================================
- (void)contactInfoViewController:(ContactInfoViewController *)contactInfoViewController responderView:(UIView *)responderView action:(ContactInfoViewControllerAction)action
{
    if (action==ContactInfoViewControllerAction_Back)
    {
        PPLogFunction(@"goBackAnimated!");
        
        [self goBackAnimated:YES];
    }
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:action:)]==YES)
    {
        [self.delegate contactInfoScrollViewController:self contactIdentifier:contactInfoViewController.userInfoObject action:action];
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoViewController:(ContactInfoViewController *)contactInfoViewController didSelectValueDictionary:(NSDictionary *)valueDictionary
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:didSelectValueDictionary:)]==YES)
    {
        [self.delegate contactInfoScrollViewController:self contactIdentifier:contactInfoViewController.userInfoObject didSelectValueDictionary:valueDictionary];
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoViewController:(ContactInfoViewController *)contactInfoViewController didSelectPhoneNumber:(NSString *)phoneNumber
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:didSelectPhoneNumber:)]==YES)
    {
        [self.delegate contactInfoScrollViewController:self contactIdentifier:contactInfoViewController.userInfoObject didSelectPhoneNumber:phoneNumber];
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoViewController:(ContactInfoViewController *)contactInfoViewController setFieldIndex:(NSUInteger)fieldIndex toDefaultWithMainType:(PPContactMainType)contactMainType
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:setFieldIndex:toDefaultWithMainType:)]==YES)
    {
        [self.delegate contactInfoScrollViewController:self contactIdentifier:contactInfoViewController.userInfoObject setFieldIndex:fieldIndex toDefaultWithMainType:contactMainType];
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoViewController:(ContactInfoViewController *)contactInfoViewController translateFromLanguage:(NSString *)fromLanguage toLanguage:(NSString *)toLanguage
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:contactIdentifier:translateFromLanguage:toLanguage:)]==YES)
    {
        [self.delegate contactInfoScrollViewController:self contactIdentifier:contactInfoViewController.userInfoObject translateFromLanguage:fromLanguage toLanguage:toLanguage];
    }
}


//================================================================================
//
//================================================================================
- (UIImage *)defaultCardImageInContactInfoViewController:(ContactInfoViewController *)contactInfoViewController
{
    UIImage *defaultCardImage = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoScrollViewController:defaultCardImageForContactIdentifier:)]==YES)
    {
        defaultCardImage = [self.delegate contactInfoScrollViewController:self defaultCardImageForContactIdentifier:contactInfoViewController.userInfoObject];
    }
    
    return defaultCardImage;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPGuideControllerDelegate

//================================================================================
//
//================================================================================
- (void)guideControllerDidFinishScript:(NSString *)guideScriptName
{
    //////////////////////////////////////////////////
    // 找出當前畫面上的Info介面
    
    ContactInfoViewController *currentInfoViewController = [self.contentViewControllerArray objectAtIndex:self.contactArrayIndex];
    
    //////////////////////////////////////////////////
    // 判斷GuideController結束哪一個Script
    
    if ([guideScriptName isEqualToString:GuideScriptName_ContactInfoNormal])
    {
        if ([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch]==1 &&
            currentInfoViewController.isContactCardSwitchShowed==YES)
        {
            [self addSwitchGuidesInViewController:currentInfoViewController];
        }
        else if ([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress]==1 &&
                 currentInfoViewController.haveLongPressCell==YES)
        {
            [self addLongPressGuidesInViewController:currentInfoViewController];
        }
        
        [PPSettingsController setIntegerValue:0 withKey:WCContactInfoScrollViewController_SettingsKey_GuideNormal];
    }
    
    //////////////////////////////////////////////////
    
    if ([guideScriptName isEqualToString:GuideScriptName_ContactInfoSwitch])
    {
        if ([PPSettingsController integerValueWithKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress]==1 &&
            currentInfoViewController.haveLongPressCell==YES)
        {
            [self addLongPressGuidesInViewController:currentInfoViewController];
        }
        
        [PPSettingsController setIntegerValue:0 withKey:WCContactInfoScrollViewController_SettingsKey_GuideSwitch];
    }
    
    //////////////////////////////////////////////////
    
    if ([guideScriptName isEqualToString:GuideScriptName_ContactInfoLongPressCell])
    {
        [PPSettingsController setIntegerValue:0 withKey:WCContactInfoScrollViewController_SettingsKey_GuideLongPress];
    }
}

@end
