//
//  WCCardImageController.m
//  WorldCardHD
//
//  Created by  Eddie on 2011/1/28.
//  Copyright 2011 Penpower. All rights reserved.
//

//#import "DumpLog.h"

#import "WCCardImageController.h"
#import "NSError+Custom.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
#if TARGET_OS_IPHONE

#import "UIImage+Additions.h"

#elif TARGET_OS_MAC

#import "NSImage+Additions.h"

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
#define WCCIC_IsSettingImage    @"WCCIC_IsSettingImage"


static inline double radians (double degrees) {return degrees * M_PI/180;}


@implementation WCCardImageController


#pragma mark - Life cycle methods

//===============================================================================
//
//===============================================================================
- (id)initWithDirPath:(NSString *)dirPath
{
    self = [super init];
    
    if (self)
    {
        baseDir_ = [dirPath retain];
        fileManager_ = [NSFileManager defaultManager];
        
        if(![fileManager_ fileExistsAtPath:baseDir_])
            [fileManager_ createDirectoryAtPath:baseDir_ withIntermediateDirectories:YES attributes:nil error:nil];
    }
    return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
    [baseDir_ release];
    [super dealloc];
}





#pragma mark - private methods

//===============================================================================
//
//===============================================================================
- (NSString *)tempPath:(NSString *)path
{
    return [path stringByAppendingString:@"~"];
}
//===============================================================================
//
//===============================================================================
- (BOOL)saveImage:(CPImage *)image compression:(CGFloat)compression toPath:(NSString *)path error:(NSError **)error
{
    if(!image)
        return NO;
    
    NSError *returnError = nil;
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    NSData *imageData = [image jpegRepresentationWithCompressQuality:compression];
    BOOL bResult = [imageData writeToFile:path options:NSDataWritingAtomic error:&returnError];
    [returnError retain];
    [pool release];
    
    if (returnError)
    {
        [returnError autorelease];
        if (error)
        {
            *error = returnError;
        }
    }
    return bResult;
}

//===============================================================================
//
//===============================================================================
- (BOOL)saveImage:(CPImage *)image toPath:(NSString *)path error:(NSError **)error
{
    return [self saveImage:image compression:WC_JPG_COMPRESS toPath:path error:error];
}


//===============================================================================
//
//===============================================================================
- (BOOL)saveImageWithoutCompress:(CPImage *)image toPath:(NSString *)path error:(NSError **)error
{
    return [self saveImage:image compression:WC_JPG_SilentRecognCompress toPath:path error:error];
}


//===============================================================================
//
//===============================================================================
- (BOOL)saveImageData:(NSData *)imageData toPath:(NSString *)path error:(NSError **)error
{
    if(imageData == nil)
    {
        if(error!=nil)
        {
            *error = PPErrorMake(NSErrorCustom_Code_ParameterInvalidity, @"imageData is empty", nil);
        }
        
        return NO;
    }
    
    return [imageData writeToFile:path options:NSDataWritingAtomic error:error];
}







#pragma mark - Instance methods

//===============================================================================
//
//===============================================================================
- (NSString *)imagePathWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    NSString *typeSuffix = @"";
    NSString *subtypeSuffix = @"";
    
    switch (type)
    {
        case WC_IT_IDPhoto:
            typeSuffix = WC_IFS_IDPhoto;
            subtype = WC_IST_Thumbnail; // 大頭照都是用Thumbnail處理
            break;
            
        case WC_IT_FrontSide:
            typeSuffix = WC_IFS_FrontSide;
            break;
            
        case WC_IT_BackSide:
            typeSuffix = WC_IFS_BackSide;
            break;
            
        default:
            return nil;
    }
    
    
    switch (subtype)
    {
        case WC_IST_ReRecogn:
            subtypeSuffix = WC_IFS_ReRecogn;
            break;
            
        case WC_IST_Original:
            subtypeSuffix = WC_IFS_Original;
            break;
            
        case WC_IST_Browse:
            subtypeSuffix = WC_IFS_Browse;
            break;
            
        case WC_IST_Thumbnail:
            subtypeSuffix = WC_IFS_Thumbnail;
            break;
            
        default:
            return nil;
    }
    
    
    NSString *imagePath = [NSString stringWithFormat:@"%@/%@_%@%@.jpg", baseDir_, cardID, typeSuffix, subtypeSuffix];
    
    return imagePath;
}


//===============================================================================
// check if image exist
//===============================================================================
- (BOOL)hasImageWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    if(![cardID length])
        return NO;
    
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
    
    return [fileManager_ fileExistsAtPath:imagePath];
}


//===============================================================================
//
//===============================================================================
- (CPImage *)copyImageWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s", __func__];
#endif
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, check and wait setting image finish
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    NSString *settingCardID = nil;
    
    for(int i=0; i<10; i++)
    {
        settingCardID = [threadDict objectForKey:WCCIC_IsSettingImage];
        
        if(settingCardID)
        {
            if([settingCardID isEqualToString:cardID])
                [NSThread sleepForTimeInterval:0.1];
            else break;
        }
        else break;
    }
    
    
    //-----------------------------------------------------------
    // get image
    //-----------------------------------------------------------
    // !! IDPhoto只有thumbnail
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;
    
    CPImage *image = nil;
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
    
    if([imagePath length])
    {
        CPImage *tempImage = [[CPImage alloc] initWithContentsOfFile:imagePath];
        
        if(tempImage)
        {
            // !! 如果直接回傳，當圖片檔被刪除時，後繼圖片的操作可能會異常，所以再產生一份出去
            // !! 要產生暫存請用png，用jpg會讓畫質減損。
            NSData *tempData = [tempImage pngRepresentation];
            image = [[CPImage alloc] initWithData:tempData];
        }
        [tempImage release];
    }
    
    return image;
}


//===============================================================================
//
//===============================================================================
- (NSData *)copyImageDataWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    //-----------------------------------------------------------
    // !! for multi-thread, check and wait setting image finish
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    NSString *settingCardID = nil;
    
    for(int i=0; i<10; i++)
    {
        settingCardID = [threadDict objectForKey:WCCIC_IsSettingImage];
        
        if(settingCardID)
        {
            if([settingCardID isEqualToString:cardID])
                [NSThread sleepForTimeInterval:0.1];
            else break;
        }
        else break;
    }
    
    
    
    //-----------------------------------------------------------
    // get image data
    //-----------------------------------------------------------
    // !! IDPhoto只有thumbnail
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;
    
    NSData *imageData = nil;
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
    
    if([imagePath length])
        imageData = [[NSData alloc] initWithContentsOfFile:imagePath];
    
    return imageData;
}



//========================
//wcm 6.0.0 新增儲存重新辨識用的名片圖 get rerecogn image
//========================
- (CPImage *)copyRerecognImageWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s", __func__];
#endif
    WC_ImageSubType subtype=WC_IST_ReRecogn;
    
    //-----------------------------------------------------------
    // !! for multi-thread, check and wait setting image finish
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    NSString *settingCardID = nil;
    
    for(int i=0; i<10; i++)
    {
        settingCardID = [threadDict objectForKey:WCCIC_IsSettingImage];
        
        if(settingCardID)
        {
            if([settingCardID isEqualToString:cardID])
                [NSThread sleepForTimeInterval:0.1];
            else break;
        }
        else break;
    }
    
    
    //-----------------------------------------------------------
    // get image
    //-----------------------------------------------------------
    // !! IDPhoto只有thumbnail
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;
    
    CPImage *image = nil;
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
    
    if([imagePath length])
        image = [[CPImage alloc] initWithContentsOfFile:imagePath];
    else
    {
        subtype =WC_IST_Original;
        imagePath=[self imagePathWithCardID:cardID type:type subtype:subtype];
        if([imagePath length])
        {
            CPImage *tempImage = [[CPImage alloc] initWithContentsOfFile:imagePath];
            
            if(tempImage)
            {
                // !! 如果直接回傳，當圖片檔被刪除時，後繼圖片的操作可能會異常，所以再產生一份出去
                NSData *tempData = [tempImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
                image = [[CPImage alloc] initWithData:tempData];
            }
            [tempImage release];
        }
    }
    return image;
}



//========================
//wcm 6.0.0 新增儲存重新辨識用的名片圖  get rerecogn image data
//========================
- (NSData *)copyRerecognImageDataWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    WC_ImageSubType subtype=WC_IST_ReRecogn;
    //-----------------------------------------------------------
    // !! for multi-thread, check and wait setting image finish
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    NSString *settingCardID = nil;
    
    for(int i=0; i<10; i++)
    {
        settingCardID = [threadDict objectForKey:WCCIC_IsSettingImage];
        
        if(settingCardID)
        {
            if([settingCardID isEqualToString:cardID])
                [NSThread sleepForTimeInterval:0.1];
            else break;
        }
        else break;
    }
    
    
    
    //-----------------------------------------------------------
    // get image data
    //-----------------------------------------------------------
    // !! IDPhoto只有thumbnail
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;
    
    NSData *imageData = nil;
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
    
    if([imagePath length])
        imageData = [[NSData alloc] initWithContentsOfFile:imagePath];
    else
    {
        subtype =WC_IST_Original;
        imagePath=[self imagePathWithCardID:cardID type:type subtype:subtype];
        if([imagePath length])
            imageData = [[NSData alloc] initWithContentsOfFile:imagePath];
    }
    return imageData;
}



//===============================================================================
//
//===============================================================================
- (BOOL)removeImageWithCardID:(NSString *)cardID type:(WC_ImageType)type
{
    switch (type)
    {
        case WC_IT_IDPhoto:
            [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail] error:nil];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:type subtype:WC_IST_ReRecogn] error:nil];
            [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:type subtype:WC_IST_Original] error:nil];
            [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse] error:nil];
            [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail] error:nil];
            break;
            
        default:
            break;
    }
    
    return YES;
}



//========================
//wcm 6.0.0 新增儲存重新辨識用的名片圖  delete image
//========================
- (BOOL)removeRerecognImageWithCardID:(NSString *)cardID
{
    [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:WC_IT_FrontSide subtype:WC_IST_ReRecogn] error:nil];
    [fileManager_ removeItemAtPath:[self imagePathWithCardID:cardID type:WC_IT_BackSide subtype:WC_IST_ReRecogn] error:nil];
    
    return YES;
    
}



//===============================================================================
//
//===============================================================================
- (BOOL)moveImageFromCardID:(NSString *)srcCardID toCardID:(NSString *)dstCardID withImageType:(WC_ImageType)type
{
    NSString *srcImagePath = nil;
    NSString *dstImagePath = nil;
    
    switch (type)
    {
        case WC_IT_IDPhoto:
            srcImagePath = [self imagePathWithCardID:srcCardID type:type subtype:WC_IST_Thumbnail];
            dstImagePath = [self imagePathWithCardID:dstCardID type:type subtype:WC_IST_Thumbnail];
            [self removeImageWithCardID:dstCardID type:type];
            [fileManager_ moveItemAtPath:srcImagePath toPath:dstImagePath error:nil];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            srcImagePath = [self imagePathWithCardID:srcCardID type:type subtype:WC_IST_Original];
            dstImagePath = [self imagePathWithCardID:dstCardID type:type subtype:WC_IST_Original];
            [self removeImageWithCardID:dstCardID type:type];
            [fileManager_ moveItemAtPath:srcImagePath toPath:dstImagePath error:nil];
            
            srcImagePath = [self imagePathWithCardID:srcCardID type:type subtype:WC_IST_Browse];
            dstImagePath = [self imagePathWithCardID:dstCardID type:type subtype:WC_IST_Browse];
            [fileManager_ moveItemAtPath:srcImagePath toPath:dstImagePath error:nil];
            
            srcImagePath = [self imagePathWithCardID:srcCardID type:type subtype:WC_IST_Thumbnail];
            dstImagePath = [self imagePathWithCardID:dstCardID type:type subtype:WC_IST_Thumbnail];
            [fileManager_ moveItemAtPath:srcImagePath toPath:dstImagePath error:nil];
            
            break;
            
        default:
            break;
    }
    
    return YES;
}


//===============================================================================
// set image
// NOTE : IDPhoto only have thumbnail image
//===============================================================================
- (BOOL)setImage:(CPImage *)image withCardID:(NSString *)cardID type:(WC_ImageType)type error:(NSError **)error
{
    NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];

    CPImage				*originalImage = nil;
    CPImage				*browseImage = nil;
    CPImage				*thumbnailImage = nil;
    NSString			*originalPath = nil;
    NSString			*browsePath = nil;
    NSString			*thumbnailPath = nil;
    BOOL				bResult = NO;
    NSError             *returnError = nil;
    
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s begin (%@-%d)", __func__, cardID, type];
#endif
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image start
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    [threadDict setObject:cardID forKey:WCCIC_IsSettingImage];
    
    
    //-----------------------------------------------------------
    // decide which type need to save
    //-----------------------------------------------------------
    switch (type)
    {
        case WC_IT_IDPhoto:
            originalImage = image;
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            
#if TARGET_OS_IPHONE
            
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            
#elif TARGET_OS_MAC
            
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            
#endif
            break;
            
        default: goto _EXIT;
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    
    //------------------------------------
    // save original image
    //------------------------------------
    if(originalPath)
    {
        originalImage = [self rotateAndResizeImage:image rotateDegree:0 maxSize:WC_IMS_Original];
        
        if(![self saveImage:originalImage toPath:[self tempPath:originalPath] error:&returnError])
        {
            [returnError retain];
            goto _EXIT;
        }
        if([[NSThread currentThread] isCancelled])
        {
            goto _EXIT;
        }
    }
    
    
    //------------------------------------
    // save browse image
    //------------------------------------
    if(browsePath)
    {
        browseImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Browse];
        
        if(![self saveImage:browseImage toPath:[self tempPath:browsePath] error:&returnError])
        {
            [returnError retain];
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // save thumbnail image
    //------------------------------------
    if(thumbnailPath)
    {
        thumbnailImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Thumbnail];
        
        if(![self saveImage:thumbnailImage toPath:[self tempPath:thumbnailPath] error:&returnError])
        {
            [returnError retain];
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // finish with success
    //------------------------------------
    bResult = YES;
    
    
    
_EXIT:
    
    // if success, delete previous files and rename temp files
    if(bResult)
    {
        // delete previous image file
        [self removeImageWithCardID:cardID type:type];
        
        // rename temp image file
        if([originalPath length])
            [fileManager_ moveItemAtPath:[self tempPath:originalPath] toPath:originalPath error:nil];
        
        if([browsePath length])
            [fileManager_ moveItemAtPath:[self tempPath:browsePath] toPath:browsePath error:nil];
        
        if([thumbnailPath length])
            [fileManager_ moveItemAtPath:[self tempPath:thumbnailPath] toPath:thumbnailPath error:nil];
    }
    else // if failed, delete temp files
    {
        if([originalPath length])
            [fileManager_ removeItemAtPath:[self tempPath:originalPath] error:nil];
        
        if([browsePath length])
            [fileManager_ removeItemAtPath:[self tempPath:browsePath] error:nil];
        
        if([thumbnailPath length])
            [fileManager_ removeItemAtPath:[self tempPath:thumbnailPath] error:nil];
        
        if([[NSThread currentThread] isCancelled] &&
           error!=nil)
            returnError = [PPErrorOperationCancel(nil) retain];
    }
    
    
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s end (%@-%d)", __func__, cardID, type];
#endif
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image stop
    //-----------------------------------------------------------
    [threadDict removeObjectForKey:WCCIC_IsSettingImage];
    
    [pool release];
    
    if (returnError)
    {
        [returnError autorelease];
        
        if (error)
        {
            *error = returnError;
        }
    }
    return bResult;
}


//===============================================================================
// set image by data
// NOTE : IDPhoto only have thumbnail image
//===============================================================================
- (BOOL)setImageData:(NSData *)imageData withCardID:(NSString *)cardID type:(WC_ImageType)type error:(NSError **)error
{
    NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];
    CPImage				*originalImage = nil;
    CPImage				*browseImage = nil;
    CPImage				*thumbnailImage = nil;
    NSString			*originalPath = nil;
    NSString			*browsePath = nil;
    NSString			*thumbnailPath = nil;
    BOOL				bResult = NO;
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image start
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    [threadDict setObject:cardID forKey:WCCIC_IsSettingImage];
    
    
    //-----------------------------------------------------------
    // decide which type need to save
    //-----------------------------------------------------------
    switch (type)
    {
        case WC_IT_IDPhoto:
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            
#if TARGET_OS_IPHONE
            
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            
#elif TARGET_OS_MAC
            
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            
#endif
            break;
            
        default: goto _EXIT;
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    
    //////////////////////////////////////////////////
    // !! 這個API主要是從同步與ImportWCXF進來時會使用,為避免佔用太多空間，所以特別處理
    //
    // 處理原則:
    // 1. 影像最長邊為 originalImage(1600)，太大要先縮小
    // 2. 影像的大小要小於150K
    // 3. browse, thumbnail以原圖壓縮比為主，不需再壓縮
    //////////////////////////////////////////////////
    
    originalImage = [CPImage imageWithData:imageData];

  
    // 原始影像若太大，要先縮小
    if([originalImage actualSize].width > WC_IMS_Original || [originalImage actualSize].height > WC_IMS_Original)
    {
        originalImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Original];
    }
    

    //------------------------------------
    // save original image
    //------------------------------------
    if(originalPath)
    {
        NSData *originalImageData = [originalImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        
        //大於150Ｋ就壓縮成0.1壓縮率的圖
        if(originalImageData.length > 150000)
        {
            // 轉為壓縮比0.1的圖
            originalImageData = [originalImage jpegRepresentationWithCompressQuality:WC_JPG_COMPRESS];
            originalImage = [CPImage imageWithData:originalImageData];
        }
        
        if(![self saveImageData:originalImageData toPath:[self tempPath:originalPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    
    //------------------------------------
    // set browse image
    //------------------------------------
    if(browsePath)
    {
        browseImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Browse];
        
        NSData *browseImageData = [browseImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];

        if(![self saveImageData:browseImageData toPath:[self tempPath:browsePath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // set thumbnail image
    //------------------------------------
    if(thumbnailPath)
    {
        thumbnailImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Thumbnail];
        
        NSData *thumbnailImageData = [thumbnailImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:thumbnailImageData toPath:[self tempPath:thumbnailPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // finish with success
    //------------------------------------
    bResult = YES;
    
    
    
_EXIT:
    
    // if success, delete previous files and rename temp files
    if(bResult)
    {
        // delete previous image file
        [self removeImageWithCardID:cardID type:type];
        
        // rename temp image file
        if([originalPath length])
            [fileManager_ moveItemAtPath:[self tempPath:originalPath] toPath:originalPath error:nil];
        
        if([browsePath length])
            [fileManager_ moveItemAtPath:[self tempPath:browsePath] toPath:browsePath error:nil];
        
        if([thumbnailPath length])
            [fileManager_ moveItemAtPath:[self tempPath:thumbnailPath] toPath:thumbnailPath error:nil];
    }
    else // if failed, delete temp files
    {
        if([originalPath length])
            [fileManager_ removeItemAtPath:[self tempPath:originalPath] error:nil];
        
        if([browsePath length])
            [fileManager_ removeItemAtPath:[self tempPath:browsePath] error:nil];
        
        if([thumbnailPath length])
            [fileManager_ removeItemAtPath:[self tempPath:thumbnailPath] error:nil];
        
        if([[NSThread currentThread] isCancelled] &&
           error!=nil)
        {
            *error = [PPErrorOperationCancel(nil) retain];
        }
    }
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image stop
    //-----------------------------------------------------------
    [threadDict removeObjectForKey:WCCIC_IsSettingImage];
    
    [pool release];
    if(error)
    {
        [*error autorelease];
    }

    return bResult;
}



//========================
//wcm 6.0.0 新增儲存重新辨識用的名片圖 set rerecogn image
//========================
- (BOOL)setRerecognImage:(CPImage *)image withCardID:(NSString *)cardID type:(WC_ImageType)type error:(NSError **)error
{
    NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];
    NSData              *compressData=nil;
    CPImage             *compressImage=nil;
    CPImage				*reRecognImage = nil;
    CPImage				*originalImage = nil;
    CPImage				*browseImage = nil;
    CPImage				*thumbnailImage = nil;
    NSString			*reRecognPath = nil;//WCM 6.0.0新增專給重新辨識用的圖片格式 caine
    NSString			*originalPath = nil;
    NSString			*browsePath = nil;
    NSString			*thumbnailPath = nil;
    BOOL				bResult = NO;
    
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s begin (%@-%d)", __func__, cardID, type];
#endif
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image start
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    [threadDict setObject:cardID forKey:WCCIC_IsSettingImage];
    
    
    //-----------------------------------------------------------
    // decide which type need to save
    //-----------------------------------------------------------
    switch (type)
    {
        case WC_IT_IDPhoto:
            originalImage = image;
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            reRecognPath= [self imagePathWithCardID:cardID type:type subtype:WC_IST_ReRecogn];
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        default: goto _EXIT;
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    //////////////////////////////////////////////////
    // !! 這個API主要是從辨識進來時會使用，進來的影像通常是原始圖(未壓縮)
    //
    // 處理原則:
    // 1. 原始影像轉為壓縮比為0.1的圖, original, browse, thumbnail由此影像儲存
    // 2. 用來再辨識的影像的圖，與其他幾個分開，由原始影像獨立產生，且儲存時壓縮比為0.7
    //////////////////////////////////////////////////
    
    //------------------------------------
    //prepare compress data
    //------------------------------------
    compressData = [image jpegRepresentationWithCompressQuality:WC_JPG_COMPRESS];
    compressImage = [CPImage imageWithData:compressData];
    
    
    //------------------------------------
    // save rerecogn image wcm 6.0.0新增的圖片格式 caine
    //------------------------------------
    if(reRecognPath)
    {
        reRecognImage = [self rotateAndResizeImage:image rotateDegree:0 maxSize:WC_IMS_ReRecogn];
        
        if(![self saveImageWithoutCompress:reRecognImage toPath:[self tempPath:reRecognPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }
        
        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    
    //------------------------------------
    // save original image
    //------------------------------------
    if(originalPath)
    {
        originalImage = [self rotateAndResizeImage:compressImage rotateDegree:0 maxSize:WC_IMS_Original];
        
        NSData *originalImageData = [originalImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:originalImageData toPath:[self tempPath:originalPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // save browse image
    //------------------------------------
    if(browsePath)
    {
        browseImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Browse];
        
        NSData *browseImageData = [browseImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:browseImageData toPath:[self tempPath:browsePath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // save thumbnail image
    //------------------------------------
    if(thumbnailPath)
    {
        thumbnailImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Thumbnail];
        
        NSData *thumbnailImageData = [thumbnailImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:thumbnailImageData toPath:[self tempPath:thumbnailPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // finish with success
    //------------------------------------
    bResult = YES;
    
    
    
_EXIT:
    
    // if success, delete previous files and rename temp files
    if(bResult)
    {
        // delete previous image file
        [self removeImageWithCardID:cardID type:type];
        
        // rename temp image file
        if([reRecognPath length])
            [fileManager_ moveItemAtPath:[self tempPath:reRecognPath] toPath:reRecognPath error:nil];
        
        if([originalPath length])
            [fileManager_ moveItemAtPath:[self tempPath:originalPath] toPath:originalPath error:nil];
        
        if([browsePath length])
            [fileManager_ moveItemAtPath:[self tempPath:browsePath] toPath:browsePath error:nil];
        
        if([thumbnailPath length])
            [fileManager_ moveItemAtPath:[self tempPath:thumbnailPath] toPath:thumbnailPath error:nil];
    }
    else // if failed, delete temp files
    {
        if([reRecognPath length])
            [fileManager_ removeItemAtPath:[self tempPath:reRecognPath] error:nil];
        
        if([originalPath length])
            [fileManager_ removeItemAtPath:[self tempPath:originalPath] error:nil];
        
        if([browsePath length])
            [fileManager_ removeItemAtPath:[self tempPath:browsePath] error:nil];
        
        if([thumbnailPath length])
            [fileManager_ removeItemAtPath:[self tempPath:thumbnailPath] error:nil];
        
        if([[NSThread currentThread] isCancelled] &&
           error!=nil)
        {
            *error = [PPErrorOperationCancel(nil) retain];
        }
    }
    
    
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s end (%@-%d)", __func__, cardID, type];
#endif
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image stop
    //-----------------------------------------------------------
    [threadDict removeObjectForKey:WCCIC_IsSettingImage];
    
    [pool release];
    if(error)
    {
        [*error autorelease];
    }

    return bResult;
}



//========================
//wcm 6.0.0 新增儲存重新辨識用的名片圖  set rerecogn image data
//========================
- (BOOL)setRerecognImageData:(NSData *)imageData withCardID:(NSString *)cardID type:(WC_ImageType)type error:(NSError **)error
{
    NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];
    NSData              *compressData=nil;
    CPImage             *compressImage=nil;
    CPImage				*reRecognImage = nil;
    CPImage				*originalImage = nil;
    CPImage				*browseImage = nil;
    CPImage				*thumbnailImage = nil;
    NSString			*reRecognPath = nil;//WCM 6.0.0新增專給重新辨識用的圖片格式 caine
    NSString			*originalPath = nil;
    NSString			*browsePath = nil;
    NSString			*thumbnailPath = nil;
    BOOL				bResult = NO;
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image start
    //-----------------------------------------------------------
    NSMutableDictionary *threadDict = [NSThread mainThread].threadDictionary;
    [threadDict setObject:cardID forKey:WCCIC_IsSettingImage];
    
    
    //-----------------------------------------------------------
    // decide which type need to save
    //-----------------------------------------------------------
    switch (type)
    {
        case WC_IT_IDPhoto:
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        case WC_IT_FrontSide:
        case WC_IT_BackSide:
            reRecognPath=[self imagePathWithCardID:cardID type:type subtype:WC_IST_ReRecogn];
            originalPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Original];
            browsePath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Browse];
            thumbnailPath = [self imagePathWithCardID:cardID type:type subtype:WC_IST_Thumbnail];
            break;
            
        default: goto _EXIT;
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    //////////////////////////////////////////////////
    // !! 這個API主要是從new Card flow進來時會使用，進來的影像是讀檔(壓縮比0.7)
    //
    // 處理原則:
    // 1. 原始影像(0.7)再轉為壓縮比為0.1的圖, original, browse, thumbnail由此影像儲存
    // 2. 用來再辨識的影像的圖，與其他幾個分開，由原始影像獨立產生，不再壓縮，因原本的圖壓縮比已是0.7
    //////////////////////////////////////////////////

    
    //------------------------------------
    //prepare compress data
    //------------------------------------
    compressImage = [CPImage imageWithData:imageData];
    compressData = [compressImage jpegRepresentationWithCompressQuality:WC_JPG_COMPRESS];
    
    // 壓縮比0.7
    reRecognImage = [CPImage imageWithData:imageData];
    
    //------------------------------------
    // save Rerecogn image
    //------------------------------------
    if(reRecognPath)
    {
        // 原始影像若太大，要先縮小再儲存
        if([reRecognImage actualSize].width > WC_IMS_Original || [reRecognImage actualSize].height > WC_IMS_Original)
        {
            reRecognImage = [self rotateAndResizeImage:reRecognImage rotateDegree:0 maxSize:WC_IMS_Original];
        }
        
        NSData *reRecognImageData = [reRecognImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:reRecognImageData toPath:[self tempPath:reRecognPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    
    
    originalImage = [CPImage imageWithData:compressData];
    
    //------------------------------------
    // save original image
    //------------------------------------
    if(originalPath)
    {
        // 原始影像若太大，要先縮小再儲存
        if([originalImage actualSize].width > WC_IMS_Original || [originalImage actualSize].height > WC_IMS_Original)
        {
            originalImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Original];
        }
        
        NSData *originalImageData = [originalImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:originalImageData toPath:[self tempPath:originalPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }
    }
    
    if([[NSThread currentThread] isCancelled])
        goto _EXIT;
    
    //------------------------------------
    // save browse image
    //------------------------------------
    if(browsePath)
    {
        browseImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Browse];
        
        NSData *browseImageData = [browseImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:browseImageData toPath:[self tempPath:browsePath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    
    //------------------------------------
    // save thumbnail image
    //------------------------------------
    if(thumbnailPath)
    {
        thumbnailImage = [self rotateAndResizeImage:originalImage rotateDegree:0 maxSize:WC_IMS_Thumbnail];
        
        NSData *thumbnailImageData = [thumbnailImage jpegRepresentationWithCompressQuality:WC_JPG_MaxCompress];
        if(![self saveImageData:thumbnailImageData toPath:[self tempPath:thumbnailPath] error:error])
        {
            if(error)
            {
                [*error retain];
            }
            goto _EXIT;
        }

        if([[NSThread currentThread] isCancelled])
            goto _EXIT;
    }
    
    //------------------------------------
    // finish with success
    //------------------------------------
    bResult = YES;
    
    
    
_EXIT:
    
    // if success, delete previous files and rename temp files
    if(bResult)
    {
        // delete previous image file
        [self removeImageWithCardID:cardID type:type];
        
        // rename temp image file
        if([reRecognPath length])
            [fileManager_ moveItemAtPath:[self tempPath:reRecognPath] toPath:reRecognPath error:nil];
        
        if([originalPath length])
            [fileManager_ moveItemAtPath:[self tempPath:originalPath] toPath:originalPath error:nil];
        
        if([browsePath length])
            [fileManager_ moveItemAtPath:[self tempPath:browsePath] toPath:browsePath error:nil];
        
        if([thumbnailPath length])
            [fileManager_ moveItemAtPath:[self tempPath:thumbnailPath] toPath:thumbnailPath error:nil];
    }
    else // if failed, delete temp files
    {
        if([reRecognPath length])
            [fileManager_ removeItemAtPath:[self tempPath:reRecognPath] error:nil];
        
        if([originalPath length])
            [fileManager_ removeItemAtPath:[self tempPath:originalPath] error:nil];
        
        if([browsePath length])
            [fileManager_ removeItemAtPath:[self tempPath:browsePath] error:nil];
        
        if([thumbnailPath length])
            [fileManager_ removeItemAtPath:[self tempPath:thumbnailPath] error:nil];
        
        if([[NSThread currentThread] isCancelled] &&
           error!=nil)
        {
            *error = [PPErrorOperationCancel(nil) retain];
        }
    }
    
    
    //-----------------------------------------------------------
    // !! for multi-thread, setting image stop
    //-----------------------------------------------------------
    [threadDict removeObjectForKey:WCCIC_IsSettingImage];
    
    [pool release];
    
    if(error)
    {
        [*error autorelease];
    }
    
    return bResult;
}



//===============================================================================
//
//===============================================================================
- (CGSize)imageSizeWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype
{
    // !! IDPhoto只有thumbnail
    if(type == WC_IT_IDPhoto)
        subtype = WC_IST_Thumbnail;
    
    // 這裡會大量使用，所以autorelease物件要立即釋放
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    CGSize imageSize = CGSizeZero;
    NSString *imagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
    
    
    if([fileManager_ fileExistsAtPath:imagePath])
    {
        NSInteger headerLength = 0xff;
        NSFileHandle *fileHandle = [NSFileHandle fileHandleForReadingAtPath:imagePath];
        NSData *headerData = [fileHandle readDataOfLength:0xff]; // 0xff只是抓大概，不是真的header長度。
        NSRange searchRange = {0, headerLength};
        NSRange resultRange;
        NSRange dataRange;
        unsigned char dataTemp[2] = {0};
        
        char widthTag[] = {0xa0, 0x02};
        char heightTag[] = {0xa0, 0x03};
        
        
        //---------------------------------
        // find width
        //---------------------------------
        resultRange = [headerData rangeOfData:[NSData dataWithBytes:widthTag length:2] options:0 range:searchRange];
        
        if(resultRange.length == 0)
            return CGSizeZero;
        
        // get width
        dataRange.location = resultRange.location+10;
        dataRange.length = 2;
        [headerData getBytes:dataTemp range:dataRange];
        imageSize.width = dataTemp[0]*16*16 + dataTemp[1];
        
        
        
        //---------------------------------
        // find height
        //---------------------------------
        searchRange.location = resultRange.location + 12;
        searchRange.length = headerLength-searchRange.location;
        resultRange = [headerData rangeOfData:[NSData dataWithBytes:heightTag length:2] options:0 range:searchRange];
        
        if(resultRange.length == 0)
            return CGSizeZero;
        
        // get width
        dataRange.location = resultRange.location + 10;
        dataRange.length = 2;
        [headerData getBytes:dataTemp range:dataRange];
        imageSize.height = dataTemp[0]*16*16 + dataTemp[1];;
    }
    
    [pool release];
    
    return imageSize;
}


//===============================================================================
//
//===============================================================================
- (BOOL)resetSmallImageWithCardID:(NSString *)cardID type:(WC_ImageType)type subtype:(WC_ImageSubType)subtype error:(NSError **)error
{
#ifdef _DUMP_LOG_
    [g_dumpLog logByFormatWithMemSize:@"%s", __func__];
#endif
    BOOL result = NO;
    
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    CPImage     *originImage = [self copyImageWithCardID:cardID type:type subtype:WC_IST_Original];
    CPImage     *smallImage = nil;
    NSString    *smallImagePath = nil;
    NSInteger   maxSize;
    
    
    if(originImage)
    {
        switch (subtype)
        {
            case WC_IST_Browse: maxSize = WC_IMS_Browse; break;
            default: maxSize = WC_IMS_Thumbnail; break;
        }
        
        if((smallImage = [self rotateAndResizeImage:originImage rotateDegree:0 maxSize:maxSize]))
        {
            smallImagePath = [self imagePathWithCardID:cardID type:type subtype:subtype];
            result = [self saveImage:smallImage toPath:smallImagePath error:error];
            if(error)
            {
                [*error retain];
            }
        }
        
        [originImage release];
    }
    
    [pool release];
    if(error)
    {
        [*error autorelease];
    }
    return result;
}




//===============================================================================
// maxSize == 0 表示不縮放
//===============================================================================
- (CPImage *)rotateAndResizeImage:(CPImage *)srcImage rotateDegree:(NSInteger)rotateDegree maxSize:(NSInteger)maxSize
{
    if(!srcImage)
        return nil;
    
    CGSize srcSize = [srcImage actualSize];
    
    //---------------------------------------------------
    // calculate new size (0表示不縮放)
    //---------------------------------------------------
    if(maxSize > 0)
    {
        if ([srcImage actualSize].width > [srcImage actualSize].height)
        {
            if([srcImage actualSize].width <= maxSize)
            {
                srcSize = [srcImage actualSize];
            }
            else
            {
                srcSize.width = maxSize;
                srcSize.height = (NSInteger)(maxSize * [srcImage actualSize].height / [srcImage actualSize].width);
            }
        }
        else
        {
            if([srcImage actualSize].height <= maxSize)
            {
                srcSize = [srcImage actualSize];
            }
            else
            {
                srcSize.height = maxSize;
                srcSize.width = (NSInteger)(maxSize * [srcImage actualSize].width / [srcImage actualSize].height);
            }
        }
    }
    
    //---------------------------------------------------
    // Quartz的旋轉是逆時針方向，所以要轉換。
    //---------------------------------------------------
    NSInteger	quartzRotate;
    
    switch(rotateDegree)
    {
        case 0: quartzRotate = 0; break;
        case 90: quartzRotate = 270; break;
        case 180: quartzRotate = 180; break;
        case 270: quartzRotate = 90; break;
        default: return nil;
    }
    
    
    //---------------------------------------------------
    // check if need resize and rotate image
    //---------------------------------------------------
    if(quartzRotate == 0 && srcSize.width == [srcImage actualSize].width && srcSize.height == [srcImage actualSize].height)
        return srcImage;
    
    
    //---------------------------------------------------
    // resize and rotate image
    //---------------------------------------------------
    CGContextRef	bitmap = nil;
    CGImageRef		imageRef = [srcImage CGImage];
    CGSize			newSize;
    
    
    // use device setting
    CGBitmapInfo	bitmapInfo = (CGBitmapInfo)kCGImageAlphaNoneSkipLast;
    CGColorSpaceRef colorSpaceInfo = CGColorSpaceCreateDeviceRGB();
    size_t			componentCount = 4;
    size_t			bitsPerComponent = 8;
    size_t			bytesPerRow = 0;
    
    /*
     // use image setting
     CGBitmapInfo	bitmapInfo = CGImageGetBitmapInfo(imageRef);
     CGColorSpaceRef colorSpaceInfo = CGImageGetColorSpace(imageRef);
     size_t			componentCount = CGColorSpaceGetNumberOfComponents(colorSpaceInfo);
     size_t			bitsPerComponent = CGImageGetBitsPerComponent(imageRef);
     size_t			bytesPerRow = 0;
     */
    
    switch(quartzRotate)
    {
        case 0:
        {
            newSize.width = srcSize.width;
            newSize.height = srcSize.height;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            
            break;
        }
            
        case 90:
        {
            newSize.width = srcSize.height;
            newSize.height = srcSize.width;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            
            if(bitmap && rotateDegree != 0)
            {
                CGContextRotateCTM(bitmap, radians(90));
                CGContextTranslateCTM(bitmap, 0, -srcSize.height);
            }
            
            break;
        }
            
        case 180:
        {
            newSize.width = srcSize.width;
            newSize.height = srcSize.height;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            
            if(bitmap && rotateDegree != 0)
            {
                CGContextRotateCTM(bitmap, radians(180));
                CGContextTranslateCTM(bitmap, -srcSize.width, -srcSize.height);
            }
            
            break;
        }
            
        case 270:
        {
            newSize.width = srcSize.height;
            newSize.height = srcSize.width;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            
            if(bitmap && rotateDegree != 0)
            {
                CGContextRotateCTM(bitmap, radians(270));
                CGContextTranslateCTM(bitmap, -srcSize.width, 0);
            }
            
            break;
        }
    }
    
    
    if(!bitmap)
    {
        CGColorSpaceRelease(colorSpaceInfo);
        return nil;
    }
    
    
    if(rotateDegree == 0)
        CGContextDrawImage(bitmap, CGRectMake(0, 0, newSize.width, newSize.height), imageRef);
    else CGContextDrawImage(bitmap, CGRectMake(0, 0, srcSize.width, srcSize.height), imageRef);
    
    
    CGImageRef ref = CGBitmapContextCreateImage(bitmap);
    CPImage* newImage = [CPImage imageWithCGImage:ref];
    
    CGColorSpaceRelease(colorSpaceInfo);
    CGContextRelease(bitmap);
    CGImageRelease(ref);
    
    return newImage;
}

@end
