//
//  CVCQRCodeCaptureController.m
//  Pods
//
//  Created by Duke on 2015/4/7.
//
//

#import "WCCaptureViewController+ResourceDefine.h"
#import "CVCQRCodeCaptureController.h"
#import "NSString+Additions.h"
#import "PPQRCodeController.h"
#import "PPButton+Factory.h"
#import "PPCameraView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, WCQRCodeMode_ButtonTag)
{
    ButtonTag_None  = 0,
    ButtonTag_Cancel,
    ButtonTag_Torch,
};


////////////////////////////////////////////////////////////////////////////////////////////////////

@interface CVCQRCodeCaptureController() <PPQRCodeControllerDelegate>

// Settings Bar
@property (nonatomic, retain) PPBarView *settingsBar;
@property (nonatomic, retain) PPButton *ppButtonTorch;

// Capture Bar
@property (nonatomic, retain) PPButton *ppButtonCancel;

// Operation
@property (atomic, assign) BOOL isRecognized;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation CVCQRCodeCaptureController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        self.isRecognized = NO;

        //////////////////////////////////////////////////
        // Capture ToolBar
        
        self.ppButtonCancel = [PPButton ppButtonWithIconImageName:WCCVC_IFP_CancelButton
                                                              tag:ButtonTag_Cancel
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:nil
                                                  imageEdgeInsets:WCCVC_LD_CaptureBarImageEdgeInset];
        if (self.ppButtonCancel != nil)
        {
            [self.ppButtonCancel addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        if (self.captureBar != nil)
        {
            self.captureBar.ignoreEdgeInset = YES;
            [self.captureBar setStyle:PPBarViewStyle_Average];
            [self.captureBar setViews:@[self.ppButtonCancel] forBlockType:PPBarViewBlockType_Center];
        }

       
        //////////////////////////////////////////////////
        // Settings ToolBar
        
        self.ppButtonTorch = [PPButton ppButtonWithIconImageName:WCCVC_IFP_TorchButton
                                                             tag:ButtonTag_Torch
                                                        delegate:self
                                           normalBackgroundColor:nil
                                      highlightedBackgroundColor:nil
                                                 imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];
        
        if (self.ppButtonTorch != nil)
        {
            [self.ppButtonTorch addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        _settingsBar = [[PPBarView alloc] init];
        if(self.settingsBar)
        {
            self.settingsBar.ignoreEdgeInset = YES;
            [self.settingsBar setStyle:PPBarViewStyle_Average];
            [self.settingsBar setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self.settingsBar setBackgroundColor:[UIColor clearColor]];
            [self.overlayView addSubview:self.settingsBar];
            
            self.settingsBar.backgroundImageView.backgroundColor = WCCVC_LD_DarkBlueColor;
            self.settingsBar.backgroundImageView.alpha = WCCVC_LD_SettingsBarAlpha;
            
            if([PPCameraView canUseTorchForCameraType:PPCameraType_Back] == YES)
            {
                [self.settingsBar setViews:@[self.ppButtonTorch] forBlockType:PPBarViewBlockType_Center];
            }
        }

        
        //////////////////////////////////////////////////
        // target view
        if(self.targetView)
        {
            CGFloat minSideLength = MIN([UIScreen mainScreen].bounds.size.width,
                                        [UIScreen mainScreen].bounds.size.height);
            CGFloat boxSideLength = WCCVC_LD_QRCodeTargetBoxRatio * minSideLength;
            
            self.targetView.fixedBoxSize = CGSizeMake(boxSideLength, boxSideLength);
            [self.overlayView addSubview:self.targetView];
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.ppButtonCancel = nil;
    self.ppButtonTorch = nil;
    self.settingsBar = nil;

    // !! 不要wait，切換時會卡。
//    [self.ppQRCodeRecognitionTaskQueue waitUntilAllOperationsAreFinished];

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview layout

//================================================================================
//
//================================================================================
- (void)setupSubviewsInterfaceOrientation:(UIInterfaceOrientation)orientation
{
    [super setupSubviewsInterfaceOrientation:orientation];
    
    //////////////////////////////////////////////////
    
    [self rotateView:self.ppButtonTorch];
    [self rotateView:self.ppButtonCancel];
}


//================================================================================
//
//================================================================================
- (void)setupOverlayViewLayoutConstraintWithStyle:(CVCBaseCaptureController_LayoutStyle)layoutStyle
{
    CGFloat settingsBarHeight = (layoutStyle == LayoutStyle_AnimationStart) ? 1 : WCCVC_LD_SettingsBarHeight;
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    NSDictionary *views = @{@"settingsBar":self.settingsBar,
                            @"targetView":self.targetView};
    NSDictionary *metrics = @{@"settingsBarHeight": @(settingsBarHeight)};
    
    
    // 水平佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[targetView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[settingsBar]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    // 垂直佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[settingsBar(settingsBarHeight)][targetView]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];


    //////////////////////////////////////////////////
    // synchronize overlayViewLayoutConstraints
    
    @synchronized(self)
    {
        if ([layoutConstraints count] > 0)
        {
            if(self.overlayViewLayoutConstraints != nil)
            {
                [self.overlayView removeConstraints:self.overlayViewLayoutConstraints];
            }
            
            [self.overlayView addConstraints:layoutConstraints];
            self.overlayViewLayoutConstraints = layoutConstraints;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview content/status

//================================================================================
//
//================================================================================
- (void)setupTorchState:(BOOL)enableTorch
{
    [self.ppButtonTorch setSelected:enableTorch];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - Flow action methods

//================================================================================
//
//================================================================================
- (void)actionWhenReceivePreviewImage:(UIImage *)previewImage
{
    if (previewImage == nil || self.isRecognized == YES)
    {
        return;
    }
    
    //////////////////////////////////////////////////

    // MARK: QRCode 辨識，不偵測 Error
    if([[PPQRCodeController qrCodeShareInstance] startRecognitionForQRCodeImage:previewImage
                                                                       delegate:self
                                                                          error:nil]==NO)
    {
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
        case ButtonTag_Cancel:
        {
            [[PPQRCodeController qrCodeShareInstance] cancelQRCodeImageOperation];
            
            //////////////////////////////////////////////////

            if ([self.delegate respondsToSelector:@selector(captureControllerRequestCancel:)])
            {
                [self.delegate captureControllerRequestCancel:self];
            }
            break;
        }
            
        case ButtonTag_Torch:
        {
            [ppButton setSelected:([ppButton isSelected] == NO)];
            if ([self.delegate respondsToSelector:@selector(captureController:requestUpdateTorchState:)])
            {
                [self.delegate captureController:self requestUpdateTorchState:[ppButton isSelected]];
            }
            break;
        }

        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPQRCodeCaptureOperationDelegate

//================================================================================
//
//================================================================================
- (void)ppQRCodeViewController:(PPQRCodeController *)ppQRCodeViewController withQRCodeString:(NSString *)string
{
    if([string length] > 0)
    {
        self.isRecognized = YES;
        
        if([self.delegate respondsToSelector:@selector(captureController:requestSendQRCodeString:)]==YES)
        {
            [self.delegate captureController:self requestSendQRCodeString:string];
        }
    }
}



@end
