//
//  CVCBaseCaptureController.m
//  Pods
//
//  Created by Duke on 2015/4/7.
//
//

#import "CVCBaseCaptureController.h"
#import "WCCaptureViewController+ResourceDefine.h"
#import "NSString+Additions.h"
#import "PPAutoResizedLabel+Factory.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define DEGREES_TO_RADIANS(degrees) (degrees * M_PI / 180)

#define CVCBaseCaptureController_InfoLabelFont              [UIFont systemFontOfSize:18]
#define CVCBaseCaptureController_InfoLabelTextColor         [UIColor whiteColor]
#define CVCBaseCaptureController_InfoLabelBGColor           WCCVC_LD_DarkBlueColor

#define CVCBaseCaptureController_CaptureBarBGColor          WCCVC_LD_DarkBlueColor

#define CVCBaseCaptureController_CaptureCountLabelFont      [UIFont boldSystemFontOfSize:15]
#define CVCBaseCaptureController_CaptureCountLabelTextColor WCCVC_LD_DarkBlueColor
#define CVCBaseCaptureController_CaptureCountLabelBGColor   WCCVC_LD_OrangeColor

CGFloat const CVCBaseCaptureController_InfoLabelAlphaMax        = 0.6;
CGFloat const CVCBaseCaptureController_InfoLabelAnimateDuration = 0.5;
CGFloat const CVCBaseCaptureController_ShowLastingInfoDuration  = 1.0;


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - CVCBaseCaptureController()

@interface CVCBaseCaptureController()

@property (atomic,    retain) UIImage *previewImage;
@property (nonatomic, assign) UIInterfaceOrientation interfaceOrientation;
@property (nonatomic, retain) NSString *recogLanguage;

@property (nonatomic, retain) PPAutoResizedLabel *infoLabel;
@property (nonatomic, retain) NSMutableArray *infoLabelLayoutConstraints;
@property (nonatomic, retain) NSString *lastingInfo;

@property (nonatomic, retain) PPAutoResizedLabel *captureCountLabel;
@property (nonatomic, retain) NSMutableArray *captureCountLayoutConstraints;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - CVCBaseCaptureController

@implementation CVCBaseCaptureController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Init & Dealloc

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        //////////////////////////////////////////////////
        // 提供給外部操作的subview

        _overlayView = [[CVCOverlayView alloc] init];
        if (self.overlayView)
        {
            self.overlayView.translatesAutoresizingMaskIntoConstraints = NO;
        }
        
        _captureBar = [[PPBarView alloc] init];
        if (self.captureBar)
        {
            self.captureBar.backgroundColor = CVCBaseCaptureController_CaptureBarBGColor;
            self.captureBar.translatesAutoresizingMaskIntoConstraints = NO;
        }

        //////////////////////////////////////////////////
        // 內部操作的subview
        
        _targetView = [[CVCTargetView alloc] init];
        
        if(self.targetView)
        {
            self.targetView.backgroundColor = [UIColor clearColor];
            self.targetView.userInteractionEnabled = NO;
            self.targetView.translatesAutoresizingMaskIntoConstraints = NO;
        }
        
        //////////////////////////////////////////////////

        self.focusMode = AVCaptureFocusModeContinuousAutoFocus;
        self.recogLanguage = @"";        
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [NSObject cancelPreviousPerformRequestsWithTarget:self];
    
    //////////////////////////////////////////////////
    self.delegate = nil;
    
    @synchronized(self)
    {
        self.overlayViewLayoutConstraints = nil;
        self.infoLabelLayoutConstraints = nil;
        self.captureCountLayoutConstraints = nil;
    }
    
    self.overlayView = nil;
    self.captureBar = nil;
    self.targetView = nil;
    self.infoLabel = nil;
    self.lastingInfo = nil;
    self.captureCountLabel = nil;
    self.recogLanguage = nil;
    self.previewImage = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Methods

//================================================================================
//
//================================================================================
- (CGAffineTransform)transformWithOrientation:(UIInterfaceOrientation)orientation
{
    switch (orientation)
    {
        case UIInterfaceOrientationLandscapeLeft:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(-90));
            
        case UIInterfaceOrientationLandscapeRight:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(90));
            
        case UIInterfaceOrientationPortraitUpsideDown:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(180));
            
        case UIInterfaceOrientationPortrait:
        default:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(0));
    }
}


//================================================================================
//
//================================================================================
- (void)rotateView:(UIView *)view
{
    view.transform = [self transformWithOrientation:self.interfaceOrientation];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Methods - Optional subviews

//================================================================================
//
//================================================================================
- (void)addInfoLabelWithSuperView:(UIView *)superview
{
    //////////////////////////////////////////////////
    // instance
    
    if(self.infoLabel == nil)
    {
        _infoLabel = [[PPAutoResizedLabel alloc] init];
        if (self.infoLabel)
        {
            self.infoLabel.maxSize = WCCVC_LD_InfoLabelMaxSize;
            self.infoLabel.contentInsets = UIEdgeInsetsMake(5, 8, 5, 8);
            self.infoLabel.textAlignment = NSTextAlignmentCenter;
            self.infoLabel.alpha = CVCBaseCaptureController_InfoLabelAlphaMax;
            self.infoLabel.layer.masksToBounds = YES;
            self.infoLabel.layer.cornerRadius = WCCVC_LD_CornerRadius;
            self.infoLabel.translatesAutoresizingMaskIntoConstraints = NO;
            self.infoLabel.textColor = CVCBaseCaptureController_InfoLabelTextColor;
            self.infoLabel.backgroundColor = CVCBaseCaptureController_InfoLabelBGColor;
            self.infoLabel.font = CVCBaseCaptureController_InfoLabelFont;
            self.infoLabel.numberOfLines = 0;
            
            [superview addSubview:self.infoLabel];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setupInfoLabelLayoutConstraintsWithMaxSize:(CGSize)maxSize
{
    UIView *superview = self.infoLabel.superview;
    NSDictionary *views = @{@"infoLabel":self.infoLabel};
    NSDictionary *metrics = @{@"maxWidth": @(maxSize.width),
                              @"maxHeight":@(maxSize.height)};
    NSMutableArray *layoutConstraints = [NSMutableArray array];

    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.infoLabel
                                                               attribute:NSLayoutAttributeCenterX
                                                               relatedBy:NSLayoutRelationEqual
                                                                  toItem:superview
                                                               attribute:NSLayoutAttributeCenterX
                                                              multiplier:1.0f
                                                                constant:0.0]];
    
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.infoLabel
                                                               attribute:NSLayoutAttributeTop
                                                               relatedBy:NSLayoutRelationEqual
                                                                  toItem:superview
                                                               attribute:NSLayoutAttributeBottom
                                                              multiplier:1.0f
                                                                constant:30.0f]];

    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:[infoLabel(<=maxWidth)]"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:[infoLabel(<=maxHeight)]"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    
    //////////////////////////////////////////////////
    // synchronize infoLabelLayoutConstraints
    
    @synchronized(self)
    {
        if ([layoutConstraints count] > 0)
        {
            if(self.infoLabelLayoutConstraints != nil)
            {
                [superview removeConstraints:self.infoLabelLayoutConstraints];
            }
            
            [superview addConstraints:layoutConstraints];
            self.infoLabelLayoutConstraints = layoutConstraints;
        }
    }
}


//================================================================================
// 永久更新info
//================================================================================
- (void)updateInfoLabelLastingWithText:(NSString *)text
{
    if([text length] > 0)
    {
        self.lastingInfo = text;
    }
    
    if(self.infoLabel == nil)
    {
        return;
    }
    
    // 沒有訊息時隱藏
    self.infoLabel.hidden = ([self.lastingInfo length] == 0);
    self.infoLabel.alpha = 0.0;
    self.infoLabel.text = self.lastingInfo;
        
//    NSLog(@"%s %p %@ %@", __func__, self.infoLabel, self.infoLabel.text, NSStringFromCGRect(self.infoLabel.frame));
    
    __block typeof(self) blockSelf = self;

    [UIView animateWithDuration:CVCBaseCaptureController_InfoLabelAnimateDuration
                     animations:^{
                         
                         blockSelf.infoLabel.alpha = CVCBaseCaptureController_InfoLabelAlphaMax;
                     }
                     completion:^(BOOL finished){
                         
                     }];
}


//================================================================================
// 暫時更新info
//================================================================================
- (void)updateInfoLabelTemporaryWithText:(NSString *)text
{
    // 沒有訊息時隱藏
    [self.infoLabel setHidden:([text length] == 0)];
    
    if(self.infoLabel == nil || [text length] == 0)
    {
        return;
    }

    //////////////////////////////////////////////////

    [NSObject cancelPreviousPerformRequestsWithTarget:self];
    
    //////////////////////////////////////////////////
    
    self.infoLabel.alpha = 0.0;
    self.infoLabel.text = text;
    
//    NSLog(@"%s %p %@ %@", __func__, self.infoLabel, self.infoLabel.text, NSStringFromCGRect(self.infoLabel.frame));
    
    __block typeof(self) blockSelf = self;
    
    [UIView animateWithDuration:CVCBaseCaptureController_InfoLabelAnimateDuration
                     animations:^{
                         
                         blockSelf.infoLabel.alpha = CVCBaseCaptureController_InfoLabelAlphaMax;
                     }
                     completion:^(BOOL finished){
                         
                         [self performSelector:@selector(updateInfoLabelLastingWithText:)
                                    withObject:nil
                                    afterDelay:CVCBaseCaptureController_ShowLastingInfoDuration];
                     }];
}



//================================================================================
//
//================================================================================
- (void)addCaptureCountLabelWithSuperview:(UIView *)superview
{
    //////////////////////////////////////////////////
    // instance
    
    if(self.captureCountLabel == nil)
    {
        self.captureCountLabel = [PPAutoResizedLabel counterStyleLableWithFont:CVCBaseCaptureController_CaptureCountLabelFont
                                                                     textColor:CVCBaseCaptureController_CaptureCountLabelTextColor
                                                               backgroundColor:CVCBaseCaptureController_CaptureCountLabelBGColor];
        
        if (self.captureCountLabel != nil)
        {
            self.captureCountLabel.translatesAutoresizingMaskIntoConstraints = NO;
            self.captureCountLabel.hidden = YES;
            
            [superview addSubview:self.captureCountLabel];
        }
    }

    
    //////////////////////////////////////////////////
    // layout constraints
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.captureCountLabel
                                                               attribute:NSLayoutAttributeRight
                                                               relatedBy:NSLayoutRelationLessThanOrEqual
                                                                  toItem:superview
                                                               attribute:NSLayoutAttributeRight
                                                              multiplier:1.0
                                                                constant:-1.0]];
    
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.captureCountLabel
                                                               attribute:NSLayoutAttributeLeft
                                                               relatedBy:NSLayoutRelationLessThanOrEqual
                                                                  toItem:superview
                                                               attribute:NSLayoutAttributeCenterX
                                                              multiplier:1.0
                                                                constant:WCCVC_LD_CaptureCountLabelOffset.x]];
    
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.captureCountLabel
                                                               attribute:NSLayoutAttributeBottom
                                                               relatedBy:NSLayoutRelationEqual
                                                                  toItem:superview
                                                               attribute:NSLayoutAttributeCenterY
                                                              multiplier:1.0
                                                                constant:WCCVC_LD_CaptureCountLabelOffset.y]];
    
    
    //////////////////////////////////////////////////
    // synchronize captureCountLayoutConstraints
    
    @synchronized(self)
    {
        if ([layoutConstraints count] > 0)
        {
            if(self.captureCountLayoutConstraints != nil)
            {
                [superview removeConstraints:self.captureCountLayoutConstraints];
            }
            
            [superview addConstraints:layoutConstraints];
            self.captureCountLayoutConstraints = layoutConstraints;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)updateCaptureCountLabelWithNumber:(NSInteger)count
{
    self.captureCountLabel.hidden = (count <= 0);
    self.captureCountLabel.text = [NSString stringWithInteger:count];
    self.captureCountLabel.layer.cornerRadius = self.captureCountLabel.bounds.size.height/2;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - Guide

//================================================================================
//
//================================================================================
- (NSArray *)guideDataModelsWithClassName:(NSString *)className
{
    return nil;
}


//================================================================================
//
//================================================================================
- (void)manuallyShowGuide
{
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview layout

//================================================================================
//
//================================================================================
- (void)addOptionalSubviews
{
    // CaptureCountLabel的constraint不能在init時設定，時機點需配合viewController，
    // 所以加入此函式讓外部可以決定什麼時候加入optional subviews。
}


//================================================================================
//
//================================================================================
- (void)setupSubviewsInterfaceOrientation:(UIInterfaceOrientation)orientation
{
    self.interfaceOrientation = orientation;
    
    // TODO:[繼承物件實作]subview的方向調整
}


//================================================================================
//
//================================================================================
- (void)setupOverlayViewLayoutConstraintWithStyle:(CVCBaseCaptureController_LayoutStyle)layoutStyle
{
    // TODO:[繼承物件實作]設定不同狀態的Layout
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview content/status

//================================================================================
//
//================================================================================
- (void)setupCaptureCount:(NSInteger)captureCount
{

}


//================================================================================
//
//================================================================================
- (void)setupCaptureSide:(WCCaptureViewController_CaptureSide)captureSide
{

}


//================================================================================
//
//================================================================================
- (void)setupFlashState:(BOOL)enableFlash
{

}


//================================================================================
//
//================================================================================
- (void)setupTorchState:(BOOL)enableTorch
{
    
}


//================================================================================
//
//================================================================================
- (void)setupImageEnhanceState:(BOOL)enableImageEnhance
{

}


//================================================================================
//
//================================================================================
- (void)setupAutoCropState:(BOOL)enableAutoCrop
{

}


//================================================================================
//
//================================================================================
- (void)setupDoubleSideState:(BOOL)enableDoubleSide
{

}


//================================================================================
//
//================================================================================
- (void)setupRecogLanguageState:(NSString *)recogLanguage
{

}


//================================================================================
//
//================================================================================
- (void)setupAutoDetectionState:(BOOL)enableAutoDetection
{
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - Flow Controll

//================================================================================
//
//================================================================================
- (void)actionWhenActive
{
}


//================================================================================
//
//================================================================================
- (void)actionWhenDeactive
{
}


//================================================================================
//
//================================================================================
- (void)actionWhenReceivePreviewImage:(UIImage *)previewImage
{
}


//================================================================================
//
//================================================================================
- (void)actionWhenGetStillImage
{
}


//================================================================================
//
//================================================================================
- (void)actionWhenSkipBackSide
{
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
}





@end
