//
//  WCCaptureFlowController.m
//  
//
//  Created by Howard on 2016/1/12.
//
//

#import "WCCaptureFlowController.h"

// Define
#import "WCCaptureFlowController+SettingKey.h"
#import "WCCaptureFlowController+ResourceDefine.h"
#import "WCCaptureViewController+SettingsKey.h"

// Controller
#import "PPLogController.h"
#import "PPSettingsController.h"
#import "WCImageBrowseViewController.h"
#import "PPNavigationController.h"
#import "WCVCFDataController.h"

// Category
#import "NSError+Custom.h"
#import "UIImage+Additions.h"

// View
#import "PPAlertView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Define Function Pointer

typedef void (^AlertCallBack)(BOOL didBuyNow);

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface WCCaptureFlowController

@interface WCCaptureFlowController () <UIAlertViewDelegate,WCCaptureViewControllerDelegate,WCImageBrowseViewControllerDelegate>

@property (nonatomic,assign) BOOL singleManualMode; //單張拍照模式
@property (nonatomic,assign) BOOL captureFrontCard;

@property (nonatomic,copy)   AlertCallBack alertComplete;
@property (nonatomic,retain) UIViewController *superViewController;
@property (nonatomic,assign) WCCaptureViewController *captureViewController;
@property (nonatomic,retain) WCImageBrowseViewController *imageBrowseViewController;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////


@implementation WCCaptureFlowController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, CopyZing, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        [PPSettingsController setDefaultIntegerValue:CaptureMode_Manual
                                             withKey:WCCaptureFlowController_SettingsKey_LastCaptureMode];
     
        //////////////////////////////////////////////////

        //代表沒有上限
        _maxCaptureCardCount = WCCaptureFlowController_UnlimitedCount;
        
        _defaultCapturedImageSize = WCFC_DefaultImageSize;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_alertComplete release];
    _alertComplete = nil;
    
    _imageBrowseViewController.delegate = nil;
    [_imageBrowseViewController release];
    _imageBrowseViewController = nil;
    
    [_superViewController release];
    _superViewController = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
// 檢查是否可以新增
//================================================================================
- (BOOL)canCaptureNewCardWithError:(NSError **)error
{
    BOOL result = YES;
   
    do
    {
        //數量檢查
        if(self.maxCaptureCardCount<0 ||
           self.maxCaptureCardCount>self.curCaptureCardCount)
        {
            break;
        }

        //////////////////////////////////////////////////

        // !! 不要用PPErrorMake來產生，因為domian需要統一 !!
        if (error)
        {
            (*error) = [NSError errorWithDomain:WCCFC
                                           code:WCCaptureFlowControllerErrorCode_ReachMaximumCount
                                     lineNumber:(__LINE__)
                         localizedFailureReason:@"MLS_ReachMaxNewCardCount"
                                         object:nil];
        }
        
        result = NO;
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)showAlertWithError:(NSError *)error completion:(AlertCallBack)completion
{
    do
    {
        if(error==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        BOOL showAlert = YES;
        
        if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowRequestShowError:)]==YES)
        {
            showAlert = [self.captureFlowDelegate captureFlowRequestShowError:error];
        }
        
        if(showAlert==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        self.alertComplete = completion;

        //////////////////////////////////////////////////

        NSString *title = @"";
        
        if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowRequestTitleWithError:)])
        {
            title = [self.captureFlowDelegate captureFlowRequestTitleWithError:error];
        }
        
        //////////////////////////////////////////////////

        NSString *message = nil;
        
        // 先跟外部詢問有沒有特別的解釋
        if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowRequestMessageWithError:)])
        {
            message = [self.captureFlowDelegate captureFlowRequestMessageWithError:error];
        }
        
        // 沒有的話就直接使用error code敘述
        if([message length] == 0)
        {
            message = [error alertMessage];
        }
        
        //////////////////////////////////////////////////

        // show alert
        if (self.showSubscribeWhenReachMaxCount)
        {
            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                 title:title
                               message:message
                              delegate:self
                     cancelButtonTitle:WCCFC_MLS_OK
                     otherButtonTitles:WCCFC_MLS_SubscribeNow,nil];
        }
        else
        {
            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                 title:title
                               message:message
                              delegate:self
                     cancelButtonTitle:WCCFC_MLS_OK
                     otherButtonTitles:nil];
        }
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Show WCBrowseImageViewController Method

//================================================================================
//
//================================================================================
- (void)showBrowseImageViewContgrollerWithImage:(UIImage *)image
                                      cardCount:(NSUInteger)cardCount
                               recogSourceModel:(WCRecogSourceModel *)recogSourceModel
                             fromViewController:(UIViewController *)fromViewController
{
    do
    {
        if(fromViewController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        WCImageBrowseViewController *browseImageViewController = [[WCImageBrowseViewController alloc] init];
        
        if(browseImageViewController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        browseImageViewController.delegate = self;
       
        //////////////////////////////////////////////////
        
        // !! iphone全方向，ipad看global
        if (UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPhone)
        {
            browseImageViewController.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_All;
        }
        else
        {
            browseImageViewController.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_LandscapeLeft|PPInterfaceOrientationControllerMask_LandscapeRight;
        }
            
        
        browseImageViewController.sourceImage = image;
        
        //////////////////////////////////////////////////

        if(recogSourceModel==nil)
        {
            browseImageViewController.imageBrowseViewControllerMode = WCImageBrowseViewControllerMode_AddCard;
        }

        browseImageViewController.userInfo = recogSourceModel;
        
        //////////////////////////////////////////////////

        if(self.singleManualMode==NO &&
           self.captureFrontCard==YES)
        {
            browseImageViewController.showRearCaptureButton = YES;
        }
        else
        {
            browseImageViewController.showRearCaptureButton = NO;
        }

        // !! 幫忙掃描的流程會用到這個模式
        browseImageViewController.showNextCaptureButton = !self.hideNextCardButton;
        //////////////////////////////////////////////////
        
        //設定數量
        browseImageViewController.totalNumberOfImage = cardCount;
        
        //////////////////////////////////////////////////
        
        //設定影像優化
        browseImageViewController.imageLight = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableImageEnhance];
        
        browseImageViewController.imageSegment = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableAutoCrop];
        
        //////////////////////////////////////////////////
        
        PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:browseImageViewController];
        
        if(navigationController!=nil)
        {
            [fromViewController presentViewController:navigationController
                                             animated:YES
                                           completion:nil];
        }
        
        //////////////////////////////////////////////////
        
        self.imageBrowseViewController = browseImageViewController;
        
        [browseImageViewController release];
        [navigationController release];
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Request Delegate Method

//================================================================================
//
//================================================================================
- (NSUInteger)requestNumberOfCard
{
    NSUInteger count = 0;
    
    if([self.captureFlowDatasource respondsToSelector:@selector(numberOfCard)]==YES)
    {
        //設定數量
        count = [self.captureFlowDatasource numberOfCard];
    }
    
    return count;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIAlertView Delegate Method

//================================================================================
//
//================================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    NSString *buttonTitle = [alertView buttonTitleAtIndex:buttonIndex];
    
    self.alertComplete([buttonTitle isEqualToString:WCCFC_MLS_SubscribeNow]);
    self.alertComplete = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCImageBrowseViewController method

//================================================================================
//
//================================================================================
- (void)browseImageViewController:(WCImageBrowseViewController *)browseImageViewController
                           action:(WCImageBrowseViewControllerAction)action
{
    switch (action)
    {
        case WCImageBrowseViewControllerAction_NextCard:
        case WCImageBrowseViewControllerAction_CaptureBackCard:
        case WCImageBrowseViewControllerAction_Done:
        {
            if(self.singleManualMode==YES)
            {
                 if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowController:singleManualSelectImage:)]==YES)
                {
                    [self.captureFlowDelegate captureFlowController:self singleManualSelectImage:browseImageViewController.sourceImage];
                }
                
                //////////////////////////////////////////////////

                __block typeof(self) blockself = self;
                
                //////////////////////////////////////////////////

                [self.superViewController dismissAnimated:YES completion:^{
                    
                    //////////////////////////////////////////////////

                    if([blockself.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==YES)
                    {
                        [blockself.captureFlowDelegate captureFlowControllerDidFinish:blockself];
                    }
                }];
            }
            //理論上只有手動新增名片進的來
            else
            {
                //結束拍照
                if(action==WCImageBrowseViewControllerAction_Done)
                {
                    if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowController:fromCaptureViewControler:manualModeDidSelectImage:recogSourceModel:isFinal:)]==YES)
                    {
                        [self.captureFlowDelegate captureFlowController:self
                                               fromCaptureViewControler:self.captureViewController
                                               manualModeDidSelectImage:browseImageViewController.sourceImage
                                                       recogSourceModel:browseImageViewController.userInfo
                                                                isFinal:YES];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==YES)
                    {
                        [self.captureFlowDelegate captureFlowControllerDidFinish:self];
                    }
                
                }
                //續拍
                else
                {
                    if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowController:fromCaptureViewControler:manualModeDidSelectImage:recogSourceModel:isFinal:)]==YES)
                    {
                        [self.captureFlowDelegate captureFlowController:self
                                               fromCaptureViewControler:self.captureViewController
                                               manualModeDidSelectImage:browseImageViewController.sourceImage
                                                       recogSourceModel:browseImageViewController.userInfo
                                                                isFinal:NO];
                    }
                    //////////////////////////////////////////////////
                    
                    if(action==WCImageBrowseViewControllerAction_CaptureBackCard)
                    {
                        self.captureViewController.captureCount = browseImageViewController.totalNumberOfImage;
                        
                        self.captureViewController.captureSide = CaptureSide_Back;
                        
                        self.captureFrontCard = NO;
                    }
                    else if(action==WCImageBrowseViewControllerAction_NextCard)
                    {
                        self.captureViewController.captureCount = [self requestNumberOfCard];
                        
                        self.captureViewController.captureSide = CaptureSide_Front;
                        
                        self.captureFrontCard = YES;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    [browseImageViewController goBackAnimated:YES];
                }
            }
            
            break;
        }
        case WCImageBrowseViewControllerAction_Exit:
        {
            if(self.singleManualMode==NO)
            {
                if(self.captureFrontCard==YES)
                {
                    self.curCaptureCardCount--;
                }
            }
            
            //////////////////////////////////////////////////
            
            [browseImageViewController goBackAnimated:YES];
            
            break;
        }
        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCaptureViewControllerDelegate methods

//================================================================================
//
//================================================================================
- (NSArray *)captureViewController:(WCCaptureViewController *)captureViewController
     recogLanguagesWithCaptureSide:(WCCaptureViewController_CaptureSide)captureSide
{
    NSArray *recogLanguages = nil;
    
    do
    {
        //單張新增圖片，不用顯示辨識語系
        if(self.singleManualMode==YES)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.captureFlowDatasource==nil ||
           [self.captureFlowDatasource respondsToSelector:@selector(recogLanguagesWithCaptureSide:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        // MARK: 前後辨識語系
        recogLanguages = [self.captureFlowDatasource recogLanguagesWithCaptureSide:captureSide];

    }
    while (0);
    
    return recogLanguages;
}


//================================================================================
//
//================================================================================
- (void)captureViewController:(WCCaptureViewController *)captureViewController
       didSelectRecogLangauge:(NSString *)recogLanguage
              withCaptureSide:(WCCaptureViewController_CaptureSide)captureSide
{
    //紀錄辨識語系
    do
    {
        if([self.captureFlowDatasource respondsToSelector:@selector(recogLangCodeWithRecogTitle:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        NSUInteger recogLanguageValue = [self.captureFlowDatasource recogLangCodeWithRecogTitle:recogLanguage];
        
        //////////////////////////////////////////////////

        if(captureSide==CaptureSide_Front)
        {
            [PPSettingsController setIntegerValue:recogLanguageValue
                                          withKey:WCCaptureFlowController_SettingsKey_FrontCardRecogLanguage];
        }
        else
        {
            [PPSettingsController setIntegerValue:recogLanguageValue
                                          withKey:WCCaptureFlowController_SettingsKey_RearCardRecogLanguage];
        }

    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)captureViewController:(WCCaptureViewController *)captureViewController
              didCaptureImage:(UIImage *)image
                recogLanguage:(NSString *)recogLanguage
             isDoubleSideMode:(BOOL)isDoubleSideMode

{
    @autoreleasepool
    {
        //縮圖，做前背景分離才不會慢
        if(MAX(self.defaultCapturedImageSize.width, self.defaultCapturedImageSize.height)<MAX(image.size.width, image.size.height) ||
           MIN(self.defaultCapturedImageSize.width, self.defaultCapturedImageSize.height)<MIN(image.size.width, image.size.height))
        {
            image = [image imageScalingMaxLength:MAX(self.defaultCapturedImageSize.width,self.defaultCapturedImageSize.height)];
        }
        
        //////////////////////////////////////////////////
        
        if(self.singleManualMode==YES)
        {
            self.captureViewController = captureViewController;
            
            //////////////////////////////////////////////////

            [self showBrowseImageViewContgrollerWithImage:image
                                                cardCount:0
                                         recogSourceModel:nil
                                       fromViewController:captureViewController];
        }
        else
        {
            if(self.captureFrontCard==YES)
            {
                self.curCaptureCardCount++;
            }
            
            //////////////////////////////////////////////////

            if(captureViewController.activeCaptureMode==CaptureMode_Docking)
            {
                do
                {
                    if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowController:dockModeDidCaptureImage:recogLanguage:captureSide:isDoubleSideMode:)]==NO)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    [self.captureFlowDelegate captureFlowController:self
                                            dockModeDidCaptureImage:image
                                                      recogLanguage:recogLanguage
                                                        captureSide:captureViewController.captureSide
                                                   isDoubleSideMode:isDoubleSideMode];
                    
                    //////////////////////////////////////////////////
                    
                    //目前這張圖背景，所以下次拍照是正面
                    if(isDoubleSideMode==YES)
                    {
                        if(captureViewController.captureSide==CaptureSide_Back)
                        {
                            self.captureFrontCard = YES;
                        }
                        else
                        {
                            self.captureFrontCard = NO;
                        }
                    }
                    else
                    {
                        self.captureFrontCard = YES;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    //設定數量
                    captureViewController.captureCount = [self requestNumberOfCard];
                }
                while (0);
            }
            else
            {
                NSUInteger cardCount = [self requestNumberOfCard];
        
                //////////////////////////////////////////////////
                
                if(self.hideCaptureCount==NO)
                {
                    if(self.captureFrontCard==YES)
                    {
                        // 圖片顯示時，雖然資料還沒存到暫存區，可是計數要加1(含目前顯示的這張)
                        cardCount++;
                    }
                }

                //////////////////////////////////////////////////
                
                WCRecogSourceModel *recogSourceModel = nil;
                
                if([self.captureFlowDatasource respondsToSelector:@selector(recogSourceModelForImageType:recogLanguage:activeCaptureMode:)]==YES)
                {
                    WC_ImageType imageType = WC_IT_FrontSide;
                    
                    if(captureViewController.captureSide==CaptureSide_Back)
                    {
                        imageType = WC_IT_BackSide;
                    }
                    
                    //////////////////////////////////////////////////

                    recogSourceModel = [self.captureFlowDatasource recogSourceModelForImageType:imageType
                                                                                  recogLanguage:recogLanguage
                                                                              activeCaptureMode:CaptureMode_Manual];
                }
                
                //////////////////////////////////////////////////
                
                self.captureViewController = captureViewController;
                
                //////////////////////////////////////////////////

                // MARK: 圖片預覽
                [self showBrowseImageViewContgrollerWithImage:image
                                                    cardCount:cardCount
                                             recogSourceModel:recogSourceModel
                                           fromViewController:captureViewController];
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)captureViewController:(WCCaptureViewController *)captureViewController didRecongizeQRCodeString:(NSString *)string
{
    // !! 因為後面流程會用到VCF所以這邊先檢查，不然後面流程不好控制
    [WCVCFDataController checkAccessPermissionWithCompletion:^(BOOL authorized) {
        
        if(authorized)
        {
            __block typeof(self) blockself = self;
            
            dispatch_async(dispatch_get_main_queue(), ^{
                [captureViewController dismissAnimated:YES completion:^{
                    do
                    {
                        if(blockself.captureFlowDelegate==nil ||
                           [blockself.captureFlowDelegate respondsToSelector:@selector(captureFlowController:didRecongizeQRCodeString:)]==NO)
                        {
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        [blockself.captureFlowDelegate captureFlowController:blockself
                                                    didRecongizeQRCodeString:string];
                        
                        //////////////////////////////////////////////////
                        
                        if([blockself.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==NO)
                        {
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        [blockself.captureFlowDelegate captureFlowControllerDidFinish:blockself];
                    }
                    while (0);
                }];
            });
        }
    }];
}


//================================================================================
//
//================================================================================
- (BOOL)captureViewController:(WCCaptureViewController *)captureViewController
    shouldChangeToCaptureMode:(WCCaptureViewController_CaptureMode)captureMode
{
    BOOL shouldChange = NO;
    
    // MARK:如果有什麼情況不允許切換模式，在這裡處理。
    do
    {
        //超過一張圖片暫存，則不能在改變拍照模式
        if([self requestNumberOfCard]>0)
        {
            break;
        }
        
        shouldChange = YES;
    }
    while (0);
    
    //////////////////////////////////////////////////
    // !! 如果是切換到CaptureMode_QRCode, 而且沒有系統聯絡人權限，要跳出alert
    
    if (captureMode==CaptureMode_QRCode)
    {
        [WCVCFDataController checkAccessPermissionWithCompletion:^(BOOL authorized) {
            
            if(!authorized)
            {
                UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@""
                                                                                         message:WCCFC_MLS_NoAddressBookPermission
                                                                                  preferredStyle:UIAlertControllerStyleAlert];
                [alertController addAction:[UIAlertAction actionWithTitle:WCCFC_MLS_OK
                                                                    style:UIAlertActionStyleDefault
                                                                  handler:nil]];
                
                [captureViewController presentViewController:alertController animated:YES completion:nil];
            }
        }];
    }
    
    //////////////////////////////////////////////////
    if(shouldChange == YES)
    {
        // 紀錄目前的選取模式
        [PPSettingsController setIntegerValue:captureMode
                                      withKey:WCCaptureFlowController_SettingsKey_LastCaptureMode];
    }
    
    return shouldChange;
}


//================================================================================
//
//================================================================================
- (void)captureViewControllerDidClickCancel:(WCCaptureViewController *)captureViewController
{
    if(self.singleManualMode==YES)
    {
        __block typeof(self) blockself = self;
        
        [captureViewController dismissViewControllerAnimated:YES completion:^{
            if([blockself.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==YES)
            {
                [blockself.captureFlowDelegate captureFlowControllerDidFinish:blockself];
            }
        }];
    }
    else
    {
        do
        {
            if(self.captureFlowDelegate==nil ||
               [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidClickCancel:fromCaptureViewController:)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.captureFlowDelegate captureFlowControllerDidClickCancel:self
                                                fromCaptureViewController:captureViewController];
            
            //////////////////////////////////////////////////

            if(self.captureFlowDelegate==nil ||
               [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.captureFlowDelegate captureFlowControllerDidFinish:self];
        }
        while (0);
    }
}


//================================================================================
//
//================================================================================
- (void)captureViewControllerDidClickDone:(WCCaptureViewController *)captureViewController
{
    do
    {
        if(self.captureFlowDelegate==nil ||
           [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidClickDone:fromCaptureViewController:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.captureFlowDelegate captureFlowControllerDidClickDone:self
                                          fromCaptureViewController:captureViewController];
        
        //////////////////////////////////////////////////
        
        if(self.captureFlowDelegate==nil ||
           [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.captureFlowDelegate captureFlowControllerDidFinish:self];
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)captureViewControllerDidClickDoneWhenSubscribeNow:(WCCaptureViewController *)captureViewController
{
    do
    {
        if(self.captureFlowDelegate==nil ||
           [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidClickDone:fromCaptureViewController:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.captureFlowDelegate captureFlowControllerDidClickDone:self
                                          fromCaptureViewController:captureViewController];
        
        //////////////////////////////////////////////////
        
        if(self.captureFlowDelegate==nil ||
           [self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinishWithSubscribe:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.captureFlowDelegate captureFlowControllerDidFinishWithSubscribe:self];
    }
    while (0);
}

//================================================================================
//
//================================================================================
- (void)captureViewControllerSkipBackSide:(WCCaptureViewController *)captureViewController
{
    // 略過反面
    
    self.captureFrontCard = YES;
}


//================================================================================
//
//================================================================================
- (BOOL)captureViewController:(WCCaptureViewController *)captureViewController shouldCaptureImageWithCaptureSide:(WCCaptureViewController_CaptureSide)captureSide isDoubleSideMode:(BOOL)isDoubleSideMode
{
    
    NSError *error = nil;
    
    if(self.captureFrontCard==YES &&
       [self canCaptureNewCardWithError:&error] == NO)
    {
        __block typeof(self) blockself = self;
        
        [captureViewController retain];
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            [blockself showAlertWithError:error completion:^(BOOL didBuyNow) {
                
                if (didBuyNow)
                {
                    [blockself captureViewControllerDidClickDoneWhenSubscribeNow:captureViewController];
                }
                else
                {
                    [blockself captureViewControllerDidClickDone:captureViewController];
                }
                [captureViewController release];
            }];
        });
        
        return NO;
    }
    else
    {
        return YES;
    }
}


//================================================================================
//
//================================================================================
- (void)captureViewControllerShowDockPromotion:(WCCaptureViewController *)captureViewController
{
    do
    {
        if([self.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerShowDockPromotion:fromCaptureViewController:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        [self.captureFlowDelegate captureFlowControllerShowDockPromotion:self
                                               fromCaptureViewController:captureViewController];
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (void)presentMultiModeFromViewController:(UIViewController *)viewController
                               captureSide:(WCCaptureViewController_CaptureSide)captureSide
                              captureCount:(NSInteger)captureCount
                       supportCaptureModes:(NSArray *)supportCaptureModes
                         activeCaptureMode:(WCCaptureViewController_CaptureMode)activeCaptureMode
                         presentCompletion:(void (^)(BOOL success))presentCompletion
{
    NSError *error = nil;
    
    __block typeof(self) blockSelf = self;
    
    if([self canCaptureNewCardWithError:&error] == NO)
    {
        [self showAlertWithError:error completion:^(BOOL didBuyNow) {
            
            if(didBuyNow)
            {
                if([blockSelf.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinishWithSubscribe:)]==YES)
                {
                    [blockSelf.captureFlowDelegate captureFlowControllerDidFinishWithSubscribe:blockSelf];
                }
            }
            else
            {
                if([blockSelf.captureFlowDelegate respondsToSelector:@selector(captureFlowControllerDidFinish:)]==YES)
                {
                    [blockSelf.captureFlowDelegate captureFlowControllerDidFinish:blockSelf];
                }
            }
        }];
    }
    else
    {
        self.singleManualMode = NO;
        
        //////////////////////////////////////////////////
        
        self.superViewController = viewController;
        
        //////////////////////////////////////////////////
        
        if(captureSide==CaptureSide_Front)
        {
            self.captureFrontCard = YES;
        }
        else
        {
            self.captureFrontCard = NO;
        }
        
        //////////////////////////////////////////////////
        
        [WCCaptureViewController presentFromViewController:viewController
                                                  delegate:self
                                               captureSide:CaptureSide_Front
                                              captureCount:captureCount
                                       supportCaptureModes:supportCaptureModes
                                         activeCaptureMode:activeCaptureMode
                                         presentCompletion:presentCompletion];
    }
}


//================================================================================
//
//================================================================================
- (void)presentSingleManualModeFromViewController:(UIViewController *)viewController
                                      captureSide:(WCCaptureViewController_CaptureSide)captureSide
                                presentCompletion:(void (^)(BOOL success))presentCompletion
{
    self.singleManualMode = YES;
    
    //////////////////////////////////////////////////

    self.superViewController = viewController;
    
    //////////////////////////////////////////////////
    
    [WCCaptureViewController presentSingleManualModeFromViewController:viewController
                                                              delegate:self
                                                           captureSide:captureSide
                                                     presentCompletion:presentCompletion];
}
@end
