//
//  BRFCMainViewController.m
//
//

#import "BRFCMainViewController.h"
#import "WCBackupRestoreFlowController+ResourceDefine.h"
#import "WCBackupRestoreFlowController+SettingsKey.h"
#import "PPSettingsController.h"
#import "PPAlertView.h"
#import "PPNavigationBarView.h"
#import "PPButton+Factory.h"
#import "UIColor+HexString.h"
#import "WCAppearanceDefine.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

CGFloat const BRFCMainViewController_SectionFooterHeight = 60.0;

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, BRFCMainViewButtonTag)
{
    BRFCMainViewButtonTag_None = 0,
    BRFCMainViewButtonTag_Back,
    BRFCMainViewButtonTag_StartBackup,
    BRFCMainViewButtonTag_StartRestore,
};

typedef NS_OPTIONS(NSInteger, BRFCMainViewSection)
{
    BRFCMainViewSection_SpaceInfo,
    BRFCMainViewSection_Backup,
    BRFCMainViewSection_Restore
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface BRFCMainViewController () <UITableViewDataSource, PPButtonDelegate>
@property (nonatomic, assign) id<BRFCMainViewControllerDelegate> delegate;
@property (nonatomic, assign) PPStorageSpaceSelectControllerSpaceType spaceType;
@property (nonatomic, retain) UILabel *backupInfoLabel;
@property (nonatomic, retain) NSString *spaceTotalBytesText;
@property (nonatomic, retain) NSString *spaceFreeBytesText;
@property (nonatomic, retain) PPNavigationBarView *navigationBarView;
@property (nonatomic, retain) NSArray *sections;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation BRFCMainViewController

//================================================================================
//
//================================================================================
- (instancetype)initWithDelegate:(id<BRFCMainViewControllerDelegate>)delegate
                       spaceType:(PPStorageSpaceSelectControllerSpaceType)spaceType
{
    if(self = [super init])
    {
        self.delegate = delegate;
        self.spaceType = spaceType;
        self.tableView.scrollEnabled = NO;
        self.view.backgroundColor = [UIColor whiteColor];
        
        switch (spaceType)
        {
            case PPStorageSpaceSelectControllerSpaceType_iCloud:
            case PPStorageSpaceSelectControllerSpaceType_Huawei:
                self.sections = @[@(BRFCMainViewSection_Backup),
                                  @(BRFCMainViewSection_Restore)];
                break;
                
            default:
                self.sections = @[@(BRFCMainViewSection_SpaceInfo),
                                  @(BRFCMainViewSection_Backup),
                                  @(BRFCMainViewSection_Restore)];
                break;
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;
    
    [self.backupInfoLabel removeFromSuperview];
    self.backupInfoLabel = nil;
    
    self.spaceTotalBytesText = nil;
    self.spaceFreeBytesText = nil;
    
    self.navigationBarView = nil;
    
    self.sections = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIViewController overwrite methods

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];

    //////////////////////////////////////////////////
    
    if([self.delegate respondsToSelector:@selector(mainViewControllerRequestSapceInfo:)])
    {
        [self.delegate mainViewControllerRequestSapceInfo:self];
    }
    
    //////////////////////////////////////////////////
    
    if(self.navigationController != nil)
    {
        if(self.navigationBarView == nil)
        {
            _navigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
        }
        
        if(self.navigationBarView != nil)
        {
            PPButton *backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                                   tag:BRFCMainViewButtonTag_Back
                                                              delegate:self
                                                 normalBackgroundColor:nil
                                            highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                      imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
            
            self.navigationBarView.style = PPBarViewStyle_CenteredAbsolute;
            self.navigationBarView.titleLabel.text = WCBRFC_MLS_BackupRestore;
            [self.navigationBarView setViews:@[backButton] forBlockType:PPBarViewBlockType_Left];
            
            UIBarButtonItem *leftButtonItem = [[UIBarButtonItem alloc] initWithCustomView:self.navigationBarView];
            
            if(leftButtonItem != nil)
            {
                self.navigationItem.leftBarButtonItem = leftButtonItem;
                [leftButtonItem release];
            }            
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UITableViewDataSource methods

//================================================================================
//
//================================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return [self.sections count];
}


//================================================================================
//
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    BRFCMainViewSection sectionID = [[self.sections objectAtIndex:section] integerValue];
    NSInteger rowCount = 0;
    
    switch (sectionID)
    {
        case BRFCMainViewSection_SpaceInfo:
            rowCount = 2;
            break;

        case BRFCMainViewSection_Backup:
            rowCount = 1;
            break;
            
        case BRFCMainViewSection_Restore:
            rowCount = 1;
            break;
            
        default:
            break;
    }

    return rowCount;
}


//================================================================================
//
//================================================================================
- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    BRFCMainViewSection sectionID = [[self.sections objectAtIndex:section] integerValue];
    NSString *title = @"";
    
    switch (sectionID)
    {
        case BRFCMainViewSection_SpaceInfo:
            title = WCBRFC_MLS_SpaceInfo;
            break;
            
        default:
            break;
    }
    
    return title;
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    static NSString *reuseIdentifier = @"BRFCMainView";
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:reuseIdentifier];
    
    if(cell == nil)
    {
        cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleValue1 reuseIdentifier:reuseIdentifier];
        [cell autorelease];
    }
    
    //////////////////////////////////////////////////

    BRFCMainViewSection sectionID = [[self.sections objectAtIndex:[indexPath section]] integerValue];
    
    switch (sectionID)
    {
        case BRFCMainViewSection_SpaceInfo:
        {
            switch ([indexPath row])
            {
                case 0:
                {
                    NSString *detailText = @"";
                    
                    if([self.spaceTotalBytesText length] > 0)
                    {
                        detailText = self.spaceTotalBytesText;
                    }

                    cell.textLabel.text = WCBRFC_MLS_TotalSpace;
                    cell.detailTextLabel.text = detailText;

                    break;
                }

                case 1:
                {
                    NSString *detailText = @"";
                    
                    if([self.spaceFreeBytesText length] > 0)
                    {
                        detailText = self.spaceFreeBytesText;
                    }
                    
                    cell.textLabel.text = WCBRFC_MLS_FreeSpace;
                    cell.detailTextLabel.text = detailText;

                    break;
                }
                    
                default:
                    break;
            }
        
            break;
        }
            
        case BRFCMainViewSection_Backup:
        {
            UIButton *button = [self buttonWithBounds:cell.contentView.bounds
                                                 text:WCBRFC_MLS_StartBackup
                                                  tag:BRFCMainViewButtonTag_StartBackup];
            
            [cell.contentView addSubview:button];
            
            break;
        }
            
        case BRFCMainViewSection_Restore:
        {
            UIButton *button = [self buttonWithBounds:cell.contentView.bounds
                                                 text:WCBRFC_MLS_StartRestore
                                                  tag:BRFCMainViewButtonTag_StartRestore];
            
            [cell.contentView addSubview:button];
            
            break;
        }
            
        default:
            break;
    }
 
    return cell;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UITableViewDelegate methods

//================================================================================
//
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForFooterInSection:(NSInteger)section
{
    return BRFCMainViewController_SectionFooterHeight;
}


//================================================================================
//
//================================================================================
- (UIView *)tableView:(UITableView *)tableView viewForFooterInSection:(NSInteger)section
{
    UIView *footerView = nil;
    BRFCMainViewSection sectionID = [[self.sections objectAtIndex:section] integerValue];
    
    switch (sectionID)
    {
        case BRFCMainViewSection_Backup:
        {
            footerView = [[[UIView alloc] initWithFrame:CGRectMake(0, 0, self.tableView.bounds.size.width, BRFCMainViewController_SectionFooterHeight)] autorelease];
            
            if(footerView != nil)
            {
                footerView.backgroundColor = [UIColor whiteColor];
                
                [self.backupInfoLabel removeFromSuperview];
                self.backupInfoLabel = nil;
                
                _backupInfoLabel = [[UILabel alloc] initWithFrame:CGRectZero];
                
                if(self.backupInfoLabel != nil)
                {
                    [footerView addSubview:self.backupInfoLabel];
                }
                
                if([self.delegate respondsToSelector:@selector(mainViewControllerRequestBackupInfo:)])
                {
                    [self.delegate mainViewControllerRequestBackupInfo:self];
                }        
            }
            
            break;
        }
         
        case BRFCMainViewSection_Restore:
        {
            footerView = [[[UIView alloc] initWithFrame:CGRectMake(0, 0, self.tableView.bounds.size.width, BRFCMainViewController_SectionFooterHeight)] autorelease];
            
            if(footerView != nil)
            {
                footerView.backgroundColor = [UIColor whiteColor];
            }
            
            break;
        }

        default:
            break;
    }
    
    return footerView;
}


//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    [tableView deselectRowAtIndexPath:indexPath animated:NO];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate methods

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
        case BRFCMainViewButtonTag_Back:
            [self.navigationController popViewControllerAnimated:YES];
            break;
            
        case BRFCMainViewButtonTag_StartBackup:
            
            if([self.delegate respondsToSelector:@selector(mainViewControllerDidClickBackup:)])
            {
                [self.delegate mainViewControllerDidClickBackup:self];
            }
            
            break;
            
        case BRFCMainViewButtonTag_StartRestore:
            
            if([self.delegate respondsToSelector:@selector(mainViewControllerDidClickRestore:)])
            {
                [self.delegate mainViewControllerDidClickRestore:self];
            }
            
            break;

        default:
            break;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (UIButton *)buttonWithBounds:(CGRect)bounds text:(NSString *)text tag:(NSInteger)tag
{
    CGRect frame = CGRectInset(bounds, 5, 5);
    PPButton *button = [PPButton ppButtonWithFrame:frame
                                       IconImageName:nil
                                            delegate:self
                                                 tag:tag
                                     backgroundColor:WCAppearanceDefine_LightBlueColor];

    
    if(button != nil)
    {
        [button setTitle:text forState:UIControlStateNormal];
        [button setTitleColor:[UIColor whiteColor] forState:UIControlStateNormal];
        
        button.layer.masksToBounds = YES;
        button.layer.cornerRadius = 3.0;
        
        button.autoresizingMask = UIViewAutoresizingFlexibleWidth|UIViewAutoresizingFlexibleHeight;
    }
    
    return button;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods


//==============================================================================
//
//==============================================================================
- (void)hideSpaceInfo
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
    
        NSMutableArray *oriSections = [NSMutableArray arrayWithArray:blockSelf.sections];
        
        [oriSections removeObject:@(BRFCMainViewSection_SpaceInfo)];
        
        blockSelf.sections = [NSArray arrayWithArray:oriSections];
        
        [blockSelf.tableView reloadData];
    });
}

//================================================================================
//
//================================================================================
- (void)updateBackupInfo:(NSString *)backupInfo
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        blockSelf.backupInfoLabel.numberOfLines = 1;
        blockSelf.backupInfoLabel.textAlignment = NSTextAlignmentCenter;
        blockSelf.backupInfoLabel.adjustsFontSizeToFitWidth = YES;
        blockSelf.backupInfoLabel.font = [UIFont systemFontOfSize:14];
        blockSelf.backupInfoLabel.text = backupInfo;
        [blockSelf.backupInfoLabel sizeToFit];
        
        // 說明文字置中並下移
        CGRect frame = blockSelf.backupInfoLabel.frame;
        frame.size.width = blockSelf.tableView.frame.size.width;
        blockSelf.backupInfoLabel.frame = frame;
    });
}


//================================================================================
//
//================================================================================
- (void)updateSpaceInfoWithTotalBytesText:(NSString *)totalBytesText freeBytesText:(NSString *)freeBytesText
{
    self.spaceTotalBytesText = totalBytesText;
    self.spaceFreeBytesText = freeBytesText;
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        [blockSelf.tableView reloadData];
    });
}




@end
