//
//  SearchDateViewController.m
//  Extension
//
//  Created by AndrewLai on 2018/4/18.
//

//Define
#import "WCAdvancedSearchViewController+ParameterDefine.h"
#import "WCAdvancedSearchViewController+ResourceDefine.h"
#import "PPDateSectionController+ParameterDefine.h"
#import "NSDate+Format.h"
#import "ContactApperanceDefine.h"
#import "WCAppearanceDefine.h"

//View
#import "PPNavigationBarView.h"
#import "PPTableView.h"
#import "PPButton+Factory.h"
#import "PPTableViewCell.h"
#import "ContactDateAndTimeTextFieldTableViewCell.h"
#import "SearchTableSectionHeaderView.h"

//Controller
#import "SearchDateViewController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static NSInteger const SearchDateViewController_MaxRowCount = 2;

static CGFloat const DateKeyboardToolBarViewButtonHeight = 24;
static CGFloat const SearchDateViewControllerTableViewCellTextFontSize = 14;

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, SearchDateViewController_ButtonTag)
{
    SearchDateViewController_ButtonTag_Back = 0,
    SearchDateViewController_ButtonTag_Finish
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface SearchDateViewController ()
<
PPButtonDelegate,
UITableViewDataSource,
UITableViewDelegate,
PPTextFieldTableViewCellDelegate,
ContactKeyboardToolBarViewDelegate
>

@property (nonatomic, assign) BOOL   keyboardShown;
@property (nonatomic, assign) CGSize keyboardSize;

@property (nonatomic, retain) NSIndexPath *currentIndexPath;
@property (nonatomic, retain) PPNavigationBarView *ppNavigationBarView;
@property (nonatomic, retain) PPButton *finishButton;
@property (nonatomic, retain) PPButton *backButton;

@property (nonatomic, retain) PPTableView *dateTableView;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation SearchDateViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if (self = [super init])
    {
        _currentDateType = WCSearchType_CreateTime;
        
        //////////////////////////////////////////////////
        self.dateTypeArray = [NSMutableArray array];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [self.dateTableView removeFromSuperview];
    self.dateTableView.delegate = nil;
    self.dateTableView.dataSource = nil;
    self.dateTableView = nil;
    
    [self.ppNavigationBarView removeFromSuperview];
    self.ppNavigationBarView = nil;
    
    self.backButton = nil;
    self.finishButton = nil;

    [_dateTypeArray release];
    _dateTypeArray = nil;
    
    [_currentIndexPath release];
    _currentIndexPath = nil;
    
    [_fromDate release];
    _fromDate = nil;
    
    [_toDate release];
    _toDate = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    [self registerNotifications];
    
    //////////////////////////////////////////////////
    self.ppNavigationBarView = [[[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds] autorelease];
    if (self.ppNavigationBarView!=nil)
    {
        self.ppNavigationBarView.style      = PPBarViewStyle_CenteredAbsolute;
        self.ppNavigationBarView.tintColor  = self.navigationController.navigationBar.tintColor;
        self.ppNavigationBarView.titleLabel.text = WCASVC_MLS_NavigationDateAdd;
        
        UIBarButtonItem *leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.ppNavigationBarView] autorelease];
        if (leftBarButtonItem!=nil)
        {
            self.navigationItem.leftBarButtonItem = leftBarButtonItem;
        }
        
        //////////////////////////////////////////////////
        self.backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                               tag:SearchDateViewController_ButtonTag_Back
                                                          delegate:self
                                             normalBackgroundColor:nil
                                        highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                   imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        {
            [self.backButton addControlEvents:UIControlEventTouchUpInside];
            [self.ppNavigationBarView setViews:@[self.backButton]
                                  forBlockType:PPBarViewBlockType_Left];
        }
        
        //////////////////////////////////////////////////
        self.finishButton = [PPButton ppButtonWithIconImageName:WCAdvancedSearchViewControllerButtonName_Done
                                                                  tag:SearchDateViewController_ButtonTag_Finish
                                                             delegate:self
                                                normalBackgroundColor:nil
                                           highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                      imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        if(self.finishButton!=nil)
        {
            [self.finishButton addControlEvents:UIControlEventTouchUpInside];
            
            [self.ppNavigationBarView setViews:@[self.finishButton]
                                  forBlockType:PPBarViewBlockType_Right];
        }
    }
    
    //////////////////////////////////////////////////
    _dateTableView = [[PPTableView alloc] initWithFrame:CGRectZero style:UITableViewStylePlain];
    if(self.dateTableView!=nil)
    {
        self.dateTableView.separatorInset = UIEdgeInsetsZero;
        self.dateTableView.dataSource = self;
        self.dateTableView.delegate   = self;
        self.dateTableView.separatorColor = WCAppearanceDefine_SeparatorColor;
        
        if([self.dateTableView respondsToSelector:@selector(setEstimatedRowHeight:)]==YES)
        {
            self.dateTableView.rowHeight = UITableViewAutomaticDimension;
            [self.dateTableView setEstimatedRowHeight:PPTableViewDefaultRowHeight];
        }
        
        [self.view addSubview:self.dateTableView];
        [self.dateTableView reloadData];
    }
    
    //////////////////////////////////////////////////
    if (self.dateTypeArray.count==0)
    {
        self.currentDateType = WCSearchType_UserDefine;
    }
}


//================================================================================
//
//================================================================================
- (void)viewWillLayoutSubviews
{
    [super viewWillLayoutSubviews];
    
    //////////////////////////////////////////////////
    self.dateTableView.frame = CGRectMake(0, 0,self.view.bounds.size.width, self.view.bounds.size.height);
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [self unRegisterNotifications];
    
    //////////////////////////////////////////////////
    [super viewWillDisappear:animated];
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [self.dateTableView removeFromSuperview];
    self.dateTableView.delegate = nil;
    self.dateTableView.dataSource = nil;
    self.dateTableView = nil;
    
    [self.ppNavigationBarView removeFromSuperview];
    self.ppNavigationBarView = nil;
    
    self.backButton = nil;
    self.finishButton = nil;
    //////////////////////////////////////////////////
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource

//================================================================================
//
//================================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    if (self.currentDateType==WCSearchType_UserDefine)
    {
        //未包含dateType選擇
        return 1;
    }
    else
    {
        //包含dateType選擇
        return 2;
    }
}


//================================================================================
// 兩個row數量皆為2
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return SearchDateViewController_MaxRowCount;
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    UITableViewCell *cell = nil;
    
    if (self.currentDateType!=WCSearchType_UserDefine && indexPath.section==0)
    {
        cell = [tableView dequeueReusableCellWithIdentifier:[NSString stringWithFormat:@"%td-%td",indexPath.section,indexPath.row]];
        if(cell==nil)
        {
            cell = [[[PPTableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                           reuseIdentifier:[NSString stringWithFormat:@"%td-%td",indexPath.section,indexPath.row]] autorelease];
        }
        
        [cell.textLabel setTextColor:WCAppearanceDefine_CellTitleColor];

        //////////////////////////////////////////////////
        NSNumber *typeNumber = [self.dateTypeArray objectAtIndex:indexPath.row];
        switch (typeNumber.integerValue)
        {
            case WCSearchType_CreateTime:
            {
                cell.textLabel.text = WCCV_MLS_CreateTime;
                break;
            }
            case WCSearchType_ModifiedTime:
            default:
            {
                cell.textLabel.text = WCCV_MLS_ModifyTime;
                break;
            }
        }
        
        //////////////////////////////////////////////////
        if (typeNumber.integerValue==self.currentDateType)
        {
            cell.accessoryType = UITableViewCellAccessoryCheckmark;
        }
        else
        {
            cell.accessoryType = UITableViewCellAccessoryNone;
        }
    }
    else
    {
        cell = [self.dateTableView dequeueReusableCellWithIdentifier:DateAndTimeTextFieldCellIdentifier];
        if(cell==nil)
        {
            cell = [[[ContactDateAndTimeTextFieldTableViewCell alloc] initWithStyle:UITableViewCellStyleValue1 reuseIdentifier:DateAndTimeTextFieldCellIdentifier] autorelease];
        }
        else if([cell respondsToSelector:@selector(reuse)]==YES)
        {
            [cell performSelector:@selector(reuse)];
        }
        
        //////////////////////////////////////////////////
        //!!讓游標顏色消失
        ContactDateAndTimeTextFieldTableViewCell *dateTextFieldCell = (ContactDateAndTimeTextFieldTableViewCell *)cell;
        [dateTextFieldCell.textLabel setTextColor:WCAppearanceDefine_CellTitleColor];

        dateTextFieldCell.textField.tintColor = [UIColor clearColor];
        
        dateTextFieldCell.delegate = self;
        dateTextFieldCell.contactKeyboardToolBarViewDelegate = self;
        
        //////////////////////////////////////////////////
        dateTextFieldCell.textField.inputAccessoryView.frame = CGRectMake(0, 0, self.view.bounds.size.width,DateKeyboardToolBarViewButtonHeight);
        
        //////////////////////////////////////////////////
        dateTextFieldCell.ignoreCustomLabelFrame = YES;
        
        //////////////////////////////////////////////////
        [dateTextFieldCell.textLabel setText:WCASVC_MLS_DateFieldStart];
        if (indexPath.row==1)
        {
            [dateTextFieldCell.textLabel setText:WCASVC_MLS_DateFieldEnd];
        }
        
        //////////////////////////////////////////////////
        dateTextFieldCell.datePicker.datePickerMode = UIDatePickerModeDate;
        
        //////////////////////////////////////////////////
        NSDateFormatter *dateFormatter = [[[NSDateFormatter alloc] init] autorelease];
        [dateFormatter setDateFormat:NSDateFormat_Day];
        
        NSString *dateString = [dateFormatter stringFromDate:self.fromDate];
        if (indexPath.row==1)
        {
            dateString = [dateFormatter stringFromDate:self.toDate];
        }
        
        [dateTextFieldCell.textField setPlaceholder:WCASVC_MLS_ChoiceStartDate];
        if (indexPath.row==1)
        {
            [dateTextFieldCell.textField setPlaceholder:WCASVC_MLS_ChoiceEndDate];
        }
        
        [dateTextFieldCell.textField setText:dateString];
        [dateTextFieldCell.textField setFont:[UIFont systemFontOfSize:SearchDateViewControllerTableViewCellTextFontSize]];
        [dateTextFieldCell.textField setClearButtonMode:UITextFieldViewModeWhileEditing];
    }
    
    return cell;
}


//================================================================================
//
//================================================================================
- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    NSString *titleSection = WCASVC_MLS_SectionTitleDateRange;
    
    if (self.currentDateType!=WCSearchType_UserDefine && section==0)
    {
        titleSection = WCASVC_MLS_SectionTitleDateProperty;
    }
    
    return titleSection;
}


//==============================================================================
//
//==============================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForHeaderInSection:(NSInteger)section
{
    return WCAppearanceDefine_SectionHeaderHeight;
}


//================================================================================
//
//================================================================================
- (UIView *)tableView:(UITableView *)tableView viewForHeaderInSection:(NSInteger)section
{
    SearchTableSectionHeaderView *sectionHeaderView = [[[SearchTableSectionHeaderView alloc] initWithFrame:CGRectZero] autorelease];
    if (sectionHeaderView!=nil)
    {
        sectionHeaderView.backgroundColor = WCAppearanceDefine_SectionHeaderBGColor;
        [sectionHeaderView.sectionLabel setTextColor:WCAppearanceDefine_SectionHeaderTitleColor];
        [sectionHeaderView.sectionLabel setFont:[UIFont systemFontOfSize:WCAppearanceDefine_SectionHeaderFontSize]];
        
        sectionHeaderView.sectionLabel.text =  [self tableView:tableView titleForHeaderInSection:section];
    }
    
    return sectionHeaderView;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - AdvanceSearchViewDelegate

//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    [tableView deselectRowAtIndexPath:indexPath animated:NO];
    
    //////////////////////////////////////////////////
    if (self.currentDateType!=WCSearchType_UserDefine)
    {
        NSNumber *typeNumber = [self.dateTypeArray objectAtIndex:indexPath.row];
        self.currentDateType = typeNumber.integerValue;
    }
    
    //////////////////////////////////////////////////
    [tableView reloadSections:[NSIndexSet indexSetWithIndex:indexPath.section] withRowAnimation:UITableViewRowAnimationNone];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPTextFieldTableViewCellDelegate

//================================================================================
//
//================================================================================
- (void)ppTextFieldTableViewCellDidBeginEditing:(PPTextFieldTableViewCell *)textFieldTableViewCell
{
    self.currentIndexPath = [self.dateTableView indexPathForCell:textFieldTableViewCell];
    
    //////////////////////////////////////////////////
    if([textFieldTableViewCell isKindOfClass:[ContactDateAndTimeTextFieldTableViewCell class]]==YES)
    {

        // MARK: 日期欄位 Pre Focus
        ContactDateAndTimeTextFieldTableViewCell *dateAndTimeTextFieldTableViewCell = (ContactDateAndTimeTextFieldTableViewCell *)textFieldTableViewCell;
        
        //////////////////////////////////////////////////
        // 設定上下限
        NSIndexPath *fromDateIndex = [NSIndexPath indexPathForRow:0 inSection:0];
        NSIndexPath *toDateIndex   = [NSIndexPath indexPathForRow:1 inSection:0];
        if (self.currentDateType!=WCSearchType_UserDefine)
        {
            fromDateIndex = [NSIndexPath indexPathForRow:0 inSection:1];
            toDateIndex   = [NSIndexPath indexPathForRow:1 inSection:1];
        }
        
        ContactDateAndTimeTextFieldTableViewCell *fromDateCell = [self.dateTableView cellForRowAtIndexPath:fromDateIndex];
        ContactDateAndTimeTextFieldTableViewCell *toDateCell   = [self.dateTableView cellForRowAtIndexPath:toDateIndex];
        
        NSString *fromDateString = fromDateCell.textField.text;
        NSString *toDateString = toDateCell.textField.text;
        
        if(dateAndTimeTextFieldTableViewCell==fromDateCell)
        {
            // from cell，要設定上限
            if ([toDateString length]>0)
            {
                NSDate *toDate = [NSDate dateFromString:toDateString format:NSDateFormat_Day];
                [dateAndTimeTextFieldTableViewCell.datePicker setMaximumDate:toDate];
            }
            else
            {
                [dateAndTimeTextFieldTableViewCell.datePicker setMinimumDate:nil];
                [dateAndTimeTextFieldTableViewCell.datePicker setMaximumDate:nil];
            }
        }
        else
        {
            // to cell，要設定下限
            if ([fromDateString length]>0)
            {
                NSDate *fromDate = [NSDate dateFromString:fromDateString format:NSDateFormat_Day];
                [dateAndTimeTextFieldTableViewCell.datePicker setMinimumDate:fromDate];
            }
            else
            {
                [dateAndTimeTextFieldTableViewCell.datePicker setMinimumDate:nil];
                [dateAndTimeTextFieldTableViewCell.datePicker setMaximumDate:nil];
            }
        }
        
        
        
        NSString *dateFormat = NSDateFormat_Day;
        CustomRectTextField *currentTextField = dateAndTimeTextFieldTableViewCell.textField;
        
        //給定預設時間字串
        do
        {
            if([currentTextField.text length]>0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            if(dateAndTimeTextFieldTableViewCell.datePicker.datePickerMode==UIDatePickerModeDateAndTime)
            {
                dateFormat = NSDateFormat_Minute;
            }
            
            currentTextField.text = [[NSDate date] stringWithFormat:dateFormat];
        }
        while (0);
        
        //////////////////////////////////////////////////
        [dateAndTimeTextFieldTableViewCell setDatePickerDateWithDateString:currentTextField.text format:dateFormat];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    BOOL isGoBack = YES;
    
    if (ppButton.tag==SearchDateViewController_ButtonTag_Finish)
    {
        NSDateFormatter *dateFormatter = [[[NSDateFormatter alloc] init] autorelease];
        [dateFormatter setDateFormat:NSDateFormat_Day];
        
        NSIndexPath *fromDateIndex = [NSIndexPath indexPathForRow:0 inSection:0];
        NSIndexPath *toDateIndex   = [NSIndexPath indexPathForRow:1 inSection:0];
        if (self.currentDateType!=WCSearchType_UserDefine)
        {
            fromDateIndex = [NSIndexPath indexPathForRow:0 inSection:1];
            toDateIndex   = [NSIndexPath indexPathForRow:1 inSection:1];
        }
        
        ContactDateAndTimeTextFieldTableViewCell *fromDateCell = [self.dateTableView cellForRowAtIndexPath:fromDateIndex];
        ContactDateAndTimeTextFieldTableViewCell *toDateCell   = [self.dateTableView cellForRowAtIndexPath:toDateIndex];
        
        self.fromDate = [dateFormatter dateFromString:fromDateCell.textField.text];
        self.toDate   = [dateFormatter dateFromString:toDateCell.textField.text];
        
        //////////////////////////////////////////////////
        // !! 改為可以按Done，不秀訊息
//        if (self.fromDate==nil && self.toDate==nil)
//        {
//            isGoBack = NO;
//            
//            __block typeof(self) blockSelf = self;
//            dispatch_async(dispatch_get_main_queue(), ^{
//
//                NSString *message = [NSString stringWithFormat:WCASVC_MLS_ErrorSearchContent, WCASVC_MLS_SectionTitleDate];
//                UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@""
//                                                                                         message:message
//                                                                                  preferredStyle:UIAlertControllerStyleAlert];
//                [alertController addAction:[UIAlertAction actionWithTitle:WCASVC_MLS_Confirm
//                                                                    style:UIAlertActionStyleDefault
//                                                                  handler:nil]];
//
//                [blockSelf presentViewController:alertController animated:YES completion:nil];
//            });
//        }
//        else
        {
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(didFinishInSearchDateViewController:)]==YES)
            {
                [self.delegate didFinishInSearchDateViewController:self];
            }
        }
    }
    
    //////////////////////////////////////////////////
    if (isGoBack==YES)
    {
        [self goBackAnimated:YES];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactKeyboardToolBarViewDelegate

//================================================================================
//
//================================================================================
- (void)contactEditKeyboardToolBarView:(ContactKeyboardToolBarView *)contactEditKeyboardToolBarView
                              ppButton:(PPButton *)ppButton
                          controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
        case ContactKeyboardToolBarViewButtonTag_HideKeyboard:
        {
            [self.view endEditing:YES];
            
            break;
        }
        case ContactKeyboardToolBarViewButtonTag_SwitchInput:
        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Keyboard Events

//================================================================================
//
//================================================================================
- (void)keyboardWillChangeFrame:(NSNotification *)notification
{
    NSLog(@"%s", __FUNCTION__);
    self.keyboardShown = YES;
    
    //////////////////////////////////////////////////
    NSDictionary* info = [notification userInfo];
    self.keyboardSize  = [[info objectForKey:UIKeyboardFrameEndUserInfoKey] CGRectValue].size;
    
    //////////////////////////////////////////////////
    if(self.keyboardShown==YES)
    {
        self.dateTableView.frame = CGRectMake(0, 0, MIN(self.view.bounds.size.width,self.view.bounds.size.height), MAX(self.view.bounds.size.width,self.view.bounds.size.height));
        
    }
    else
    {
        self.dateTableView.frame = CGRectMake(0, 0,self.view.bounds.size.width, self.view.bounds.size.height);
    }
    
    //////////////////////////////////////////////////
    UIEdgeInsets contentInsets;
    if(self.keyboardSize.height>self.keyboardSize.width)
    {
        contentInsets = UIEdgeInsetsMake(0.0, 0.0, self.keyboardSize.width, 0.0);
    }
    else
    {
        contentInsets = UIEdgeInsetsMake(0.0, 0.0, self.keyboardSize.height, 0.0);
    }
    
    //////////////////////////////////////////////////
    self.dateTableView.contentInset = contentInsets;
    self.dateTableView.scrollIndicatorInsets = contentInsets;
    
    //////////////////////////////////////////////////
    UITableViewCell *cell = [self.dateTableView cellForRowAtIndexPath:self.currentIndexPath];
    
    //////////////////////////////////////////////////
    CGRect cellFrame = cell.frame;
    
    //////////////////////////////////////////////////
    if(cell!=nil)
    {
        CGFloat height = self.view.bounds.size.height;
        
        height -= MIN(self.keyboardSize.width, self.keyboardSize.height);
        
        //////////////////////////////////////////////////
        CGRect rect = CGRectMake(0, self.dateTableView.contentOffset.y, self.view.bounds.size.width, height);
        
        //////////////////////////////////////////////////
        if (!CGRectContainsPoint(rect, cellFrame.origin))
        {
            __block typeof(self) blockself = self;
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [blockself.dateTableView beginUpdates];
                
                [blockself.dateTableView scrollRectToVisible:cellFrame
                                                                               animated:NO];
                
                [blockself.dateTableView endUpdates];
            });
            
        }
    }
}


//================================================================================
//
//================================================================================
- (void)keyboardWillHidden:(NSNotification *)notification
{
    NSLog(@"%s", __FUNCTION__);
    self.keyboardShown = NO;
    
    //////////////////////////////////////////////////
    UIEdgeInsets contentInsets = UIEdgeInsetsZero;
    self.dateTableView.contentInset = contentInsets;
    self.dateTableView.scrollIndicatorInsets = contentInsets;
    
    //////////////////////////////////////////////////
    self.dateTableView.frame = CGRectMake(0, 0,self.view.bounds.size.width, self.view.bounds.size.height);
}


//================================================================================
//
//================================================================================
- (void)registerNotifications
{
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(keyboardWillChangeFrame:)
                                                 name:UIKeyboardWillChangeFrameNotification
                                               object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(keyboardWillHidden:)
                                                 name:UIKeyboardWillHideNotification
                                               object:nil];
}


//================================================================================
//
//================================================================================
- (void)unRegisterNotifications
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

@end
