//
//  WCABCardModel+CNContact.m
//  
//
//  Created by Howard on 2018/1/3.
//

#import "WCABCardModel+CNContact.h"

// Framework
#import <Contacts/Contacts.h>

// Controller
#import "PPCountryCodeConvert.h"

// Category
#import "NSDate+Format.h"
#import "NSString+Additions.h"
#import "NSDictionary+Additions.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Parameter Define

#define WCABDC_WCDateEmptyYear [NSString stringWithInteger:WC_ValidMinYear]

static NSString * const WCABDC_ABDateEmptyYear = @"1604";

//static NSString * const WCABDC_ErrorReasonMessage = @"Get NSError retrun by system in userInfo with key NSErrorCustom_Key_Object.";
static NSString * const WCABDC_GoogleVoiceLabelKey = @"Google Voice";
static NSString * const WCABDC_BlogLabelKey = @"Blog";
static NSString * const WCABDC_ProfileLabelKey = @"Profile";
//static NSString * const WCABDC_SocialNetworkOtherLabelKey = @"Other";
//static NSString * const WCABDC_InstantMessageOtherLabelKey = @"Other";
static NSString * const WCABDC_SocialNetworkdDisplayNameKey = @"displayname";

static NSInteger const WCABDC_FakeType_keyForGetSubType1 = 1;
static NSInteger const WCABDC_FakeType_keyForGetValue = 2;
static NSInteger const WCABDC_FakeType_keyForGetAlternateValue = 3;
static WC_FieldType const WCABDC_FakeFieldType_Birthday = WC_FT_None+1111;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation

@implementation WCABCardModel (CNContact)

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//==============================================================================
//
//==============================================================================
- (NSString *)keyForConvertFieldCountWithFieldType:(WC_FieldType)fieldType subType1:(WC_FieldSubType1)subType1 subType2:(WC_FieldSubType2)subType2
{
    return [NSString stringWithFormat:@"%td-%td-%td", fieldType, subType1, subType2];
}


//==============================================================================
// MARK: 上限在這邊設定
//==============================================================================
- (NSInteger)maxFieldCountForExchangeWithFieldType:(WC_FieldType)fieldType subType1:(WC_FieldSubType1)subType1 subType2:(WC_FieldSubType2)subType2
{
    switch (fieldType)
    {
        case WC_FT_Name:
        case WC_FT_Company:
        {
            // 讀音不能匯出，其他都是1個
            if (subType2==WC_FST2_Name_FirstPhonetic ||
                subType2==WC_FST2_Name_LastPhonetic ||
                subType2==WC_FST2_Company_Phonetic)
            {
                return 0;
            }
            else
            {
                return 1;
            }
            break;
        }
        case WC_FT_Phone:
        {
            if(subType1==WC_FST1_Phone_Home||
               subType1==WC_FST1_Phone_Work)
            {
                return 2;
            }
            else if(subType1==WC_FST1_Phone_Mobile||
                    subType1==WC_FST1_Phone_HomeFax||
                    subType1==WC_FST1_Phone_WorkFax||
                    subType1==WC_FST1_Phone_Pager)
            {
                return 1;
            }
            
            return 0;
        }
        case WC_FT_Email:
        {
            if(subType1==WC_FST1_Email_Work||
               subType1==WC_FST1_Email_Home||
               subType1==WC_FST1_Email_Other)
            {
                return 1;
            }
            return 0;
        }
        case WC_FT_URL:
        {
            if(subType1==WC_FST1_URL_HomePage)
            {
                return 1;
            }
            
            return 0;
        }
        case WC_FT_Address:
        {
            if(subType1==WC_FST1_Address_Home||
               subType1==WC_FST1_Address_Work||
               subType1==WC_FST1_Address_Other)
            {
                return 1;
            }
            
            return 0;
        }
        case WC_FT_Note:
        case WC_FT_Nickname:
        {
            return 1;
        }
        case WCABDC_FakeFieldType_Birthday:
        default:
            return 0;
    }
}


//==============================================================================
// MARK:檢查是否要轉為CNContact的邏輯
//==============================================================================
- (BOOL)canConvertWithInfo:(NSMutableDictionary *)convertInfo
                 fieldType:(WC_FieldType)fieldType
                  subType1:(WC_FieldSubType1)subType1
                  subType2:(WC_FieldSubType2)subType2
{
    NSInteger curFieldCount = 0;
    NSString *convertFieldKey = [self keyForConvertFieldCountWithFieldType:fieldType subType1:subType1 subType2:subType2];
    if(convertInfo)
    {
        NSInteger maxFieldCount = [self maxFieldCountForExchangeWithFieldType:fieldType subType1:subType1 subType2:subType2];
        curFieldCount = [(NSNumber *)[convertInfo objectForKey:convertFieldKey] integerValue];
        if (maxFieldCount<=curFieldCount)
        {
            return NO;
        }
        else
        {
            //可以新增，count+1
            curFieldCount++;
            [convertInfo setObject:@(curFieldCount) forKey:convertFieldKey];
            
            return YES;
        }
    }
    
    return YES;
}


//==============================================================================
// convert date
//==============================================================================
- (NSDate *)validDateFromDate:(NSDate *)date convertMode:(WCABDC_ConvertMode)convertModel
{
    if (date==nil)
    {
        return nil;
    }
    
    NSString *dateString = [date stringWithFormat:NSDateFormat_Day
                                         timeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
    
    switch (convertModel)
    {
            
        case ConvertMode_WC_to_AB:
        {
            // !! 如果WC的字串有1900，表示是沒有年，要轉為1804代表系統的沒有年的設定
            if([dateString containsString:WCABDC_WCDateEmptyYear])
            {
                dateString = [dateString stringByReplacingOccurrencesOfString:WCABDC_WCDateEmptyYear
                                                                   withString:WCABDC_ABDateEmptyYear];
            }
            break;
        }
        case ConvertMode_AB_to_WC:
        {
            // !! 如果系統的字串小於等於1900，表示是沒有年，要轉為1900代表WC的沒有年的設定
            NSString *year = [date stringWithFormat:NSDateFormat_Year
                                           timeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
            
            if([year integerValue]<=WC_ValidMinYear)
            {
                dateString = [dateString stringByReplacingOccurrencesOfString:WCABDC_ABDateEmptyYear
                                                                   withString:WCABDC_WCDateEmptyYear];
            }
            break;
        }
            
        default:
            break;
    }
    
    return [NSDate dateFromString:dateString
                           format:NSDateFormat_Day
                         timeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
}


//================================================================================
//
//================================================================================
- (void)getMappingDictWithFieldType:(WC_FieldType)fieldType
                      fieldTypeDict:(NSDictionary **)fieldTypeDict
                       subType1Dict:(NSDictionary **)subType1Dict
                    defaultSubType1:(WC_FieldSubType1 *)defaultSubtype1
                       subType2Dict:(NSDictionary **)subType2Dict
{
    NSDictionary *returnFieldTypeDict = nil;
    NSDictionary *returnSubType1Dict = nil;
    NSDictionary *returnSubType2Dict = nil;
    WC_FieldSubType1 returnDefaultSubtype1 = WC_FST1_None;
    
    switch (fieldType)
    {
            //////////////////////////////////////////////////
            // type1
            
        case WC_FT_Note:
        {
            returnFieldTypeDict = @{CNContactNoteKey:@(WC_FT_Note)};
            break;
        }
            
        case WC_FT_Nickname:
        {
            returnFieldTypeDict = @{CNContactNicknameKey:@(WC_FT_Nickname)};
            break;
        }
        case WCABDC_FakeFieldType_Birthday:
        {
            returnFieldTypeDict = @{CNContactBirthdayKey:@(WCABDC_FakeFieldType_Birthday)};
            break;
        }
            
            
            //////////////////////////////////////////////////
            // type2
            
        case WC_FT_Name:
        {
            returnSubType2Dict = @{CNContactGivenNameKey:@(WC_FST2_Name_First),
                                   CNContactMiddleNameKey:@(WC_FST2_Name_Middle),
                                   CNContactFamilyNameKey:@(WC_FST2_Name_Last),
                                   
                                   CNContactPhoneticGivenNameKey:@(WC_FST2_Name_FirstPhonetic),
                                   CNContactPhoneticFamilyNameKey:@(WC_FST2_Name_LastPhonetic),
                                   
                                   CNContactNamePrefixKey:@(WC_FST2_Name_Prefix),
                                   CNContactNameSuffixKey:@(WC_FST2_Name_Suffix)};
            break;
        }
            
        case WC_FT_Company:
        {
            
#if TARGET_OS_IPHONE
            // !! 公司讀音ios10, macOS 10.12以上才支援
            if(@available(iOS 10.0, *))
            {
                returnSubType2Dict = @{CNContactOrganizationNameKey:@(WC_FST2_Company_Name),
                                       CNContactPhoneticOrganizationNameKey:@(WC_FST2_Company_Phonetic),
                                       CNContactJobTitleKey:@(WC_FST2_Company_JobTitle),
                                       CNContactDepartmentNameKey:@(WC_FST2_Company_Department)};
                
            }
            else
            {
                returnSubType2Dict = @{CNContactOrganizationNameKey:@(WC_FST2_Company_Name),
                                       CNContactJobTitleKey:@(WC_FST2_Company_JobTitle),
                                       CNContactDepartmentNameKey:@(WC_FST2_Company_Department)};
            }
#elif TARGET_OS_MAC
            // !! 公司讀音ios10, macOS 10.12以上才支援
            if([[NSProcessInfo processInfo] operatingSystemVersion].majorVersion>=10 &&
               [[NSProcessInfo processInfo] operatingSystemVersion].minorVersion>=12)
            {
                returnSubType2Dict = @{CNContactOrganizationNameKey:@(WC_FST2_Company_Name),
                                       CNContactPhoneticOrganizationNameKey:@(WC_FST2_Company_Phonetic),
                                       CNContactJobTitleKey:@(WC_FST2_Company_JobTitle),
                                       CNContactDepartmentNameKey:@(WC_FST2_Company_Department)};
                
            }
            else
            {
                returnSubType2Dict = @{CNContactOrganizationNameKey:@(WC_FST2_Company_Name),
                                       CNContactJobTitleKey:@(WC_FST2_Company_JobTitle),
                                       CNContactDepartmentNameKey:@(WC_FST2_Company_Department)};
            }
#endif
            
    
            break;
        }
            
            //////////////////////////////////////////////////
            // type3
            
        case WC_FT_Address:
        {
            returnFieldTypeDict = @{CNContactPostalAddressesKey:@(WC_FT_Address)};
            
            returnSubType1Dict = @{CNLabelWork:@(WC_FST1_Address_Work),
                                   CNLabelHome:@(WC_FST1_Address_Home),
                                   CNLabelOther:@(WC_FST1_Address_Other)};
            
            returnDefaultSubtype1 = WC_FST1_Address_Other;
            
            returnSubType2Dict = @{CNPostalAddressStreetKey:@(WC_FST2_Address_Street),
                                   CNPostalAddressCityKey:@(WC_FST2_Address_City),
                                   CNPostalAddressStateKey:@(WC_FST2_Address_State),
                                   CNPostalAddressPostalCodeKey:@(WC_FST2_Address_ZIP),
                                   CNPostalAddressCountryKey:@(WC_FST2_Address_Country),
                                   CNPostalAddressISOCountryCodeKey:@(WC_FST2_Address_CountryCode)};
            break;
        }
            
            //////////////////////////////////////////////////
            // type4
            
        case WC_FT_Phone:
        {
            returnFieldTypeDict = @{CNContactPhoneNumbersKey:@(WC_FT_Phone)};
            
            returnSubType1Dict = @{CNLabelWork:@(WC_FST1_Phone_Work),
                                   CNLabelHome:@(WC_FST1_Phone_Home),
                                   CNLabelPhoneNumberMobile:@(WC_FST1_Phone_Mobile),
                                   CNLabelPhoneNumberiPhone:@(WC_FST1_Phone_iPhone),
                                   CNLabelPhoneNumberMain:@(WC_FST1_Phone_Main),
                                   CNLabelPhoneNumberHomeFax:@(WC_FST1_Phone_HomeFax),
                                   CNLabelPhoneNumberWorkFax:@(WC_FST1_Phone_WorkFax),
                                   CNLabelPhoneNumberPager:@(WC_FST1_Phone_Pager),
                                   WCABDC_GoogleVoiceLabelKey:@(WC_FST1_Phone_GoogleVoice),
                                   CNLabelOther:@(WC_FST1_Phone_Other)};
            
            returnDefaultSubtype1 = WC_FST1_Phone_Other;
            break;
        }
            
        case WC_FT_Email:
        {
            returnFieldTypeDict = @{CNContactEmailAddressesKey:@(WC_FT_Email)};
            
            returnSubType1Dict = @{CNLabelWork:@(WC_FST1_Email_Work),
                                   CNLabelHome:@(WC_FST1_Email_Home),
                                   CNLabelOther:@(WC_FST1_Email_Other)};
            
            returnDefaultSubtype1 = WC_FST1_Email_Other;
            break;
        }
            
        case WC_FT_URL:
        {
            returnFieldTypeDict = @{CNContactUrlAddressesKey:@(WC_FT_URL)};
            
            returnSubType1Dict = @{CNLabelWork:@(WC_FST1_URL_Work),
                                   CNLabelHome:@(WC_FST1_URL_Home),
                                   CNLabelURLAddressHomePage:@(WC_FST1_URL_HomePage),
                                   WCABDC_BlogLabelKey:@(WC_FST1_URL_Blog),
                                   WCABDC_ProfileLabelKey:@(WC_FST1_URL_Profile),
                                   CNLabelOther:@(WC_FST1_URL_Other)};
            
            returnDefaultSubtype1 = WC_FST1_URL_Other;
            break;
        }
            
            
            //////////////////////////////////////////////////
            // type5
            
        case WC_FT_InstantMessage:
        {
            returnFieldTypeDict = @{CNContactInstantMessageAddressesKey:@(WC_FT_InstantMessage)};
            
            returnSubType1Dict = @{CNInstantMessageServiceYahoo:@(WC_FST1_IM_Yahoo),
                                   CNInstantMessageServiceJabber:@(WC_FST1_IM_Jabber),
                                   CNInstantMessageServiceMSN:@(WC_FST1_IM_MSN),
                                   CNInstantMessageServiceICQ:@(WC_FST1_IM_ICQ),
                                   CNInstantMessageServiceAIM:@(WC_FST1_IM_AIM),
                                   CNInstantMessageServiceQQ:@(WC_FST1_IM_QQ),
                                   CNInstantMessageServiceGoogleTalk:@(WC_FST1_IM_GoogleTalk),
                                   CNInstantMessageServiceGaduGadu:@(WC_FST1_IM_GaduGadu),
                                   CNInstantMessageServiceSkype:@(WC_FST1_IM_Skype),
                                   CNLabelOther:@(WC_FST1_IM_Other)};
            
            returnDefaultSubtype1 = WC_FST1_IM_AIM;
            
            returnSubType2Dict = @{CNInstantMessageAddressServiceKey:@(WCABDC_FakeType_keyForGetSubType1),
                                   CNInstantMessageAddressUsernameKey:@(WCABDC_FakeType_keyForGetValue)};
            
            break;
        }
            
        case WC_FT_SocialNetwork:
        {
            returnFieldTypeDict = @{CNContactSocialProfilesKey:@(WC_FT_SocialNetwork)};
            
            returnSubType1Dict = @{CNSocialProfileServiceTwitter:@(WC_FST1_SN_Twitter),
                                   CNSocialProfileServiceSinaWeibo:@(WC_FST1_SN_SinaWebo),
                                   CNLabelOther:@(WC_FST1_SN_Other),
                                   CNSocialProfileServiceFacebook:@(WC_FST1_SN_Facebook),
                                   CNSocialProfileServiceMySpace:@(WC_FST1_SN_MySpace),
                                   CNSocialProfileServiceLinkedIn:@(WC_FST1_SN_LinkedIn),
                                   CNSocialProfileServiceFlickr:@(WC_FST1_SN_Flicker)};
            
            returnDefaultSubtype1 = WC_FST1_SN_Twitter;
            
            returnSubType2Dict = @{CNSocialProfileServiceKey:@(WCABDC_FakeType_keyForGetSubType1),
                                   CNSocialProfileUsernameKey:@(WCABDC_FakeType_keyForGetValue),
                                   WCABDC_SocialNetworkdDisplayNameKey:@(WCABDC_FakeType_keyForGetAlternateValue)};
            
            break;
        }
            
            //////////////////////////////////////////////////
            // type date
            
        case WC_FT_Date:
            returnFieldTypeDict = @{CNContactDatesKey:@(WC_FT_Date)};
            
            returnSubType1Dict = @{CNLabelDateAnniversary:@(WC_FST1_Date_Anniversary),
                                   CNLabelOther:@(WC_FST1_Date_Other)};
            
            returnDefaultSubtype1 = WC_FST1_Date_Other;
            break;
            
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(fieldTypeDict != nil)
    {
        *fieldTypeDict = returnFieldTypeDict;
    }
    
    if(subType1Dict != nil)
    {
        *subType1Dict = returnSubType1Dict;
    }
    
    if(subType2Dict != nil)
    {
        *subType2Dict = returnSubType2Dict;
    }
    
    if(defaultSubtype1 != nil)
    {
        *defaultSubtype1 = returnDefaultSubtype1;
    }
}


//================================================================================
/// AB結構：一個value
//================================================================================
- (BOOL)convertType1FieldWithCardModel:(WCABCardModel *)cardModel
                               contact:(CNContact *)contact
                             fieldType:(WC_FieldType)fieldType
                           convertMode:(WCABDC_ConvertMode)convertMode
                     convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                 error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    NSDictionary *fieldTypeDict = nil;
    
    [self getMappingDictWithFieldType:fieldType
                        fieldTypeDict:&fieldTypeDict
                         subType1Dict:nil
                      defaultSubType1:nil
                         subType2Dict:nil];
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            NSString *key = [fieldTypeDict keyForIntValue:@(fieldType)];
            id value = [contact valueForKey:key];
            
            if(value != nil)
            {
                WCFieldModel *field = [[WCFieldModel alloc]init];
                
                if(field != nil)
                {
                    field.type = fieldType;
                    field.value = value;
                    
                    //////////////////////////////////////////////////
                    // !! 如果field是時間，要檢查時間是否有空的年
                    if ([field.value isKindOfClass:[NSDateComponents class]])
                    {
                        // NSDateComponents要轉成date
                        NSDateComponents *dateComponents = value;
                        dateComponents.timeZone = [NSTimeZone timeZoneForSecondsFromGMT:0];
                        NSDate * date = [dateComponents date];
                        field.value = [self validDateFromDate:date
                                                  convertMode:convertMode];
                    }
                    else
                    {
                        // 如果是空字串，當成沒有值
                        if ([(NSString*)value length]==0)
                        {
                            field.value = nil;
                        }
                    }
                    //////////////////////////////////////////////////
                    
                    if([field isEmpty] == NO)
                    {
                        [cardModel addField:field];
                        result = YES;
                    }
                    
                    [field release];
                }
                
            }
            
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            NSString *key = [fieldTypeDict keyForIntValue:@(fieldType)];
            
            WCFieldModel *fieldModel = [[cardModel fieldArrayWithType:fieldType] firstObject];
            
            if(fieldModel != nil)
            {
                // !! 檢查欄位上限
                if([self canConvertWithInfo:convertFieldCount fieldType:fieldType subType1:fieldModel.subType1 subType2:WC_FST2_None]==NO)
                {
                    result = YES;
                    break;
                }
                
                //////////////////////////////////////////////////
                // !! 如果field是時間，要檢查時間是否有空的年
                if ([fieldModel.value isKindOfClass:[NSDate class]])
                {
                    NSDate *validDate = [self validDateFromDate:fieldModel.value
                                                    convertMode:convertMode];
                    
                    NSDate *localDateTime = [NSDate dateWithTimeInterval:[[NSTimeZone systemTimeZone] secondsFromGMT] sinceDate:validDate];
                    
                    NSCalendar *gregorian = [[NSCalendar alloc]
                                             initWithCalendarIdentifier:NSCalendarIdentifierGregorian];
                    fieldModel.value =
                    [gregorian components:(NSCalendarUnitYear|NSCalendarUnitMonth|NSCalendarUnitDay)
                                 fromDate:localDateTime];
                    [gregorian release];
                }
                //////////////////////////////////////////////////
                

                [contact setValue:fieldModel.value forKey:key];
                result = YES;
            }
            else
            {
                [contact setValue:nil forKey:key];
                
                result = YES;
            }
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
/// AB結構：一個包含label & dict的array
//================================================================================
- (BOOL)convertType2FieldWithCardModel:(WCABCardModel *)cardModel
                               contact:(CNContact *)contact
                             fieldType:(WC_FieldType)fieldType
                           convertMode:(WCABDC_ConvertMode)convertMode
                     convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                 error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    NSDictionary *subType2Dict = nil;
    
    [self getMappingDictWithFieldType:fieldType
                        fieldTypeDict:nil
                         subType1Dict:nil
                      defaultSubType1:nil
                         subType2Dict:&subType2Dict];
    
    NSArray *allSubType2 = [subType2Dict allValues];
    NSArray *allProperty = [subType2Dict allKeys];
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            WCFieldModel *field = [[WCFieldModel alloc]init];
            
            if(field != nil)
            {
                field.type = fieldType;
                
                for(int i=0; i<[allProperty count]; i++)
                {
                    NSString *key = [allProperty objectAtIndex:i];
                    WC_FieldSubType2 subType2 = [[allSubType2 objectAtIndex:i] intValue];
                    
                    NSString * value = [contact valueForKey:key];
                    
                    if([value length]>0)
                    {
                        [field setSubType2FieldWithValue:value recogRect:CGRectZero subType2:subType2];
                    }
                }
                
                if([field isEmpty] == NO)
                {
                    [cardModel addField:field];
                    result = YES;
                }
                
                [field release];
            }
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            // !! 只用第一筆資料
            WCFieldModel *fieldModel = [[cardModel fieldArrayWithType:fieldType] firstObject];
            
            if(fieldModel != nil)
            {
                for(int i=0; i<[allSubType2 count]; i++)
                {
                    NSString *key = [allProperty objectAtIndex:i];
                    WC_FieldSubType2 subType2 = [[allSubType2 objectAtIndex:i] intValue];
                    
                    //////////////////////////////////////////////////
                    if([fieldModel hasFieldWithSubType2:subType2] == YES)
                    {
                        //////////////////////////////////////////////////
                        // !! 檢查欄位上限
                        if([self canConvertWithInfo:convertFieldCount fieldType:fieldType subType1:WC_FST1_None subType2:subType2]==NO)
                        {
                            continue;
                        }
                        
                        //////////////////////////////////////////////////
                        NSString *text = (NSString *)[[fieldModel fieldWithSubType2:subType2] value];
                        [contact setValue:text forKey:key];
                        result = YES;
                    }
                }
            }
            else
            {
                for(int i=0; i<[allSubType2 count]; i++)
                {
                    NSString *key = [allProperty objectAtIndex:i];
                    
                    [contact setValue:nil forKey:key];
                }
                
                result = YES;
            }
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
/// AB結構：一個包含label & dict的array (kABMultiDictionaryPropertyType)
//================================================================================
- (BOOL)convertType3FieldWithCardModel:(WCABCardModel *)cardModel
                               contact:(CNContact *)contact
                             fieldType:(WC_FieldType)fieldType
                           convertMode:(WCABDC_ConvertMode)convertMode
                     convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                 error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    NSDictionary *fieldTypeDict = nil;
    NSDictionary *subType1Dict = nil;
    NSDictionary *subType2Dict = nil;
    WC_FieldSubType1 defaultSubType1 = WC_FST1_None;
    
    [self getMappingDictWithFieldType:fieldType
                        fieldTypeDict:&fieldTypeDict
                         subType1Dict:&subType1Dict
                      defaultSubType1:&defaultSubType1
                         subType2Dict:&subType2Dict];
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            NSString *key = [fieldTypeDict keyForIntValue:@(fieldType)];
            
            NSArray *addresses = [contact valueForKey:key];
            
            for (CNLabeledValue *addressValue in addresses)
            {
                WCFieldModel *field = [[WCFieldModel alloc]init];
                
                if(field != nil)
                {
                    field.type = fieldType;
                    
                    //////////////////////////////////////////////////
                    // set subType1
                    id subType1 = nil;
                    NSString *addressLabel = addressValue.label;
                    if(addressLabel != nil)
                    {
                        subType1 = [subType1Dict valueForKey:addressLabel];
                    }
                    
                    if(subType1 == nil)
                    {
                        subType1 = @(defaultSubType1);
                    }
                    
                    field.subType1 = [subType1 intValue];
                    
                    //////////////////////////////////////////////////
                    // set subType2 & value
                    
                    CNPostalAddress *postalAddress = addressValue.value;
                    
                    if(postalAddress != nil)
                    {
                        NSArray *allSubType2 = [subType2Dict allValues];
                        NSArray *allProperty = [subType2Dict allKeys];
                        
                        for(int i=0; i<[allProperty count]; i++)
                        {
                            NSString *propertyKey = [allProperty objectAtIndex:i];
                            WC_FieldSubType2 subType2 = [[allSubType2 objectAtIndex:i] intValue];
                            
                            NSString *text = [postalAddress valueForKey:propertyKey];
                            
                            if([text length]>0)
                            {
                                [field setSubType2FieldWithValue:text recogRect:CGRectZero subType2:subType2];
                            }
                        }
                        
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if([field isEmpty] == NO)
                    {
                        [cardModel addField:field];
                        result = YES;
                    }
                    
                    [field release];
                }
            }
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            NSMutableArray *fieldModels = [cardModel fieldArrayWithType:fieldType];
            
            if([fieldModels count] == 0)
            {
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                [contact setValue:@[] forKey:objectKey];
                
                result = YES;
                break;
            }
            
            NSMutableArray *labeledValueArray = [NSMutableArray array];
            
            if(labeledValueArray != nil)
            {
                //////////////////////////////////////////////////
                // !! forExchange 只能匯出
                
                //////////////////////////////////////////////////
                NSArray *allSubType2 = [subType2Dict allValues];
                NSArray *allPropertyKey = [subType2Dict allKeys];
                
                for(int i=0; i<[fieldModels count]; i++)
                {
                    
                    WCFieldModel *fieldModel = [fieldModels objectAtIndex:i];
                    NSString *keyForSubtype1 = [subType1Dict keyForIntValue:@(fieldModel.subType1)];
                    
                    //////////////////////////////////////////////////
                    // !! 檢查欄位上限
                    if([self canConvertWithInfo:convertFieldCount fieldType:fieldType subType1:fieldModel.subType1 subType2:WC_FST2_None]==NO)
                    {
                        continue;
                    }
                    
                    //////////////////////////////////////////////////
                    CNPostalAddress *postalAddress = [[CNPostalAddress alloc] init];
                    
                    //////////////////////////////////////////////////
                    //--------------------------------------------------
                    // !! special case for address
                    NSString *countryCode = nil;
                    //--------------------------------------------------
                    
                    for(int j=0; j<[allSubType2 count]; j++)
                    {
                        WC_FieldSubType2 subType2 = [[allSubType2 objectAtIndex:j] intValue];
                        NSString *propertyKey = [allPropertyKey objectAtIndex:j];
                        
                        if([fieldModel hasFieldWithSubType2:subType2] == YES)
                        {
                            NSString *text = [fieldModel fieldWithSubType2:subType2].value;
                            
                            //--------------------------------------------------
                            // !! special case for address
                            if(subType2 == WC_FST2_Address_CountryCode)
                            {
                                countryCode = text;
                            }
                            //--------------------------------------------------
                            [postalAddress setValue:text forKey:propertyKey];
                        }
                    }
                    
                    //--------------------------------------------------
                    // !! special case : 調整country名稱
                    if(fieldType == WC_FT_Address)
                    {
                        NSString *localizedCountryText = [PPCountryCodeConvert localizeStringFromCountryCode:countryCode];
                        
                        if([localizedCountryText length] > 0)
                        {
                            [postalAddress setValue:localizedCountryText forKey:CNPostalAddressCountryKey];
                        }
                    }
                    //--------------------------------------------------
                    
                    CNLabeledValue *labeledValue = [CNLabeledValue labeledValueWithLabel:keyForSubtype1 value:postalAddress];
                    [labeledValueArray addObject:labeledValue];
                    [postalAddress release];
                }
                
                //////////////////////////////////////////////////
                
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                
                [contact setValue:labeledValueArray forKey:objectKey];
                result = YES;
            }
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
/// AB結構：一個包含label & value的array
//================================================================================
- (BOOL)convertType4FieldWithCardModel:(WCABCardModel *)cardModel
                               contact:(CNContact *)contact
                             fieldType:(WC_FieldType)fieldType
                           convertMode:(WCABDC_ConvertMode)convertMode
                     convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                 error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    NSDictionary *fieldTypeDict = nil;
    NSDictionary *subType1Dict = nil;
    WC_FieldSubType1 defaultSubType1 = WC_FST1_None;
    
    [self getMappingDictWithFieldType:fieldType
                        fieldTypeDict:&fieldTypeDict
                         subType1Dict:&subType1Dict
                      defaultSubType1:&defaultSubType1
                         subType2Dict:nil];
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            NSString *key = [fieldTypeDict keyForIntValue:@(fieldType)];
            NSArray *value = [contact valueForKey:key];
            
            for (CNLabeledValue *object in value)
            {
                WCFieldModel *field = [[WCFieldModel alloc]init];
                
                if(field != nil)
                {
                    field.type = fieldType;
                    
                    //////////////////////////////////////////////////
                    // set subType1
                    NSString *objectLabel = object.label;
                    
                    id subType1 = nil;
                    
                    if(objectLabel != nil)
                    {
                        subType1 = [subType1Dict valueForKey:objectLabel];
                    }
                    
                    if(subType1 == nil)
                    {
                        subType1 = @(defaultSubType1);
                    }
                    
                    field.subType1 = [subType1 intValue];
                    
                    //////////////////////////////////////////////////
                    // set value
                    id objectValue = object.value;
                    if(objectValue != nil)
                    {
                        if ([objectValue isKindOfClass:[CNPhoneNumber class]])
                        {
                            field.value = [(CNPhoneNumber*)objectValue stringValue];
                            
                            
                            // !! 如果是空字串，當成沒有值
                            if([field.value  length]==0)
                            {
                                field.value = nil;
                            }
                        }
                        else if ([objectValue isKindOfClass:[NSDateComponents class]])
                        {
                            // NSDateComponents要轉成date
                            NSDateComponents *dateComponents = objectValue;
                            dateComponents.timeZone = [NSTimeZone timeZoneForSecondsFromGMT:0];
                            NSDate * date = [dateComponents date];
                            field.value = [self validDateFromDate:date
                                                      convertMode:convertMode];
                        }
                        else
                        {
                            field.value = objectValue;
                            
                            
                            // !! 如果是空字串，當成沒有值
                            if([field.value  length]==0)
                            {
                                field.value = nil;
                            }
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if([field isEmpty] == NO)
                    {
                        [cardModel addField:field];
                        result = YES;
                    }
                    
                    [field release];
                }
            }
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            NSMutableArray *fieldModels = [cardModel fieldArrayWithType:fieldType];
            
            if([fieldModels count] == 0)
            {
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                [contact setValue:@[] forKey:objectKey];
                
                result = YES;
                break;
            }
            
            NSMutableArray *labeledValueArray = [NSMutableArray array];
            
            if(labeledValueArray != nil)
            {
                
                for(int i=0; i<[fieldModels count]; i++)
                {
                    WCFieldModel *fieldModel = [fieldModels objectAtIndex:i];
                    
                    NSString *propertyLabel = [subType1Dict keyForIntValue:@(fieldModel.subType1)];
                    id propertyValue = nil;
                    
                    //////////////////////////////////////////////////
                    // !! 檢查欄位上限
                    if([self canConvertWithInfo:convertFieldCount fieldType:fieldType subType1:fieldModel.subType1 subType2:WC_FST2_None]==NO)
                    {
                        continue;
                    }
                    
                    //////////////////////////////////////////////////
                    // !! 如果field是時間，要檢查時間是否有空的年
                    if ([fieldModel.value isKindOfClass:[NSDate class]])
                    {
                        NSDate *validDate = [self validDateFromDate:fieldModel.value
                                                        convertMode:convertMode];
                        
                        NSDate *localDateTime = [NSDate dateWithTimeInterval:[[NSTimeZone systemTimeZone] secondsFromGMT] sinceDate:validDate];
                        
                        NSCalendar *gregorian = [[NSCalendar alloc]
                                                 initWithCalendarIdentifier:NSCalendarIdentifierGregorian];
                        fieldModel.value =
                        [gregorian components:(NSCalendarUnitYear|NSCalendarUnitMonth|NSCalendarUnitDay)
                                     fromDate:localDateTime];
                        [gregorian release];
                    }
                    
                    //////////////////////////////////////////////////
                    if (fieldType==WC_FT_Phone)
                    {
                        propertyValue = [CNPhoneNumber phoneNumberWithStringValue:fieldModel.value];
                    }
                    else
                    {
                        propertyValue = fieldModel.value;
                    }
                    CNLabeledValue *labeledValue = [CNLabeledValue labeledValueWithLabel:propertyLabel value:propertyValue];
                    [labeledValueArray addObject:labeledValue];
                }
                
                //////////////////////////////////////////////////
                
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                [contact setValue:labeledValueArray forKey:objectKey];
                result = YES;
            }
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
/// AB結構：一個包含dict的array，欄位名稱和內容由不同的key從dict取得。
//================================================================================
- (BOOL)convertType5FieldWithCardModel:(WCABCardModel *)cardModel
                               contact:(CNContact *)contact
                             fieldType:(WC_FieldType)fieldType
                           convertMode:(WCABDC_ConvertMode)convertMode
                     convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                 error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    NSDictionary *fieldTypeDict = nil;
    NSDictionary *subType1Dict = nil;
    NSDictionary *subType2Dict = nil;
    WC_FieldSubType1 defaultSubType1 = WC_FST1_None;
    
    [self getMappingDictWithFieldType:fieldType
                        fieldTypeDict:&fieldTypeDict
                         subType1Dict:&subType1Dict
                      defaultSubType1:&defaultSubType1
                         subType2Dict:&subType2Dict];
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            // !! InstantMessage在exchange source中取不到欄位名稱
            
            NSString *key = [fieldTypeDict keyForIntValue:@(fieldType)];
            NSArray *valueArray = [contact valueForKey:key];
            
            NSString *keyForGetSubType1 = [subType2Dict keyForIntValue:@(WCABDC_FakeType_keyForGetSubType1)];
            NSString *keyForGetValue = [subType2Dict keyForIntValue:@(WCABDC_FakeType_keyForGetValue)];
            NSString *keyForGetAlternateValue = [subType2Dict keyForIntValue:@(WCABDC_FakeType_keyForGetAlternateValue)];
            
            for (CNLabeledValue *object in valueArray)
            {
                WCFieldModel *field = [[WCFieldModel alloc]init];
                
                if(field != nil)
                {
                    field.type = fieldType;
                    
                    
                    //////////////////////////////////////////////////
                    // set subType1
                    
                    NSString *objectLabel = [object.value valueForKey:keyForGetSubType1];
                    
                    if(objectLabel != nil)
                    {
                        field.subType1 = [[subType1Dict valueForKey:objectLabel] intValue];
                    }
                    
                    if(field.subType1 == WC_FST1_None)
                    {
                        field.subType1 = defaultSubType1;
                    }
                    
                    
                    //////////////////////////////////////////////////
                    // set value
                    
                    NSString *objectValue = [object.value valueForKey:keyForGetValue];
                    
                    if([objectValue length]>0)
                    {
                        field.value = objectValue;
                    }
                    
                    // if value is empty, set alternate value
                    if (keyForGetAlternateValue!=nil &&
                        [field.value length]==0)
                    {
                        NSString *objectValue = [object.value valueForKey:keyForGetAlternateValue];
                        
                        if([objectValue length]>0)
                        {
                            field.value = objectValue;
                        }
                    }
                    
                    // !! 如果是空字串，當成沒有值
                    if([field.value  length]==0)
                    {
                        field.value = nil;
                    }
                    //////////////////////////////////////////////////
                    
                    if([field isEmpty] == NO)
                    {
                        [cardModel addField:field];
                        result = YES;
                    }
                    
                    [field release];
                }
            }
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            NSMutableArray *fieldModels = [cardModel fieldArrayWithType:fieldType];
            
            if([fieldModels count] == 0)
            {
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                [contact setValue:@[] forKey:objectKey];
                
                result = YES;
                break;
            }
            
            NSMutableArray *labeledValueArray = [NSMutableArray array];
            
            if(labeledValueArray != nil)
            {
                for(int i=0; i<[fieldModels count]; i++)
                {
                    WCFieldModel *fieldModel = [fieldModels objectAtIndex:i];
                    
                    NSString *service = [subType1Dict keyForIntValue:@(fieldModel.subType1)];
                    NSString *username = fieldModel.value;
                    
                    //////////////////////////////////////////////////
                    // !! 檢查欄位上限
                    if([self canConvertWithInfo:convertFieldCount fieldType:fieldType subType1:fieldModel.subType1 subType2:WC_FST2_None]==NO)
                    {
                        continue;
                    }
                    
                    //////////////////////////////////////////////////
                    id value = nil;
                    if (fieldType==WC_FT_InstantMessage)
                    {
                        value = [[[CNInstantMessageAddress alloc] initWithUsername:username service:service] autorelease];
                    }
                    else
                    {
                        value = [[[CNSocialProfile alloc] initWithUrlString:nil
                                                                   username:username
                                                             userIdentifier:nil
                                                                    service:service] autorelease];
                    }
                    
                    CNLabeledValue *labeledValue = [CNLabeledValue labeledValueWithLabel:CNLabelOther value:value];
                    [labeledValueArray addObject:labeledValue];
                }
                
                //////////////////////////////////////////////////
                
                NSString *objectKey = [fieldTypeDict keyForIntValue:@(fieldType)];
                [contact setValue:labeledValueArray forKey:objectKey];
                result = YES;
            }
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
// 轉換Date+Birthday+Modification欄位
//================================================================================
- (BOOL)convertDateFieldWithCardModel:(WCABCardModel *)cardModel
                              contact:(CNContact *)contact
                          convertMode:(WCABDC_ConvertMode)convertMode
                    convertFieldCount:(NSMutableDictionary *)convertFieldCount
                                error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    
    switch (convertMode)
    {
        case ConvertMode_AB_to_WC:
        {
            do
            {
                //////////////////////////////////////////////////
                // 取得birthday並轉換成 WC_FT_Date
                
                if([self convertType1FieldWithCardModel:cardModel
                                                contact:contact
                                              fieldType:WCABDC_FakeFieldType_Birthday
                                            convertMode:convertMode
                                      convertFieldCount:convertFieldCount
                                                  error:&returnError] == YES)
                {
                    WCFieldModel *fieldModel = [[cardModel fieldArrayWithType:WCABDC_FakeFieldType_Birthday] firstObject];
                    
                    if(fieldModel != nil)
                    {
                        [fieldModel retain];
                        
                        fieldModel.type = WC_FT_Date;
                        fieldModel.subType1 = WC_FST1_Date_Birthday;
                        
                        [cardModel removeFieldWithType:WCABDC_FakeFieldType_Birthday index:0];
                        [cardModel addField:fieldModel];
                        
                        [fieldModel release];
                    }
                }
                
                
                //////////////////////////////////////////////////
                // 取得其他date資料
                
                [self convertType4FieldWithCardModel:cardModel
                                             contact:contact
                                           fieldType:WC_FT_Date
                                         convertMode:convertMode
                                   convertFieldCount:convertFieldCount
                                               error:&returnError];
                
                
                //////////////////////////////////////////////////
                // 取得Creation Date
                
                //
                //                NSDate *creationDate = ABRecordCopyValue(personRef, kABPersonCreationDateProperty);
                //
                //                if(creationDate != nil)
                //                {
                //                    [cardModel setCreatedTime:creationDate];
                //                    [creationDate release];
                //                }
                //
                //                //////////////////////////////////////////////////
                //                // 取得Modification Date
                //
                //                NSDate *modificationDate = ABRecordCopyValue(personRef, kABPersonModificationDateProperty);
                //
                //                if(modificationDate != nil)
                //                {
                //                    [cardModel setModifiedTime:modificationDate];
                //                    [modificationDate release];
                //                }
                
                
                result = YES;
            }
            while (0);
            
            break;
        }
            
        case ConvertMode_WC_to_AB:
        {
            // !! modification date不用寫入
            
            NSMutableArray *dateFieldModels = [cardModel fieldArrayWithType:WC_FT_Date];
            
            if([dateFieldModels count] > 0)
            {
                NSInteger birthdayFieldIndex = -1;
                WCFieldModel *birthdayFieldModel = nil;
                
                // find birthday field
                for(int i=0; i<[dateFieldModels count]; i++)
                {
                    WCFieldModel *fieldModel = [dateFieldModels objectAtIndex:i];
                    
                    if(fieldModel.subType1 == WC_FST1_Date_Birthday)
                    {
                        birthdayFieldModel = [fieldModel retain];
                        birthdayFieldIndex = i;
                        break;
                    }
                }
                
                if(birthdayFieldModel != nil)
                {
                    // build fake field for convert
                    [dateFieldModels removeObject:birthdayFieldModel];
                    birthdayFieldModel.type = WCABDC_FakeFieldType_Birthday;
                    [cardModel addField:birthdayFieldModel];
                    
                    // 處理birthday
                    result = [self convertType1FieldWithCardModel:cardModel
                                                          contact:contact
                                                        fieldType:WCABDC_FakeFieldType_Birthday
                                                      convertMode:convertMode
                                                convertFieldCount:convertFieldCount
                                                            error:&returnError];
                    
                    if(result == YES)
                    {
                        // 處理其他日期
                        result = [self convertType4FieldWithCardModel:cardModel
                                                              contact:contact
                                                            fieldType:WC_FT_Date
                                                          convertMode:convertMode
                                                    convertFieldCount:convertFieldCount
                                                                error:&returnError];
                    }
                    
                    // restore field
                    [cardModel removeField:birthdayFieldModel];
                    birthdayFieldModel.type = WC_FT_Date;
                    [cardModel insertField:birthdayFieldModel atIndex:birthdayFieldIndex];
                    [birthdayFieldModel release];
                }
                else
                {
                    // 處理其他日期
                    result = [self convertType4FieldWithCardModel:cardModel
                                                          contact:contact
                                                        fieldType:WC_FT_Date
                                                      convertMode:convertMode
                                                convertFieldCount:convertFieldCount
                                                            error:&returnError];
                }
            }
            else
            {
                // 沒有日期欄位
                [self convertType1FieldWithCardModel:cardModel
                                             contact:contact
                                           fieldType:WCABDC_FakeFieldType_Birthday
                                         convertMode:convertMode
                                   convertFieldCount:convertFieldCount
                                               error:&returnError];
                
                [self convertType4FieldWithCardModel:cardModel
                                             contact:contact
                                           fieldType:WC_FT_Date
                                         convertMode:convertMode
                                   convertFieldCount:convertFieldCount
                                               error:&returnError];
                result = YES;
            }
            
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//==============================================================================
//
//==============================================================================
- (void)convertDataWithMode:(WCABDC_ConvertMode)convertMode
                  cardModel:(WCABCardModel *)cardModel
          convertFieldCount:(NSMutableDictionary *)convertFieldCount
                    contact:(CNContact *)contact
                convertNote:(BOOL)convertNote
{
    // convert type2 field
    // name
    [self convertType2FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Name
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    // company
    [self convertType2FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Company
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    
    //////////////////////////////////////////////////
    // convert type3 field
    
    // address
    [self convertType3FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Address
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    
    //////////////////////////////////////////////////
    // convert type4 field
    
    // phone
    [self convertType4FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Phone
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    // email
    [self convertType4FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Email
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    // url
    [self convertType4FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_URL
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    
    //////////////////////////////////////////////////
    // convert type5 field
    
    // instance message
    [self convertType5FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_InstantMessage
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    // social network
    [self convertType5FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_SocialNetwork
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    //////////////////////////////////////////////////
    // convert type1 field
    
    // note
    if(convertNote)
    {
        [self convertType1FieldWithCardModel:cardModel
                                     contact:contact
                                   fieldType:WC_FT_Note
                                 convertMode:convertMode
                           convertFieldCount:convertFieldCount
                                       error:nil];
    }
    
    // nick name
    [self convertType1FieldWithCardModel:cardModel
                                 contact:contact
                               fieldType:WC_FT_Nickname
                             convertMode:convertMode
                       convertFieldCount:convertFieldCount
                                   error:nil];
    
    
    //////////////////////////////////////////////////
    // convert date field
    
    [self convertDateFieldWithCardModel:cardModel
                                contact:contact
                            convertMode:convertMode
                      convertFieldCount:convertFieldCount
                                  error:nil];
}

@end
