//
//  PPTextView.m
//  
//
//  Created by Mike on 13/5/7.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPTextView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPTextView()

@interface PPTextView()
- (CGRect)placeHolderLabelFrame;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPTextView(Notification)

@interface PPTextView(Notification)
- (void)textViewTextDidChangeNotification:(NSNotification *)notification;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPTextView

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize placeHolderLabel    = placeHolderLabel_;
@synthesize placeholder         = placeholder_;
@synthesize menuEnabled         = menuEnabled_;
@synthesize selectionEnabled    = selectionEnabled_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithCoder:(NSCoder *)aDecoder
{
    if(self=[super initWithCoder:aDecoder])
    {
        [self commonInit];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame
{
    if((self=[super initWithFrame:frame]))
    {
        [self commonInit];
    }
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UITextViewTextDidChangeNotification object:nil];

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [self.placeHolderLabel removeFromSuperview];
	[placeHolderLabel_ release];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
	[placeholder_ release];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [super dealloc];
}


//================================================================================
//
//================================================================================
- (void)commonInit
{
    menuEnabled_        = YES;
    selectionEnabled_   = YES;
    _placeholderStyle   = PPTextViewPlaceholderStyle_Center;
    
    placeHolderLabel_ = [[UILabel alloc] initWithFrame:[self placeHolderLabelFrame]];
    if(self.placeHolderLabel!=nil)
    {
        self.placeHolderLabel.alpha             = (([self.text length]>0)?0:1);
        self.placeHolderLabel.backgroundColor   = [UIColor clearColor];
        self.placeHolderLabel.font              = self.font;
        self.placeHolderLabel.lineBreakMode     = NSLineBreakByWordWrapping;
        self.placeHolderLabel.numberOfLines     = 0;
        self.placeHolderLabel.text              = self.placeholder;
        self.placeHolderLabel.textColor         = [UIColor lightGrayColor];
        [self addSubview:self.placeHolderLabel];
    }
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(textViewTextDidChangeNotification:)
                                                 name:UITextViewTextDidChangeNotification
                                               object:nil];
}


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Configuring the Bounds and Frame Rectangles


//================================================================================
//
//================================================================================
- (void)setFrame:(CGRect)frame
{
    [super setFrame:frame];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    if(self.placeHolderLabel!=nil) 
    { 
        self.placeHolderLabel.frame = [self placeHolderLabelFrame];
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (void)setText:(NSString *)text
{
    [super setText:text];
    
    //////////////////////////////////////////////////
    
    if(self.placeHolderLabel != nil &&
       [self.placeholder length] > 0 )
    {
        self.placeHolderLabel.alpha = (([self.text length]>0)?0:1);
    }
}


//================================================================================
//
//================================================================================
- (void)setContentInset:(UIEdgeInsets)contentInset
{
    contentInset.bottom = 0;
    
    [super setContentInset:contentInset];
}


//================================================================================
//
//================================================================================
- (void)setPlaceholder:(NSString *)placeholder
{
    [placeholder retain];
    [placeholder_ release];
    placeholder_ = placeholder;
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    if(self.placeHolderLabel!=nil)
    {
        self.placeHolderLabel.text = self.placeholder;
        self.placeHolderLabel.frame = [self placeHolderLabelFrame];

        if(self.text==nil || [self.text length]==0) 
        {
            self.placeHolderLabel.alpha = 1;
        }
        else 
        {
            self.placeHolderLabel.alpha = 0;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods


//================================================================================
//
//================================================================================
- (CGRect)placeHolderLabelFrame
{
    CGRect frame = self.bounds;
    
    frame.origin.x = PPTextViewDefaultGap;
    frame.size.width -= (2*PPTextViewDefaultGap);
    
    switch (self.placeholderStyle)
    {
        case PPTextViewPlaceholderStyle_Top:
        {
            frame.origin.y = PPTextViewDefaultGap;
            CGFloat height = ceilf([self.placeholder sizeWithFont:self.placeHolderLabel.font constrainedToSize:frame.size lineBreakMode:self.placeHolderLabel.lineBreakMode].height);
            frame.size.height = height;
            break;
        }
        case PPTextViewPlaceholderStyle_Bottom:
        {
            
            CGFloat height = ceilf([self.placeholder sizeWithFont:self.placeHolderLabel.font constrainedToSize:frame.size lineBreakMode:self.placeHolderLabel.lineBreakMode].height);
            frame.origin.y = frame.size.height - height - PPTextViewDefaultGap;
           frame.size.height = height;
            break;
        }
        case PPTextViewPlaceholderStyle_Center:
        default:
            break;
    }
    return frame;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification Methods


//================================================================================
//
//================================================================================
- (void)textViewTextDidChangeNotification:(NSNotification *)notification
{
    if( self==[notification object]     && 
        self.placeHolderLabel!=nil      && 
        [self.placeholder length]>0     )
    {
        self.placeHolderLabel.alpha = (([self.text length]>0)?0:1);
    }
    
    //////////////////////////////////////////////////
    // 修正iOS7 caret換行會超過顯示區域的問題
    
    if([[[UIDevice currentDevice] systemVersion] floatValue] < 8.0)
    {
        CGRect line = [self caretRectForPosition:self.selectedTextRange.start];
        CGFloat overflow = line.origin.y + line.size.height - (self.contentOffset.y + self.bounds.size.height - self.contentInset.bottom - self.contentInset.top);
        
        if (overflow > 0)
        {
            // We are at the bottom of the visible text and introduced
            // a line feed, scroll down (iOS 7 does not do it)
            // Scroll caret to visible area
            CGPoint offset = self.contentOffset;
            
            // +2只是為了保留下方空間比較好看
            offset.y += overflow+2;
            
            [self setContentOffset:offset];
        }
    }
}



//================================================================================
//
//================================================================================
-(BOOL)canPerformAction:(SEL)action withSender:(id)sender
{
    if (self.menuEnabled==YES)
    {
        // 如果不允許selection，select:進來時要return no
        if(self.selectionEnabled==NO)
        {
            if ([NSStringFromSelector(action) isEqualToString:@"select:"]||
                [NSStringFromSelector(action) isEqualToString:@"selectAll:"])
            {
                return NO;
            }
        }
		
        return [super canPerformAction:action withSender:sender];
	}
    
	return NO;
}

//================================================================================
//
//================================================================================
-(BOOL)gestureRecognizerShouldBegin:(UIGestureRecognizer *)gestureRecognizer
{
    if(self.selectionEnabled==YES)
    {
        return YES;
    }
    
    // Check for gestures to prevent
    if ([gestureRecognizer isKindOfClass:[UITapGestureRecognizer class]])
    {
        // Check for oneFingerTripleTap , oneFingerDoubleTap, twoFingerSingleTap
        if (((UITapGestureRecognizer *)gestureRecognizer).numberOfTapsRequired >= 2||((UITapGestureRecognizer *)gestureRecognizer).numberOfTouchesRequired>=2) {
            return NO;
        }
    }
    // check twoFingerRangedSelectGesture
    else if ([gestureRecognizer isKindOfClass:[UILongPressGestureRecognizer class]]&&((UILongPressGestureRecognizer *)gestureRecognizer).numberOfTouchesRequired>=2)
    {
        return NO;
    }
    else if ([NSStringFromClass([gestureRecognizer class]) isEqualToString:@"UITapAndAHalfRecognizer"])
    {
        return NO;
    }
    
    return YES;
}

@end
