//
//  PPPhoneTextFieldTableViewCell.m
//
//  Created by Howard.Lin on 13/9/30.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPPhoneTextFieldTableViewCell.h"

//controller
#import "PPCutPhoneNumberController.h"

//tool
#import "PPCountryCodeConvert.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPPhoneTextFieldTableViewCell()

@property (nonatomic,retain) NSCharacterSet  *availableCharSet;
@property (nonatomic,assign) NSInteger       cursorPosition;
@property (nonatomic,retain) NSMutableString *charsAfterCursor;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPPhoneTextFieldTableViewCell(Notification)

@interface PPPhoneTextFieldTableViewCell(Notification)
- (void)textFieldTextDidChangeNotification:(NSNotification *)notification;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPPhoneTextFieldTableViewCell


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects


//================================================================================
//
//================================================================================
- (id)initWithStyle:(UITableViewCellStyle)style reuseIdentifier:(NSString *)reuseIdentifier
{
    if((self = [super initWithStyle:style reuseIdentifier:reuseIdentifier]))
    {
        self.selectionStyle = UITableViewCellSelectionStyleNone;
        
        //////////////////////////////////////////////////
        
        _availableCharSet = [[NSCharacterSet characterSetWithCharactersInString:@"0123456789*#+,;"] retain];

        _cursorPosition = -1;
        
        _countryCode = [[PPCountryCodeConvert defaultCountryCode] retain];
        
        //////////////////////////////////////////////////
        
        //預設輸入上限最高
        _maxLength = NSUIntegerMax;
        
        //////////////////////////////////////////////////
        
        _textField = [[CustomRectTextField alloc] init];
        if(self.textField!=nil)
        {
            self.textField.autocapitalizationType   = UITextAutocapitalizationTypeWords;
            self.textField.autocorrectionType       = UITextAutocorrectionTypeNo;
            self.textField.backgroundColor          = [UIColor clearColor];
            self.textField.borderStyle              = UITextBorderStyleNone;
            self.textField.clearButtonMode          = UITextFieldViewModeWhileEditing;
            self.textField.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
            self.textField.delegate                 = self;
//            self.textField.returnKeyType            = UIReturnKeyDone;
            
            [self.contentView addSubview:self.textField];
            
            //////////////////////////////////////////////////
            
            if(self.detailTextLabel!=nil)
            {
                self.textField.textColor = self.detailTextLabel.textColor;
            }
        }
        
        //////////////////////////////////////////////////
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(textFieldTextDidChangeNotification:)
                                                     name:UITextFieldTextDidChangeNotification
                                                   object:nil];
        
        [self layoutSubviews];
    
    }
    
    return self;
}

//================================================================================
//
//================================================================================
- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [_availableCharSet release];
    _availableCharSet = nil;

    [_charsAfterCursor release];
    _charsAfterCursor = nil;
    
    [_countryCode release];
    _countryCode = nil;
    
    [_textField removeFromSuperview];
    [_textField release];
	_textField = nil;
    
    //////////////////////////////////////////////////
    
	[super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Laying out Subviews

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
	[super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    CGRect textFieldFrame = self.contentView.bounds;
    
    //////////////////////////////////////////////////
    
    if(self.textField!=nil)
    {
        switch(self.style)
        {
            case UITableViewCellStyleValue1:
            case UITableViewCellStyleValue2:
            {
                break;
            }
            case UITableViewCellStyleSubtitle:
            {
                CGRect textLabelFrame = self.contentView.bounds;
                
                textLabelFrame.size.height  = (self.contentView.bounds.size.height-(PPTableViewCellDefaultVerticalGap+self.contentInsets.top+self.contentInsets.bottom))/2;
                self.textLabel.frame = textLabelFrame;
                
                
                textFieldFrame.origin.x     = textLabelFrame.origin.x;
                textFieldFrame.origin.y     = textLabelFrame.origin.y+textLabelFrame.size.height+PPTableViewCellDefaultVerticalGap;
                textFieldFrame.size.width   = textLabelFrame.size.width;
                textFieldFrame.size.height  = textLabelFrame.size.height;
                
                break;
            }
            case UITableViewCellStyleDefault:
            default:
            {
                textFieldFrame.origin.x     = self.contentInsets.left;
                textFieldFrame.origin.y     = self.contentInsets.top;
                textFieldFrame.size.width   -= (self.contentInsets.left+self.contentInsets.right);
                textFieldFrame.size.height  -= (self.contentInsets.top+self.contentInsets.bottom);
                
                break;
            }
        }
        
        self.textField.frame = textFieldFrame;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private methods

//===============================================================================
// 設定text後游標會移到最後，利用select的方式移回指定的位置
//===============================================================================
- (void)moveCursorWithOffset:(NSInteger)offset
{
    UITextPosition *basePosition  = [self.textField selectedTextRange].start;
    UITextPosition *beginPosition = [self.textField positionFromPosition:basePosition offset:offset];
    UITextPosition *endPosition   = [self.textField positionFromPosition:basePosition offset:offset];

    if(beginPosition && endPosition)
        self.textField.selectedTextRange  = [self.textField textRangeFromPosition:beginPosition toPosition:endPosition];
    else self.textField.selectedTextRange = [self.textField textRangeFromPosition:basePosition toPosition:basePosition];
}


//===============================================================================
// 取得移動cursor所需的參考資料
//===============================================================================
- (void)getCursorReferenceData:(NSString *)newText
{
    @autoreleasepool
    {
        self.charsAfterCursor = nil;
        
        _charsAfterCursor = [[NSMutableString alloc] init];

        //////////////////////////////////////////////////
        
        for(NSInteger i=self.cursorPosition+1; i<[newText length]; i++)
        {
            unichar oneChar = [newText characterAtIndex:i];
            
            if([self.availableCharSet characterIsMember:oneChar])
            {
                [_charsAfterCursor appendString:[NSString stringWithCharacters:&oneChar length:1]];
            }
        }
        
        if([self.charsAfterCursor length] == 0)
        {
            self.charsAfterCursor = nil;
        }
    }
}


//===============================================================================
// 還原游標位置
//===============================================================================
- (void)restoreCursorPosition:(UITextField *)textField
{
    // !! special case : 第一個字一定往後移
    if(self.cursorPosition == 0)
    {
        [self moveCursorWithOffset:(0 - [textField.text length]+1)];
        return;
    }
    
    
    NSMutableString *pureChars = [[NSMutableString alloc] init];
    NSMutableArray *charPositions = [[NSMutableArray alloc] init];
    NSInteger newPosition = -1;
    
    //////////////////////////////////////////////////

    NSCharacterSet *availableCharSet = self.availableCharSet;
    
    //////////////////////////////////////////////////

    //-------------------------------
    // 取得每個字元的位置
    //-------------------------------
    for(int i=0; i<[textField.text length]; i++)
    {
        unichar oneChar = [textField.text characterAtIndex:i];
        
        if([availableCharSet characterIsMember:oneChar])
        {
            [pureChars appendString:[NSString stringWithCharacters:&oneChar length:1]];
            
            //////////////////////////////////////////////////

            [charPositions addObject:[NSNumber numberWithInt:i]];
        }
    }
 
    //-------------------------------
    // 移動游標到新位置
    //-------------------------------
    if(self.charsAfterCursor!=nil)
    {
        NSRange range = [pureChars rangeOfString:self.charsAfterCursor options:NSBackwardsSearch];

        if(range.length!=0)
        {
            // 游標移到前一個字元之後，如果是在最前面，newPosition必須是-1。
            if(range.location>0)
            {
                newPosition = [[charPositions objectAtIndex:(range.location-1)] intValue];
            }
        }
        
        self.charsAfterCursor = nil;
    }
    else
    {
        // !! 把游標移到最後一個數字
        newPosition = [[charPositions lastObject] intValue];
    }
    
    [self moveCursorWithOffset:(newPosition - [textField.text length]+1)];
    
    [charPositions release];
    [pureChars release];
  
}


//===============================================================================
//
//===============================================================================
- (void)updateInputText:(NSString *)text
{
    [self.textField setText:text];
    
    CGFloat osVer = [[UIDevice currentDevice].systemVersion floatValue];
    
    if(osVer >= 5.0)
    {
        [self restoreCursorPosition:self.textField];
    }

    [self textFieldTextDidChange];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification Methods

//================================================================================
//
//================================================================================
- (void)textFieldTextDidChangeNotification:(NSNotification*)notification
{
    if(self.textField==[notification object] && self.textField.text.length<=self.maxLength)
    {
        [self textFieldTextDidChange];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Intended For Subclasses Method

//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldBeginEditing
{
    BOOL shouldBeginEditing = YES;
    
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellShouldBeginEditing:)]==YES)
    {
        shouldBeginEditing = [self.delegate ppPhoneTextFieldTableViewCellShouldBeginEditing:self];
    }
    
    return shouldBeginEditing;

}


//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldReturn
{
    BOOL shouldReturn = YES;
    
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellShouldReturn:)]==YES)
    {
        shouldReturn = [self.delegate ppPhoneTextFieldTableViewCellShouldReturn:self];
    }
    
    return shouldReturn;
}


//================================================================================
//
//================================================================================
- (void)textFieldDidBeginEditing
{
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellDidBeginEditing:)]==YES)
    {
        [self.delegate ppPhoneTextFieldTableViewCellDidBeginEditing:self];
    }
}


//================================================================================
//
//================================================================================
- (void)textFieldTextDidChange
{
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellDidChanged:)]==YES)
    {
        [self.delegate ppPhoneTextFieldTableViewCellDidChanged:self];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldEndEditing
{
    BOOL result = YES;
    
    if(self.delegate!=nil &&
       [self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellShouldEndEditing:)]==YES)
    {
        result = [self.delegate ppPhoneTextFieldTableViewCellShouldEndEditing:self];
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)textFieldDidEndEditing
{
    if([self.delegate respondsToSelector:@selector(ppPhoneTextFieldTableViewCellDidEndEditing:)]==YES)
    {
        [self.delegate ppPhoneTextFieldTableViewCellDidEndEditing:self];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITextFieldDelegate

//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldBeginEditing:(UITextField *)textField
{
    return [self textFieldShouldBeginEditing];
}


//================================================================================
//
//================================================================================
- (BOOL)textField:(UITextField *)textField shouldChangeCharactersInRange:(NSRange)range replacementString:(NSString *)string
{
    BOOL result = YES;
    
    // 文字超過上限時
    if(self.textField.text.length+string.length>self.maxLength)
    {
        // 新增文字，或取代文字
        if(range.length<=string.length &&
           [string length]>0)
        {
            result = NO;
        }
    }
    else
    {
        do
        {
            if(self.autoFormat==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            //特殊情況是刪除時遇到"-",不做 autoFormat
            
            else if(![string length] &&
                    range.length==1 &&
                    [[textField.text substringWithRange:range] compare:@"-"]==NSOrderedSame)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *newText = [textField.text stringByReplacingCharactersInRange:range withString:string];
            
            //////////////////////////////////////////////////
            
            PPCutPhoneNumberController  *ppCutPhoneNumberUt;
            
            ppCutPhoneNumberUt    = [[PPCutPhoneNumberController alloc] init];
            
            NSString *countryCode = self.countryCode;
            
            if(countryCode==nil)
            {
                countryCode = [PPCountryCodeConvert defaultCountryCode];
            }
            
            /////////////////////////////////////////////
            
            NSString *formatedPhoneString = [[ppCutPhoneNumberUt copyFormattedPhoneNumberWithoutExtension:newText
                                                                                          withCountryCode:[ppCutPhoneNumberUt getPhoneFormatTypeFromCountryCode:countryCode]] autorelease];
            
            [ppCutPhoneNumberUt release];
            
            //////////////////////////////////////////////////
            
            //當格式化字串與顯示字串不一致時，才需要做游標計算
            if([formatedPhoneString compare:newText]==NSOrderedSame)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            // 刪除時不是遇到"-"
            if(![string length] &&
               range.length==1)
            {
                self.cursorPosition = range.location - range.length;
            }
            // 新增
            else
            {
                self.cursorPosition = range.location;
            }
            
            //////////////////////////////////////////////////
            
            [self getCursorReferenceData:newText];
            
            //////////////////////////////////////////////////
            
            [self updateInputText:formatedPhoneString];
            
            //////////////////////////////////////////////////
            
            result = NO;
            
        }
        while (0);
    }
  
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
    BOOL shouldReturn = [self textFieldShouldReturn];
    
    if(shouldReturn==YES)
    {
        [textField resignFirstResponder];
    }
    
    return shouldReturn;
}


//================================================================================
//
//================================================================================
- (void)textFieldDidBeginEditing:(UITextField *)textField
{
    if(self.textField==textField && self.textField.text.length<=self.maxLength)
    {
        [self textFieldDidBeginEditing];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)textFieldShouldEndEditing:(UITextField *)textField
{
    return [self textFieldShouldEndEditing];
}


//================================================================================
//
//================================================================================
- (void)textFieldDidEndEditing:(UITextField *)textField
{
    [self textFieldDidEndEditing];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - subclass


//==============================================================================
//
//==============================================================================
- (void)setEnable:(BOOL)enable
{
    [super setEnable:enable];
    
    self.textField.userInteractionEnabled = enable;
    self.textField.enabled = enable;
}
@end
