//
//  PPSyncRecordController.m
//  
//
//  Created by Mike Shih on 12/03/15.
//  Copyright (c) 2011年 Penpower. All rights reserved.
//

#import "PPSyncRecordController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSyncRecordTable_Groups    @"Groups"
#define PPSyncRecordTable_Cards     @"Cards"

#define PPSyncRecordColumn_LocalID  @"LocalID"
#define PPSyncRecordColumn_RemoteID @"RemoteID"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Macro Command

#define PPSyncRecord_AlterTable_V1_0ToV2_0(tableName)                                                       \
[NSString stringWithFormat:@"ALTER TABLE %@ ADD COLUMN Version INTEGER DEFAULT 1;", tableName]

#define PPSyncRecord_Count(tableName, clauses)                                                        		\
[NSString stringWithFormat:@"SELECT COUNT(*) FROM %@ %@", tableName, clauses]

#define PPSyncRecord_CreateIndex_LocalIDWithRemoteID(tableName)                                             \
[NSString stringWithFormat:@"CREATE INDEX IF NOT EXISTS %@_LocalIDWithRemoteID ON %@(LocalID, RemoteID);", tableName, tableName]

#define PPSyncRecord_CreateTable(tableName)                                                                 \
[NSString stringWithFormat:@"CREATE TABLE IF NOT EXISTS %@(Row INTEGER PRIMARY KEY AUTOINCREMENT, LocalID TEXT UNIQUE, LocalModifiedTime double, RemoteID TEXT UNIQUE, RemoteModifiedTime double, Version INTEGER DEFAULT 1);", tableName]
//Version 1.0
//[NSString stringWithFormat:@"CREATE TABLE IF NOT EXISTS %@(Row INTEGER PRIMARY KEY AUTOINCREMENT, LocalID TEXT UNIQUE, LocalModifiedTime double, RemoteID TEXT UNIQUE, RemoteModifiedTime double);", tableName]

#define PPSyncRecord_Delete(tableName, clauses)                                                             \
[NSString stringWithFormat:@"DELETE FROM %@ %@", tableName, clauses]

#define PPSyncRecord_Insert(tableName, localID, localModifiedTime, remoteID, remoteModifiedTime, version)   \
[NSString stringWithFormat:@"INSERT INTO %@(LocalID, LocalModifiedTime, RemoteID, RemoteModifiedTime, Version) VALUES('%@', %f, '%@', %f, %lu);", tableName, localID, localModifiedTime, remoteID, remoteModifiedTime, (unsigned long)version]

#define PPSyncRecord_Select(tableName, clauses)                                                             \
[NSString stringWithFormat:@"SELECT LocalID, LocalModifiedTime, RemoteID, RemoteModifiedTime, Version FROM %@ %@", tableName, clauses]

#define PPSyncRecord_Update(tableName, localID, localModifiedTime, remoteID, remoteModifiedTime, version)   \
[NSString stringWithFormat:@"UPDATE %@ SET LocalModifiedTime=%f, RemoteModifiedTime=%f WHERE LocalID='%@' AND RemoteID='%@' AND Version=%lu;", tableName, localModifiedTime, remoteModifiedTime, localID, remoteID, (unsigned long)version]

#define PPSyncRecord_UpdateLocalID(tableName, oldLocalID, newLocalID)   \
[NSString stringWithFormat:@"UPDATE %@ SET LocalID='%@' WHERE LocalID='%@';", tableName, newLocalID, oldLocalID]

#define PPSyncRecord_UpdateLocalModifiedTime(tableName, localID, localModifiedTime)   \
[NSString stringWithFormat:@"UPDATE %@ SET LocalModifiedTime=%f WHERE LocalID='%@';", tableName, localModifiedTime, localID]

#define PPSyncRecord_UpdateRemoteModifiedTime(tableName, remoteID, remoteModifiedTime)   \
[NSString stringWithFormat:@"UPDATE %@ SET RemoteModifiedTime=%f WHERE RemoteID='%@';", tableName, remoteModifiedTime, remoteID]


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSyncRecordController(Private)

@interface PPSyncRecordController(Private)
- (BOOL)deleteSyncRecordWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error;
- (BOOL)insertSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel tableName:(NSString *)tableName error:(NSError **)error;
- (BOOL)isFindSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error;
- (BOOL)updateSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel tableName:(NSString *)tableName error:(NSError **)error;
- (PPSyncRecordModel *)syncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error;
- (NSMutableArray *)syncRecordModelsWithTableName:(NSString *)tableName clauses:(NSString *)clauses error:(NSError **)error;
- (NSString *)columnNameWithSyncRecordType:(PPSyncRecordType)syncRecordType;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSyncRecordController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithFileName:(NSString *)fileName atPath:(NSString *)path error:(NSError **)error
{
    id object = nil;
    
    //////////////////////////////////////////////////
    
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        self = [super initWithFileName:fileName atPath:path error:&returnError];
        if(self==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([super isExist]==YES)
        {
            //資料庫檔案存在,檢查版本
            
            if([self openWithError:&returnError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            //取得版號
            
            int major=0, minor=0;
            
            if([self getVersionMajor:&major minor:&minor error:&returnError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            //檢查版本是否需要升級
            
            if(major!=PPSyncRecordVersionMajor_Current || minor!=PPSyncRecordVersionMinor_Current)
            {
                switch(major)
                {
                    case PPSyncRecordVersionMajor_1:
                    {
                        switch(minor)
                        {
                            case PPSyncRecordVersionMinor_0:
                            {
                                if([self openWithError:&returnError]==NO)
                                {
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                
                                NSArray *commands = @[PPSyncRecord_AlterTable_V1_0ToV2_0(PPSyncRecordTable_Groups),
                                                      PPSyncRecord_AlterTable_V1_0ToV2_0(PPSyncRecordTable_Cards)];
                                if(commands==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                
                                if([self beginTransactionWithError:&returnError]==NO)
                                {
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                
                                for(NSString *command in commands)
                                {
                                    if([self runCommand:command error:&returnError]==NO)
                                    {
                                        break;
                                    }
                                }
                                
                                //////////////////////////////////////////////////
                                
                                if(returnError==originError)
                                {
                                    [self endTransactionWithError:&returnError];
                                }
                                else
                                {
                                    [self rollbackTransactionWithError:&returnError];
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                
                                if([self setVersionMajor:PPSyncRecordVersionMajor_Current minor:PPSyncRecordVersionMinor_Current error:&returnError]==NO)
                                {
                                    break;
                                }
                                
                                break;
                            }
                            default:
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                        }
                        
                        break;
                    }
                    default:
                    {
                        returnError = PPErrorOperationFailed(returnError);
                        break;
                    }
                }
            }
        }
        else
        {
            //建立新檔案
            
            NSArray *createCommands = @[PPSyncRecord_CreateTable(PPSyncRecordTable_Groups),
                                        PPSyncRecord_CreateIndex_LocalIDWithRemoteID(PPSyncRecordTable_Groups),
                                        PPSyncRecord_CreateTable(PPSyncRecordTable_Cards),
                                        PPSyncRecord_CreateIndex_LocalIDWithRemoteID(PPSyncRecordTable_Cards)];
            
            if([self createWithCommands:createCommands error:&returnError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self openWithError:&returnError]==NO)
            {
                break;
            }

            //////////////////////////////////////////////////
            
            if([self setVersionMajor:PPSyncRecordVersionMajor_Current minor:PPSyncRecordVersionMinor_Current error:&returnError]==NO)
            {
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        object = self;
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    if(object==nil)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}

//////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)deleteAllSyncRecordWithError:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        NSMutableArray *commands = [[[NSMutableArray alloc] init] autorelease];
        if(commands==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        [commands addObject:PPSyncRecord_Delete(PPSyncRecordTable_Groups, @"")];
        [commands addObject:PPSyncRecord_Delete(PPSyncRecordTable_Cards, @"")];
        
        //////////////////////////////////////////////////
        
        if([self beginTransactionWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(NSString *command in commands)
        {
            if([self runCommand:command error:&returnError]==NO)
            {
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        if(returnError==originError)
        {
            [self endTransactionWithError:&returnError];
        }
        else
        {
            [self rollbackTransactionWithError:&returnError];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)deleteGroupSyncRecordWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self deleteSyncRecordWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)deleteCardSyncRecordWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self deleteSyncRecordWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Cards error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)insertGroupSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel error:(NSError **)error
{
    return [self insertSyncRecordWithSyncRecordModel:syncRecordModel tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)insertCardSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel error:(NSError **)error
{
    return [self insertSyncRecordWithSyncRecordModel:syncRecordModel tableName:PPSyncRecordTable_Cards error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)isFindGroupSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self isFindSyncRecordModelWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)isFindCardSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self isFindSyncRecordModelWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Cards error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)updateGroupSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel error:(NSError **)error
{
    return [self updateSyncRecordWithSyncRecordModel:syncRecordModel tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)updateCardSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel error:(NSError **)error
{
    return [self updateSyncRecordWithSyncRecordModel:syncRecordModel tableName:PPSyncRecordTable_Cards error:error];
}

//================================================================================
//
//================================================================================
- (PPSyncRecordModel *)groupSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self syncRecordModelWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (PPSyncRecordModel *)cardSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self syncRecordModelWithUniqueID:uniqueID syncRecordType:syncRecordType tableName:PPSyncRecordTable_Cards error:error];
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)groupSyncRecordModelsWithError:(NSError **)error
{
    return [self syncRecordModelsWithTableName:PPSyncRecordTable_Groups clauses:@"" error:error];
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)groupSyncRecordModelsOfLessThanVersion:(NSUInteger)version error:(NSError **)error
{
    return [self syncRecordModelsWithTableName:PPSyncRecordTable_Groups clauses:[NSString stringWithFormat:@"WHERE version<%lu", (unsigned long)version] error:error];
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)cardSyncRecordModelsWithError:(NSError **)error
{
    return [self syncRecordModelsWithTableName:PPSyncRecordTable_Cards clauses:@"" error:error];
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)cardSyncRecordModelsOfLessThanVersion:(NSUInteger)version error:(NSError **)error
{
    return [self syncRecordModelsWithTableName:PPSyncRecordTable_Cards clauses:[NSString stringWithFormat:@"WHERE version<%lu", (unsigned long)version] error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)updateGroupSyncRecordWithOldLocalID:(NSString *)oldLocalID newLocalID:(NSString *)newLocalID error:(NSError **)error
{
    return [self updateSyncRecordWithOldLocalID:oldLocalID newLocalID:newLocalID tableName:PPSyncRecordTable_Groups error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)updateGroupSyncRecordWithUniqueID:(NSString *)uniqueID modifiedTime:(NSDate *)modifiedTime syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self updateSyncRecordWithUniqueID:uniqueID
                                 modifiedTime:modifiedTime
                               syncRecordType:syncRecordType
                                    tableName:PPSyncRecordTable_Groups
                                        error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)updateCardSyncRecordWithUniqueID:(NSString *)uniqueID modifiedTime:(NSDate *)modifiedTime syncRecordType:(PPSyncRecordType)syncRecordType error:(NSError **)error
{
    return [self updateSyncRecordWithUniqueID:uniqueID
                                 modifiedTime:modifiedTime
                               syncRecordType:syncRecordType
                                    tableName:PPSyncRecordTable_Cards
                                        error:error];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

//================================================================================
//
//================================================================================
- (BOOL)deleteSyncRecordWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do 
    {
       	if(uniqueID==nil || [uniqueID length]==0 || tableName==nil || [tableName length]==0)	
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command = PPSyncRecord_Delete(tableName, ([NSString stringWithFormat:@"WHERE %@='%@'", [self columnNameWithSyncRecordType:syncRecordType], uniqueID]));
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self runCommand:command error:&returnError];
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)insertSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel tableName:(NSString *)tableName error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do 
    {
        if(syncRecordModel==nil || tableName==nil || [tableName length]==0)		
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command = PPSyncRecord_Insert(tableName,
                                                syncRecordModel.localSyncCompareModel.uniqueID,
                                                [syncRecordModel.localSyncCompareModel.lastModifiedDate timeIntervalSince1970],
                                                syncRecordModel.remoteSyncCompareModel.uniqueID,
                                                [syncRecordModel.remoteSyncCompareModel.lastModifiedDate timeIntervalSince1970],
                                                syncRecordModel.version);
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self runCommand:command error:&returnError];
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)isFindSyncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error
{
    BOOL    result          = NO;
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do 
    {
        if(uniqueID==nil || [uniqueID length]==0 || tableName==nil || [tableName length]==0)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command = PPSyncRecord_Count(tableName, ([NSString stringWithFormat:@"WHERE %@='%@'", [self columnNameWithSyncRecordType:syncRecordType], uniqueID]));
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = ([self recordCountWithCommand:command error:&returnError]>0);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return result;
}

//================================================================================
//
//================================================================================
- (BOOL)updateSyncRecordWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel tableName:(NSString *)tableName error:(NSError **)error
{
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do 
    {
        if(syncRecordModel==nil || tableName==nil || [tableName length]==0)		
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command = PPSyncRecord_Update(tableName,
                                                syncRecordModel.localSyncCompareModel.uniqueID,
                                                [syncRecordModel.localSyncCompareModel.lastModifiedDate timeIntervalSince1970],
                                                syncRecordModel.remoteSyncCompareModel.uniqueID,
                                                [syncRecordModel.remoteSyncCompareModel.lastModifiedDate timeIntervalSince1970],
                                                syncRecordModel.version);
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self runCommand:command error:&returnError];
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)updateSyncRecordWithOldLocalID:(NSString *)oldLocalID newLocalID:(NSString *)newLocalID tableName:(NSString *)tableName error:(NSError **)error
{
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if([oldLocalID length]==0 || [newLocalID length]==0 || [tableName length]==0)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command = PPSyncRecord_UpdateLocalID(tableName, oldLocalID, newLocalID);
        
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self runCommand:command error:&returnError];
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)updateSyncRecordWithUniqueID:(NSString *)uniqueID modifiedTime:(NSDate *)modifiedTime syncRecordType:(PPSyncRecordType)syncRecordType tableName:tableName error:(NSError **)error
{
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if([uniqueID length]==0 || modifiedTime==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *command;
        
        switch (syncRecordType)
        {
            case PPSyncRecordType_Local:
                command = PPSyncRecord_UpdateLocalModifiedTime(tableName, uniqueID, [modifiedTime timeIntervalSince1970]);
                break;

            case PPSyncRecordType_Remote:
                command = PPSyncRecord_UpdateRemoteModifiedTime(tableName, uniqueID, [modifiedTime timeIntervalSince1970]);
                break;

            default:
                NSAssert(NO, @"Invalid type");
                break;
        }
        
        if(command==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self runCommand:command error:&returnError];
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (PPSyncRecordModel *)syncRecordModelWithUniqueID:(NSString *)uniqueID syncRecordType:(PPSyncRecordType)syncRecordType tableName:(NSString *)tableName error:(NSError **)error
{
    PPSyncRecordModel   *syncRecordModel    = nil;
    NSError             *originError        = ((error!=nil)?*error:nil);
    NSError             *returnError        = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        do
        {
            if(uniqueID==nil || [uniqueID length]==0 || tableName==nil || [tableName length]==0)
            {
                returnError = PPErrorParameterInvalidity(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *command = PPSyncRecord_Select(tableName, ([NSString stringWithFormat:@"WHERE %@='%@'", [self columnNameWithSyncRecordType:syncRecordType], uniqueID]));
            if(command==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self openWithError:&returnError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            sqlite3_stmt *statement = NULL;
            
            if(sqlite3_prepare_v2(self.dbHandle, [command UTF8String], -1, &statement, NULL)!=SQLITE_OK)
            {
                returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            while(sqlite3_step(statement)==SQLITE_ROW)
            {
                PPSyncCompareModel *localSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
                if(localSyncCompareModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                localSyncCompareModel.uniqueID          = [PPSQLiteController stringFromStatement:statement column:0];
                localSyncCompareModel.lastModifiedDate  = [NSDate dateWithTimeIntervalSince1970:sqlite3_column_double(statement, 1)];
                
                //////////////////////////////////////////////////
                
                PPSyncCompareModel *remoteSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
                if(remoteSyncCompareModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                remoteSyncCompareModel.uniqueID         = [PPSQLiteController stringFromStatement:statement column:2];
                remoteSyncCompareModel.lastModifiedDate = [NSDate dateWithTimeIntervalSince1970:sqlite3_column_double(statement, 3)];
                
                //////////////////////////////////////////////////
                
                syncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                if(syncRecordModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                syncRecordModel.localSyncCompareModel   = localSyncCompareModel;
                syncRecordModel.remoteSyncCompareModel  = remoteSyncCompareModel;
                syncRecordModel.version                 = [PPSQLiteController integerFromStatement:statement column:4];
            }
            
            //////////////////////////////////////////////////
            
            if(sqlite3_finalize(statement)!=SQLITE_OK)
            {
                returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(returnError!=originError)
        {
            syncRecordModel = nil;
        }
        else
        {
            //沒錯誤要保留syncRecordModel
            [syncRecordModel retain];
        }
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    if(syncRecordModel!=nil)
    {
        [syncRecordModel autorelease];
    }
    
    //////////////////////////////////////////////////
    
    return syncRecordModel;
}

//================================================================================
//
//================================================================================
- (NSMutableArray *)syncRecordModelsWithTableName:(NSString *)tableName clauses:(NSString *)clauses error:(NSError **)error
{
    NSMutableArray  *syncRecordModels   = nil;
    NSError         *originError        = ((error!=nil)?*error:nil);
    NSError         *returnError        = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        do
        {
            if(tableName==nil || [tableName length]==0)
            {
                returnError = PPErrorParameterInvalidity(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *command = PPSyncRecord_Select(tableName, clauses);
            if(command==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self openWithError:&returnError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            sqlite3_stmt *statement = NULL;
            
            if(sqlite3_prepare_v2(self.dbHandle, [command UTF8String], -1, &statement, NULL)!=SQLITE_OK)
            {
                returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            syncRecordModels = [[[NSMutableArray alloc] init] autorelease];
            if(syncRecordModels==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            while(sqlite3_step(statement)==SQLITE_ROW)
            {
                PPSyncCompareModel *localSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
                if(localSyncCompareModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                localSyncCompareModel.uniqueID          = [PPSQLiteController stringFromStatement:statement column:0];
                localSyncCompareModel.lastModifiedDate  = [NSDate dateWithTimeIntervalSince1970:sqlite3_column_double(statement, 1)];
                
                //////////////////////////////////////////////////
                
                PPSyncCompareModel *remoteSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
                if(remoteSyncCompareModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                remoteSyncCompareModel.uniqueID         = [PPSQLiteController stringFromStatement:statement column:2];
                remoteSyncCompareModel.lastModifiedDate = [NSDate dateWithTimeIntervalSince1970:sqlite3_column_double(statement, 3)];
                
                //////////////////////////////////////////////////
                
                PPSyncRecordModel *syncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                if(syncRecordModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                syncRecordModel.localSyncCompareModel   = localSyncCompareModel;
                syncRecordModel.remoteSyncCompareModel  = remoteSyncCompareModel;
                syncRecordModel.version                 = [PPSQLiteController integerFromStatement:statement column:4];
                
                //////////////////////////////////////////////////
                
                [syncRecordModels addObject:syncRecordModel];
            }
            
            //////////////////////////////////////////////////
            
            if(sqlite3_finalize(statement)!=SQLITE_OK)
            {
                returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(returnError!=originError)
        {
            syncRecordModels = nil;
        }
        else
        {
            //沒錯誤要保留syncRecordModels
            [syncRecordModels retain];
        }
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    if(syncRecordModels!=nil)
    {
        [syncRecordModels autorelease];
    }
    
    //////////////////////////////////////////////////
    
    return syncRecordModels;
}

//================================================================================
//
//================================================================================
- (NSString *)columnNameWithSyncRecordType:(PPSyncRecordType)syncRecordType
{
    NSString *columnName = nil;
    
    switch(syncRecordType) 
    {
        case PPSyncRecordType_Local:
        {
            columnName = PPSyncRecordColumn_LocalID;

            break;
        }
        case PPSyncRecordType_Remote:
        {
            columnName = PPSyncRecordColumn_RemoteID;
            
            break;
        }
        default:
        {
            break;
        }
    }

    return columnName;
}

@end
