//
//  PPSyncCardModel.m
//  
//
//  Created by Mike Shih on 12/3/19.
//  Copyright (c) 2012年 Penpower. All rights reserved.
//

#import "PPSyncCardModel.h"
#import "NSMutableDictionary+NSInteger.h"

@interface PPSyncCardModel ()
@property(nonatomic, retain, readwrite)  NSMutableDictionary *fieldArrayDict;
@property(nonatomic, retain, readwrite)  NSMutableDictionary *imageDict;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPSyncCardModel






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects


//==============================================================================
//
//==============================================================================
- (id)initWithSyncCardModel:(PPSyncCardModel *)syncCardModel
{
    self = [super init];
    
	if(self)
	{
        self.uniqueID           = [[[NSString alloc] initWithString:syncCardModel.uniqueID] autorelease];
        self.lastModifiedDate   = [[[NSDate alloc] initWithTimeInterval:0 sinceDate:syncCardModel.lastModifiedDate] autorelease];
        self.needForceUpdate    = syncCardModel.needForceUpdate;
        self.deleted            = syncCardModel.deleted;
        
        //////////////////////////////////////////////////

        [self appendFieldsFromSyncCardModel:syncCardModel];
        [self updateImageDictFromSyncCardModel:syncCardModel];
	}
	
	return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.fieldArrayDict = nil;
    self.imageDict = nil;
    self.extendedPropertyDict = nil;

    //////////////////////////////////////////////////
	[super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - NSCopying



//==============================================================================
//
//==============================================================================
- (id)copyWithZone:(nullable NSZone *)zone
{
    PPSyncCardModel *syncCardModel = [super copyWithZone:zone];
    
    syncCardModel.fieldArrayDict = [[self.fieldArrayDict mutableCopyWithZone:zone] autorelease];
    syncCardModel.imageDict = [[self.imageDict mutableCopyWithZone:zone] autorelease];
    syncCardModel.extendedPropertyDict = [[self.extendedPropertyDict mutableCopyWithZone:zone] autorelease];

    return syncCardModel;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods


//===============================================================================
// check if having field data
//===============================================================================
- (BOOL)hasSyncFieldModels
{
    return ([self.fieldArrayDict count] != 0);
}


//===============================================================================
// get field array by specified type
// return : array of PPSyncFieldModel
//===============================================================================
- (NSMutableArray *)syncFieldArrayWithFieldType:(NSInteger)fieldType
{
	if(!self.fieldArrayDict)
		self.fieldArrayDict = [[[NSMutableDictionary alloc] init] autorelease];
	
	NSMutableArray *fieldArray = [self.fieldArrayDict objectForIntegerKey:fieldType];
	
	if(!fieldArray)
	{
		fieldArray = [[NSMutableArray alloc] init];
		[self.fieldArrayDict setObject:fieldArray forIntegerKey:fieldType];
		[fieldArray release];
	}
	
	return fieldArray;
}


//===============================================================================
// get field count by specified type
// param  : subType - define value of subType. Input 0 will count with all subType.
// return : count of field
//===============================================================================
- (NSInteger)syncFieldCountWithFieldType:(NSInteger)fieldType subType:(NSInteger)subType
{
	if(!self.fieldArrayDict)
        return 0;
    
	NSMutableArray *fieldArray = [self.fieldArrayDict objectForIntegerKey:fieldType];
    
    if(subType == 0)
    {
        return [fieldArray count];
    }
    else
    {
        NSInteger counter = 0;
        
        for(PPSyncFieldModel *fieldModel in fieldArray)
        {
            if(fieldModel.subType == subType)
                counter++;
        }
        
        return counter;
    }
}


//===============================================================================
// add field
//===============================================================================
- (void)addSyncFieldModel:(PPSyncFieldModel *)syncFieldModel
{
    if(syncFieldModel.value == nil)
        return;
    
	//-------------------------------------------------
	// hadle field array
	//-------------------------------------------------
	// add to field array
	NSMutableArray *fieldArray = [self syncFieldArrayWithFieldType:syncFieldModel.type];
	[fieldArray addObject:syncFieldModel];
}


//===============================================================================
// remove field
//===============================================================================
- (void)removeSyncFieldModel:(PPSyncFieldModel *)syncFieldModel
{
    NSInteger fieldType = syncFieldModel.type;
    
	// delete from field array
	NSMutableArray *fieldArray = [self syncFieldArrayWithFieldType:fieldType];
	[fieldArray removeObject:syncFieldModel];
    
	// delete fieldArray in dict if is empty
    if(![fieldArray count])
        [self.fieldArrayDict removeObjectForIntegerKey:fieldType];
}


//===============================================================================
// get field
//===============================================================================
- (PPSyncFieldModel *)syncFieldModelWithFieldType:(NSInteger)fieldType atIndex:(NSInteger)index
{
	NSMutableArray *fieldArray = [self syncFieldArrayWithFieldType:fieldType];
	
	if(![fieldArray count])
		return nil;
    
    return [fieldArray objectAtIndex:index];
}


//===============================================================================
// append fields from other dataModel
//===============================================================================
- (void)appendFieldsFromSyncCardModel:(PPSyncCardModel *)syncCardModel
{
    NSArray         *keyArray   = [syncCardModel.fieldArrayDict allKeys];
    NSArray         *valueArray = [syncCardModel.fieldArrayDict allValues];
    NSUInteger      count       = [keyArray count];

    NSMutableArray  *fieldModelArray = nil;

    for(int i=0; i<count; i++)
    {
        fieldModelArray = [self syncFieldArrayWithFieldType:[[keyArray objectAtIndex:i] intValue]];
        [fieldModelArray addObjectsFromArray:[valueArray objectAtIndex:i]];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - ImageDict Methods


//===============================================================================
// update image dict from other dataModel
//===============================================================================
- (void)updateImageDictFromSyncCardModel:(PPSyncCardModel *)syncCardModel
{
    if (self.imageDict==nil)
    {
        self.imageDict = [[[NSMutableDictionary alloc] init] autorelease];
    }

    NSArray *keyArray   = [syncCardModel.imageDict allKeys];
    NSArray *valueArray = [syncCardModel.imageDict allValues];
    NSUInteger count    = [keyArray count];

    for(int i=0; i<count; i++)
    {
        [self.imageDict setObject:[valueArray objectAtIndex:i] forIntegerKey:[[keyArray objectAtIndex:i] intValue]];
    }
}


//===============================================================================
// set identfiy at index
//===============================================================================
- (void)setImageIdentify:(NSString *)identfiy forIndex:(NSInteger)index
{
    if(self.imageDict==nil)
    {
        self.imageDict = [[[NSMutableDictionary alloc] init] autorelease];
    }

    [self.imageDict setObject:identfiy forIntegerKey:index];
}


//===============================================================================
// get identfiy at index
//===============================================================================
- (NSString *)imageIdentifyForIndex:(NSInteger)index
{
    return [self.imageDict objectForIntegerKey:index];
}


//===============================================================================
// check key in imageDict
//===============================================================================
- (BOOL)hasKeyForImageDict:(NSInteger)intKey
{
    if(self.imageDict!=nil)
    {
        NSString *keyString = [[NSString alloc] initWithFormat:@"%@", @(intKey)];
        for(NSString *key in [self.imageDict allKeys])
        {
            if([key isEqualToString:keyString])
            {
                [keyString release];
                return YES;
            }
        }
        [keyString release];
    }
    return NO;
}


//===============================================================================
// Has photos
//===============================================================================
- (BOOL)hasFrontPhoto
{
    return [self hasKeyForImageDict:PPSyncCardImageType_FrontSide];
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasBackPhoto
{
    return [self hasKeyForImageDict:PPSyncCardImageType_BackSide];
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasCardPhoto
{
    return [self hasKeyForImageDict:PPSyncCardImageType_IDPhoto];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - debug


//===============================================================================
// dump data
//===============================================================================
- (void)dumpSyncFieldModels
{
    NSLog(@"=== %@ === dump begin", self.uniqueID);
    
    NSArray *valueArray = [self.fieldArrayDict allValues];
    
    if([valueArray count] == 0)
        NSLog(@"No data");
    
    for(int i=0; i<[valueArray count]; i++)
    {
        NSArray *fieldArray = [valueArray objectAtIndex:i];
        
        for(PPSyncFieldModel *fieldModel in fieldArray)
        {
            NSLog(@"- type : %@", @(fieldModel.type));
            NSLog(@"- subType : %@", @(fieldModel.subType));
            NSLog(@"- value : %@", fieldModel.value);
            NSLog(@"-----------------------------");
        }
    }
    
    NSLog(@"=== %@ === dump end", self.uniqueID);
    NSLog(@" ");
    
}

@end
