//
//  PPSelectController.m
//
//  Fixed   by howard on 13-12-17.
//  Modified   by Egg on 15-02-16.
//  Modified   by Andrew on 15-04-28.
//  Copyright 2009 __MyCompanyName__. All rights reserved.
//

#import "PPSelectController.h"

// Model
#import <ReactiveCocoa/ReactiveCocoa.h>
#import <QuartzCore/QuartzCore.h>

// View
#import "ContentClickView.h"
#import "UIView+Appearance.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define DEGREES_TO_RADIANS(degrees) (degrees * M_PI / 180)

#define PPSelectController_ContentTableHeaderColor [UIColor yellowColor]
#define PPSelectController_ContentTableSeparatorColor [UIColor colorWithWhite:0.5 alpha:0.5]
#define PPSelectController_ContentTableBackgroundColor [UIColor colorWithRed:0 green:0 blue:0 alpha:0.8]

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Constant

static NSUInteger   PPSelectController_DefaultCellFontSize               = 20;
static UIEdgeInsets PPSelectController_DefaultContentTableInsetPortrait  = {23.0f, 35.0f, 23.0f, 35.0f};
static UIEdgeInsets PPSelectController_DefaultContentTableInsetLandScape = {23.0f, 110.0f, 23.0f, 110.0f};

static NSUInteger   PPSelectController_TableAndLabelVerticalMargin = 10;
static NSUInteger   PPSelectController_TableCellVerticalMargin     = 10;
static NSUInteger   PPSelectController_TableCellHorizontalMargin   = 15;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - global attributes

static NSMutableDictionary *PPSelectControllerAttributes = nil;

static NSString *const PPSelectControllerAttributesKey_TitleHeight = @"PPSelectControllerAttributesKey_TitleHeight";
static NSString *const PPSelectControllerAttributesKey_TitleColor = @"PPSelectControllerAttributesKey_TitleColor";
static NSString *const PPSelectControllerAttributesKey_TitleBackgroundColor = @"PPSelectControllerAttributesKey_TitleBackgroundColor";
static NSString *const PPSelectControllerAttributesKey_ContentTableBackgroundColor = @"PPSelectControllerAttributesKey_ContentTableBackgroundColor";
static NSString *const PPSelectControllerAttributesKey_CellNormalTextColor = @"PPSelectControllerAttributesKey_CellNormalTextColor";
static NSString *const PPSelectControllerAttributesKey_CellHighlightedTextColor = @"PPSelectControllerAttributesKey_CellHighlightedTextColor";
static NSString *const PPSelectControllerAttributesKey_CellHighlightedBackgroundColor = @"PPSelectControllerAttributesKey_CellHighlightedBackgroundColor";

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSelectController Extension

@interface PPSelectController () <UITableViewDataSource, UITableViewDelegate, PPButtonDelegate>
//!! Simple Clicked anywhere View
@property (nonatomic, retain) ContentClickView *contentClickView;
//!! Keep viewController which show current ItemSelecView
@property (nonatomic, assign) UIViewController *curViewController;
@property (nonatomic, copy) SelectCompletion selectCompleteHandler;

@property (nonatomic, retain) RACDisposable *animatedRotate;
@property (nonatomic, retain) RACDisposable *viewDidLayout;
@property (nonatomic, retain) RACDisposable *viewDidDisappear;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSelectController Implementation

@implementation PPSelectController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
// MARK: 如果需要設定appearance的預設值，請覆寫NSObject的+initialize。
//================================================================================
+ (void)initialize
{
    [[PPSelectController appearance] setContentTableViewPortraitInset:PPSelectController_DefaultContentTableInsetPortrait];
    [[PPSelectController appearance] setContentTableViewLandScapeInset:PPSelectController_DefaultContentTableInsetLandScape];
}


//================================================================================
// 盡量使用私有變數初始化，避免使用Property導致直接呼叫後續添加的覆寫Set函式
//================================================================================
- (id)init
{
    if (self = [super init])
    {
        // MARK: 套用appearance設定，儘量放在init一開始的地方，除非有別的需求。
        [[PPSelectController appearance] applyInvocationTo:self];
        
        //////////////////////////////////////////////////
        
        _customCellHeight  = 0;
        _selectedIndex     = -1;
        _cellFontSize      = [[UIFont systemFontOfSize:PPSelectController_DefaultCellFontSize] retain];
        
        _title = @"";
        _titleColor = [[PPSelectController titleColor] retain]?:[[UIColor blackColor] retain];
        _titleHeight = [PPSelectController titleHeight]?:0.0;
        _titleBackgroundColor = [[PPSelectController titleBackgroundColor] retain]?:[PPSelectController_ContentTableHeaderColor retain];
        _contentTableBackgroundColor = [[PPSelectController contentTableBackgroundColor] retain]?:[PPSelectController_ContentTableBackgroundColor retain];
        _cellHighlightedBackgroundColor = [[PPSelectController cellHighlightedBackgroundColor] retain]?:[PPSelectController_ContentTableBackgroundColor retain];
        
        _cellNormalTextColor = [[PPSelectController cellNormalTextColor] retain]?:[[UIColor whiteColor] retain];
        _cellHighlightedTextColor = [[PPSelectController cellHighlightedTextColor] retain]?:[[UIColor whiteColor] retain];
        
        _tableViewCellSelectionStyle = UITableViewCellSelectionStyleDefault;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.selectCompleteHandler = nil;
    
    self.delegate = nil;
    self.userInfo = nil;
    self.sourceItemStringArray = nil;
    self.cellFontSize = nil;
    
    self.title = nil;
    self.titleColor = nil;
    self.titleBackgroundColor = nil;
    self.contentTableBackgroundColor = nil;
    
    self.cellHighlightedTextColor = nil;
    self.cellNormalTextColor = nil;
    self.cellHighlightedBackgroundColor = nil;

    //////////////////////////////////////////////////
    
    [self.contentClickView removeControlEvents:UIControlEventTouchDown];
    [self.contentClickView removeFromSuperview];
    self.contentClickView = nil;
    
    //////////////////////////////////////////////////
    
    self.curViewController = nil;
    
    //////////////////////////////////////////////////
    // 釋放ReactiveCocoa的監控
    
    self.animatedRotate = nil;
    self.viewDidLayout = nil;
    self.viewDidDisappear = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Method

//================================================================================
//
//================================================================================
- (void)setSourceItemStringArray:(NSArray *)sourceItemStringArray
{
    if(sourceItemStringArray==nil)
    {
        [_sourceItemStringArray release];
        
        _sourceItemStringArray = nil;
    }
    else
    {
        [sourceItemStringArray retain];
        
        [_sourceItemStringArray release];
        
        _sourceItemStringArray = sourceItemStringArray;
    }
    
    //////////////////////////////////////////////////
    
    if (self.contentClickView!=nil && self.contentClickView.contentTableView!=nil)
    {
        [self.contentClickView.contentTableView reloadData];
        [self layoutViewConstraint];
    }
}


//================================================================================
//
//================================================================================
- (void)setTitleHeight:(CGFloat)titleHeight
{
    _titleHeight = titleHeight;
    
    //////////////////////////////////////////////////
    
    [self layoutViewConstraint];
}


//================================================================================
//
//================================================================================
- (void)setContentTableViewLandScapeInset:(UIEdgeInsets)contentTableViewLandScapeInset
{
    _contentTableViewLandScapeInset = contentTableViewLandScapeInset;
}


//================================================================================
//
//================================================================================
- (void)setContentTableViewPortraitInset:(UIEdgeInsets)contentTableViewPortraitInset
{
    _contentTableViewPortraitInset = contentTableViewPortraitInset;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

//================================================================================
//
//================================================================================
- (CGAffineTransform)transformWithOrientation:(UIInterfaceOrientation)orientation
{
    CGAffineTransform affineTransform;
    
    switch (orientation)
    {
        case UIInterfaceOrientationLandscapeLeft:
        {
            affineTransform = CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(-90));
            
            break;
        }
        case UIInterfaceOrientationLandscapeRight:
        {
            affineTransform = CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(90));
            
            break;
        }
        case UIInterfaceOrientationPortraitUpsideDown:
        {
            affineTransform = CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(180));
            
            break;
        }
        case UIInterfaceOrientationPortrait:
        default:
        {
            affineTransform = CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(0));
            
            break;
        }
    }
    
    return affineTransform;
}


//================================================================================
//
//================================================================================
- (CGFloat)cellHeightWithText:(NSString *)text
{
    CGFloat cellHeight = 0;
    
    if (self.customCellHeight==0)
    {
        CGFloat cellTextFieldWidth = [self maxCellContentWidth];
        
        if([[[UIDevice currentDevice]systemVersion]floatValue]>=7.0)
        {
            NSMutableParagraphStyle *paragraphStyle = [[NSMutableParagraphStyle alloc] init];
            
            if(paragraphStyle!=nil)
            {
                paragraphStyle.lineBreakMode = NSLineBreakByCharWrapping;
                paragraphStyle.alignment     = NSTextAlignmentLeft;
                
                NSDictionary * attributes = @{NSFontAttributeName : self.cellFontSize,
                                              NSParagraphStyleAttributeName : paragraphStyle};
                
                [paragraphStyle release];
                
                CGRect textRect = [text boundingRectWithSize:(CGSize){cellTextFieldWidth, 10000}
                                                     options:NSStringDrawingUsesLineFragmentOrigin attributes:attributes context:nil];
                CGSize textSize=textRect.size;
                
                cellHeight = textSize.height + 2*PPSelectController_TableCellVerticalMargin;
            }
        }
        
        //////////////////////////////////////////////////
        
        if(cellHeight<=0)
        {
            NSMutableParagraphStyle *style = [[NSParagraphStyle defaultParagraphStyle] mutableCopy];
            
            [style setLineBreakMode:NSLineBreakByWordWrapping];
            
            NSDictionary *sizeAttributes = @{NSFontAttributeName:self.cellFontSize, NSParagraphStyleAttributeName:style};
            [style release];
            
            //////////////////////////////////////////////////
            
            CGRect rect = [text boundingRectWithSize:CGSizeMake(cellTextFieldWidth, 10000)
                                             options:NSStringDrawingUsesLineFragmentOrigin
                                          attributes:sizeAttributes
                                             context:nil];
            CGSize textSize = rect.size;
            
            cellHeight = (textSize.height + 2*PPSelectController_TableCellVerticalMargin);
        }
    }
    else
    {
        cellHeight = self.customCellHeight;
    }
    
    //////////////////////////////////////////////////
    
    return cellHeight;
}


//================================================================================
//
//================================================================================
- (CGFloat)contentHeight
{
    CGFloat contentHeight = 0;
    
    //-------------------------------
    // count total cell height
    //-------------------------------
    for(NSString *item in self.sourceItemStringArray)
    {
        contentHeight += [self cellHeightWithText:item];
    }
    
    //-------------------------------
    // HeaderView height
    //-------------------------------
    contentHeight += (self.titleHeight+PPSelectController_TableAndLabelVerticalMargin);
    
    return contentHeight;
}


//================================================================================
// 添加選擇界面於keyWindow上方
//================================================================================
- (void)layoutItemSelectView
{
    UIWindow *keyWindow = [UIApplication sharedApplication].keyWindow;
    if (keyWindow==nil)
    {
        keyWindow = [[UIApplication sharedApplication].windows objectAtIndex:0];
    }
    
    //////////////////////////////////////////////////
    // 製作最底層點選任意鍵關閉界面的按鈕
    
    if (self.contentClickView==nil)
    {
        _contentClickView = [[ContentClickView alloc] initWithFrame:CGRectZero];
        if (self.contentClickView!=nil)
        {
            [self.contentClickView addControlEvents:UIControlEventTouchDown];
            
            self.contentClickView.delegate = self;
            
            self.contentClickView.contentTableView.dataSource     = self;
            self.contentClickView.contentTableView.delegate       = self;
            
            [keyWindow addSubview:self.contentClickView];
            [self layoutViewConstraint];
        }
    }
    else
    {
        if (self.contentClickView.superview==nil)
        {
            [keyWindow addSubview:self.contentClickView];
        }
        
        [keyWindow bringSubviewToFront:self.contentClickView];
        [self layoutViewConstraint];
    }
    
    self.contentClickView.contentTableView.separatorColor = PPSelectController_ContentTableSeparatorColor;
        
    self.contentClickView.tableTitleLabel.text      = self.title;
    self.contentClickView.tableTitleLabel.textColor = self.titleColor;
    self.contentClickView.tableTitleLabel.backgroundColor  = self.titleBackgroundColor;
    self.contentClickView.contentTableView.backgroundColor = self.contentTableBackgroundColor;
    self.contentClickView.contentTableView.separatorInset  = UIEdgeInsetsMake(0.0f, PPSelectController_TableCellHorizontalMargin, 0.0f, PPSelectController_TableCellHorizontalMargin);
    
    //////////////////////////////////////////////////
    // 呈現選擇頁面(可能重複使用，需取消hidden)
    
    self.contentClickView.hidden = NO;
}


//================================================================================
//
//================================================================================
- (void)layoutViewConstraint
{
    //////////////////////////////////////////////////
    // reset contentClickView position
    
    if ([[[UIDevice currentDevice] systemVersion] floatValue] < 8.0)
    {
        self.contentClickView.transform = [self transformWithOrientation:self.curViewController.interfaceOrientation];
    }
    
    self.contentClickView.frame = [[UIScreen mainScreen] bounds];
    
    //////////////////////////////////////////////////
    // 因為ios 7的keyWindow不會進行旋轉
    
    CGRect superViewFrame = self.contentClickView.superview.frame;
    
    if ([[[UIDevice currentDevice] systemVersion] floatValue]<8.0)
    {
        superViewFrame = [self.contentClickView.superview convertRect:self.contentClickView.superview.frame toView:self.contentClickView];
    }
    
    CGFloat superViewHeight    = superViewFrame.size.height;
    CGFloat maxTableViewHeight = superViewHeight-self.contentTableViewPortraitInset.top-self.contentTableViewPortraitInset.bottom;
    
    //////////////////////////////////////////////////
    // 對呈現選單內容的contentTableView進行layout
    
    self.contentClickView.contentTableView.scrollEnabled = YES;
    
    if (UIInterfaceOrientationIsPortrait(self.curViewController.interfaceOrientation)==YES)
    {
        CGRect clickViewFrame    = self.contentClickView.bounds;
        CGRect titleLabelFrame   = self.contentClickView.tableTitleLabel.frame;
        CGRect contentTableFrame = self.contentClickView.contentTableView.frame;
        
        titleLabelFrame.origin.x     = self.contentTableViewPortraitInset.left;
        titleLabelFrame.size.width   = clickViewFrame.size.width-titleLabelFrame.origin.x-self.contentTableViewPortraitInset.right;
        contentTableFrame.origin.x   = self.contentTableViewPortraitInset.left;
        contentTableFrame.size.width = clickViewFrame.size.width-contentTableFrame.origin.x-self.contentTableViewPortraitInset.right;

        //////////////////////////////////////////////////
        // 垂直分佈需觀察內容高度，進行TableView高度調整
        
        if ([self contentHeight]>maxTableViewHeight)
        {
            titleLabelFrame.origin.y    = self.contentTableViewPortraitInset.top;
            titleLabelFrame.size.height = self.titleHeight;
            
            if (self.titleHeight==0)
            {
                contentTableFrame.origin.y = self.contentTableViewPortraitInset.top;
                contentTableFrame.size.height = clickViewFrame.size.height-contentTableFrame.origin.y-self.contentTableViewPortraitInset.bottom;
            }
            else
            {
                contentTableFrame.origin.y = self.contentTableViewPortraitInset.top+self.titleHeight+PPSelectController_TableAndLabelVerticalMargin;
                contentTableFrame.size.height = clickViewFrame.size.height-contentTableFrame.origin.y-self.contentTableViewPortraitInset.bottom;
            }
        }
        else
        {
            self.contentClickView.contentTableView.scrollEnabled = NO;
            CGFloat contentBottomHeight = (superViewHeight-[self contentHeight])/2;
            
            titleLabelFrame.origin.y      = contentBottomHeight;
            titleLabelFrame.size.height   = self.titleHeight;
            contentTableFrame.origin.y    = contentBottomHeight+self.titleHeight+PPSelectController_TableAndLabelVerticalMargin;
            contentTableFrame.size.height = [self contentHeight]-self.titleHeight-PPSelectController_TableAndLabelVerticalMargin;
        }
        
        self.contentClickView.tableTitleLabel.frame = titleLabelFrame;
        self.contentClickView.contentTableView.frame = contentTableFrame;
    }
    else
    {
        CGRect clickViewFrame    = self.contentClickView.bounds;
        CGRect titleLabelFrame   = self.contentClickView.tableTitleLabel.frame;
        CGRect contentTableFrame = self.contentClickView.contentTableView.frame;
        
        titleLabelFrame.origin.x   = self.contentTableViewLandScapeInset.left;
        titleLabelFrame.size.width = clickViewFrame.size.width-self.contentTableViewLandScapeInset.right-titleLabelFrame.origin.x;

        contentTableFrame.origin.x   = self.contentTableViewLandScapeInset.left;
        contentTableFrame.size.width = clickViewFrame.size.width-self.contentTableViewLandScapeInset.right-contentTableFrame.origin.x;

        //////////////////////////////////////////////////
        // 垂直分佈需觀察內容高度，進行TableView高度調整
        
        if ([self contentHeight]>maxTableViewHeight)
        {
            titleLabelFrame.origin.y    = self.contentTableViewLandScapeInset.top;
            titleLabelFrame.size.height = self.titleHeight;
            
            if (self.titleHeight==0)
            {
                contentTableFrame.origin.y = self.contentTableViewLandScapeInset.top;
                contentTableFrame.size.height = clickViewFrame.size.height-contentTableFrame.origin.y-self.contentTableViewLandScapeInset.bottom;
            }
            else
            {
                contentTableFrame.origin.y = self.contentTableViewLandScapeInset.top+self.titleHeight+PPSelectController_TableAndLabelVerticalMargin;
                contentTableFrame.size.height = clickViewFrame.size.height-contentTableFrame.origin.y-self.contentTableViewLandScapeInset.bottom;
            }
        }
        else
        {
            self.contentClickView.contentTableView.scrollEnabled = NO;
            CGFloat contentBottomHeight = (superViewHeight-[self contentHeight])/2;
            
            titleLabelFrame.origin.y      = contentBottomHeight;
            titleLabelFrame.size.height    = self.titleHeight;
            contentTableFrame.origin.y    = contentBottomHeight+self.titleHeight+self.titleHeight+PPSelectController_TableAndLabelVerticalMargin;
            contentTableFrame.size.height = [self contentHeight]-self.titleHeight-PPSelectController_TableAndLabelVerticalMargin;
        }
        
        self.contentClickView.tableTitleLabel.frame = titleLabelFrame;
        self.contentClickView.contentTableView.frame = contentTableFrame;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Show Methods


//==============================================================================
//
//==============================================================================
- (void)showFromViewController:(UIViewController *)viewController animated:(BOOL)animated completeHendler:(SelectCompletion)completeHandler
{
    self.curViewController = viewController;
    self.selectCompleteHandler = completeHandler;
    
    //////////////////////////////////////////////////
    // do not monitor if no source ItemString found
    
    if(self.sourceItemStringArray == nil)
    {
        self.curViewController = nil;
        return;
    }
    
    //////////////////////////////////////////////////
    
    [self layoutItemSelectView];
    
    //////////////////////////////////////////////////
    // start monitor view controller
    
    if([self.sourceItemStringArray count] > 0)
    {
        __block PPSelectController *blockSelf = self;
        
        ////////////////////////////////////////////////
        // 抓取呼叫的viewController旋轉通知，進行layout
        
        RACSignal *willAnimateRotateToInterfaceOrientation = [viewController rac_signalForSelector:@selector(willAnimateRotationToInterfaceOrientation:duration:)];
        
        self.animatedRotate = [[willAnimateRotateToInterfaceOrientation subscribeNext:^(id _) {
            if ([blockSelf isVisible]==YES)
            {
                [blockSelf layoutItemSelectView];
                [blockSelf.contentClickView.contentTableView reloadData];
            }
        }] asScopedDisposable];
        
        //////////////////////////////////////////////////
        // 為了避免出現keyWindow在旋轉時，出現不夠大＆頁面背景露出的問題
        
        RACSignal *viewDidLayoutSubviews = [viewController rac_signalForSelector:@selector(viewDidLayoutSubviews)];
        
        self.viewDidLayout = [[viewDidLayoutSubviews subscribeNext:^(id _) {
            if ([blockSelf isVisible]==YES)
            {
                [blockSelf layoutItemSelectView];
            }
        }] asScopedDisposable];
        
        //////////////////////////////////////////////////
        // 為了避免出現keyWindow在旋轉時，出現不夠大＆頁面背景露出的問題
        
        RACSignal *viewDidDisappear = [viewController rac_signalForSelector:@selector(viewDidDisappear:)];
        
        self.viewDidDisappear = [[viewDidDisappear subscribeNext:^(id _) {
            if ([blockSelf isVisible]==YES)
            {
                self.contentClickView.hidden = YES;
            }
        }] asScopedDisposable];
    }
}

//================================================================================
//
//================================================================================
- (void)showFromViewController:(UIViewController *)viewController animated:(BOOL)animated
{
    [self showFromViewController:viewController animated:animated completeHendler:nil];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - View Status Methods


//==============================================================================
//
//==============================================================================
- (void)hideContentClickView
{
    self.contentClickView.hidden = YES;
    //////////////////////////////////////////////////
    // 釋放ReactiveCocoa的監控
    // !! 畫面隱藏時就要先釋收，不然如果建立新的PPSelectController時，會無法釋放舊的，要等self.curViewController release時，才會一起釋放
    
    self.animatedRotate = nil;
    self.viewDidLayout = nil;
    self.viewDidDisappear = nil;
}


//================================================================================
//
//================================================================================
- (BOOL)isVisible
{
    return !(self.contentClickView.hidden);
}


//================================================================================
//
//================================================================================
- (CGFloat)maxCellContentWidth
{
    CGFloat contentTableWidth = 0.0;
    
    UIWindow *keyWindow = [UIApplication sharedApplication].keyWindow;
    if (keyWindow==nil)
    {
        keyWindow = [[UIApplication sharedApplication].windows objectAtIndex:0];
    }
    
    //////////////////////////////////////////////////
    
    if (UIInterfaceOrientationIsPortrait(self.curViewController.interfaceOrientation)==YES)
    {
        contentTableWidth = keyWindow.frame.size.width-self.contentTableViewPortraitInset.left-self.contentTableViewPortraitInset.right;
    }
    else
    {
        contentTableWidth = keyWindow.frame.size.width-self.contentTableViewLandScapeInset.left-self.contentTableViewLandScapeInset.right;
        
        //////////////////////////////////////////////////
        // 因為ios 7的keyWindow不會進行旋轉
        
        if ([[[UIDevice currentDevice] systemVersion] floatValue]<8.0)
        {
            CGRect transFrame = [keyWindow convertRect:keyWindow.frame toView:self.contentClickView];
            
            contentTableWidth = transFrame.size.width-self.contentTableViewLandScapeInset.left-self.contentTableViewLandScapeInset.right;
        }
    }
    
    return contentTableWidth - (PPSelectController_TableCellHorizontalMargin * 2);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate Method

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    [self hideContentClickView];
    
    //////////////////////////////////////////////////
    // 通知元件消失
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppSelectControllerDidDismiss:)]==YES)
    {
        [self.delegate ppSelectControllerDidDismiss:self];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPAppearance protocol methods

//================================================================================
//
//================================================================================
+ (id)appearance
{
    // MARK: 實作+appearance的方式固定如下
    return [PPAppearance appearanceForClass:[self class]];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIScrollViewDelegate Methods

//================================================================================
// 滾動期間禁止點選
//================================================================================
- (void)scrollViewWillBeginDragging:(UIScrollView *)scrollView
{
    self.contentClickView.enabled = NO;
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDragging:(UIScrollView *)scrollView willDecelerate:(BOOL)decelerate
{
    self.contentClickView.enabled = YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource Methods

//================================================================================
//
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [self.sourceItemStringArray count];
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    UITableViewCell *cell = nil;
    
    //////////////////////////////////////////////////
    // 發生例外問題cell的索引超出資料數量
    
    if(self.sourceItemStringArray.count <= indexPath.row)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    // 客製化Cell
    
    NSString *itemString = [self.sourceItemStringArray objectAtIndex:indexPath.row];
    
    if([self.delegate respondsToSelector:@selector(ppSelectController:tableView:cellForRowAtIndexPath:itemString:)] == YES)
    {
        cell = [self.delegate ppSelectController:self tableView:tableView cellForRowAtIndexPath:indexPath itemString:itemString];
    }
    
    //////////////////////////////////////////////////
    // 若無客製化Cell，則使用預設的樣式
    
    if (cell == nil)
    {
        static NSString *CellIdentifier = @"PPSelectControllerCellIdentifier";
        
        cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
        if (cell==nil)
        {
            cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier] autorelease];
            if (cell==nil)
            {
                return nil;
            }
        }
        
        cell.textLabel.text = [self.sourceItemStringArray objectAtIndex:indexPath.row];
        
        cell.backgroundColor     = [UIColor clearColor];
        cell.textLabel.textColor = self.cellNormalTextColor;
        cell.textLabel.highlightedTextColor = self.cellHighlightedTextColor;
        
        //////////////////////////////////////////////////
        // 使Cell內部的textLabel會自動換行
        
        cell.textLabel.font          = self.cellFontSize;
        
        if (self.customCellHeight==0)
        {
            cell.textLabel.lineBreakMode = NSLineBreakByCharWrapping;
            cell.textLabel.numberOfLines = 0;
        }
        
        //////////////////////////////////////////////////
        //!! seletionStyle不是UITableViewCellSelectionStyleNone，設定Cell選擇顏色才有效
        
        cell.selectionStyle = self.tableViewCellSelectionStyle;
        
        cell.selectedBackgroundView = [[[UIView alloc] init] autorelease];
        cell.selectedBackgroundView.backgroundColor = self.cellHighlightedBackgroundColor;
    }

    //////////////////////////////////////////////////
    // 選擇狀態設定
    
    if(indexPath.row==self.selectedIndex)
    {
        cell.accessoryType = UITableViewCellAccessoryCheckmark;
    }
    else
    {
        cell.accessoryType = UITableViewCellAccessoryNone;
    }
    
    return cell;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDelegate Methods

//================================================================================
//
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath
{
    NSString *text = [self.sourceItemStringArray objectAtIndex:indexPath.row];
    
    return [self cellHeightWithText:text];
}


//================================================================================
// 變更Cell選取的樣式＆記錄selectIndex
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{    
    UITableViewCell* selectedCell = [tableView cellForRowAtIndexPath:indexPath];
    
    if(self.selectedIndex>-1)
    {
        if (self.selectedIndex>=0 && indexPath.row!=self.selectedIndex)
        {
            selectedCell.accessoryType = UITableViewCellAccessoryCheckmark;
            
            NSUInteger oldIndex[] = {0, self.selectedIndex};
            NSIndexPath* oldIndexPath = [[NSIndexPath alloc] initWithIndexes:oldIndex length:2];
            UITableViewCell *oldCell = [tableView cellForRowAtIndexPath:oldIndexPath];
            oldCell.accessoryType = UITableViewCellAccessoryNone;
            [oldIndexPath release];
        }
        
        self.selectedIndex = indexPath.row;
    }
    
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
    
    //////////////////////////////////////////////////
    // 通知選擇哪一個SelectItem
    
    if([self.delegate respondsToSelector:@selector(ppSelectController:itemSelectedIndex:withString:)]==YES)
    {
        [self.delegate ppSelectController:self itemSelectedIndex:indexPath.row withString:[self.sourceItemStringArray objectAtIndex:indexPath.row]];
    }
    
    if (self.selectCompleteHandler)
    {
        self.selectCompleteHandler([self.sourceItemStringArray objectAtIndex:indexPath.row], indexPath.row);
    }
    //////////////////////////////////////////////////
    // 判斷點選後是否隱藏選擇界面,預設為YES
    
    BOOL shouldHide = YES;
    
    if ([self.delegate respondsToSelector:@selector(ppSelectControllerShouldHideAfterSelected:)] == YES)
    {
        shouldHide = [self.delegate ppSelectControllerShouldHideAfterSelected:self];
    }
    
    if (shouldHide==YES)
    {
        [self hideContentClickView];
        
        //////////////////////////////////////////////////
        // 通知元件消失
        if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppSelectControllerDidDismiss:)]==YES)
        {
            [self.delegate ppSelectControllerDidDismiss:self];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Global attributes settings

//================================================================================
//
//================================================================================
+ (CGFloat)titleHeight
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [[PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_TitleHeight] floatValue];
}


//================================================================================
//
//================================================================================
+ (void)setTitleHeight:(CGFloat)titleHeight
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    [PPSelectControllerAttributes setObject:@(titleHeight) forKey:PPSelectControllerAttributesKey_TitleHeight];
}


//================================================================================
//
//================================================================================
+ (UIColor *)titleColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_TitleColor];
}


//================================================================================
//
//================================================================================
+ (void)setTitleColor:(UIColor *)titleColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (titleColor)
    {
        [PPSelectControllerAttributes setObject:titleColor forKey:PPSelectControllerAttributesKey_TitleColor];
    }
}


//================================================================================
//
//================================================================================
+ (UIColor *)titleBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_TitleBackgroundColor];
}


//================================================================================
//
//================================================================================
+ (void)setTitleBackgroundColor:(UIColor *)titleBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (titleBackgroundColor)
    {
        [PPSelectControllerAttributes setObject:titleBackgroundColor forKey:PPSelectControllerAttributesKey_TitleBackgroundColor];
    }
}


//================================================================================
//
//================================================================================
+ (UIColor *)contentTableBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_ContentTableBackgroundColor];
}


//================================================================================
//
//================================================================================
+ (void)setContentTableBackgroundColor:(UIColor *)contentTableBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (contentTableBackgroundColor)
    {
        [PPSelectControllerAttributes setObject:contentTableBackgroundColor forKey:PPSelectControllerAttributesKey_ContentTableBackgroundColor];
    }
}


//================================================================================
//
//================================================================================
+ (UIColor *)cellNormalTextColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_CellNormalTextColor];
}


//================================================================================
//
//================================================================================
+ (void)setCellNormalTextColor:(UIColor *)cellNormalTextColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (cellNormalTextColor)
    {
        [PPSelectControllerAttributes setObject:cellNormalTextColor forKey:PPSelectControllerAttributesKey_CellNormalTextColor];
    }
}


//================================================================================
//
//================================================================================
+ (UIColor *)cellHighlightedTextColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_CellHighlightedTextColor];
}


//================================================================================
//
//================================================================================
+ (void)setCellHighlightedTextColor:(UIColor *)cellHighlightedTextColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (cellHighlightedTextColor)
    {
        [PPSelectControllerAttributes setObject:cellHighlightedTextColor forKey:PPSelectControllerAttributesKey_CellHighlightedTextColor];
    }
}


//================================================================================
//
//================================================================================
+ (UIColor *)cellHighlightedBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    return [PPSelectControllerAttributes objectForKey:PPSelectControllerAttributesKey_CellHighlightedBackgroundColor];
}


//================================================================================
//
//================================================================================
+ (void)setCellHighlightedBackgroundColor:(UIColor *)cellHighlightedBackgroundColor
{
    if (PPSelectControllerAttributes == nil)
    {
        PPSelectControllerAttributes = [[NSMutableDictionary alloc] init];
    }
    
    if (cellHighlightedBackgroundColor)
    {
        [PPSelectControllerAttributes setObject:cellHighlightedBackgroundColor forKey:PPSelectControllerAttributesKey_CellHighlightedBackgroundColor];
    }
}

@end
