//
//  PPSectionIndexView.m
//

#import "PPSectionIndexView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

CGFloat const PPSectionIndexViewDefaultVerticalWidth = 35;
CGFloat const PPSectionIndexViewDefaultHorizontalHeight = 35;

////////////////////////////////////////////////////////////////////////////////////////////////////

CGFloat const PPSIV_DefaultFontSize     = 11;
CGFloat const PPSIV_TitleBeginMargin	= 5;
CGFloat const PPSIV_TitleEndMargin      = 5;

NSString * const PPSIV_DotSymbol = @"•";

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPSectionIndexView ()

@property (nonatomic, retain) PPSectionIndexModel *preSelectedModel;

@property (nonatomic, retain) UIImage *searchIconNormal;
@property (nonatomic, retain) UIImage *searchIconDown;
@property (nonatomic, assign) CGSize searchIconSize;
@property (nonatomic, assign) CGSize switchIconSize;
@property (nonatomic, assign) CGFloat oneTitleMaxSize;
@property (nonatomic, assign) CGFloat titlesEndMargin;

@property (nonatomic, assign) NSInteger curTouchDownIndex;
@property (nonatomic, retain) NSMutableArray *displayIndexTitleArray;
@property (nonatomic, retain) NSMutableArray *fullIndexTitleArray;
@property (nonatomic, retain) NSMutableArray *indexDataArray;

@property (nonatomic, retain) UILongPressGestureRecognizer *pressGestureRecognizer;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSectionIndexView


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIAppearance apply methods

//================================================================================
//
//================================================================================
+ (void)initialize
{
    [[PPSectionIndexView appearance] setTitleFont:[UIFont boldSystemFontOfSize:PPSIV_DefaultFontSize]];
    [[PPSectionIndexView appearance] setTitleNormalColor:[UIColor blackColor]];
    [[PPSectionIndexView appearance] setTitleHighlightColor:[UIColor whiteColor]];
    [[PPSectionIndexView appearance] setBackgroundHighlightColor:[UIColor blackColor]];
}


//================================================================================
//
//================================================================================
- (void)setTitleFont:(UIFont *)titleFont
{
    if(_titleFont != titleFont)
    {
        [_titleFont release];
        _titleFont = [titleFont retain];
        
        [self resetIndexData];
    }
}


//================================================================================
//
//================================================================================
- (void)setTitleNormalColor:(UIColor *)titleNormalColor
{
    if(_titleNormalColor != titleNormalColor)
    {
        [_titleNormalColor release];
        _titleNormalColor = [titleNormalColor retain];
    }
}


//================================================================================
//
//================================================================================
- (void)setTitleHighlightColor:(UIColor *)titleHighlightColor
{
    if(_titleHighlightColor != titleHighlightColor)
    {
        [_titleHighlightColor release];
        _titleHighlightColor = [titleHighlightColor retain];
    }
}


//================================================================================
//
//================================================================================
- (void)setBackgroundHighlightColor:(UIColor *)backgroundHighlightColor
{
    if(_backgroundHighlightColor != backgroundHighlightColor)
    {
        [_backgroundHighlightColor release];
        _backgroundHighlightColor = [backgroundHighlightColor retain];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//===============================================================================
// 
//===============================================================================
- (instancetype)init
{
    self = [super init];
    
    if (self) 
	{				
        // Initialization code
        self.backgroundColor = [UIColor clearColor];
        
        self.curTouchDownIndex = -1;
		self.style = PPSectionIndexViewStyleVertical;
		self.titlesEndMargin = PPSIV_TitleEndMargin;
        
        //////////////////////////////////////////////////

        _pressGestureRecognizer = [[UILongPressGestureRecognizer alloc] initWithTarget:self action:@selector(onPressGesture:)];
        
        if(self.pressGestureRecognizer != nil)
        {
            self.pressGestureRecognizer.minimumPressDuration = 0.0;
            [self addGestureRecognizer:self.pressGestureRecognizer];
        }

        //////////////////////////////////////////////////
    }
    
    return self;
}


//===============================================================================
// 
//===============================================================================
- (void)dealloc 
{
    if(self.pressGestureRecognizer != nil)
    {
        [self removeGestureRecognizer:self.pressGestureRecognizer];
        self.pressGestureRecognizer = nil;
    }
    
    //////////////////////////////////////////////////
    
    self.titleFont = nil;
    self.titleNormalColor = nil;
    self.titleHighlightColor = nil;
    self.backgroundHighlightColor = nil;
    
    self.preSelectedModel = nil;
    self.searchIconDown = nil;
    self.searchIconNormal = nil;
    
    self.displayIndexTitleArray = nil;
    self.fullIndexTitleArray = nil;
    self.indexDataArray = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Action methods

//================================================================================
//
//================================================================================
- (void)onPressGesture:(UIGestureRecognizer *)gestureRecognizer
{
    CGPoint point = [gestureRecognizer locationInView:self];
    
    switch (gestureRecognizer.state)
    {
        case UIGestureRecognizerStateBegan:
        {
            self.preSelectedModel = nil;            
            [self touchedTitleWithPoint:point];
            break;
        }
            
        case UIGestureRecognizerStateChanged:
        {
            [self touchedTitleWithPoint:point];
            break;
        }
            
        default:
        {
            self.curTouchDownIndex = -1;
            break;
        }
    }
    
    [self setNeedsDisplay];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIView override methods

//===============================================================================
// 
//===============================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    [self resetIndexData];
}


//===============================================================================
// 
//===============================================================================
- (void)drawRect:(CGRect)rect 
{
    CGRect              circleRect;
	PPSectionIndexModel	*sectionIndexModel;
	CGContextRef		context = UIGraphicsGetCurrentContext();
    NSArray             *indexDataArray = [self.indexDataArray retain];
	
	
	CGContextSaveGState(context);

	// draw index characters
	for(int i=0; i<[indexDataArray count]; i++)
	{
		sectionIndexModel = [indexDataArray objectAtIndex:i];
		
		// draw image
		if([sectionIndexModel.title isEqualToString:PPSectionIndexTitleForSearch])
		{            
            if(self.curTouchDownIndex == i)
            {
                circleRect = [self touchDownCircleRectWithBounds:sectionIndexModel.drawRect];
                CGContextSetFillColorWithColor(context, self.backgroundHighlightColor.CGColor);
                CGContextFillEllipseInRect(context, circleRect);
                
                [self.searchIconDown drawInRect:sectionIndexModel.drawRect];
            }
            else [self.searchIconNormal drawInRect:sectionIndexModel.drawRect];
		}
		else // draw text & circle background
		{
            UIColor *textColor = self.titleNormalColor;
            
            // draw circle background
            if(self.curTouchDownIndex == i && ![sectionIndexModel.title isEqualToString:PPSIV_DotSymbol])
            {
                circleRect = [self touchDownCircleRectWithBounds:sectionIndexModel.drawRect];
                CGContextSetFillColorWithColor(context, self.backgroundHighlightColor.CGColor);
                CGContextFillEllipseInRect(context, circleRect);
                
                textColor = self.titleHighlightColor;
            }
            
            // draw text
            if ([sectionIndexModel.title respondsToSelector:@selector(drawInRect:withAttributes:)]==YES)
            {
                [sectionIndexModel.title drawInRect:sectionIndexModel.drawRect
                                     withAttributes:@{NSFontAttributeName:_titleFont, NSForegroundColorAttributeName:textColor}];
            }
            else
            {
                CGContextSetFillColorWithColor(context, textColor.CGColor);
                [sectionIndexModel.title drawInRect:sectionIndexModel.drawRect withAttributes:@{NSFontAttributeName:_titleFont}];
            }
		}
        
        //test : show touch rect
//        CGContextSetFillColorWithColor(context, [UIColor redColor].CGColor);
//        CGContextFillRect(context, sectionIndexModel.touchRect);        
	}
	
	CGContextRestoreGState(context);
    
    [indexDataArray release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private methods


//==============================================================================
//
//==============================================================================
- (NSInteger)displayTitleIndexWithTitle:(NSString *)title
{
    for (NSInteger i = 0 ; i < [self.displayIndexTitleArray count]; i++)
    {
        NSString *displayTitle = [self.displayIndexTitleArray objectAtIndex:i];
        if ([displayTitle isEqualToString:title])
        {
            return i;
        }
    }
    
    return -1;
}



//===============================================================================
//
//===============================================================================
- (CGSize)drawSizeWithImage:(UIImage *)image
{
    CGSize drawSize = image.size;
    
    drawSize.width = image.size.width/2;
    drawSize.height = image.size.height/2;
    
    return drawSize;
}


//===============================================================================
//
//===============================================================================
- (void)resetIndexData
{
    NSInteger			titleCount = [self.displayIndexTitleArray count];
    PPSectionIndexModel	*sectionIndexModel;
    CGRect				touchRect;
    CGRect				drawRect;
    CGRect				dispFrame = self.bounds;
    CGSize				titleSize;
    
    
    //////////////////////////////////////////////////
    // ignore if no title exist
    if(titleCount == 0 || self.titleFont == nil)
    {
        self.indexDataArray = nil;
        return;
    }
    
    
    //////////////////////////////////////////////////
    // caculate index data
    self.indexDataArray = [NSMutableArray array];
    
    switch (self.style)
    {
        case PPSectionIndexViewStyleVertical:
            self.oneTitleMaxSize = (dispFrame.size.height - PPSIV_TitleBeginMargin - self.titlesEndMargin)/titleCount;
            break;
            
        case PPSectionIndexViewStyleHorizontal:
            self.oneTitleMaxSize = (dispFrame.size.width - PPSIV_TitleBeginMargin - _titlesEndMargin)/titleCount;
            break;
            
        default:
            self.indexDataArray = nil;
            return;
    }
    
    CGFloat fontSize = [self.titleFont pointSize];
    if(self.oneTitleMaxSize < fontSize)
    {
        self.oneTitleMaxSize = fontSize;
    }
    
    //////////////////////////////////////////////////

    for(int i=0; i<titleCount; i++)
    {
        // set title
        sectionIndexModel = [[PPSectionIndexModel alloc] init];
        sectionIndexModel.title = [self.displayIndexTitleArray objectAtIndex:i];
        
        switch (self.style)
        {
            case PPSectionIndexViewStyleVertical:
            {
                // set draw rect
                if([sectionIndexModel.title isEqualToString:PPSectionIndexTitleForSearch])
                {
                    drawRect.size.width = fminf(dispFrame.size.width, self.searchIconSize.width);
                    drawRect.size.height = fminf(dispFrame.size.height, self.searchIconSize.height);
                }
                else
                {
                    titleSize = [sectionIndexModel.title sizeWithAttributes:@{NSFontAttributeName:self.titleFont}];
                    
                    drawRect.size.width = fminf(dispFrame.size.width, titleSize.width);
                    drawRect.size.height = fminf(dispFrame.size.height, titleSize.height);
                }
                
                drawRect.origin.x = (dispFrame.size.width - drawRect.size.width)/2;
                drawRect.origin.y = PPSIV_TitleBeginMargin + self.oneTitleMaxSize*i + (self.oneTitleMaxSize-drawRect.size.height)/2;
                sectionIndexModel.drawRect = drawRect;
                
                // set touch frame
                touchRect.origin.x = 0;
                touchRect.size.width = dispFrame.size.width;
                
                // 這邊先計算出每一個display index的touchRect
                if(i == 0)
                {
                    touchRect.origin.y = 0;
                    touchRect.size.height = PPSIV_TitleBeginMargin + self.oneTitleMaxSize;
                }
                else if(i == titleCount-1)
                {
                    touchRect.origin.y = drawRect.origin.y - (self.oneTitleMaxSize-drawRect.size.height)/2;
                    touchRect.size.height = PPSIV_TitleEndMargin + self.oneTitleMaxSize;
                }
                else
                {
                    touchRect.origin.y = drawRect.origin.y - (self.oneTitleMaxSize-drawRect.size.height)/2;
                    touchRect.size.height = self.oneTitleMaxSize;
                }
                
                sectionIndexModel.touchRect = touchRect;
                break;
            }
                
            case PPSectionIndexViewStyleHorizontal:
            {
                // set draw rect
                if([sectionIndexModel.title isEqualToString:PPSectionIndexTitleForSearch])
                {
                    drawRect.size.width = fminf(dispFrame.size.width, self.searchIconSize.width);
                    drawRect.size.height = fminf(dispFrame.size.height, self.searchIconSize.height);
                }
                else
                {
                    titleSize = [sectionIndexModel.title sizeWithAttributes:@{NSFontAttributeName:_titleFont}];
                    
                    drawRect.size.width = fminf(dispFrame.size.width, titleSize.width);
                    drawRect.size.height = fminf(dispFrame.size.height, titleSize.height);
                }
                
                drawRect.origin.x = PPSIV_TitleBeginMargin + self.oneTitleMaxSize*i + (self.oneTitleMaxSize-drawRect.size.width)/2;
                drawRect.origin.y = (dispFrame.size.height - drawRect.size.height)/2;
                sectionIndexModel.drawRect = drawRect;
                
                
                // set touch frame
                touchRect.origin.y = 0;
                touchRect.size.height = dispFrame.size.height;
                
                // 這邊先計算出每一個display index的touchRect
                if(i == 0)
                {
                    touchRect.origin.x = 0;
                    touchRect.size.width = PPSIV_TitleBeginMargin + self.oneTitleMaxSize;
                }
                else if(i == titleCount-1)
                {
                    touchRect.origin.x = drawRect.origin.x - (self.oneTitleMaxSize-drawRect.size.width)/2;
                    touchRect.size.width = PPSIV_TitleEndMargin + self.oneTitleMaxSize;
                }
                else
                {
                    touchRect.origin.x = drawRect.origin.x - (self.oneTitleMaxSize-drawRect.size.width)/2;
                    touchRect.size.width = self.oneTitleMaxSize;
                }

                sectionIndexModel.touchRect = touchRect;
                break;
            }
                
            default:
            {
                break;
            }
        }

        //////////////////////////////////////////////////

        // add index object
        [self.indexDataArray addObject:sectionIndexModel];
        [sectionIndexModel release];
    }
    
    //////////////////////////////////////////////////
    // 計算full index title的touchRect
    NSInteger fullTitleCount = [self.fullIndexTitleArray count];
    NSMutableArray *tempMappinArray = [NSMutableArray array];

    
    for (NSInteger j=0; j<fullTitleCount; j++)
    {
        // 建立一個Section index model
        sectionIndexModel = [[PPSectionIndexModel alloc] init];
        sectionIndexModel.title = [self.fullIndexTitleArray objectAtIndex:j];
        
        // 先找出目前的title,在display index中是第幾個
        NSInteger displayTitleIndex = [self displayTitleIndexWithTitle:sectionIndexModel.title];

        // -1表示沒有在displayIndexTitleArray中
        // 如果有在display中，表示有顯示出來，不用再處理
        // 如果沒有，表示是點，要把對應的點塞進去
        if (displayTitleIndex==-1)
        {
            // 對應到點，先記錄下來
            [tempMappinArray addObject:sectionIndexModel];
        }
        else
        {
            // 當有對應到display title時，將之前對應到dot symbol的那些section index
            // 加到 displayTitleIndex-1 所對應的section index model中
            
            // 第0個沒有前面的indexModel所以不用處理
            if (displayTitleIndex<1)
            {
                [sectionIndexModel release];
                continue;
            }
            
            //有對應其他點才要處理
            if ([tempMappinArray count]>0)
            {
                // 取得要附加資料的sectionIndexModel (目前displayTitleIndex的前一個)
                PPSectionIndexModel * matchedIndexModel = [self.indexDataArray objectAtIndex:displayTitleIndex-1];

                // 將之前對應到的資料寫入indexModel
                CGRect touchRect = matchedIndexModel.touchRect;
                CGFloat mappedRectHeight = touchRect.size.height/[tempMappinArray count];
                CGFloat mappedRectWidth = touchRect.size.width/[tempMappinArray count];
                
                // 計算對應到點的每一個index model的touchRect
                NSInteger mappedIndex = 0;
                for (PPSectionIndexModel *mappedIndexModel in tempMappinArray)
                {
                    if (self.style==PPSectionIndexViewStyleVertical)
                    {
                        mappedIndexModel.touchRect = CGRectMake(touchRect.origin.x,
                                                                touchRect.origin.y+mappedIndex*mappedRectHeight,
                                                                touchRect.size.width,
                                                                mappedRectHeight);
                    }
                    else if (self.style==PPSectionIndexViewStyleHorizontal)
                    {
                        mappedIndexModel.touchRect = CGRectMake(touchRect.origin.x+mappedIndex*mappedRectWidth,
                                                                touchRect.origin.y,
                                                                mappedRectWidth,
                                                                touchRect.size.height);
                    }
                    mappedIndex ++;
                }
                
                matchedIndexModel.mappedIndexArray = [NSArray arrayWithArray:tempMappinArray];
                
                [tempMappinArray removeAllObjects];
            }

        }
        
        [sectionIndexModel release];
    }
    
    //////////////////////////////////////////////////

    dispatch_async(dispatch_get_main_queue(), ^{
        
        [self setNeedsDisplay];
    });
}


//===============================================================================
// 
//===============================================================================
- (CGRect)touchDownCircleRectWithBounds:(CGRect)bounds
{
    CGRect circleRect;    
    NSInteger midX = CGRectGetMidX(bounds);
    NSInteger midY = CGRectGetMidY(bounds);
    NSInteger radius = 10;
    
    
    circleRect.origin.x = midX-radius;
    circleRect.origin.y = midY-radius;
    circleRect.size.width = 2*radius;
    circleRect.size.height = 2*radius;
    
    return circleRect;
}


//===============================================================================
// 
//===============================================================================
- (BOOL)touchedTitleWithPoint:(CGPoint)point
{
	PPSectionIndexModel *oneModel;
    
    self.curTouchDownIndex = -1;    
	
	for(int i=0; i<[self.indexDataArray count]; i++)
	{
		oneModel = [self.indexDataArray objectAtIndex:i];
		
        // 如果有sub section index model，要用sub section index model來處理
        if (oneModel.mappedIndexArray!=nil)
        {
            for (PPSectionIndexModel *subIndexModel in oneModel.mappedIndexArray)
            {
                if([self checkAndDelegatetIfPoint:point inSectionIndexModel:subIndexModel])
                {
                    self.curTouchDownIndex = i;
                    return YES;
                }
            }
        }
        else
        {
            if([self checkAndDelegatetIfPoint:point inSectionIndexModel:oneModel])
            {
                self.curTouchDownIndex = i;
                return YES;
            }
        }
    }
	
	return NO;
}


//==============================================================================
//
//==============================================================================
- (BOOL)checkAndDelegatetIfPoint:(CGPoint)point inSectionIndexModel:(PPSectionIndexModel *)sectionIndexModel
{
    if(CGRectContainsPoint(sectionIndexModel.touchRect, point))
    {
        // check previous select
        if(self.preSelectedModel != sectionIndexModel)
        {
            // 不是點才要通知
            if([sectionIndexModel.title isEqualToString:PPSIV_DotSymbol]==NO &&
               [self.delegate respondsToSelector:@selector(sectionIndexView:didSelectedIndexTitle:)])
            {
                [self.delegate sectionIndexView:self didSelectedIndexTitle:sectionIndexModel.title];
            }
            
            self.preSelectedModel = sectionIndexModel;
        }
        
        return YES;
    }
    return NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//===============================================================================
//
//===============================================================================
- (void)enableView:(BOOL)enable
{
    // !! 一定要設定，不然隱藏時還是會攔截touch event，下面的元件會收不到。
    self.userInteractionEnabled = enable;
    self.hidden = !enable;
}


//===============================================================================
// keep index char and caculate display rect
//===============================================================================
- (void)setIndexTitles:(NSArray *)indexTitles withExtraTitle:(PPSectionIndexViewExtraTitle)extraTitle
{
    [self setDisplayIndexTitles:indexTitles fullIndexTitles:indexTitles withExtraTitle:extraTitle];
}


//===============================================================================
// keep index char and caculate display rect
//===============================================================================
- (void)setDisplayIndexTitles:(NSArray *)displayIndexTitles fullIndexTitles:(NSArray *)fullIndexTitles withExtraTitle:(PPSectionIndexViewExtraTitle)extraTitle
{
    [displayIndexTitles retain];
    [_displayIndexTitleArray release];
    _displayIndexTitleArray = [[NSMutableArray alloc] initWithArray:displayIndexTitles];
    [displayIndexTitles release];

    
    [fullIndexTitles retain];
    [_fullIndexTitleArray release];
    _fullIndexTitleArray = [[NSMutableArray alloc] initWithArray:fullIndexTitles];
    [fullIndexTitles release];

    //////////////////////////////////////////////////
    
    if(extraTitle & PPSectionIndexViewExtraTitleSearch)
    {
        NSBundle *bundle = [NSBundle mainBundle];
        
        if(self.searchIconNormal == nil)
        {
            self.searchIconNormal = [UIImage imageWithContentsOfFile:[bundle pathForResource:@"PPSectionIndexView-Search-N" ofType:@"png"]];
            self.searchIconDown = [UIImage imageWithContentsOfFile:[bundle pathForResource:@"PPSectionIndexView-Search-D" ofType:@"png"]];
            self.searchIconSize = [self drawSizeWithImage:self.searchIconNormal];
        }
        
        // search icon會加在最前方
        [self.displayIndexTitleArray insertObject:PPSectionIndexTitleForSearch atIndex:0];
    }
    else
    {
        self.searchIconNormal = nil;
        self.searchIconDown = nil;
        self.searchIconSize = CGSizeZero;
    }
    
    //////////////////////////////////////////////////
    
    [self resetIndexData];
}

@end
