//
//  PPScrollTabController.m
//  Example
//
//  Created by sanhue on 2015/6/9.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "PPScrollTabController.h"
#import "PPBarView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPScrollTabController ()
@property (nonatomic, retain) UIScrollView *tabIndicatorContainer;
@property (nonatomic, retain) UIView *tabIndicator;

@property (nonatomic, retain) PPBarView *topBarView;
@property (nonatomic, retain) UIScrollView *pageScrollView;
@property (nonatomic, retain) NSMutableArray *buttons;
@property (nonatomic, retain, readwrite) UIViewController *selectedViewController;

@property (nonatomic, retain) NSArray *layoutConstraints;
@property (nonatomic, retain, readwrite) NSArray *viewControllers;

/// 最後的tabIndex, 用來與current比較看是否要切換layout
@property (nonatomic, assign) NSInteger lastTabIndex;

#pragma mark view controller flag
@property (nonatomic, assign) BOOL isFirstAppear;
@property (nonatomic, assign) BOOL willShowNext;


#pragma mark style
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPScrollTabController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        self.buttons = [NSMutableArray array];
        self.isFirstAppear = NO;
        self.willShowNext = YES;
        self.enableScroll = YES;
        
        self.tabButtonColor = [UIColor whiteColor];
        self.tabButtonTitleColor = [UIColor blackColor];
        self.tabButtonTitleSelectedColor = [UIColor blueColor];
        self.buttonImageEdgeInsets = UIEdgeInsetsZero;
        self.tabIndicatorColor = [UIColor whiteColor];
        self.tabIndicatorBackgroundColor = [UIColor whiteColor];
        self.tabIndicatorHeight = 3;
        self.bottomToolbarHeight = 0;
        self.lastTabIndex = -1;
        self.currentTabIndex = 0;
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self removeLayoutConstraints];
    self.buttons = nil;
    self.selectedViewController = nil;
    self.viewControllers = nil;
    self.pageScrollView = nil;
    self.tabButtonColor = nil;
    self.tabButtonTitleColor = nil;
    self.tabButtonTitleSelectedColor = nil;
    self.tabIndicator = nil;
    self.tabIndicatorColor = nil;
    self.tabIndicatorBackgroundColor = nil;
    self.tabIndicatorContainer = nil;
    self.topBarView = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    self.view.backgroundColor = [UIColor whiteColor];

    //////////////////////////////////////////////////
    // reset flag
    self.isViewDidAppear = NO;
    
    //////////////////////////////////////////////////
    // 建立介面
    [self prepareScrollTabUI];
    
    //////////////////////////////////////////////////
    [self resetLayoutConstraints];
    
    //////////////////////////////////////////////////
    // delegate
    if ([self.delegate respondsToSelector:@selector(scrollTabController:willAppear:)])
    {
        [self.delegate scrollTabController:self willAppear:animated];
    }
    
    [self sendWillShowPageAtIndex:self.currentTabIndex];
    
    [self setTabIndicatorWithIndex:self.currentTabIndex animated:NO];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    //////////////////////////////////////////////////
    // delegate
    if ([self.delegate respondsToSelector:@selector(scrollTabController:didAppear:)])
    {
        [self.delegate scrollTabController:self didAppear:animated];
    }
    
    [self sendDidShowPageAtIndex:self.currentTabIndex];
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    //////////////////////////////////////////////////
    // delegate
    if ([self.delegate respondsToSelector:@selector(scrollTabController:willDisappear:)])
    {
        [self.delegate scrollTabController:self willDisappear:animated];
    }
    
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    self.lastTabIndex = -1;

    // !! 直接做的話，會造成present其他view controller時，childViewController的ViewDidDisappear收不到
    // !! 用dispatch會造成release時先進dealloc再進dispatch block造成當機，所以先手動送出viewdiddisAppear
    for (UIViewController *viewController in self.childViewControllers)
    {
        if (viewController.isViewLoaded)
        {
            [viewController viewDidDisappear:animated];
        }
    }
    
    //////////////////////////////////////////////////
    
    [self removeScrollTabUI];
    
    //////////////////////////////////////////////////
    // delegate
    if ([self.delegate respondsToSelector:@selector(scrollTabController:didDisappear:)])
    {
        [self.delegate scrollTabController:self didDisappear:animated];
    }
    
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}
//- (void)viewWillLayoutSubviews
//{
//    [super viewWillLayoutSubviews];
//
//    [self resetLayoutConstraints];
//}

//==============================================================================
//
//==============================================================================
- (void)viewDidLayoutSubviews
{
    [super viewDidLayoutSubviews];
    
    //////////////////////////////////////////////////
    // 重新排列與設定 content view controller的大小
    
    [self.pageScrollView setContentSize:CGSizeMake(self.view.bounds.size.width * [self.viewControllers count], self.pageScrollView.bounds.size.height)];
    
    NSInteger i = 0;
    for (UIViewController *viewController in self.viewControllers)
    {
        viewController.view.frame = CGRectMake(self.pageScrollView.bounds.size.width * i++, 0, self.pageScrollView.bounds.size.width, self.pageScrollView.bounds.size.height);
    }
    
    // 調整topBarView大小
    [self.topBarView setFrame:CGRectMake(0, 0, self.view.bounds.size.width, PPBarViewDefaultHeightForNavigationBarNormal)];
    
    // 調整indicator大小
    [self adjustTabIndecatorSize];
    
    [self layoutWithIndex:self.currentTabIndex oldIndex:self.lastTabIndex animated:NO];
    
    
    // !!! 一定要在這邊call [self.view layoutIfNeeded];
    // 不然會出現 crash
    // ** Assertion failure in -[UIView layoutSublayersOfLayer:], /SourceCache/UIKit_Sim/UIKit-2935.137/UIView.m:8803
    // 參考 http://www.tagwith.com/question_486877_nslayoutconstraints-crashing-on-ios7-but-not-on-ios8
    [self.view layoutIfNeeded];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private layout constraints


//==============================================================================
//
//==============================================================================
- (void)resetLayoutConstraints
{
    
    if (self.topBarView==nil||
        self.tabIndicatorContainer==nil||
        self.pageScrollView==nil)
    {
        return ;
    }
    
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    //////////////////////////////////////////////////
    CGFloat navigationBarHeight = 0;
    if ([self.navigationController.navigationBar isTranslucent]==YES)
    {
        if (self.navigationController.navigationBar &&
            [self.navigationController isNavigationBarHidden] == NO)
        {
            navigationBarHeight = PPBarViewDefaultHeightForNavigationBarNormal;
        }

        if ([[UIApplication sharedApplication] isStatusBarHidden] == NO)
        {
            CGRect statusBarFrame =[[UIApplication sharedApplication] statusBarFrame];
            navigationBarHeight += MIN(statusBarFrame.size.width, statusBarFrame.size.height);
        }
    }

    //////////////////////////////////////////////////
    NSDictionary *views = @{@"bottomLayoutGuide":self.bottomLayoutGuide,
                            @"topBarView":self.topBarView,
                            @"tabIndicatorContainer":self.tabIndicatorContainer,
                            @"pageScrollView":self.pageScrollView};
    NSDictionary *metrics = @{@"BarHeight":@(PPBarViewDefaultHeightForNavigationBarNormal),
                              @"NavigationBarHeight":@(navigationBarHeight),
                              @"TabIndecatorHeight":@(self.tabIndicatorHeight),
                              @"BottomToolbarHeight":@(self.bottomToolbarHeight)};

   
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[topBarView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[tabIndicatorContainer]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[pageScrollView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];

    // 一頁以上才要顯示scroll tab
    NSString *vertFormat = [NSString stringWithFormat:@"V:|-(NavigationBarHeight)-%@[pageScrollView]%@-(BottomToolbarHeight)-|",
                            ([self.viewControllers count]>1)?@"[topBarView(BarHeight)][tabIndicatorContainer(TabIndecatorHeight)]":@"",
                            (self.layoutByBottomLayoutGuide==NO)?@"":@"[bottomLayoutGuide]"];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:vertFormat
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    //////////////////////////////////////////////////
    if ([layoutConstraints count] >0)
    {
        self.layoutConstraints = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraints];
//        [self.view layoutIfNeeded];
    }
}



//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private prepare UI


//==============================================================================
//
//==============================================================================
- (PPButton *)buttonWithTitle:(NSString *)title tag:(NSInteger)tag
{
    PPButton * button = [[[PPButton alloc] init] autorelease];
    
    [button addControlEvents:UIControlEventTouchUpInside];
    [button setDelegate:self];
    [button setBackgroundColor:self.tabButtonColor];
    
    [button setTitleColor:self.tabButtonTitleColor forState:UIControlStateNormal];
    [button setTitleColor:self.tabButtonTitleSelectedColor forState:UIControlStateSelected];
    [button setImageEdgeInsets:self.buttonImageEdgeInsets];
    [button setMinimumImageHeight:self.buttonImageMinimumHeight];
    
    [button setTitle:title maxWidth:0];
    [button setTag:tag];
    
    return button;
}


//==============================================================================
//
//==============================================================================
- (void)prepareScrollTabUI
{
    for (UIViewController *viewController in self.viewControllers)
    {
        [viewController.view removeFromSuperview];
        [viewController removeFromParentViewController];
    }
    
    self.viewControllers = nil;
    
    [self.buttons removeAllObjects];
    
    //////////////////////////////////////////////////
    self.topBarView = [[[PPBarView alloc] init] autorelease];
    if (self.topBarView)
    {
        [self.topBarView setStyle:PPBarViewStyle_Average];
        [self.topBarView setBackgroundColor:self.tabButtonColor];
        [self.topBarView setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.topBarView setUserInteractionEnabled:self.enableScroll];
        [self.view addSubview:self.topBarView];
    }
    
    self.pageScrollView = [[[UIScrollView alloc] init] autorelease];
    if (self.pageScrollView)
    {
        [self.pageScrollView setDelegate:self];
        [self.pageScrollView setBackgroundColor:[UIColor clearColor]];
        [self.pageScrollView setPagingEnabled:YES];
        [self.pageScrollView setBounces:NO];
        [self.pageScrollView setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.pageScrollView setShowsVerticalScrollIndicator:NO];
        [self.pageScrollView setShowsHorizontalScrollIndicator:NO];
        [self.pageScrollView setScrollEnabled:self.enableScroll];
        [self.view addSubview:self.pageScrollView];
    }
    
    //////////////////////////////////////////////////
    // 建立 按鈕與介面
    if ([self.dataSource respondsToSelector:@selector(numberOfTabAtController:)])
    {
        NSInteger numberOfTab = [self.dataSource numberOfTabAtController:self];
        NSAssert(numberOfTab>0, @"至少要有一個tab");
        
        NSMutableArray *viewControllers = [NSMutableArray array];
        for (NSInteger i = 0; i < numberOfTab; i++)
        {
            // 取得每一個頁面
            if ([self.dataSource respondsToSelector:@selector(scrollTabController:viewControllerAtIndex:)])
            {
                UIViewController *viewController = [self.dataSource scrollTabController:self viewControllerAtIndex:i];
                [viewControllers addObject:viewController];
                
                [viewController willMoveToParentViewController:self];
                [self addChildViewController:viewController];
                [self.pageScrollView addSubview:viewController.view];
                [viewController didMoveToParentViewController:self];
                
                //////////////////////////////////////////////////
                UIButton *tabButton = nil;
                // 如果有實作 |scrollTabController:buttonAtIndex:|, 優先使用
                if ([self.dataSource respondsToSelector:@selector(scrollTabController:buttonAtIndex:)])
                {
                    tabButton = [self.dataSource scrollTabController:self buttonAtIndex:i];
                }
                else
                {
                    //////////////////////////////////////////////////
                    NSString *viewTitle = @"No Title";
                    if ([self.dataSource respondsToSelector:@selector(scrollTabController:tabTitleAtIndex:)])
                    {
                        viewTitle = [self.dataSource scrollTabController:self tabTitleAtIndex:i];
                    }
                    
                    //////////////////////////////////////////////////
                    tabButton = [self buttonWithTitle:viewTitle tag:i];
                    
                    // 設定icon
                    if ([self.dataSource respondsToSelector:@selector(scrollTabController:tabImageAtIndex:controlState:)])
                    {
                        UIImage *normalTabImage = [self.dataSource scrollTabController:self tabImageAtIndex:i controlState:UIControlStateNormal];
                        [tabButton setImage:normalTabImage forState:UIControlStateNormal];

                        //////////////////////////////////////////////////
                        UIImage *highlightedTabImage = [self.dataSource scrollTabController:self tabImageAtIndex:i controlState:UIControlStateHighlighted];
                        if (highlightedTabImage==nil)
                        {
                            highlightedTabImage = normalTabImage;
                        }
                        [tabButton setImage:highlightedTabImage forState:UIControlStateHighlighted];
                        
                        //////////////////////////////////////////////////
                        UIImage *selectedTabImage = [self.dataSource scrollTabController:self tabImageAtIndex:i controlState:UIControlStateSelected];
                        if (selectedTabImage==nil)
                        {
                            selectedTabImage = normalTabImage;
                        }
                        [tabButton setImage:selectedTabImage forState:UIControlStateSelected];
                        
                        //////////////////////////////////////////////////
                        UIImage *disabledTabImage = [self.dataSource scrollTabController:self tabImageAtIndex:i controlState:UIControlStateDisabled];
                        if (disabledTabImage==nil)
                        {
                            disabledTabImage = normalTabImage;
                        }
                        [tabButton setImage:disabledTabImage forState:UIControlStateDisabled];
                    }
                }
                
                if (tabButton)
                {
                    [self.buttons addObject:tabButton];
                }
                //////////////////////////////////////////////////
                
            }
        }

        self.viewControllers = [NSArray arrayWithArray:viewControllers];
        
        [self.topBarView setViews:self.buttons forBlockType:PPBarViewBlockType_Left];
        
        //////////////////////////////////////////////////
        [self prepareTabIndicator];
        
        //////////////////////////////////////////////////
        [self.topBarView setHidden:([self.viewControllers count]==1)];
        [self.tabIndicatorContainer setHidden:([self.viewControllers count]==1)];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeScrollTabUI
{
    for (UIViewController *viewController in self.viewControllers)
    {
        [viewController.view removeFromSuperview];
        [viewController removeFromParentViewController];
    }

    self.viewControllers = nil;
    
    [self.buttons removeAllObjects];
    
    [self.topBarView removeFromSuperview];
    self.topBarView  = nil;
    
    [self.pageScrollView removeFromSuperview];
    self.pageScrollView = nil;
    
    //////////////////////////////////////////////////
    [self removeTabIndicator];
}


//==============================================================================
//
//==============================================================================
- (void)prepareTabIndicator
{
    _tabIndicatorContainer = [[UIScrollView alloc] init];
    if (self.tabIndicatorContainer)
    {
        [self.tabIndicatorContainer setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.tabIndicatorContainer setScrollEnabled:NO];
        [self.tabIndicatorContainer setBackgroundColor:self.tabIndicatorBackgroundColor?:self.tabButtonColor];
        [self.view addSubview:self.tabIndicatorContainer];

        //////////////////////////////////////////////////
        self.tabIndicator = [[[UIView alloc] init] autorelease];
        [self.tabIndicator setBackgroundColor:self.tabIndicatorColor];
        [self.tabIndicatorContainer addSubview:self.tabIndicator];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeTabIndicator
{
    [self.tabIndicator removeFromSuperview];
    self.tabIndicator = nil;
    
    [self.tabIndicatorContainer removeFromSuperview];
    self.tabIndicatorContainer = nil;
}


//==============================================================================
//
//==============================================================================
- (void)adjustTabIndecatorSize
{
    if (self.tabIndicatorContainer==nil ||
        [self.buttons count]==0)
    {
        return ;
    }
    
    CGRect containerFrame = self.tabIndicatorContainer.bounds;
    containerFrame.size.width /= [self.buttons count];
    self.tabIndicator.frame = containerFrame;
}


//==============================================================================
//
//==============================================================================
- (void)setTabIndicatorWithIndex:(NSInteger)index animated:(BOOL)animated
{
    void (^moveTabIndicator)(void) = ^{
        CGRect indecatorFrame = self.tabIndicator.frame;
        indecatorFrame.origin.x = index * indecatorFrame.size.width;
        [self.tabIndicator setFrame:indecatorFrame];
    };
    
    if (animated)
    {
        [UIView animateWithDuration:0.3 animations:moveTabIndicator];
    }
    else
    {
        moveTabIndicator ();
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Private) override preperty


//==============================================================================
//
//==============================================================================
- (void)setEnableScroll:(BOOL)enableScroll
{
    _enableScroll = enableScroll;
    
    //////////////////////////////////////////////////
    
    [self.pageScrollView setScrollEnabled:enableScroll];
    [self.topBarView setUserInteractionEnabled:enableScroll];
}


//==============================================================================
//
//==============================================================================
- (void)setBottomToolbarHeight:(CGFloat)bottomToolbarHeight
{
    _bottomToolbarHeight = bottomToolbarHeight;
    
    [self resetLayoutConstraints];
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - handle scroll view


//==============================================================================
//
//==============================================================================
- (void)sendWillShowPageAtIndex:(NSInteger)index
{
    if ([self.delegate respondsToSelector:@selector(scrollTabController:willShowPageAtIndex:)])
    {
        [self.delegate scrollTabController:self willShowPageAtIndex:index];
    }
}


//==============================================================================
//
//==============================================================================
- (void)sendDidShowPageAtIndex:(NSInteger)index
{
    if ([self.delegate respondsToSelector:@selector(scrollTabController:didShowPageAtIndex:)])
    {
        [self.delegate scrollTabController:self didShowPageAtIndex:index];
    }
}




//==============================================================================
//
//==============================================================================
- (void)setCurrentTabIndex:(NSInteger)currentTabIndex
{
    [self setCurrentTabIndex:currentTabIndex animated:NO];
}


//==============================================================================
//
//==============================================================================
- (void)setCurrentTabIndex:(NSInteger)currentTabIndex animated:(BOOL)animated
{
    NSInteger oldTabIndex = _currentTabIndex;
    if (_currentTabIndex!=currentTabIndex)
    {
        [self sendWillShowPageAtIndex:currentTabIndex];
        
        _currentTabIndex = currentTabIndex;

        [self layoutWithIndex:currentTabIndex oldIndex:oldTabIndex animated:animated];
    }
}


//==============================================================================
//
//==============================================================================
- (void)layoutWithIndex:(NSInteger)index oldIndex:(NSInteger)oldIndex animated:(BOOL)animated
{
    if(index==NSNotFound||
       [self.viewControllers count]<index)
    {
        return ;
    }
    
    NSInteger buttonCount = (NSInteger)[self.buttons count];
    if (buttonCount>index && buttonCount>oldIndex)
    {
        //////////////////////////////////////////////////
        // 設定content offset
        void (^offsetViewController)(void) = ^{
            [self.pageScrollView setContentOffset:CGPointMake(self.pageScrollView.bounds.size   .width * index, 0)];
            
            self.selectedViewController = [self.viewControllers objectAtIndex:index];
            self.lastTabIndex = index;
        };
        
        if (animated)
        {
            [UIView animateWithDuration:0.3 animations:offsetViewController];
        }
        else
        {
            offsetViewController ();
        }
        
        
        [self setTabIndicatorWithIndex:index animated:animated];
        
        //////////////////////////////////////////////////
        // 設定tab button state
        if (oldIndex>=0)
        {
            PPButton *oldButton = [self.buttons objectAtIndex:oldIndex];
            [oldButton setSelected:NO];
        }

        PPButton *currentButton = [self.buttons objectAtIndex:index];
        [currentButton setSelected:YES];
        
        //////////////////////////////////////////////////
        if (self.isViewDidAppear && index!=oldIndex)
        {
            [self sendDidShowPageAtIndex:self.currentTabIndex];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate


//==============================================================================
//
//==============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent;
{
    [self setCurrentTabIndex:ppButton.tag animated:YES];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIScrollViewDelegate

//==============================================================================
//
//==============================================================================
- (void)scrollViewDidScroll:(UIScrollView *)scrollView
{
//    NSLog(@"%s - %@", __PRETTY_FUNCTION__,NSStringFromCGPoint(scrollView.contentOffset));
    if ([self.delegate respondsToSelector:@selector(scrollTabController:willShowPageAtIndex:)])
    {
        if (self.willShowNext)
        {
            CGFloat currentTabIndex = scrollView.contentOffset.x/scrollView.bounds.size.width;
            CGFloat offset = _currentTabIndex-currentTabIndex;
            if (ABS(offset)>0)
            {
                if (offset>0)
                {
                    // 上一頁
                    [self.delegate scrollTabController:self willShowPageAtIndex:(NSInteger)(currentTabIndex-1)];
                }
                else
                {
                    // 下一頁
                    [self.delegate scrollTabController:self willShowPageAtIndex:(NSInteger)(currentTabIndex+1)];

                }
                self.willShowNext = NO;
            }
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)scrollViewDidEndDecelerating:(UIScrollView *)scrollView
{
    NSInteger oldTabIndex = _currentTabIndex;
    NSInteger currentTabIndex = scrollView.contentOffset.x/scrollView.bounds.size.width;
    
    //////////////////////////////////////////////////
    if (_currentTabIndex!=currentTabIndex)
    {
        _currentTabIndex = currentTabIndex;
        
        //////////////////////////////////////////////////
        [self layoutWithIndex:currentTabIndex oldIndex:oldTabIndex animated:YES];
        
    }
    self.willShowNext = YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - public methods


//==============================================================================
//
//==============================================================================
- (void)pageScrollViewSetRequireGestureRecognizerToFail:(UIGestureRecognizer *)gestureRecognizer
{
    [self.pageScrollView.panGestureRecognizer requireGestureRecognizerToFail:gestureRecognizer];

}


//==============================================================================
//
//==============================================================================
- (PPButton *)tabButtonWithIndex:(NSInteger)index
{
    if (index>=[self.buttons count])
    {
        return nil;
    }
    
    return [self.buttons objectAtIndex:index];
}

@end
