//
//  PPSQLiteController.m
//  
//
//  Created by Mike on 13/5/13.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPSQLiteController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSQLiteControllerFileFormat_SQLite             @"%@.db"
#define PPSQLiteControllerFileFormat_PropertyList       @"%@.plist"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSQLiteControllerCommand_CreateTable_Version	@"CREATE TABLE IF NOT EXISTS Version(Row INTEGER PRIMARY KEY AUTOINCREMENT, Major INTEGER, Minor INTEGER);"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSQLiteControllerStatus_Key_DeleteCount		@"DeleteCount"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSQLiteControllerMaxDeleteTimes                3000

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSQLiteController()

@interface PPSQLiteController()
@property(nonatomic,retain)             NSString    *dbPath;
@property(nonatomic,retain)             NSString    *plistPath;
@property(nonatomic,readwrite,assign)   sqlite3     *dbHandle;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSQLiteController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
dbPath      = dbPath_,
plistPath   = plistPath_,
dbHandle    = dbHandle_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithFileName:(NSString *)fileName atPath:(NSString *)path error:(NSError **)error
{
    id object = nil;
    
    //////////////////////////////////////////////////
    
    if((self=[super init]))
    {
        NSError *originError = ((error!=nil)?*error:nil);
        NSError *returnError = originError;
        
        //////////////////////////////////////////////////
        
        do
        {
            if(fileName==nil || [fileName length]<=0 || path==nil || [path length]<=0)
            {
                returnError = PPErrorParameterInvalidity(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dbPath_         = [[path stringByAppendingPathComponent:[NSString stringWithFormat:PPSQLiteControllerFileFormat_SQLite, fileName]] retain];
            plistPath_      = [[path stringByAppendingPathComponent:[NSString stringWithFormat:PPSQLiteControllerFileFormat_PropertyList, fileName]] retain];
            dbHandle_       = NULL;
            
            //////////////////////////////////////////////////
            
            object = self;
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = returnError;
        }
    }
    
    //////////////////////////////////////////////////
    
    if(object==nil)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}

//================================================================================
//
//================================================================================
- (void)dealloc 
{
	[self closeWithError:nil];
    
    //////////////////////////////////////////////////
    
	[plistPath_ release];
    [dbPath_ release];
    
    //////////////////////////////////////////////////
    
	[super dealloc];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Open & Close Methods

//================================================================================
//
//================================================================================
- (BOOL)openWithError:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    do
    {
        //已經開啟資料庫了
        if(self.dbHandle!=NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
   
        int code = sqlite3_open([self.dbPath UTF8String], &dbHandle_);
        if(code!=SQLITE_OK)
        {
            if(self.dbHandle!=nil)
            {
                returnError = PPErrorMake(code, [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
                
                self.dbHandle = nil;
            }
            else
            {
                returnError = PPErrorMake(code, @"SQLite is unable to allocate memory to hold the sqlite3 object", returnError);
            }
            
            break;
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)closeWithError:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    do
    {
        if(self.dbHandle==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        int code = sqlite3_close(self.dbHandle);
        if(code!=SQLITE_OK)
        {
            returnError = PPErrorMake(code, [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
        self.dbHandle = NULL;
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)removeWithError:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    do
    {
        if(self.dbHandle!=NULL && [self closeWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([[NSFileManager defaultManager] removeItemAtPath:self.dbPath error:nil]==NO)
        {
            returnError = PPErrorOperationFailed(returnError);
            
            break;
        }

        //////////////////////////////////////////////////
        
        if([[NSFileManager defaultManager] removeItemAtPath:self.plistPath error:nil]==NO)
        {
            returnError = PPErrorOperationFailed(returnError);
            
            break;
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - DB Methods

//================================================================================
//
//================================================================================
- (BOOL)isExist
{
	return [[NSFileManager defaultManager] fileExistsAtPath:self.dbPath];
}

//================================================================================
//
//================================================================================
- (BOOL)createWithCommands:(NSArray *)commands error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    do 
    {
        //連線資料庫
        if([self openWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(NSString *command in commands)
		{
			if([self runCommand:command error:&returnError]==NO)
			{
				break;
			}
		}
        
        //////////////////////////////////////////////////
        
        if(returnError==nil)
		{
			[self runCommand:PPSQLiteControllerCommand_CreateTable_Version error:&returnError];
		}
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        //建立失敗則要刪除剛建立的資料庫
        [self removeWithError:&returnError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Compact Methods

//================================================================================
//
//================================================================================
- (BOOL)compactWithError:(NSError **)error
{
    BOOL result = [self runCommand:@"VACUUM;" error:error];
    
    if(result==YES)
    {
        [self setStatusObject:[NSNumber numberWithUnsignedInteger:0] forKey:PPSQLiteControllerStatus_Key_DeleteCount];
    }
    
    return result;
}

//================================================================================
//
//================================================================================
- (BOOL)needsCompact
{
	return ([(NSNumber *)[self statusObjectForKey:PPSQLiteControllerStatus_Key_DeleteCount] unsignedIntValue]>PPSQLiteControllerMaxDeleteTimes);
}

//================================================================================
//
//================================================================================
- (void)incrementDeleteCount
{
    [self setStatusObject:[NSNumber numberWithUnsignedInteger:[[self statusObjectForKey:PPSQLiteControllerStatus_Key_DeleteCount] unsignedIntValue]+1]
                   forKey:PPSQLiteControllerStatus_Key_DeleteCount];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Status Dictionary Read&Write Operations

//================================================================================
//
//================================================================================
- (id)statusObjectForKey:(id)key
{
    id statusObject = nil;
    
    NSDictionary *statusDictionary = [[NSDictionary alloc] initWithContentsOfFile:self.plistPath];
    if(statusDictionary!=nil)
    {
        statusObject = [[[statusDictionary objectForKey:key] copy] autorelease];
        
        [statusDictionary release];
    }
    
	return statusObject;
}

//================================================================================
//
//================================================================================
- (void)setStatusObject:(id)object forKey:(id<NSCopying>)key
{
    @autoreleasepool
    {
        do
        {
            if(key==nil || self.plistPath==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSMutableDictionary *statusDictionary = [[[NSMutableDictionary alloc] initWithContentsOfFile:self.plistPath] autorelease];
            if(statusDictionary==nil)
            {
                statusDictionary = [[[NSMutableDictionary alloc] init] autorelease];
                if(statusDictionary==nil)
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(object!=nil)
            {
                [statusDictionary setObject:object forKey:key];
            }
            else
            {
                [statusDictionary removeObjectForKey:key];
            }
            
            //////////////////////////////////////////////////
            
            [statusDictionary writeToFile:self.plistPath atomically:YES];
            
        }while(0);
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Version Control Methods

//================================================================================
//
//================================================================================
- (BOOL)setVersionMajor:(int)major minor:(int)minor error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    [self runCommand:[NSString stringWithFormat:@"INSERT INTO Version(Major, Minor) VALUES(%d, %d);", major, minor]
               error:&returnError];
	 
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)getVersionMajor:(int *)major minor:(int *)minor error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if([self openWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString		*command    = @"SELECT Major, Minor FROM Version WHERE Row=(SELECT MAX(Row) FROM Version);";
        sqlite3_stmt	*statement  = NULL;
        
        if(sqlite3_prepare_v2(self.dbHandle, [command UTF8String], -1, &statement, NULL)!=SQLITE_OK)
        {
            returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
		if(sqlite3_step(statement)==SQLITE_ROW)
		{
			*major = sqlite3_column_int(statement, 0);
			*minor = sqlite3_column_int(statement, 1);
		}
		else
        {
            //這邊不能直接離開statement需要結束
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
        }
		
        //////////////////////////////////////////////////
        
        if(sqlite3_finalize(statement)!=SQLITE_OK)
		{
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
		}
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Transaction Methods

//================================================================================
//
//================================================================================
- (BOOL)beginTransactionWithError:(NSError **)error
{
    return [self runCommand:@"BEGIN;" error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)commitTransactionWithError:(NSError **)error
{
    return [self runCommand:@"COMMIT;" error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)endTransactionWithError:(NSError **)error
{
    return [self runCommand:@"END;" error:error];
}

//================================================================================
//
//================================================================================
- (BOOL)rollbackTransactionWithError:(NSError **)error
{
    return [self runCommand:@"ROLLBACK;" error:error];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)runCommand:(NSString *)command error:(NSError **)error
{
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    do 
    {
        if(command==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
     
        if([self openWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        sqlite3_stmt *statement = NULL;
        
        if(sqlite3_prepare_v2(self.dbHandle, [command UTF8String], -1, &statement, NULL)!=SQLITE_OK)
        {
            returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
		if(sqlite3_step(statement)!=SQLITE_DONE)
		{
            //這邊不能直接離開statement需要結束
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
		}
		
        //////////////////////////////////////////////////
        
        if(sqlite3_finalize(statement)!=SQLITE_OK)
		{
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
		}
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (NSInteger)integerResultWithCommand:(NSString *)command error:(NSError **)error
{
    NSInteger   integerResult   = -1;
    NSError     *returnError    = ((error!=nil)?*error:nil);
    
    do
    {
        if(command==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([self openWithError:&returnError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        sqlite3_stmt *statement = NULL;
        
        if(sqlite3_prepare_v2(self.dbHandle, [command UTF8String], -1, &statement, NULL)!=SQLITE_OK)
        {
            returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
		if(sqlite3_step(statement)==SQLITE_ROW)
		{
			integerResult = sqlite3_column_int(statement, 0);
		}
		else
        {
            //這邊不能直接離開statement需要結束
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
        }
        
        //////////////////////////////////////////////////
        
        if(sqlite3_finalize(statement)!=SQLITE_OK)
		{
			returnError = PPErrorMake(sqlite3_errcode(self.dbHandle), [NSString stringWithCString:sqlite3_errmsg(self.dbHandle) encoding:NSUTF8StringEncoding], returnError);
		}
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return integerResult;
}

//================================================================================
//
//================================================================================
- (NSInteger)recordCountWithCommand:(NSString *)command error:(NSError **)error
{
    return [self integerResultWithCommand:command error:error];
}

//================================================================================
//
//================================================================================
- (NSInteger)recordCountWithTableName:(NSString *)tableName error:(NSError **)error
{
    return [self recordCountWithCommand:[NSString stringWithFormat:@"SELECT COUNT(*) FROM %@;", tableName] error:error];
}

//================================================================================
//
//================================================================================
- (void)cancel
{
    sqlite3_interrupt(dbHandle_);
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//================================================================================
//
//================================================================================
+ (NSString *)formattedString:(NSString *)string
{
    NSString *formattedString = @"";
    
    if(string!=nil)
    {
        char *cString = sqlite3_mprintf("%q", [string UTF8String]);
        if(cString!=NULL)
        {
            formattedString = [NSString stringWithCString:cString encoding:NSUTF8StringEncoding];
            
            sqlite3_free(cString);
        }
    }
    
    return formattedString;
}

//================================================================================
//
//================================================================================
+ (BOOL)isExistFileName:(NSString *)fileName atPath:(NSString *)atPath
{
	return [[NSFileManager defaultManager] fileExistsAtPath:[atPath stringByAppendingPathComponent:[NSString stringWithFormat:PPSQLiteControllerFileFormat_SQLite, fileName]]];
}

//================================================================================
//
//================================================================================
+ (NSInteger)integerFromStatement:(sqlite3_stmt *)statement column:(int)column
{
    return (NSInteger)sqlite3_column_int(statement, column);
}

//================================================================================
//
//================================================================================
+ (double)doubleFromStatement:(sqlite3_stmt *)statement column:(int)column
{
    return sqlite3_column_double(statement, column);
}

//================================================================================
//
//================================================================================
+ (NSString *)stringFromStatement:(sqlite3_stmt *)statement column:(int)column
{
    NSString *string = nil;
    
    const char *utf8String = (const char *)sqlite3_column_text(statement, column);
    if(utf8String!=NULL)
    {
        string = [NSString stringWithUTF8String:utf8String];
    }
    
    return string;
}

//================================================================================
//
//================================================================================
+ (NSData *)dataFromStatement:(sqlite3_stmt *)statement column:(int)column
{
    NSData *data = nil;
    
    const void *dataBytes = sqlite3_column_blob(statement, column);
    if(dataBytes!=NULL)
    {
        data = [NSData dataWithBytes:dataBytes length:sqlite3_column_bytes(statement, column)];
    }
    
    return data;
}


@end
