//
//  PPRestClient.m
//  
//
//  Created by Howard on 2016/3/9.
//
//

#import "PPRestClient.h"
#import "PPRestClient+ParameterDefine.h"

#import <ifaddrs.h>
#import <arpa/inet.h>


#if TARGET_OS_IPHONE

#elif TARGET_OS_MAC

#import "NSImage+Additions.h"

#endif

@interface PPRestClient ()

@property (nonatomic, retain) RKObjectManager *jsonObjectManager;
@property (nonatomic, retain) RKObjectManager *urlEncodedObjectManager;
@property (nonatomic, retain) RKObjectManager *imageObjectManager;

@end
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation

@implementation PPRestClient

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithBaseURL:(NSURL *)url
{
    id object = nil;
    
    //////////////////////////////////////////////////
    
    if((self=[super init]))
    {
        do
        {
            _jsonObjectManager = [[RKObjectManager alloc] initWithHTTPClient:[AFRKHTTPClient clientWithBaseURL:url]];
            
            if(self.jsonObjectManager==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.jsonObjectManager.HTTPClient registerHTTPOperationClass:[AFRKJSONRequestOperation class]];
            [self.jsonObjectManager setAcceptHeaderWithMIMEType:RKMIMETypeJSON];
            [self.jsonObjectManager setRequestSerializationMIMEType:RKMIMETypeJSON];
            
            //////////////////////////////////////////////////
            _urlEncodedObjectManager = [[RKObjectManager alloc] initWithHTTPClient:[AFRKHTTPClient clientWithBaseURL:url]];
            
            if(self.urlEncodedObjectManager==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.urlEncodedObjectManager.HTTPClient registerHTTPOperationClass:[AFRKJSONRequestOperation class]];
            [self.urlEncodedObjectManager setAcceptHeaderWithMIMEType:RKMIMETypeJSON];
            [self.urlEncodedObjectManager setRequestSerializationMIMEType:RKMIMETypeFormURLEncoded];
            
            //////////////////////////////////////////////////

            _imageObjectManager = [[RKObjectManager alloc] initWithHTTPClient:[AFRKHTTPClient clientWithBaseURL:url]];
            
            if(self.imageObjectManager==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.imageObjectManager.HTTPClient registerHTTPOperationClass:[AFRKImageRequestOperation class]];
            [self.imageObjectManager setRequestSerializationMIMEType:RKMIMETypeJSON];
            
            //////////////////////////////////////////////////
            
            _timeOutInterval = PPRestClient_DefaultTimeOutInterval;
            
            //////////////////////////////////////////////////

            object = self;
            
        }while(0);
    }
    
    //////////////////////////////////////////////////
    
    if(object!=self)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}


//==============================================================================
//
//==============================================================================
- (RKObjectManager *)objectManagerForType:(PPRCObjectManagerType)type
{
    if(type==PPRCObjectManagerType_JSON)
    {
        return self.jsonObjectManager;
    }
    else if(type==PPRCObjectManagerType_URLEncoding)
    {
        return self.urlEncodedObjectManager;
    }
    else
    {
        return self.imageObjectManager;
    }
}


//================================================================================
//
//================================================================================
- (id)init
{
    return [self initWithBaseURL:[NSURL URLWithString:@""]];
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_jsonObjectManager release];
    _jsonObjectManager = nil;

    [_urlEncodedObjectManager release];
    _urlEncodedObjectManager = nil;

    [_imageObjectManager  release];
    _imageObjectManager = nil;
    
    [_lastOperationDate release];
    _lastOperationDate = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

//================================================================================
// Dictionary 移除 [NSNULL null]
//================================================================================
- (NSDictionary *)dictionaryRemovingNullsWithStringsFromDictionary:(NSDictionary *)dictionary
{
    NSMutableDictionary *replacedDictionary = nil;
    
    do
    {
        if(dictionary==nil ||
           dictionary.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        replacedDictionary = [NSMutableDictionary dictionaryWithDictionary:dictionary];
        
        for (NSString *key in dictionary)
        {
            const id object = [dictionary objectForKey:key];
  
            if([object isKindOfClass:[NSNull class]]==YES)
            {
                [replacedDictionary removeObjectForKey:key];
            }
            else if([object isKindOfClass:[NSDictionary class]]==YES)
            {
                NSDictionary *nextDictionary = [self dictionaryRemovingNullsWithStringsFromDictionary:object];
                
                if(nextDictionary.count>0)
                {
                    [replacedDictionary setObject:nextDictionary
                                           forKey:key];
                }

            }
            else if([object isKindOfClass:[NSArray class]]==YES)
            {
                NSMutableArray *newArray = [NSMutableArray array];
                
                for(id arrayComponent in object)
                {
                    if([arrayComponent isKindOfClass:[NSDictionary class]]==YES)
                    {
                        NSDictionary *newDictionary = [self dictionaryRemovingNullsWithStringsFromDictionary:arrayComponent];
                        
                        if(newDictionary.count>0)
                        {
                            [newArray addObject:newDictionary];
                        }
                    }
                    else if([arrayComponent isKindOfClass:[NSNull class]]==NO)
                    {
                        [newArray addObject:arrayComponent];
                    }
                }
                
                if(newArray.count>0)
                {
                    [replacedDictionary setObject:newArray
                                           forKey:key];
                }
            }
        }
    }
    while (0);
   
    return replacedDictionary;
}



//================================================================================
//
//================================================================================
- (NSArray *)ipAddresses
{
    NSMutableArray *ipAddresses = [NSMutableArray array];
    
    struct ifaddrs *interfaces = NULL;
    struct ifaddrs *temp_addr = NULL;
    struct sockaddr_in *s4;
    struct sockaddr_in6 *s6;
    char buf[64];
    int success = 0;
    
    // retrieve the current interfaces - returns 0 on success
    success = getifaddrs(&interfaces);
    if (success == 0)
    {
        // Loop through linked list of interfaces
        temp_addr = interfaces;
        while(temp_addr != NULL)
        {
            if(temp_addr->ifa_addr->sa_family == AF_INET)
            {
                
                s4 = (struct sockaddr_in *)temp_addr->ifa_addr;
                
                if (inet_ntop(temp_addr->ifa_addr->sa_family, (void *)&(s4->sin_addr), buf, sizeof(buf)) == NULL)
                {
                    NSLog(@"%s: inet_ntop failed for v4!\n",temp_addr->ifa_name);
                }
                else
                {
                    NSString *address = [NSString stringWithUTF8String:buf];
                    if (address)
                    {
                        [ipAddresses addObject:address];
                    }
                    
                }
                
            }
            
            else if(temp_addr->ifa_addr->sa_family == AF_INET6)
            {
                s6 = (struct sockaddr_in6 *)(temp_addr->ifa_addr);
                
                if (inet_ntop(temp_addr->ifa_addr->sa_family, (void *)&(s6->sin6_addr), buf, sizeof(buf)) == NULL)
                {
                    NSLog(@"%s: inet_ntop failed for v6!\n",temp_addr->ifa_name);
                }
                else
                {
                    NSString *address = [NSString stringWithUTF8String:buf];
                    if (address)
                    {
                        [ipAddresses addObject:address];
                    }
                }
            }
            
            temp_addr = temp_addr->ifa_next;
        }
    }
    
    // Free memory
    freeifaddrs(interfaces);
    
    return ipAddresses;
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)requestMethod:(RKRequestMethod)requestMethod
                                       path:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                   httpBody:(NSData *)httpBody
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:nil method:requestMethod path:path parameters:nil];
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        NSString *charset = (__bridge NSString *)CFStringConvertEncodingToIANACharSetName(CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding));
        [request setValue:[NSString stringWithFormat:@"application/json; charset=%@", charset] forHTTPHeaderField:@"Content-Type"];
        
        //////////////////////////////////////////////////
        
        if(httpBody!=nil)
        {
            [request setHTTPBody:httpBody];
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    return operation;
}





//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)requestMethod:(RKRequestMethod)requestMethod
                                       path:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                     object:(id)object
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:requestMethod path:path parameters:nil];
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        NSString *charset = (__bridge NSString *)CFStringConvertEncodingToIANACharSetName(CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding));
        [request setValue:[NSString stringWithFormat:@"application/json; charset=%@", charset] forHTTPHeaderField:@"Content-Type"];
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    return operation;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance register Methods

//================================================================================
//這是因為RestKit底層判斷mapping是否已經註冊過的機制有bug而改的
//================================================================================
- (BOOL)addRequestDescriptorForObjectClass:(Class)objectClass
                           toObjectManager:(RKObjectManager *)objectManager
                               rootKeyPath:(NSString *)rootKeyPath
                                    method:(RKRequestMethod)method
{
    BOOL result = NO;
    
    @synchronized(self)
    {
        do
        {
            if(objectClass==Nil || [objectClass isSubclassOfClass:[PPRestObject class]]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            RKRequestDescriptor *objectRequestDescriptor = [RKRequestDescriptor requestDescriptorWithMapping:[objectClass requestMapping]
                                                                                                 objectClass:objectClass
                                                                                                 rootKeyPath:rootKeyPath
                                                                                                      method:method];
            
            //////////////////////////////////////////////////
            
            NSArray *requestDescriptors = objectManager.requestDescriptors;
            
            for(RKRequestDescriptor *requestDescriptor in requestDescriptors)
            {
                if([requestDescriptor isKindOfClass:[objectRequestDescriptor class]]==YES               &&
                   [requestDescriptor.mapping isEqualToMapping:objectRequestDescriptor.mapping]==YES    &&
                   requestDescriptor.method==objectRequestDescriptor.method                             &&
                   (requestDescriptor.rootKeyPath==objectRequestDescriptor.rootKeyPath || [requestDescriptor.rootKeyPath isEqualToString:objectRequestDescriptor.rootKeyPath]==YES))
                {
                    result = YES;
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(result==NO)
            {
                [objectManager addRequestDescriptor:objectRequestDescriptor];
                
                result = YES;
            }
            
        }while(0);
    }
    
    return result;
}

//================================================================================
//
//================================================================================
- (BOOL)addRequestDescriptorForObjectClass:(Class)objectClass
                           toObjectManager:(RKObjectManager *)objectManager
{
    return [self addRequestDescriptorForObjectClass:objectClass toObjectManager:objectManager rootKeyPath:nil method:RKRequestMethodAny];
}

//================================================================================
//這是因為RestKit底層判斷mapping是否已經註冊過的機制有bug而改的
//================================================================================
- (BOOL)addResponseDescriptorForObjectClass:(Class)objectClass
                            toObjectManager:(RKObjectManager *)objectManager
                                     method:(RKRequestMethod)method
                                pathPattern:(NSString *)pathPattern
                                    keyPath:(NSString *)keyPath
                                statusCodes:(NSIndexSet *)statusCodes
{
    BOOL result = NO;
    
    @synchronized(self)
    {
        do
        {
            if(objectClass==Nil ||
               [objectClass isSubclassOfClass:[PPRestObject class]]==NO||
               [pathPattern length]==0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            RKResponseDescriptor *objectResponseDescriptor = [RKResponseDescriptor responseDescriptorWithMapping:[objectClass responseMapping]
                                                                                                          method:method
                                                                                                     pathPattern:pathPattern
                                                                                                         keyPath:keyPath
                                                                                                     statusCodes:statusCodes];
            
            //////////////////////////////////////////////////
            
            NSArray *responseDescriptors = objectManager.responseDescriptors;
            
//            //////////////////////////////////////////////////WCAM
//            
//            NSInteger targetIndex = -1;
//            
//            //Mark: Howard 沒辦法特殊解，如果mapping 是參考本身的，則要將舊的mapping 從 objectManager 移除
//            if([objectResponseDescriptor.pathPattern compare:WCClientPath_Category_QueryCategoryTree]==NSOrderedSame)
//            {
//                for(RKResponseDescriptor *responseDescriptor in responseDescriptors)
//                {
//                    if([responseDescriptor.pathPattern compare:WCClientPath_Category_QueryCategoryTree]==NSOrderedSame)
//                    {
//                        targetIndex = [responseDescriptors indexOfObject:responseDescriptor];
//                        
//                        break;
//                    }
//                }
//            }
//            
//            if(targetIndex>=0)
//            {
//                [self.objectManager removeResponseDescriptor:[responseDescriptors objectAtIndex:targetIndex]];
//                
//                responseDescriptors = self.objectManager.responseDescriptors;
//            }
            
            //////////////////////////////////////////////////
            
            for(RKResponseDescriptor *responseDescriptor in responseDescriptors)
            {
                if([responseDescriptor isEqualToResponseDescriptor:objectResponseDescriptor])
                {
                    result = YES;
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(result==NO)
            {
                [objectManager addResponseDescriptor:objectResponseDescriptor];
                
                result = YES;
            }
            
        }while(0);
    }
    
    return result;
}

//================================================================================
//
//================================================================================
- (BOOL)addResponseDescriptorForObjectClass:(Class)objectClass
                            toObjectManager:(RKObjectManager *)objectManager
                                pathPattern:(NSString *)pathPattern
{
    return [self addResponseDescriptorForObjectClass:objectClass
                                     toObjectManager:objectManager
                                              method:RKRequestMethodAny
                                         pathPattern:pathPattern
                                             keyPath:nil
                                         statusCodes:RKStatusCodeIndexSetForClass(RKStatusCodeClassSuccessful)];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance UnRegister Method

//================================================================================
//
//================================================================================
- (BOOL)removeResponseDescriptorForObjectClass:(Class)objectClass
                               toObjectManager:(RKObjectManager *)objectManager
                                        method:(RKRequestMethod)method
                                   pathPattern:(NSString *)pathPattern
                                       keyPath:(NSString *)keyPath
                                   statusCodes:(NSIndexSet *)statusCodes
{
    BOOL result = NO;
    
    @synchronized(self)
    {
        do
        {
            if(objectClass==Nil || [objectClass isSubclassOfClass:[PPRestObject class]]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            RKResponseDescriptor *objectResponseDescriptor = [RKResponseDescriptor responseDescriptorWithMapping:[objectClass responseMapping]
                                                                                                          method:method
                                                                                                     pathPattern:pathPattern
                                                                                                         keyPath:keyPath
                                                                                                     statusCodes:statusCodes];
            
            //////////////////////////////////////////////////
            
            NSArray *responseDescriptors = objectManager.responseDescriptors;
            
            //////////////////////////////////////////////////
            
            for(NSUInteger index=0; index<responseDescriptors.count; index++)
            {
                RKResponseDescriptor *responseDescriptor = [objectManager.responseDescriptors objectAtIndex:index];
                
                if([responseDescriptor isEqualToResponseDescriptor:objectResponseDescriptor]==NO)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////

                result = YES;
                
                [objectManager removeResponseDescriptor:responseDescriptor];
                
                break;
            }
            
        }while(0);
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance GET Method

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self getObject:object
           byObjectManager:objectManager
                      path:path
                parameters:parameters
      httpHeaderParameters:nil
                   success:success
                   failure:failure];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                   httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }

        //////////////////////////////////////////////////
       
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:RKRequestMethodGET path:path parameters:parameters];
        
        if(request==nil)
        {
            break;
        }
    
        request.timeoutInterval = self.timeOutInterval;
    
        //////////////////////////////////////////////////

        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        NSDictionary *routingMetadata = nil;
        
        if(parameters!=nil)
        {
            routingMetadata = @{ @"query": @{ @"parameters": parameters } };
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            operation.mappingMetadata = routingMetadata;
            
            //////////////////////////////////////////////////
            
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;

            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance POST Method

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postObject:(id)object
                         byObjectManager:(RKObjectManager *)objectManager
                                    path:(NSString *)path
                              parameters:(NSDictionary *)parameters
                                 success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                 failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self postObject:object
            byObjectManager:objectManager
                       path:path
                 parameters:parameters
       httpHeaderParameters:nil
                    success:success
                    failure:failure];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postObject:(id)object
                         byObjectManager:(RKObjectManager *)objectManager
                                    path:(NSString *)path
                              parameters:(NSDictionary *)parameters
                    httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                 success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                 failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:RKRequestMethodPOST path:path parameters:parameters];
       
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }

        //////////////////////////////////////////////////

        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////

        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////

        // HTTPBody ，針對 Dictionary [NSNull Null] 的 Value，重新解析
        if(request.HTTPBody!=nil)
        {
            NSDictionary *noNullBodyDictinary = [self dictionaryRemovingNullsWithStringsFromDictionary:[NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil]];
        
            if(noNullBodyDictinary!=nil)
            {
                [request setHTTPBody:[NSJSONSerialization dataWithJSONObject:noNullBodyDictinary
                                                                     options:(NSJSONWritingOptions)0
                                                                       error:nil]];
            }
        }
        
        //////////////////////////////////////////////////
        
#ifdef DEBUG
        //打印 HTTPBody 內容
//        if(request.HTTPBody!=nil)
//        {
//            NSDictionary *bodyDictinary = [NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil];
//            
//            // json format
//            if(bodyDictinary!=nil)
//            {
//                NSLog(@"%s, bodyDictinary:%@",__func__,bodyDictinary);
//            }
//            // url encoding
//            else
//            {
//                NSLog(@"%s, bodyString:%@",__func__,[NSString stringWithUTF8String:[request.HTTPBody bytes]]);
//            }
//        }
#endif
        //////////////////////////////////////////////////
        
        NSDictionary *routingMetadata = nil;
        
        if(parameters!=nil)
        {
            routingMetadata = @{ @"query": @{ @"parameters": parameters } };
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            operation.mappingMetadata = routingMetadata;
            
            //////////////////////////////////////////////////
            
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);

    //////////////////////////////////////////////////

    return operation;
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postWithPath:(NSString *)path
                           byObjectManager:(RKObjectManager *)objectManager
                            parameterArray:(NSArray *)parameterArray
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:nil method:RKRequestMethodPOST path:path parameters:nil];
        
        //////////////////////////////////////////////////

        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////

        NSString *charset = (__bridge NSString *)CFStringConvertEncodingToIANACharSetName(CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding));
        [request setValue:[NSString stringWithFormat:@"application/json; charset=%@", charset] forHTTPHeaderField:@"Content-Type"];

        //////////////////////////////////////////////////

        if(parameterArray.count>0)
        {
            NSData *data = [NSJSONSerialization dataWithJSONObject:parameterArray options:NSJSONWritingPrettyPrinted error:nil];
            
            if(data!=nil)
            {
                [request setHTTPBody:data];
            }
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;

            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);

    return operation;
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postWithPath:(NSString *)path
                           byObjectManager:(RKObjectManager *)objectManager
                                    object:(id)object
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self requestMethod:RKRequestMethodPOST
                          path:path
               byObjectManager:objectManager
                        object:object
          httpHeaderParameters:httpHeaderParameters
                       success:success
                       failure:failure];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance DELETE Method


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)deleteObject:(id)object
                           byObjectManager:(RKObjectManager *)objectManager
                                      path:(NSString *)path
                                parameters:(NSDictionary *)parameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self deleteObject:object
              byObjectManager:objectManager
                         path:path
                   parameters:parameters
         httpHeaderParameters:nil
                      success:success
                      failure:failure];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)deleteObject:(id)object
                           byObjectManager:(RKObjectManager *)objectManager
                                      path:(NSString *)path
                                parameters:(NSDictionary *)parameters
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:RKRequestMethodDELETE path:path parameters:parameters];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        // HTTPBody ，針對 Dictionary [NSNull Null] 的 Value，重新解析
        if(request.HTTPBody!=nil)
        {
            NSDictionary *noNullBodyDictinary = [self dictionaryRemovingNullsWithStringsFromDictionary:[NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil]];
            
            if(noNullBodyDictinary!=nil)
            {
                [request setHTTPBody:[NSJSONSerialization dataWithJSONObject:noNullBodyDictinary
                                                                     options:(NSJSONWritingOptions)0
                                                                       error:nil]];
            }
        }
        
        //////////////////////////////////////////////////
        
#ifdef DEBUG
        //打印 HTTPBody 內容
        //        if(request.HTTPBody!=nil)
        //        {
        //            NSDictionary *bodyDictinary = [NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil];
        //
        //            // json format
        //            if(bodyDictinary!=nil)
        //            {
        //                NSLog(@"%s, bodyDictinary:%@",__func__,bodyDictinary);
        //            }
        //            // url encoding
        //            else
        //            {
        //                NSLog(@"%s, bodyString:%@",__func__,[NSString stringWithUTF8String:[request.HTTPBody bytes]]);
        //            }
        //        }
#endif
        //////////////////////////////////////////////////
        
        NSDictionary *routingMetadata = nil;
        
        if(parameters!=nil)
        {
            routingMetadata = @{ @"query": @{ @"parameters": parameters } };
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            operation.mappingMetadata = routingMetadata;
            
            //////////////////////////////////////////////////
            
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace PUT Method

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)putObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self putObject:object
           byObjectManager:objectManager
                      path:path
                parameters:parameters
      httpHeaderParameters:nil
                   success:success
                   failure:failure];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)putObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                   httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:RKRequestMethodPUT path:path parameters:parameters];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        // HTTPBody ，針對 Dictionary [NSNull Null] 的 Value，重新解析
        if(request.HTTPBody!=nil)
        {
            NSDictionary *noNullBodyDictinary = [self dictionaryRemovingNullsWithStringsFromDictionary:[NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil]];
            
            if(noNullBodyDictinary!=nil)
            {
                [request setHTTPBody:[NSJSONSerialization dataWithJSONObject:noNullBodyDictinary
                                                                     options:(NSJSONWritingOptions)0
                                                                       error:nil]];
            }
        }
        
        //////////////////////////////////////////////////
        
#ifdef DEBUG
        //打印 HTTPBody 內容
        //        if(request.HTTPBody!=nil)
        //        {
        //            NSDictionary *bodyDictinary = [NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil];
        //
        //            // json format
        //            if(bodyDictinary!=nil)
        //            {
        //                NSLog(@"%s, bodyDictinary:%@",__func__,bodyDictinary);
        //            }
        //            // url encoding
        //            else
        //            {
        //                NSLog(@"%s, bodyString:%@",__func__,[NSString stringWithUTF8String:[request.HTTPBody bytes]]);
        //            }
        //        }
#endif
        //////////////////////////////////////////////////
        
        NSDictionary *routingMetadata = nil;
        
        if(parameters!=nil)
        {
            routingMetadata = @{ @"query": @{ @"parameters": parameters } };
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            operation.mappingMetadata = routingMetadata;
            
            //////////////////////////////////////////////////
            
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)putWithPath:(NSString *)path
                          byObjectManager:(RKObjectManager *)objectManager
                                 httpBody:(NSData *)httpBody
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self requestMethod:RKRequestMethodPUT
                          path:path
               byObjectManager:objectManager
                      httpBody:httpBody
          httpHeaderParameters:httpHeaderParameters
                       success:success
                       failure:failure];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)putWithPath:(NSString *)path
                          byObjectManager:(RKObjectManager *)objectManager
                                   object:(id)object
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self requestMethod:RKRequestMethodPUT
                          path:path
               byObjectManager:objectManager
                        object:object
          httpHeaderParameters:httpHeaderParameters
                       success:success
                       failure:failure];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance PATCH Method

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self patchObject:object
             byObjectManager:objectManager
                        path:path
                  parameters:parameters
        httpHeaderParameters:nil
           removeNullsInBody:YES
                     success:success
                     failure:failure];
}



//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self patchObject:object
             byObjectManager:objectManager
                        path:path
                  parameters:parameters
        httpHeaderParameters:httpHeaderParameters
           removeNullsInBody:YES
                     success:success
                     failure:failure];
}

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                        removeNullsInBody:(BOOL)removeNullsInBody
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //客製化HTTP Header
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object method:RKRequestMethodPATCH path:path parameters:parameters];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        // HTTPBody ，針對 Dictionary [NSNull Null] 的 Value，重新解析
        if(removeNullsInBody==YES &&
           request.HTTPBody!=nil)
        {
            NSDictionary *noNullBodyDictinary = [self dictionaryRemovingNullsWithStringsFromDictionary:[NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil]];
            
            if(noNullBodyDictinary!=nil)
            {
                [request setHTTPBody:[NSJSONSerialization dataWithJSONObject:noNullBodyDictinary
                                                                     options:(NSJSONWritingOptions)0
                                                                       error:nil]];
            }
        }
        
        //////////////////////////////////////////////////
        
#ifdef DEBUG
        //打印 HTTPBody 內容
        //        if(request.HTTPBody!=nil)
        //        {
        //            NSDictionary *bodyDictinary = [NSJSONSerialization JSONObjectWithData:request.HTTPBody options:NSJSONReadingMutableLeaves error:nil];
        //
        //            // json format
        //            if(bodyDictinary!=nil)
        //            {
        //                NSLog(@"%s, bodyDictinary:%@",__func__,bodyDictinary);
        //            }
        //            // url encoding
        //            else
        //            {
        //                NSLog(@"%s, bodyString:%@",__func__,[NSString stringWithUTF8String:[request.HTTPBody bytes]]);
        //            }
        //        }
#endif
        //////////////////////////////////////////////////
        
        NSDictionary *routingMetadata = nil;
        
        if(parameters!=nil)
        {
            routingMetadata = @{ @"query": @{ @"parameters": parameters } };
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            operation.mappingMetadata = routingMetadata;
            
            //////////////////////////////////////////////////
            
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;
            
            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)patchWithPath:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                   httpBody:(NSData *)httpBody
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self requestMethod:RKRequestMethodPATCH
                          path:path
               byObjectManager:objectManager
                      httpBody:httpBody
          httpHeaderParameters:httpHeaderParameters
                       success:success
                       failure:failure];
}





//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)patchWithPath:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                     object:(id)object
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    return [self requestMethod:RKRequestMethodPATCH
                          path:path
               byObjectManager:objectManager
                        object:object
          httpHeaderParameters:httpHeaderParameters
                       success:success
                       failure:failure];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Batch Method

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)multipartFormRequestWithObject:(id)object
                                             byObjectManager:(RKObjectManager *)objectManager
                                                      method:(RKRequestMethod)method
                                                        path:(NSString *)path
                                                        data:(NSData *)data
                                                        name:(NSString *)name
                                                    fileName:(NSString *)fileName
                                                    mimeType:(NSString *)mimeType
                                                  parameters:(NSDictionary *)parameters
                                        httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                                     success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                                     failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    RKObjectRequestOperation *operation = nil;
    
    do
    {
        if(path==nil ||
           data==nil ||
           [name length]<=0 ||
           [fileName length]<=0 ||
           [mimeType length]<=0)
        {
            break;
        }
        
        if(objectManager==nil)
        {
            break;
        }

        //////////////////////////////////////////////////
        
        NSMutableURLRequest *request = [objectManager multipartFormRequestWithObject:object
                                                                              method:method
                                                                                path:path
                                                                          parameters:parameters
                                                           constructingBodyWithBlock:^(id<AFRKMultipartFormData> formData){
                                                               
                                                               [formData appendPartWithFileData:data
                                                                                           name:name
                                                                                       fileName:fileName
                                                                                       mimeType:mimeType];
                                                           }];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////

        //客製化HTTP Header
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        operation = [objectManager objectRequestOperationWithRequest:request success:nil failure:nil];
        
        if(operation!=nil)
        {
            // callbackQueue是nil時，底層會用mainQueue回傳。
            operation.successCallbackQueue = self.successCallbackQueue;
            operation.failureCallbackQueue = self.failureCallbackQueue;

            operation.targetObject = nil;
            [operation setCompletionBlockWithSuccess:success failure:failure];
            [objectManager enqueueObjectRequestOperation:operation];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Image or Data Method

//================================================================================
//
//================================================================================
- (AFRKImageRequestOperation *)getImageWithObject:(id)object
                                    objectManager:(RKObjectManager *)objectManager
                                             path:(NSString *)path
                                       parameters:(NSDictionary *)parameters
                             httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                          success:(void (^)(NSURLRequest *request, NSHTTPURLResponse *response, CPImage *image))success
                                          failure:(void (^)(NSURLRequest *request, NSHTTPURLResponse *response, NSError *error))failure
{
    AFRKImageRequestOperation *operation = nil;
    
    do
    {
        if(path==nil ||
           httpHeaderParameters==nil ||
           httpHeaderParameters.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
     
        NSMutableURLRequest *request = [objectManager requestWithObject:object
                                                                 method:RKRequestMethodGET
                                                                   path:path
                                                             parameters:parameters];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }

        request.timeoutInterval = self.timeOutInterval;
    
        //////////////////////////////////////////////////

        //客製化HTTP Header
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////

        operation = [AFRKImageRequestOperation imageRequestOperationWithRequest:request
                                                           imageProcessingBlock:^CPImage *(CPImage *image)
                     {
                         return image;
                     }
                                                                        success:success
                                                                        failure:failure];
      
        if(operation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        operation.imageScale = 1.0;
#endif
        
        [objectManager.HTTPClient.operationQueue addOperation:operation];
        
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}



//================================================================================
//
//================================================================================
- (AFRKHTTPRequestOperation *)getDataWithObject:(id)object
                                    objectManager:(RKObjectManager *)objectManager
                                             path:(NSString *)path
                                       parameters:(NSDictionary *)parameters
                             httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                          success:(void (^)(AFRKHTTPRequestOperation *operation, NSData *responseData))success
                                          failure:(void (^)(AFRKHTTPRequestOperation *operation, NSError *error))failure
{
    AFRKHTTPRequestOperation *operation = nil;
    
    do
    {
        if(path==nil ||
           httpHeaderParameters==nil ||
           httpHeaderParameters.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(objectManager==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSMutableURLRequest *request = [objectManager requestWithObject:object
                                                                 method:RKRequestMethodGET
                                                                   path:path
                                                             parameters:parameters];
        
        //////////////////////////////////////////////////
        
        if(request==nil)
        {
            break;
        }
        
        request.timeoutInterval = self.timeOutInterval;
        
        //////////////////////////////////////////////////

        //客製化HTTP Header
        
        for(id key in httpHeaderParameters)
        {
            [request setValue:[httpHeaderParameters objectForKey:key] forHTTPHeaderField:key];
        }
        
        //////////////////////////////////////////////////
        
        operation =  [objectManager.HTTPClient HTTPRequestOperationWithRequest:request
                                                                       success:success
                                                                       failure:failure];
        operation.successCallbackQueue = self.successCallbackQueue;
        operation.failureCallbackQueue = self.failureCallbackQueue;
        
        if(operation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        [objectManager.HTTPClient.operationQueue addOperation:operation];
        
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    return operation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance IP Method

//================================================================================
//
//================================================================================
- (NSString *)ipAddress
{
    NSString *resultIPString = @"";
    
    NSArray *ipAddresses = [self ipAddresses];
    
    //////////////////////////////////////////////////
    // 過濾ipV6
    // 過濾127.0.0.1
    NSArray *sortedIPAddresses = [ipAddresses sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)];
    
    NSNumberFormatter *numberFormatter = [[[NSNumberFormatter alloc] init] autorelease];
    numberFormatter.allowsFloats = NO;
    
    for (NSString *potentialIPAddress in sortedIPAddresses)
    {
        if ([potentialIPAddress isEqualToString:@"127.0.0.1"])
        {
            continue;
        }
        
        NSArray *ipParts = [potentialIPAddress componentsSeparatedByString:@"."];
        BOOL isMatch = YES;
        
        for (NSString *ipPart in ipParts)
        {
            if ([numberFormatter numberFromString:ipPart]==nil)
            {
                isMatch = NO;
                break;
            }
        }
        
        if (isMatch)
        {
            if ([resultIPString length]>0)
            {
                resultIPString = [resultIPString stringByAppendingString:@"    "];
            }
            resultIPString = [resultIPString stringByAppendingString:potentialIPAddress];
            
        }
    }
    
    //////////////////////////////////////////////////
    if ([resultIPString length]>0)
    {
        return resultIPString;
    }
    else
    {
        // No IP found
        return @"?.?.?.?";
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Operation Method

//================================================================================
//
//================================================================================
- (void)cancellAllRestOperation
{
    [[[self objectManagerForType:PPRCObjectManagerType_JSON] operationQueue] cancelAllOperations];
    [[[self objectManagerForType:PPRCObjectManagerType_URLEncoding] operationQueue] cancelAllOperations];
    [[[self objectManagerForType:PPRCObjectManagerType_Image] operationQueue] cancelAllOperations];
    
    [[[[self objectManagerForType:PPRCObjectManagerType_JSON] HTTPClient] operationQueue] cancelAllOperations];
    [[[[self objectManagerForType:PPRCObjectManagerType_URLEncoding] HTTPClient] operationQueue] cancelAllOperations];
    [[[self objectManagerForType:PPRCObjectManagerType_Image] operationQueue] cancelAllOperations];
}


//================================================================================
//
//================================================================================
- (void)waitAllRestOperaitonAreFinished
{
    [[[self objectManagerForType:PPRCObjectManagerType_JSON] operationQueue] waitUntilAllOperationsAreFinished];
    [[[self objectManagerForType:PPRCObjectManagerType_URLEncoding] operationQueue] waitUntilAllOperationsAreFinished];
    [[[self objectManagerForType:PPRCObjectManagerType_Image] operationQueue] waitUntilAllOperationsAreFinished];
    
    [[[[self objectManagerForType:PPRCObjectManagerType_JSON] HTTPClient] operationQueue] waitUntilAllOperationsAreFinished];
    [[[[self objectManagerForType:PPRCObjectManagerType_URLEncoding] HTTPClient] operationQueue] waitUntilAllOperationsAreFinished];
    [[[[self objectManagerForType:PPRCObjectManagerType_Image] HTTPClient] operationQueue] waitUntilAllOperationsAreFinished];
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Image Data Method

//================================================================================
//
//================================================================================
- (NSData *)dataForImage:(CPImage *)image mineType:(NSString *)mineType
{
    NSData *data = nil;
 
    if([mineType isEqualToString:PPRestClientImageType_Jpg] == YES)
    {
#if TARGET_OS_IPHONE
        
        data = UIImageJPEGRepresentation(image, 0.9);
        
#elif TARGET_OS_MAC
        
        data = [image jpegRepresentationWithCompressQuality:0.9];
#endif
        
    }
    else if([mineType isEqualToString:PPRestClientImageType_Png] == YES)
    {
#if TARGET_OS_IPHONE
        
        data = UIImagePNGRepresentation(image);
        
#elif TARGET_OS_MAC
        
        data = [image pngRepresentation];
#endif
        
    }
    
    return data;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//==============================================================================
//
//==============================================================================
+ (NSHTTPURLResponse *)responseFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil) return nil;
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self responseFromAFRKNetworkingError:subError];
            }
        }
        
    }
    
    return [[error userInfo] objectForKey:AFRKNetworkingOperationFailingURLResponseErrorKey];
}


//==============================================================================
//
//==============================================================================
+ (NSInteger)statusCodeFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil) return NSNotFound;
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self statusCodeFromAFRKNetworkingError:subError];
            }
        }
        
    }
    else
    {
        return [error code];
    }
    
    NSHTTPURLResponse *response = [[error userInfo] objectForKey:AFRKNetworkingOperationFailingURLResponseErrorKey];
    if (response==nil)
    {
        return NSNotFound;
    }
    
    return [response statusCode];
}


//==============================================================================
//
//==============================================================================
+ (NSString *)domainFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil)
    {
        return nil;
    }
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self domainFromAFRKNetworkingError:subError];
            }
        }
        
    }
    else
    {
        return [error domain];
    }
    
    return [error domain];
}

@end
