//
//  PPRefreshController.m
//  

#import "PPRefreshController.h"
#import "PPLogController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

//#define _DUMP_LOG_

#ifdef _DUMP_LOG_
// !! 因為要檢查物件life cycle所以logController要放在global
PPLogController *g_logController = nil;
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

NSTimeInterval const PPRC_DefaultTriggerHeight = 80.0;
NSTimeInterval const PPRC_DefaultAnimationDuration = 0.3;

NSString * const PPRC_ObserveKey_ContentInset = @"contentInset";
NSString * const PPRC_ObserveKey_ContentOffset = @"contentOffset";
NSString * const PPRC_ObserveKey_ContentSize = @"contentSize";
NSString * const PPRC_ObserveKey_Frame = @"frame";

CGFloat const PPRC_ActivityIndicatorMargin = 5.0;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPRefreshController ()
@property (nonatomic, assign) BOOL canRefresh;
@property (nonatomic, assign) BOOL isTriggered;
@property (nonatomic, retain) UIView *refreshView;
@property (nonatomic, retain) PPActivityIndicatorView *activityIndicatorView;
@property (nonatomic, assign) UIEdgeInsets originalContentInset;
@property (nonatomic, assign) CGPoint originalContentOffset;
@property (nonatomic, assign) CGFloat originalContentHeieght;
@property (nonatomic, assign) CGFloat activityIndicatorSize;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPRefreshController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)initWithDelegate:(id<PPRefreshControllerDelegate>)delegate
{
    if((self=[super init]))
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

        self.delegate = delegate;
        self.enabled = NO;
        self.canRefresh = YES;
        self.triggerHeight = PPRC_DefaultTriggerHeight;
        self.infoPosition = PPRefreshControllerInfoPosition_Top;
        
        //////////////////////////////////////////////////
        
        _refreshView = [[UIView alloc] init];
        self.refreshView.backgroundColor = [UIColor clearColor];
        self.refreshView.clipsToBounds = YES;
        
        switch ([PPActivityIndicatorView appearance].style)
        {
            case PPActivityIndicatorViewStyleSystem:
                self.activityIndicatorSize = PPAIV_SystemStyleDefaultSize;
                break;

            case PPActivityIndicatorViewStyleGradientCircle:
                self.activityIndicatorSize = PPAIV_GradientCircleStyleDefaultSize;
                break;

            default:
                break;
        }
        
        CGRect frame = CGRectMake(0, 0, self.activityIndicatorSize, self.activityIndicatorSize);
        self.activityIndicatorView = [[[PPActivityIndicatorView alloc] initWithFrame:frame] autorelease];
        [self.refreshView addSubview:self.activityIndicatorView];
    }
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];
	
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    [self.refreshView removeFromSuperview];
    self.refreshView = nil;
    
    [self.activityIndicatorView removeFromSuperview];
    self.activityIndicatorView = nil;
    
    [self.infoView removeFromSuperview];
    self.infoView = nil;
 
    self.targetScrollView = nil;
    self.backgroundColor = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Property methods

//================================================================================
//
//================================================================================
- (void)setTriggerHeight:(CGFloat)triggerHeight
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    _triggerHeight = triggerHeight;
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];

}


//================================================================================
//
//================================================================================
- (void)setTargetScrollView:(UIScrollView *)targetScrollView
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(_targetScrollView == targetScrollView)
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) out 1", __func__, self];

        return;
    }

    self.originalContentInset = targetScrollView.contentInset;
    self.originalContentOffset = targetScrollView.contentOffset;
    
    
    //////////////////////////////////////////////////
    // 移除舊監測
    
    [_targetScrollView removeObserver:self forKeyPath:PPRC_ObserveKey_ContentInset];
    [_targetScrollView removeObserver:self forKeyPath:PPRC_ObserveKey_ContentOffset];
    [_targetScrollView removeObserver:self forKeyPath:PPRC_ObserveKey_ContentSize];
    [_targetScrollView removeObserver:self forKeyPath:PPRC_ObserveKey_Frame];

    
    //////////////////////////////////////////////////
    
    [self.refreshView removeFromSuperview];
    [_targetScrollView release];
    _targetScrollView = [targetScrollView retain];

    
    //////////////////////////////////////////////////
    // 加入新監測

    [_targetScrollView addObserver:self forKeyPath:PPRC_ObserveKey_ContentInset options:NSKeyValueObservingOptionNew context:NULL];
    [_targetScrollView addObserver:self forKeyPath:PPRC_ObserveKey_ContentOffset options:NSKeyValueObservingOptionNew context:NULL];
    [_targetScrollView addObserver:self forKeyPath:PPRC_ObserveKey_ContentSize options:NSKeyValueObservingOptionNew context:NULL];
    [_targetScrollView addObserver:self forKeyPath:PPRC_ObserveKey_Frame options:NSKeyValueObservingOptionNew context:NULL];
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out 2", __func__, self];
}


//================================================================================
//
//================================================================================
- (void)setBackgroundColor:(UIColor *)backgroundColor
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];
    
    // test
//    backgroundColor = [UIColor redColor];

    [_backgroundColor release];
    _backgroundColor = [backgroundColor retain];
    
    self.refreshView.backgroundColor = backgroundColor;
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];
}


//================================================================================
//
//================================================================================
- (void)setInfoView:(UIView *)infoView
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(_infoView == infoView)
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) out 1", __func__, self];

        return;
    }
    
    [_infoView release];
    _infoView = [infoView retain];
    _infoView.clipsToBounds = YES;
    
    // test
//    [_infoView addObserver:self forKeyPath:PPRC_ObserveKey_Frame options:NSKeyValueObservingOptionNew context:nil];
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out 2", __func__, self];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - NSKeyValueObserving

//================================================================================
//
//================================================================================
- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{
    do
    {
        //////////////////////////////////////////////////
        // info顯示在下方模式時，如果內容高度比框架還少就不用處理了。
        
        if(self.infoPosition == PPRefreshControllerInfoPosition_Bottom &&
           self.targetScrollView.contentSize.height <= self.targetScrollView.frame.size.height)
        {
            break;
        }

        
        //////////////////////////////////////////////////
        
        if([keyPath isEqualToString:PPRC_ObserveKey_Frame]==YES)
        {
            // test
//            NSLog(@"change = %@", change);
        }
        else if([keyPath isEqualToString:PPRC_ObserveKey_ContentInset]==YES)
        {
            // 記錄外部contentInset變更（要避開hide/show infoView時的內部變更）
            if(self.isTriggered == NO)
            {
                self.originalContentInset = self.targetScrollView.contentInset;
            }
        }
        else if([keyPath isEqualToString:PPRC_ObserveKey_ContentOffset]==YES)
        {
            if(self.enabled == NO || self.canRefresh == NO)
            {
                break;
            }


            //////////////////////////////////////////////////

            if(self.infoPosition == PPRefreshControllerInfoPosition_Top)
            {
                //////////////////////////////////////////////////
                // MARK: 往下拉 - 注意refreshView在上方顯示時是固定在最上方，和在下方顯示時不同。
             
                // 上下算法不一樣
                CGFloat refreshVisibleHeight = self.originalContentOffset.y - self.targetScrollView.contentOffset.y;
                
//                NSLog(@"refreshVisibleHeightdistance=%f (%f, %f)", refreshVisibleHeight, self.originalContentOffset.y, self.targetScrollView.contentOffset.y);
                
                if(refreshVisibleHeight < 0)
                {
                    // 處理拖曳時瞬間反向滑動的情況
                    self.isTriggered = NO;
                    
                    if(self.refreshView.superview != nil)
                    {
                        [self.refreshView removeFromSuperview];
                    }
                    
                    break;
                }
                

                //////////////////////////////////////////////////
                // 檢查是否已觸發
                
                if (refreshVisibleHeight > self.triggerHeight && self.targetScrollView.isTracking == YES)
                {
                    self.isTriggered = YES;
                }
                
                
                //////////////////////////////////////////////////
                // 加入或移除refreshView (parent是self.targetScrollView.superview)

                // refreshView顯示範圍
                CGRect frame = CGRectMake(0,
                                          -(self.originalContentOffset.y+self.originalContentInset.top)+self.refreshDisplayOffsetY,
                                          self.targetScrollView.bounds.size.width,
                                          refreshVisibleHeight);
                
                // test
//                NSLog(@"refreshView.frame = %@", NSStringFromCGRect(frame));
                
                self.refreshView.frame = frame;
                
                if(self.targetScrollView.isTracking == YES && self.refreshView.superview == nil)
                {
                    [self.targetScrollView.superview addSubview:self.refreshView];
                }
                else if(refreshVisibleHeight < PPRC_ActivityIndicatorMargin && self.refreshView.superview != nil)
                {
                    [self.refreshView removeFromSuperview];
                }
                
                
                //////////////////////////////////////////////////
                // 設定indicator

                [self updateIndicatorWithRefreshVisibleHeight:refreshVisibleHeight];
               
                
                //////////////////////////////////////////////////
                // 顯示InfoView或固定refreshView
                
                [self dockViewWithRefreshVisibleHeight:refreshVisibleHeight];
            }
            else // self.infoPosition == PPRefreshControllerInfoPosition_Bottom
            {
                //////////////////////////////////////////////////
                // MARK: 往上拉 - 注意refreshView在下方顯示時，是加在scrollView的尾端，和在上方顯示時不同。
                
                // 上下算法不一樣
                CGFloat refreshVisibleHeight =  self.targetScrollView.contentOffset.y +
                                                self.targetScrollView.frame.size.height -
                                                self.targetScrollView.contentSize.height -
                                                self.originalContentInset.bottom;

                if(@available(iOS 11.0, *))
                {
                    refreshVisibleHeight -= self.targetScrollView.safeAreaInsets.bottom;
                }

//                NSLog(@"refreshVisibleHeightdistance=%f (%f, %f)", refreshVisibleHeight, self.originalContentOffset.y, self.originalContentInset.bottom);

                if(refreshVisibleHeight < 0)
                {
                    // 處理拖曳時瞬間反向滑動的情況
                    self.isTriggered = NO;
                    
                    if(self.refreshView.superview != nil)
                    {
                        [self.refreshView removeFromSuperview];
                    }

                    break;
                }

                
                //////////////////////////////////////////////////
                // 檢查是否已觸發
                
                if (refreshVisibleHeight > self.triggerHeight)
                {
                    if(self.canBeTriggeredWithoutTracking == YES ||
                       (self.canBeTriggeredWithoutTracking == NO && self.targetScrollView.isTracking == YES))
                    {
                        self.isTriggered = YES;
                    }
                }

                
                //////////////////////////////////////////////////
                // 加入或移除refreshView (parent是self.targetScrollView)
                
                // refreshView顯示範圍
                self.refreshView.frame = CGRectMake(0,
                                                    self.targetScrollView.contentSize.height,
                                                    self.targetScrollView.bounds.size.width,
                                                    refreshVisibleHeight);
                
                if(self.refreshView.superview == nil)
                {
                    if(self.canBeTriggeredWithoutTracking == YES ||
                       (self.canBeTriggeredWithoutTracking == NO && self.targetScrollView.isTracking == YES))
                    {
                        [self.targetScrollView addSubview:self.refreshView];
                    }
                }
                else if(refreshVisibleHeight < PPRC_ActivityIndicatorMargin && self.refreshView.superview != nil)
                {
                    [self.refreshView removeFromSuperview];
                }

                
                //////////////////////////////////////////////////
                // 設定indicator
                
                [self updateIndicatorWithRefreshVisibleHeight:refreshVisibleHeight];
                
                
                //////////////////////////////////////////////////
                // 判斷是否要做dock動作
                
                [self dockViewWithRefreshVisibleHeight:refreshVisibleHeight];
             }
        }
        
    }while(0);
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private  Methods

//================================================================================
//
//================================================================================
- (void)updateIndicatorWithRefreshVisibleHeight:(CGFloat)refreshVisibleHeight
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(self.refreshView.superview != nil)
    {
        self.activityIndicatorView.center = CGPointMake(self.refreshView.bounds.size.width/2,
                                                        PPRC_ActivityIndicatorMargin+self.activityIndicatorView.bounds.size.height/2);
        
        if (refreshVisibleHeight >= self.triggerHeight)
        {
            [self.activityIndicatorView startAnimating];
        }
        else if(self.isTriggered == NO)
        {
            [self.activityIndicatorView stopAnimating];
            [self.activityIndicatorView updateCurrentProgress:(refreshVisibleHeight/self.triggerHeight)
                                            withVisibleHeight:@(refreshVisibleHeight-2*PPRC_ActivityIndicatorMargin)];
        }
    }
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];
}


//================================================================================
//
//================================================================================
- (CGFloat)refreshViewDockHeight
{
    [PPRefreshController logMessageWithFormat:@"%s (%p)", __func__, self];

    return self.activityIndicatorSize+2*PPRC_ActivityIndicatorMargin;
}


//================================================================================
//
//================================================================================
- (void)dockViewWithRefreshVisibleHeight:(CGFloat)refreshVisibleHeight
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(self.isTriggered == NO || self.targetScrollView.isTracking == YES)
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) out 1", __func__, self];
        return;
    }

    
    //////////////////////////////////////////////////
    // 由refreshView的高度判斷是否要進行dock動作
    
    CGFloat dockViewHeight;
    
    if(self.infoView != nil)
    {
        dockViewHeight = self.infoView.frame.size.height;
    }
    else
    {
        dockViewHeight = [self refreshViewDockHeight];
    }
    
    if(refreshVisibleHeight > dockViewHeight+3)
    {
        return;
    }
    else
    {
        if([self.delegate respondsToSelector:@selector(refreshControllerInfoViewWillAppear:)])
        {
            [self.delegate refreshControllerInfoViewWillAppear:self];
        }
        
        self.canRefresh = NO;
    }

    // 記錄顯示dockView時的內容高度，供之後隱藏dockView時進行判斷。
    self.originalContentHeieght = self.targetScrollView.contentSize.height;

    
    //////////////////////////////////////////////////
    // 計算dock位置

    UIEdgeInsets contentInset = self.targetScrollView.contentInset;
    UIView *dockView = self.infoView ? self.infoView : self.refreshView;
    
    dockView.backgroundColor = self.backgroundColor;
    
    if(self.infoPosition == PPRefreshControllerInfoPosition_Top)
    {
        contentInset.top = dockViewHeight;
        self.targetScrollView.contentInset = contentInset;
        
        if(dockView == self.refreshView)
        {
            CGRect frame = self.refreshView.frame;
            frame.size.height = [self refreshViewDockHeight];
            self.refreshView.frame = frame;
            self.activityIndicatorView.center = CGPointMake(CGRectGetMidX(self.refreshView.bounds), CGRectGetMidY(self.refreshView.bounds));
            [self.targetScrollView.superview addSubview:self.refreshView];
        }
        else
        {
            CGRect frame = dockView.frame;
            frame.origin.y = self.refreshView.frame.origin.y;
            dockView.frame = frame;
            
            [self.targetScrollView.superview addSubview:dockView];
            [self.refreshView removeFromSuperview];
        }
    }
    else // PPRefreshControllerInfoPosition_Bottom
    {
        if(dockView == self.refreshView)
        {
            [self.refreshView removeFromSuperview];
            
            CGRect frame = self.refreshView.frame;
            frame.origin.y = self.targetScrollView.contentSize.height;
            frame.size.height = [self refreshViewDockHeight];
            self.refreshView.frame = frame;
            self.activityIndicatorView.center = CGPointMake(CGRectGetMidX(self.refreshView.bounds), CGRectGetMidY(self.refreshView.bounds));
        }
        else
        {
            CGRect frame = self.infoView.frame;
            frame.origin.y = self.targetScrollView.contentSize.height;
            self.infoView.frame = frame;
        }

        [self.targetScrollView addSubview:dockView];
        
        contentInset.bottom = dockView.bounds.size.height+self.originalContentInset.bottom;
        self.targetScrollView.contentInset = contentInset;
    }


    //////////////////////////////////////////////////
    // 使用infoView做dock動作，完畢時要把refreshView移開。
    
    if(dockView == self.infoView)
    {
        [self.refreshView removeFromSuperview];
    }
    
    
    //////////////////////////////////////////////////
    
    if([self.delegate respondsToSelector:@selector(refreshControllerInfoViewDidAppear:)])
    {
        [self.delegate refreshControllerInfoViewDidAppear:self];
    }
    
    [PPRefreshController logMessageWithFormat:@"%s (%p) out", __func__, self];

}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (void)showInfoView
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(self.isTriggered == YES)
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) out 1", __func__, self];
        return;
    }

    self.isTriggered = YES;
    
    //////////////////////////////////////////////////

    __block UIView *dockView = self.infoView ? self.infoView:self.refreshView;
    
    dockView.hidden = YES;
    [self dockViewWithRefreshVisibleHeight:0];
    [self.activityIndicatorView startAnimating];
    
    if(self.infoPosition == PPRefreshControllerInfoPosition_Top)
    {
        __block CGFloat originalHeight;
        
        originalHeight = dockView.frame.size.height;
        
        CGRect frame = dockView.frame;
        frame.size.height = 0;
        dockView.frame = frame;
        
        dockView.hidden = NO;
        
        [UIView animateWithDuration:PPRC_DefaultAnimationDuration animations:^{
            
            CGRect frame = dockView.frame;
            frame.size.height = originalHeight;
            dockView.frame = frame;
            
        } completion:^(BOOL finished) {
            
            [PPRefreshController logMessageWithFormat:@"%s (%p) out 2", __func__, self];

        }];
    }
    else
    {
        dockView.alpha = 0;
        
        [UIView animateWithDuration:PPRC_DefaultAnimationDuration animations:^{
            
            dockView.alpha = 1;
            dockView.hidden = NO;

        } completion:^(BOOL finished) {

            [PPRefreshController logMessageWithFormat:@"%s (%p) out 3", __func__, self];

        }];
    }
}


//================================================================================
//
//================================================================================
- (void)hideInfoView
{
    [PPRefreshController logMessageWithFormat:@"%s (%p) in", __func__, self];

    if(self.isTriggered == NO)
    {
        [PPRefreshController logMessageWithFormat:@"%s (%p) out 1", __func__, self];
        return;
    }

    //////////////////////////////////////////////////

    if([self.delegate respondsToSelector:@selector(refreshControllerInfoViewWillDisappear:)])
    {
        [self.delegate refreshControllerInfoViewWillDisappear:self];
    }
    
    //////////////////////////////////////////////////

    __block UIView *dockView = self.infoView ? self.infoView:self.refreshView;
    __block BOOL needResetOffset = NO;
    __block CGFloat originalHeight;

    // !! 避免animation過程中self被釋放
    __block typeof(self) blockSelf = [self retain];

    // !! 避免animation過程中infoView被釋放
    [dockView retain];
    
    if(self.infoPosition == PPRefreshControllerInfoPosition_Top)
    {
        // scrollContent上方在refreshView範圍內時，隱藏時要一起調整offset。
        if(self.targetScrollView.contentOffset.y < self.originalContentOffset.y &&
           self.targetScrollView.contentOffset.y >= self.originalContentOffset.y-dockView.frame.size.height)
        {
            needResetOffset = YES;
        }
        
        
        [UIView animateWithDuration:PPRC_DefaultAnimationDuration animations:^{
            
            blockSelf.canRefresh = NO;
            originalHeight = dockView.frame.size.height;

            CGRect frame = dockView.frame;
            frame.size.height = 0;
            dockView.frame = frame;
            
            if(needResetOffset == YES)
            {
                blockSelf.targetScrollView.contentOffset = blockSelf.originalContentOffset;
            }

            // scrollView要動態上移需要在這裡設定inset
            blockSelf.targetScrollView.contentInset = blockSelf.originalContentInset;
            
        } completion:^(BOOL finished) {
            
            // 還原參數
            blockSelf.isTriggered = NO;
            [blockSelf.activityIndicatorView stopAnimating];
            [blockSelf.activityIndicatorView updateCurrentProgress:0 withVisibleHeight:nil];
            [dockView removeFromSuperview];

            if(dockView == blockSelf.infoView)
            {
                CGRect frame = dockView.frame;
                frame.size.height = originalHeight;
                dockView.frame = frame;
            }

            if([blockSelf.delegate respondsToSelector:@selector(refreshControllerInfoViewDidDisappear:)])
            {
                [blockSelf.delegate refreshControllerInfoViewDidDisappear:blockSelf];
            }

            blockSelf.canRefresh = YES;
            [blockSelf release];
            [dockView release];

            [PPRefreshController logMessageWithFormat:@"%s (%p) out 2", __func__, self];
        }];
    }
    else // PPRefreshControllerInfoPosition_Bottom
    {
        [UIView animateWithDuration:PPRC_DefaultAnimationDuration animations:^{
            
            blockSelf.canRefresh = NO;
            originalHeight = dockView.frame.size.height;
            
            CGRect frame = dockView.frame;
            frame.size.height = 0;
            dockView.frame = frame;
            
            // 內容沒變時，隱藏動作要特別處理
            if(blockSelf.originalContentHeieght == blockSelf.targetScrollView.contentSize.height)
            {
                // 縮起來的動作不要超過原始下拉範圍
                CGPoint offset = blockSelf.targetScrollView.contentOffset;
                offset.y = MAX(blockSelf.originalContentOffset.y, offset.y-originalHeight);
                blockSelf.targetScrollView.contentOffset = offset;
            }
            
        } completion:^(BOOL finished) {

            // contentInset一定要在這裡調整，不然畫面會跳動。
            UIEdgeInsets contentInset = blockSelf.targetScrollView.contentInset;
            contentInset.bottom -= originalHeight;
            blockSelf.targetScrollView.contentInset = contentInset;

            // 還原參數
            blockSelf.isTriggered = NO;
            [blockSelf.activityIndicatorView stopAnimating];
            [blockSelf.activityIndicatorView updateCurrentProgress:0 withVisibleHeight:nil];
            [dockView removeFromSuperview];

            if(dockView == blockSelf.infoView)
            {
                CGRect frame = dockView.frame;
                frame.size.height = originalHeight;
                dockView.frame = frame;
            }

            if([self.delegate respondsToSelector:@selector(refreshControllerInfoViewDidDisappear:)])
            {
                [self.delegate refreshControllerInfoViewDidDisappear:self];
            }
            
            blockSelf.canRefresh = YES;
            [blockSelf release];
            [dockView release];
            
            [PPRefreshController logMessageWithFormat:@"%s (%p) out 3", __func__, self];
        }];
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (instancetype)refreshControllerWithDelegate:(id<PPRefreshControllerDelegate>)delegate
{
    [self initLogController];
    
    PPRefreshController *controller = [[PPRefreshController alloc] initWithDelegate:delegate];
    
    return [controller autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - log methods

//================================================================================
//
//================================================================================
+ (PPLogController *)initLogController
{
#ifdef _DUMP_LOG_
    
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        
        g_logController = [[PPLogController alloc] init];
        
        NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        NSString *logDirPath = [paths[0] stringByAppendingString:@"/PPRefresController"];
        
        if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
        {
            [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                      withIntermediateDirectories:YES
                                                       attributes:nil
                                                            error:nil];
        }
        
        g_logController.toFile = YES;
        g_logController.toConsole = YES;
        [g_logController setFileName:@"log.txt" atPath:logDirPath];
        [g_logController setMask:PPLogControllerMask_Normal];
    });
    
    return g_logController;
#else
    return nil;
#endif
}


//===============================================================================
//
//===============================================================================
+ (void)logMessageWithFormat:(NSString *)format, ...
{
#ifdef _DUMP_LOG_
    va_list arguments;
    va_start(arguments, format);
    [g_logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
#endif
}

@end
