//
//  PPPatternOCRController.m
//  CameraTest
//
//  Created by cheng sanhue on 12/7/3.
//  Copyright (c) 2012年 Penpower Technology LTD. All rights reserved.
//

#import "PPPatternOCRController.h"
#import "WorldCardock.h"
#import "UIImage+BITMAPPTR.h"
#import "UIDevice+DeviceModel.h"
#import "PPImageEnhanceController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define DEFAULT_STABLE_FRAME_ForA4        2
#define DEFAULT_STABLE_FRAME_ForA5        5
#define DEFAULT_STABLE_FRAME_ForA6        15
#define DEFAULT_STABLE_FRAME_ForA7        20

#define NO_PATTERN_ALARM_COUNT            10

static PPPatternOCRController * g_ppPatternOCRController = nil;

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, PPPatternOCRControllerHoughTransform)
{
    PPPatternOCRControllerHoughTransform_Link,
    PPPatternOCRControllerHoughTransform_LinkQR,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPPatternOCRController ()
{
 @private
    void *			captram_;
}

@property (nonatomic, assign) NSUInteger noPatternCount;
@property (nonatomic, assign) NSUInteger stableFrame;			// 幾張後認定為穩定
@property (nonatomic, assign) BOOL readyToCapture;
@property (nonatomic, assign) CGPoint focusPointForPattern;     // pattern所在的對焦點
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPPatternOCRController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - init/dealloc

//==============================================================================
//
//==============================================================================
- (id)init
{
    self = [super init];
    if(self)
    {
        //////////////////////////////////////////////////
        // 初始化設定值
        self.currentState = CaptureState_Nothing;
		self.noPatternCount = 0;
        self.readyToCapture = NO;
        
        // !! 偵測不到時要用CGPointZero，讓外部可以判斷。
        self.focusPointForPattern = CGPointZero;
        
        [self setupStableFrame];
        //////////////////////////////////////////////////
        // 初始化核心，核心資料會記在captram_，所以要KEEP住
        CreateCaptram(&captram_);
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    // 釋放核心
    DestoryCaptram(&captram_);
    //////////////////////////////////////////////////
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - private methods


//==============================================================================
// 只有舊版dock在使用（辨識PENPOWER字串）
//==============================================================================
- (void)setupStableFrame
{
    self.stableFrame = DEFAULT_STABLE_FRAME_ForA7;
    
    // !! iPhone4 以後的機型因為速度太快，為了延遲判斷時間，
    //    所以要增加stableFrame數量。
    //------------------------------------------------
    
    PPDeviceModelID deviceModelID = [UIDevice deviceModelID];
    
    if((deviceModelID>=PPDeviceModelID_iPhone_4S&&deviceModelID<PPDeviceModelID_iPhone_5)||
       (deviceModelID>=PPDeviceModelID_iPod_5G&&deviceModelID<PPDeviceModelID_AppleTV))
    {
        self.stableFrame = DEFAULT_STABLE_FRAME_ForA5;
    }
    else if((deviceModelID>=PPDeviceModelID_iPhone_5&&deviceModelID<PPDeviceModelID_iPhone_5S)||
            (deviceModelID>=PPDeviceModelID_iPad_4&&deviceModelID<PPDeviceModelID_iPad_Air))
    {
        self.stableFrame = DEFAULT_STABLE_FRAME_ForA6;
    }
    else if((deviceModelID>=PPDeviceModelID_iPhone_5S&&deviceModelID<PPDeviceModelID_iPad)||
            (deviceModelID>=PPDeviceModelID_iPad_Air&&deviceModelID<PPDeviceModelID_iPod_1G))
    {
        self.stableFrame = DEFAULT_STABLE_FRAME_ForA7;
    }

}


//==============================================================================
// !! 核心的回傳的IMAGEREMOVE就是no pattern的意思，但因為時間太久，所以另外做一個判斷機制。
//==============================================================================
- (CaptureState)captureStateWithImage:(UIImage*)image
{
    if(image==nil)
    {
        return CaptureState_Nothing;
    }
    
    //////////////////////////////////////////////////
    // 影像轉為bitmap
    
    NSInteger result = CaptureState_Nothing;
    BITMAPPTR bitmapPTR = {0};
    [UIImage createBitmapPTR:&bitmapPTR withBitCount:32 image:image];
    
    //////////////////////////////////////////////////
    
    int imgStableResult = IMAGENOCAPTURE;

    if (self.forLinkVersion == YES)
    {
        // 核心判斷公式為 stable image >= (fr/2+1)，輸入的值要換算後再給核心
        imgStableResult = ImgStable(&captram_, bitmapPTR, (int)((self.stableFrame-1)*2));
    }
    else
    {
        // TODO: time_th為何設定為500? (eddie)
        imgStableResult = ImgStableQR(&captram_, bitmapPTR, 500);
    }

    //////////////////////////////////////////////////
    
//    //test
//    NSString *fileName = [NSString stringWithFormat:@"QRDockImage_%f_%d.png",[[NSDate new] timeIntervalSince1970],imgStableResult];
//    NSData *saveData = UIImagePNGRepresentation(image);
//    NSArray	*paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
//    NSString *saveFile = [NSString stringWithFormat:@"%@/%@", [paths firstObject], fileName];
//    [saveData writeToFile:saveFile atomically:YES];
    
    //////////////////////////////////////////////////
    
    switch (imgStableResult)
    {
        case IMAGESTART:
		{
			result = CaptureState_ReadyToCapture;
            self.noPatternCount = 0;
            self.readyToCapture = YES;
            
            
            //////////////////////////////////////////////////
            // !! 只有QRCode才能判斷對焦點，link不行
            
            if (!self.forLinkVersion)
            {
                // 計算對焦點
                CGFloat posYOfQRCode = (CGFloat)getQRPosy(&captram_);
                
                if (posYOfQRCode>0)
                {
                    self.focusPointForPattern = CGPointMake(0.5, posYOfQRCode/image.size.height);
                }
            }

			break;
		}
            
        case IMAGECAPTURE:
        {
            if(self.readyToCapture == YES)
            {
                result = CaptureState_CaptureNow;
                self.noPatternCount = 0;
                self.readyToCapture = NO;
            }
            
            break;
        }
            
        case IMAGENOCAPTURE:
        {
            break;
        }
            
        case IMAGEREMOVE:
        {
            // !! 核心的回傳的IMAGEREMOVE就是no pattern的意思，但因為時間太久，所以另外做一個判斷機制。
			break;
        }
            
        default:
		{
			result = CaptureState_Error;
            break;
		}
    }

    
    //////////////////////////////////////////////////
    // !! 核心的回傳的IMAGEREMOVE就是no pattern的意思，但因為時間太久，所以另外做一個判斷機制。
    
    if(self.readyToCapture==NO && result == CaptureState_Nothing)
    {
        self.noPatternCount++;
        
        if(self.noPatternCount >= NO_PATTERN_ALARM_COUNT)
        {
            // 'NO_PATTERN_ALARM_COUNT' (10)次沒有偵測到 Pattern
            result = CaptureState_NoPatternAlarm;
            self.noPatternCount = 0;
        }
    }
    
    //////////////////////////////////////////////////

	if(result != CaptureState_Nothing)
	{
		self.currentState = result;
	}

    //////////////////////////////////////////////////

    [UIImage releaseBitmapPTR:&bitmapPTR];
    
	return result;
}


//==============================================================================
//
//==============================================================================
+ (PPQuadrilateralModel *)houghTransformQuadrilateralModelForImage:(UIImage *)image
                          withPPPatternOCRControllerHoughTransform:(PPPatternOCRControllerHoughTransform)houghTransform
{
    PPQuadrilateralModel *ppQuadrilateralModel = nil;
    
    do
    {
        if(image==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(image.imageOrientation!=UIImageOrientationUp)
        {
            image = [image imageRotatedByDegrees:0];
        }
        
        //////////////////////////////////////////////////
        
        BITMAPPTR inputBitmapPTR;
        
        NSInteger result = 0;
        
        memset(&inputBitmapPTR, 0, sizeof(BITMAPPTR));
        
        if([UIImage createBitmapPTR:&inputBitmapPTR image:image]==YES)
        {
            MYPOINT myPoint[4];
            
            switch (houghTransform)
            {
                case PPPatternOCRControllerHoughTransform_Link:
                {
                    result = HoughTrans_Color_Link(&inputBitmapPTR, myPoint);
                    
                    break;
                }
                case PPPatternOCRControllerHoughTransform_LinkQR:
                default:
                {
                    result = HoughTrans_Color_LinkQR(&inputBitmapPTR, myPoint);
                    
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(result==0)
            {
                ppQuadrilateralModel = [[[PPQuadrilateralModel alloc] init] autorelease];
                if(ppQuadrilateralModel!=nil)
                {
                    ppQuadrilateralModel.bottomLeft  = CGPointMake(myPoint[0].x, abs(inputBitmapPTR.pHeader->biHeight)-myPoint[0].y);
                    ppQuadrilateralModel.bottomRight = CGPointMake(myPoint[1].x, abs(inputBitmapPTR.pHeader->biHeight)-myPoint[1].y);
                    ppQuadrilateralModel.topLeft     = CGPointMake(myPoint[2].x, abs(inputBitmapPTR.pHeader->biHeight)-myPoint[2].y);
                    ppQuadrilateralModel.topRight    = CGPointMake(myPoint[3].x, abs(inputBitmapPTR.pHeader->biHeight)-myPoint[3].y);
                }
            }
            
            [UIImage releaseBitmapPTR:&inputBitmapPTR];
        }
        
    }while(0);
    
    return ppQuadrilateralModel;
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - class methods


//==============================================================================
//
//==============================================================================
+ (PPPatternOCRController*)sharedPatternOCRController
{
    if (g_ppPatternOCRController==nil) {
        g_ppPatternOCRController = [[PPPatternOCRController alloc] init];
    }
    return g_ppPatternOCRController;
}


//==============================================================================
//
//==============================================================================
+ (CaptureState)captureStateWithImage:(UIImage*)image
{
    PPPatternOCRController * ppPatternOCRController = [PPPatternOCRController sharedPatternOCRController];
    if(ppPatternOCRController==nil)
    {
        return CaptureState_Error;
    }
    return [ppPatternOCRController captureStateWithImage:image];
}


//==============================================================================
//
//==============================================================================
+ (void)releasePatternOCRControler
{
    [g_ppPatternOCRController release];
    g_ppPatternOCRController = nil;
}


//==============================================================================
//
//==============================================================================
+ (CGPoint)focusPointForQRCodePosition
{
    PPPatternOCRController * ppPatternOCRController = [PPPatternOCRController sharedPatternOCRController];
    if(ppPatternOCRController==nil)
    {
        return CGPointZero;
    }

    return ppPatternOCRController.focusPointForPattern;
}


//==============================================================================
//
//==============================================================================
+ (PPQuadrilateralModel *)houghTransLinkQRformQuadrilateralModelForImage:(UIImage *)image;
{
    return [self houghTransformQuadrilateralModelForImage:image
                 withPPPatternOCRControllerHoughTransform:PPPatternOCRControllerHoughTransform_LinkQR];
}


//==============================================================================
//
//==============================================================================
+ (PPQuadrilateralModel *)houghTransLinkformQuadrilateralModelForImage:(UIImage *)image
{
    return [self houghTransformQuadrilateralModelForImage:image
                 withPPPatternOCRControllerHoughTransform:PPPatternOCRControllerHoughTransform_Link];
}
@end
