//
//  PPMagnifierView.m
//  
//
//  Created by Mike on 13/5/16.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPMagnifierView.h"
#if TARGET_OS_IPHONE
#import "UIViewController+Additions.h"
#import "UIViewController+ShareApplication.h"
#elif TARGET_OS_MAC
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPMagnifierView

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize style           = style_;
@synthesize	magnifyRatio    = magnifyRatio_;
@synthesize	offset          = offset_;
@synthesize	targetPoint     = targetPoint_;
@synthesize	targetView      = targetView_;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
// 初始化
//================================================================================
- (id)initWithFrame:(CGRect)frame
{
    if(CGRectIsEmpty(frame)==true)
    {
        frame.size = CGSizeMake(PPMagnifierViewDefaultRadius * 2, PPMagnifierViewDefaultRadius * 2);
    }
    
    //////////////////////////////////////////////////
    
    if((self = [super initWithFrame:frame]))
    {
        self.layer.masksToBounds = YES;
#if TARGET_OS_IPHONE
#elif TARGET_OS_MAC
        self.wantsLayer = YES;
#endif

        //////////////////////////////////////////////////
        
        magnifyRatio_   = PPMagnifierViewDefaultMagnifyRation;
        offset_         = PPMagnifierViewDefaultOffset;
        
        //////////////////////////////////////////////////
        
#if TARGET_OS_IPHONE
        [self setBorderColor:[CPColor lightGrayColor]];
#elif TARGET_OS_MAC
#endif
        
        //////////////////////////////////////////////////
        
        _magnifierViewBackgroundColor = [[CPColor blackColor] retain];
    }
    
    return self;
}


//================================================================================
// release retain, copy objects
//================================================================================
- (void)dealloc
{
    [targetView_ release];
    
    [_magnifierViewBackgroundColor release];
    _magnifierViewBackgroundColor = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Configuring the Bounds and Frame Rectangles

//================================================================================
// 設定放大鏡的大小及外觀
//================================================================================
- (void)setFrame:(CGRect)frame
{
    if(self.style == PPMagnifierViewStyle_Default && frame.size.width != frame.size.height)
    {
        frame.size.width    = MIN(frame.size.width, frame.size.height);
        frame.size.height   = MIN(frame.size.width, frame.size.height);
    }
    
    //////////////////////////////////////////////////
    
    [super setFrame:frame];
    
    //////////////////////////////////////////////////
    
    if(self.style == PPMagnifierViewStyle_Default)
    {
        self.layer.cornerRadius = self.bounds.size.width/2;
    }
    else
    {
        self.layer.cornerRadius = 5;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Drawing and Updating the View

//================================================================================
// 更新放大鏡呈現的畫面
//================================================================================
- (void)drawRect:(CGRect)rect 
{
    if(self.targetView != nil && self.superview != nil)
    {
#if TARGET_OS_IPHONE
        //////////////////////////////////////////////////
        //設為透明，這樣在放大鏡中就不會看到自己
        self.alpha = 0;
        
        CGContextRef context = UIGraphicsGetCurrentContext();
        
        if(context != NULL)
        {
            //先填滿色(預設黑色),避免邊界沒有畫面
            if(self.magnifierViewBackgroundColor!=nil)
            {
                CGContextSetFillColorWithColor(context, self.magnifierViewBackgroundColor.CGColor);
            }
            
            //////////////////////////////////////////////////
            
            CGContextFillRect(context, rect);
            CGContextTranslateCTM(context, self.frame.size.width/2, self.frame.size.height/2);
            CGContextScaleCTM(context, self.magnifyRatio, self.magnifyRatio);
            CGContextTranslateCTM(context, -self.targetPoint.x, -self.targetPoint.y);
            
            [self.targetView.layer renderInContext:context];
        }
        
        UIGraphicsEndImageContext();
        
        //////////////////////////////////////////////////
        //再改回去非透明
        self.alpha = 1;
#elif TARGET_OS_MAC
        
        CGContextRef context = [NSGraphicsContext currentContext].CGContext;
        
        if(self.screenMagnifier==YES)
        {
            if(context != NULL)
            {
                self.alphaValue = 0;
                //先填滿色(預設黑色),避免邊界沒有畫面
                if(self.magnifierViewBackgroundColor!=nil)
                {
                    CGContextSetFillColorWithColor(context, self.magnifierViewBackgroundColor.CGColor);
                }
                CGContextFillRect(context, rect);

                //////////////////////////////////////////////////
                NSImage *image = [self getScreenShotImage];
                
                CGImageSourceRef source = CGImageSourceCreateWithData((CFDataRef)[image TIFFRepresentation], NULL);
                CGImageRef maskRef =  CGImageSourceCreateImageAtIndex(source, 0, NULL);
                
                CGFloat scaledWidth = ceilf(image.size.width*self.magnifyRatio);
                CGFloat scaledHeight = ceilf(image.size.height*self.magnifyRatio);
                CGFloat scaledOffsetX = 0-ABS(scaledWidth-image.size.width)/2;
                CGFloat scaledOffsetY = 0-ABS(scaledHeight-image.size.height)/2;
                CGContextDrawImage(context, NSMakeRect(scaledOffsetX, scaledOffsetY, scaledWidth, scaledHeight), maskRef);
                
                CGImageRelease(maskRef);
                CFRelease(source);
                
                //////////////////////////////////////////////////
                
                [self.targetView.layer renderInContext:context];
                
                //////////////////////////////////////////////////
                //再改回去非透明
                self.alphaValue = 1;
            }
        }
        else
        {
            if(context != NULL)
            {
                self.alphaValue = 0;
                
                //先填滿色(預設黑色),避免邊界沒有畫面
                if(self.magnifierViewBackgroundColor!=nil)
                {
                    CGContextSetFillColorWithColor(context, self.magnifierViewBackgroundColor.CGColor);
                }
                
                //////////////////////////////////////////////////
                
                CGContextFillRect(context, rect);
                CGContextTranslateCTM(context, self.frame.size.width/2, self.frame.size.height/2);
                CGContextScaleCTM(context, self.magnifyRatio, self.magnifyRatio);
                CGContextTranslateCTM(context, -self.targetPoint.x, -self.targetPoint.y);
                
                //////////////////////////////////////////////////
                
                [self.targetView.layer renderInContext:context];
                
                //////////////////////////////////////////////////
                //再改回去非透明
                self.alphaValue = 1;
            }
        }
        
#endif
        
        
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property Method

#if TARGET_OS_IPHONE
#elif TARGET_OS_MAC
//================================================================================
//
//================================================================================
- (NSImage *)getScreenShotImage
{
    CFArrayRef windowsRef = CGWindowListCreate(kCGWindowListOptionOnScreenOnly, kCGNullWindowID);
    CFIndex windowsCount = CFArrayGetCount( windowsRef );
    CFMutableArrayRef adjustWindowsRef = CFArrayCreateMutableCopy(NULL, windowsCount, windowsRef);
    
    //////////////////////////////////////////////////
    // 移除snapshot window
    
    int snapshotWindowIndex = -1;
    for( int i = 0; i < windowsCount ; ++i  )
    {
        CGWindowID windowID = (CGWindowID) (uintptr_t) CFArrayGetValueAtIndex(windowsRef, i);
        
        if(windowID==[self.window windowNumber])
        {
            snapshotWindowIndex = i;
        }
    }
    
    if(snapshotWindowIndex!=-1)
    {
        CFArrayRemoveValueAtIndex(adjustWindowsRef, snapshotWindowIndex);
    }
    
    //////////////////////////////////////////////////
    // 裁圖是以滑鼠為中心(target point)
    NSRect trasferRect = self.frame;
    trasferRect.origin.x = self.targetPoint.x - (self.bounds.size.width/2);
    trasferRect.origin.y = self.targetPoint.y + self.offset.y - (self.bounds.size.height/2);
//    NSLog(@"trasferRect: %@", NSStringFromRect(trasferRect));
    
    NSRect rect = [self.window convertRectToScreen:trasferRect];
    rect.origin.y = CGRectGetMaxY(self.window.frame) - CGRectGetMaxY(rect);
    rect = NSInsetRect(rect, 1, 1);
    
    CGImageRef imgRef = CGWindowListCreateImageFromArray(NSRectToCGRect(rect), adjustWindowsRef, kCGWindowImageDefault);
    CFRelease(windowsRef);
    
    NSImage *img = nil;
    
    if (CGRectIsEmpty(NSRectToCGRect(rect))==NO &&
        CGRectIsNull(NSRectToCGRect(rect))==NO)
    {
        img = [[[NSImage alloc] initWithCGImage:imgRef size:rect.size] autorelease];
    }
    
    CGImageRelease(imgRef);
    
    return img;
}
#endif



//================================================================================
//
//================================================================================
- (void)setMagnifierViewBackgroundColor:(CPColor *)magnifierViewBackgroundColor
{
    [magnifierViewBackgroundColor retain];
    [_magnifierViewBackgroundColor release];
    _magnifierViewBackgroundColor = magnifierViewBackgroundColor;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
// 設定放大鏡的位置
//================================================================================
- (void)setTargetPoint:(CGPoint)targetPoint
{
    targetPoint_ = targetPoint;
    
    //////////////////////////////////////////////////
    CGPoint convertTargetPoint = [self.targetView convertPoint:self.targetPoint toView:self.superview];
//    NSLog(@"convertTargetPoint: %@", NSStringFromPoint(convertTargetPoint));

    //////////////////////////////////////////////////
    
#if TARGET_OS_IPHONE
    self.center = CGPointMake(convertTargetPoint.x+self.offset.x,
                              convertTargetPoint.y+self.offset.y-(self.bounds.size.height/2));
#elif TARGET_OS_MAC
    // !! 這邊主要先算出放大鏡的位置 (原點在左下)
    NSPoint point = NSMakePoint(convertTargetPoint.x+self.offset.x - (self.bounds.size.width/2),
                                convertTargetPoint.y-self.offset.y );

    NSRect rect = self.frame;
    rect.origin.x = point.x;
    rect.origin.y = point.y;
//    NSLog(@"self.frame: %@", NSStringFromRect(rect));

    //!!  如果在靠近上方時，要調整放大鏡位置，不然會被擋住
    if(CGRectGetMaxY(rect)>self.window.frame.size.height)
    {
        rect.origin.y = convertTargetPoint.y + self.offset.y - CGRectGetHeight(rect);
    }
    self.frame = rect;

    
#endif
    
    
}


//================================================================================
// 設定放大鏡要呈現的view
//================================================================================
- (void)setTargetView:(CPView *)targetView
{
    [targetView retain];
    [targetView_ release];
    targetView_ = targetView;
    
    //////////////////////////////////////////////////
    //!! 2015/10/19-發佈iOS9後，設定windowLevel會導致statusBar消失
    
    //    static UIWindowLevel keepWindowLevel;
    
    if(self.targetView == nil && self.superview != nil)
    {
        [self removeFromSuperview];
        
        //        [UIApplication sharedApplication].keyWindow.windowLevel = keepWindowLevel;
    }
    else if(self.targetView != nil)
    {
        //        keepWindowLevel = [UIApplication sharedApplication].keyWindow.windowLevel;
        //        [UIApplication sharedApplication].keyWindow.windowLevel = UIWindowLevelStatusBar;
        
        //////////////////////////////////////////////////
#if TARGET_OS_IPHONE
        [[UIViewController topMostViewController].view addSubview:self];
#elif TARGET_OS_MAC
        [self.targetView addSubview:self];
#endif
        
    }
}

@end
