//
//  PPLogController.m
//  
//
//  Created by Mike on 13/5/7.
//  Modified by Egg on 14/6/27
//  Copyright (c) 2013年 Penpower. All rights reserved.
//
// Model
#import "NSDate+Format.h"

// View

// Controller
#import "PPLogController.h"




static NSMutableDictionary *staticCostTimeKey = nil;
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPLogController()

@interface PPLogController()






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic,retain           ) NSFileHandle *fileHandle;
@property (nonatomic,retain           ) NSString     *atPath;
@property (nonatomic,retain, readwrite) NSString     *fileName;
@property (nonatomic,retain, readwrite) NSString     *filePath;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

- (void)closeFile;
- (BOOL)createFile;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation BaseShape

@implementation PPLogController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
atPath      = atPath_,
fileHandle  = fileHandle_,
fileName    = fileName_,
filePath    = filePath_,
mask        = mask_,
toConsole   = toConsole_,
toFile      = toFile_;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Init & dealloc

//================================================================================
//
//================================================================================
- (id)init
{
	if((self = [super init]))
	{
		atPath_     = nil;
		fileName_   = nil;
        toConsole_  = YES;
        toFile_     = YES;
        mask_       = PPLogControllerMask_Normal;
	}
	
	return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
	[self closeFile];
	
    //////////////////////////////////////////////////

    self.atPath = nil;
    self.fileName = nil;
    self.filePath = nil;
    
    [staticCostTimeKey release];
    staticCostTimeKey = nil;
    //////////////////////////////////////////////////
    
	[super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods


//==============================================================================
//
//==============================================================================
- (BOOL)isFileExist
{
    @synchronized(self)
    {
        BOOL isDir = NO;
        BOOL isFileExist = [[NSFileManager defaultManager] fileExistsAtPath:self.filePath isDirectory:&isDir];
        return isFileExist;
    }
}


//================================================================================
//
//================================================================================
- (void)closeFile
{
    @synchronized(self)
    {
        if(self.fileHandle!=nil)
        {
            [self.fileHandle synchronizeFile];
            [self.fileHandle closeFile];
            self.fileHandle = nil;
        }
    }
}


//================================================================================
//
//================================================================================
- (BOOL)createFile
{
    @synchronized(self)
    {
        BOOL result = NO;
        
        do
        {
            if(self.atPath==nil || [self.atPath length]==0 || self.fileName==nil || [self.fileName length]==0)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSFileManager *fileManager = [NSFileManager defaultManager];
            if(fileManager==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //檢查並建立指定目錄
            
            if([fileManager fileExistsAtPath:self.atPath]==NO && [fileManager createDirectoryAtPath:self.atPath withIntermediateDirectories:YES attributes:nil error:nil]==NO)
            {
                break;
            }
			
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            self.filePath = [self.atPath stringByAppendingPathComponent:[NSString stringWithFormat:@"%@-%@.txt",
                                                                         [[NSDate date] stringWithFormat:@"yyMMdd_HHmmss"],
                                                                         self.fileName]];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            if([fileManager createFileAtPath:self.filePath contents:nil attributes:nil]==NO)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [self closeFile];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            self.fileHandle = [NSFileHandle fileHandleForWritingAtPath:self.filePath];
            if(self.fileHandle==nil)
            {
                break;
            }
            
			////////////////////////////////////////////////////////////////////////////////////////////////////
			// 有限定數量時，超過限制要一時間順序刪除舊檔案。
			
			if(self.remainCount > 0)
			{
				NSArray *filelist = [fileManager contentsOfDirectoryAtPath:self.atPath error:nil];
				NSInteger removeCount = ([filelist count] - self.remainCount);
				
				if(removeCount > 0)
				{
					// 依時間排序（直接用檔名）
					NSArray* sortedFiles = [filelist sortedArrayUsingComparator:
					^(id path1, id path2)
					{
						// compare
						NSComparisonResult comp = [[path1 lastPathComponent] compare:
												   [path2 lastPathComponent]];
						return comp;
					}];
					
					for(int i=0; i<removeCount; i++)
					{
						NSString *file = [self.atPath stringByAppendingPathComponent:sortedFiles[i]];
						[fileManager removeItemAtPath:file error:nil];
					}
				}
			}
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            result = YES;
            
        }while(0);
        
        return result;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)logWithMask:(PPLogControllerMask)mask dataName:(NSString *)dataName data:(NSData *)data
{
    @synchronized(self)
    {
        BOOL result = NO;
        
        do
        {
            if((mask&self.mask)==PPLogControllerMask_None || dataName==nil || data==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSString *dataPath = [self.atPath stringByAppendingPathComponent:[NSString stringWithFormat:@"%@-%@",
                                                                              [[NSDate date] stringWithFormat:@"HHmmss"],
                                                                              dataName]];
            if(dataPath==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            result = [data writeToFile:dataPath atomically:YES];
            
        }while(0);
        
        return result;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format, ...
{
    BOOL result = NO;
    
    va_list arguments;
    va_start(arguments, format);
    result = [self logWithMask:mask format:format arguments:arguments];
    va_end(arguments);
    
	return result;
}


//================================================================================
//
//================================================================================
- (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format arguments:(va_list)arguments
{
    @synchronized(self)
    {
        BOOL result = NO;
        
        do
        {
            if((mask&self.mask)==PPLogControllerMask_None || format==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSString *message = [[NSString alloc] initWithFormat:format arguments:arguments];
            if(message!=nil)
            {
                NSString *log = [[NSString alloc] initWithFormat:@"[%@] %@\r\n", [[NSDate date] stringWithFormat:PPLogController_DateFormat], message];
                if(log!=nil)
                {
                    if(self.toConsole==YES)
                    {
                        printf("%s", [log UTF8String]);
                    }
                    
                    ////////////////////////////////////////////////////////////////////////////////////////////////////
                    
                    if(self.toFile==YES)
                    {
                        // !!不存在就要建立，不然如果被刪除就寫不進去了
                        if([self isFileExist]==NO)
                        {
                            self.fileHandle = nil;
                            
                            [self createFile];
                        }
                        
                        if(self.fileHandle!=nil)
                        {
                            NSData *logData = [log dataUsingEncoding:NSUTF8StringEncoding];
                            
                            if([logData length] > 0)
                            {
                                [self.fileHandle seekToEndOfFile];
                                
                                @try
                                {
                                    [self.fileHandle writeData:logData];
                                }
                                @catch (NSException *e)
                                {
                                    // 不做處理，只是為了避免crash。
                                }
                            }
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            if([self.fileHandle offsetInFile]>PPLogController_FileMaxSize)
                            {
                                [self createFile];
                            }
                        }
                    }
                    
                    ////////////////////////////////////////////////////////////////////////////////////////////////////
                    
                    result = YES;
                    
                    ////////////////////////////////////////////////////////////////////////////////////////////////////
                    
                    [log release];
                }
                
                [message release];
            }
            
        }while(0);
        
        return result;
    }
}


//================================================================================
//
//================================================================================
- (void)setFileName:(NSString *)fileName atPath:(NSString *)atPath
{
    @synchronized(self)
    {
        if (fileName_   == nil &&
            atPath_     == nil &&
            fileName    != nil &&
            atPath      != nil)
        {
            self.fileName = fileName;
            self.atPath   = atPath;
        }
        else
        {
#ifdef DEBUG
            NSLog(@"<%s> cannot set filename & path:\n\
                  self.fileName:%@\n\
                  self.atPath:%@\n\
                  fileName:%@\n\
                  atPath:%@",
                  __PRETTY_FUNCTION__,
                  fileName_,
                  atPath_,
                  fileName,
                  atPath);
#endif
        }
    }
}


//================================================================================
//
//================================================================================
- (void)renewLogFile
{
    [self closeFile];
    [self createFile];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - log cost time


//==============================================================================
//
//==============================================================================
- (void)startLogCostTime:(NSString *)logKey
{
    @synchronized (self)
    {
        if([logKey length]==0)
        {
            NSAssert([logKey length]==0, @"logKey 應該要有值");
            return ;
        }
        
        if (staticCostTimeKey==nil)
        {
            staticCostTimeKey = [[NSMutableDictionary alloc] init];
        }
        
        [staticCostTimeKey setObject:@([NSDate timeIntervalSinceReferenceDate]) forKey:logKey];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)logWithKey:(NSString *)logKey mask:(PPLogControllerMask)mask format:(NSString *)format, ...
{
    @synchronized (self)
    {
        BOOL result = NO;
        if([logKey length]==0)
        {
            NSAssert([logKey length]==0, @"logKey 應該要有值");
            return result;
        }
        
        NSTimeInterval curTimeInterval = [NSDate timeIntervalSinceReferenceDate];
        NSNumber *prevTimeIntervalNumber = [staticCostTimeKey objectForKey:logKey];
        
        // 沒有值表示沒有start
        if(prevTimeIntervalNumber==nil)
        {
            return result;
        }
        
        NSTimeInterval prevTimeInterval = [prevTimeIntervalNumber doubleValue];
        format = [format stringByAppendingFormat:@" - (cost: %f s)", curTimeInterval - prevTimeInterval];
        [staticCostTimeKey setObject:@(curTimeInterval) forKey:logKey];
        
        //////////////////////////////////////////////////
        va_list arguments;
        va_start(arguments, format);
        result = [self logWithMask:mask format:format arguments:arguments];
        va_end(arguments);
        
        return result;
    }
}


//==============================================================================
//
//==============================================================================
- (void)stopLogCostTime:(NSString *)logKey
{
    @synchronized (self)
    {
        [staticCostTimeKey removeObjectForKey:logKey];
        
        if ([staticCostTimeKey count]==0)
        {
            [staticCostTimeKey release];
            staticCostTimeKey = nil;
        }
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//================================================================================
// Singleton
//================================================================================
+ (instancetype)shareLogController
{
    static PPLogController *shareLogController = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^ {
        shareLogController = [[self alloc] init];
    });
    return shareLogController;
}


//================================================================================
//
//================================================================================
+ (BOOL)logWithMask:(PPLogControllerMask)mask dataName:(NSString *)dataName data:(NSData *)data
{
    BOOL result = NO;

    PPLogController *shareLogController = [self shareLogController];
    
	if(shareLogController!=nil)
	{
        result = [shareLogController logWithMask:mask dataName:dataName data:data];
	}
	
	return result;
}


//================================================================================
//
//================================================================================
+ (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format, ...
{
    BOOL result = NO;
    
    PPLogController *shareLogController = [self shareLogController];
    
	if(shareLogController!=nil)
	{
		va_list arguments;
		va_start(arguments, format);
        result = [shareLogController logWithMask:mask format:format arguments:arguments];
        va_end(arguments);
	}
	
	return result;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - (Class) Log cost time


//==============================================================================
//
//==============================================================================
+ (void)startLogCostTime:(NSString *)logKey
{
    PPLogController *shareLogController = [self shareLogController];
    
    if(shareLogController!=nil)
    {
        [shareLogController startLogCostTime:logKey];
    }
}


//================================================================================
//
//================================================================================
+ (BOOL)logWithKey:(NSString *)logKey mask:(PPLogControllerMask)mask format:(NSString *)format, ...
{
    BOOL result = NO;
    
    PPLogController *shareLogController = [self shareLogController];
    
    if(shareLogController!=nil)
    {
        if([logKey length]==0)
        {
            NSAssert([logKey length]==0, @"logKey 應該要有值");
            return result;
        }
        
        NSTimeInterval curTimeInterval = [NSDate timeIntervalSinceReferenceDate];
        
        NSNumber *prevTimeIntervalNumber = [staticCostTimeKey objectForKey:logKey];
        if (prevTimeIntervalNumber==nil)
        {
            return result;
        }
        NSTimeInterval prevTimeInterval = [prevTimeIntervalNumber doubleValue];
        
        format = [format stringByAppendingFormat:@" - (cost: %f s)", curTimeInterval - prevTimeInterval];
        [staticCostTimeKey setObject:@(curTimeInterval) forKey:logKey];
        
        //////////////////////////////////////////////////
        
        va_list arguments;
        va_start(arguments, format);
        result = [shareLogController logWithMask:mask format:format arguments:arguments];
        va_end(arguments);
    }
    
    return result;
}


//==============================================================================
//
//==============================================================================
+ (void)stopLogCostTime:(NSString *)logKey
{
    PPLogController *shareLogController = [self shareLogController];
    
    if(shareLogController!=nil)
    {
        [shareLogController stopLogCostTime:logKey];
    }
}
@end
