//
//  BKPasscodeInputView.m
//  BKPasscodeViewDemo
//
//  Created by Byungkook Jang on 2014. 4. 20..
//  Copyright (c) 2014년 Byungkook Jang. All rights reserved.
//

#import "PPPincodeView.h"

#define kLabelPasscodeSpacePortrait         (20.0f)
#define kLabelPasscodeSpaceLandscape        (10.0f)

#define kTextLeftRightSpace                 (20.0f)

#define kErrorMessageLeftRightPadding       (10.0f)
#define kErrorMessageTopBottomPadding       (5.0f)

#define kDefaultNumericPasscodeMaximumLength        (4)
#define kDefaultNormalPasscodeMaximumLength         (20)

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPPincodeView ()

@property (nonatomic, retain) UILabel   *titleLabel;
@property (nonatomic, retain) UILabel   *messageLabel;
@property (nonatomic, retain) UILabel   *errorMessageLabel;
@property (nonatomic, retain) UIControl *pincodeField;

@property (nonatomic, assign) BOOL      isKeyboardTypeSet;
@property (nonatomic, retain) NSArray   *layoutConstraints;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPPincodeView
@synthesize pincodeField = _pincodeField;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - init/dealloc


//==============================================================================
//
//==============================================================================
- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self)
    {
        [self initialize];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (instancetype)initWithCoder:(NSCoder *)coder
{
    self = [super initWithCoder:coder];
    if (self) {
        [self initialize];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)initialize
{
    self.backgroundColor = [UIColor lightTextColor];
    
    _enabled = YES;
    _viewStyle = PPPincodeViewStyle_Simple;
    _keyboardType = UIKeyboardTypeNumberPad;
    _maximumLength = 0;
    
    _titleLabel = [[UILabel alloc] init];
    if (self.titleLabel)
    {
        [self.titleLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [[self class] configureTitleLabel:self.titleLabel];
        [self addSubview:self.titleLabel];
    }
    
    _messageLabel = [[UILabel alloc] init];
    if (self.messageLabel)
    {
        [self.messageLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [[self class] configureMessageLabel:self.messageLabel];
        [self addSubview:self.messageLabel];
    }
    
    _errorMessageLabel = [[UILabel alloc] init];
    if (self.errorMessageLabel)
    {
        [self.errorMessageLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [[self class] configureErrorMessageLabel:self.errorMessageLabel];
        self.errorMessageLabel.hidden = YES;
        [self addSubview:self.errorMessageLabel];
    }
}



//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.delegate = nil;
    
    self.layoutConstraints = nil;
    
    self.title = nil;
    self.message = nil;
    self.errorMessage = nil;
    self.pincode = nil;
    
    [_pincodeField removeFromSuperview];
    [_pincodeField release];
    _pincodeField = nil;
    
    [self.titleLabel removeFromSuperview];
    self.titleLabel = nil;
    
    [self.messageLabel removeFromSuperview];
    self.messageLabel = nil;
    
    [self.errorMessageLabel removeFromSuperview];
    self.errorMessageLabel = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - NSCopying


//==============================================================================
//
//==============================================================================
- (id)copyWithZone:(NSZone *)zone
{
    PPPincodeView *view = [[[self class] alloc] initWithFrame:self.frame];
    view.delegate = self.delegate;
    view.autoresizingMask = self.autoresizingMask;
    view.viewStyle = self.viewStyle;
    view.keyboardType = self.keyboardType;
    view.maximumLength = self.maximumLength;
    
    return view;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - class methods


//==============================================================================
//
//==============================================================================
+ (void)configureTitleLabel:(UILabel *)aLabel
{
    aLabel.backgroundColor = [UIColor clearColor];
    aLabel.numberOfLines = 1;
    aLabel.textAlignment = NSTextAlignmentCenter;
    aLabel.lineBreakMode = NSLineBreakByTruncatingTail;
    aLabel.font = [UIFont boldSystemFontOfSize:15.0f];
}


//==============================================================================
//
//==============================================================================
+ (void)configureMessageLabel:(UILabel *)aLabel
{
    aLabel.backgroundColor = [UIColor clearColor];
    aLabel.numberOfLines = 0;
    aLabel.textAlignment = NSTextAlignmentCenter;
    aLabel.lineBreakMode = NSLineBreakByWordWrapping;
    aLabel.font = [UIFont systemFontOfSize:15.0f];
}


//==============================================================================
//
//==============================================================================
+ (void)configureErrorMessageLabel:(UILabel *)aLabel
{
    aLabel.backgroundColor = [UIColor clearColor];
    aLabel.numberOfLines = 0;
    aLabel.textAlignment = NSTextAlignmentCenter;
    aLabel.lineBreakMode = NSLineBreakByWordWrapping;
    aLabel.backgroundColor = [UIColor colorWithRed:0.63 green:0.2 blue:0.13 alpha:1];
    aLabel.textColor = [UIColor whiteColor];
    aLabel.font = [UIFont systemFontOfSize:15.0f];
    
    aLabel.layer.cornerRadius = 10.0f;
    aLabel.layer.masksToBounds = YES;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - property override


//==============================================================================
//
//==============================================================================
- (void)setViewStyle:(PPPincodeViewStyle)viewStyle
{
    if (_viewStyle != viewStyle) {
        _viewStyle = viewStyle;

        if (_pincodeField)
        {
            _pincodeField = nil;
            // 重建pincodeField
            [self pincodeField];
        }
    }
}


//==============================================================================
//
//==============================================================================
- (UIControl *)pincodeField
{
    if (_pincodeField==nil)
    {
        switch (_viewStyle)
        {
            case PPPincodeViewStyle_Simple:
            {
                if (_maximumLength == 0)
                {
                    _maximumLength = kDefaultNumericPasscodeMaximumLength;
                }
                
                if (_isKeyboardTypeSet==NO)
                {
                    _keyboardType = UIKeyboardTypeNumberPad;
                }
                
                PPDigitalPinCodeField *pincodeField = [[PPDigitalPinCodeField alloc] init];
                pincodeField.delegate = self;
                pincodeField.keyboardType = _keyboardType;
                pincodeField.maximumLength = _maximumLength;
                [pincodeField addTarget:self action:@selector(pincodeControlEditingChanged:) forControlEvents:UIControlEventEditingChanged];
                
                [self setPincodeField:pincodeField];
                [pincodeField release];
                break;
            }
                
            case PPPincodeViewStyle_Normal:
            {
                if (_maximumLength == 0)
                {
                    _maximumLength = kDefaultNormalPasscodeMaximumLength;
                }

                if (_isKeyboardTypeSet==NO)
                {
                    _keyboardType = UIKeyboardTypeASCIICapable;
                }
                
                UITextField *textField = [[UITextField alloc] init];
                textField.delegate = self;
                textField.borderStyle = UITextBorderStyleRoundedRect;
                textField.autocapitalizationType = UITextAutocapitalizationTypeNone;
                textField.autocorrectionType = UITextAutocorrectionTypeNo;
                textField.spellCheckingType = UITextSpellCheckingTypeNo;
                textField.enablesReturnKeyAutomatically = YES;
                textField.keyboardType = _keyboardType;
                textField.secureTextEntry = YES;
                textField.font = [UIFont systemFontOfSize:25.0f];
                textField.clearButtonMode = UITextFieldViewModeWhileEditing;
                textField.returnKeyType = UIReturnKeyDone;
                
                [self setPincodeField:textField];
                [textField release];
                break;
            }
        }
    }
    
    return _pincodeField;
}


//==============================================================================
//
//==============================================================================
- (void)setPincodeField:(UIControl *)pincodeField
{
    if (_pincodeField != pincodeField)
    {
        [pincodeField retain];
        [_pincodeField removeFromSuperview];
        [_pincodeField release];
        _pincodeField = pincodeField;
        
        if (_pincodeField)
        {
            [self addSubview:_pincodeField];
        }
        [self setNeedsLayout];
    }
}


//==============================================================================
//
//==============================================================================
- (void)setMaximumLength:(NSUInteger)maximumLength
{
    _maximumLength = maximumLength;
    
    if ([self.pincodeField isKindOfClass:[PPDigitalPinCodeField class]]) {
        [(PPDigitalPinCodeField *)self.pincodeField setMaximumLength:maximumLength];
    }
}


//==============================================================================
//
//==============================================================================
- (void)setKeyboardType:(UIKeyboardType)keyboardType
{
    _isKeyboardTypeSet = YES;
    _keyboardType = keyboardType;
    [(id<UITextInputTraits>)self.pincodeField setKeyboardType:keyboardType];
}


//==============================================================================
//
//==============================================================================
- (void)setTitle:(NSString *)title
{
    self.titleLabel.text = title;
    [self setNeedsLayout];
}


//==============================================================================
//
//==============================================================================
- (NSString *)title
{
    return self.titleLabel.text;
}


//==============================================================================
//
//==============================================================================
- (void)setMessage:(NSString *)message
{
    self.messageLabel.text = message;
    self.messageLabel.hidden = NO;
    
    self.errorMessageLabel.text = nil;
    self.errorMessageLabel.hidden = YES;
    
    [self setNeedsLayout];
}


//==============================================================================
//
//==============================================================================
- (NSString *)message
{
    return self.messageLabel.text;
}


//==============================================================================
//
//==============================================================================
- (void)setErrorMessage:(NSString *)errorMessage
{
    self.errorMessageLabel.text = errorMessage;
    self.errorMessageLabel.hidden = NO;
    
    self.messageLabel.text = nil;
    self.messageLabel.hidden = YES;
    
    [self setNeedsLayout];
}


//==============================================================================
//
//==============================================================================
- (NSString *)errorMessage
{
    return self.errorMessageLabel.text;
}


//==============================================================================
//
//==============================================================================
- (NSString *)pincode
{
    switch (self.viewStyle)
    {
        case PPPincodeViewStyle_Simple:
        {
            return [(PPDigitalPinCodeField *)self.pincodeField pincode];
        }
        case PPPincodeViewStyle_Normal:
        {
            return [(UITextField *)self.pincodeField text];
        }
        default:
            return nil;
    }
}


//==============================================================================
//
//==============================================================================
- (void)setPincode:(NSString *)pincode
{
    switch (self.viewStyle)
    {
        case PPPincodeViewStyle_Simple:
            [(PPDigitalPinCodeField *)self.pincodeField setPincode:pincode];
            break;
        case PPPincodeViewStyle_Normal:
             [(UITextField *)self.pincodeField setText:pincode];
             break;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - layout constraints


//==============================================================================
//
//==============================================================================
- (CGFloat)labelPasscodeSpace
{
    return UIInterfaceOrientationIsPortrait([[UIApplication sharedApplication] statusBarOrientation]) ? kLabelPasscodeSpacePortrait : kLabelPasscodeSpaceLandscape;
}


//==============================================================================
//
//==============================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    // layout passcode control to center
    [self.pincodeField sizeToFit];
    
    if ([self.pincodeField isKindOfClass:[UITextField class]]) {
        self.pincodeField.frame = CGRectMake(0, 0, self.frame.size.width - kTextLeftRightSpace * 2.0f, CGRectGetHeight(self.pincodeField.frame) + 10.0f);
    }

    self.pincodeField.center = CGPointMake(CGRectGetWidth(self.frame) * 0.5f, CGRectGetHeight(self.frame) * 0.5f);
    
    CGFloat maxTextWidth = self.frame.size.width - (kTextLeftRightSpace * 2.0f);
    CGFloat labelPasscodeSpace = [self labelPasscodeSpace];
    
    // layout title label
    _titleLabel.frame = CGRectMake(kTextLeftRightSpace, 0, maxTextWidth, self.frame.size.height);
    [_titleLabel sizeToFit];
    
    CGRect rect = _titleLabel.frame;
    rect.origin.x = floorf((self.frame.size.width - CGRectGetWidth(rect)) * 0.5f);
    rect.origin.y = CGRectGetMinY(self.pincodeField.frame) - labelPasscodeSpace - CGRectGetHeight(_titleLabel.frame);

    _titleLabel.frame = rect;
    
    // layout message label
    if (!_messageLabel.hidden) {
        _messageLabel.frame = CGRectMake(kTextLeftRightSpace, CGRectGetMaxY(self.pincodeField.frame) + labelPasscodeSpace, maxTextWidth, self.frame.size.height);
        [_messageLabel sizeToFit];
        
        rect = _messageLabel.frame;
        rect.origin.x = floorf((self.frame.size.width - CGRectGetWidth(rect)) * 0.5f);
        _messageLabel.frame = rect;
    }
    
    // layout error message label
    if (!_errorMessageLabel.hidden) {
        _errorMessageLabel.frame = CGRectMake(0, CGRectGetMaxY(self.pincodeField.frame) + labelPasscodeSpace,
                                              maxTextWidth - kErrorMessageLeftRightPadding * 2.0f,
                                              self.frame.size.height);
        [_errorMessageLabel sizeToFit];
        
        rect = _errorMessageLabel.frame;
        rect.size.width += (kErrorMessageLeftRightPadding * 2.0f);
        rect.size.height += (kErrorMessageTopBottomPadding * 2.0f);
        rect.origin.x = floorf((self.frame.size.width - rect.size.width) * 0.5f);
        
        _errorMessageLabel.frame = rect;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UIResponder


//==============================================================================
//
//==============================================================================
- (BOOL)canBecomeFirstResponder
{
    [super canBecomeFirstResponder];
    return [self.pincodeField canBecomeFirstResponder];
}


//==============================================================================
//
//==============================================================================
- (BOOL)becomeFirstResponder
{
    [super becomeFirstResponder];
    return [self.pincodeField becomeFirstResponder];
}


//==============================================================================
//
//==============================================================================
- (BOOL)canResignFirstResponder
{
    [super canResignFirstResponder];
    return [self.pincodeField canResignFirstResponder];
}

//==============================================================================
//
//==============================================================================
- (BOOL)resignFirstResponder
{
    [super resignFirstResponder];
    return [self.pincodeField resignFirstResponder];
}


//==============================================================================
// touch to focus
//==============================================================================
- (void)touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
    [super touchesBegan:touches withEvent:event];
    [self.pincodeField becomeFirstResponder];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - Actions


//==============================================================================
//
//==============================================================================
- (void)pincodeControlEditingChanged:(id)sender
{
    if (![self.pincodeField isKindOfClass:[PPDigitalPinCodeField class]])
    {
        return;
    }
    
    PPDigitalPinCodeField *pincodeField = (PPDigitalPinCodeField *)self.pincodeField;
    
    if (pincodeField.pincode.length == pincodeField.maximumLength)
    {
        if ([self.delegate respondsToSelector:@selector(inputViewDidFinish:)])
        {
            [self.delegate inputViewDidFinish:self];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - PPDigitalPinCodeFieldDelegate


//==============================================================================
//
//==============================================================================
- (BOOL)pincodeField:(PPDigitalPinCodeField *)pincodeField shouldInsertText:(NSString *)text
{
    return self.isEnabled;
}


//==============================================================================
//
//==============================================================================
- (BOOL)pincodeFieldShouldDeleteBackward:(PPDigitalPinCodeField *)pincodeField;
{
    return self.isEnabled;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UITextFieldDelegate


//==============================================================================
//
//==============================================================================
- (BOOL)textField:(UITextField *)textField shouldChangeCharactersInRange:(NSRange)range replacementString:(NSString *)string
{
    if (self.isEnabled == NO)
    {
        return NO;
    }
    
    NSUInteger length = textField.text.length - range.length + string.length;
    if (length > self.maximumLength)
    {
        return NO;
    }
    
    return YES;
}


//==============================================================================
//
//==============================================================================
- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
    if (self.isEnabled == NO)
    {
        return NO;
    }
    
    if ([self.delegate respondsToSelector:@selector(inputViewDidFinish:)])
    {
        [self.delegate inputViewDidFinish:self];
        return NO;
    } else {
        return YES; // default behavior
    }
}


@end
