//
//  PPLocationController.m
//
//
//  Created by Howard on 2015/12/29.
//
//

#import "PPLocationController.h"

// Controller
#import "PPLocationManager.h"
#import "AppleLocationManager.h"
#import "BaiduLocationManager.h"
#import "GoogleLocationManager.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface PPLocationController

@interface PPLocationController()

@property (nonatomic,readonly) NSOperationQueue *operationQueue;
@property (nonatomic,retain) NSMutableDictionary *managerDictionarys;
@property (atomic,retain) NSMutableArray *readyLocationOperations;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation  PPLocationController

@implementation PPLocationController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        _operationQueue = [[NSOperationQueue alloc] init];
        
        if(_operationQueue!=nil)
        {
            [_operationQueue setMaxConcurrentOperationCount:PPLocationControllerDefaultConcurrencyOperationCount];
        }
        
        //////////////////////////////////////////////////
        
        _managerDictionarys = [[NSMutableDictionary alloc] init];
        
        //////////////////////////////////////////////////
        
        _readyLocationOperations = [[NSMutableArray alloc] init];
        
        //////////////////////////////////////////////////
        
        _locationManagerOrders = [[NSMutableArray alloc] initWithObjects:[AppleLocationManager class],[BaiduLocationManager class],[GoogleLocationManager class], nil];
        
        //////////////////////////////////////////////////
        
        _operationTimeInterval = PPLocationController_OperationTimeInterval;
        
        _operationTimeOutInterval = PPLocationController_OperationTimeOutInterval;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_operationQueue cancelAllOperations];
    [_operationQueue release];
    _operationQueue = nil;
    
    [_managerDictionarys release];
    _managerDictionarys = nil;
    
    [_locationManagerOrders release];
    _locationManagerOrders = nil;
    
    [_readyLocationOperations release];
    _readyLocationOperations = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private LocationManager Method

//================================================================================
//
//================================================================================
- (PPLocationManager *)ppLocationManagerForLocationManager:(Class)locationManager
{
    PPLocationManager *ppLocationManager = nil;
    
    do
    {
        if(locationManager==Nil)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        ppLocationManager = [self.managerDictionarys objectForKey:NSStringFromClass(locationManager)];
        
        if(ppLocationManager==nil && [locationManager isSubclassOfClass:[PPLocationManager class]]==YES)
        {
            ppLocationManager = [[locationManager alloc] init];
            
            if(ppLocationManager!=nil)
            {
                [self.managerDictionarys setObject:ppLocationManager forKey:NSStringFromClass(locationManager)];
                
                [ppLocationManager autorelease];
            }
        }
        
    }while(0);
    
    return ppLocationManager;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Next Operation Method

//================================================================================
//
//================================================================================
- (void)removeRedundancyOperationBaseOnTargetOperation:(LocationBlockOperation *)operation
{
    for(NSUInteger operationIndex=0; operationIndex<[[self readyLocationOperations] count]; operationIndex++)
    {
        LocationBlockOperation *checkLocationBlockOperation = [[self readyLocationOperations] objectAtIndex:operationIndex];
        
        //!! 辨識成功時，要將其它同樣地址反查Location，且 identity 一樣時，不同的做法(Ex apple, google, baidu) 給移除
        if([checkLocationBlockOperation.identity isEqualToString:operation.identity] &&
           [checkLocationBlockOperation.actionName isEqualToString:operation.actionName])
        {
            [[self readyLocationOperations] removeObjectAtIndex:operationIndex];
            
            operationIndex--;
        }
        else
        {
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (BOOL)runNextOperationWithTimeInterval:(CGFloat)timeInterval
{
    BOOL result = NO;
    
    if([self readyLocationOperations].count>0)
    {
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            @synchronized([self readyLocationOperations])
            {
                [NSThread sleepForTimeInterval:timeInterval];
                
                //////////////////////////////////////////////////
                
                id operation = [[self readyLocationOperations] firstObject];
                
                [[self operationQueue] addOperation:operation];
                
                [[self readyLocationOperations] removeObject:operation];
            }

        });

        result = YES;
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (instancetype)sharedInstance
{
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
+ (BOOL)checkLocationAccess
{
    if([CLLocationManager authorizationStatus]==kCLAuthorizationStatusDenied ||
       [CLLocationManager authorizationStatus]==kCLAuthorizationStatusRestricted)
    {
        return NO;
    }
    else
    {
        return YES;
    }
}


//================================================================================
//
//================================================================================
+ (BOOL)updatingCurrentLocation
{
    BOOL result = NO;
    
    do
    {
        __block id sharedInstance = [self sharedInstance];
        
        if([sharedInstance operationQueue]==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *orders = [sharedInstance locationManagerOrders];
        
        if(orders==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(id locationManager in orders)
        {
            PPLocationManager *curLocation = [sharedInstance ppLocationManagerForLocationManager:locationManager];
            
            if(curLocation==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            LocationBlockOperation *operation = [curLocation startUpdatingCurrentLocationBlockOperationWithSuccess:^{
               [sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]];
            }
                                                                                                           failure:^(NSError *error)
                                                 {
                                                     [sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]];
                                                 }];
            
            if(operation!=nil)
            {
                @synchronized([sharedInstance operationQueue])
                {
                    if([[sharedInstance operationQueue] operationCount]<=0)
                    {
                        [[sharedInstance operationQueue] addOperation:operation];
                    }
                    else
                    {
                        [[sharedInstance readyLocationOperations] addObject:operation];
                    }
                }
                
                //////////////////////////////////////////////////
                
                result = YES;
            }
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (BOOL)fetchLocationFromAddressDictionary:(NSDictionary *)addressDictionary
                                  identity:(NSString *)identity
{
    BOOL result = NO;
    
    do
    {
        if(addressDictionary==nil ||
           addressDictionary.count<=0 ||
           identity==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block PPLocationController *sharedInstance = [self sharedInstance];
        
        //////////////////////////////////////////////////
        
        if(sharedInstance==nil ||
           [sharedInstance operationQueue]==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *orders = [sharedInstance locationManagerOrders];
        
        if(orders==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(id locationManager in orders)
        {
            PPLocationManager *curLocation = [[self sharedInstance] ppLocationManagerForLocationManager:locationManager];
            
            if(curLocation==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block LocationBlockOperation *operation = [curLocation fetchLocationFromAddressDictionary:addressDictionary
                                                                                               identity:identity
                                                                                                success:^{
                                                                                                
                                                                                                [sharedInstance removeRedundancyOperationBaseOnTargetOperation:operation];
                                                                                                
                                                                                                //////////////////////////////////////////////////
                                                                                                
                                                                                                if([sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]]==NO)
                                                                                                {
                                                                                                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                                                                                        object:self
                                                                                                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationEnd)}];
                                                                                                }
                                                                                                
                                                                                            }
                                                                                            failure:^(NSError *error){
                                                                                                
                                                                                                NSTimeInterval interval = [sharedInstance operationTimeInterval];
                                                                                                
                                                                                                if(error.code==kCLErrorNetwork)
                                                                                                {
                                                                                                    interval = [sharedInstance operationTimeOutInterval];
                                                                                                }
                                                                                                
                                                                                                //////////////////////////////////////////////////
                                                                                                
                                                                                                if([sharedInstance runNextOperationWithTimeInterval:interval]==NO)
                                                                                                {
                                                                                                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                                                                                        object:self
                                                                                                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationEnd)}];
                                                                                                }
                                                                                                
                                                                                            }];
            
            if(operation!=nil)
            {
                operation.identity = identity;
                operation.actionName = PPLocationControllerUserInfoKey_Location;
                
                //////////////////////////////////////////////////
                
                @synchronized([sharedInstance operationQueue])
                {
                    if([[sharedInstance operationQueue] operationCount]<=0)
                    {
                        [[sharedInstance operationQueue] addOperation:operation];
                    }
                    else
                    {
                        [[sharedInstance readyLocationOperations] addObject:operation];
                    }
                }
                
                result = YES;
            }
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (BOOL)fetchLocationFromAddressString:(NSString *)addressString
                                region:(CLCircularRegion *)region
                              identity:(NSString *)identity;
{
    BOOL result = NO;
    
    do
    {
        if(addressString==nil ||
           addressString.length<=0 ||
           identity==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block PPLocationController *sharedInstance = [self sharedInstance];
        
        //////////////////////////////////////////////////
        
        if(sharedInstance==nil ||
           [sharedInstance operationQueue]==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *orders = [sharedInstance locationManagerOrders];
        
        if(orders==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(id locationManager in orders)
        {
            PPLocationManager *curLocation = [[self sharedInstance] ppLocationManagerForLocationManager:locationManager];
            
            if(curLocation==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block LocationBlockOperation *operation = [curLocation fetchLocationFromAddressString:addressString
                                                                                             region:region
                                                                                           identity:identity
                                                                                            success:^{
                                                                                                
                                                                                                [sharedInstance removeRedundancyOperationBaseOnTargetOperation:operation];
                                                                                                
                                                                                                //////////////////////////////////////////////////
                                                                                                
                                                                                                if([sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]]==NO)
                                                                                                {
                                                                                                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                                                                                        object:self
                                                                                                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationEnd)}];
                                                                                                }
                                                                                     
                                                                                            }
                                                                                            failure:^(NSError *error){
                                                                                                
                                                                                                NSTimeInterval interval = [sharedInstance operationTimeInterval];
                                                                                                
                                                                                                if(error.code==kCLErrorNetwork)
                                                                                                {
                                                                                                    interval = [sharedInstance operationTimeOutInterval];
                                                                                                }
                                                                                                
                                                                                                //////////////////////////////////////////////////

                                                                                                if([sharedInstance runNextOperationWithTimeInterval:interval]==NO)
                                                                                                {
                                                                                                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                                                                                        object:self
                                                                                                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationEnd)}];
                                                                                                }

                                                                                            }];
            
            if(operation!=nil)
            {
                operation.identity = identity;
                operation.actionName = PPLocationControllerUserInfoKey_Location;
                
                //////////////////////////////////////////////////
                
                @synchronized([sharedInstance operationQueue])
                {
                    if([[sharedInstance operationQueue] operationCount]<=0)
                    {
                        [[sharedInstance operationQueue] addOperation:operation];
                    }
                    else
                    {
                        [[sharedInstance readyLocationOperations] addObject:operation];
                    }
                }
                
                result = YES;
            }
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (BOOL)fetchAddressStringFromLocation:(CLLocation *)location identity:(NSString *)identity
{
    BOOL result = NO;
    
    do
    {
        if(location==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block PPLocationController *sharedInstance = [self sharedInstance];
        
        //////////////////////////////////////////////////
        
        if(sharedInstance==nil ||
           [sharedInstance operationQueue]==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *orders = [sharedInstance locationManagerOrders];
        
        if(orders==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(id locationManager in orders)
        {
            PPLocationManager *curLocationManager = [[self sharedInstance] ppLocationManagerForLocationManager:locationManager];
            
            if(curLocationManager==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block LocationBlockOperation *operation = [curLocationManager fetchAddressStringFromLocation:location
                                                                                                  identity:identity
                                                                                                   success:^{
                                                                                                       
                                                                                                       [sharedInstance removeRedundancyOperationBaseOnTargetOperation:operation];
                                                                                                       
                                                                                                       [sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]];
                                                                                                       
                                                                                                   } failure:^(NSError *error){
                                                                                                       [sharedInstance runNextOperationWithTimeInterval:[sharedInstance operationTimeInterval]];
                                                                                                   }];
            
            
            if(operation!=nil)
            {
                operation.identity = identity;
                operation.actionName = PPLocationControllerUserInfoKey_Location;
                
                //////////////////////////////////////////////////
                
                @synchronized([sharedInstance operationQueue])
                {
                    if([[sharedInstance operationQueue] operationCount]<=0)
                    {
                        [[sharedInstance operationQueue] addOperation:operation];
                    }
                    else
                    {
                        [[sharedInstance readyLocationOperations] addObject:operation];
                    }
                }
                
                result = YES;
            }
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (void)cancelAllOperation
{
    for(id key in [[self sharedInstance] managerDictionarys])
    {
        PPLocationManager *manager = [[[self sharedInstance] managerDictionarys] objectForKey:key];
        
        [manager cancelGeocode];
    }
    
    //////////////////////////////////////////////////

    [[[self sharedInstance] readyLocationOperations] removeAllObjects];
    
    //////////////////////////////////////////////////

    [[[self sharedInstance] operationQueue] cancelAllOperations];
}


//================================================================================
//
//================================================================================
+ (void)waitUntilAllLocationOperationsAreFinished
{
    [[[self sharedInstance] operationQueue] waitUntilAllOperationsAreFinished];
}


//================================================================================
//
//================================================================================
+ (NSUInteger)locationOperationCount
{
    return [[[self sharedInstance] operationQueue] operationCount];
}


//================================================================================
//
//================================================================================
+ (NSUInteger)readyOperationCount
{
    return [[[self sharedInstance] readyLocationOperations] count];
}


//================================================================================
//
//================================================================================
+ (double)meterDistanceBetweenLocation1:(CLLocation *)location1 location2:(CLLocation *)location2
{
    CLLocationDistance distance = [location1 distanceFromLocation:location2];
    
    return distance;
}


//================================================================================
//
//================================================================================
+ (NSString *)stringForLocation:(CLLocation *)location
{
    NSString *locationString = nil;
    
    do
    {
        if(location==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([self isValidLocation:location]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        locationString = [NSString stringWithFormat:@"%f,%f", location.coordinate.latitude, location.coordinate.longitude];
    }
    while (0);
    
    return locationString;
}


//================================================================================
//
//================================================================================
+ (CLLocation *)copyLocationFromString:(NSString *)GPSString
{
    CLLocation *location = nil;
    
    do
    {
        NSArray *paramArray = [GPSString componentsSeparatedByString:@","];
        
        if([paramArray count]!=2)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *latitude = [paramArray firstObject];
        NSString *longitude = [paramArray lastObject];
        
        CLLocationCoordinate2D coordinate = CLLocationCoordinate2DMake([latitude floatValue], [longitude floatValue]);
        
        if(CLLocationCoordinate2DIsValid(coordinate))
        {
            location = [[CLLocation alloc] initWithLatitude:coordinate.latitude longitude:coordinate.longitude];
        }
    }
    while (0);
    
    return location;
}


//================================================================================
//
//================================================================================
+ (BOOL)isValidLocation:(CLLocation *)location
{
    BOOL result = NO;
    
    do
    {
        if(location==nil)
        {
            break;
        }
        else if(location.coordinate.latitude==0 && location.coordinate.longitude==0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = CLLocationCoordinate2DIsValid(location.coordinate);
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (CLLocation *)invalidLocation
{
    return [[[CLLocation alloc] initWithLatitude:0 longitude:0] autorelease];
    
}


//================================================================================
//
//================================================================================
+ (NSString *)invalidLocationString
{
    return [NSString stringWithFormat:@"%f,%f",0.0f,0.0f];
}
@end
