//
//  GoogleLocationManager.m
//
//
//  Created by Howard on 2015/12/31.
//
//

#import "GoogleLocationManager.h"

// Define
#import "PPLocationController+ParameterDefine.h"

// Category
#import "NSError+Custom.h"

@implementation GoogleLocationManager

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (LocationBlockOperation *)fetchLocationFromAddressString:(NSString *)addressString
                                                    region:(CLCircularRegion *)region
                                                  identity:(NSString *)identity
                                                   success:(void (^)(void))success
                                                   failure:(void (^)(NSError *error))failure
{
    __block LocationBlockOperation *blockOperation = nil;
    __block NSError *error = nil;
    
    do
    {
        if(addressString==nil ||
           addressString.length<=0)
        {
            error = PPErrorParameterInvalidity(error);
            break;
        }
        
        //////////////////////////////////////////////////
        
        blockOperation = [[LocationBlockOperation alloc] init];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block id blockIdentity = identity;
        
        if(blockIdentity==nil)
        {
            blockIdentity = [NSNull null];
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                if(blockOperation.isCancelled==YES)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(NSErrorCustom_Code_OperationCancel, @"fetchLocationFromCardModel Operatoin Cancel", nil)}];
                    
                    break;
                }
                else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                    
                    break;
                }
                
                //////////////////////////////////////////////////
                
                
                if([addressString length]<=0)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                @autoreleasepool
                {
                
                    //google map v3
                    NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"https://maps.googleapis.com/maps/api/geocode/json?address=%@&sensor=false", [addressString stringByAddingPercentEscapesUsingEncoding:NSUTF8StringEncoding]]];
                    
                    if(url==nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSError	*error = nil;
                    NSData *resultData = [NSData dataWithContentsOfURL:url options:NSDataReadingMappedIfSafe error:&error];
                  
                    if(error!=nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////

                    NSDictionary *JSONObject = [NSJSONSerialization JSONObjectWithData:resultData options:NSJSONReadingMutableContainers error:&error];
                    
                    if(JSONObject==nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////

                    NSString *status = [JSONObject valueForKey:@"status"];
                  
                    if([status isEqualToString:@"OK"]==NO)
                    {
                        NSString *errorMessage = [JSONObject valueForKey:@"error_message"];

                        NSInteger errorCode = PPLocationControllerErrorCode_Other;
                        
                        // time out
                        if([status isEqualToString:@"OVER_QUERY_LIMIT"]==YES)
                        {
                            errorCode = PPLocationControllerErrorCode_RequestTimeOut;
                        }

                        //////////////////////////////////////////////////

                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Address:addressString,
                                                                                     PPLocationControllerUserInfoKey_Error:PPErrorMake(errorCode, errorMessage, nil)}];
                        failure(error);
                        
                        break;
                    }
            
                    //////////////////////////////////////////////////

                    NSArray *results = [JSONObject valueForKey:@"results"];
                    NSNumber *lat = [[[[results objectAtIndex:0] valueForKey:@"geometry"]valueForKey:@"location" ] valueForKey:@"lat"];
                    NSNumber *lng = [[[[results objectAtIndex:0] valueForKey:@"geometry"]valueForKey:@"location" ] valueForKey:@"lng"];

                    //////////////////////////////////////////////////

                    CLLocation *locaiton = [[CLLocation alloc] initWithLatitude:[lat floatValue]
                                                                      longitude:[lng floatValue]];
                    
                    if(locaiton==nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                 PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationSuccess),
                                                                                 PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Address:addressString,
                                                                                 PPLocationControllerUserInfoKey_Location:locaiton}];
                    
                    [locaiton autorelease];
                    
                    success();
                }
            }
            while (0);
 
            //operation end
            
        }];
        // while loop end
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil && failure!=NULL)
    {
        failure(error);
    }
    
    //////////////////////////////////////////////////
    
    return [blockOperation autorelease];
}


//================================================================================
//
//================================================================================
- (LocationBlockOperation *)fetchAddressStringFromLocation:(CLLocation *)location
                                                  identity:(NSString *)identity
                                                   success:(void (^)(void))success
                                                   failure:(void (^)(NSError *error))failure
{
    __block LocationBlockOperation *blockOperation = nil;
    __block NSError *error = nil;
    
    do
    {
        if(location==nil)
        {
            error = PPErrorParameterInvalidity(error);
            break;
        }
        
        //////////////////////////////////////////////////
        
        blockOperation = [[LocationBlockOperation alloc] init];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block id blockIdentity = identity;
        
        if(blockIdentity==nil)
        {
            blockIdentity = [NSNull null];
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                if(blockOperation.isCancelled==YES)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(NSErrorCustom_Code_OperationCancel, @"fetchLocationFromCardModel Operatoin Cancel", nil)}];
                    
                    break;
                }
                else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                    
                    break;
                }
                //////////////////////////////////////////////////
                
                @autoreleasepool
                {
                    
                    //google map v3
                    NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"https://maps.googleapis.com/maps/api/geocode/json?latlng=%f,%f",location.coordinate.latitude,location.coordinate.longitude]];
                    
                    if(url==nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSError	*error = nil;
                    NSData *resultData = [NSData dataWithContentsOfURL:url options:NSDataReadingMappedIfSafe error:&error];
                    
                    if(error!=nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSDictionary *JSONObject = [NSJSONSerialization JSONObjectWithData:resultData options:NSJSONReadingMutableContainers error:&error];
                    
                    if(JSONObject==nil)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSString *status = [JSONObject valueForKey:@"status"];
                    
                    if([status isEqualToString:@"OK"]==NO)
                    {
                        NSString *errorMessage = [JSONObject valueForKey:@"error_message"];
                        
                        NSInteger errorCode = PPLocationControllerErrorCode_Other;
                        
                        // time out
                        if([status isEqualToString:@"OVER_QUERY_LIMIT"]==YES)
                        {
                            errorCode = PPLocationControllerErrorCode_RequestTimeOut;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Location:location,
                                                                                     PPLocationControllerUserInfoKey_Error:PPErrorMake(errorCode, errorMessage, nil)}];
                        failure(error);
                        
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSArray *results = [JSONObject valueForKey:@"results"];
                    
                    if(results.count>0)
                    {
                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringSuccess),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity,
                                                                                     PPLocationControllerUserInfoKey_AddressDictionary:[results firstObject]}];
                    }
                    else
                    {
                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringSuccess),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity}];
                    }
                    
                    //////////////////////////////////////////////////

                    success();
                }
            }
            while (0);
            
            //operation end
            
        }];
        // while loop end
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil && failure!=NULL)
    {
        failure(error);
    }
    
    //////////////////////////////////////////////////
    
    return [blockOperation autorelease];
}

@end
