//
//  PPIndexingController.m
//  
//
//  Created by Mike on 13/5/28.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPIndexingController.h"
#import "Indexing.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPIndexingControllerHiraganaVoiceBufferSize 512
#define PPIndexingControllerTrimmingCharactersSet   [NSCharacterSet characterSetWithCharactersInString:@" -/:;()$&@\".,?!'[]{}#%^*+=_\\|~<>€£¥•"]

static PPIndexingController *staticIndexingController = nil;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPIndexingController()

@interface PPIndexingController()
+ (NSDictionary *)indexLookupDictionaryForStyle:(PPIndexingStyle)style;
+ (NSDictionary *)indexReverseLookupDictionaryForStyle:(PPIndexingStyle)style;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPIndexingController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

+ (NSDictionary *)indexLookupDictionaryForStyle:(PPIndexingStyle)style
{
    NSMutableDictionary *indexLookupDictionary = [[[NSMutableDictionary alloc] init] autorelease];
    if(indexLookupDictionary!=nil)
    {
        switch(style)
        {
            case PPIndexingStyle_English:
            {
                [indexLookupDictionary setObject:@"aAāĀáÁàÀâÂåÅäÄãÃąĄæÆ"    forKey:@"A"];
                [indexLookupDictionary setObject:@"bB"                      forKey:@"B"];
                [indexLookupDictionary setObject:@"cCćĆčČçÇ"                forKey:@"C"];
                [indexLookupDictionary setObject:@"dD"                      forKey:@"D"];
                [indexLookupDictionary setObject:@"eEēĒéÉèÈeEêÊëËėĖęĘ"      forKey:@"E"];
                [indexLookupDictionary setObject:@"fF"                      forKey:@"F"];
                [indexLookupDictionary setObject:@"gG"                      forKey:@"G"];
                [indexLookupDictionary setObject:@"hH"                      forKey:@"H"];
                [indexLookupDictionary setObject:@"iIīĪíÍìÌîÎïÏįĮ"          forKey:@"I"];
                [indexLookupDictionary setObject:@"jJ"                      forKey:@"J"];
                [indexLookupDictionary setObject:@"kK"                      forKey:@"K"];
                [indexLookupDictionary setObject:@"lLłŁ"                    forKey:@"L"];
                [indexLookupDictionary setObject:@"mM"                      forKey:@"M"];
                [indexLookupDictionary setObject:@"nNńŃñÑ"                  forKey:@"N"];
                [indexLookupDictionary setObject:@"oOōŌóÓòÒôÔöÖõÕøØœŒ"      forKey:@"O"];
                [indexLookupDictionary setObject:@"pP"                      forKey:@"P"];
                [indexLookupDictionary setObject:@"qQ"                      forKey:@"Q"];
                [indexLookupDictionary setObject:@"rR"                      forKey:@"R"];
                [indexLookupDictionary setObject:@"sSśŚšŠß"                 forKey:@"S"];
                [indexLookupDictionary setObject:@"tT"                      forKey:@"T"];
                [indexLookupDictionary setObject:@"uUūŪúÚùÙûÛüÜ"			forKey:@"U"];
                [indexLookupDictionary setObject:@"vV"                      forKey:@"V"];
                [indexLookupDictionary setObject:@"wW"                      forKey:@"W"];
                [indexLookupDictionary setObject:@"xX"                      forKey:@"X"];
                [indexLookupDictionary setObject:@"yYÿŸ"					forKey:@"Y"];
                [indexLookupDictionary setObject:@"zZźŹžŽżŻ"				forKey:@"Z"];
                break;
            }
            case PPIndexingStyle_Zhuyin:
            {
                [indexLookupDictionary setObject:@"ㄅ"                     forKey:@"ㄅ"];
                [indexLookupDictionary setObject:@"ㄆ"                     forKey:@"ㄆ"];
                [indexLookupDictionary setObject:@"ㄇ"                     forKey:@"ㄇ"];
                [indexLookupDictionary setObject:@"ㄈ"                     forKey:@"ㄈ"];
                [indexLookupDictionary setObject:@"ㄉ"                     forKey:@"ㄉ"];
                [indexLookupDictionary setObject:@"ㄊ"                     forKey:@"ㄊ"];
                [indexLookupDictionary setObject:@"ㄋ"                     forKey:@"ㄋ"];
                [indexLookupDictionary setObject:@"ㄌ"                     forKey:@"ㄌ"];
                [indexLookupDictionary setObject:@"ㄍ"                     forKey:@"ㄍ"];
                [indexLookupDictionary setObject:@"ㄎ"                     forKey:@"ㄎ"];
                [indexLookupDictionary setObject:@"ㄏ"                     forKey:@"ㄏ"];
                [indexLookupDictionary setObject:@"ㄐ"                     forKey:@"ㄐ"];
                [indexLookupDictionary setObject:@"ㄑ"                     forKey:@"ㄑ"];
                [indexLookupDictionary setObject:@"ㄒ"                     forKey:@"ㄒ"];
                [indexLookupDictionary setObject:@"ㄓ"                     forKey:@"ㄓ"];
                [indexLookupDictionary setObject:@"ㄔ"                     forKey:@"ㄔ"];
                [indexLookupDictionary setObject:@"ㄕ"                     forKey:@"ㄕ"];
                [indexLookupDictionary setObject:@"ㄖ"                     forKey:@"ㄖ"];
                [indexLookupDictionary setObject:@"ㄗ"                     forKey:@"ㄗ"];
                [indexLookupDictionary setObject:@"ㄘ"                     forKey:@"ㄘ"];
                [indexLookupDictionary setObject:@"ㄙ"                     forKey:@"ㄙ"];
                [indexLookupDictionary setObject:@"ㄚ"                     forKey:@"ㄚ"];
                [indexLookupDictionary setObject:@"ㄛ"                     forKey:@"ㄛ"];
                [indexLookupDictionary setObject:@"ㄜ"                     forKey:@"ㄜ"];
                [indexLookupDictionary setObject:@"ㄝ"                     forKey:@"ㄝ"];
                [indexLookupDictionary setObject:@"ㄞ"                     forKey:@"ㄞ"];
                [indexLookupDictionary setObject:@"ㄟ"                     forKey:@"ㄟ"];
                [indexLookupDictionary setObject:@"ㄠ"                     forKey:@"ㄠ"];
                [indexLookupDictionary setObject:@"ㄡ"                     forKey:@"ㄡ"];
                [indexLookupDictionary setObject:@"ㄢ"                     forKey:@"ㄢ"];
                [indexLookupDictionary setObject:@"ㄣ"                     forKey:@"ㄣ"];
                [indexLookupDictionary setObject:@"ㄤ"                     forKey:@"ㄤ"];
                [indexLookupDictionary setObject:@"ㄥ"                     forKey:@"ㄥ"];
                [indexLookupDictionary setObject:@"ㄦ"                     forKey:@"ㄦ"];
                [indexLookupDictionary setObject:@"ㄧ"                     forKey:@"ㄧ"];
                [indexLookupDictionary setObject:@"ㄨ"                     forKey:@"ㄨ"];
                [indexLookupDictionary setObject:@"ㄩ"                     forKey:@"ㄩ"];
                
                break;
            }
            case PPIndexingStyle_Hanpin:
            {
                [indexLookupDictionary setObject:@"aA"                    forKey:@"A"];
                [indexLookupDictionary setObject:@"bB"                    forKey:@"B"];
                [indexLookupDictionary setObject:@"cC"                    forKey:@"C"];
                [indexLookupDictionary setObject:@"dD"                    forKey:@"D"];
                [indexLookupDictionary setObject:@"eE"                    forKey:@"E"];
                [indexLookupDictionary setObject:@"fF"                    forKey:@"F"];
                [indexLookupDictionary setObject:@"gG"                    forKey:@"G"];
                [indexLookupDictionary setObject:@"hH"                    forKey:@"H"];
                [indexLookupDictionary setObject:@"iI"                    forKey:@"I"];
                [indexLookupDictionary setObject:@"jJ"                    forKey:@"J"];
                [indexLookupDictionary setObject:@"kK"                    forKey:@"K"];
                [indexLookupDictionary setObject:@"lL"                    forKey:@"L"];
                [indexLookupDictionary setObject:@"mM"                    forKey:@"M"];
                [indexLookupDictionary setObject:@"nN"                    forKey:@"N"];
                [indexLookupDictionary setObject:@"oO"                    forKey:@"O"];
                [indexLookupDictionary setObject:@"pP"                    forKey:@"P"];
                [indexLookupDictionary setObject:@"qQ"                    forKey:@"Q"];
                [indexLookupDictionary setObject:@"rR"                    forKey:@"R"];
                [indexLookupDictionary setObject:@"sS"                    forKey:@"S"];
                [indexLookupDictionary setObject:@"tT"                    forKey:@"T"];
                [indexLookupDictionary setObject:@"uU"                    forKey:@"U"];
                [indexLookupDictionary setObject:@"vV"                    forKey:@"V"];
                [indexLookupDictionary setObject:@"wW"                    forKey:@"W"];
                [indexLookupDictionary setObject:@"xX"                    forKey:@"X"];
                [indexLookupDictionary setObject:@"yY"                    forKey:@"Y"];
                [indexLookupDictionary setObject:@"zZ"                    forKey:@"Z"];
                break;
            }
            case PPIndexingStyle_Hiragana:
            {	// 最後小字為半形
                [indexLookupDictionary setObject:@"ぁあァアｧｱ"                forKey:@"あ"];// a
                [indexLookupDictionary setObject:@"ぃいィイｨｲ"                forKey:@"い"];// i
                [indexLookupDictionary setObject:@"ぅうゥウｩｳ"                forKey:@"う"];// u
                [indexLookupDictionary setObject:@"ぇえェエｪｴ"                forKey:@"え"];// e
                [indexLookupDictionary setObject:@"ぉおォオｫｵ"                forKey:@"お"];// o
                [indexLookupDictionary setObject:@"かがカガｶ"                 forKey:@"か"];// ka
                [indexLookupDictionary setObject:@"きぎキギｷ"                 forKey:@"き"];// ki
                [indexLookupDictionary setObject:@"くぐクグｸ"                 forKey:@"く"];// ku
                [indexLookupDictionary setObject:@"けげケゲｹ"                 forKey:@"け"];// ke
                [indexLookupDictionary setObject:@"こごコゴｺ"                 forKey:@"こ"];// ko
                [indexLookupDictionary setObject:@"さざサザｻ"                 forKey:@"さ"];// sa
                [indexLookupDictionary setObject:@"しじシジｼ"                 forKey:@"し"];// si
                [indexLookupDictionary setObject:@"すずスズｽ"                 forKey:@"す"];// su
                [indexLookupDictionary setObject:@"せぜセゼｾ"                 forKey:@"せ"];// se
                [indexLookupDictionary setObject:@"そぞソゾｿ"                 forKey:@"そ"];// so
                [indexLookupDictionary setObject:@"ただタダﾀ"                 forKey:@"た"];// ta
                [indexLookupDictionary setObject:@"ちぢチヂﾁ"                 forKey:@"ち"];// ti
                [indexLookupDictionary setObject:@"っつづッツヅｯﾂ"             forKey:@"つ"];// tu
                [indexLookupDictionary setObject:@"てでテデﾃ"                 forKey:@"て"];// te
                [indexLookupDictionary setObject:@"とどトドﾄ"                 forKey:@"と"];// to
                [indexLookupDictionary setObject:@"なナﾅ"                    forKey:@"な"];// na
                [indexLookupDictionary setObject:@"にニﾆ"                    forKey:@"に"];// ni
                [indexLookupDictionary setObject:@"ぬヌﾇ"                    forKey:@"ぬ"];// nu
                [indexLookupDictionary setObject:@"ねネﾈ"                    forKey:@"ね"];// ne
                [indexLookupDictionary setObject:@"のノﾉ"                    forKey:@"の"];// no
                [indexLookupDictionary setObject:@"はばぱハバパﾊ"             forKey:@"は"];// ha
                [indexLookupDictionary setObject:@"ひびぴヒビピﾋ"             forKey:@"ひ"];// hi
                [indexLookupDictionary setObject:@"ふぶぷフブプﾌ"             forKey:@"ふ"];// hu
                [indexLookupDictionary setObject:@"へべぺヘベペﾍ"             forKey:@"へ"];// he
                [indexLookupDictionary setObject:@"ほぼぽホボポﾎ"             forKey:@"ほ"];// ho
                [indexLookupDictionary setObject:@"まマﾏ"                    forKey:@"ま"];// ma
                [indexLookupDictionary setObject:@"みミﾐ"                    forKey:@"み"];// mi
                [indexLookupDictionary setObject:@"むムﾑ"                    forKey:@"む"];// mu
                [indexLookupDictionary setObject:@"めメﾒ"                    forKey:@"め"];// me
                [indexLookupDictionary setObject:@"もモﾓ"                    forKey:@"も"];// mo
                [indexLookupDictionary setObject:@"ゃやャヤﾔ"                forKey:@"や"];// ya
                [indexLookupDictionary setObject:@"ゅゆュユｭ"                forKey:@"ゆ"];// yu
                [indexLookupDictionary setObject:@"ょよョヨｮ"                forKey:@"よ"];// yo
                [indexLookupDictionary setObject:@"らラﾗ"                   forKey:@"ら"];// ra
                [indexLookupDictionary setObject:@"りリﾘ"                   forKey:@"り"];// ri
                [indexLookupDictionary setObject:@"るルﾙ"                   forKey:@"る"];// ru
                [indexLookupDictionary setObject:@"れレﾚ"                   forKey:@"れ"];// ra
                [indexLookupDictionary setObject:@"ろロﾛ"                   forKey:@"ろ"];// ro
                [indexLookupDictionary setObject:@"ゎわヮワﾜ"                forKey:@"わ"];// wa
                [indexLookupDictionary setObject:@"ゐヰ"                    forKey:@"ゐ"];// wi
                [indexLookupDictionary setObject:@"ゑヱ"                    forKey:@"ゑ"];// we
                [indexLookupDictionary setObject:@"をヲｦ"                   forKey:@"を"];// wo
                [indexLookupDictionary setObject:@"んンﾝ"                   forKey:@"ん"];// n
                break;
            }
            case PPIndexingStyle_Hangul:
            {	// 以下各組韓文看起來一樣，其實是不同unicode。('ᄀ' & 'ㄱ' ) // 最後小字為半形
                [indexLookupDictionary setObject:@"ᄀㄱﾡ"                   forKey:@"ᄀ"];
                [indexLookupDictionary setObject:@"ᄁㄲﾢ"                   forKey:@"ᄁ"];
                [indexLookupDictionary setObject:@"ᄂㄴﾤ"                   forKey:@"ᄂ"];
                [indexLookupDictionary setObject:@"ᄃㄷﾧ"                   forKey:@"ᄃ"];
                [indexLookupDictionary setObject:@"ᄄㄸﾨ"                   forKey:@"ᄄ"];
                [indexLookupDictionary setObject:@"ᄅㄹﾩ"                   forKey:@"ᄅ"];
                [indexLookupDictionary setObject:@"ᄆㅁﾱ"                   forKey:@"ᄆ"];
                [indexLookupDictionary setObject:@"ᄇㅂﾲ"                   forKey:@"ᄇ"];
                [indexLookupDictionary setObject:@"ᄈㅃﾳ"                   forKey:@"ᄈ"];
                [indexLookupDictionary setObject:@"ᄉㅅﾵ"                   forKey:@"ᄉ"];
                [indexLookupDictionary setObject:@"ᄊㅆﾶ"                   forKey:@"ᄊ"];
                [indexLookupDictionary setObject:@"ᄋㅇﾷ"                   forKey:@"ᄋ"];
                [indexLookupDictionary setObject:@"ᄌㅈﾸ"                   forKey:@"ᄌ"];
                [indexLookupDictionary setObject:@"ᄍㅉﾹ"                   forKey:@"ᄍ"];
                [indexLookupDictionary setObject:@"ᄎㅊﾺ"                   forKey:@"ᄎ"];
                [indexLookupDictionary setObject:@"ᄏㅋﾻ"                   forKey:@"ᄏ"];
                [indexLookupDictionary setObject:@"ᄐㅌﾼ"                   forKey:@"ᄐ"];
                [indexLookupDictionary setObject:@"ᄑㅍﾽ"                   forKey:@"ᄑ"];
                [indexLookupDictionary setObject:@"ᄒㅎﾾ"                   forKey:@"ᄒ"];
                break;
            }
            case PPIndexingStyle_Thai:
            {
                [indexLookupDictionary setObject:@"ก" forKey:@"ก"];
                [indexLookupDictionary setObject:@"ข" forKey:@"ข"];
                [indexLookupDictionary setObject:@"ฃ" forKey:@"ฃ"];
                [indexLookupDictionary setObject:@"ค" forKey:@"ค"];
                [indexLookupDictionary setObject:@"ฅ" forKey:@"ฅ"];
                [indexLookupDictionary setObject:@"ฆ" forKey:@"ฆ"];
                [indexLookupDictionary setObject:@"ง" forKey:@"ง"];
                [indexLookupDictionary setObject:@"จ" forKey:@"จ"];
                [indexLookupDictionary setObject:@"ฉ" forKey:@"ฉ"];
                [indexLookupDictionary setObject:@"ช" forKey:@"ช"];
                [indexLookupDictionary setObject:@"ซ" forKey:@"ซ"];
                [indexLookupDictionary setObject:@"ฌ" forKey:@"ฌ"];
                [indexLookupDictionary setObject:@"ญ" forKey:@"ญ"];
                [indexLookupDictionary setObject:@"ฎ" forKey:@"ฎ"];
                [indexLookupDictionary setObject:@"ฏ" forKey:@"ฏ"];
                [indexLookupDictionary setObject:@"ฐ" forKey:@"ฐ"];
                [indexLookupDictionary setObject:@"ฑ" forKey:@"ฑ"];
                [indexLookupDictionary setObject:@"ฒ" forKey:@"ฒ"];
                [indexLookupDictionary setObject:@"ณ" forKey:@"ณ"];
                [indexLookupDictionary setObject:@"ด" forKey:@"ด"];
                [indexLookupDictionary setObject:@"ต" forKey:@"ต"];
                [indexLookupDictionary setObject:@"ถ" forKey:@"ถ"];
                [indexLookupDictionary setObject:@"ท" forKey:@"ท"];
                [indexLookupDictionary setObject:@"ธ" forKey:@"ธ"];
                [indexLookupDictionary setObject:@"น" forKey:@"น"];
                [indexLookupDictionary setObject:@"บ" forKey:@"บ"];
                [indexLookupDictionary setObject:@"ป" forKey:@"ป"];
                [indexLookupDictionary setObject:@"ผ" forKey:@"ผ"];
                [indexLookupDictionary setObject:@"ฝ" forKey:@"ฝ"];
                [indexLookupDictionary setObject:@"พ" forKey:@"พ"];
                [indexLookupDictionary setObject:@"ฟ" forKey:@"ฟ"];
                [indexLookupDictionary setObject:@"ภ" forKey:@"ภ"];
                [indexLookupDictionary setObject:@"ม" forKey:@"ม"];
                [indexLookupDictionary setObject:@"ย" forKey:@"ย"];
                [indexLookupDictionary setObject:@"ร" forKey:@"ร"];
                [indexLookupDictionary setObject:@"ล" forKey:@"ล"];
                [indexLookupDictionary setObject:@"ว" forKey:@"ว"];
                [indexLookupDictionary setObject:@"ศ" forKey:@"ศ"];
                [indexLookupDictionary setObject:@"ษ" forKey:@"ษ"];
                [indexLookupDictionary setObject:@"ส" forKey:@"ส"];
                [indexLookupDictionary setObject:@"ห" forKey:@"ห"];
                [indexLookupDictionary setObject:@"ฬ" forKey:@"ฬ"];
                [indexLookupDictionary setObject:@"อ" forKey:@"อ"];
                [indexLookupDictionary setObject:@"ฮ" forKey:@"ฮ"];


                break;
            }
            case PPIndexingStyle_Swedish:
            {
                [indexLookupDictionary setObject:@"aAÀà"      forKey:@"A"];
                [indexLookupDictionary setObject:@"bB"        forKey:@"B"];
                [indexLookupDictionary setObject:@"cC"        forKey:@"C"];
                [indexLookupDictionary setObject:@"dD"        forKey:@"D"];
                [indexLookupDictionary setObject:@"eEÉËÊèëê"  forKey:@"E"];
                [indexLookupDictionary setObject:@"fF"        forKey:@"F"];
                [indexLookupDictionary setObject:@"gG"        forKey:@"G"];
                [indexLookupDictionary setObject:@"hH"        forKey:@"H"];
                [indexLookupDictionary setObject:@"iI"        forKey:@"I"];
                [indexLookupDictionary setObject:@"jJ"        forKey:@"J"];
                [indexLookupDictionary setObject:@"kK"        forKey:@"K"];
                [indexLookupDictionary setObject:@"lL"        forKey:@"L"];
                [indexLookupDictionary setObject:@"mM"        forKey:@"M"];
                [indexLookupDictionary setObject:@"nN"        forKey:@"N"];
                [indexLookupDictionary setObject:@"oO"        forKey:@"O"];
                [indexLookupDictionary setObject:@"pP"        forKey:@"P"];
                [indexLookupDictionary setObject:@"qQ"        forKey:@"Q"];
                [indexLookupDictionary setObject:@"rR"        forKey:@"R"];
                [indexLookupDictionary setObject:@"sS"        forKey:@"S"];
                [indexLookupDictionary setObject:@"tT"        forKey:@"T"];
                [indexLookupDictionary setObject:@"uUÜü"      forKey:@"U"];
                [indexLookupDictionary setObject:@"vV"        forKey:@"V"];
                [indexLookupDictionary setObject:@"wW"        forKey:@"W"];
                [indexLookupDictionary setObject:@"xX"        forKey:@"X"];
                [indexLookupDictionary setObject:@"yY"        forKey:@"Y"];
                [indexLookupDictionary setObject:@"zZ"        forKey:@"Z"];
                [indexLookupDictionary setObject:@"Åå"        forKey:@"Å"];
                [indexLookupDictionary setObject:@"Ää"        forKey:@"Ä"];
                [indexLookupDictionary setObject:@"Öö"        forKey:@"Ö"];
                
                break;
            }
            case PPIndexingStyle_Stroke:
            {
                for(unsigned short stroke=1; stroke<=28; stroke++)
                {
                    //這邊是特殊型態處理
                    NSString *key = [NSString stringWithCharacters:&stroke length:1];
                    
                    [indexLookupDictionary setObject:key forKey:key];
                }
                
                break;
            }
            default:
            {
                break;
            }
        }
    }
    
    return indexLookupDictionary;
}

+ (NSDictionary *)indexReverseLookupDictionaryForStyle:(PPIndexingStyle)style
{
    NSMutableDictionary *indexReverseLookupDictionary = nil;
    
    do
    {
        NSDictionary *indexLookupDictionary = [PPIndexingController indexLookupDictionaryForStyle:style];
        if(indexLookupDictionary==nil)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        indexReverseLookupDictionary = [[[NSMutableDictionary alloc] init] autorelease];
        if(indexReverseLookupDictionary==nil)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        NSArray *indexLookupKeys = [indexLookupDictionary allKeys];
        
        for(NSString *indexLookupKey in indexLookupKeys)
        {
            NSString    *indexLookupCondition      = [indexLookupDictionary objectForKey:indexLookupKey];
            NSInteger   indexLookupConditionLength = [indexLookupCondition length];
            
            for(int location=0; location<indexLookupConditionLength; location++)
            {
                [indexReverseLookupDictionary setObject:indexLookupKey forKey:[indexLookupCondition substringWithRange:NSMakeRange(location, 1)]];
            }
        }
        
    }while(0);
    
    return indexReverseLookupDictionary;
}


//==============================================================================
//
//==============================================================================
- (unsigned short)thaiIndexAliasFromResult:(unsigned char)result
{
    NSArray *thaiIndexes = @[@"ก",@"ข",@"ฃ",@"ค",@"ฅ",@"ฆ",@"ง",@"จ",@"ฉ",@"ช",@"ซ",@"ฌ",@"ญ",@"ฎ",@"ฏ",@"ฐ",@"ฑ",@"ฒ",@"ณ",@"ด",@"ต",@"ถ",@"ท",@"ธ",@"น",@"บ",@"ป",@"ผ",@"ฝ",@"พ",@"ฟ",@"ภ",@"ม",@"ย",@"ร",@"ล",@"ว",@"ศ",@"ษ",@"ส",@"ห",@"ฬ",@"อ",@"ฮ"];
    if (result>=0 && result<=43)
    {
        NSString *index = [thaiIndexes objectAtIndex:result];
        if ([index length]>0)
        {
            unichar indexChar = [index characterAtIndex:0];
            return (unsigned short)indexChar;
        }
    }

    return PPIndexAliasOfOther;
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - instance Methods


//==============================================================================
//
//==============================================================================
- (NSArray *)indexsForStyle:(PPIndexingStyle)style
{
    @synchronized (self)
    {
        NSMutableArray *indexs = nil;
        
        do
        {
            NSDictionary *indexLookupDictionary = [PPIndexingController indexLookupDictionaryForStyle:style];
            if(indexLookupDictionary==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            indexs = [[[NSMutableArray alloc] init] autorelease];
            if(indexs==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSArray *indexLookupDictionaryAllKeys = [indexLookupDictionary allKeys];
            
            switch(style)
            {
                case PPIndexingStyle_Stroke:
                {
                    //這邊型態是特殊處理
                    for(NSString *key in indexLookupDictionaryAllKeys)
                    {
                        [indexs addObject:[NSString stringWithFormat:@"%d", [key characterAtIndex:0]]];
                    }
                    
                    break;
                }
                case PPIndexingStyle_English:
                case PPIndexingStyle_Zhuyin:
                case PPIndexingStyle_Hanpin:
                case PPIndexingStyle_Hiragana:
                case PPIndexingStyle_Hangul:
                default:
                {
                    [indexs addObjectsFromArray:indexLookupDictionaryAllKeys];
                    
                    break;
                }
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [indexs sortUsingSelector:@selector(localizedStandardCompare:)];
            [indexs addObject:PPIndexStringOfOther];
            
        }while(0);
        
        return indexs;
        
    }
}



//==============================================================================
//
//==============================================================================
- (unsigned short)indexAliasOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    @synchronized (self)
    {
        unsigned short indexAlias = PPIndexAliasOfOther;
        
        do
        {
            if(string==nil || [string length]==0)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //先過濾掉符號那些
            
            NSString *trimmingString = [string stringByTrimmingCharactersInSet:PPIndexingControllerTrimmingCharactersSet];
            if(trimmingString==nil || [trimmingString length]==0)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //取出首字
            
            UniChar firstCharacter = [trimmingString characterAtIndex:0];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //先給核心判斷細分
            
            switch(style)
            {
                case PPIndexingStyle_English:
                case PPIndexingStyle_Swedish:
                {
                    indexAlias = firstCharacter;
                    
                    break;
                }
                    
                case PPIndexingStyle_Zhuyin:
                case PPIndexingStyle_Hanpin:
                {
                    if([NSString isCJKCharacter:firstCharacter]==YES)
                    {
                        indexAlias = CHFindVoice(&firstCharacter, (int)style);
                    }
                    
                    break;
                }
                    
                case PPIndexingStyle_Stroke:
                {
                    if([NSString isCJKCharacter:firstCharacter]==YES)
                    {
                        indexAlias = CHFindVoice(&firstCharacter, (int)style);
                    }
                    else
                    {
                        // !! 找不到index的，alias都要設為0。
                        indexAlias = 0;
                    }
                    
                    break;
                }
                    
                case PPIndexingStyle_Hiragana:
                {
                    NSString *phonetic = [PPIndexingController phoneticOfString:trimmingString forStyle:style];
                    
                    if([phonetic length] > 0)
                    {
                        indexAlias = [phonetic characterAtIndex:0];
                    }
                    
                    break;
                }
                    
                case PPIndexingStyle_Hangul:
                {
                    //韓文是取部首，不需判斷是否為漢字
                    indexAlias = KSFindVoice(firstCharacter, 0);
                    
                    break;
                }
                case PPIndexingStyle_Thai:
                {
                    // !! 泰文要整個字串傳進去，才能取得正確的indexing
                    NSUInteger stringLength = [trimmingString length];
					
                    if(stringLength>0)
                    {
                        NSUInteger codeBufferLength = (stringLength+1)*sizeof(UniChar);
                        UniChar *codeBuffer = (UniChar *)malloc(codeBufferLength);
						
                        if(codeBuffer!=NULL)
                        {
							[trimmingString getCharacters:codeBuffer];
							unsigned char thaiResult = THThai2index(codeBuffer);
							indexAlias = [self thaiIndexAliasFromResult:thaiResult];
							free(codeBuffer);
						}
                    }
                    break;
                }
                default:
                {
                    break;
                }
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            switch(style)
            {
                case PPIndexingStyle_English:
                case PPIndexingStyle_Zhuyin:
                case PPIndexingStyle_Hanpin:
                case PPIndexingStyle_Hiragana:
                case PPIndexingStyle_Hangul:
                case PPIndexingStyle_Stroke:
                case PPIndexingStyle_Thai:
                case PPIndexingStyle_Swedish:

                {
                    //漢拼的情況下，文字又不是中日韓，直接退開
                    if(style==PPIndexingStyle_Hanpin && [NSString isCJKCharacter:firstCharacter]==NO)
                    {
                        break;
                    }
                    else
                    {
                        NSDictionary *indexReverseLookupDictionary = [PPIndexingController indexReverseLookupDictionaryForStyle:style];
                        if(indexReverseLookupDictionary!=nil)
                        {
                            //核心如果找不到,則用第一個字去規則找
                            
                            if(indexAlias==PPIndexAliasOfOther)
                            {
                                indexAlias = firstCharacter;
                            }
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            NSString *index = [indexReverseLookupDictionary objectForKey:[NSString stringWithCharacters:&indexAlias length:1]];
                            
                            if([index length] > 0)
                            {
                                indexAlias = [index characterAtIndex:0];
                            }
                            else
                            {
                                indexAlias = PPIndexAliasOfOther;
                            }
                        }
                        
                        break;
                    }
                }
                default:
                {
                    break;
                }
            }
            
        }while(0);
        
        return indexAlias;
    }
}



//==============================================================================
//
//==============================================================================
- (NSString *)indexOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    @synchronized (self)
    {
        NSString *index = PPIndexStringOfOther;
        
        do
        {
            unsigned short indexAlias = [PPIndexingController indexAliasOfString:string forStyle:style];
            if(indexAlias==PPIndexAliasOfOther)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            switch(style)
            {
                case PPIndexingStyle_English:
                case PPIndexingStyle_Zhuyin:
                case PPIndexingStyle_Hanpin:
                case PPIndexingStyle_Hiragana:
                case PPIndexingStyle_Hangul:
                case PPIndexingStyle_Thai:
                case PPIndexingStyle_Swedish:
                {
                    index = [NSString stringWithCharacters:&indexAlias length:1];
                    
                    break;
                }
                case PPIndexingStyle_Stroke:
                {
                    if(indexAlias>0)
                    {
                        index = [NSString stringWithFormat:@"%d", indexAlias];
                    }
                    
                    break;
                }
                default:
                {
                    break;
                }
            }
            
        }while(0);
        
        return index;
    }
}



//==============================================================================
//
//==============================================================================
- (NSString *)phoneticOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    @synchronized (self)
    {
        NSString *phonetic = nil;
        
        do
        {
            if(string==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSUInteger stringLength = [string length];
            if(stringLength==0)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            if([NSString isCJKCharacter:[string characterAtIndex:0]]==NO)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSUInteger  codeBufferLength    = (stringLength+1)*sizeof(UniChar);
            UniChar     *codeBuffer         = (UniChar *)malloc(codeBufferLength);
            
            if(codeBuffer!=NULL)
            {
                memset(codeBuffer, 0, codeBufferLength);
                
                ////////////////////////////////////////////////////////////////////////////////////////////////////
                
                switch(style)
                {
                    case PPIndexingStyle_Hiragana:
                    {
                        UniChar *voiceBuffer = (UniChar *)malloc(PPIndexingControllerHiraganaVoiceBufferSize);
                        
                        if(voiceBuffer!=NULL)
                        {
                            memset(voiceBuffer, 0, PPIndexingControllerHiraganaVoiceBufferSize);
                            
                            [string getCharacters:codeBuffer];
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            NSString *path = [[NSBundle mainBundle] resourcePath];
                            
                            if(JPFindVoice([path cStringUsingEncoding:NSASCIIStringEncoding], codeBuffer, voiceBuffer)>0)
                            {
                                for(int index=0; index<PPIndexingControllerHiraganaVoiceBufferSize; index++)
                                {
                                    if(voiceBuffer[index]==0x7c || voiceBuffer[index]==0)
                                    {
                                        phonetic = [NSString stringWithCharacters:voiceBuffer length:index];
                                        
                                        break;
                                    }
                                }
                            }
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            free(voiceBuffer);
                        }
                        
                        break;
                    }
                    case PPIndexingStyle_Hangul:
                    {
                        UniChar *voiceBuffer = (UniChar *)malloc(codeBufferLength);
                        
                        if(voiceBuffer!=NULL)
                        {
                            memset(voiceBuffer, 0, codeBufferLength);
                            
                            [string getCharacters:codeBuffer];
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            for(int index=0; index<stringLength; index++)
                            {
                                UniChar currentChar = KSFindVoice(codeBuffer[index], 1);
                                if (currentChar==0xffff)
                                {
                                    voiceBuffer[index] = codeBuffer[index];
                                }
                                else
                                {
                                    voiceBuffer[index] = currentChar;
                                }
                            }
                            
                            phonetic = [NSString stringWithCharacters:voiceBuffer length:stringLength];
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            free(voiceBuffer);
                        }
                        
                        break;
                    }
                    default:
                    {
                        break;
                    }
                }
                
                free(codeBuffer);
            }
            
        }while(0);
        
        return phonetic;
    }
}


//==============================================================================
//
//==============================================================================
+ (NSArray <NSString *>*)phoneticsOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    @synchronized (self)
    {
        NSMutableArray *phonetics = nil;
        
        do
        {
            if(string==nil)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSUInteger stringLength = [string length];
            if(stringLength==0)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            if([NSString isCJKCharacter:[string characterAtIndex:0]]==NO)
            {
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSUInteger  codeBufferLength    = (stringLength+1)*sizeof(UniChar);
            UniChar     *codeBuffer         = (UniChar *)malloc(codeBufferLength);
            
            if(codeBuffer!=NULL)
            {
                memset(codeBuffer, 0, codeBufferLength);
                
                ////////////////////////////////////////////////////////////////////////////////////////////////////
                
                switch(style)
                {
                    case PPIndexingStyle_Hiragana:
                    {
                        UniChar *voiceBuffer = (UniChar *)malloc(PPIndexingControllerHiraganaVoiceBufferSize);
                        
                        if(voiceBuffer!=NULL)
                        {
                            memset(voiceBuffer, 0, PPIndexingControllerHiraganaVoiceBufferSize);
                            
                            [string getCharacters:codeBuffer];
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            NSString *path = [[NSBundle mainBundle] resourcePath];
                            
                            if(JPFindVoice([path cStringUsingEncoding:NSASCIIStringEncoding], codeBuffer, voiceBuffer)>0)
                            {
                                int index = 0;
                                for(index=0; index<PPIndexingControllerHiraganaVoiceBufferSize; index++)
                                {
                                    if(voiceBuffer[index] == 0)
                                        break;
                                }
                                
                                if (index>0)
                                {
                                    phonetics = [NSMutableArray array];
                                    
                                    NSString *phonetic = [[NSString alloc] initWithCharacters:voiceBuffer length:index];
                                    
                                    if([phonetic length])
                                    {
                                        NSArray *tempArray = [phonetic componentsSeparatedByString:@"|"];
                                        
                                        for(NSString *onePhonetic in tempArray)
                                        {
                                            if([onePhonetic length] && ![onePhonetic isEqualToString:string])
                                                [phonetics addObject:onePhonetic];
                                        }
                                    }
                                    [phonetic release];
                                }
                            }
                            
                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                            
                            free(voiceBuffer);
                        }
                        
                        break;
                    }
                        //                    case PPIndexingStyle_Hangul:
                        //                    {
                        //                        UniChar *voiceBuffer = (UniChar *)malloc(codeBufferLength);
                        //
                        //                        if(voiceBuffer!=NULL)
                        //                        {
                        //                            memset(voiceBuffer, 0, codeBufferLength);
                        //
                        //                            [string getCharacters:codeBuffer];
                        //
                        //                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                        //
                        //                            for(int index=0; index<stringLength; index++)
                        //                            {
                        //                                UniChar currentChar = KSFindVoice(codeBuffer[index], 1);
                        //                                if (currentChar==0xffff)
                        //                                {
                        //                                    voiceBuffer[index] = codeBuffer[index];
                        //                                }
                        //                                else
                        //                                {
                        //                                    voiceBuffer[index] = currentChar;
                        //                                }
                        //                            }
                        //
                        //                            phonetic = [NSString stringWithCharacters:voiceBuffer length:stringLength];
                        //
                        //                            ////////////////////////////////////////////////////////////////////////////////////////////////////
                        //
                        //                            free(voiceBuffer);
                        //                        }
                        //
                        //                        break;
                        //                    }
                    default:
                    {
                        break;
                    }
                }
                
                free(codeBuffer);
            }
            
        }while(0);
        
        if ([phonetics count]>0)
        {
            return [NSArray arrayWithArray:phonetics];
        }
        else
        {
            return nil;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods


//==============================================================================
//
//==============================================================================
+ (instancetype)shareInstance
{
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        staticIndexingController = [[PPIndexingController alloc] init];
    });
    
    return staticIndexingController;
}


//==============================================================================
//
//==============================================================================
+ (NSArray *)indexsForStyle:(PPIndexingStyle)style
{
    return [[PPIndexingController shareInstance] indexsForStyle:style];
}

//==============================================================================
//
//==============================================================================
+ (unsigned short)indexAliasOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    return [[PPIndexingController shareInstance] indexAliasOfString:string forStyle:style];
}


//==============================================================================
//
//==============================================================================
+ (NSString *)indexOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    return [[PPIndexingController shareInstance] indexOfString:string forStyle:style];
}


//==============================================================================
//
//==============================================================================
+ (NSString *)phoneticOfString:(NSString *)string forStyle:(PPIndexingStyle)style
{
    return [[PPIndexingController shareInstance] phoneticOfString:string forStyle:style];
}
@end
