//
//  PPImageBrowseViewController.m
//
//  NOTE:
//  ------------------------------
//  1. 縮放中心點是依照touch位置決定，不是在image中央。
//  2. 旋轉影像利用imageOrientation，不要建立一個新的image。
//  3. UIScrollView雙指縮放時本身就會處理縮放中心，所以只有點擊縮放時需要特別處理。
//

#import "PPImageBrowseViewController.h"
#import "UIImage+Additions.h"
#import "math.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPImageBrowseViewController () <UIScrollViewDelegate>

@property (nonatomic, retain) UIImageView *imageView;
@property (nonatomic, retain) UIScrollView *browseView;
@property (nonatomic, retain) NSMutableArray *browseViewlayoutConstraints;

@property (nonatomic, assign) NSInteger rotateDegree;
@property (nonatomic, assign) BOOL isAnimating;
@property (nonatomic, assign) CGPoint zoomCenterInView;
@property (nonatomic, assign) CGPoint zoomCenterInImageView;

@end


////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPImageBrowseViewController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//==============================================================================
//
//==============================================================================
+ (void)initialize
{
    // do nothing, 元件預設值看起來還是在init做，在initialize做的話，如果子類別沒有實作initialize就會呼叫父類別的initialize造成，appearance重覆設定
}


//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        [[PPImageBrowseViewController appearance] applyInvocationTo:self];
        
        //////////////////////////////////////////////////

        if(self.browseViewBackgroundColor==nil)
        {
            self.browseViewBackgroundColor = [UIColor blackColor];
        }
        
        //////////////////////////////////////////////////

        self.numberOfTouchesForMoving = 1;
        self.doubleTapZoomingEnabled = YES;
        
        //////////////////////////////////////////////////
        
        _browseView = [[UIScrollView alloc] init];
        
        if(self.browseView != nil)
        {
            self.browseView.backgroundColor = self.browseViewBackgroundColor;
            self.browseView.translatesAutoresizingMaskIntoConstraints = NO;
            self.browseView.delegate = self;
            [self.view addSubview:self.browseView];
            
            //////////////////////////////////////////////////
            
            _imageView = [[UIImageView alloc] init];
            
            if(self.imageView)
            {
                self.imageView.contentMode = UIViewContentModeScaleAspectFit;
                [self.browseView addSubview:self.imageView];
            }
            
            //////////////////////////////////////////////////

            _doubleTapGestureRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(onDoubleTapGesture:)];
            
            if(self.doubleTapGestureRecognizer)
            {
                self.doubleTapGestureRecognizer.numberOfTapsRequired = 2;
                self.doubleTapGestureRecognizer.enabled = self.doubleTapZoomingEnabled;
                [self.browseView addGestureRecognizer:self.doubleTapGestureRecognizer];
            }
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [self.browseView removeGestureRecognizer:self.doubleTapGestureRecognizer];
    self.doubleTapGestureRecognizer = nil;
    
    self.image = nil;
    self.imageView.image = nil; // iOS8要自行將image設定為nil，否則記憶體會無法釋放。
    self.imageView = nil;
    self.browseView = nil;
    self.browseViewlayoutConstraints = nil;
    
    self.browseViewBackgroundColor = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override methods

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    [self assignBrowseViewLayuotConstraints];
}


//================================================================================
// 在這裡才能取得browseView的gestureRecognizer
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////

    // 設定移動需要的touches
    self.browseView.panGestureRecognizer.minimumNumberOfTouches = self.numberOfTouchesForMoving;
    self.browseView.panGestureRecognizer.maximumNumberOfTouches = self.numberOfTouchesForMoving;
}


//================================================================================
// 在這裡才能取得browseView大小
//================================================================================
- (void)viewDidLayoutSubviews
{
    [super viewDidLayoutSubviews];
    
    //////////////////////////////////////////////////
    // assign image
    
    [self updateImageContent];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Property methods

//================================================================================
//
//================================================================================
- (void)setImage:(UIImage *)image
{
    if(_image != image)
    {
        [_image release];
        _image = [image retain];
        self.imageView.image = image;

        switch (_image.imageOrientation)
        {
            case UIImageOrientationRight: self.rotateDegree = 90 ; break;
            case UIImageOrientationDown: self.rotateDegree = 180 ; break;
            case UIImageOrientationLeft: self.rotateDegree = 270 ; break;
            default: self.rotateDegree = 0; break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setDoubleTapZoomingEnabled:(BOOL)doubleTapZoomingEnabled
{
    if(_doubleTapZoomingEnabled != doubleTapZoomingEnabled)
    {
        _doubleTapZoomingEnabled = doubleTapZoomingEnabled;
        self.doubleTapGestureRecognizer.enabled = doubleTapZoomingEnabled;
    }
}


//================================================================================
//
//================================================================================
- (void)setBrowseViewBackgroundColor:(UIColor *)browseViewBackgroundColor
{
    [browseViewBackgroundColor retain];
    [_browseViewBackgroundColor release];
    _browseViewBackgroundColor = browseViewBackgroundColor;
    
    //////////////////////////////////////////////////
    if (self.browseView!=nil)
    {
        self.browseView.backgroundColor = self.browseViewBackgroundColor;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIScrollViewDelegate methods

//================================================================================
//
//================================================================================
- (UIView *)viewForZoomingInScrollView:(UIScrollView *)scrollView
{
    return self.imageView;
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidZoom:(UIScrollView *)scrollView
{
    [self updateScrollContent];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Action methods

//================================================================================
//
//================================================================================
- (void)onDoubleTapGesture:(UIGestureRecognizer *)gestureRecognizer
{
//    CGPoint contentOffset = self.scrollView.contentOffset;
//    UIEdgeInsets contentInset = self.scrollView.contentInset;
//    CGSize contentSize = self.scrollView.contentSize;
    
//    NSLog(@"contentOffset %@", NSStringFromCGPoint(contentOffset));
//    NSLog(@"contentInset %@", NSStringFromUIEdgeInsets(contentInset));
//    NSLog(@"contentSize %@", NSStringFromCGSize(contentSize));

    //////////////////////////////////////////////////
    // 檢查點擊有效範圍
    
    CGPoint pointInImageView = [gestureRecognizer locationInView:self.imageView];

    // pointInImageView的座標是對應到原始影像大小，不是目前縮放過的大小。
    // imageView的frame是目前縮放過的大小，所以要自己產生檢查範圍。
    CGRect imageFrame = CGRectMake(0, 0, self.imageView.image.size.width, self.imageView.image.size.height);
    CGFloat newScale = 1.0;
    
    if(CGRectContainsPoint(imageFrame, pointInImageView) == true)
    {
        if(self.browseView.zoomScale > self.browseView.minimumZoomScale)
        {
            newScale = self.browseView.minimumZoomScale;
        }
        else
        {
            newScale = self.browseView.maximumZoomScale;
        }
        
        [self prepareDataBeforZoomingWithCenterPoint:[gestureRecognizer locationInView:self.view]];
        
        [self.browseView setZoomScale:newScale animated:YES];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPAppearance 

//==============================================================================
//
//==============================================================================
+ (instancetype)appearance
{
    return [PPAppearance appearanceForClass:[self class]];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (void)prepareDataBeforZoomingWithCenterPoint:(CGPoint)centerPoint
{
    self.zoomCenterInView = centerPoint;
    self.zoomCenterInImageView = [self.view convertPoint:centerPoint toView:self.imageView];
    
//    NSLog(@"self.zoomCenterInImageView %@", NSStringFromCGPoint(self.zoomCenterInImageView));
//    NSLog(@"self.zoomCenterInView %@", NSStringFromCGPoint(self.zoomCenterInView));
}


//================================================================================
// (contentOffset 左+ 右- 上+ 下-)
// (contentInset  左- 右+ 上- 下+)
//================================================================================
- (void)updateScrollContent
{
    CGSize scrollViewSize = self.browseView.bounds.size;
    CGSize contentSize = self.browseView.contentSize;
    UIEdgeInsets contentInset = UIEdgeInsetsZero;
    CGPoint contentOffset = CGPointZero;

//    NSLog(@"contentInset %@", NSStringFromUIEdgeInsets(contentInset));
//    NSLog(@"contentSize %@", NSStringFromCGSize(contentSize));
//    NSLog(@"scrollViewSize %@", NSStringFromCGSize(scrollViewSize));

    
    //////////////////////////////////////////////////
    // 影像比顯示區域小時，設定contentInset。
    
    if(contentSize.width < scrollViewSize.width)
    {
        contentInset.left = (scrollViewSize.width-contentSize.width)/2;
        contentInset.right = contentInset.left;
    }
    
    if(contentSize.height < scrollViewSize.height)
    {
        contentInset.top = (scrollViewSize.height-contentSize.height)/2;
        contentInset.bottom = contentInset.top;
    }
    
    self.browseView.contentInset = contentInset;

    // !! 設定inset後，offset也會變更，所以要重新取值。
    contentOffset = self.browseView.contentOffset;
    
    
    //////////////////////////////////////////////////
    // 觸碰點為中心的縮放效果
    
    if(CGPointEqualToPoint(self.zoomCenterInImageView, CGPointZero) == false)
    {
        // 移動contentOffset，讓縮放後的viewZoomCenter移回原位
        CGPoint curPointOfZoomCenterInImageView = [self.imageView convertPoint:self.zoomCenterInImageView toView:self.view];
        CGPoint diff = CGPointMake(curPointOfZoomCenterInImageView.x-self.zoomCenterInView.x,
                                   curPointOfZoomCenterInImageView.y-self.zoomCenterInView.y);
        
        
        contentOffset.x += diff.x;
        contentOffset.y += diff.y;
        
        // !! 設定完要清除，避免影像後續的行為
        self.zoomCenterInImageView = CGPointZero;
        self.zoomCenterInView = CGPointZero;
    }
    

    //////////////////////////////////////////////////
    // 修正contentOffset讓影像貼齊邊緣
    if(-contentOffset.x > contentInset.left)
    {
        contentOffset.x = -contentInset.left;
    }
    else if(-contentOffset.x + contentSize.width < scrollViewSize.width-contentInset.right)
    {
        contentOffset.x = contentSize.width - scrollViewSize.width + contentInset.right;
    }
    
    
    if(-contentOffset.y > contentInset.top)
    {
        contentOffset.y = -contentInset.top;
    }
    else if(-contentOffset.y + contentSize.height < scrollViewSize.height-contentInset.bottom)
    {
        contentOffset.y = contentSize.height - scrollViewSize.height + contentInset.bottom;
    }
    
    //////////////////////////////////////////////////
    
    self.browseView.contentOffset = contentOffset;
}


//================================================================================
//
//================================================================================
- (void)updateImageContent
{
    //////////////////////////////////////////////////
    // 設定影像及大小
    
    self.imageView.image = self.image;
    self.imageView.transform = CGAffineTransformIdentity;
    self.imageView.frame = CGRectMake(0, 0, self.image.size.width, self.image.size.height);
    self.browseView.contentSize = self.image.size;
    
    self.zoomCenterInImageView = CGPointZero;
    self.zoomCenterInView = CGPointZero;

    
    //////////////////////////////////////////////////
    // 設定縮放比

    CGRect scrollViewFrame = self.browseView.frame;
    CGFloat scaleWidth = scrollViewFrame.size.width / self.browseView.contentSize.width;
    CGFloat scaleHeight = scrollViewFrame.size.height / self.browseView.contentSize.height;
    CGFloat minScale = MIN(scaleWidth, scaleHeight);
    
    self.browseView.minimumZoomScale = minScale;
    self.browseView.maximumZoomScale = 1.0f;
    self.browseView.zoomScale = minScale;
    
    //////////////////////////////////////////////////
    
    [self updateScrollContent];
}


//================================================================================
//
//================================================================================
- (void)updateRotateContentWithOrientation:(PPImageBrowseViewController_Orientation)orientation
{
    switch (orientation)
    {
        case PPImageBrowseViewController_Orientation_Right:
            self.rotateDegree = (self.rotateDegree+90)%360;
            break;
            
        case PPImageBrowseViewController_Orientation_Left:
            self.rotateDegree = (self.rotateDegree+270)%360;
            break;
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    UIImageOrientation imageOrientation;
    
    switch (self.rotateDegree)
    {
        case 90: imageOrientation = UIImageOrientationRight; break;
        case 180: imageOrientation = UIImageOrientationDown; break;
        case 270: imageOrientation = UIImageOrientationLeft; break;
        default: imageOrientation = UIImageOrientationUp; break;
    }
    
    self.image = [UIImage imageWithCGImage:self.image.CGImage
                                     scale:1.0
                               orientation:imageOrientation];
    
    //////////////////////////////////////////////////
    
    [self updateImageContent];
    
}


//================================================================================
//
//================================================================================
- (CGFloat)scaleOfView:(UIView *)view
{
    return sqrt(view.transform.a*view.transform.a + view.transform.c*view.transform.c);
}


//================================================================================
//
//================================================================================
- (void)rotateWithOrientation:(PPImageBrowseViewController_Orientation)orientation
{
    [self.browseView setZoomScale:self.browseView.minimumZoomScale animated:NO];
    
    //////////////////////////////////////////////////
    // handle transform
    
    CGFloat angle = 0;
    
    switch (orientation)
    {
        case PPImageBrowseViewController_Orientation_Right:
            angle = M_PI/2;
            break;
            
        case PPImageBrowseViewController_Orientation_Left:
            angle = M_PI*3/2;
            break;
            
        default:
            break;
    }
    
    CGAffineTransform transform = CGAffineTransformMakeRotation(angle);
    
    // scale
    CGFloat scale = MIN(self.browseView.bounds.size.width/self.imageView.bounds.size.height,
                        self.browseView.bounds.size.height/self.imageView.bounds.size.width);
    
    transform = CGAffineTransformConcat(transform, CGAffineTransformMakeScale(scale, scale));
    self.imageView.transform = transform;
    
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (BOOL)animateRotateImageViewWithOrientation:(PPImageBrowseViewController_Orientation)orientation
{
    //    NSLog(@"contentSize %@", NSStringFromCGSize(self.cardImageScrollView.contentSize));
    //    NSLog(@"imageViewSize %@", NSStringFromCGSize(self.imageView.bounds.size));
    
    if(self.isAnimating == YES)
    {
        return NO;
    }
    
    self.isAnimating = YES;
    
    __block typeof(self) blockSelf = self;
    
    [UIView animateWithDuration:0.3 animations:^{
                         
        [blockSelf rotateWithOrientation:orientation];
        
    } completion:^(BOOL finished) {
                         
        [blockSelf updateRotateContentWithOrientation:orientation];
        blockSelf.isAnimating = NO;
    }];

    return YES;
}


//================================================================================
//
//================================================================================
- (UIImage *)imageWithApplyRotation
{
    UIImage *image;
    
    if(self.image.imageOrientation != UIImageOrientationUp)
    {
        image = [self.image imageByAdjustOrientationWithMaxLength:0];
    }
    else
    {
        image = self.image;
    }
    
//    // test
//    NSData *data = UIImageJPEGRepresentation(image, 0.5);
//    NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
//    NSString *path = [NSString stringWithFormat:@"%@/test.jpg", [paths objectAtIndex:0]];
//    
//    [data writeToFile:path atomically:YES];
//    NSLog(@"%@", path);

    return image;
}


//================================================================================
//
//================================================================================
- (void)assignBrowseViewLayuotConstraints
{
    if (self.browseViewlayoutConstraints)
    {
        [self.view removeConstraints:self.browseViewlayoutConstraints];
        self.browseViewlayoutConstraints = nil;
    }
    
    //////////////////////////////////////////////////
    // LayoutConstraints
    
    NSDictionary *views = @{@"browseView":self.browseView};
    
    NSDictionary *metrics = @{@"topInset": @(self.browseAreaInset.top),
                              @"bottomInset": @(self.browseAreaInset.bottom),
                              @"leftInset": @(self.browseAreaInset.left),
                              @"rightInset": @(self.browseAreaInset.right)};
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|-(leftInset)-[browseView]-(rightInset)-|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(topInset)-[browseView]-(bottomInset)-|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    if ([layoutConstraints count] > 0)
    {
        [self.view addConstraints:layoutConstraints];
        self.browseViewlayoutConstraints = layoutConstraints;
    }
}

@end
