//
//  PPGuideView.m
//
//  Created by eddie on 2015/4/2.
//  Copyright (c) 2015年 eddie. All rights reserved.
//

#import "PPGuideView.h"
#import "PPGuideMaskView.h"
#import "PPGuideLabel.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPGuideView ()
@property (nonatomic, assign) id<PPGuideViewDelegate> delegate;
@property (nonatomic, retain) PPGuideAttributeModel* guideAttribute;
@property (nonatomic, retain) PPGuideDataModel* guideData;
@property (nonatomic, retain) PPGuideMaskView *privMaskView;
@property (nonatomic, retain) PPGuideLabel *titleLabel;
@property (nonatomic, retain) PPGuideLabel *messageLabel;
@property (nonatomic, retain) UIButton *confirmButton;
@property (nonatomic, retain) UIButton *customViewConfirmButton;
@property (nonatomic, retain) UITapGestureRecognizer *tapGestureRecognizer;
@property (nonatomic, assign) NSTimeInterval timestamoOfSendDelegate;
@property (nonatomic, retain) NSArray *guideViewConstraints;
@property (nonatomic, retain) UIView *infoAreaView;
@property (nonatomic, retain) NSArray *infoAreaViewConstraints;
@property (nonatomic, retain) NSArray *infoAreaSubviewConstraints;

//@property (nonatomic, retain) UIView *testView;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPGuideView

////////////////////////////////////////////////////////////////////////////////////////////////////

// 較窄範圍的infoArea最大寬度
static const NSInteger PPGuideView_MaxInfoAreaWidthInNarrowBounds = 250;

// 較寬範圍的infoArea最大寬度
static const NSInteger PPGuideView_MaxInfoAreaWidthInWideBounds = 400;

// InfoArea與螢幕邊界的最小距離
static const NSInteger PPGuideView_InfoAreaMinMargin = 10.0;

// InfoArea中物件之間的最小距離
static const NSInteger PPGuideView_InfoAreaMinGap = 5.0;

// 調整字型時的最大值
static const CGFloat PPGuideView_MaxAdjustFontSize = 25.0;

// 調整字型時的最小值
static const CGFloat PPGuideView_MinAdjustFontSize = 5.0;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods


//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame delegate:(id<PPGuideViewDelegate>)delegate;
{
    if(self = [super initWithFrame:frame])
    {
        self.delegate = delegate;
        self.confirmButton = nil;
        self.timestamoOfSendDelegate = 0;
        self.backgroundColor = [UIColor clearColor];
        self.autoresizesSubviews = YES;
        self.userInteractionEnabled = YES;
        
        //////////////////////////////////////////////////
        // !! 只有maskView使用AutoLayout

        _privMaskView = [[PPGuideMaskView alloc] init];
        
        if(self.privMaskView)
        {
            self.privMaskView.translatesAutoresizingMaskIntoConstraints = YES;
            self.privMaskView.backgroundColor = [UIColor clearColor];
            self.privMaskView.contentMode = UIViewContentModeRedraw;
            self.privMaskView.autoresizingMask = UIViewAutoresizingFlexibleWidth|UIViewAutoresizingFlexibleHeight;
            
            [self addSubview:self.privMaskView];
        }

        //////////////////////////////////////////////////
        
        _infoAreaView = [[UIView alloc] init];
        
        if(self.infoAreaView)
        {
            self.infoAreaView.translatesAutoresizingMaskIntoConstraints = NO;
            [self addSubview:self.infoAreaView];
        }
        
        //////////////////////////////////////////////////
        
        _titleLabel = [[PPGuideLabel alloc] init];
        
        if(self.titleLabel)
        {
            self.titleLabel.alignment = PPGuideLabelTextAlignmentBottom;
            self.titleLabel.translatesAutoresizingMaskIntoConstraints = NO;
            self.titleLabel.backgroundColor = [UIColor clearColor];
            self.titleLabel.contentMode = UIViewContentModeScaleAspectFit;
            self.titleLabel.autoresizingMask = UIViewAutoresizingNone;
            self.titleLabel.numberOfLines = 1;
            self.titleLabel.adjustsFontSizeToFitWidth = YES;
            self.titleLabel.lineBreakMode = NSLineBreakByTruncatingTail;
            
            [self.infoAreaView addSubview:self.titleLabel];
        }
        
        //////////////////////////////////////////////////

        _messageLabel = [[PPGuideLabel alloc] init];
        
        if(self.messageLabel)
        {
            self.messageLabel.alignment = PPGuideLabelTextAlignmentTop;
            self.messageLabel.translatesAutoresizingMaskIntoConstraints = NO;
            self.messageLabel.backgroundColor = [UIColor clearColor];
            self.messageLabel.contentMode = UIViewContentModeScaleAspectFit;
            self.messageLabel.autoresizingMask = UIViewAutoresizingNone;
            self.messageLabel.numberOfLines = 0;
            self.messageLabel.lineBreakMode = NSLineBreakByWordWrapping;
            
            [self.infoAreaView addSubview:self.messageLabel];
        }
        
        //////////////////////////////////////////////////
        
        _tapGestureRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(onTapAction:)];
        
        if(self.tapGestureRecognizer)
        {
            self.tapGestureRecognizer.numberOfTapsRequired = 1;
            self.tapGestureRecognizer.numberOfTouchesRequired = 1;
            [self addGestureRecognizer:self.tapGestureRecognizer];
        }
        
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [self.infoAreaView removeFromSuperview];
    self.infoAreaView = nil;
    self.infoAreaViewConstraints = nil;
    self.infoAreaSubviewConstraints = nil;
    
    if([self.guideViewConstraints count] > 0)
    {
        [self.superview removeConstraints:self.guideViewConstraints];
    }
    
    self.guideViewConstraints = nil;
    
    [self.privMaskView removeFromSuperview];
    self.privMaskView = nil;
    
    [self.messageLabel removeFromSuperview];
    self.messageLabel = nil;

    [self.titleLabel removeFromSuperview];
    self.titleLabel = nil;
    
    if(self.confirmButton != nil)
    {
        // !! 這樣移不乾淨
//        [self.confirmButton removeTarget:self action:@selector(onConfirmAction:) forControlEvents:UIControlEventTouchUpInside];
        
        [self.confirmButton removeTarget:nil action:nil forControlEvents:UIControlEventTouchUpInside];
        [self.confirmButton removeFromSuperview];
        self.confirmButton = nil;
    }
    
    if(self.tapGestureRecognizer)
    {
        [self removeGestureRecognizer:self.tapGestureRecognizer];
        self.tapGestureRecognizer = nil;
    }
    
    if(self.customViewConfirmButton != nil)
    {
        [self.customViewConfirmButton removeTarget:nil action:NULL forControlEvents:UIControlEventTouchUpInside];
        self.customViewConfirmButton = nil;
    }

    self.delegate = nil;
    self.guideData = nil;
    self.guideAttribute = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Action methods

//================================================================================
//
//================================================================================
- (void)onTapAction:(UITapGestureRecognizer *)recognizer
{
    // !! confirm button存在時tap無效。
    if(self.confirmButton == nil)
    {
        CGPoint point = [recognizer locationInView:self];
        
        if([self.delegate respondsToSelector:@selector(guideView:didTapWithPoint:)])
        {
            [self.delegate guideView:self didTapWithPoint:point];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)onConfirmAction:(id)sender
{
    // MARK: PPGuideFocusStyle_View關閉處理
    if(self.guideData.guideFocusModel.style == PPGuideFocusStyle_View)
    {
        if(self.customViewConfirmButton != nil)
        {
            [self.customViewConfirmButton removeTarget:nil action:NULL forControlEvents:UIControlEventTouchUpInside];
            self.customViewConfirmButton = nil;
        }

        [self.guideData.guideFocusModel.view removeFromSuperview];
    }
    
    //////////////////////////////////////////////////

    if([self.delegate respondsToSelector:@selector(guideView:didTapWithPoint:)])
    {
        [self.delegate guideView:self didTapWithPoint:CGPointZero];
    }
    
    if([self.delegate respondsToSelector:@selector(guideView:didClickConfirmButton:)])
    {
        [self.delegate guideView:self didClickConfirmButton:sender];
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Overwrite methods

//================================================================================
// 
//================================================================================
- (UIView *)hitTest:(CGPoint)point withEvent:(UIEvent *)event
{
//    NSLog(@"%s %@", __func__, event);
    
    //////////////////////////////////////////////////
    
    if([self.delegate respondsToSelector:@selector(guideView:canTouchThroughWithPoint:)])
    {
        if([self.delegate guideView:self canTouchThroughWithPoint:point] == YES)
        {
//            NSLog(@"%@", [self.guideData.targetView hitTest:point withEvent:event]);
            
            // !! 因為會收到連續多個hitTest，但delegate不能一直發，會影響流程，所以用時間過濾。
            // !! For touch through mode
            if(event.timestamp - self.timestamoOfSendDelegate > 0.5)
            {
                self.timestamoOfSendDelegate = event.timestamp;

                __block PPGuideView *blockSelf = self;
                __block CGPoint blockPoint = point;

                // 不要卡在這裡
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    if([blockSelf.delegate respondsToSelector:@selector(guideView:didTapWithPoint:)])
                    {
                        [blockSelf.delegate guideView:blockSelf didTapWithPoint:blockPoint];
                    }
                });
            }
            
            // Pass event to view under guideView
            return nil;
        }
    }

    // Do not pass event to view under guideView
    return [super hitTest:point withEvent:event];
}


//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
//    NSLog(@"%s %@", __func__, NSStringFromCGRect(self.frame));

    [super layoutSubviews];
}


//================================================================================
//
//================================================================================
- (void)updateConstraints
{
//    NSLog(@"%s %@", __func__, NSStringFromCGRect(self.frame));
    
    return [super updateConstraints];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (CGSize)sizeWithText:(NSString *)text font:(UIFont *)font maxWidth:(CGFloat)maxWidth
{
    if([text length] == 0 || font == nil)
    {
        return CGSizeZero;
    }
    
    NSDictionary *attributes = @{NSFontAttributeName:font};
    NSAttributedString *attributedText = [[NSAttributedString alloc] initWithString:text
                                                                         attributes:attributes];
    
    CGRect rect = [attributedText boundingRectWithSize:CGSizeMake(maxWidth, 999)
                                               options:NSStringDrawingUsesLineFragmentOrigin
                                               context:nil];
    
    [attributedText release];
    
    // MARK: 必須無條件進位，否則label會無法換行
    rect.size.width = ceilf(rect.size.width);
    rect.size.height = ceilf(rect.size.height);
    
    return rect.size;
}


//================================================================================
//
//================================================================================
- (void)adjustFontSizeWithLabel:(UILabel *)label maxSize:(CGSize)maxSize
{
    for (CGFloat i = PPGuideView_MaxAdjustFontSize; i>PPGuideView_MinAdjustFontSize; i--)
    {
        UIFont *newFont = [UIFont fontWithName:label.font.fontName size:i];
        CGSize newSize = [self sizeWithText:label.text font:newFont maxWidth:maxSize.width];
        
        if (newSize.height <= maxSize.height)
        {
            label.font = newFont;
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setupInfoAreaSubviewConstraintsWithInfoAreaSize:(CGSize)infoAreaSize
                                         titleLabelSize:(CGSize)titleLabelSize
                                       messageLabelSize:(CGSize)messageLabelSize
                                      confirmButtonSize:(CGSize)confirmButtonSize
{
    NSString *format;
    NSMutableArray *constraints = [NSMutableArray array];
    NSDictionary *views;
    NSDictionary *metrics = @{@"titleLabelHeight":@(titleLabelSize.height),
                              @"messageLabelHeight":@(messageLabelSize.height),
                              @"confirmButtonWidth":@(confirmButtonSize.width),
                              @"confirmButtonHeight":@(confirmButtonSize.height),
                              @"gap":@(PPGuideView_InfoAreaMinGap)};
    
    if(self.confirmButton != nil)
    {
        views = @{@"titleLabel":self.titleLabel,
                  @"messageLabel":self.messageLabel,
                  @"confirmButton":self.confirmButton};
    }
    else
    {
        views = @{@"titleLabel":self.titleLabel,
                  @"messageLabel":self.messageLabel};
    }
    
    //NSLog(@"metrics %@", metrics);
    
    format = @"H:|[titleLabel]|";
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                             options:NSLayoutFormatDirectionLeftToRight
                                                                             metrics:metrics
                                                                               views:views]];
    
    format = @"H:|[messageLabel]|";
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                             options:NSLayoutFormatDirectionLeftToRight
                                                                             metrics:metrics
                                                                               views:views]];
    
    if(self.confirmButton != nil)
    {
        format = @"H:[confirmButton(confirmButtonWidth)]|";
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                                 options:NSLayoutFormatDirectionLeftToRight
                                                                                 metrics:metrics
                                                                                   views:views]];
    }
    
    //////////////////////////////////////////////////
    
    NSMutableString *vertFormat = [NSMutableString stringWithString:@"V:|"];

    // title
    if([self.titleLabel.text length] > 0)
    {
        [vertFormat appendString:@"[titleLabel(titleLabelHeight)]-(gap)-"];
    }

    // message
    [vertFormat appendString:@"[messageLabel(messageLabelHeight)]"];

    // confirm
    if(self.confirmButton != nil)
    {
        [vertFormat appendString:@"-(gap)-[confirmButton(confirmButtonHeight)]"];
    }
    
    [vertFormat appendString:@"|"];

    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:vertFormat
                                                                             options:NSLayoutFormatDirectionLeadingToTrailing
                                                                             metrics:metrics
                                                                               views:views]];
    //////////////////////////////////////////////////
    
    if(self.infoAreaSubviewConstraints != nil)
    {
        [self.infoAreaView removeConstraints:self.infoAreaSubviewConstraints];
        self.infoAreaSubviewConstraints = nil;
    }
    
    [self.infoAreaView addConstraints:constraints];
    self.infoAreaSubviewConstraints = constraints;
}


//================================================================================
//
//================================================================================
- (void)setupInfoAreaViewConstraintsWithFocusFrame:(CGRect)focusFrame
                                            bounds:(CGRect)bounds
                                       infoAreaGap:(CGFloat)infoAreaGap
                                   layoutDirection:(PPGuideDataModel_LayoutDirection)layoutDirection
{
//    NSLog(@"bounds:%@", NSStringFromCGRect(bounds));
    
    //////////////////////////////////////////////////
    // 依據focusFrame位置計算InfoArea位置
    
    CGFloat maxInfoAreaWidth;
    CGFloat infoAreaTopMargin;
    CGFloat infoAreaBottomMargin;
    CGFloat infoAreaLeftMargin;
    CGFloat infoAreaRightMargin;
    
    CGSize infoAreaSize;
    CGSize titleLabelSize;
    CGSize messageLabelSize;
    CGSize confirmButtonSize = self.confirmButton ? self.confirmButton.bounds.size:CGSizeZero;
    
    
    // 最大可用寬度
    if(bounds.size.width < PPGuideView_MaxInfoAreaWidthInWideBounds+2*PPGuideView_InfoAreaMinMargin)
    {
        maxInfoAreaWidth = PPGuideView_MaxInfoAreaWidthInNarrowBounds;
    }
    else
    {
        maxInfoAreaWidth = PPGuideView_MaxInfoAreaWidthInWideBounds;
    }
    
    // 顯示資訊放在目標的水平方向
    if(layoutDirection == PPGuideDataModel_LayoutDirection_Horz)
    {
        CGFloat leftSideSpace = CGRectGetMinX(focusFrame);
        CGFloat rightSideSpace = bounds.size.width - CGRectGetMaxX(focusFrame);
        
        // 寬度處理
        titleLabelSize = [self sizeWithText:self.titleLabel.text font:self.titleLabel.font maxWidth:maxInfoAreaWidth];
        messageLabelSize = [self sizeWithText:self.messageLabel.text font:self.messageLabel.font maxWidth:maxInfoAreaWidth];
        infoAreaSize.width = fmaxf(titleLabelSize.width, messageLabelSize.width);
        infoAreaSize.width = fmaxf(infoAreaSize.width, confirmButtonSize.width);
        
        if(leftSideSpace > rightSideSpace)
        {
            // 顯示資訊放在目標的左方
            leftSideSpace -= (2*PPGuideView_InfoAreaMinMargin);
            infoAreaSize.width = fminf(infoAreaSize.width, maxInfoAreaWidth);
            infoAreaSize.width = fminf(infoAreaSize.width, leftSideSpace);
            
            CGFloat maxGap = fmaxf(PPGuideView_InfoAreaMinMargin, infoAreaGap);
            
            infoAreaRightMargin = bounds.size.width - (CGRectGetMinX(focusFrame) - maxGap);
            infoAreaLeftMargin = bounds.size.width - infoAreaRightMargin - infoAreaSize.width;
        }
        else
        {
            // 顯示資訊放在目標的右方
            rightSideSpace -= (2*PPGuideView_InfoAreaMinMargin);
            infoAreaSize.width = fminf(infoAreaSize.width, maxInfoAreaWidth);
            infoAreaSize.width = fminf(infoAreaSize.width, rightSideSpace);

            CGFloat maxGap = fmaxf(PPGuideView_InfoAreaMinMargin, infoAreaGap);

            infoAreaLeftMargin = CGRectGetMaxX(focusFrame) + maxGap;
            infoAreaRightMargin = bounds.size.width - infoAreaLeftMargin - infoAreaSize.width;
        }

        // 高度處理
        titleLabelSize = [self sizeWithText:self.titleLabel.text font:self.titleLabel.font maxWidth:infoAreaSize.width];
        messageLabelSize = [self sizeWithText:self.messageLabel.text font:self.messageLabel.font maxWidth:infoAreaSize.width];
        infoAreaSize.height = 0.0;
        
        if([self.titleLabel.text length] > 0)
        {
            infoAreaSize.height += titleLabelSize.height;
            infoAreaSize.height += PPGuideView_InfoAreaMinGap;
        }
        
        infoAreaSize.height += messageLabelSize.height;
        
        if(self.confirmButton != nil)
        {
            infoAreaSize.height += PPGuideView_InfoAreaMinGap;
            infoAreaSize.height += confirmButtonSize.height;
        }
        
        infoAreaTopMargin = CGRectGetMidY(focusFrame) - infoAreaSize.height/2.0;
        infoAreaBottomMargin = bounds.size.height - infoAreaTopMargin - infoAreaSize.height;
    }
    else // PPGuideDataModel_LayoutDirection_Vert (顯示資訊放在目標的垂直方向)
    {
        CGFloat topSideSpace = CGRectGetMinY(focusFrame);
        CGFloat bottomSideSpace = bounds.size.height - CGRectGetMaxY(focusFrame);
        

        // 寬度處理
        titleLabelSize = [self sizeWithText:self.titleLabel.text font:self.titleLabel.font maxWidth:maxInfoAreaWidth];
        messageLabelSize = [self sizeWithText:self.messageLabel.text font:self.messageLabel.font maxWidth:maxInfoAreaWidth];
        infoAreaSize.width = fmaxf(titleLabelSize.width, messageLabelSize.width);
        infoAreaSize.width = fmaxf(infoAreaSize.width, confirmButtonSize.width);
        infoAreaLeftMargin = CGRectGetMidX(focusFrame) - infoAreaSize.width/2.0;
        infoAreaRightMargin = bounds.size.width - infoAreaLeftMargin - infoAreaSize.width;

        // 高度處理
        titleLabelSize = [self sizeWithText:self.titleLabel.text font:self.titleLabel.font maxWidth:infoAreaSize.width];
        messageLabelSize = [self sizeWithText:self.messageLabel.text font:self.messageLabel.font maxWidth:infoAreaSize.width];
        infoAreaSize.height = 0.0;
        
        if([self.titleLabel.text length] > 0)
        {
            infoAreaSize.height += titleLabelSize.height;
            infoAreaSize.height += PPGuideView_InfoAreaMinGap;
        }
        
        infoAreaSize.height += messageLabelSize.height;
        
        if(self.confirmButton != nil)
        {
            infoAreaSize.height += PPGuideView_InfoAreaMinGap;
            infoAreaSize.height += confirmButtonSize.height;
        }

        if(topSideSpace > bottomSideSpace)
        {
            // 顯示資訊放在目標的上方
            CGFloat maxGap = fmaxf(PPGuideView_InfoAreaMinMargin, infoAreaGap);

            infoAreaBottomMargin = bounds.size.height - CGRectGetMinY(focusFrame) + maxGap;
            infoAreaTopMargin = bounds.size.height - infoAreaBottomMargin - infoAreaSize.height;
        }
        else
        {
            // 顯示資訊放在目標的下方
            CGFloat maxGap = fmaxf(PPGuideView_InfoAreaMinMargin, infoAreaGap);

            infoAreaTopMargin = CGRectGetMaxY(focusFrame) + maxGap;
            infoAreaBottomMargin = bounds.size.height - infoAreaTopMargin - infoAreaSize.height;
        }
    }
    
    //////////////////////////////////////////////////
    // 左右調整不要超過畫面
    
    if(infoAreaRightMargin < PPGuideView_InfoAreaMinMargin)
    {
        infoAreaRightMargin = PPGuideView_InfoAreaMinMargin;
        infoAreaLeftMargin = bounds.size.width - infoAreaSize.width - infoAreaRightMargin;
    }
    
    if(infoAreaLeftMargin < PPGuideView_InfoAreaMinMargin)
    {
        infoAreaLeftMargin = PPGuideView_InfoAreaMinMargin;
        infoAreaRightMargin = bounds.size.width - infoAreaSize.width - infoAreaLeftMargin;
    }
    
    //////////////////////////////////////////////////
    // 上下調整不要超過畫面
    
    CGFloat minButtonMargin = PPGuideView_InfoAreaMinMargin;
    
    if(infoAreaBottomMargin < minButtonMargin)
    {
        infoAreaBottomMargin = minButtonMargin;
        infoAreaTopMargin = bounds.size.height - infoAreaSize.height - infoAreaBottomMargin;
    }
    
    if(infoAreaTopMargin < PPGuideView_InfoAreaMinMargin)
    {
        infoAreaTopMargin = PPGuideView_InfoAreaMinMargin;
        infoAreaBottomMargin = bounds.size.height - infoAreaSize.height - infoAreaTopMargin;
        
        // !! 如果調整後BottomMargin又小於PPGuideView_InfoAreaMinMargin，就要縮小infoArea的高度
        if(infoAreaBottomMargin < minButtonMargin)
        {
            // message 一旦大小有變動，就要調整fontSize避免切字
            CGFloat heightDiff = minButtonMargin-infoAreaBottomMargin;
            messageLabelSize.height -= heightDiff;
            [self adjustFontSizeWithLabel:self.messageLabel maxSize:messageLabelSize];
            
            // font變動後，重跑一次計算，位置才會正確。
            [self setupInfoAreaViewConstraintsWithFocusFrame:focusFrame
                                                      bounds:bounds
                                                 infoAreaGap:infoAreaGap
                                             layoutDirection:layoutDirection];
            
            return;
        }
    }
    
    //////////////////////////////////////////////////
    // 建立constraints

    NSString *format;
    NSMutableArray *constraints = [NSMutableArray array];
    NSDictionary *views = @{@"infoAreaView":self.infoAreaView};
    NSDictionary *metrics = @{@"infoAreaTopMargin":@(infoAreaTopMargin),
                              @"infoAreaBottomMargin":@(infoAreaBottomMargin),
                              @"infoAreaLeftMargin":@(infoAreaLeftMargin),
                              @"infoAreaRightMargin":@(infoAreaRightMargin),
                              @"infoAreaWidth":@(infoAreaSize.width),
                              @"infoAreaHeight":@(infoAreaSize.height)};
    
    //NSLog(@"metrics %@", metrics);
    
    format = @"H:|-(infoAreaLeftMargin)-[infoAreaView(infoAreaWidth)]-(infoAreaRightMargin)-|";
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                             options:NSLayoutFormatDirectionLeftToRight
                                                                             metrics:metrics
                                                                               views:views]];
    
    format = @"V:|-(infoAreaTopMargin)-[infoAreaView(infoAreaHeight)]-(infoAreaBottomMargin)-|";
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                             options:NSLayoutFormatDirectionLeadingToTrailing
                                                                             metrics:metrics
                                                                               views:views]];
    
    //////////////////////////////////////////////////

    // 避免使用transform轉向的模式時出現conflict
    for(NSLayoutConstraint *constraint in constraints)
    {
        constraint.priority = UILayoutPriorityRequired-1;
    }

    //////////////////////////////////////////////////

    if(self.infoAreaViewConstraints != nil)
    {
        [self removeConstraints:self.infoAreaViewConstraints];
        self.infoAreaViewConstraints = nil;
    }
    
    [self addConstraints:constraints];
    self.infoAreaViewConstraints = constraints;
    
    //////////////////////////////////////////////////

    [self setupInfoAreaSubviewConstraintsWithInfoAreaSize:infoAreaSize
                                           titleLabelSize:titleLabelSize
                                         messageLabelSize:messageLabelSize
                                        confirmButtonSize:confirmButtonSize];
}


//================================================================================
//
//================================================================================
- (void)updateGuideLayout
{
    CGRect focusFrame = CGRectZero;
    PPGuideDataModel_LayoutDirection layoutDirection = self.guideData.layoutDirection;
    CGFloat infoAreaGap = 0.0;
    
    
    switch (self.guideData.specialStyle)
    {
        case PPGuideDataModel_SpecialStyle_GuideInfoInTargetView:
        {
            //////////////////////////////////////////////////
            // 限定條件使用此特殊樣式
            
            NSAssert((([self.guideData.guideFocusModel style] == PPGuideFocusStyle_Image ||
                       [self.guideData.guideFocusModel style] == PPGuideFocusStyle_View) &&
                      [self.guideData.guideFocusModel shape] == PPGuideFocusShape_None), @"Focus樣式不符合PPGuideDataModel_SpecialStyle_GuideInfoInTargetView要求！");

            
            //////////////////////////////////////////////////
            // 取得targetView範圍，PPGuideDataModel_ReservedIndex_NoTarget代表全螢幕。
            
            CGRect targetViewFrame;
            
            if(self.guideData.guideIndex == PPGuideDataModel_ReservedIndex_NoTarget)
            {
                targetViewFrame = CGRectMake(0, 0, self.bounds.size.width, self.bounds.size.height);
            }
            else
            {
                targetViewFrame.origin = [self.guideData.targetView convertPoint:CGPointZero toView:self];
                targetViewFrame.size = self.guideData.targetView.frame.size;
            }
            
            
            //////////////////////////////////////////////////
            // 計算guide顯示區域

            switch (self.guideData.guideFocusModel.style)
            {
                case PPGuideFocusStyle_Image:
                {
                    CGSize imageSize = self.guideData.guideFocusModel.image.size;
                    
                    focusFrame = CGRectMake(targetViewFrame.origin.x + (targetViewFrame.size.width - imageSize.width)/2,
                                            targetViewFrame.origin.y + (targetViewFrame.size.height - imageSize.height)/2,
                                            imageSize.width,
                                            imageSize.height);
                    
                    infoAreaGap = self.guideData.guideFocusModel.infoAreaGap;
                    
                    if(self.guideData.layoutDirection == PPGuideDataModel_LayoutDirection_Horz)
                    {
                        focusFrame.origin.x -= (infoAreaGap/2);
                    }
                    else
                    {
                        focusFrame.origin.y -= (infoAreaGap/2);
                    }
                    
                    self.privMaskView.transparentPath = nil;
                    self.privMaskView.maskColor = self.guideAttribute.maskColor;
                    self.privMaskView.focusFrame = focusFrame;
                    self.privMaskView.focusImage = self.guideData.guideFocusModel.image;
                    self.privMaskView.focusStyle = PPGuideFocusStyle_Image;
                    [self.privMaskView setNeedsDisplay];
                    
                    break;
                }

                case PPGuideFocusStyle_View:
                {
                    // MARK: PPGuideFocusStyle_View顯示處理
                    UIView *customView = self.guideData.guideFocusModel.view;
                    CGSize viewSize = customView.bounds.size;
                    
                    customView.frame = CGRectMake(targetViewFrame.origin.x + (targetViewFrame.size.width - viewSize.width)/2,
                                                  targetViewFrame.origin.y + (targetViewFrame.size.height - viewSize.height)/2,
                                                  viewSize.width,
                                                  viewSize.height);
                    
                    [self.privMaskView addSubview:customView];

                    self.privMaskView.transparentPath = nil;
                    self.privMaskView.maskColor = self.guideAttribute.maskColor;
                    self.privMaskView.focusFrame = focusFrame;
                    self.privMaskView.focusStyle = PPGuideFocusStyle_View;
                    [self.privMaskView setNeedsDisplay];
                    
                    self.confirmButton.hidden = YES;

                    break;
                }

                default:
                    break;
            }
            
            break;
        }
            
        default:
        {
            //////////////////////////////////////////////////
            // Normal focus mode
            
            
            //////////////////////////////////////////////////
            // 計算targetView在guideView中對應的位置
            
            CGRect targetViewFrame;
            
            targetViewFrame.origin = [self.guideData.targetView convertPoint:CGPointZero toView:self];
            targetViewFrame.size = self.guideData.targetView.frame.size;
            
            
            //////////////////////////////////////////////////
            // MARK: !! special case for monitorSignalWithView
            
            // 處理sourceView不旋轉，但targetView會旋轉的狀況下，targetViewFrame位置錯誤的問題。
            // 例如WCCaptureViewController的manualCapture畫面。
            
            // 被旋轉90/270度（找不到解法，先硬轉回來）
            if(self.guideData.targetView.transform.b * self.guideData.targetView.transform.c == -1)
            {
                targetViewFrame.origin = [self.guideData.targetView convertPoint:CGPointZero toView:self];
                targetViewFrame.size.width = self.guideData.targetView.frame.size.height;
                targetViewFrame.size.height = self.guideData.targetView.frame.size.width;
                
                if(self.guideData.layoutDirection == PPGuideDataModel_LayoutDirection_Vert)
                {
                    layoutDirection = PPGuideDataModel_LayoutDirection_Horz;
                }
                else if(self.guideData.layoutDirection == PPGuideDataModel_LayoutDirection_Horz)
                {
                    layoutDirection = PPGuideDataModel_LayoutDirection_Vert;
                }
            }
            // 被旋轉180度
            else if(self.guideData.targetView.transform.a==-1 && self.guideData.targetView.transform.d == -1)
            {
                targetViewFrame.origin = [self.guideData.targetView convertPoint:CGPointZero toView:self];
            }
            
            
            //    if(!self.testView)
            //    {
            //        _testView = [[UIView alloc] initWithFrame:CGRectZero];
            //        self.testView.backgroundColor = [UIColor greenColor];
            //        self.testView.alpha = 0.5;
            //        [self addSubview:self.testView];
            //    }
            //
            //    self.guideData.targetView.backgroundColor = [UIColor redColor];
            //    self.testView.frame = targetViewFrame;
            
            
            
            //////////////////////////////////////////////////
            // 計算包含targetView的最小透明區域
            
            PPGuideFocusModel *focusModel = self.guideData.guideFocusModel ? self.guideData.guideFocusModel:self.guideAttribute.guideFocus;
            CGRect transparentFrame = CGRectZero;
            
            infoAreaGap = focusModel.infoAreaGap;
            
            
            switch (focusModel.shape)
            {
                case PPGuideFocusShape_Circle:
                {
                    CGPoint targetCenter = CGPointMake(CGRectGetMidX(targetViewFrame), CGRectGetMidY(targetViewFrame));
                    CGFloat length = ceil(sqrt(pow(targetViewFrame.size.width, 2) + pow(targetViewFrame.size.height, 2)));
                    
                    transparentFrame = CGRectMake(targetCenter.x-length/2,
                                                  targetCenter.y-length/2,
                                                  length, length);
                    
                    transparentFrame = CGRectInset(transparentFrame,
                                                   focusModel.targetAreaInset.width,
                                                   focusModel.targetAreaInset.height);
                    
                    self.privMaskView.transparentPath = [UIBezierPath bezierPathWithOvalInRect:transparentFrame];
                    break;
                }
                    
                case PPGuideFocusShape_Rectangle:
                {
                    transparentFrame = CGRectInset(targetViewFrame,
                                                   focusModel.targetAreaInset.width,
                                                   focusModel.targetAreaInset.height);
                    
                    self.privMaskView.transparentPath = [UIBezierPath bezierPathWithRect:transparentFrame];
                    break;
                }
                    
                default:
                    self.privMaskView.transparentPath = nil;
                    break;
            }
            
            
            //////////////////////////////////////////////////
            // 計算focusFrame
            
            switch (focusModel.style)
            {
                case PPGuideFocusStyle_Image:
                {
                    CGFloat scaleX = transparentFrame.size.width/focusModel.imageTransparentFrame.size.width;
                    CGFloat scaleY = transparentFrame.size.height/focusModel.imageTransparentFrame.size.height;
                    CGSize sizeInset = CGSizeMake((focusModel.image.size.width-focusModel.imageTransparentFrame.size.width)*scaleX,
                                                  (focusModel.image.size.height-focusModel.imageTransparentFrame.size.height)*scaleY);
                    
                    focusFrame = CGRectInset(transparentFrame, -sizeInset.width/2, -sizeInset.height/2);
                    
                    break;
                }
                    
                case PPGuideFocusStyle_DoubleRing:
                {
                    // 設定FocusFrame來調整InfoArea位置
                    CGFloat expandSize = -30;
                    focusFrame = CGRectInset(transparentFrame, expandSize, expandSize);
                    
                    self.privMaskView.focusColor = focusModel.color;
                    break;
                }
                    
                default:
                {
                    focusFrame = transparentFrame;
                    break;
                }
            }
            
            self.privMaskView.focusFrame = focusFrame;
            self.privMaskView.focusStyle = focusModel.style;
            [self.privMaskView setNeedsDisplay];
            break;
        }
    }
        
        
    
    //////////////////////////////////////////////////
    // 更新layout
    
    // 先清除constraint避免出現conflict warning
    [self removeAllConstraints];        
    [self setupInfoAreaViewConstraintsWithFocusFrame:focusFrame
                                              bounds:self.bounds
                                         infoAreaGap:infoAreaGap
                                     layoutDirection:layoutDirection];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)setupLayoutConstraintWithTransform:(CGAffineTransform)transform
{
    //////////////////////////////////////////////////
    // MARK: !! special case for monitorSignalWithView
    
    if(CGAffineTransformEqualToTransform(self.transform, transform) == YES)
    {
        return;
    }
    else
    {
        self.transform = transform;
    }

    
    //////////////////////////////////////////////////
    // 水平開啟時位置會偏移，所以要指定constraint來解決。
    
    self.translatesAutoresizingMaskIntoConstraints = NO;

    NSMutableArray *constraints = [NSMutableArray array];
    
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                        attribute:NSLayoutAttributeCenterX
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:self.superview
                                                        attribute:NSLayoutAttributeCenterX
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                        attribute:NSLayoutAttributeCenterY
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:self.superview
                                                        attribute:NSLayoutAttributeCenterY
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    
    // 旋轉90/270度
    if(transform.b * transform.c == -1)
    {
        if([[[UIDevice currentDevice] systemVersion] floatValue] < 8.0)
        {
            [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                                attribute:NSLayoutAttributeWidth
                                                                relatedBy:NSLayoutRelationEqual
                                                                   toItem:self.superview
                                                                attribute:NSLayoutAttributeWidth
                                                               multiplier:1.0
                                                                 constant:0.0]];
            
            [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                                attribute:NSLayoutAttributeHeight
                                                                relatedBy:NSLayoutRelationEqual
                                                                   toItem:self.superview
                                                                attribute:NSLayoutAttributeHeight
                                                               multiplier:1.0
                                                                 constant:0.0]];
        }
        else // iOS8 寬高要對調
        {
            [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                                attribute:NSLayoutAttributeWidth
                                                                relatedBy:NSLayoutRelationEqual
                                                                   toItem:self.superview
                                                                attribute:NSLayoutAttributeHeight
                                                               multiplier:1.0
                                                                 constant:0.0]];
            
            [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                                attribute:NSLayoutAttributeHeight
                                                                relatedBy:NSLayoutRelationEqual
                                                                   toItem:self.superview
                                                                attribute:NSLayoutAttributeWidth
                                                               multiplier:1.0
                                                                 constant:0.0]];
        }
    }
    else
    {
        [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                            attribute:NSLayoutAttributeWidth
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:self.superview
                                                            attribute:NSLayoutAttributeWidth
                                                           multiplier:1.0
                                                             constant:0.0]];
        
        [constraints addObject:[NSLayoutConstraint constraintWithItem:self
                                                            attribute:NSLayoutAttributeHeight
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:self.superview
                                                            attribute:NSLayoutAttributeHeight
                                                           multiplier:1.0
                                                             constant:0.0]];
    }

    //////////////////////////////////////////////////

    if([constraints count] > 0)
    {
        if(self.guideViewConstraints != nil)
        {
            [self.superview removeConstraints:self.guideViewConstraints];
        }
        
        [self.superview addConstraints:constraints];
        self.guideViewConstraints = constraints;
    }
}


//================================================================================
//
//================================================================================
- (void)assignGuideAttribute:(PPGuideAttributeModel *)guideAttribute guideData:(PPGuideDataModel *)guideData
{
//    NSLog(@"%s", __func__);
    
    self.guideData = guideData;
    self.guideAttribute = guideAttribute;
    
    // !! focus的相關資料在updateConstraints時才指定
    self.privMaskView.maskColor = guideAttribute.maskColor;
    self.privMaskView.useBlurBackground = guideAttribute.useBlurBackground;
    self.privMaskView.focusImage = guideAttribute.guideFocus.image;
    
    self.titleLabel.font = guideAttribute.titleFont;
    self.titleLabel.textColor = guideAttribute.titleColor;
    self.titleLabel.text = guideData.title;
    
    self.messageLabel.font = guideAttribute.messageFont;
    self.messageLabel.textColor = guideAttribute.messageColor;
    self.messageLabel.text = guideData.message;
    
    
    //////////////////////////////////////////////////
    // handle confirm button
    
    if(guideData.guideFocusModel.style == PPGuideFocusStyle_View)
    {
        // MARK: PPGuideFocusStyle_View有自己的confirm button (第一個UIButton)        
        if(self.confirmButton != nil)
        {
            [self.confirmButton removeTarget:nil action:NULL forControlEvents:UIControlEventTouchUpInside];
            [self.confirmButton removeFromSuperview];
            self.confirmButton = nil;
        }

        for(UIView *subView in guideData.guideFocusModel.view.subviews)
        {
            if([subView isKindOfClass:[UIButton class]] == YES)
            {
                // 這裡不要使用self.confirmButton，因為superView是不同的。
                self.customViewConfirmButton = (UIButton *)subView;
                [self.customViewConfirmButton addTarget:self action:@selector(onConfirmAction:) forControlEvents:UIControlEventTouchUpInside];
                break;
            }
        }
    }
    else
    {
        if(self.confirmButton != guideAttribute.confirmButton)
        {
            if(self.confirmButton != nil)
            {
                [self.confirmButton removeTarget:nil action:NULL forControlEvents:UIControlEventTouchUpInside];
                [self.confirmButton removeFromSuperview];
                self.confirmButton = nil;
            }
            
            if(guideAttribute.confirmButton != nil)
            {
                self.confirmButton = guideAttribute.confirmButton;
                self.confirmButton.userInteractionEnabled = YES;
                self.confirmButton.translatesAutoresizingMaskIntoConstraints = NO;
                [self.confirmButton addTarget:self action:@selector(onConfirmAction:) forControlEvents:UIControlEventTouchUpInside];
                [self.infoAreaView addSubview:self.confirmButton];
            }
        }
    }
    
    
    //////////////////////////////////////////////////
    
    [self updateGuideLayout];
}


//================================================================================
//
//================================================================================
- (void)hideGudieTargetAndInfo:(BOOL)hidden
{
    if(hidden)
    {
        self.infoAreaView.alpha = 0.0;

        self.privMaskView.hideTarget = YES;
        [self.privMaskView setNeedsDisplay];
    }
    else
    {
        // !! 先全部關掉，後面animation效果才會清楚
        [self hideGudieTargetAndInfo:YES];
        
        //////////////////////////////////////////////////
        
        CATransition *animation = [CATransition animation];
        [animation setType:kCATransitionFade];
        [animation setDuration:0.3];
        [animation setRemovedOnCompletion:YES];
        
        [self.privMaskView.layer addAnimation:animation forKey:@"ShowMaskAnimation"];
        self.privMaskView.hideTarget = NO;
        [self.privMaskView setNeedsDisplay];
        
        self.infoAreaView.alpha = 1.0;
    }
}


//================================================================================
//
//================================================================================
- (void)removeAllConstraints
{
    if(self.infoAreaSubviewConstraints)
    {
        [self.infoAreaView removeConstraints:self.infoAreaSubviewConstraints];
        self.infoAreaSubviewConstraints = nil;
    }
    
    if(self.infoAreaViewConstraints)
    {
        [self removeConstraints:self.infoAreaViewConstraints];
        self.infoAreaViewConstraints = nil;
    }
}


@end
