//
//  PPGuideController.m
//
//  Created by eddie on 2015/4/3.
//  Copyright (c) 2015年 eddie. All rights reserved.
//

#import <ReactiveCocoa/ReactiveCocoa.h>
#import "PPGuideController.h"
#import "PPGuideView.h"
#import "PPGuideScriptModel.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define DEGREES_TO_RADIANS(degrees) (degrees * M_PI / 180)

typedef NS_OPTIONS(NSInteger, PPGuideController_ViewControllerState)
{
    PPGuideController_VCS_None = 0,
    PPGuideController_VCS_viewDidAppear = 1 << 0,
    PPGuideController_VCS_viewDidLayoutSubviews = 1 << 1,
    PPGuideController_VCS_viewWillDisappear = 1 << 2,
};


////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPGuideController () <PPGuideViewDelegate>
@property (nonatomic, assign) id<PPGuideControllerDelegate> delegate;
@property (nonatomic, retain) PPGuideAttributeModel *guideAttributeModel;
@property (nonatomic, retain) NSMutableArray *guideScriptModels;
@property (nonatomic, retain) PPGuideScriptModel *curGuideScriptModel;
@property (nonatomic, retain) PPGuideView *guideView;
@property (atomic, assign) PPGuideController_ViewControllerState viewControllerState;

@property (nonatomic, retain) UIViewController *monitorViewController;
@property (nonatomic, retain) UIView *monitorView;

// for ReactiveCocoa
@property (nonatomic, retain) NSMutableArray *racDisposables;
@property (nonatomic, retain) RACScopedDisposable *targetViewRacDisposable; // only for monitorView

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPGuideController


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        self.delegate = nil;
        self.guideScriptModels = [NSMutableArray array];
        _guideAttributeModel = [[PPGuideAttributeModel alloc] init];
        
        self.racDisposables = [NSMutableArray array];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;
    
    
    [self.guideView removeFromSuperview];
    self.guideView = nil;
    
    self.guideAttributeModel = nil;
    self.guideScriptModels = nil;
    self.curGuideScriptModel = nil;
    
    self.monitorViewController = nil;
    self.monitorView = nil;
    
    self.racDisposables = nil;
    self.targetViewRacDisposable = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPGuideViewDelegate methods

//================================================================================
//
//================================================================================
- (BOOL)guideView:(PPGuideView *)guideView canTouchThroughWithPoint:(CGPoint)point
{
    if(self.guideAttributeModel.guideFocus.canTouchThrough == YES)
    {
        //////////////////////////////////////////////////
        // MARK:注意TouchThrough的做法(1)
        // 判斷點擊是否可以穿透
        PPGuideDataModel *dataModel = [self.curGuideScriptModel.dataModels firstObject];
        
        if(dataModel != nil && dataModel.targetView != nil)
        {
            CGPoint pointInTargetView = [guideView convertPoint:point toView:dataModel.targetView];
            
            if(CGRectContainsPoint(dataModel.targetView.bounds, pointInTargetView) == true)
            {
                return YES;
            }
        }
    }
    
    return NO;
}


//================================================================================
//
//================================================================================
- (void)guideView:(PPGuideView *)guideView didTapWithPoint:(CGPoint)point
{
//    NSLog(@"%s", __func__);
    
    //////////////////////////////////////////////////
    
    if(self.guideAttributeModel.guideFocus.canTouchThrough == YES)
    {
        //////////////////////////////////////////////////
        // MARK:注意TouchThrough的做法(2)
        // 穿透後的處理
        PPGuideDataModel *dataModel = [self.curGuideScriptModel.dataModels firstObject];
        
        if(dataModel != nil && dataModel.targetView != nil)
        {
            CGPoint pointInTargetView = [guideView convertPoint:point toView:dataModel.targetView];
            
            if(CGRectContainsPoint(dataModel.targetView.bounds, pointInTargetView) == true)
            {
                [self showNextGuideData];
            }
        }
    }
    else
    {
        // show next guide or close guide
        [self showNextGuideData];
    }
}


//================================================================================
//
//================================================================================
- (void)guideView:(PPGuideView *)guideView didClickConfirmButton:(UIButton *)button
{
	__block typeof(self) blockSelf = self;
	
    dispatch_async(dispatch_get_main_queue(), ^{
        
        if([blockSelf.delegate respondsToSelector:@selector(guideControllerDidClickConfirmButton:)])
        {
            [blockSelf.delegate guideControllerDidClickConfirmButton:button];
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods


//================================================================================
//
//================================================================================
- (void)showNextGuideData
{
    [self removeFirstGuideData];

    if([self.curGuideScriptModel.dataModels count] > 0)
    {
        PPGuideDataModel *nextDataModel = [self.curGuideScriptModel.dataModels firstObject];
        NSString *curClassName = self.monitorView ? self.monitorView.ppGuideClassName : NSStringFromClass([self.monitorViewController class]);
        
        
        if([nextDataModel.className isEqualToString:curClassName])
        {
            [self.guideView hideGudieTargetAndInfo:YES];
            [self handleGuideTarget];
            [self.guideView hideGudieTargetAndInfo:NO];
        }
        else
        {
            [self.guideView removeFromSuperview];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)removeFirstGuideData
{
    if([self.curGuideScriptModel.dataModels count] > 0)
    {
        [self.curGuideScriptModel.dataModels removeObjectAtIndex:0];
    }
    
    //////////////////////////////////////////////////
    // MARK: !! special case for monitorSignalWithView
    
    if(self.monitorView != nil)
    {
        self.targetViewRacDisposable = nil;
    }

    //////////////////////////////////////////////////
    
    if([self.curGuideScriptModel.dataModels count] == 0)
    {
        [self finishCurrentGuideScript];
    }
}


//================================================================================
//
//================================================================================
- (void)finishCurrentGuideScript
{
	if(self.curGuideScriptModel == nil)
	{
		return;
	}

	
    //////////////////////////////////////////////////
	// 結束目前顯示的guide

    __block NSString *finishScriptName = [self.curGuideScriptModel.name copy];
    
    [self.guideScriptModels removeObject:self.curGuideScriptModel];
    [self.racDisposables removeAllObjects];
    self.curGuideScriptModel = nil;
    self.monitorViewController = nil;
    
    
    //////////////////////////////////////////////////
    // !! 移除guideView和發送finish delegate要做兩層dispatch。
    //    不這樣做的話，接收端如果在delegate method裡又用手動方式
    //    開啟另一個script，guideView會先被assign script的資料，
    //    然後才被release，顯示變得不正常。
	
	__block typeof(self) blockSelf = [self retain];
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        [blockSelf.guideView removeFromSuperview];
        blockSelf.guideView = nil;
        
        dispatch_async(dispatch_get_main_queue(), ^{

            if([blockSelf.delegate respondsToSelector:@selector(guideControllerDidFinishScript:)])
            {
                [blockSelf.delegate guideControllerDidFinishScript:finishScriptName];
            }
            
            [finishScriptName release];
			[blockSelf release];
        });
    });
}


//==============================================================================
// 使用這個關閉不會記錄為已顯示過
//==============================================================================
- (void)forceFinishGuide
{
    if([PPGuideController isGuideDisplaying]==YES)
    {
        // 強制關閉所以不收delegate
        [PPGuideController setDelegate:nil];
        [PPGuideController finishCurrentGuideScript];
    }
}

//================================================================================
//
//================================================================================
- (CGAffineTransform)transformWithOrientation:(UIInterfaceOrientation)orientation
{
    switch (orientation)
    {
        case UIInterfaceOrientationLandscapeLeft:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(-90));
            
        case UIInterfaceOrientationLandscapeRight:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(90));
            
        case UIInterfaceOrientationPortraitUpsideDown:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(180));
            
        case UIInterfaceOrientationPortrait:
        default:
            return CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(0));
    }
}


//================================================================================
//
//================================================================================
- (UIView *)findTargetViewWithGuideIndex:(NSInteger)guideIndex parentView:(UIView *)parentView
{
    if(parentView.ppGuideIndex == guideIndex)
    {
        return parentView;
    }
    
    for(UIView *subview in parentView.subviews)
    {
        //////////////////////////////////////////////////
        // MARK:如果要支援UIBarButtonItem，取得系統樣式item的view方式如下

//        if([subview isKindOfClass:[UIToolbar class]])
//        {
//            UIToolbar *toolbar = (UIToolbar *)subview;
//
//            for (UIBarButtonItem *item in toolbar.items)
//            {
//                UIView *itemView = (UIView *)[item performSelector:@selector(view)];
//            }
//        }
        
        //////////////////////////////////////////////////
        
//        NSLog(@"%s subview (%d) %@ %@", __func__, (int)subview.ppGuideIndex, NSStringFromClass([subview class]), NSStringFromCGRect(subview.frame));
        
        if([subview isKindOfClass:[UIView class]])
        {
            if(subview.ppGuideIndex == guideIndex)
            {
                return subview;
            }
            else
            {
                UIView *targetView = [self findTargetViewWithGuideIndex:guideIndex parentView:subview];
                
                // 有值才回傳，沒有的話繼續找。
                if(targetView != nil)
                {
                    return targetView;
                }
            }
        }
    }
    
    return nil;
}


//================================================================================
// !! 順序有關聯性，不可亂調
//================================================================================
- (void)modifyGuideView
{
    //////////////////////////////////////////////////
    // prepare guide data

    PPGuideDataModel *dataModel = [self.curGuideScriptModel.dataModels firstObject];

    
    //////////////////////////////////////////////////
    // prepare guide view

    if(self.monitorView != nil)
    {
        if(self.guideView == nil)
        {
            _guideView = [[PPGuideView alloc] initWithFrame:CGRectZero delegate:self];

            [self.monitorView addSubview:self.guideView];
        }
        else
        {
            [self.monitorView bringSubviewToFront:self.guideView];
        }
        
        [self.guideView setupLayoutConstraintWithTransform:dataModel.targetView.transform];
        
        // MARK: 這裏一定要指定大小才會正常，應該是流程有問題，後續要再找原因。
        self.guideView.frame = self.monitorView.bounds;
    }
    else
    {
        UIWindow *keyWindow = [UIApplication sharedApplication].keyWindow;
        
        if (!keyWindow)
        {
            keyWindow = [[UIApplication sharedApplication].windows objectAtIndex:0];
        }
        
        if(self.guideView == nil)
        {
            _guideView = [[PPGuideView alloc] initWithFrame:CGRectZero delegate:self];
            [keyWindow addSubview:self.guideView];
        }
        else
        {
            if(self.guideView.superview == nil)
            {
                [keyWindow addSubview:self.guideView];
            }
            
            [keyWindow bringSubviewToFront:self.guideView];
        }
        
        //////////////////////////////////////////////////
        // reset guide view position
        
        // !! 旋轉時要先移除constraint，否則設定transform時就會出現constraint conflict的情況。
        [self.guideView removeAllConstraints];
        
        if([[[UIDevice currentDevice] systemVersion] floatValue] < 8.0)
        {
            self.guideView.transform = [self transformWithOrientation:self.monitorViewController.interfaceOrientation];
        }
        
        self.guideView.frame = keyWindow.bounds;
    }

    
    //////////////////////////////////////////////////

    [self.guideView assignGuideAttribute:self.guideAttributeModel guideData:dataModel];
    [self.guideView layoutIfNeeded];
}


//================================================================================
//
//================================================================================
- (void)handleGuideTarget
{
    if([self.curGuideScriptModel.dataModels count] == 0)
    {
        return;
    }
    
    //////////////////////////////////////////////////

    PPGuideDataModel *dataModel = [self.curGuideScriptModel.dataModels firstObject];
    
    
    if(dataModel.guideIndex == PPGuideDataModel_ReservedIndex_NoTarget)
    {
        //////////////////////////////////////////////////
        // No target, show in screen center
    
        [self modifyGuideView];
    }
    else
    {
        //////////////////////////////////////////////////
        // Find target view
        
        if(self.monitorView != nil)
        {
            dataModel.targetView = [self findTargetViewWithGuideIndex:dataModel.guideIndex
                                                           parentView:self.monitorView];
        }
        else
        {
            // 先找是不是在navigationBar上面
            if(self.monitorViewController.navigationController != nil)
            {
                dataModel.targetView = [self findTargetViewWithGuideIndex:dataModel.guideIndex
                                                               parentView:self.monitorViewController.navigationController.navigationBar];
            }
            
            // 沒有繼續找viewController的view
            if(dataModel.targetView == nil)
            {
                dataModel.targetView = [self findTargetViewWithGuideIndex:dataModel.guideIndex
                                                               parentView:self.monitorViewController.view];
            }
        }
        
        
        //////////////////////////////////////////////////
        // Show guide if found visible target view
        
        if(dataModel.targetView != nil && dataModel.targetView.hidden == NO)
        {
            [self modifyGuideView];
            
            
            //////////////////////////////////////////////////
            // MARK: !! special case for monitorSignalWithView
            
            // 監測target的rotate動作
            if(self.monitorView != nil)
            {
                // 不要重複監測
                if(self.targetViewRacDisposable == nil)
                {
                    RACSignal *signal = [dataModel.targetView rac_valuesForKeyPath:@"transform" observer:self];
                    __block typeof(self) blockSelf = self;
                    
                   self.targetViewRacDisposable = [[signal subscribeNext:^(id x) {
                        
                        [blockSelf modifyGuideView];
                        
                    }] asScopedDisposable];
                }
            }
        }
        else
        {
            __block typeof(self) blockSelf = self;
            
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [blockSelf showNextGuideData];
            });
        }
    }
        
}


//================================================================================
//
//================================================================================
- (void)findCurGuideScriptModelWithClassName:(NSString *)className
{
    if(self.curGuideScriptModel == nil)
    {
        for(PPGuideScriptModel *scriptModel in self.guideScriptModels)
        {
            PPGuideDataModel *firstDataModel = [scriptModel.dataModels firstObject];
            
            if([[firstDataModel className] isEqualToString:className])
            {
                self.curGuideScriptModel = scriptModel;
                break;
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)monitorRotateSignal
{
    __block PPGuideController *blockSelf = self;
    RACDisposable *disposable = nil;
    
    
    ////////////////////////////////////////////////
    // 因為旋轉後的位置是在didRotateFromInterfaceOrientationy才確定，
    // 為了避免在viewDidLayoutSubviews到didRotateFromInterfaceOrientationy之間
    // 距離差距過大造成顯示時會有跳動的感覺，所以在
    // willRotateToInterfaceOrientation先將一些UI隱藏
    
    RACSignal *willRotateToInterfaceOrientation = [self.monitorViewController rac_signalForSelector:@selector(willRotateToInterfaceOrientation:duration:)];
    
    disposable = [[willRotateToInterfaceOrientation subscribeNext:^(id _) {
        
        if((blockSelf.viewControllerState & PPGuideController_VCS_viewWillDisappear) == 0)
        {
            [blockSelf.guideView hideGudieTargetAndInfo:YES];
        }
        
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];
    
    
    //////////////////////////////////////////////////
    // 收到旋轉完成後重新顯示相關的UI
    
    RACSignal *didRotateFromInterfaceOrientation = [self.monitorViewController rac_signalForSelector:@selector(didRotateFromInterfaceOrientation:)];
    
    disposable = [[didRotateFromInterfaceOrientation subscribeNext:^(id _) {
        
        if((blockSelf.viewControllerState & PPGuideController_VCS_viewWillDisappear) == 0)
        {
            [blockSelf handleGuideTarget];
            [blockSelf.guideView hideGudieTargetAndInfo:NO];
        }
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];
}


//================================================================================
//
//================================================================================
- (void)monitorSignalWithViewController:(UIViewController *)viewController
{
//    NSLog(@"!!! monitorSignalWithViewController %p", viewController);
    
    //////////////////////////////////////////////////
    // find script that contain current viewController
    
    [self findCurGuideScriptModelWithClassName:NSStringFromClass([viewController class])];

    
    //////////////////////////////////////////////////
    // start monitor view controller
    
    self.viewControllerState = PPGuideController_VCS_None;
    self.monitorViewController = viewController;
    self.monitorView = nil;
    
    __block PPGuideController *blockSelf = self;
    RACDisposable *disposable = nil;

    
    // MARK:請注意監測的流程處理

    //////////////////////////////////////////////////
    // 畫面開啟及旋轉過程的處理
    // 1. ios 7/8 viewDidAppear和viewDidLayoutSubviews收到的順序不一樣，
    //    所以要確定兩者都收到才開始顯示。
    // 2. 旋轉過程中viewDidLayoutSubviews收到的並不是最後的位置，
    //    didRotateFromInterfaceOrientation才是最後的位置。
    // 3. 旋轉過程中若介面要跟隨旋轉一定要處理viewDidLayoutSubviews。
    
    RACSignal *viewDidAppear = [viewController rac_signalForSelector:@selector(viewDidAppear:)];

    disposable = [[viewDidAppear subscribeNext:^(id _) {
        
        blockSelf.viewControllerState |= PPGuideController_VCS_viewDidAppear;
        
        if((blockSelf.viewControllerState & PPGuideController_VCS_viewDidLayoutSubviews) > 0 &&
           (blockSelf.viewControllerState & PPGuideController_VCS_viewWillDisappear) == 0)
        {
            [blockSelf handleGuideTarget];
        }
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];
    

    RACSignal *viewDidLayoutSubviews = [viewController rac_signalForSelector:@selector(viewDidLayoutSubviews)];
    
    disposable = [[viewDidLayoutSubviews subscribeNext:^(id _) {
        
        blockSelf.viewControllerState |= PPGuideController_VCS_viewDidLayoutSubviews;
        
        if((blockSelf.viewControllerState & PPGuideController_VCS_viewDidAppear) > 0 &&
           (blockSelf.viewControllerState & PPGuideController_VCS_viewWillDisappear) == 0)
        {
            [blockSelf handleGuideTarget];
        }
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];


    //////////////////////////////////////////////////
    // viewWillDisappear後還是會收到signal，但已不用處理，先用flag方式濾除。

    RACSignal *viewWillDisappear = [viewController rac_signalForSelector:@selector(viewWillDisappear:)];
    
    disposable = [[viewWillDisappear subscribeNext:^(id _) {
        
        blockSelf.viewControllerState |= PPGuideController_VCS_viewWillDisappear;
        
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];

    
    //////////////////////////////////////////////////
    // 旋轉處理
    
    [self monitorRotateSignal];
}


//================================================================================
//
//================================================================================
- (void)manuallyShowGuideWithViewController:(UIViewController *)viewController
{
    [self findCurGuideScriptModelWithClassName:NSStringFromClass([viewController class])];

    //////////////////////////////////////////////////

    self.monitorViewController = viewController;
    self.monitorView = nil;

    
    //////////////////////////////////////////////////
    // 顯示guide
    
    [self handleGuideTarget];
    
    
    //////////////////////////////////////////////////
    // 手動開啟時需要監測旋轉
    
    __block PPGuideController *blockSelf = self;
    RACDisposable *disposable = nil;
    
    
    //////////////////////////////////////////////////
    // 旋轉過程中若介面要跟隨旋轉一定要處理viewDidLayoutSubviews。

    RACSignal *viewDidLayoutSubviews = [viewController rac_signalForSelector:@selector(viewDidLayoutSubviews)];
    
    disposable = [[viewDidLayoutSubviews subscribeNext:^(id _) {
        
        [blockSelf handleGuideTarget];
    }] asScopedDisposable];
    
    [self.racDisposables addObject:disposable];

    
    //////////////////////////////////////////////////
    // 旋轉處理
    
    [self monitorRotateSignal];
}


//================================================================================
//
//================================================================================
- (void)manuallyShowGuideWithView:(UIView *)view
{
    NSAssert([view.ppGuideClassName length]>0, @"Show guide with view must assign ppGuideClassName");
    
    //////////////////////////////////////////////////
    
    self.monitorViewController = nil;
    self.monitorView = view;
    
    if(self.guideView != nil)
    {
        [self.monitorView addSubview:self.guideView];
    }    
    
    //////////////////////////////////////////////////
    // 顯示guide
    
    [self findCurGuideScriptModelWithClassName:self.monitorView.ppGuideClassName];
    [self handleGuideTarget];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods : Common use

//================================================================================
//
//================================================================================
+ (instancetype)sharedInstance
{
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
+ (void)setDelegate:(id<PPGuideControllerDelegate>)delegate
{
    PPGuideController *controller = [self sharedInstance];

    controller.delegate = delegate;
}


//================================================================================
//
//================================================================================
+ (void)setAttributeWithMaskColor:(UIColor *)maskColor
                useBlurBackground:(BOOL)useBlurBackground
                       guideFocus:(PPGuideFocusModel *)guideFocus
                        titleFont:(UIFont *)titleFont
                      titleCollor:(UIColor *)titleColor
                      messageFont:(UIFont *)messageFont
                     messageColor:(UIColor *)messageColor
                    confirmButton:(UIButton *)confirmButton
{
    PPGuideController *controller = [self sharedInstance];

    controller.guideAttributeModel.maskColor = maskColor;
    controller.guideAttributeModel.useBlurBackground = useBlurBackground;
    controller.guideAttributeModel.guideFocus = guideFocus;
    controller.guideAttributeModel.titleFont = titleFont;
    controller.guideAttributeModel.titleColor = titleColor;
    controller.guideAttributeModel.messageFont = messageFont;
    controller.guideAttributeModel.messageColor = messageColor;
    
    if(guideFocus.canTouchThrough == YES)
    {
        controller.guideAttributeModel.confirmButton = nil;
    }
    else
    {
        controller.guideAttributeModel.confirmButton = confirmButton;
    }
}


//================================================================================
//
//================================================================================
+ (void)addGuideScriptWithName:(NSString *)guideScriptName
               guideDataModels:(NSArray *)guideDataModels;
{
    if([guideDataModels count] == 0)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
    PPGuideController *controller = [self sharedInstance];
    
    if(controller.guideScriptModels == nil)
    {
        controller.guideScriptModels = [NSMutableArray array];
    }
    
    //////////////////////////////////////////////////
    
    PPGuideScriptModel *existScriptModel = nil;
    
    // 先找有沒有相同名稱的script
    for (PPGuideScriptModel *scriptModel in controller.guideScriptModels)
    {
        if([scriptModel.name isEqualToString:guideScriptName])
        {
            existScriptModel = scriptModel;
            break;
        }
    }
    
    if(existScriptModel != nil)
    {
        // 有相同名稱的script，把要手動顯示的資料插入到最前面。
        NSIndexSet *indexSet = [NSIndexSet indexSetWithIndexesInRange:NSMakeRange(0, [guideDataModels count])];
        
        [existScriptModel.dataModels insertObjects:guideDataModels atIndexes:indexSet];
    }
    else
    {
        // 沒有相同名稱的script，加入顯示資料。
        PPGuideScriptModel *model = [[PPGuideScriptModel alloc] init];
        
        model.name = guideScriptName;
        model.dataModels = [NSMutableArray arrayWithArray:guideDataModels];
        [controller.guideScriptModels addObject:model];
        [model release];
    }
}


//================================================================================
//
//================================================================================
+ (void)clearAllGuideScripts
{
    PPGuideController *controller = [self sharedInstance];
    
    controller.guideScriptModels = nil;
    controller.curGuideScriptModel = nil;
    
    controller.monitorViewController = nil;
    controller.monitorView = nil;
}


//================================================================================
//
//================================================================================
+ (BOOL)isGuideDisplaying
{
    PPGuideController *controller = [self sharedInstance];
    
    return (controller != nil &&
            controller.guideView != nil &&
            controller.guideView.hidden == NO &&
            controller.guideView.superview != nil);
}


//================================================================================
//
//================================================================================
+ (void)finishCurrentGuideScript
{
    [[self sharedInstance] finishCurrentGuideScript];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods : Guide-In-ViewController (不可和Guide-In-View函式混用)

//================================================================================
//
//================================================================================
+ (void)monitorSignalWithViewController:(UIViewController *)viewController
{
    [[self sharedInstance] monitorSignalWithViewController:viewController];
}


//================================================================================
//
//================================================================================
+ (void)manuallyShowGuideWithViewController:(UIViewController *)viewController
{
    [[self sharedInstance] manuallyShowGuideWithViewController:viewController];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods : Guide-In-View (不可和Guide-In-ViewController函式混用)

//================================================================================
//
//================================================================================
+ (void)manuallyShowGuideWithView:(UIView *)view
{
    [[self sharedInstance] manuallyShowGuideWithView:view];
}



@end
