//
//  PPDataConvertFlowController.m
//
//  Created by eddie on 2015/9/7.
//  Copyright (c) 2015年 penpower. All rights reserved.
//

#import "PPDataConvertFlowController.h"
#import "PPDataConvertFlowController+ResourceDefine.h"
#import "PPDataConverter.h"
#import "PPZipController.h"
#import "PPLogController.h"
#import "PPSettingsController.h"
#import "PPSupportController.h"
#import "NSError+Custom.h"
#import "NSThread+Additions.h"

#if TARGET_OS_IPHONE

#import "PPBackgroundTaskController.h"
#import "UIApplication+Idle.h"
#import "PPViewController.h"
#import "PPBusyView.h"
#import "PPAlertView.h"

#elif TARGET_OS_MAC

#import "PPWaitingWindowController.h"

#endif




////////////////////////////////////////////////////////////////////////////////////////////////////

/// 紀錄需要轉換步驟
NSString * const SettingsKey_TodoConverterKeys = @"PPDataConvertFlowController_SettingsKey_TodoConverterKeys";

/// 紀錄轉換成功的步驟
NSString * const SettingsKey_FinishedConverterKeys = @"PPDataConvertFlowController_SettingsKey_FinishedConverterKeys";

/// 備份成功
NSString * const SettingsKey_BackupSuccess = @"PPDataConvertFlowController_SettingsKey_BackupSuccess";

/// 紀錄轉換成功的app版號
NSString * const SettingsKey_LastConvertVersion = @"PPDataConvertFlowController_SettingsKey_LastConvertVersion";

/// 紀錄檔名稱
NSString * const Log_FileNamePrefix = @"ConvertLog";

/// 備份檔名稱
NSString * const Backup_FileNamePrefix = @"ConvertBackup";

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, ErrorCode)
{
    ErrorCode_None = 0,
    ErrorCode_NotEnoughSpace,
    ErrorCode_FailedToBackup,
    ErrorCode_FailedToConvert,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, AlertTag)
{
    AlertTag_None = 0,
    AlertTag_OkWithExit,
    AlertTag_AbortConvertOrRepoprtError,
    AlertTag_NoEmailSetting,
    AlertTag_ConvertSuccess,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef void (^CompletionBlock)(BOOL success);

////////////////////////////////////////////////////////////////////////////////////////////////////


#if TARGET_OS_IPHONE

@interface PPDataConvertFlowController () <PPDataConverterDelegate, UIAlertViewDelegate, MFMailComposeViewControllerDelegate>

@property (nonatomic, retain) PPBusyView *busyView;

#elif TARGET_OS_MAC

@interface PPDataConvertFlowController () <PPDataConverterDelegate>

@property (nonatomic, retain) PPWaitingWindowController *waitingWindowController;

#endif

@property (nonatomic, retain) NSString *dataDirPath;
@property (nonatomic, retain) NSString *backupFileSuffix;
@property (nonatomic, retain) NSArray *converters;
@property (nonatomic, retain) CPColor *messageColor;
@property (nonatomic, copy) CompletionBlock completionBlock;
@property (nonatomic, retain) PPLogController *logController;
@property (nonatomic, retain) NSError *lastError;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPDataConvertFlowController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)init
{
#if TARGET_OS_IPHONE
    // 檢查iTunes file sharing 有沒有設定
    NSAssert([[NSBundle mainBundle] objectForInfoDictionaryKey:@"UIFileSharingEnabled"], @"!! App需要加入FileSharing設定 !!");
#endif

    //////////////////////////////////////////////////

    if(self = [super init])
    {
        _logController = [[PPLogController alloc] init];
        
        if(self.logController)
        {
            NSString *logDirPath = [PPDataConvertFlowController logDirPath];
            
            if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
            {
                [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                          withIntermediateDirectories:YES
                                                           attributes:nil
                                                                error:nil];
            }
            
            [self.logController setFileName:Log_FileNamePrefix atPath:logDirPath];
            self.logController.toFile = YES;
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
#if TARGET_OS_IPHONE
    self.busyView = nil;
#elif TARGET_OS_MAC
    self.waitingWindowController = nil;
#endif
    
    self.lastError = nil;
    self.dataDirPath = nil;
    self.converters = nil;
    self.messageColor = nil;
    self.completionBlock = nil;
    self.logController = nil;
    self.backupFileSuffix = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPDataConverterDelegate methods

//================================================================================
//
//================================================================================
- (void)dataConverter:(PPDataConverter *)dataConverter accessProgress:(float)accessProgress
{
    if(accessProgress > 0.0)
    {
        int percentage = accessProgress*100;

#if TARGET_OS_IPHONE
        
        [self showBusyMessage:[NSString stringWithFormat:@"%@ %d%%", PPDCFC_MLS_ConvertingData, percentage]];
        
#elif TARGET_OS_MAC
        
            [self.waitingWindowController setProgressBarRate:percentage];
        
#endif


    }
}


//================================================================================
//
//================================================================================
- (void)dataConverter:(PPDataConverter *)dataConverter addSubLog:(NSString *)log
{
    [self addSubLog:log];
}






#if TARGET_OS_IPHONE

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - MFMailComposeViewControllerDelegate


//===============================================================================
//
//===============================================================================
- (void)mailComposeController:(MFMailComposeViewController *)controller didFinishWithResult:(MFMailComposeResult)result error:(NSError *)error
{
    [controller dismissViewControllerAnimated:YES completion:nil];
    
    //////////////////////////////////////////////////
    
    [self showConvertFailedAlertView];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIAlertViewDelegate

//==============================================================================
//
//==============================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    switch (alertView.tag)
    {
        case AlertTag_OkWithExit:
        {
            // 請使用者重新開啟app
            // 強制關閉，避免使用者不知道要從task把app移除再重開。
            exit(0);
            break;
        }
            
        case AlertTag_AbortConvertOrRepoprtError:
        {
            NSString *buttonTitle = [alertView buttonTitleAtIndex:buttonIndex];
            
            if ([buttonTitle isEqualToString:PPDCFC_MLS_AbortConvert])
            {
                // 放棄轉換，變成重新開始。(備份檔會留著)
                [self finishConvertFlowWithSuccess:NO];
            }
            else if ([buttonTitle isEqualToString:PPDCFC_MLS_ReportError])
            {
                // 通知錯誤
                [self addMainLog:@"Report error"];

                if(![MFMailComposeViewController canSendMail])
                {
                    [self showNoEmailSettingAlert];
                    break;
                }
                else
                {
                    [self reportErrorLog];
                }
            }
            break;
        }
            
        case AlertTag_ConvertSuccess:
        {
            [self finishConvertFlowWithSuccess:YES];
            break;
        }
            
        case AlertTag_NoEmailSetting:
        {
            [self showConvertFailedAlertView];
            break;
        }
            
        default:
            break;
    }
}

#endif // end of TARGET_OS_IPHONE



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (NSString *)appVersion
{
    return [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
}


//================================================================================
// appVersion : nil表示使用目前版號
//================================================================================
- (NSString *)backupFilePathWithAppVersion:(NSString *)appVersion
{
    if([appVersion length] == 0)
    {
        appVersion = [self appVersion];
    }
    
    NSString *filePath = [NSString stringWithFormat:@"%@/%@_%@", [PPDataConvertFlowController backupDirPath], Backup_FileNamePrefix, appVersion];
    
    // check if need add suffix
    if([self.backupFileSuffix length] > 0)
    {
        filePath = [filePath stringByAppendingFormat:@"_%@", self.backupFileSuffix];
    }

    return filePath;
}


//================================================================================
//
//================================================================================
- (NSString *)logFilePath
{
    return self.logController.filePath;
}


//================================================================================
//
//================================================================================
- (unsigned long long)diskFreeSize
{
    unsigned long long freeSize = 0;
    NSError *error = nil;
    NSDictionary *attribute = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[PPDataConvertFlowController backupDirPath] error: &error];
    
    freeSize = [[attribute objectForKey:NSFileSystemFreeSize] unsignedLongLongValue];
    
    return freeSize;
}


//================================================================================
//
//================================================================================
- (unsigned long long)totalFileSizeInDirPath:(NSString *)dirPath
{
    NSString *fileName;
    NSError *error = nil;
    unsigned long long fileSize = 0;
    NSDirectoryEnumerator *dirEnum = [[NSFileManager defaultManager] enumeratorAtPath:dirPath];
    
    while (fileName = [dirEnum nextObject])
    {
        NSDictionary* fileAttributes = [[NSFileManager defaultManager] attributesOfItemAtPath:[dirPath stringByAppendingPathComponent:fileName] error:&error];
        fileSize += [fileAttributes fileSize];

#ifdef DEBUG
        NSLog(@"%@ %lld", fileName, [fileAttributes fileSize]);
#endif
        
    }
    
    return fileSize;
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasEnoughDiskSpace:(unsigned long long *)notEnoughSize
{
    BOOL result = NO;
    BOOL hasBackedUp = [self hasBackedUpData]; // !!要先執行，若有備份失敗的檔案才能先清除。
    
    unsigned long long fileSize = [self totalFileSizeInDirPath:self.dataDirPath];
    unsigned long long freeSize = [self diskFreeSize];
    unsigned long long needSize = 0;
    
    if(hasBackedUp == NO)
    {
        needSize = fileSize*2;
    }
    else
    {
        needSize = fileSize*1;
    }
    
    if(freeSize > needSize)
    {
        *notEnoughSize = 0;
        result = YES;
    }
    else
    {
        *notEnoughSize = needSize-freeSize;
        result = NO;
    }

    return result;
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasBackedUpData
{
    NSString *backupFilePath = [self backupFilePathWithAppVersion:nil];
    BOOL fileExist = [[NSFileManager defaultManager] fileExistsAtPath:backupFilePath];
    
    if([PPSettingsController integerValueWithKey:SettingsKey_BackupSuccess] == 1)
    {
        return fileExist;
    }
    else
    {
        if(fileExist == YES)
        {
            [[NSFileManager defaultManager] removeItemAtPath:backupFilePath error:nil];
        }

        return NO;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)backupData
{
    //////////////////////////////////////////////////
    // 移除舊版備份資料
    
    NSString *lastBackupVersion = [PPSettingsController stringValueWithKey:SettingsKey_LastConvertVersion];
    NSString *appVersion = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
    
    if([lastBackupVersion length] > 0 &&
       [lastBackupVersion isEqualToString:appVersion] == NO)
    {
        NSString *oldFilePath = [self backupFilePathWithAppVersion:lastBackupVersion];
        
        if([[NSFileManager defaultManager] fileExistsAtPath:oldFilePath] == YES)
        {
            [[NSFileManager defaultManager] removeItemAtPath:oldFilePath error:nil];
            [self addMainLog:[NSString stringWithFormat:@"remove old backup with version %@", lastBackupVersion]];
        }
    }
    
    
    //////////////////////////////////////////////////
    // 建立目前的備份資料
    
    [PPSettingsController setIntegerValue:0 withKey:SettingsKey_BackupSuccess];
    
    NSArray *excludeDirPaths = @[@"temp", @"backup"];
    BOOL result = [PPZipController zipWithSrcDir:self.dataDirPath
                              excludeDirPatterns:excludeDirPaths
                                         dstFile:[self backupFilePathWithAppVersion:nil]
                                        password:PPDataConvertFlowController_ZipPassword
                                 progressHandler:^(CGFloat progress) {
                                     
                                     if(progress > 0.0)
                                     {
                                         int percentage = progress*100;

#if TARGET_OS_IPHONE
                                         [self showBusyMessage:[NSString stringWithFormat:@"%@ %d%%", PPDCFC_MLS_BackingUpData, percentage]];
                                         
#elif TARGET_OS_MAC
                                         [self.waitingWindowController setProgressBarRate:percentage];
#endif
                                     }
                                 }];
    
    if(result == YES)
    {
        [PPSettingsController setIntegerValue:1 withKey:SettingsKey_BackupSuccess];
        [self addMainLog:[NSString stringWithFormat:@"build new backup with version %@", [self appVersion]]];
    }
    
    return result;
}


//==============================================================================
//
//==============================================================================
- (void)addMainLog:(NSString *)log
{
    [self.logController logWithMask:PPLogControllerMask_Normal format:@"## %@ ##", log];
}


//==============================================================================
//
//==============================================================================
- (void)addSubLog:(NSString *)log
{
    [self.logController logWithMask:PPLogControllerMask_Normal format:@"   - %@", log];
}


//==============================================================================
//
//==============================================================================
- (void)reportErrorLog
{
    dispatch_async(dispatch_get_main_queue(), ^{
        
        NSArray	*paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        NSString *zipFilePath = [NSString stringWithFormat:@"%@/%@.zip", [paths firstObject], Log_FileNamePrefix];
        NSString *srcDir = [PPDataConvertFlowController logDirPath];
        
        if([PPZipController zipWithSrcDir:srcDir
                       excludeDirPatterns:nil
                                  dstFile:zipFilePath
                                 password:PPDataConvertFlowController_ZipPassword] == YES)
        {
            
#if TARGET_OS_IPHONE
            
            UIWindow* keyWindow = [[UIApplication sharedApplication] keyWindow];
            
            [PPSupportController reportMailWithViewController:keyWindow.rootViewController
                                                      license:LicenseString_Active
                                          attachmentFilePaths:@[zipFilePath]
                                              additionalInfos:nil
                                                     delegate:self];

#elif TARGET_OS_MAC
            
            [PPSupportController reportMailWithLicense:LicenseString_Active
                                            recipients:nil
                                   attachmentFilePaths:@[zipFilePath]];
#endif
        }
    });
}


//================================================================================
//
//================================================================================
- (NSString *)sizeTextWithSize:(double)size
{
    if(size == 0)
    {
        return nil;
    }
    
    CGFloat divisor = 1024.0;
    NSString *unit = @"bytes";
    
    if(size/divisor > 1.0)
    {
        size /= divisor;
        unit = @"KB";
        
        if(size/divisor > 1.0)
        {
            size /= divisor;
            unit = @"MB";
            
            if(size/divisor > 1.0)
            {
                size /= divisor;
                unit = @"GB";
            }
        }
    }
    
    return [NSString stringWithFormat:@"%.2f %@", size, unit];
}


//================================================================================
//
//================================================================================
- (NSArray *)convertersNeedTodo
{
    NSMutableArray *todoConverters = [NSMutableArray array];
    
    
    //////////////////////////////////////////////////
    // 檢查是否有新的converter
    
    BOOL hasNewConverter = NO;
    NSArray *preTodoConverterKeys = [PPSettingsController arrayValueWithKey:SettingsKey_TodoConverterKeys];
    
    // test add
//    preTodoConverterKeys = nil;
    
    for (PPDataConverter *converter in self.converters)
    {
        if([preTodoConverterKeys containsObject:converter.key] == NO)
        {
            hasNewConverter = YES;
            break;
        }
    }

    
    //////////////////////////////////////////////////
    // 有新的converter，更新TodoConverterKeys並清除FinishedConverterKeys，
    // 重新使用needConvert詢問要不要執行。
    // (只要加入新的converter就會全部重新詢問一次)
    //
    // !! 原則上converter只有在沒有執行過的前提下才能利用needConvert詢問要不要執行。
    //    因為當converter執行到一半失敗時，下次再詢問needConvert的結果有可能是錯誤的。
    //
    
    if(hasNewConverter == YES)
    {
        NSMutableArray *todoConverterKeys = [NSMutableArray array];
        NSMutableArray *finishedConverterKeys = [NSMutableArray array];
        
        for (PPDataConverter *converter in self.converters)
        {
            [todoConverterKeys addObject:converter.key];
            
            if([converter needConvert] == NO)
            {
                [finishedConverterKeys addObject:converter.key];
                
                [self addMainLog:[NSString stringWithFormat:@"skip converter - %@", converter.key]];
            }
            else
            {
                [todoConverters addObject:converter];
            }
        }
        
        [PPSettingsController setArrayValue:todoConverterKeys withKey:SettingsKey_TodoConverterKeys];
        [PPSettingsController setArrayValue:finishedConverterKeys withKey:SettingsKey_FinishedConverterKeys];
        [PPSettingsController setIntegerValue:0 withKey:SettingsKey_BackupSuccess];
    }
    else
    {
        NSArray *finishedConverterKeys = [PPSettingsController arrayValueWithKey:SettingsKey_FinishedConverterKeys];
     
        for (PPDataConverter *converter in self.converters)
        {
            if([finishedConverterKeys containsObject:converter.key] == NO)
            {
                [todoConverters addObject:converter];
            }
        }
    }

    //////////////////////////////////////////////////

    return todoConverters;
}


//================================================================================
//
//================================================================================
- (void)showBusyMessage:(NSString *)message
{
#if TARGET_OS_IPHONE
   // !!不能在dispatch_sync到dispatch_get_main_queue()的地方call
    dispatch_sync(dispatch_get_main_queue(), ^{
        [PPBusyView postMessage:message];
    });
    
#elif TARGET_OS_MAC
    
    [self.waitingWindowController setTitleString:message];
    
#endif
}


//================================================================================
//
//================================================================================
- (BOOL)runBackup
{
    //////////////////////////////////////////////////
    // !! Check disk space
    
    [self addMainLog:@"hasEnoughDiskSpace (begin)"];
    
    unsigned long long notEnoughSize = 0;
    
    if([self hasEnoughDiskSpace:&notEnoughSize] == NO)
    {
        NSString *message = [NSString stringWithFormat:PPDCFC_MLS_NotEnoughStorageToConvert, [self sizeTextWithSize:notEnoughSize]];
        
        self.lastError = PPErrorMake(ErrorCode_NotEnoughSpace, message, nil);
        [self addMainLog:@"hasEnoughDiskSpace (failed)"];
        [self addSubLog:PPDCFC_MLS_NotEnoughStorageToConvert];
        
        return NO;
    }
    else
    {
        [self addMainLog:@"hasEnoughDiskSpace (success)"];
    }
    
        
    //////////////////////////////////////////////////
    // !! backup data
    
    if([self hasBackedUpData] == YES)
    {
        [self addMainLog:@"backupData (already backed up)"];
    }
    else
    {
        [self addMainLog:@"backupData (begin)"];
        
        if([self backupData] == NO)
        {
            self.lastError = PPErrorMake(ErrorCode_FailedToBackup, PPDCFC_MLS_BackupFailed, nil);
            [self addMainLog:@"backupData (failed)"];
            [self addSubLog:PPDCFC_MLS_BackupFailed];
            
            return NO;
        }
        else
        {
            [self addMainLog:@"backupData (success)"];
        }
    }
 
    //////////////////////////////////////////////////

    return YES;
}


//================================================================================
//
//================================================================================
- (BOOL)runConvertWithConverters:(NSArray *)todoConverters
{
    NSMutableArray *curFinishedKeys = [NSMutableArray array];
    NSArray *finishedConverterKeys = [PPSettingsController arrayValueWithKey:SettingsKey_FinishedConverterKeys];
    
    // prepare array to save finished keys
    if([finishedConverterKeys count])
    {
        [curFinishedKeys addObjectsFromArray:finishedConverterKeys];
    }
    
    // start convert
    for (PPDataConverter *converter in todoConverters)
    {
        [self addMainLog:[NSString stringWithFormat:@"run converter - %@ (begin)", converter.key]];
        
        NSError *error = nil;
        
        if([converter doConvertWithDelegate:self error:&error] == YES)
        {
            [curFinishedKeys addObject:converter.key];
            
            // test remove
            [PPSettingsController setArrayValue:curFinishedKeys withKey:SettingsKey_FinishedConverterKeys];
            
            [self addMainLog:[NSString stringWithFormat:@"run converter - %@ (success)", converter.key]];
        }
        else
        {
            self.lastError = PPErrorMake(ErrorCode_FailedToConvert, PPDCFC_MLS_ConvertFailed, error);
            [self addMainLog:[NSString stringWithFormat:@"run converter - %@ (failed)", converter.key]];
            [self addSubLog:self.lastError.description];
            break;
        }
        
        if(self.lastError != nil)
        {
            return NO;
        }
    }
    
    return YES;
}


//================================================================================
//
//================================================================================
- (void)startConvertFlowWithCompletion:(void (^)(BOOL success))completion
{
    //////////////////////////////////////////////////
    // 取得所有需要執行的converter
    
    __block NSArray *todoConverters = [[self convertersNeedTodo] retain];
    __block BOOL needBackup = NO;
    
    if([todoConverters count] == 0)
    {
        [todoConverters release];
        completion(YES);
        return;
    }
    else
    {
        // !! 檢查是否需要進行備份
        for (PPDataConverter *converter in todoConverters)
        {
            if([converter respondsToSelector:@selector(needBackup)] == YES)
            {
                needBackup = [converter needBackup];
            }
            else
            {
                // !! 沒有實作代表要備份
                needBackup = YES;
            }
            
            if(needBackup == YES)
            {
                break;
            }
        }
    }
    
    //////////////////////////////////////////////////
    
    self.completionBlock = completion;
    self.lastError = nil;
    __block typeof(self) blockSelf = self;
    
    //////////////////////////////////////////////////
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{

        // !! 注意這裡的dispatch，iOS用"sync"；macOS用"async"，所以才需另外處理finish。

#if TARGET_OS_IPHONE
        
        dispatch_sync(dispatch_get_main_queue(), ^{
            
            blockSelf.busyView = [[[PPBusyView alloc] initWithSuperView:nil] autorelease];
            
            if(blockSelf.busyView != nil)
            {
                blockSelf.busyView.backgroundColor = [UIColor clearColor];
                blockSelf.busyView.messageLabel.shadowColor = [UIColor clearColor];
                blockSelf.busyView.messageLabel.shadowOffset = CGSizeZero;
                blockSelf.busyView.messageLabel.textColor = blockSelf.messageColor ? blockSelf.messageColor : PPDCFC_DefaultMessageColor;
            }
            
            [[UIApplication sharedApplication] enableIdle:NO];
        });

        
        [blockSelf showBusyMessage:PPDCFC_MLS_BackingUpData];

        NSString *taskKey = @"startConvertFlow";
        [PPBackgroundTaskController addTaskWithKey:taskKey terminate:nil];
        
        BOOL canConvert = YES;
        
        // backup
        if(needBackup == YES)
        {
            [blockSelf showBusyMessage:PPDCFC_MLS_BackingUpData];
            canConvert = [blockSelf runBackup];
        }
        
        // convert
        if(canConvert == YES)
        {
            [blockSelf showBusyMessage:PPDCFC_MLS_ConvertingData];            
            [blockSelf runConvertWithConverters:todoConverters];
        }
        
        // finish
        [PPBackgroundTaskController removeTaskWithKey:taskKey];
        
        dispatch_sync(dispatch_get_main_queue(), ^{
            
            [[UIApplication sharedApplication] enableIdle:YES];
            [blockSelf.busyView removeFromSuperview];
            blockSelf.busyView = nil;
        });
        
#elif TARGET_OS_MAC

        __block BOOL isFinish = NO;
        
        dispatch_async(dispatch_get_main_queue(), ^{

            blockSelf.waitingWindowController = [[[PPWaitingWindowController alloc] init] autorelease];
            blockSelf.waitingWindowController.isIndeterminateProgress = NO;

            [blockSelf.waitingWindowController isCancelButtonShowed:NO];
            [blockSelf.waitingWindowController runModalWithWindowDidLoadHandler:^{

                dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_BACKGROUND, 0), ^{
                    
                    BOOL canConvert = YES;
                    
                    // backup
                    if(needBackup == YES)
                    {
                        [blockSelf performSelectorOnMainThread:@selector(showBusyMessage:) withObject:PPDCFC_MLS_BackingUpData waitUntilDone:YES];
                        canConvert = [blockSelf runBackup];
                    }
                    
                    // convert
                    if(canConvert == YES)
                    {
                        [blockSelf performSelectorOnMainThread:@selector(showBusyMessage:) withObject:PPDCFC_MLS_ConvertingData waitUntilDone:YES];
                        [blockSelf runConvertWithConverters:todoConverters];
                    }
                    
                    // finish (dispatch到mainThread無法關閉，要用performSelectorOnMainThread)
                    [blockSelf.waitingWindowController performSelectorOnMainThread:@selector(stopModal) withObject:nil waitUntilDone:YES];
                    blockSelf.waitingWindowController = nil;
                    isFinish = YES;
                });
            }];
        });
        
        while (isFinish == NO)
        {
            [NSThread waitWithTimeInterval:0.5];
        }
        
#endif
        
        //////////////////////////////////////////////////
        
        [todoConverters release];
        
        //////////////////////////////////////////////////

        dispatch_sync(dispatch_get_main_queue(), ^{

            if(blockSelf.lastError != nil)
            {
                switch (blockSelf.lastError.code)
                {
                    case ErrorCode_NotEnoughSpace:
                    case ErrorCode_FailedToBackup:
                    {
                        [blockSelf showRetryAlertView];
                        break;
                    }
                        
                    default:
                    {
                        [blockSelf showConvertFailedAlertView];
                        break;
                    }
                }
            }
            else
            {
                [blockSelf showConvertSuccessAlertView];
            }
        });
    });
}


//================================================================================
//
//================================================================================
- (void)finishConvertFlowWithSuccess:(BOOL)success
{
    if(success == YES)
    {
        [self addMainLog:@"convert success"];
    }
    else
    {
        [self addMainLog:@"convert abort"];
        
        // !! special case : 因為放棄轉換，所以要當作所有converter都已經完成。
        NSArray *converterKeys = [PPSettingsController arrayValueWithKey:SettingsKey_TodoConverterKeys];
        [PPSettingsController setArrayValue:converterKeys withKey:SettingsKey_FinishedConverterKeys];
    }
    
    [PPSettingsController setStringValue:[self appVersion] withKey:SettingsKey_LastConvertVersion];
    
    self.completionBlock(success);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods - show alert

//==============================================================================
//
//==============================================================================
- (void)showRetryAlertView
{
#if TARGET_OS_IPHONE
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:@""
                       message:self.lastError.localizedFailureReason
                      delegate:self
                           tag:AlertTag_OkWithExit
             cancelButtonTitle:PPDCFC_MLS_OK
             otherButtonTitles:nil];

#elif TARGET_OS_MAC
    
    NSAlert *alert = [[NSAlert alloc] init];
    
    [alert setMessageText:self.lastError.localizedFailureReason];
    [alert addButtonWithTitle:PPDCFC_MLS_OK];
    [alert runModal];
    [alert release];

#endif
}


//==============================================================================
//
//==============================================================================
- (void)showConvertFailedAlertView
{
#if TARGET_OS_IPHONE
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:PPDCFC_MLS_ConvertFailed
                       message:PPDCFC_MLS_ConvertFailedMessage
                      delegate:self
                           tag:AlertTag_AbortConvertOrRepoprtError
             cancelButtonTitle:PPDCFC_MLS_ReportError
             otherButtonTitles:PPDCFC_MLS_AbortConvert, nil];
    
#elif TARGET_OS_MAC
    
    NSAlert *alert = [[NSAlert alloc] init];
    
    [alert setMessageText:PPDCFC_MLS_ConvertFailed];
    [alert setInformativeText:PPDCFC_MLS_ConvertFailedMessage];
    [alert addButtonWithTitle:PPDCFC_MLS_ReportError];
    [alert addButtonWithTitle:PPDCFC_MLS_AbortConvert];
    
    switch ([alert runModal])
    {
        case NSAlertFirstButtonReturn: // ReportError
        {
            // 通知錯誤 (不可呼叫finishConvertFlowWithSuccess:回傳結果)
            [self addMainLog:@"Report error"];
            [self reportErrorLog];
            break;
        }

        case NSAlertSecondButtonReturn: // AbortConvert
        {
            // 放棄轉換，變成重新開始。(備份檔會留著)
            [[NSFileManager defaultManager] removeItemAtPath:self.dataDirPath error:nil];
            [self finishConvertFlowWithSuccess:YES];
            break;
        }

        default:
            break;
    }

    [alert release];

#endif
}


//==============================================================================
//
//==============================================================================
- (void)showConvertSuccessAlertView
{
#if TARGET_OS_IPHONE
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:@""
                       message:PPDCFC_MLS_ConvertSuccess
                      delegate:self
                           tag:AlertTag_ConvertSuccess
             cancelButtonTitle:PPDCFC_MLS_OK
             otherButtonTitles:nil];

#elif TARGET_OS_MAC
    
    NSAlert *alert = [[NSAlert alloc] init];
    
    [alert setMessageText:PPDCFC_MLS_ConvertSuccess];
    [alert addButtonWithTitle:PPDCFC_MLS_OK];
    [alert runModal];
    [alert release];

    [self finishConvertFlowWithSuccess:YES];
    
#endif
    
}

//==============================================================================
//
//==============================================================================
- (void)showNoEmailSettingAlert
{
#if TARGET_OS_IPHONE
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:@""
                       message:PPDCFC_MLS_NotSettingEmail
                      delegate:self
                           tag:AlertTag_NoEmailSetting
             cancelButtonTitle:PPDCFC_MLS_OK
             otherButtonTitles:nil];

#elif TARGET_OS_MAC
    
    NSAlert *alert = [[NSAlert alloc] init];
    
    [alert setMessageText:PPDCFC_MLS_NotSettingEmail];
    [alert addButtonWithTitle:PPDCFC_MLS_OK];
    [alert runModal];
    [alert release];

#endif
}





///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (void)startWithDataDirPath:(NSString *)dataDirPath
                  converters:(NSArray *)converters
                messageColor:(CPColor *)messageColor
             backgroundColor:(CPColor *)backgroundColor
                  completion:(void (^)(BOOL success))completion
{
    
#if TARGET_OS_IPHONE
    
    if([converters count] == 0 || [dataDirPath length] == 0)
    {
        completion(NO);
        return;
    }
    
    
    //////////////////////////////////////////////////
    // 路徑最後如果有"/"，先移除。
    
    if([dataDirPath hasSuffix:@"/"] == YES)
    {
        dataDirPath = [dataDirPath substringToIndex:[dataDirPath length]-1];
    }
    
    
    //////////////////////////////////////////////////
    // 建立轉換背景
    
    __block UIWindow *window = [[UIWindow alloc] initWithFrame:[[UIScreen mainScreen] bounds]];
    
    if(window != nil)
    {
        PPViewController *viewController = [[[PPViewController alloc] init] autorelease];
        viewController.view.backgroundColor = backgroundColor ? backgroundColor : PPDCFC_DefaultBackgroundColor;
        
        window.rootViewController = viewController;
        [window makeKeyAndVisible];
        
        //        window.rootViewController = [[[UIViewController alloc] init] autorelease];
        //        window.rootViewController.view.backgroundColor = backgroundColor ? backgroundColor : PPDCFC_DefaultBackgroundColor;
        //        [window makeKeyAndVisible];
    }
    
    
    //////////////////////////////////////////////////
    // 開始轉換流程
    
    PPDataConvertFlowController *controller = [[PPDataConvertFlowController alloc] init];
    
    if(controller != nil)
    {
        controller.dataDirPath = dataDirPath;
        controller.converters = converters;
        controller.messageColor = messageColor;
        
        [controller startConvertFlowWithCompletion:^(BOOL success){
            
            completion(success);
            [controller release];
            [window release];
        }];
    }
    else
    {
        completion(NO);
        [window release];
    }

#elif TARGET_OS_MAC
    
    NSAssert(NO, @"macOS請用 -startWithDataDirPath:converters:completion:");
    
#endif
}


//================================================================================
//
//================================================================================
+ (void)startWithDataDirPath:(NSString *)dataDirPath
            backupFileSuffix:(NSString *)backupFileSuffix
                  converters:(NSArray *)converters
                  completion:(void (^)(BOOL success))completion
{
#if TARGET_OS_IPHONE
    
    NSAssert(NO, @"iOS請用 -startWithDataDirPath:converters:messageColor:backgroundColor:completion:");
    
#elif TARGET_OS_MAC

    if([converters count] == 0 || [dataDirPath length] == 0)
    {
        completion(NO);
        return;
    }
    
    
    //////////////////////////////////////////////////
    // 路徑最後如果有"/"，先移除。
    
    if([dataDirPath hasSuffix:@"/"] == YES)
    {
        dataDirPath = [dataDirPath substringToIndex:[dataDirPath length]-1];
    }
    
    
    //////////////////////////////////////////////////
    // 開始轉換流程
    
    PPDataConvertFlowController *controller = [[PPDataConvertFlowController alloc] init];
    
    if(controller != nil)
    {
        controller.dataDirPath = dataDirPath;
        controller.backupFileSuffix = backupFileSuffix;
        controller.converters = converters;
        
        [controller startConvertFlowWithCompletion:^(BOOL success){
            
            completion(success);
            [controller release];
        }];
    }
    else
    {
        completion(NO);
    }
    
#endif
    
}


//===============================================================================
//
//===============================================================================
+ (NSString *)logDirPath
{
    NSArray	*paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
    NSString *dirPath = [[paths firstObject] stringByAppendingPathComponent:Log_FileNamePrefix];
    
    return dirPath;
}


//================================================================================
//
//================================================================================
+ (NSString *)backupDirPath
{
    NSArray	*paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
    NSString *dirPath = [[paths firstObject] stringByAppendingPathComponent:Backup_FileNamePrefix];
    
    if([[NSFileManager defaultManager] fileExistsAtPath:dirPath] == NO)
    {
        [[NSFileManager defaultManager] createDirectoryAtPath:dirPath withIntermediateDirectories:YES attributes:nil error:nil];
    }
    
    return dirPath;
}



@end
