//
//  PPCountrySelectorViewController.m
//
//  Created by  Eddie on 2011/6/28.
//  Fixed   by Howard on 2013/10/15.
//  Copyright 2011年 Penpower. All rights reserved.
//


#import "PPCountrySelectorViewController.h"

// View
#import "PPNavigationBarView.h"
#import "PPTableView.h"

// Controller
#import "PPCountryCodeConvert.h"

// Category
#import "NSString+Additions.h"
#import "PPButton+Factory.h"
#import "UIViewController+ShareApplication.h"

//controller
#import "PPSystemInfoController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger,PPCountrySelectorViewControllerButtonTag)
{
    PPCountrySelectorViewControllerButtonTag_Back,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPCountrySelectorViewController() <UITableViewDataSource,UITableViewDelegate>

@property (nonatomic,assign)   id<PPCountrySelectorViewControllerDelegate> delegate;
@property (nonatomic,retain)   NSArray      *regionCountryArray;
@property (nonatomic,retain)   NSArray      *countryCodesArray;

@property (nonatomic,retain)   PPButton            *backButton;
@property (nonatomic,retain)   PPTableView         *tableView;
@property (nonatomic,retain)   PPNavigationBarView *countrySelectorNavigationBarView;
@property (nonatomic,retain)   NSDictionary *countryNameDict;
@property (nonatomic,retain)   NSDictionary *regionNameDict;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPCountrySelectorViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//===============================================================================
//
//===============================================================================
- (id)init
{
    self = [super init];
    
    if(self!=nil)
    {
        NSString* systemLang = [PPSystemInfoController stringWithLanguage:[PPSystemInfoController language]];
        
        [self prepareCountryDataWithLanguageCode:systemLang];
    }
    
    return self;
}


//===============================================================================
//
//===============================================================================
- (id)initWithFrame:(CGRect)frame delegate:(id)delegate;
{
    self = [self init];
    
    if (self) 
    {
        _delegate = delegate;
        _regionCountryArray = nil;
        _countryCodesArray = nil;
        _currentCountryCode = nil;
        
        NSString* systemLang = [PPSystemInfoController stringWithLanguage:[PPSystemInfoController language]];

        [self prepareCountryDataWithLanguageCode:systemLang];
    }
    return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
    [self.tableView setDataSource:nil];
    [self.tableView setDelegate:nil];
    [self.tableView removeFromSuperview];
    self.tableView = nil;
    
    [_backButton removeControlEvents:UIControlEventTouchUpInside];
    [_backButton removeFromSuperview];
    [_backButton release];
    _backButton = nil;
    
    [_countrySelectorNavigationBarView removeFromSuperview];
    [_countrySelectorNavigationBarView release];
    _countrySelectorNavigationBarView = nil;
    
    [_currentCountryCode release];
    _currentCountryCode = nil;
    
    [_backButtonHighlightedBackgroundColor release];
    _backButtonHighlightedBackgroundColor = nil;

    [self freeCountryData];
    
    //////////////////////////////////////////////////
    
    [super dealloc];    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//===============================================================================
//
//===============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    
    if(self.tableView==nil)
    {
        _tableView = [[PPTableView alloc] initWithFrame:CGRectZero style:UITableViewStylePlain];
    }
    
    if(self.tableView!=nil)
    {
        self.tableView.dataSource = self;
        self.tableView.delegate = self;
        self.tableView.rowHeight = PPCS_RowHeight;
        [self.view addSubview:self.tableView];
    }
    
    //////////////////////////////////////////////////
    
    if(self.navigationController!=nil)
    {
        if(self.countrySelectorNavigationBarView==nil)
        {
            _countrySelectorNavigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
        }
    
        if(self.countrySelectorNavigationBarView!=nil)
        {
            self.countrySelectorNavigationBarView.style = PPBarViewStyle_CenteredAbsolute;
            
            //////////////////////////////////////////////////
            
            self.backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                              tag:0
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:self.backButtonHighlightedBackgroundColor
                                                  imageEdgeInsets:self.imageEdgeInsets];
            
            if(self.backButton!=nil)
            {
                self.backButton.delegate = self;
                
                [self.backButton addControlEvents:UIControlEventTouchUpInside];
                
                if(self.backButton.imageView.image==nil)
                {
                    [self.backButton setTitle:[@"MLS_Back" localizedFromTable:NSStringFromClass(self.backButton.class)] maxWidth:0.0];
                }
                
                //////////////////////////////////////////////////

                PPBarViewItemModel *backButtonItemModel = [PPBarViewItemModel ppBarViewItemModelWithView:_backButton];
                
                if(backButtonItemModel!=nil)
                {
                    backButtonItemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                }
                
                //////////////////////////////////////////////////
                
                if(backButtonItemModel!=nil)
                {
                    [self.countrySelectorNavigationBarView setItemModels:@[backButtonItemModel] forBlockType:PPBarViewBlockType_Left];
                }
            }
            
            //////////////////////////////////////////////////
       
            UIBarButtonItem *leftBarButtonItem = [[UIBarButtonItem alloc] initWithCustomView:self.countrySelectorNavigationBarView];
            
            self.navigationItem.leftBarButtonItem = leftBarButtonItem;
            
            [leftBarButtonItem release];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    self.tableView.frame = [self layoutFrame];

    //////////////////////////////////////////////////
    
    NSIndexPath *indexPath = nil;
    
    //------------------------------------------
    // find and move to current country code
    //------------------------------------------
    if([self.currentCountryCode length])
    {
        int section=0, row=0;
        
        for(NSArray *codeArray in _countryCodesArray)
        {
            for(NSString *code in codeArray)
            {
                if([code isEqualToString:[self.currentCountryCode lowercaseString]])
                {
                    indexPath = [NSIndexPath indexPathForRow:row inSection:section];
                    break;
                }
                
                row++;
            }
            
            section++;
            row = 0;
        }
    }
    
    if(indexPath)
    {
        [self.tableView selectRowAtIndexPath:indexPath animated:NO scrollPosition:UITableViewScrollPositionMiddle];
    }
}


//===============================================================================
//
//===============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [super viewDidDisappear:animated];
    
    //////////////////////////////////////////////////
    
    [self.tableView setDataSource:nil];
    [self.tableView setDelegate:nil];
    [self.tableView removeFromSuperview];
    self.tableView = nil;
    
    //////////////////////////////////////////////////

    [self.countrySelectorNavigationBarView removeFromSuperview];
    self.countrySelectorNavigationBarView = nil;
    
    [self.backButton removeControlEvents:UIControlEventTouchUpInside];
    [self.backButton removeFromSuperview];
    self.backButton = nil;
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//===============================================================================
//
//===============================================================================
- (void)prepareCountryDataWithLanguageCode:(NSString *)languageCode
{
    [self freeCountryData];

    //////////////////////////////////////////////////

    __block typeof(self) blockself = self;
    
    [PPCountryCodeConvert countryListForLanguageCode:languageCode
                                          completion:^(NSDictionary *countryNameDictionary, NSDictionary *regionNameDictionary, NSArray *countryCodes, NSArray *regionCountries) {
                                             
                                              blockself.countryNameDict = countryNameDictionary;
                                              blockself.regionNameDict = regionNameDictionary;
                                              blockself.countryCodesArray = countryCodes;
                                              blockself.regionCountryArray = regionCountries;
                                          }];
}


//===============================================================================
//
//===============================================================================
- (void)freeCountryData
{
    [_regionNameDict release];
    _regionNameDict = nil;
    
    [_regionCountryArray release];
    _regionCountryArray = nil;
    
    [_countryCodesArray release];
    _countryCodesArray = nil;
    
    [_countryNameDict release];
    _countryNameDict = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//===============================================================================
//
//===============================================================================
- (void)setDisplayLanguageWithLanguageCode:(NSString *)languageCode
{
    if([languageCode length])
        [self prepareCountryDataWithLanguageCode:languageCode];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource Method

//===============================================================================
//
//===============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
        case PPCountrySelectorViewControllerButtonTag_Back:
        {
            if(self.navigationController!=nil)
            {
                [self.navigationController popViewControllerAnimated:YES];
            }
            else
            {
                if([self respondsToSelector:@selector(dismissModalViewControllerAnimated:)]==YES)
                {
                    [self performSelector:@selector(dismissModalViewControllerAnimated:)
                               withObject:[NSNumber numberWithBool:YES]];
                }
                else
                {
                    [self dismissViewControllerAnimated:YES completion:nil];
                }
            }
            break;
        }
        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource Method

//===============================================================================
//
//===============================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return [self.regionCountryArray count];
}


//===============================================================================
//
//===============================================================================
- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    NSString *regionID = [self.regionCountryArray objectAtIndex:section];
    return [self.regionNameDict objectForKey:regionID];
}


//===============================================================================
//
//===============================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [[self.countryCodesArray objectAtIndex:section] count];
}


//===============================================================================
//
//===============================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    static NSString *CellIdentifier = @"Cell";
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    
    if (cell == nil)
        cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier] autorelease];

    
    // Configure the cell...
    NSArray *countryCodes = [self.countryCodesArray objectAtIndex:[indexPath section]];
    NSString *code = [countryCodes objectAtIndex:[indexPath row]];
    
    cell.textLabel.text = [self.countryNameDict objectForKey:code];
    
    if([code isEqualToString:[self.currentCountryCode lowercaseString]])
        cell.accessoryType = UITableViewCellAccessoryCheckmark;
    else cell.accessoryType = UITableViewCellAccessoryNone;
    
    return cell;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDelegate Method

//===============================================================================
//
//===============================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
    
    if([self.delegate respondsToSelector:@selector(ppCountrySelectorTableViewDidSelectCountry:countryName:)])
    {
        NSArray *codeArray = [self.countryCodesArray objectAtIndex:[indexPath section]];
        
        NSString *code    = [codeArray objectAtIndex:[indexPath row]];
        NSString *country = [self.countryNameDict objectForKey:code];
        
        [self.delegate ppCountrySelectorTableViewDidSelectCountry:[code lowercaseString] countryName:country];
    }        
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - class methods

//===============================================================================
//
//===============================================================================
+ (NSLocale *)copyLocaleFromLanguageCode:(NSString *)langCode
{
    if([langCode length])
    {
        // find localIdentifier by country code
        NSArray *localeArray = [NSLocale availableLocaleIdentifiers];
        NSString *foundLocale = nil;
        NSRange range = [langCode rangeOfString:@"-"];
        
        
        if(range.length)
        {
            langCode = [langCode stringByReplacingOccurrencesOfString:@"-" withString:@"_"];
            
            for(NSString *localeString in localeArray)
            {
                if([[localeString lowercaseString] isEqualToString:[langCode lowercaseString]])
                {
                    foundLocale = localeString;
                    break;
                }
                
                if(foundLocale)
                    break;
            }
        }
        else
        {
            for(NSString *localeString in localeArray)
            {
                NSArray *compArray = [localeString componentsSeparatedByString:@"_"];
                
                for(NSString *comp in compArray)
                {
                    if([[comp lowercaseString] isEqualToString:[langCode lowercaseString]])
                    {
                        foundLocale = localeString;
                        break;
                    }
                }
                
                if(foundLocale)
                    break;
            }       
        }
        
        
        
        if(foundLocale)
        {
            //test
                        //NSLog(@"foundLocale : %@", foundLocale);
            
            return [[NSLocale alloc] initWithLocaleIdentifier:foundLocale];
        }
    }
    
    return [[NSLocale currentLocale] retain];
}

@end
