//
//  PPGoogleDriveOperation_CreateFolder.m
//  
//

#import "PPGoogleDriveOperation_CreateFolder.h"
#import "PPCloud_GoogleDrive.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPGoogleDriveOperation_LoadMetadata()

@interface PPGoogleDriveOperation_CreateFolder ()
@property (atomic, assign) BOOL isLastAction;
@property (atomic, assign) BOOL isDelegateReceived;
@property (atomic, retain) GTLRDrive_File * delegateFile;
@property (atomic, retain) NSError *delegateError;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPGoogleDriveOperation_CreateFolder

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.createFolder = nil;
    self.delegateFile = nil;
    self.delegateError = nil;
	
	[super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Executing the Operation

//================================================================================
//
//================================================================================
- (void)main
{
    @autoreleasepool
    {
        NSError *error = nil;
        
        do
        {
            if(self.createFolder == nil)
            {
                error = PPErrorParameterInvalidity(nil);
                break;
            }
            
            //////////////////////////////////////////////////

            NSDictionary *identifierForFolderDictionary = [GTLRDriveService sharedServiceDrive].identifierForFolderDictionary;
            
            NSString *fileID = [identifierForFolderDictionary objectForKey:self.createFolder];
            
            
            GTLRDrive_File *file = [[GTLRDriveService sharedServiceDrive] fileForIdentifier:fileID
                                                                                      error:nil];
            
            //!!目錄已存在
            if(file!=nil)
            {
                error = PPErrorMake(PPCloudCommonError_PathAlreadyExist, @"Folder already exist", nil);
        
                break;
            }
            
            //////////////////////////////////////////////////

            NSArray *pathComponents = [self.createFolder pathComponents];
            
            //////////////////////////////////////////////////
            // 目前看起來還是只能一層一層建立
            
            NSMutableArray *createComponents = [NSMutableArray array];
            
            [createComponents addObject:[pathComponents firstObject]];
            
            for (int i=1; i<[pathComponents count]; i++)
            {
                [createComponents addObject:[pathComponents objectAtIndex:i]];
                
                //////////////////////////////////////////////////
                // Do not create folder if name exist
                
                // !! 這裡要用獨立的error，不然會干擾。
                
                NSError *createFolderError = nil;
                NSArray *createFolderFiles = [[GTLRDriveService sharedServiceDrive] filesForPathComponents:createComponents
                                                                                   currentPathComponentID:nil
                                                                                                    error:&createFolderError];
                
                //////////////////////////////////////////////////

                // 詢問目錄存在失敗，不允許繼續建立流程(有可能 Google 回傳沒建立成功)
                
                if(createFolderError!=nil)
                {
                    error = PPErrorMake(PPCloudCommonError_CheckIfFolderExists, @"Check if a folder exists fail", createFolderError);
                    
                    break;
                }
                
                //////////////////////////////////////////////////

                if([createFolderFiles count]>=1)
                {
                    error = PPErrorMake(PPCloudCommonError_PathAlreadyExist, @"Folder already exist", nil);
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                self.isDelegateReceived = NO;
                self.delegateFile = nil;
                self.delegateError = nil;
                
                error = [self createFolder:createComponents];
                
                if(error != nil)
                {
                    break;
                }
                else
                {
                    
                    while (self.isDelegateReceived == NO)
                    {
                        [self waitWithTimeInterval:PPCloudOperation_WaitInterval];
                    }
                    
                    if(self.delegateError != nil)
                    {
                        error = self.delegateError;
                        break;
                    }
                }
            }
            
        }while(0);
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        if(error != nil)
        {
            if([self.delegate respondsToSelector:@selector(ppGoogleDriveOperation:createFolderFailedWithError:)]==YES)
            {
                [self.delegate ppGoogleDriveOperation:self createFolderFailedWithError:error];
            }
            
        }
        else
        {
            if([self.delegate respondsToSelector:@selector(ppGoogleDriveOperation:createdFolder:)]==YES)
            {
                [self.delegate ppGoogleDriveOperation:self createdFolder:[NSDictionary dictionaryWithObjectsAndKeys:self.delegateFile, NSStringFromClass([self.delegateFile class]), nil]];
            }
        }

        [self completion];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private

//================================================================================
//
//================================================================================
- (NSError *)createFolder:(NSArray *)pathComponents
{
    NSError *error = nil;
    
    do
    {
        //////////////////////////////////////////////////
        
        NSMutableArray *parentComponents = [NSMutableArray arrayWithArray:pathComponents];
        [parentComponents removeLastObject];
        
        NSArray *createFolderParentFiles = [[GTLRDriveService sharedServiceDrive] filesForPathComponents:parentComponents
                                                                                 currentPathComponentID:nil
                                                                                                  error:&error];
        if(error!=nil)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        if([createFolderParentFiles count]<1)
        {
            error = PPErrorMake(PPCloudCommonError_PathAlreadyExist, @"Folder not found", nil);
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        if([createFolderParentFiles count]>1)
        {
            error = PPErrorMake(PPCloudCommonError_MultipleItemsWithSamePath, @"Folder already exist", nil);
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        GTLRDrive_File *createFolderParentFile = [createFolderParentFiles objectAtIndex:0];
        
        //Parent一定要是目錄
        if([createFolderParentFile.mimeType isEqualToString:GTLDriveFile_MimeType_Folder]==NO)
        {
            error = PPErrorParameterInvalidity(nil);
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        GTLRDrive_File *createFolderFile = [GTLRDrive_File object];
        if(createFolderFile==nil)
        {
            error = PPErrorOperationFailed(nil);
            break;
        }
        
        createFolderFile.name       = [pathComponents lastObject];
        createFolderFile.mimeType   = GTLDriveFile_MimeType_Folder;
        createFolderFile.parents    = [NSArray arrayWithObjects:createFolderParentFile.identifier, nil];
        
        if(createFolderFile.parents==nil)
        {
            error = PPErrorOperationFailed(nil);
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        GTLRDriveQuery_FilesCreate *createFolderQuery= [GTLRDriveQuery_FilesCreate queryWithObject:createFolderFile uploadParameters:nil];
   
        if(createFolderQuery==nil)
        {
            error = PPErrorOperationFailed(nil);
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        GTLRServiceTicket *ticket = [[GTLRDriveService sharedServiceDrive] executeQuery:createFolderQuery
                                                                             delegate:self
                                                                    didFinishSelector:@selector(serviceTicket:finishedWithFile:error:)];
        if(ticket==nil)
        {
            error = PPErrorOperationFailed(nil);
            break;
        }
        
    }
    while(0);
    
    return error;
}


//================================================================================
//
//================================================================================
- (void)serviceTicket:(GTLRServiceTicket *)ticket finishedWithFile:(GTLRDrive_File *)file error:(NSError *)error
{
    self.delegateFile = file;
    self.delegateError = error;
    self.isDelegateReceived = YES;
}

@end
