//
//  PPCameraView+iOS.m
//  
//
//  Created by Howard on 2017/7/31.
//
//

#import "PPCameraView+iOS.h"
#import "UIImage+Additions.h"
#import "PPCameraView+ResourceDefine.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static inline double radians (double degrees) {return degrees * M_PI/180;}
static NSArray *s_CameraAvailableDeviceTypes = nil;
@implementation PPCameraView (iOS)





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Class Method

//================================================================================
//
//================================================================================
+ (void)showCameraAuthorizationAlertWithViewController:(UIViewController *)viewController
{
    NSString *message = PPCV_MLS_CameraAuthorizationAlert;
    NSString *okTitle = PPCV_MLS_OK;
    __block UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"" message:message preferredStyle:UIAlertControllerStyleAlert];
    
    if(alertController != nil)
    {
        UIAlertAction *okAction = [UIAlertAction actionWithTitle:okTitle style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
            
            [[UIApplication sharedApplication] openURL:[NSURL URLWithString:UIApplicationOpenSettingsURLString] options:@{} completionHandler:nil];
        }];
        
        [alertController addAction:okAction];
        [viewController presentViewController:alertController animated:YES completion:nil];
    }
}


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public Method

//===============================================================================
// Create a UIImage from sample buffer
//===============================================================================
- (UIImage *)copyImageFromSampleBuffer:(CMSampleBufferRef)sampleBuffer
{
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    
    // Get a CMSampleBuffer's Core Video image buffer for the media data
    CVImageBufferRef imageBuffer = CMSampleBufferGetImageBuffer(sampleBuffer);
    // Lock the base address of the pixel buffer
    CVPixelBufferLockBaseAddress(imageBuffer, 0);
    
    // Get the number of bytes per row for the pixel buffer
    void *baseAddress = CVPixelBufferGetBaseAddress(imageBuffer);
    
    // Get the number of bytes per row for the pixel buffer
    size_t bytesPerRow = CVPixelBufferGetBytesPerRow(imageBuffer);
    
    // Get the pixel buffer width and height
    size_t width = CVPixelBufferGetWidth(imageBuffer);
    size_t height = CVPixelBufferGetHeight(imageBuffer);
    
    // Create a device-dependent RGB color space
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    
    // Create a bitmap graphics context with the sample buffer data
    CGContextRef context = CGBitmapContextCreate(baseAddress, width, height, 8, bytesPerRow, colorSpace, kCGBitmapByteOrder32Little | kCGImageAlphaPremultipliedFirst);
    
    // Create a Quartz image from the pixel data in the bitmap graphics context
    CGImageRef quartzImage = CGBitmapContextCreateImage(context);
    
    // Unlock the pixel buffer
    CVPixelBufferUnlockBaseAddress(imageBuffer,0);
    
    // Free up the context and color space
    CGContextRelease(context);
    CGColorSpaceRelease(colorSpace);
    
    // Create an image object from the Quartz image
    UIImage *image = [[UIImage alloc] initWithCGImage:quartzImage];
    
    // Release the Quartz image
    CGImageRelease(quartzImage);
    
    
    UIImage *adjustedImage = nil;
    
    //Dock 模式不轉
    if(self.adjustOutputImageOrientation==YES)
    {
        adjustedImage = [self copyOrientationAdjustedImageFromImage:image];
    }
    else
    {
        adjustedImage = image;
        
        [adjustedImage retain];
    }
    
    [image release];
    [pool release];
    
    return adjustedImage;
}


//===============================================================================
// capture影像需要以interface Orientation進行修正
// RETURNED: UIImage, must free manually
//===============================================================================
- (UIImage *)copyOrientationAdjustedImageFromImage:(UIImage *)srcImage
{
    //----------------------------------------------
    // handle special condition
    //----------------------------------------------
    if(!srcImage)
        return nil;
    
    // !! 因為回傳物件被始用完畢後會被release，所以統一將srcImage先retain一次
    [srcImage retain];
    
    
    //----------------------------------------------
    // adjust image
    //----------------------------------------------
    CGContextRef		bitmap = nil;
    CGImageRef			imageRef = [srcImage CGImage];
    CGSize				srcSize = srcImage.size;
    CGSize              newSize;
    CGBitmapInfo		bitmapInfo = kCGBitmapAlphaInfoMask & kCGImageAlphaNoneSkipLast;
    CGColorSpaceRef		colorSpaceInfo = CGColorSpaceCreateDeviceRGB();
    size_t				componentCount = 4;
    size_t				bitsPerComponent = 8;
    size_t				bytesPerRow = 0;
    
    
    switch(self.interfaceOrientation)
    {
        case UIInterfaceOrientationPortrait:
        {
            newSize.width = srcSize.height;
            newSize.height = srcSize.width;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            CGContextRotateCTM(bitmap, radians(270));
            CGContextTranslateCTM(bitmap, -srcSize.width, 0);
            
            break;
        }
            
        case UIInterfaceOrientationPortraitUpsideDown:
        {
            newSize.width = srcSize.height;
            newSize.height = srcSize.width;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            
            CGContextRotateCTM(bitmap, radians(90));
            CGContextTranslateCTM(bitmap, 0, -srcSize.height);
            
            break;
        }
            
        case UIInterfaceOrientationLandscapeLeft:
        {
            newSize.width = srcSize.width;
            newSize.height = srcSize.height;
            bytesPerRow = componentCount * newSize.width;
            bitmap = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpaceInfo, bitmapInfo);
            CGContextRotateCTM(bitmap, radians(180));
            CGContextTranslateCTM(bitmap, -srcSize.width, -srcSize.height);
            
            break;
        }
            
        default : //UIInterfaceOrientationLandscapeRight
        {
            CGColorSpaceRelease(colorSpaceInfo);
            return srcImage;
        }
    }
    
    
    if(!bitmap)
    {
        CGColorSpaceRelease(colorSpaceInfo);
        return srcImage;
    }
    
    // !! 回傳新影像，把srcImage增加retain count還原
    [srcImage release];
    
    // 建立新影像
    CGContextDrawImage(bitmap, CGRectMake(0, 0, srcSize.width, srcSize.height), imageRef);
    CGImageRef ref = CGBitmapContextCreateImage(bitmap);
    UIImage* newImage = [[UIImage alloc] initWithCGImage:ref];
    CGColorSpaceRelease(colorSpaceInfo);
    CGContextRelease(bitmap);
    CGImageRelease(ref);
    
    return newImage;
}


//==============================================================================
// 傳入的image應該與preview方向相同
//==============================================================================
- (UIImage *)cropImageWithPreviewSize:(CGSize)previewSize image:(UIImage *)image
{
    CGSize imageSize = [image size];
    
    CGFloat widthScale = MIN(imageSize.width,imageSize.height)/MIN(previewSize.width, previewSize.height);
    CGFloat heightScale = MAX(imageSize.width,imageSize.height)/MAX(previewSize.width, previewSize.height);
    
    // 如果比例接近就不調整
    if (fabs(widthScale-heightScale)<0.01)
    {
        return image;
    }
    
    CGSize outputImageSize = CGSizeZero;
    CGFloat scale = fmin(widthScale, heightScale);
    
    // !! cameraView跟著轉與沒有轉的計算方向不同
    if (self.viewOrientationFix==NO)
    {
        outputImageSize = CGSizeMake(previewSize.width * scale, previewSize.height * scale);
    }
    else
    {
        if (imageSize.width<imageSize.height)
        {
            outputImageSize = CGSizeMake(previewSize.width * scale, previewSize.height * scale);
        }
        else
        {
            outputImageSize = CGSizeMake(previewSize.height * scale, previewSize.width * scale);
        }
    }
    
    CGRect cropRect = CGRectMake((imageSize.width-outputImageSize.width)/2, (imageSize.height-outputImageSize.height)/2, outputImageSize.width, outputImageSize.height);
    return [image imageCroppedByRect:cropRect];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public class methods


//================================================================================
//
//================================================================================
+ (void)checkCameraAuthorizationWithViewController:(UIViewController *)viewController
                                completionHanadler:(CheckAuthorizationCompletionHandler)handler
{
    AVAuthorizationStatus status = [AVCaptureDevice authorizationStatusForMediaType:AVMediaTypeVideo];
    
    switch(status)
    {
        case AVAuthorizationStatusAuthorized:
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                
                handler(YES);
            });
            
            break;
        }
            
        case AVAuthorizationStatusNotDetermined:
        {
            [AVCaptureDevice requestAccessForMediaType:AVMediaTypeVideo completionHandler:^(BOOL granted) {
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    if(granted == YES)
                    {
                        handler(YES);
                    }
                    else
                    {
                        [self showCameraAuthorizationAlertWithViewController:viewController];
                        handler(NO);
                    }
                });
                
            }];
            
            break;
        }
            
        default:
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [self showCameraAuthorizationAlertWithViewController:viewController];
                handler(NO);
            });
            
            break;
        }
            
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - 鏡頭相關


//==============================================================================
//
//==============================================================================
+ (AVCaptureDevice *)ultraWideCamera
{
    if(@available(iOS 13, *))
    {
        return [AVCaptureDevice defaultDeviceWithDeviceType:AVCaptureDeviceTypeBuiltInUltraWideCamera
                                                  mediaType:AVMediaTypeVideo
                                                   position:AVCaptureDevicePositionUnspecified];
    }
    else
    {
        return nil;
    }
}


//==============================================================================
//
//==============================================================================
+ (AVCaptureDevice *)wideAngleCamera
{
    if(@available(iOS 13, *))
    {
        return [AVCaptureDevice defaultDeviceWithDeviceType:AVCaptureDeviceTypeBuiltInWideAngleCamera
                                                  mediaType:AVMediaTypeVideo
                                                   position:AVCaptureDevicePositionUnspecified];
    }
    else
    {
        return nil;
    }
}


//==============================================================================
//
//==============================================================================
+ (BOOL)isSupportTripleCamera
{
    if(@available(iOS 13, *))
    {
        return ([AVCaptureDevice defaultDeviceWithDeviceType:AVCaptureDeviceTypeBuiltInTripleCamera
                                                    mediaType:AVMediaTypeVideo
                                                     position:AVCaptureDevicePositionBack]!=nil);
    }
    else
    {
        return NO;
    }
}


//==============================================================================
//
//==============================================================================
+ (void)setAvailableDeviceTypes:(NSArray *)availableDeviceTypes
{
    [availableDeviceTypes retain];
    [s_CameraAvailableDeviceTypes release];
    s_CameraAvailableDeviceTypes = availableDeviceTypes;
}


//==============================================================================
//
//==============================================================================
+ (NSArray *)availableDeviceTypes
{
    return s_CameraAvailableDeviceTypes;
}


@end
