//
//  PPCameraMedia.m
//
//  Created by pptai on 13/4/3.
//  Copyright (c) 2013年 penpower. All rights reserved.


#import "PPCameraView.h"

#if TARGET_OS_IPHONE

#import "UIImage+Additions.h"
#import "PPCameraView+iOS.h"
#import <CoreMotion/CoreMotion.h>

// HDR required
#import "PPHDRGenerator.h"
#import "UIDevice+DeviceModel.h"
#import <MetalKit/MTKTextureLoader.h>

#elif TARGET_OS_MAC

#import "NSImage+Additions.h"
#import "PPCameraView+macOS.h"

#endif



NSString *const PPCameraView_focus = @"PPCameraView_focus";

#if TARGET_OS_IPHONE
static float PPCameraView_LastBias = 1.0;
#elif TARGET_OS_MAC


#endif
////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPCameraView() <
AVCaptureVideoDataOutputSampleBufferDelegate,
AVCapturePhotoCaptureDelegate,
AVCaptureMetadataOutputObjectsDelegate>

#pragma mark - Common Property


@property (nonatomic,retain) AVCaptureDevice            *cameraDevice;
@property (nonatomic,retain) AVCaptureVideoDataOutput   *captureVideoDataOutput;
@property (nonatomic,retain) AVCaptureMetadataOutput    *captureMetadataOutput;

@property (nonatomic,retain) AVCaptureVideoPreviewLayer *previewLayer;

@property (nonatomic,assign) BOOL                       canCaptureStillImage;
#if TARGET_OS_IPHONE

@property (nonatomic,retain) AVCapturePhotoOutput       *capturePhotoOutput;

// Flash
@property (nonatomic,assign) AVCaptureFlashMode         flashMode;

// HDR
@property (nonatomic,assign) PPCameraHDRMode           hdrMode;
@property (nonatomic,retain) id<MTLDevice>             mtlDevice;
@property (nonatomic,retain) PPHDRGenerator            *hdrGenerator;
@property (nonatomic, retain) MTKTextureLoader         *loader; // 纹理加载器
@property (nonatomic, retain) NSMutableArray           *bracketImages;  //暫存bracket image

@property (nonatomic,retain) UIImageView               *focusImageView;
@property (nonatomic,retain) UITapGestureRecognizer    *singleTapGestureRecognizer;

#pragma mark capture stable check

@property (nonatomic,assign) NSInteger                 stableCounter;
@property (nonatomic,retain) CMMotionManager           *motionManager;
@property (nonatomic,retain) CMAccelerometerData       *preAccelerationData;

#pragma mark preview stable check

@property (atomic, assign) BOOL isDeviceStable;
@property (nonatomic,assign) NSInteger                 previewStableCounter;
@property (nonatomic,retain) CMMotionManager           *previewMotionManager;
@property (nonatomic,retain) CMAccelerometerData       *prePreviewAccelerationData;

#elif TARGET_OS_MAC

//captureStillImageOutput 改capturePhotoOutput，方便整合code
@property (nonatomic,retain) AVCaptureStillImageOutput *capturePhotoOutput;

#endif

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPCameraView

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Creating, Copying, and Creating Objects


//==============================================================================
//
//==============================================================================
- (id)initWithCoder:(NSCoder *)coder
{
    self = [super initWithCoder:coder];
    if(self)
    {
        [self initProcess];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    if(self!=nil)
    {
        [self initProcess];
    }

    return self;
}


//==============================================================================
//
//==============================================================================
- (void)initProcess
{
    self.sessionPreset = AVCaptureSessionPresetPhoto;
    self.needCropImageByPreviewSize = YES;
    
    
#if TARGET_OS_IPHONE
    self.hdrMode = PPCameraHDRMode_Off;
    self.bracketImages = [NSMutableArray array];
    // hdr
#if !(TARGET_IPHONE_SIMULATOR)
    self.mtlDevice = [MTLCreateSystemDefaultDevice() autorelease];
    self.hdrGenerator = [[[PPHDRGenerator alloc] initWithDevice:self.mtlDevice] autorelease];
    self.loader = [[[MTKTextureLoader alloc] initWithDevice:self.mtlDevice] autorelease];
#endif
    self.interfaceOrientation = UIInterfaceOrientationPortrait;
    self.ppCameraType         = PPCameraType_Back;
    self.enableFocusTouch     = YES;
    self.motionManager        = [[[CMMotionManager alloc] init] autorelease];
    self.previewMotionManager = [[[CMMotionManager alloc] init] autorelease];
   
    // 這三個在init時不能用property做, 因為self.camerdDevice還沒建好，會設定不進去
    // init只單純設值，所以需要在cameraDevice建好後，再call applyCameraSettings
    _focusMode            = AVCaptureFocusModeContinuousAutoFocus;
    _exposureMode         = AVCaptureExposureModeContinuousAutoExposure;
    _whiteBalanceMode     = AVCaptureWhiteBalanceModeContinuousAutoWhiteBalance;
    _videoZoomFactor      = 1.0;
    //////////////////////////////////////////////////
    
    NSString *focusImagePath = [[NSBundle mainBundle] pathForResource:PPCameraView_focus ofType:@"png"];
    self.focusImageView = [[[UIImageView alloc] initWithImage:[UIImage imageWithContentsOfFile:focusImagePath]] autorelease];
    
    if(_focusImageView!=nil)
    {
        _focusImageView.alpha           = 0;
        [self addSubview:_focusImageView];
    }
    
    //////////////////////////////////////////////////
    // register tap gesture
    
    self.singleTapGestureRecognizer = [[[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(handleTap:)] autorelease];
    
    if(self.singleTapGestureRecognizer!=nil)
    {
        self.singleTapGestureRecognizer.numberOfTapsRequired = 1;
        [self addGestureRecognizer:self.singleTapGestureRecognizer];
    }

#elif TARGET_OS_MAC
    self.wantsLayer = YES;
#endif
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
#if TARGET_OS_IPHONE
    //////////////////////////////////////////////////
    
    [[self class] setAvailableDeviceTypes:nil];
    self.bracketImages = nil;
    
    self.mtlDevice = nil;
    self.hdrGenerator = nil;
    
    [self stopStableCheck];
    self.motionManager = nil;

    [self setEnablePreviewStableCheck:NO];
    self.previewMotionManager = nil;
    
    
    [self.focusImageView removeFromSuperview];
    self.focusImageView = nil;
    
    [self removeGestureRecognizer:self.singleTapGestureRecognizer];
    self.singleTapGestureRecognizer = nil;

    self.prePreviewAccelerationData = nil;

    self.loader = nil;
    //////////////////////////////////////////////////
    
#if !(TARGET_IPHONE_SIMULATOR)
    [self stopCameraPreview];
    [self turnOnFlash:NO useAutoMode:NO];
    [self turnOnTorch:NO];
#endif
    
#else
    [self.previewLayer removeFromSuperlayer];
    self.previewLayer = nil;
    self.cameraDevice = nil;
    
    [self.session stopRunning];
    self.session = nil;

    //////////////////////////////////////////////////
#endif
    self.sessionPreset = nil;
    self.cameraDeviceName = nil;
    self.capturePhotoOutput = nil;
    self.captureVideoDataOutput = nil;
    self.captureMetadataOutput = nil;
    self.captureMovieFileOutput = nil;
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Layout of subview

#if TARGET_OS_IPHONE
//================================================================================
// iOS
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    // 變更preview大小
    if(self.previewLayer != nil)
    {
        self.previewLayer.frame = self.bounds;
    }
}

#elif TARGET_OS_MAC
//================================================================================
//
//================================================================================
- (void)layout
{
    [super layout];
    
    //////////////////////////////////////////////////
    
    self.previewLayer.frame = self.bounds;
}

#endif



//================================================================================
//
//================================================================================
- (void)mainThreadDelegateDidGetStillImage:(CPImage *)image
{
	if([self.delegate respondsToSelector:@selector(ppCameraView:didGetStillImage:)])
	{
		[self.delegate ppCameraView:self didGetStillImage:image];
	}
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private still image Method

#if TARGET_OS_IPHONE


//================================================================================
//
//================================================================================
- (CPImage *)imageWithPhoto:(AVCapturePhoto *)photo
{
    CPImage *resultImage = nil;
    
    if(photo == nil)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    
    // 調整影像速度約0.02sec
    NSData *imageData = [photo fileDataRepresentation];

    // !! 取得的影像都是UIInterfaceOrientationLandscapeRight的方向，要依據目前的實際方向進行調整
    UIImageOrientation adjustOrientation = [self currentImageOrientation];
    CPImage *oriImage = [CPImage imageWithData:imageData];
    
    resultImage = [CPImage imageWithCGImage:oriImage.CGImage scale:1.0 orientation:adjustOrientation];

    
    //////////////////////////////////////////////////
    // 調整輸出影像大小
    // (原生image在核心處理時方向有問題，所以都要透過imageByAdjustOrientationWithMaxLength重建bmp)
    
    if (self.maxStillImageLength!=0)
    {
        CGFloat maxImageLength = fmaxf([resultImage size].width, [resultImage size].height);
        
        maxImageLength = fminf(maxImageLength, self.maxStillImageLength);
        resultImage = [resultImage imageByAdjustOrientationWithMaxLength:maxImageLength];
    }
    else
    {
        resultImage = [resultImage imageByAdjustOrientationWithMaxLength:0];
    }
    
    
    //////////////////////////////////////////////////
    // 是否依據preview大小調整輸出
    
    if(self.needCropImageByPreviewSize == YES)
    {
        
        CGSize previewSize = [self previewInView:self].bounds.size;
        
        // 傳入的image應該與preview方向相同
        resultImage = [self cropImageWithPreviewSize:previewSize image:resultImage];
    }
    
    
    //////////////////////////////////////////////////
    // test
    
    // NSLog(@"still image : (%d, %f, %f)", resultImage.imageOrientation, resultImage.size.width, resultImage.size.height);
    
    // !! test hight quility image
    //             UIImage *jpgImage = [[UIImage alloc] initWithData:imageData];
    //             UIImage *image = [self copyOrientationAdjustedImageFromImage:jpgImage];
    //             [jpgImage release];
    //             UIImage *image = [self copyImageFromSampleBuffer:imageSampleBuffer];
    
    
    //////////////////////////////////////////////////
    // finish
    
    return resultImage;
}


#elif TARGET_OS_MAC

//================================================================================
//
//================================================================================
- (CPImage *)imageWithStillImageSampleBuffer:(CMSampleBufferRef)sampleBuffer
{
    CPImage *resultImage = nil;
    
    if(sampleBuffer == nil)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    
    // 調整影像速度約0.02sec
    NSData *imageData = [AVCaptureStillImageOutput jpegStillImageNSDataRepresentation:sampleBuffer];
    resultImage = [CPImage imageWithData:imageData];
 
    
    //////////////////////////////////////////////////
    // 調整輸出影像大小
    // (原生image在核心處理時方向有問題，所以都要透過imageByAdjustOrientationWithMaxLength重建bmp)
    
    if (self.maxStillImageLength!=0)
    {
        CGFloat maxImageLength = fmaxf([resultImage size].width, [resultImage size].height);
        
        maxImageLength = fminf(maxImageLength, self.maxStillImageLength);
        resultImage = [resultImage imageByAdjustOrientationWithMaxLength:maxImageLength];
    }
    else
    {
        resultImage = [resultImage imageByAdjustOrientationWithMaxLength:0];
    }
    
    //////////////////////////////////////////////////
    // test
    
    // NSLog(@"still image : (%d, %f, %f)", resultImage.imageOrientation, resultImage.size.width, resultImage.size.height);
    
    // !! test hight quility image
    //             UIImage *jpgImage = [[UIImage alloc] initWithData:imageData];
    //             UIImage *image = [self copyOrientationAdjustedImageFromImage:jpgImage];
    //             [jpgImage release];
    //             UIImage *image = [self copyImageFromSampleBuffer:imageSampleBuffer];
    
    
    //////////////////////////////////////////////////
    // finish
    
    return resultImage;
}
#endif



//================================================================================
//
//================================================================================
- (void)getStillImageAndSendDelegate
{
    if(self.canCaptureStillImage == NO)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
    AVCaptureConnection *videoConnection = nil;
    for (AVCaptureConnection *connection in [self.capturePhotoOutput connections])
    {
        for (AVCaptureInputPort *port in [connection inputPorts])
        {
            if ([[port mediaType] isEqual:AVMediaTypeVideo])
            {
                videoConnection = connection;
                break;
            }
        }
        
        if (videoConnection != nil)
        {
            break;
        }
    }
    
    if (videoConnection==nil)
    {
        return ;
    }
    
    // !!https://fabric.io/penpower/ios/apps/com.penpower.worldcardmobileen/issues/5ab0b6538cb3c2fa63785882?time=last-ninety-days
    // solve: https://github.com/codezero-jp/PBJVision/commit/e89d5ad4fc287b8a449f58c54862b5edee98a5be
    if (!videoConnection.enabled)
    {
        return;
    }
    
    //https://fabric.io/penpower/ios/apps/com.penpower.worldictionarylite/issues/f1f77f1a59d29152f44ff2a5eeeedaa9?time=last-ninety-days
    // solution: https://codeday.me/bug/20190514/1097253.html
    if (!videoConnection.active)
    {
        // Raise error here / warn user...
        return;
    }
    
    //////////////////////////////////////////////////
    
#if TARGET_OS_IPHONE
    
    if(self.cameraDevice.activeFormat.isVideoHDRSupported == NO &&
       (self.hdrMode == PPCameraHDRMode_Auto || self.hdrMode == PPCameraHDRMode_On))
    {
        // MARK:沒有支援videoHDR又需要拍HDR時，自行合成。
        [self.capturePhotoOutput capturePhotoWithSettings:[self capturePhotoBracketSettings] delegate:self];
        // NEXT: captureOutput:didFinishProcessingPhoto:error
    }
    else
    {
        // 一般拍照
        [self.capturePhotoOutput capturePhotoWithSettings:[self capturePhotoSettings] delegate:self];
        // NEXT: captureOutput:didFinishProcessingPhoto:error
        
    }
    
#elif TARGET_OS_MAC
    __block typeof(self) blockSelf = self;
    
    [self.capturePhotoOutput captureStillImageAsynchronouslyFromConnection:videoConnection completionHandler:^(CMSampleBufferRef  _Nullable imageDataSampleBuffer, NSError * _Nullable error) {
        
        CPImage *image = [self imageWithStillImageSampleBuffer:imageDataSampleBuffer];
        [blockSelf performSelectorOnMainThread:@selector(mainThreadDelegateDidGetStillImage:) withObject:image waitUntilDone:YES];
    }];
    
#endif
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Camera Setting Method

//===============================================================================
//
//===============================================================================
- (void)applyCameraSettings
{
#if TARGET_OS_IPHONE
    NSError *error = nil;
    
    if([self.cameraDevice lockForConfiguration:&error])
    {
        // set focus mode
        if ([self.cameraDevice isFocusModeSupported:self.focusMode])
        {
            self.cameraDevice.focusMode = self.focusMode;
        }
        
        // set exposure mode
        if ([self.cameraDevice isExposureModeSupported:self.exposureMode] == YES)
        {
            self.cameraDevice.exposureMode = self.exposureMode;
        }
        
        // set white balance mode
        if ([self.cameraDevice isWhiteBalanceModeSupported:self.whiteBalanceMode] == YES)
        {
            self.cameraDevice.whiteBalanceMode = self.whiteBalanceMode;
        }
        [self.cameraDevice unlockForConfiguration];
    }
#endif
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private capture define methods


//==============================================================================
// macOS上position沒有用
//==============================================================================
+ (NSArray *)captureDeviceWithPosition:(AVCaptureDevicePosition)position
{
    NSArray *devices = nil;
    
#if TARGET_OS_IPHONE

    NSMutableArray *deviceTypes = [NSMutableArray array];
    if([self availableDeviceTypes])
    {
        [deviceTypes addObjectsFromArray:[self availableDeviceTypes]];
    }
    else
    {
        [deviceTypes addObject:AVCaptureDeviceTypeBuiltInWideAngleCamera];
    }
    
    AVCaptureDeviceDiscoverySession *deviceDiscoverySession = [AVCaptureDeviceDiscoverySession discoverySessionWithDeviceTypes:deviceTypes mediaType:AVMediaTypeVideo position:position];
    devices = [deviceDiscoverySession devices];
#elif TARGET_OS_MAC
    devices = [AVCaptureDevice devicesWithMediaType:AVMediaTypeVideo];
#endif
    return devices;
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - AVCaptureSession Initialization

//================================================================================
//
//================================================================================
- (BOOL)addPreviewImageOutput
{
    BOOL result = NO;
    
    do
    {
        //不需要預覽畫面輸出圖
        if([self.delegate respondsToSelector:@selector(ppCameraView:didReceivePreviewImage:)]==NO)
        {
            result = YES;
            break;
        }
    
        //////////////////////////////////////////////////

        if(self.captureVideoDataOutput==nil)
        {
            // Create capture output
            // Update thanks to Jake Marsh who points out not to use the main queue
            char *queueName = "com.penpower.tasks.grabFrames";
            dispatch_queue_t queue = dispatch_queue_create(queueName, NULL);
            self.captureVideoDataOutput = [[[AVCaptureVideoDataOutput alloc] init] autorelease];
            self.captureVideoDataOutput.alwaysDiscardsLateVideoFrames = YES;
            [self.captureVideoDataOutput setSampleBufferDelegate:self queue:queue];
            dispatch_release(queue);
            
            // Establish settings
            NSDictionary *settings = @{(NSString *)kCVPixelBufferPixelFormatTypeKey:@(kCVPixelFormatType_32BGRA)};
            [self.captureVideoDataOutput setVideoSettings:settings];
            
            @synchronized (self.session)
            {
                [self.session beginConfiguration];
                [self.session addOutput:self.captureVideoDataOutput];
                [self.session commitConfiguration];
            }
        }
        
        result = YES;

    } while (0);
       
    return result;
}


//================================================================================
//
//================================================================================
- (void)removePreviewImageOutput
{
//    NSLog(@"debug removePreviewImageOutput before synchronized, %@", self);

    @synchronized (self.session)
    {
//        NSLog(@"debug removePreviewImageOutput before removeOutput, %@", self);
        [self.session beginConfiguration];
        [self.session removeOutput:_captureVideoDataOutput];
        [self.session commitConfiguration];
//        NSLog(@"debug removePreviewImageOutput after removeOutput, %@", self);

        self.captureVideoDataOutput = nil;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)addPhotoOutput
{
    BOOL result = NO;

#if TARGET_OS_IPHONE
    
    if(!self.capturePhotoOutput)
    {
        self.capturePhotoOutput = [[[AVCapturePhotoOutput alloc] init] autorelease];
    }
    
#elif TARGET_OS_MAC
    
    if(!self.capturePhotoOutput)
    {
        self.capturePhotoOutput = [[[AVCaptureStillImageOutput alloc] init] autorelease];
    }
#endif
    
    do {
        
        if(self.capturePhotoOutput==nil)
        {
            break;
        }
        
#if TARGET_OS_IPHONE
#elif TARGET_OS_MAC
        NSDictionary *outputSettings = [[NSDictionary alloc] initWithObjectsAndKeys:AVVideoCodecJPEG, AVVideoCodecKey, nil];
        
        // !! test hight quility image
        //    NSDictionary *outputSettings = [NSDictionary dictionaryWithObject:[NSNumber numberWithInt:kCVPixelFormatType_32BGRA]
        //                                                       forKey:(id)kCVPixelBufferPixelFormatTypeKey];
        
        
        [self.capturePhotoOutput setOutputSettings:outputSettings];
        [outputSettings release];
#endif
        @synchronized (self.session)
        {
            if([self.session canAddOutput:self.capturePhotoOutput])
            {
                [self.session beginConfiguration];
                [self.session addOutput:self.capturePhotoOutput];
                [self.session commitConfiguration];
            }
        }
        result = YES;
    } while (0);

    return result;
}


//==============================================================================
//
//==============================================================================
- (void)removeCapturePhotoOutput
{
    @synchronized (self.session)
    {
        if(!self.capturePhotoOutput)
        {
            [self.session beginConfiguration];
            [self.session removeOutput:self.capturePhotoOutput];
            [self.session commitConfiguration];
        }
        self.capturePhotoOutput = nil;
    }
}



//================================================================================
//
//================================================================================
- (BOOL)addMetaOutput
{
    BOOL result = NO;

    if(!self.captureMetadataOutput)
    {
        self.captureMetadataOutput = [[[AVCaptureMetadataOutput alloc] init] autorelease];
    }

    
    do {
        
        if(self.captureMetadataOutput==nil)
        {
            break;
        }
    
        
        char *queueName = "com.penpower.tasks.metaoutput";
        dispatch_queue_t queue = dispatch_queue_create(queueName, NULL);
        [self.captureMetadataOutput setMetadataObjectsDelegate:self queue:queue];

        @synchronized (self.session)
        {
            if([self.session canAddOutput:self.captureMetadataOutput])
            {
                [self.session beginConfiguration];
                [self.session addOutput:self.captureMetadataOutput];
                [self.session commitConfiguration];
                
                // 需要addOutput後availableMetadataObjectTypes才會有效，這時才可以設定setMetadataObjectTypes
                //            NSLog(@"self.captureMetadataOutput:%@", [self.captureMetadataOutput availableMetadataObjectTypes]);
                [self.captureMetadataOutput setMetadataObjectTypes:@[AVMetadataObjectTypeQRCode]];
            }
        }
        
        result = YES;
    } while (0);

    return result;
}


//==============================================================================
//
//==============================================================================
- (void)removeMetaOutput
{
    @synchronized (self.session)
    {
        if(!self.captureMetadataOutput)
        {
            [self.session beginConfiguration];
            [self.session removeOutput:self.captureMetadataOutput];
            [self.session commitConfiguration];
        }
        self.captureMetadataOutput = nil;
    }
}










////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance control Method


//==============================================================================
//
//==============================================================================
- (BOOL)startCameraPreview;
{
    return [self startCameraPreviewWithCompletion:nil];
}


//================================================================================
//
//================================================================================
- (BOOL)startCameraPreviewWithCompletion:(void(^)(void))completion
{
    BOOL result = NO;
    
    do
    {
        if([self.session isRunning] == YES)
        {
            result = YES;
            break;
        }
        
        //////////////////////////////////////////////////
        
        self.session = [[[AVCaptureSession alloc] init] autorelease];
        
        if(self.session == nil)
        {
            result = NO;
            break;
        }
  
        //////////////////////////////////////////////////
        
        if([self addVideoInputCamera]==YES
           && [self addPhotoOutput]==YES
           && [self addPreviewImageOutput]==YES)
        {
            self.canCaptureStillImage = YES;
#if TARGET_OS_IPHONE
            [self applyCameraSettings];
            
            //////////////////////////////////////////////////
            
            [self.previewLayer removeFromSuperlayer];
            self.previewLayer = nil;
            
            self.previewLayer = [AVCaptureVideoPreviewLayer layerWithSession:self.session];
            
            self.previewLayer.videoGravity = AVLayerVideoGravityResizeAspectFill;
            [self.layer insertSublayer:self.previewLayer atIndex:0];
            
            if(self.supportVideoMirrored==YES)
            {
                self.previewLayer.connection.automaticallyAdjustsVideoMirroring = !self.supportVideoMirrored;
                self.previewLayer.connection.videoMirrored = self.supportVideoMirrored;
            }
            
            //////////////////////////////////////////////////

            __block typeof(self) blockSelf = self;
            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(1.0 * NSEC_PER_SEC)), dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
//                    NSLog(@"debug startRunning before synchronized, %@", self);
                    @synchronized (self.session)
                    {
//                        NSLog(@"debug before startRunning, %@", self);
                        [blockSelf.session startRunning];
//                        NSLog(@"debug after startRunning, %@", self);
                        if(completion)
                        {
                            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                                completion();
                            }] ;
                        }
                    }
                });
            }];
            
            //////////////////////////////////////////////////
            
            result = YES;
#elif TARGET_OS_MAC
            [self.session startRunning];
            
            [self applyCameraSettings];
            
            //////////////////////////////////////////////////
            
            [self.previewLayer removeFromSuperlayer];
            self.previewLayer = nil;
            
            self.previewLayer = [AVCaptureVideoPreviewLayer layerWithSession:self.session];
            
            self.previewLayer.videoGravity = AVLayerVideoGravityResizeAspectFill;
            [self.layer insertSublayer:self.previewLayer atIndex:0];
            
            if(self.supportVideoMirrored==YES)
            {
                self.previewLayer.connection.automaticallyAdjustsVideoMirroring = !self.supportVideoMirrored;
                self.previewLayer.connection.videoMirrored = self.supportVideoMirrored;
            }
            
            //////////////////////////////////////////////////
            
            result = YES;
 #endif

        }
        else
        {
            [self stopCameraPreview];
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)stopCameraPreview
{
#if !(TARGET_IPHONE_SIMULATOR)
    [self removePreviewImageOutput];
    [self removeCapturePhotoOutput];

    [self.previewLayer removeFromSuperlayer];
    self.previewLayer = nil;
        
    [self.session stopRunning];
    self.session = nil;
    
    self.capturePhotoOutput = nil;
    self.cameraDevice = nil;
    self.canCaptureStillImage = NO;
#endif
    
}


//================================================================================
//
//================================================================================
- (void)pauseCameraPreview
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    // !! 暫停時不呼叫removePreviewImageOutput是因為resume會太久，
    //    收到影像時利用connection.enabled來決定是否往外丟即可。
    
    if(self.previewLayer != nil)
    {
        self.previewLayer.connection.enabled = NO;
    }
    
#endif
    
}


//===============================================================================
//
//===============================================================================
- (void)resumeCameraPreview
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    if(self.previewLayer != nil)
    {
        self.previewLayer.connection.enabled = YES;
    }
    
#endif
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property Method (Common)

//================================================================================
//
//================================================================================
- (void)setSupportVideoMirrored:(BOOL)supportVideoMirrored
{
    _supportVideoMirrored = supportVideoMirrored;
    
    //////////////////////////////////////////////////

    if(self.supportVideoMirrored==YES)
    {
        self.previewLayer.connection.automaticallyAdjustsVideoMirroring = NO;
        self.previewLayer.connection.videoMirrored = self.supportVideoMirrored;
    }
    else
    {
        self.previewLayer.connection.videoMirrored = self.supportVideoMirrored;
        self.previewLayer.connection.automaticallyAdjustsVideoMirroring = YES;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Input Method (Common)


//================================================================================
//
//================================================================================
- (BOOL)addVideoInputCamera
{
    NSError *error = nil;
    AVCaptureDeviceInput *deviceInput = nil;
    BOOL result = NO;
    
    do
    {
        if([PPCameraView numberOfCameras] == 0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        // prepare input device
        if([self.cameraDeviceName length]>0)
        {
            NSArray *devices = [PPCameraView captureDeviceWithPosition:AVCaptureDevicePositionUnspecified];
            
            for (AVCaptureDevice *device in devices)
            {
                if([device.localizedName containsString:self.cameraDeviceName]==YES)
                {
                    self.cameraDevice = device;
                    break;
                }
            }
        }
        else
        {
            if(self.ppCameraType==PPCameraType_Front)
            {
                self.cameraDevice = [PPCameraView frontCamera];
            }
            else
            {
                self.cameraDevice = [PPCameraView backCamera];
            }
        }
                
        //////////////////////////////////////////////////
      
        if(self.cameraDevice)
        {
            deviceInput = [AVCaptureDeviceInput deviceInputWithDevice:self.cameraDevice error:&error];
        }
        
        if(!deviceInput)
        {
            self.cameraDevice = nil;
            break;
        }
        
        
        //////////////////////////////////////////////////
        // initial capture session

        self.session.sessionPreset = self.sessionPreset;
        
        @synchronized (self.session)
        {
            [self.session beginConfiguration];
            [self.session addInput:deviceInput];
            [self.session commitConfiguration];
        }
        result = YES;
        
    } while (0);
    
    return result;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - IOS Releated Method

#if TARGET_OS_IPHONE


#pragma mark - capture setting


//==============================================================================
//
//==============================================================================
- (AVCapturePhotoSettings *)capturePhotoSettings
{
    AVCapturePhotoSettings *photoSettings = nil;
    
    if([[self.capturePhotoOutput availablePhotoCodecTypes] containsObject:AVVideoCodecTypeHEVC]==YES)
    {
        // 不支援HEVC時會用JPEG
        photoSettings = [AVCapturePhotoSettings photoSettingsWithFormat:@{AVVideoCodecKey:AVVideoCodecTypeHEVC}];
    }
    else
    {
        photoSettings = [AVCapturePhotoSettings photoSettingsWithFormat:@{AVVideoCodecKey:AVVideoCodecTypeJPEG}];
    }

    if([self.capturePhotoOutput.supportedFlashModes containsObject:@(AVCaptureFlashModeAuto)]||
       [self.capturePhotoOutput.supportedFlashModes containsObject:@(AVCaptureFlashModeOn)]||
       [self.capturePhotoOutput.supportedFlashModes containsObject:@(AVCaptureFlashModeOff)])
    {
        photoSettings.flashMode = self.flashMode;
    }
    
    if (@available(iOS 13.0, *)) {
    } else if (@available(iOS 10.0, *)) {
        // Fallback on earlier versions
        photoSettings.autoStillImageStabilizationEnabled = self.capturePhotoOutput.isStillImageStabilizationSupported;
    }

    return photoSettings;
}

//==============================================================================
//
//==============================================================================
- (AVCapturePhotoBracketSettings *)capturePhotoBracketSettings
{
    NSArray *settings = @[
        [AVCaptureAutoExposureBracketedStillImageSettings autoExposureSettingsWithExposureTargetBias:-4],
        [AVCaptureAutoExposureBracketedStillImageSettings autoExposureSettingsWithExposureTargetBias:0],
        [AVCaptureAutoExposureBracketedStillImageSettings autoExposureSettingsWithExposureTargetBias:PPCameraView_LastBias]
    ];
    
    AVCapturePhotoBracketSettings *bracketSettings = nil;
    
    
    if([[self.capturePhotoOutput availablePhotoCodecTypes] containsObject:AVVideoCodecTypeHEVC]==YES)
    {
        // 不支援HEVC時會用JPEG
        bracketSettings = [AVCapturePhotoBracketSettings photoBracketSettingsWithRawPixelFormatType:0 processedFormat:@{AVVideoCodecKey:AVVideoCodecTypeHEVC} bracketedSettings:settings];
    }
    else
    {
        bracketSettings = [AVCapturePhotoBracketSettings photoBracketSettingsWithRawPixelFormatType:0 processedFormat:@{AVVideoCodecKey:AVVideoCodecTypeJPEG} bracketedSettings:settings];
    }
    
    if (@available(iOS 13.0, *)) {
        bracketSettings.photoQualityPrioritization = AVCapturePhotoQualityPrioritizationBalanced;
    } else {
        // Fallback on earlier versions
        bracketSettings.autoStillImageStabilizationEnabled = NO;
        bracketSettings.autoDualCameraFusionEnabled = NO;
    }
    return bracketSettings;
}

#pragma mark - HDR

//================================================================================
//
//================================================================================
- (BOOL)isSupportHDR
{
#if !(TARGET_IPHONE_SIMULATOR)
    return YES;
#else
    return NO;
#endif

}


//================================================================================
//
//================================================================================
- (void)setHDRMode:(PPCameraHDRMode)hdrMode
{
    self.hdrMode = hdrMode;
    
    if(self.cameraDevice.activeFormat.isVideoHDRSupported == NO)
    {
        return;
    }
    
    switch (hdrMode)
    {
        case PPCameraHDRMode_Auto:
        {
            if(self.cameraDevice.automaticallyAdjustsVideoHDREnabled == NO)
            {
                if([self.cameraDevice lockForConfiguration:nil] == YES)
                {
                    self.cameraDevice.automaticallyAdjustsVideoHDREnabled = YES;
                    
                    [self.cameraDevice unlockForConfiguration];
                }
            }
            
            break;
        }
            
        case PPCameraHDRMode_On:
        {
            if(self.cameraDevice.automaticallyAdjustsVideoHDREnabled == YES ||
               self.cameraDevice.videoHDREnabled == NO)
            {
                if([self.cameraDevice lockForConfiguration:nil] == YES)
                {
                    self.cameraDevice.automaticallyAdjustsVideoHDREnabled = NO;
                    self.cameraDevice.videoHDREnabled = YES;
                    
                    [self.cameraDevice unlockForConfiguration];
                }
            }
            
            break;
        }
            
        case PPCameraHDRMode_Off:
        {
            if(self.cameraDevice.automaticallyAdjustsVideoHDREnabled == YES ||
               self.cameraDevice.videoHDREnabled == YES)
            {
                if([self.cameraDevice lockForConfiguration:nil] == YES)
                {
                    self.cameraDevice.automaticallyAdjustsVideoHDREnabled = NO;
                    self.cameraDevice.videoHDREnabled = NO;
                    
                    [self.cameraDevice unlockForConfiguration];
                }
            }
            
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)generateHDRImageWithBracketImages:(NSArray *)bracketImages completionHandler:(void (^)(CPImage *hdrImage))completionHandler
{
    CPImage *resultImage = [self.hdrGenerator hdrImageByImages:bracketImages];
    
    if(completionHandler != nil)
    {
        completionHandler(resultImage);
    }
}



//==============================================================================
//
//==============================================================================
- (id<MTLTexture>)textureFromImage:(CPImage *)image
{
    if(image==nil)
    {
        return nil;
    }
    
    id<MTLTexture> texture = nil;
    
    @autoreleasepool {
        // MARK:若要檢查aligment過程的數值是否和android一樣，要用bgraData。
        //        NSData *imageData = [image bgraData];
        
        // MARK:單純HDR用jpg data即可，速度較快。
        
        NSData *imageData = UIImageJPEGRepresentation(image, 0.8);
        
        if(imageData==nil)
        {
            return nil;
        }
        
        NSError* err = nil;
        texture = [self.loader newTextureWithData:imageData options:@{} error:&err];
    }
    return [texture autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Stable Check

//==============================================================================
//
//==============================================================================
- (void)setEnablePreviewStableCheck:(BOOL)enablePreviewStableCheck
{
    _enablePreviewStableCheck = enablePreviewStableCheck;
    
    //////////////////////////////////////////////////
    if (self.isEnablePreviewStableCheck)
    {
        self.previewMotionManager.accelerometerUpdateInterval = 0.1;
        self.previewStableCounter = 0;
        
        // !! 這邊accelerometerQueue要設定maxConcurrentOperationCount為1，以免有東西卡住時，
        // 可能會造成 prePreviewAccelerationData 出現overrelease, 不確定與crashlytics上的當機是否一樣問題
        // https://fabric.io/penpower/ios/apps/com.penpower.worldictionary/issues/5c0748ecf8b88c296393c463?time=last-ninety-days
        
        NSOperationQueue *accelerometerQueue = [[[NSOperationQueue alloc] init] autorelease];
        accelerometerQueue.maxConcurrentOperationCount = 1;
        
        [self.previewMotionManager startAccelerometerUpdatesToQueue:accelerometerQueue
                                                        withHandler:^(CMAccelerometerData *data, NSError *error)
         {
             float shakingLevel = 0;
             
             if(self.prePreviewAccelerationData)
             {
                 shakingLevel = fabs(self.prePreviewAccelerationData.acceleration.x - data.acceleration.x) +
                 fabs(self.prePreviewAccelerationData.acceleration.y - data.acceleration.y) +
                 fabs(self.prePreviewAccelerationData.acceleration.z - data.acceleration.z);
                 
                 //test
                 //             NSLog(@"%s shakingLevel:%f", __func__, shakingLevel);
                 
                 // update stable counter
                 if(shakingLevel>=0.05)
                 {
                     self.previewStableCounter=0;
                     self.isDeviceStable = NO;
                 }
                 else
                 {
                     self.previewStableCounter++;
                 }
                 
                 if(self.previewStableCounter>2)
                 {
                     self.isDeviceStable = YES;
                 }
             }
             
             self.prePreviewAccelerationData = data;
         }];
    }
    else
    {
        [self.previewMotionManager stopAccelerometerUpdates];
        
        self.prePreviewAccelerationData = nil;
        
        self.previewStableCounter = 0;
    }
    
}


//================================================================================
//
//================================================================================
- (void)startStableCheck
{
    self.motionManager.accelerometerUpdateInterval = 0.1;
    self.stableCounter = 0;
    
    // !! 這邊accelerometerQueue要設定maxConcurrentOperationCount為1，以免有東西卡住時，

    NSOperationQueue *accelerometerQueue = [[[NSOperationQueue alloc] init] autorelease];
    accelerometerQueue.maxConcurrentOperationCount = 1;
    
    [self.motionManager startAccelerometerUpdatesToQueue:accelerometerQueue
                                             withHandler:^(CMAccelerometerData *data, NSError *error)
     {
         float shakingLevel = 0;
         
         if(self.preAccelerationData)
         {
             shakingLevel = fabs(self.preAccelerationData.acceleration.x - data.acceleration.x) +
             fabs(self.preAccelerationData.acceleration.y - data.acceleration.y) +
             fabs(self.preAccelerationData.acceleration.z - data.acceleration.z);
             
             //test
             //             NSLog(@"%s shakingLevel:%f", __func__, shakingLevel);
             
             if([self.delegate respondsToSelector:@selector(ppCameraView:shakingLevel:)])
             {
                 [self.delegate ppCameraView:self shakingLevel:shakingLevel];
             }
             
             // update stable counter
             if(shakingLevel>=0.04)
             {
                 self.stableCounter=0;
             }
             else
             {
                 self.stableCounter++;
             }
             
             if(self.stableCounter>6)
             {
                 [self getStillImageAndSendDelegate];
                 [self stopStableCheck];
                 return;
             }
         }
         
         self.preAccelerationData = data;
     }];
}


//================================================================================
//
//================================================================================
- (void)stopStableCheck
{
    self.preAccelerationData = nil;
    
    self.stableCounter = 0;
    
    [self.motionManager stopAccelerometerUpdates];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (UIImageOrientation)currentImageOrientation
{
    UIImageOrientation orientation = UIImageOrientationUp;
    
    switch (self.interfaceOrientation)
    {
        case UIDeviceOrientationPortrait:
        {
            if(self.ppCameraType==PPCameraType_Front)
            {
                orientation = UIImageOrientationLeftMirrored;
            }
            else if(self.ppCameraType==PPCameraType_Back)
            {
                orientation = UIImageOrientationRight;
            }
            
            break;
        }
        case UIDeviceOrientationPortraitUpsideDown:
        {
            if(self.ppCameraType==PPCameraType_Front)
            {
                orientation = UIImageOrientationRightMirrored;
            }
            else if(self.ppCameraType==PPCameraType_Back)
            {
                orientation = UIImageOrientationLeft;
            }
            
            break;
        }
        case UIDeviceOrientationLandscapeLeft:
        {
            if(self.ppCameraType==PPCameraType_Front)
            {
                orientation = UIImageOrientationDownMirrored;
            }
            else if(self.ppCameraType==PPCameraType_Back)
            {
                orientation = UIImageOrientationUp;
            }
            
            break;
        }
        case UIDeviceOrientationLandscapeRight:
        {
            if(self.ppCameraType==PPCameraType_Front)
            {
                orientation = UIImageOrientationUpMirrored;
            }
            else if(self.ppCameraType==PPCameraType_Back)
            {
                orientation = UIImageOrientationDown;
            }
            
            break;
        }
        default:
        {
            orientation = UIImageOrientationUp;
            break;
        }
    }
    
    return orientation;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private focus animation methods


//===============================================================================
//
//===============================================================================
- (void)focusAnimationDidStop:(NSString*)animationID finished:(NSNumber*)finished context:(void*)context
{
    if([animationID isEqualToString:@"focusFlick"])
    {
        self.focusImageView.alpha = 0.0;
    }
    else if([animationID isEqualToString:@"centerTransition"])
    {
        self.focusImageView.alpha = 0.5;
        
        [UIView beginAnimations:@"focusFlick" context:nil];
        [UIView setAnimationDuration:.2];
        [UIView setAnimationDelegate:self];
        [UIView setAnimationDidStopSelector:@selector(focusAnimationDidStop:finished:context:)];
        [UIView setAnimationRepeatCount:3];
        [UIView setAnimationRepeatAutoreverses:YES];
        self.focusImageView.alpha = 1.0;
        [UIView commitAnimations];
    }
}


//===============================================================================
//
//===============================================================================
- (void)animateFocusAtPoint:(CGPoint)location
{
    self.focusImageView.center = location;
    self.focusImageView.alpha = 1.0;
    
    [UIView beginAnimations:@"centerTransition" context:nil];
    [UIView setAnimationBeginsFromCurrentState:YES];
    [UIView setAnimationDuration:0.2];
    [UIView setAnimationDelegate:self];
    [UIView setAnimationDidStopSelector:@selector(focusAnimationDidStop:finished:context:)];
    self.focusImageView.alpha = 0.0;
    [UIView commitAnimations];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private focus Gesture methods

//===============================================================================
//
//===============================================================================
- (void)handleTap:(UITapGestureRecognizer *)sender
{
    if (self.enableFocusTouch==NO)
    {
        return ;
    }
    
    //////////////////////////////////////////////////
    
    if (sender.state == UIGestureRecognizerStateEnded)
    {
        CGPoint point = [sender locationInView:self];
        
        [self setFocusPoint:point];
        
        //////////////////////////////////////////////////
        
        if(self.alwaysShowFocusImageWhenSetFoucsPoint == NO)
        {
            if(CGRectContainsPoint(self.bounds, point))
            {
                [self bringSubviewToFront:self.focusImageView];
                [self animateFocusAtPoint:point];
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property override

//==============================================================================
//
//==============================================================================
- (UIImage *)focusImage
{
    return self.focusImageView.image;
}


//==============================================================================
//
//==============================================================================
- (void)setFocusImage:(UIImage *)focusImage
{
    self.focusImageView.image = focusImage;
}


//===============================================================================
// 設定白平衡的模式，預設為 AVCaptureWhiteBalanceModeContinuousAutoWhiteBalance
//===============================================================================
- (void)setFocusMode:(AVCaptureFocusMode)focusMode
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    NSError * error = nil;
    
    if (self.cameraDevice!= nil && [self.cameraDevice lockForConfiguration:&error])
    {
        if ([self.cameraDevice isFocusModeSupported:focusMode] == YES)
        {
            self.cameraDevice.focusMode = focusMode;
            _focusMode = focusMode;
        }
        
        [self.cameraDevice unlockForConfiguration];
    }
    
#endif
}


//===============================================================================
// 設定白平衡的模式，預設為 AVCaptureWhiteBalanceModeContinuousAutoWhiteBalance
//===============================================================================
- (void)setWhiteBalanceMode:(AVCaptureWhiteBalanceMode)whiteBalanceMode
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    NSError * error = nil;
    
    if (self.cameraDevice!= nil && [self.cameraDevice lockForConfiguration:&error])
    {
        if ([self.cameraDevice isWhiteBalanceModeSupported:whiteBalanceMode] == YES)
        {
            self.cameraDevice.whiteBalanceMode = whiteBalanceMode;
            _whiteBalanceMode = whiteBalanceMode;
        }
        
        [self.cameraDevice unlockForConfiguration];
    }
    
#endif
}


//===============================================================================
// 設定曝光的模式，預設為 AVCaptureExposureModeContinuousAutoExposure，回傳原本的模式
//===============================================================================
- (void)setExposureMode:(AVCaptureExposureMode)exposureMode
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    NSError * error = nil;
    
    if (self.cameraDevice!= nil && [self.cameraDevice lockForConfiguration:&error])
    {
        if ([self.cameraDevice isExposureModeSupported:exposureMode] == YES)
        {
            self.cameraDevice.exposureMode = exposureMode;
            _exposureMode = exposureMode;
        }
        
        [self.cameraDevice unlockForConfiguration];
    }
    
#endif
}


//===============================================================================
// 設定縮放倍率
//===============================================================================
- (void)setVideoZoomFactor:(CGFloat)videoZoomFactor
{
#if !(TARGET_IPHONE_SIMULATOR)
    @synchronized (self)
    {
        NSError * error = nil;
        
        if (self.cameraDevice!= nil && [self.cameraDevice lockForConfiguration:&error])
        {
            self.cameraDevice.videoZoomFactor = videoZoomFactor;
            _videoZoomFactor = videoZoomFactor;
            [self.cameraDevice unlockForConfiguration];
        }
    }
#endif
}


//==============================================================================
//
//==============================================================================
- (AVCaptureDeviceType)deviceType
{
    return self.cameraDevice.deviceType;
}


//===============================================================================
//
//===============================================================================
- (void)setFocusPoint:(CGPoint)focusPoint
{
    if(CGRectContainsPoint(self.bounds, focusPoint))
    {
        _focusPoint = focusPoint;
        
        //        NSLog(@"--------setFocusPoint:%@", NSStringFromCGPoint(focusPoint));
        
        CGPoint devicePoint;
        
        //----------------------------------
        // !! 取得實際對應的device point
        // !! devicePoint計算是以LandscapeRight方向，左上角為(0,0)。
        //    目前iPhone WCM的Camera UI是固定方向，所以只要找出固定規則即可。
        //    後續iPad WCHD要檢查是否會有問題
        //----------------------------------
        devicePoint.x = focusPoint.y/self.bounds.size.height;
        devicePoint.y = 1.0 - (focusPoint.x/self.bounds.size.width);
        
        [self clickDevicePoint:devicePoint];
        
        //////////////////////////////////////////////////
        
        if(self.alwaysShowFocusImageWhenSetFoucsPoint == YES)
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [self bringSubviewToFront:self.focusImageView];
                [self animateFocusAtPoint:focusPoint];
            });
        }
    }
}










////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - Instance Method


//================================================================================
//
//================================================================================
- (void)switchCameras
{
    @synchronized (self.session)
    {
        if ([PPCameraView numberOfCameras] <= 1)
        {
            return;
        }
        
        if(self.ppCameraType!=PPCameraType_Back)
        {
            self.ppCameraType = PPCameraType_Back;
        }
        else if(self.ppCameraType!=PPCameraType_Front)
        {
            self.ppCameraType = PPCameraType_Front;
        }
        
        AVCaptureDevice *newDevice = (self.ppCameraType==PPCameraType_Back) ? [PPCameraView backCamera] : [PPCameraView frontCamera];
        
        if(newDevice==nil)
        {
            return;
        }
        

        [self.session beginConfiguration];
        
        // Remove existing inputs
        for (AVCaptureInput *input in [self.session inputs])
        {
            [self.session removeInput:input];
        }
        
        // Change the input
        AVCaptureDeviceInput *captureInput = [AVCaptureDeviceInput deviceInputWithDevice:newDevice error:nil];
        [self.session addInput:captureInput];
        
        [self.session commitConfiguration];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance UltraWide camera methods



//================================================================================
//
//================================================================================
- (void)ultraWideCameraEnable:(BOOL)enable
{
    @synchronized (self.session)
    {
        if ([PPCameraView numberOfCameras] <= 1)
        {
            return;
        }
        
        
        AVCaptureDevice *newDevice = nil;
        if(enable)
        {
            newDevice = [PPCameraView ultraWideCamera];
        }
        else
        {
            newDevice = self.ppCameraType ? [PPCameraView backCamera] : [PPCameraView frontCamera];
        }
    

        [self.session beginConfiguration];
        
        // Remove existing inputs
        for (AVCaptureInput *input in [self.session inputs])
        {
            [self.session removeInput:input];
        }
        
        // Change the input
        AVCaptureDeviceInput *captureInput = [AVCaptureDeviceInput deviceInputWithDevice:newDevice error:nil];
        [self.session addInput:captureInput];
        
        [self.session commitConfiguration];
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)isSupportUltraWideCamera
{
    AVCaptureDevice *device = [PPCameraView ultraWideCamera];
    
    return (device!=nil);
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Flash methods


//================================================================================
//
//================================================================================
- (void)turnOnFlash:(BOOL)turnOn useAutoMode:(BOOL)useAutoMode
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    if([self.cameraDevice hasFlash])
    {
        if(turnOn)
        {
            if(useAutoMode)
            {
                self.flashMode = AVCaptureFlashModeAuto;
            }
            else
            {
                self.flashMode = AVCaptureFlashModeOn;
            }
        }
        else
        {
            self.flashMode = AVCaptureFlashModeOff;
        }
    }
    
#endif
}


//================================================================================
//
//================================================================================
- (void)turnOnTorch:(BOOL)turnOn
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    if([self.cameraDevice hasTorch])
    {
        if(turnOn == YES && [self.cameraDevice torchMode] == AVCaptureTorchModeOff)
        {
            [self.cameraDevice lockForConfiguration:nil];
            [self.cameraDevice setTorchMode:AVCaptureTorchModeOn];
            [self.cameraDevice unlockForConfiguration];
        }
        else if(turnOn == NO && [self.cameraDevice torchMode] == AVCaptureTorchModeOn)
        {
            [self.cameraDevice lockForConfiguration:nil];
            [self.cameraDevice setTorchMode:AVCaptureTorchModeOff];
            [self.cameraDevice unlockForConfiguration];
        }
    }
    
#endif
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance focus Method

//================================================================================
//
//================================================================================
- (BOOL)isFocusing
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    return [self.cameraDevice isAdjustingFocus];
    
#else
    
    return NO;
    
#endif
}


//================================================================================
// !! 注意座標系不同，是FocusPointOfInterest !!
//================================================================================
- (void)clickDevicePoint:(CGPoint)devicePoint
{
    NSError *error = nil;
    
    if([self.cameraDevice lockForConfiguration:&error])
    {
        // set auto-focus point
        if([self.cameraDevice isFocusPointOfInterestSupported])
        {
            [self.cameraDevice setFocusPointOfInterest:devicePoint];
        }
        
        // set auto-exposure point
        if([self.cameraDevice isExposurePointOfInterestSupported])
        {
            [self.cameraDevice setExposurePointOfInterest:devicePoint];
        }
        
        [self.cameraDevice unlockForConfiguration];
    }
    
    [self applyCameraSettings];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance layout Method

//================================================================================
//
//================================================================================
- (AVCaptureVideoPreviewLayer *)previewInView:(CPView *)view
{
    for (CALayer *layer in view.layer.sublayers)
    {
        if ([layer isKindOfClass:[AVCaptureVideoPreviewLayer class]])
        {
            return (AVCaptureVideoPreviewLayer *)layer;
        }
    }
    
    return nil;
}


//================================================================================
//
//================================================================================
- (void)layoutPreviewInView:(CPView *)view
{
    [self layoutPreviewInView:view withDefaultOrientation:UIInterfaceOrientationPortrait];
}


//================================================================================
//
//================================================================================
- (void)layoutPreviewInView:(CPView *)view withDefaultOrientation:(UIInterfaceOrientation)orientation
{
    do
    {
        if(view==nil || [PPCameraView numberOfCameras]<=0)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        AVCaptureVideoPreviewLayer *layer = [self previewInView:view];
        if (!layer)
        {
            NSLog(@"Warning. AvCaptureVideoPreviewLayer doesn't exit in the view");
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        if (orientation == UIInterfaceOrientationPortrait)
        {
            if([[UIDevice currentDevice].systemVersion floatValue]>=6.0)
            {
                [layer.connection setVideoOrientation:AVCaptureVideoOrientationPortrait];
            }
            else
            {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
                layer.orientation = AVCaptureVideoOrientationPortrait;
#pragma clang diagnostic pop
            }
        }
        else if (orientation == UIInterfaceOrientationLandscapeLeft)
        {
            if([[UIDevice currentDevice].systemVersion floatValue]>=6.0)
            {
                [layer.connection setVideoOrientation:AVCaptureVideoOrientationLandscapeLeft];
            }
            else
            {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
                layer.orientation = AVCaptureVideoOrientationLandscapeLeft;
#pragma clang diagnostic pop
            }
        }
        else if (orientation == UIInterfaceOrientationLandscapeRight)
        {
            if([[UIDevice currentDevice].systemVersion floatValue]>=6.0)
            {
                [layer.connection setVideoOrientation:AVCaptureVideoOrientationLandscapeRight];
            }
            else
            {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
                layer.orientation = AVCaptureVideoOrientationLandscapeRight;
#pragma clang diagnostic pop
            }
        }
        else
        {
            if([[UIDevice currentDevice].systemVersion floatValue]>=6.0)
            {
                [layer.connection setVideoOrientation:AVCaptureVideoOrientationPortraitUpsideDown];
            }
            else
            {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
                layer.orientation = AVCaptureVideoOrientationPortraitUpsideDown;
#pragma clang diagnostic pop
            }
        }
        
        layer.frame = view.bounds;
        
    } while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Capture Method

//===============================================================================
//
//===============================================================================
- (void)captureStillImagWithStableCheck:(BOOL)needStableCheck
{
#if !(TARGET_IPHONE_SIMULATOR)
    
    if(needStableCheck)
    {
        [self startStableCheck];
    }
    else
    {
        [self getStillImageAndSendDelegate];
    }
    
#else
    
    UIImage *image = [UIImage imageNamed:@"PPCV_SimCapturedCard.jpg"];
    [self performSelectorOnMainThread:@selector(mainThreadDelegateDidGetStillImage:) withObject:image waitUntilDone:NO];
    
#endif
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - QRCODE methods


//==============================================================================
//
//==============================================================================
- (void)enableQrcodeRecognize:(BOOL)enable
{
    if(enable)
    {
        [self addMetaOutput];
    }
    else
    {
        [self removeMetaOutput];
    }
    
}


//==============================================================================
//
//==============================================================================
- (void)captureOutput:(AVCaptureOutput *)output didOutputMetadataObjects:(NSArray<__kindof AVMetadataObject *> *)metadataObjects fromConnection:(AVCaptureConnection *)connection
{
    AVMetadataObject* metadataObject = [metadataObjects firstObject];
    if([metadataObject type]==AVMetadataObjectTypeQRCode)
    {
        if([self.delegate respondsToSelector:@selector(ppCameraView:didGetQrcode:)])
        {
            [self.delegate ppCameraView:self didGetQrcode:[(AVMetadataMachineReadableCodeObject*)metadataObject stringValue]];
        }
    }
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - AVCapturePhotoCaptureDelegate


//==============================================================================
// HDR phto 會進來多次
//==============================================================================
- (void)captureOutput:(AVCapturePhotoOutput *)output
didFinishProcessingPhoto:(AVCapturePhoto *)photo
                error:(NSError *)error
{
    if(error)
    {
        return ;
    }
    
    if(photo.bracketSettings!=nil)
    {
        // MARK: 拍多張自行產生HDR
        @autoreleasepool {
            CPImage *image = [self imageWithPhoto:photo];
            if(image)
            {
                id <MTLTexture> texture = [self textureFromImage:image];
                [self.bracketImages addObject:texture];
            }
        }
        
        // 拍完最後一張送HDR (用最後一張的bias做判斷)
        if(((AVCaptureAutoExposureBracketedStillImageSettings *)photo.bracketSettings).exposureTargetBias==PPCameraView_LastBias)
        {
            __block typeof(self) blockSelf = self;
            
            [blockSelf generateHDRImageWithBracketImages:self.bracketImages completionHandler:^(UIImage *hdrImage) {
                
                [blockSelf performSelectorOnMainThread:@selector(mainThreadDelegateDidGetStillImage:) withObject:hdrImage waitUntilDone:YES];
                [self.bracketImages removeAllObjects];
            }];
        }
    }
    else
    {
        // MARK: 單張拍照
        CPImage *image = [self imageWithPhoto:photo];
        if(image)
        {
            [self performSelectorOnMainThread:@selector(mainThreadDelegateDidGetStillImage:) withObject:image waitUntilDone:YES];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - macOS Instace Method
#elif TARGET_OS_MAC

//================================================================================
//
//================================================================================
- (void)captureStillImage
{
    [self getStillImageAndSendDelegate];
}

#endif






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Send Delegate Method

//================================================================================
//
//================================================================================
- (void)mainThreadDelegateDidGetPreviewImage:(CPImage *)image
{
    [self.delegate ppCameraView:self didReceivePreviewImage:image];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - AVCaptureVideoDataOutputSampleBufferDelegate methods

//================================================================================
//
//================================================================================
- (void)captureOutput:(AVCaptureOutput *)output didOutputSampleBuffer:(CMSampleBufferRef)sampleBuffer fromConnection:(AVCaptureConnection *)connection;
{
    //////////////////////////////////////////////////
    
    if(self.previewLayer.connection.enabled == NO)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
#if TARGET_OS_IPHONE
    
    // 如果有開始穩定偵測, 且還沒有穩定，就不做
    if (self.isEnablePreviewStableCheck== YES && self.isDeviceStable == NO)
    {
        return;
    }
    
#endif
    
    //////////////////////////////////////////////////
    
    if([self.delegate respondsToSelector:@selector(ppCameraView:didReceivePreviewImage:)])
    {
        CPImage *image = [self copyImageFromSampleBuffer:sampleBuffer];
        
        [self performSelectorOnMainThread:@selector(mainThreadDelegateDidGetPreviewImage:) withObject:image waitUntilDone:YES];
        [image release];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method


//================================================================================
//
//================================================================================
+ (NSInteger)numberOfCameras
{
    // iphone or ipad simulator
    if(TARGET_IPHONE_SIMULATOR)
    {
        return  0;
    }
    else
    {
        return [PPCameraView captureDeviceWithPosition:AVCaptureDevicePositionUnspecified].count;
    }
    
}


//================================================================================
//
//================================================================================
+ (BOOL)backCameraAvailable
{
    return ([self backCamera]!=nil);
}


//================================================================================
//
//================================================================================
+ (BOOL)frontCameraAvailable
{
    return ([self frontCamera]!=nil);
}


//================================================================================
//
//================================================================================
+ (AVCaptureDevice *)backCamera
{
    AVCaptureDevice *backCameraDevice = nil;
    NSArray *videDevices = [PPCameraView captureDeviceWithPosition:AVCaptureDevicePositionBack];

    for(AVCaptureDevice *device in videDevices)
    {
        if(device.position==AVCaptureDevicePositionBack)
        {
            backCameraDevice = device;
            break;
        }
    }
    return backCameraDevice;
}


//================================================================================
//
//================================================================================
+ (AVCaptureDevice *)frontCamera
{
    AVCaptureDevice *frontCameraDevice = nil;
    NSArray *videDevices = [PPCameraView captureDeviceWithPosition:AVCaptureDevicePositionFront];
    
    for(AVCaptureDevice *device in videDevices)
    {
        if(device.position==AVCaptureDevicePositionFront)
        {
            frontCameraDevice = device;
            break;
        }
    }
    
    return frontCameraDevice;
}


//================================================================================
//
//================================================================================
+ (BOOL)canUseFlashForCameraType:(PPCameraType)cameraType
{
    BOOL hasFlash = NO;
    
    do
    {
        if ([PPCameraView numberOfCameras]<=0)
        {
            break;
        }
        
        AVCaptureDevice *device = nil;
        
        if(cameraType==PPCameraType_Front)
        {
            device = [PPCameraView frontCamera];
        }
        else if(cameraType==PPCameraType_Back)
        {
            device = [PPCameraView backCamera];
        }
        
        if(device!=nil)
        {
            hasFlash = [device hasFlash];
        }
        
    } while (0);
    
    return hasFlash;
}


//================================================================================
//
//================================================================================
+ (BOOL)canUseTorchForCameraType:(PPCameraType)cameraType
{
    BOOL hasTorch = NO;
    
    do
    {
        if ([PPCameraView numberOfCameras]<=0)
        {
            break;
        }
        
        AVCaptureDevice *device = nil;
        
        if(cameraType==PPCameraType_Front)
        {
            device = [PPCameraView frontCamera];
        }
        else if(cameraType==PPCameraType_Back)
        {
            device = [PPCameraView backCamera];
        }
        
        if(device!=nil)
        {
            hasTorch = [device hasTorch];
        }
        
    } while (0);
    
    return hasTorch;
}

@end
