//
//  MetalFunctionCreateMapping.m
//  Extension
//
//  Created by sanhue cheng on 2020/5/29.
//

#import "MetalFunctionCreateMapping.h"
#import "PPCVKernelTypes.h"

@interface MetalFunctionCreateMapping ()

@property (nonatomic, strong) id<MTLDevice> mtlDevice;
@property (nonatomic, strong) id <MTLComputePipelineState> computingPipelineState;
@end


////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation MetalFunctionCreateMapping


//==============================================================================
//
//==============================================================================
- (nullable instancetype)initWithDevice:(nonnull id <MTLDevice>)device libray:(nonnull id<MTLLibrary>)library
{
    self = [super init];
    if (self)
    {
        self.mtlDevice = device;

        // compute pipline
        id <MTLFunction> function = [library newFunctionWithName:@"createMapping"];
        self.computingPipelineState = [[self.mtlDevice newComputePipelineStateWithFunction:function error:nil] autorelease];
        [function release];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.mtlDevice = nil;
    self.computingPipelineState = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)encodeToCommandBuffer:(nonnull id <MTLCommandBuffer>)commandBuffer
            withStartPosition:(CGPoint)startPosition
                 stopPosition:(CGPoint)stopPosition
                       median:(int)median
                sourceTexture:(id<MTLTexture>)sourceTexture
                resultTexture:(id<MTLTexture>)resultTexture
                    imageSize:(CGSize)imageSize
{
    id<MTLComputeCommandEncoder> computingCommandEncoder = [commandBuffer computeCommandEncoder]; // 通过渲染描述符构建 encoder

    id<MTLBuffer> creatingMappingParams = [self.mtlDevice newBufferWithLength:sizeof(PPCVCreateMappingParams) options:MTLResourceStorageModeShared];
    PPCVCreateMappingParams *param = (PPCVCreateMappingParams *)creatingMappingParams.contents;
    param->startX = startPosition.x;
    param->startY = startPosition.y;
    param->stopX = stopPosition.x;
    param->stopY = stopPosition.y;
    param->medianValue = median;

    [computingCommandEncoder setComputePipelineState:self.computingPipelineState];
    [computingCommandEncoder setTexture:sourceTexture atIndex:0];
    [computingCommandEncoder setTexture:resultTexture atIndex:1];
    [computingCommandEncoder setBuffer:creatingMappingParams offset:0 atIndex:0];

    [creatingMappingParams release];
    //////////////////////////////////////////////////
    // Calculate a threadgroup size.
    
    NSUInteger w = self.computingPipelineState.threadExecutionWidth;
    NSUInteger h = self.computingPipelineState.maxTotalThreadsPerThreadgroup / w;
    MTLSize threadsPerThreadgroup = MTLSizeMake(w, h, 1);
    MTLSize threadgroupsPerGrid = MTLSizeMake((imageSize.width + w - 1) / w,
                                              (imageSize.height + h - 1) / h,
                                              1);

    // Encode the compute command.
    [computingCommandEncoder dispatchThreadgroups: threadgroupsPerGrid
                            threadsPerThreadgroup: threadsPerThreadgroup];
    
    // End the compute pass.
    [computingCommandEncoder endEncoding];
}

@end
