//
//  MetalFunctionAlignmentOffset.m
//  PPCameraView
//
//  Created by sanhue cheng on 2020/6/2.
//

#import "MetalFunctionAlignmentOffset.h"
#import "PPCVKernelTypes.h"


@interface MetalFunctionAlignmentOffset ()

@property (nonatomic, strong) id<MTLDevice> mtlDevice;
@property (nonatomic, strong) id <MTLComputePipelineState> computingPipelineState;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation MetalFunctionAlignmentOffset


//==============================================================================
//
//==============================================================================
- (nullable instancetype)initWithDevice:(nonnull id <MTLDevice>)device libray:(nonnull id<MTLLibrary>)library
{
    self = [super init];
    if (self)
    {
        self.mtlDevice = device;
        
        // compute pipline
        id <MTLFunction> function = [library newFunctionWithName:@"alignmentOffset"];
        self.computingPipelineState = [[self.mtlDevice newComputePipelineStateWithFunction:function error:nil] autorelease];
        [function release];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.mtlDevice = nil;
    self.computingPipelineState = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)encodeToCommandBuffer:(nonnull id <MTLCommandBuffer>)commandBuffer
            withStartPosition:(CGPoint)startPosition
                 stopPosition:(CGPoint)stopPosition
               offsetPosition:(CGPoint)offsetPosition
                     stepSize:(int)stepSize
                  baseTexture:(id<MTLTexture>)baseTexture
               compareTexture:(id<MTLTexture>)compareTexture
                 resultBuffer:(id<MTLBuffer>)resultBuffer
                    imageSize:(CGSize)imageSize
{
    id<MTLComputeCommandEncoder> computingCommandEncoder = [commandBuffer computeCommandEncoder]; // 通过渲染描述符构建 encoder
    id<MTLBuffer> calculateOffsetParams = [self.mtlDevice newBufferWithLength:sizeof(PPCVCalculateOffsetParams) options:MTLResourceStorageModeShared];
    PPCVCalculateOffsetParams *param = (PPCVCalculateOffsetParams *)calculateOffsetParams.contents;
    
    param->startX = startPosition.x;
    param->startY = startPosition.y;
    param->stopX = stopPosition.x;
    param->stopY = stopPosition.y;
    param->offsetX = offsetPosition.x;
    param->offsetY = offsetPosition.y;
    param->stepSize = stepSize;

    [computingCommandEncoder setComputePipelineState:self.computingPipelineState];
    [computingCommandEncoder setTexture:baseTexture atIndex:0];
    [computingCommandEncoder setTexture:compareTexture atIndex:1];
    [computingCommandEncoder setBuffer:calculateOffsetParams offset:0 atIndex:0];
    [computingCommandEncoder setBuffer:resultBuffer offset:0 atIndex:1];

    [calculateOffsetParams release];
    //////////////////////////////////////////////////
    // Calculate a threadgroup size.
    
    NSUInteger w = self.computingPipelineState.threadExecutionWidth;
    NSUInteger h = self.computingPipelineState.maxTotalThreadsPerThreadgroup / w;
    MTLSize threadsPerThreadgroup = MTLSizeMake(w, h, 1);
    MTLSize threadgroupsPerGrid = MTLSizeMake((imageSize.width + w - 1) / w,
                                              (imageSize.height + h - 1) / h,
                                              1);
    
    // Encode the compute command.
    [computingCommandEncoder dispatchThreadgroups: threadgroupsPerGrid
                            threadsPerThreadgroup: threadsPerThreadgroup];
    
    // End the compute pass.
    [computingCommandEncoder endEncoding];
}

@end
