//
//  PPButton.m
//  
//
//  Created by Mike on 13/5/17.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPButton.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Resource

NSString *ImageNamePPButtonBackgroundForBack       		= @"PPButtonBackgroundForBack";
NSString *ImageNamePPButtonBackgroundForNavigationBar	= @"PPButtonBackgroundForNavigationBar";
NSString *ImageNamePPButtonIconForSystemBackImitation	= @"PPButtonIconForSystemBackImitation";

////////////////////////////////////////////////////////////////////////////////////////////////////

static UIColor  *defaultTitleColor          = nil;
static UIColor  *defaultTitleShadowColor    = nil;
static UIFont   *defaultTitleLabelFont      = nil;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPButton()
@property (nonatomic,retain) UILongPressGestureRecognizer *longPressGestureRecognizer;
@property (nonatomic, assign) UIEdgeInsets oringalImageEdgeInsets;
- (void)orientationDidChangeNotification:(NSNotification *)notification;
- (void)onTouchCancel;
- (void)onTouchDown;
- (void)onTouchDownRepeat;
- (void)onTouchDragInside;
- (void)onTouchDragOutside;
- (void)onTouchDragEnter;
- (void)onTouchDragExit;
- (void)onTouchUpInside;
- (void)onTouchUpOutside;
- (void)reportControlEvent:(UIControlEvents)controlEvent;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation PPButton

@implementation PPButton

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize delegate           = delegate_;
@synthesize orientationDetect  = orientationDetect_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame
{
	if((self = [super initWithFrame:frame]))
	{
        //////////////////////////////////////////////////
        [[PPButton appearance] applyInvocationTo:self];
        //////////////////////////////////////////////////

        self.clipsToBounds				= YES;
		self.imageView.contentMode		= UIViewContentModeScaleAspectFit;
		self.titleLabel.adjustsFontSizeToFitWidth = YES;
        
        //////////////////////////////////////////////////

        if(defaultTitleColor!=nil)
        {
            [self setTitleColor:defaultTitleColor];
        }
        
        if(defaultTitleShadowColor!=nil)
        {
            [self setTitleShadowColor:defaultTitleColor];
        }
        
        if(defaultTitleLabelFont!=nil)
        {
            self.titleLabel.font = defaultTitleLabelFont;
        }
        
        self.oringalImageEdgeInsets = UIEdgeInsetsZero;
	}
	
	return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    // !! 這邊要設為nil，不然ios8可能會leak
    [self setImage:nil forState:UIControlStateNormal];
    [self setImage:nil forState:UIControlStateHighlighted];
    [self setImage:nil forState:UIControlStateDisabled];
    [self setImage:nil forState:UIControlStateSelected];
     
    self.orientationDetect = NO;
   
    //////////////////////////////////////////////////
 
    _enableLongPress = NO;
    
    //////////////////////////////////////////////////

    [self removeGestureRecognizer:self.longPressGestureRecognizer];
    [self.longPressGestureRecognizer removeTarget:self action:@selector(longPress:)];
    
    self.longPressGestureRecognizer = nil;

    //////////////////////////////////////////////////
    
    [super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    // 如果還沒有大小，先不layout
    if (self.bounds.size.width==0||
        self.bounds.size.height==0)
    {
        return;
    }
    //////////////////////////////////////////////////
    // !! 如果不是0才要調整大小
    if (self.minimumImageHeight!=0)
    {
        // 先以按鈕大小來計算|minimumImageHeight|是否合法
        CGFloat adjustedMinimumImageHeight = self.minimumImageHeight;
        
        if (adjustedMinimumImageHeight>self.bounds.size.width)
        {
            adjustedMinimumImageHeight=self.bounds.size.width;
        }

        if (adjustedMinimumImageHeight>self.bounds.size.height)
        {
            adjustedMinimumImageHeight=self.bounds.size.height;
        }

        //////////////////////////////////////////////////
        // 第一次進來應該是空的，所以強制記錄，之後才能再用來計算
        if (UIEdgeInsetsEqualToEdgeInsets(self.oringalImageEdgeInsets, UIEdgeInsetsZero))
        {
            self.oringalImageEdgeInsets = self.imageEdgeInsets;
        }
        
        CGFloat originalImageHeight = self.frame.size.height-(self.oringalImageEdgeInsets.top+self.oringalImageEdgeInsets.bottom);
        
        if (originalImageHeight<adjustedMinimumImageHeight)
        {
            CGFloat adjustInset = (self.frame.size.height-adjustedMinimumImageHeight)/2.0;
            UIEdgeInsets adjustEdgeInsets = self.oringalImageEdgeInsets;
            adjustEdgeInsets.top = adjustInset;
            adjustEdgeInsets.bottom = adjustInset;
            self.imageEdgeInsets = adjustEdgeInsets;
        }
        else
        {
            // 這邊是為了在layout回到原本的狀況以，可以用原本的imageEdgeInset所以要設定回去
            self.imageEdgeInsets = self.oringalImageEdgeInsets;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Methods

//================================================================================
//
//================================================================================
- (void)setLayoutType:(PPButtonLayoutType)layoutType
{
    _layoutType = layoutType;
    
    //////////////////////////////////////////////////
    switch (layoutType)
    {
        case PPButtonLayoutType_Bottom:
        {
            self.titleLabel.textAlignment = NSTextAlignmentCenter;
            break;
        }
        case PPButtonLayoutType_Right:
        {
            self.titleLabel.textAlignment = NSTextAlignmentLeft;
            break;
        }
        default:
            break;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Methods

//================================================================================
// 參考 https://blog.csdn.net/lqq200912408/article/details/51323336，修改比例
//================================================================================
- (CGRect)titleRectForContentRect:(CGRect)contentRect
{
    CGRect titleRect = contentRect;
    
    switch (self.layoutType)
    {
        case PPButtonLayoutType_Bottom:
        {
            CGFloat inteval = CGRectGetHeight(contentRect)/16.0;
            inteval = MIN(inteval, 6);
            
            CGFloat imageH = CGRectGetHeight(contentRect) - 8*inteval;
            
            titleRect = CGRectMake(self.titleEdgeInsets.left+CGRectGetMinX(contentRect),
                                   self.titleEdgeInsets.top+CGRectGetMinY(contentRect)+2*inteval+imageH,
                                   CGRectGetWidth(contentRect)-(self.titleEdgeInsets.left+self.titleEdgeInsets.right),
                                   CGRectGetHeight(contentRect)-(imageH+2*inteval)-(self.titleEdgeInsets.top+self.titleEdgeInsets.bottom));
            
            break;
        }
        case PPButtonLayoutType_Right:
        {
            CGFloat imageH = CGRectGetHeight(contentRect);
            
            titleRect = CGRectMake(self.titleEdgeInsets.left+CGRectGetMinX(contentRect)+imageH,
                                   self.titleEdgeInsets.top+CGRectGetMinY(contentRect),
                                   CGRectGetWidth(contentRect)-imageH-(self.titleEdgeInsets.left+self.titleEdgeInsets.right),
                                   CGRectGetHeight(contentRect)-(self.titleEdgeInsets.top+self.titleEdgeInsets.bottom));
            
            break;
        }
        default:
        {
            titleRect = [super titleRectForContentRect:contentRect];
            break;
        }
    }
    
    return titleRect;
}


//================================================================================
// 參考 https://blog.csdn.net/lqq200912408/article/details/51323336，修改比例
//================================================================================
- (CGRect)imageRectForContentRect:(CGRect)contentRect
{
    CGRect imageRect = contentRect;
    
    switch (self.layoutType)
    {
        case PPButtonLayoutType_Bottom:
        {
            CGFloat inteval = CGRectGetHeight(contentRect)/16.0;
            inteval = MIN(inteval, 8);
            
            CGFloat imageH = CGRectGetHeight(contentRect) - 8*inteval;
            
            imageRect = CGRectMake(self.imageEdgeInsets.left+CGRectGetMidX(contentRect)-imageH/2,
                                   self.imageEdgeInsets.top+CGRectGetMinY(contentRect)+2*inteval,
                                   imageH-(self.imageEdgeInsets.left+self.imageEdgeInsets.right),
                                   imageH-(self.imageEdgeInsets.top+self.imageEdgeInsets.bottom));
            
            break;
        }
        case PPButtonLayoutType_Right:
        {
            CGFloat imageH = CGRectGetHeight(contentRect);
            
            imageRect = CGRectMake(self.imageEdgeInsets.left+CGRectGetMinX(contentRect),
                                   self.imageEdgeInsets.top+CGRectGetMinY(contentRect),
                                   imageH-(self.imageEdgeInsets.left+self.imageEdgeInsets.right),
                                   imageH-(self.imageEdgeInsets.top+self.imageEdgeInsets.bottom));
            
            break;
        }
        default:
        {
            imageRect = [super imageRectForContentRect:contentRect];
            break;
        }
            
    }
    
    return imageRect;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property Methods


//================================================================================
//
//================================================================================
- (void)setOrientationDetect:(BOOL)detect;
{
    if(orientationDetect_!=detect)
    {
        orientationDetect_ = detect;
        
        if(orientationDetect_==YES)
        {
            [[NSNotificationCenter defaultCenter] addObserver:self
                                                     selector:@selector(orientationDidChangeNotification:)
                                                         name:UIDeviceOrientationDidChangeNotification
                                                       object:nil];
        }
        else
        {
            [[NSNotificationCenter defaultCenter] removeObserver:self];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setEnableLongPress:(BOOL)enableLongPress
{
    if(_enableLongPress!=enableLongPress)
    {
        _enableLongPress = enableLongPress;
        
        if(self.enableLongPress==YES)
        {
            UILongPressGestureRecognizer *longPress = [[UILongPressGestureRecognizer alloc] initWithTarget:self action:@selector(longPress:)];
            
            [self addGestureRecognizer:longPress];
            
            self.longPressGestureRecognizer = longPress;
            
            [longPress release];
        }
        else
        {
            [self removeGestureRecognizer:self.longPressGestureRecognizer];
            [self.longPressGestureRecognizer removeTarget:self action:@selector(longPress:)];
            
            self.longPressGestureRecognizer = nil;
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)setMinimumImageHeight:(CGFloat)minimumImageHeight
{
    _minimumImageHeight = minimumImageHeight;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Gesture Method

//================================================================================
//
//================================================================================
- (void)longPress:(UILongPressGestureRecognizer*)gesture
{
    if([self.delegate respondsToSelector:@selector(ppButton:longPressWithState:)]==YES)
    {
        [self.delegate ppButton:self
             longPressWithState:gesture.state];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (UIControlEvents)addControlEvents:(UIControlEvents)controlEvents
{
    UIControlEvents succuessControlEvents = 0;
    
    if((controlEvents&UIControlEventTouchCancel)==UIControlEventTouchCancel)
    {
        [self addTarget:self
                 action:@selector(onTouchCancel)
       forControlEvents:UIControlEventTouchCancel];
        
        succuessControlEvents |= UIControlEventTouchCancel;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDown)==UIControlEventTouchDown)
    {
        [self addTarget:self
                 action:@selector(onTouchDown)
       forControlEvents:UIControlEventTouchDown];
        
        succuessControlEvents |= UIControlEventTouchDown;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDownRepeat)==UIControlEventTouchDownRepeat)
    {
        [self addTarget:self
                 action:@selector(onTouchDownRepeat)
       forControlEvents:UIControlEventTouchDownRepeat];
        
        succuessControlEvents |= UIControlEventTouchDownRepeat;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragInside)==UIControlEventTouchDragInside)
    {
        [self addTarget:self
                 action:@selector(onTouchDragInside)
       forControlEvents:UIControlEventTouchDragInside];
        
        succuessControlEvents |= UIControlEventTouchDragInside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragOutside)==UIControlEventTouchDragOutside)
    {
        [self addTarget:self
                 action:@selector(onTouchDragOutside)
       forControlEvents:UIControlEventTouchDragOutside];
        
        succuessControlEvents |= UIControlEventTouchDragOutside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragEnter)==UIControlEventTouchDragEnter)
    {
        [self addTarget:self
                 action:@selector(onTouchDragEnter)
       forControlEvents:UIControlEventTouchDragEnter];
        
        succuessControlEvents |= UIControlEventTouchDragEnter;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragExit)==UIControlEventTouchDragExit)
    {
        [self addTarget:self
                 action:@selector(onTouchDragExit)
       forControlEvents:UIControlEventTouchDragExit];
        
        succuessControlEvents |= UIControlEventTouchDragExit;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchUpInside)==UIControlEventTouchUpInside)
    {
        [self addTarget:self
                 action:@selector(onTouchUpInside)
       forControlEvents:UIControlEventTouchUpInside];
        
        succuessControlEvents |= UIControlEventTouchUpInside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchUpOutside)==UIControlEventTouchUpOutside)
    {
        [self addTarget:self
                 action:@selector(onTouchUpOutside)
       forControlEvents:UIControlEventTouchUpOutside];
        
        succuessControlEvents |= UIControlEventTouchUpOutside;
    }

    //////////////////////////////////////////////////

    return succuessControlEvents;
}


//================================================================================
//
//================================================================================
- (UIControlEvents)removeControlEvents:(UIControlEvents)controlEvents;
{
    UIControlEvents succuessControlEvents = 0;
    
    if((controlEvents&UIControlEventTouchCancel)==UIControlEventTouchCancel)
    {
        [self removeTarget:self
                    action:@selector(onTouchCancel)
          forControlEvents:UIControlEventTouchCancel];
        
        succuessControlEvents |= UIControlEventTouchCancel;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDown)==UIControlEventTouchDown)
    {
        [self removeTarget:self
                    action:@selector(onTouchDown)
          forControlEvents:UIControlEventTouchDown];
        
        succuessControlEvents |= UIControlEventTouchDown;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDownRepeat)==UIControlEventTouchDownRepeat)
    {
        [self removeTarget:self
                    action:@selector(onTouchDownRepeat)
          forControlEvents:UIControlEventTouchDownRepeat];
        
        succuessControlEvents |= UIControlEventTouchDownRepeat;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragInside)==UIControlEventTouchDragInside)
    {
        [self removeTarget:self
                    action:@selector(onTouchDragInside)
          forControlEvents:UIControlEventTouchDragInside];
        
        succuessControlEvents |= UIControlEventTouchDragInside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragOutside)==UIControlEventTouchDragOutside)
    {
        [self removeTarget:self
                    action:@selector(onTouchDragOutside)
          forControlEvents:UIControlEventTouchDragOutside];
        
        succuessControlEvents |= UIControlEventTouchDragOutside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragEnter)==UIControlEventTouchDragEnter)
    {
        [self removeTarget:self
                    action:@selector(onTouchDragEnter)
          forControlEvents:UIControlEventTouchDragEnter];
        
        succuessControlEvents |= UIControlEventTouchDragEnter;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchDragExit)==UIControlEventTouchDragExit)
    {
        [self removeTarget:self
                    action:@selector(onTouchDragExit)
          forControlEvents:UIControlEventTouchDragExit];
        
        succuessControlEvents |= UIControlEventTouchDragExit;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchUpInside)==UIControlEventTouchUpInside)
    {
        [self removeTarget:self
                    action:@selector(onTouchUpInside)
          forControlEvents:UIControlEventTouchUpInside];
        
        succuessControlEvents |= UIControlEventTouchUpInside;
    }
    
    //////////////////////////////////////////////////
    
    if((controlEvents&UIControlEventTouchUpOutside)==UIControlEventTouchUpOutside)
    {
        [self removeTarget:self
                    action:@selector(onTouchUpOutside)
          forControlEvents:UIControlEventTouchUpOutside];
        
        succuessControlEvents |= UIControlEventTouchUpOutside;
    }
    
    //////////////////////////////////////////////////
    
    return succuessControlEvents;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

//================================================================================
//
//================================================================================
- (void)orientationDidChangeNotification:(NSNotification *)notification
{
	static UIDeviceOrientation preDeviceOrientation = UIDeviceOrientationUnknown;
	
    UIDeviceOrientation currentDeviceOrientation = [UIDevice currentDevice].orientation;
    
    switch(currentDeviceOrientation)
    {
        case UIDeviceOrientationUnknown:
        case UIDeviceOrientationFaceDown:
        case UIDeviceOrientationFaceUp:
        {
            if(preDeviceOrientation!=UIDeviceOrientationPortraitUpsideDown)
            {
                break;
            }
            
            currentDeviceOrientation = UIDeviceOrientationPortrait;
        }
        default:
        {
            preDeviceOrientation = currentDeviceOrientation;
            
            [self setImageTransformByOrientation:currentDeviceOrientation];
            
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)onTouchCancel
{
    [self reportControlEvent:UIControlEventTouchCancel];
}


//================================================================================
//
//================================================================================
- (void)onTouchDown
{
    [self reportControlEvent:UIControlEventTouchDown];
}


//================================================================================
//
//================================================================================
- (void)onTouchDownRepeat
{
    [self reportControlEvent:UIControlEventTouchDownRepeat];
}


//================================================================================
//
//================================================================================
- (void)onTouchDragInside
{
    [self reportControlEvent:UIControlEventTouchDragInside];
}


//================================================================================
//
//================================================================================
- (void)onTouchDragOutside
{
    [self reportControlEvent:UIControlEventTouchDragOutside];
}


//================================================================================
//
//================================================================================
- (void)onTouchDragEnter
{
    [self reportControlEvent:UIControlEventTouchDragEnter];
}


//================================================================================
//
//================================================================================
- (void)onTouchDragExit
{
    [self reportControlEvent:UIControlEventTouchDragExit];
}


//================================================================================
//
//================================================================================
- (void)onTouchUpInside
{
    [self reportControlEvent:UIControlEventTouchUpInside];
}


//================================================================================
//
//================================================================================
- (void)onTouchUpOutside
{ 
    [self reportControlEvent:UIControlEventTouchUpOutside];
}


//================================================================================
//
//================================================================================
- (void)reportControlEvent:(UIControlEvents)controlEvent
{
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppButton:controlEvent:)]==YES)
    {
        [self.delegate ppButton:self controlEvent:controlEvent];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//================================================================================
//
//================================================================================
+ (void)setDefaultTitleColor:(UIColor *)color
{
    [color retain];
    [defaultTitleColor release];
    defaultTitleColor = color;
}


//================================================================================
//
//================================================================================
+ (void)setDefaultTitleShadowColor:(UIColor *)color
{
    [color retain];
    [defaultTitleShadowColor release];
    defaultTitleShadowColor = color;
}


//================================================================================
//
//================================================================================
+ (void)setDefaultTitleLabelFont:(UIFont *)font
{
    [font retain];
    [defaultTitleLabelFont release];
    defaultTitleLabelFont = font;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPAppearance


//==============================================================================
//
//==============================================================================
+ (instancetype)appearance
{
    return [PPAppearance appearanceForClass:[self class]];
}
@end
