//
//  PPBackgroundTaskController.m
//
//  NOTE
//  ---------------------------------
//  1. app到背景時，如果有audio正在播放，不會啟動背景播音延長機制，task 180秒後結束。
//  2. app背景延長機制如果正在播音時，開系統音樂播音會被中斷，task 180秒後結束。
//  3. 原則上背景播音延長機制不可影響原有播放的狀態。
//

////////////////////////////////////////////////////////////////////////////////////////////////////

#import "PPBackgroundTaskController.h"
#import <AVFoundation/AVFoundation.h>
#import "PPLogController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef void (^VoidBlock)(void);

////////////////////////////////////////////////////////////////////////////////////////////////////

/// 背景要被強制結束前幾秒進行通知 (一定要大於5秒，不然就直接進入ExpirationHandler)
CGFloat const PPBackgroundTaskController_TerminateInterval = 10.0;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPBackgroundTaskController ()
@property (atomic, retain) NSMutableDictionary *taskDict;
@property (atomic, assign) UIBackgroundTaskIdentifier bgTaskID;
@property (atomic, retain) AVAudioPlayer *bgMusicPlayer;
@property (atomic, retain) PPLogController *logController;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPBackgroundTaskController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)init
{
    if(self = [super init])
    {
        self.taskDict = [NSMutableDictionary dictionary];

        //////////////////////////////////////////////////

        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvDidBecomeActive:)
                                                     name:UIApplicationDidBecomeActiveNotification
                                                   object:nil];
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvWillResignActive:)
                                                     name:UIApplicationWillResignActiveNotification
                                                   object:nil];

        //////////////////////////////////////////////////

        if([self hasBackgroundModeWithAudio] == YES)
        {
            [[AVAudioSession sharedInstance] setCategory:AVAudioSessionCategoryPlayback error:nil];
        }
        
        //////////////////////////////////////////////////

#ifdef DEBUG
//        self.logController = [[[PPLogController alloc] init] autorelease];
#endif
        
        if(self.logController != nil)
        {
            self.logController.toConsole = NO;
            self.logController.toFile = YES;
         
            NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
            [self.logController setFileName:@"PPBackgroundTaskController" atPath:[paths firstObject]];
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    //////////////////////////////////////////////////

    self.logController = nil;
    self.taskDict = nil;
    
    [self.bgMusicPlayer stop];
    self.bgMusicPlayer = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (BOOL)hasBackgroundModeWithAudio
{
    NSArray *backgroundModes = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"UIBackgroundModes"];
    
    return ([backgroundModes containsObject:@"audio"] == YES);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notification recv methods

//===============================================================================
//
//===============================================================================
- (void)recvDidBecomeActive:(NSNotification *)notify
{
    UIApplication *application = (UIApplication *)notify.object;
    
    // !! terminate background task running in applicationDidEnterBackground
    if(self.bgTaskID != UIBackgroundTaskInvalid)
    {
        [application endBackgroundTask:self.bgTaskID];
        self.bgTaskID = UIBackgroundTaskInvalid;
    }
    
    if([self hasBackgroundModeWithAudio] == YES)
    {
        [self.bgMusicPlayer stop];
        self.bgMusicPlayer = nil;
    }
}


//===============================================================================
//
//===============================================================================
- (void)recvWillResignActive:(NSNotification *)notify
{
    UIApplication *application = (UIApplication *)notify.object;
    __block typeof(self) blockSelf = self;
    
    
    //////////////////////////////////////////////////
    // !! 到達背景模式時間上限前5秒就會進入ExpirationHandler
    
    self.bgTaskID = [application beginBackgroundTaskWithExpirationHandler:^{
        
        [application endBackgroundTask:self.bgTaskID];
        blockSelf.bgTaskID = UIBackgroundTaskInvalid;
    }];
    
    
    //////////////////////////////////////////////////
    // Start the long-running task and return immediately.
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_LOW, 0), ^{
        
        // !! 進入背景模式時，繼續執行thread。
        while ([blockSelf.taskDict count] > 0 &&                    // 1.有thread需要執行
               blockSelf.bgTaskID != UIBackgroundTaskInvalid)        // 2.還在背景模式
        {
            [NSThread sleepForTimeInterval:0.5];
        
            __block NSTimeInterval backgroundTimeRemaining = 0;
            
            // !! backgroundTimeRemaining必須在mainThread取得
            dispatch_sync(dispatch_get_main_queue(), ^{
                backgroundTimeRemaining = application.backgroundTimeRemaining;
            });
            
//            NSLog(@"backgroundTimeRemaining : %f", backgroundTimeRemaining);
            
            // !! 結束的通知 (一定要大於5秒，不然就直接進入ExpirationHandler)
            if(backgroundTimeRemaining < PPBackgroundTaskController_TerminateInterval)
            {
                for (id value in [blockSelf.taskDict allValues])
                {
                    if([value isKindOfClass:[NSNull class]] == NO)
                    {
                        VoidBlock terminateBlock = value;
                        
                        terminateBlock();
                    }
                }
                
                blockSelf.taskDict = nil;
                
                break;
            }
        }
        
        // !! 未達背景模式時間上限前離開要先關閉task
        if(blockSelf.bgTaskID != UIBackgroundTaskInvalid)
        {
            [application endBackgroundTask:blockSelf.bgTaskID];
            blockSelf.bgTaskID = UIBackgroundTaskInvalid;
        }
        
        // !! 背景沒有需要執行的task時，關閉播音延長機制。
        if([blockSelf hasBackgroundModeWithAudio] == YES)
        {
            [blockSelf.bgMusicPlayer stop];
            blockSelf.bgMusicPlayer = nil;
        }
    });

    
    //////////////////////////////////////////////////
    // MARK: play audio to extend background time
    
    if([self hasBackgroundModeWithAudio] == YES)
    {
        //////////////////////////////////////////////////
        // 如果有其他音樂正在播放就不執行播音延長機制

        BOOL isOtherAudioPlaying = [[AVAudioSession sharedInstance] isOtherAudioPlaying];
        
        if(isOtherAudioPlaying == YES)
        {
            return;
        }
        
        
        //////////////////////////////////////////////////
        // 執行播音延長機制
        
        [self.bgMusicPlayer stop];
        self.bgMusicPlayer = nil;
        
        NSString *filePath = [[NSBundle mainBundle] pathForResource:@"PPBTC" ofType:@"wav"];
        self.bgMusicPlayer = [[[AVAudioPlayer alloc] initWithData:[NSData dataWithContentsOfFile:filePath] error:nil] autorelease];
        
        if(self.bgMusicPlayer != nil)
        {
            self.bgMusicPlayer.numberOfLoops = -1;
            
            if([self.bgMusicPlayer prepareToPlay] == YES)
            {
                [self.bgMusicPlayer play];
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (instancetype)sharedInstance
{
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
+ (BOOL)addTaskWithKey:(NSString *)key terminate:(void (^)(void))teminate
{
//    NSLog(@"%s : %@", __func__, key);

    BOOL result = NO;
    
    if([key length] > 0)
    {
        PPBackgroundTaskController *controller = [self sharedInstance];
        
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"addTaskWithKey : %@", key];
        
        if([controller.taskDict objectForKey:key] == nil)
        {
            if(teminate != nil)
            {
                [controller.taskDict setObject:teminate forKey:key];
            }
            else
            {
                [controller.taskDict setObject:[NSNull null] forKey:key];
            }
            
            result = YES;
        }
        
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"taskDict : %@", controller.taskDict];
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"##########"];
    }
    
    return result;
}


//================================================================================
//
//================================================================================
+ (void)removeTaskWithKey:(NSString *)key
{
    if([key length] > 0)
    {
        PPBackgroundTaskController *controller = [self sharedInstance];
        
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"removeTaskWithKey : %@", key];
        
        [controller.taskDict removeObjectForKey:key];
        
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"taskDict : %@", controller.taskDict];
        [controller.logController logWithMask:PPLogControllerMask_Normal format:@"##########"];
    }
}


@end
