//
//  PPAppearance.m
//

#import "PPAppearance.h"

@interface PPAppearance ()
@property (nonatomic, retain) Class mainClass;
@property (nonatomic, retain) NSMutableArray *invocations;
@end


@implementation PPAppearance

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)initWithClass:(Class)aClass
{
    if(self = [super init])
    {
        self.mainClass = aClass;
        self.invocations = [NSMutableArray array];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.mainClass = nil;
    self.invocations = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - NSObject override methods

//================================================================================
// MARK: 1.instance被呼叫但沒有實作的method系統都會在此詢問，所以呼叫setXXXX時都會先進來這裡。
//       (Forward messages that the receiving object does not respond to)
//================================================================================
- (NSMethodSignature *)methodSignatureForSelector:(SEL)aSelector
{
    NSMethodSignature *signature = [self.mainClass instanceMethodSignatureForSelector:aSelector];
    
//    NSLog(@"%ld", [signature numberOfArguments]);
    
    return signature;
}


//================================================================================
// MARK: 2.記錄這些invocaton，等後面呼叫applyInvocationTo時再執行。
//================================================================================
- (void)forwardInvocation:(NSInvocation *)anInvocation;
{
    [anInvocation setTarget:nil];
    [anInvocation retainArguments];
    
    [self.invocations addObject:anInvocation];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)applyInvocationTo:(id)target
{
    for (NSInvocation *invocation in self.invocations)
    {

        //////////////////////////////////////////////////
        // 3rd-party參考 (from MZAppearance)

//        // Create a new copy of the stored invocation,
//        // otherwise setting the new target, this will never be released
//        // because the invocation in the array is still alive after the call
//        
//        NSInvocation *targetInvocation = [invocation copy];
//        [targetInvocation setTarget:target];
//        [targetInvocation invoke];
//        targetInvocation = nil;

        
        //////////////////////////////////////////////////
        // !! invocation需要複製出來執行並清除，否則target因為invocation一直存在而無法清除。
        
        NSInvocation *applyInvocation = [NSInvocation invocationWithMethodSignature:invocation.methodSignature];
        
        [applyInvocation setTarget:target];
        [applyInvocation setSelector:invocation.selector];
        

        NSUInteger numberOfArguments = [invocation.methodSignature numberOfArguments];
        
        // (參考Objective-C Runtime Programming Guide -> Type Encodings)
        // 第一個argumentType一定是'@' (An object (whether statically typed or typed id))
        // 第二個argumentType一定是':' (A method selector (SEL))
        // 設定applyInvocation時要略過前兩個argument，從第三個開始設定。
        
        for(NSUInteger i=2; i<numberOfArguments; i++)
        {
            const char *argumentType = [invocation.methodSignature getArgumentTypeAtIndex:i];
            NSUInteger argumentLength;
            NSGetSizeAndAlignment(argumentType, &argumentLength, NULL);
            void *buffer = malloc(argumentLength);
            
            if (buffer)
            {
                [invocation getArgument:buffer atIndex:i];
                [applyInvocation setArgument:buffer atIndex:i];
                free(buffer);
            }
        }
        
        [applyInvocation invoke];
        applyInvocation = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (NSMutableDictionary *)sharedAppearanceDictionary
{
    static NSMutableDictionary *dict = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        dict = [[NSMutableDictionary alloc] init];
    });
    
    return dict;
}


//================================================================================
//
//================================================================================
+ (id)appearanceForClass:(Class)aClass
{
    NSMutableDictionary *appearanceDictionary = [self sharedAppearanceDictionary];
    NSString *className = NSStringFromClass(aClass);
    PPAppearance *appearance = [appearanceDictionary objectForKey:className];

    
    if (appearance == nil)
    {
        appearance = [[PPAppearance alloc] initWithClass:aClass];
        [appearanceDictionary setObject:appearance forKey:className];
        [appearance release];
    }
    
    return appearance;
}


@end
