//
//  PPCollectionView.m
//
//  Created by howard.lin on 13/3/11.
//  Copyright (c) 2013年 penpower. All rights reserved.
//

#import "PPCollectionView.h"

// Category
#import "PPAlbumContentViewController+ResourceDefine.h"

// View
#import "AlbumContentSectionHeaderView.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

static CGFloat const PPCollectionView_SectionHeadHeight = 30;

#define kPSCollectionViewDidRelayoutNotification @"kPSCollectionViewDidRelayoutNotification"
#define kMargin 10

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface UIView (PSCollectionView)

@property(nonatomic, assign) CGFloat           left;
@property(nonatomic, assign) CGFloat           top;
@property(nonatomic, assign, readonly) CGFloat right;
@property(nonatomic, assign, readonly) CGFloat bottom;
@property(nonatomic, assign) CGFloat           width;
@property(nonatomic, assign) CGFloat           height;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation UIView (PPCollectionView)

//================================================================================
//
//================================================================================
- (CGFloat)left
{
    return self.frame.origin.x;
}


//================================================================================
//
//================================================================================
- (void)setLeft:(CGFloat)x
{
    CGRect frame = self.frame;
    frame.origin.x = x;
    self.frame = frame;
}


//================================================================================
//
//================================================================================
- (CGFloat)top
{
    return self.frame.origin.y;
}


//================================================================================
//
//================================================================================
- (void)setTop:(CGFloat)y
{
    CGRect frame = self.frame;
    frame.origin.y = y;
    self.frame = frame;
}


//================================================================================
//
//================================================================================
- (CGFloat)right
{
    return self.frame.origin.x + self.frame.size.width;
}


//================================================================================
//
//================================================================================
- (CGFloat)bottom
{
    return self.frame.origin.y + self.frame.size.height;
}


//================================================================================
//
//================================================================================
- (CGFloat)width
{
    return self.frame.size.width;
}


//================================================================================
//
//================================================================================
- (void)setWidth:(CGFloat)width
{
    CGRect frame = self.frame;
    frame.size.width = width;
    self.frame = frame;
}


//================================================================================
//
//================================================================================
- (CGFloat)height
{
    return self.frame.size.height;
}


//================================================================================
//
//================================================================================
- (void)setHeight:(CGFloat)height
{
    CGRect frame = self.frame;
    frame.size.height = height;
    self.frame = frame;
}

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Gesture Recognizer

@interface PPCollectionViewLongGestureRecognizer : UILongPressGestureRecognizer
@end

@implementation PPCollectionViewLongGestureRecognizer
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCollectionView

@interface PPCollectionView() <UIGestureRecognizerDelegate,AlbumColletcionViewCellDelegate>
{
    @private
     BOOL      editMode_;
     CGFloat   lastOffset_;
     CGFloat   offsetThreshold_;
     CGFloat   lastWidth_;
     CGFloat   columnWidth_;
     NSInteger numColumns_;
    
     UIInterfaceOrientation orientation_;
     NSMutableDictionary    *reuseableViews_;
     NSMutableDictionary    *visibleViews_;
     NSMutableArray         *viewKeysToRemove_;
     NSMutableDictionary    *indexToRectMap_;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

@property (nonatomic, assign, readwrite) CGFloat     lastOffset;
@property (nonatomic, assign, readwrite) CGFloat     offsetThreshold;
@property (nonatomic, assign, readwrite) CGFloat     lastWidth;
@property (nonatomic, assign, readwrite) CGFloat     columnWidth;
@property (nonatomic, assign, readwrite) NSInteger   numColumns;
@property (nonatomic, assign) UIInterfaceOrientation orientation;

@property (nonatomic, retain) NSMutableDictionary *reuseableViews;
@property (nonatomic, retain) NSMutableDictionary *visibleViews;
@property (nonatomic, retain) NSMutableArray      *viewKeysToRemove;
@property (nonatomic, retain) NSMutableDictionary *indexToRectMap;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCollectionView

@implementation PPCollectionView

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize editMode         = editMode_;
@synthesize reuseableViews   = reuseableViews_;
@synthesize visibleViews     = visibleViews_;
@synthesize viewKeysToRemove = viewKeysToRemove_;
@synthesize indexToRectMap   = indexToRectMap_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    
    if (self)
    {
        self.alwaysBounceVertical = YES;
        
        self.lastOffset          = 0.0;
        self.offsetThreshold     = floorf(self.height / 4.0);
        
        self.columnWidth         = 0.0;
        self.numColumns          = 0;
        self.numColumnsPortrait  = 0;
        self.numColumnsLandscape = 0;
        
        orientation_          = [UIApplication sharedApplication].statusBarOrientation;
        reuseableViews_       = [[NSMutableDictionary dictionary]retain];
        visibleViews_         = [[NSMutableDictionary dictionary]retain];
        viewKeysToRemove_     = [[NSMutableArray array]retain];
        indexToRectMap_       = [[NSMutableDictionary dictionary]retain];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [reuseableViews_ removeAllObjects];
    [reuseableViews_   release];
    
    [visibleViews_ removeAllObjects];
    [visibleViews_     release];


    [viewKeysToRemove_ removeAllObjects];
    [viewKeysToRemove_ release];
    
    [indexToRectMap_ removeAllObjects];
    [indexToRectMap_   release];
    
    [_headerTitleColor release];
    _headerTitleColor = nil;
    
    [_headerViewBackgroundColor release];
    _headerViewBackgroundColor = nil;

    [_footerView removeFromSuperview];
    [_footerView release];

    [_headerView removeFromSuperview];
    [_headerView release];

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Layout View

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    UIInterfaceOrientation orientation = [UIApplication sharedApplication].statusBarOrientation;
    
    if (self.orientation != orientation)
    {
        self.orientation = orientation;
        [self relayoutViews];
    }
    else if(self.lastWidth != self.width)
    {
        [self relayoutViews];
    }
    else
    {
        //判斷是否變動cell的顯示
        CGFloat diff = fabs(self.lastOffset - self.contentOffset.y);
        
        if (diff > self.offsetThreshold)
        {
            self.lastOffset = self.contentOffset.y;
            
            [self removeAndAddCellsIfNecessary];
        }
    }
    
    self.lastWidth = self.width;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (void)relayoutViews
{
    do
    {
        if([self.collectionViewDataSource respondsToSelector:@selector(numberOfSectionInPPCollectionView:)]==NO ||
           [self.collectionViewDataSource respondsToSelector:@selector(collectionView:numberOfRowsInSection:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        self.numColumns = UIInterfaceOrientationIsPortrait(self.orientation) ? self.numColumnsPortrait : self.numColumnsLandscape;
        
        // Reset all state
        [self.visibleViews enumerateKeysAndObjectsUsingBlock:^(id key, id obj, BOOL *stop)
         {
             [self enqueueReusableView:obj];
         }];
        
        [self.visibleViews     removeAllObjects];
        [self.viewKeysToRemove removeAllObjects];
        [self.indexToRectMap   removeAllObjects];
        
        //////////////////////////////////////////////////
    
        // header
        CGFloat totalHeight = 0.0;
        CGFloat top = kMargin;
        
        // Add headerView if it exists
        if(self.headerView!=nil)
        {
            top += self.headerView.top;
            self.headerView.width = self.width;
            [self addSubview:self.headerView];
            top += self.headerView.height;
        }
        
        //////////////////////////////////////////////////
        
        totalHeight = self.height;

        //////////////////////////////////////////////////

        NSInteger numberOfSection = [self.collectionViewDataSource numberOfSectionInPPCollectionView:self];
        
        for(NSUInteger section = 0; section<numberOfSection; section++)
        {
            //section header height
            
            if([self.collectionViewDataSource respondsToSelector:@selector(collectionView:titleForHeaderInSection:)]==YES)
            {
                NSString *title = [self.collectionViewDataSource collectionView:self
                                                        titleForHeaderInSection:section];
                
                if(title!=nil && title.length>0)
                {
                    top+=PPCollectionView_SectionHeadHeight;
                    
                    if(section!=0)
                    {
                        top+=kMargin;
                    }
                }
            }
            
            //////////////////////////////////////////////////

            // This array determines the last height offset on a column
            NSMutableArray *colOffsets = [NSMutableArray arrayWithCapacity:self.numColumns];
            
            for (int i = 0; i < self.numColumns; i++)
            {
                [colOffsets addObject:[NSNumber numberWithFloat:top]];
            }
    
            //////////////////////////////////////////////////

            // Calculate index to rect mapping
            self.columnWidth = floorf((self.width - kMargin * (self.numColumns + 1)) / self.numColumns);
            
            //////////////////////////////////////////////////

            // 得到多少grid
            NSInteger numOfRow = [self.collectionViewDataSource collectionView:self numberOfRowsInSection:section];

            for(NSUInteger row = 0; row<numOfRow; row++)
            {
                //NSString *key = PSCollectionKeyForIndex(i);
                NSString *key = [NSString stringWithFormat:@"%td-%td",section,row];
                
                // Find the shortest column
                NSInteger col = 0;
                
                CGFloat minHeight = [[colOffsets objectAtIndex:col] floatValue];
                
                for (NSUInteger count = 1; count < [colOffsets count]; count++)
                {
                    CGFloat colHeight = [[colOffsets objectAtIndex:count] floatValue];
                    
                    if (colHeight < minHeight)
                    {
                        col = count;
                        minHeight = colHeight;
                    }
                }
                
                CGFloat left      = kMargin + (col * kMargin) + (col * self.columnWidth);
                CGFloat top       = [[colOffsets objectAtIndex:col] floatValue];
                CGFloat colHeight = [self.collectionViewDataSource collectionView:self heightForRowAtIndex:row];
                
                CGRect viewRect = CGRectMake(left, top, self.columnWidth, colHeight);
       
                // Add to index rect map
                [self.indexToRectMap setObject:NSStringFromCGRect(viewRect) forKey:key];
              
                //////////////////////////////////////////////////

                // Update the last height offset for this column
                CGFloat heightOffset = colHeight > 0 ? top + colHeight + kMargin : top;
                
                [colOffsets replaceObjectAtIndex:col withObject:[NSNumber numberWithFloat:heightOffset]];

                if(row==0)
                {
                    CGRect labelRect = CGRectZero;
                    
                    if(section!=0)
                    {
                        labelRect.origin.y = top-PPCollectionView_SectionHeadHeight-kMargin;
                    }
                    else
                    {
                        
                    }
                    labelRect.size.height = PPCollectionView_SectionHeadHeight;
                    labelRect.size.width  = self.bounds.size.width;
                    
                    // Add to sectionHeader rect map
                    [self.indexToRectMap setObject:NSStringFromCGRect(labelRect) forKey:[NSString stringWithFormat:@"s-%td-%d",section,0]];
                }
            }

            
            //////////////////////////////////////////////////
            
            //紀錄同一個sectionMax高度
            
            for(NSNumber *height in colOffsets)
            {
                if([height unsignedIntegerValue]>top)
                {
                    top = [height unsignedIntegerValue];
                }
            }
        
            //////////////////////////////////////////////////
            
            for (NSNumber *colHeight in colOffsets)
            {
                totalHeight = (totalHeight < [colHeight floatValue]) ? [colHeight floatValue] : totalHeight;
            }
        }

        //////////////////////////////////////////////////
        
        // Add footerView if exists
        if (self.footerView) {
            self.footerView.top = totalHeight;
            self.footerView.width = self.width;
            [self addSubview:self.footerView];
            totalHeight += self.footerView.height;
        }
        
        //////////////////////////////////////////////////
    
        self.contentSize = CGSizeMake(self.width, totalHeight);
        
        // MARK: 調整 contentOffset
        if(self.scrollToEnd==YES)
        {
            self.scrollToEnd = NO;
            
            //////////////////////////////////////////////////

            self.contentOffset = CGPointMake(0, totalHeight-self.bounds.size.height);
        }
        
        //////////////////////////////////////////////////

        [self removeAndAddCellsIfNecessary];
        
        [[NSNotificationCenter defaultCenter] postNotificationName:kPSCollectionViewDidRelayoutNotification object:self];

    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)enqueueReusableView:(UIView *)view
{
    if ([view respondsToSelector:@selector(prepareForReuse)])
    {
        [view performSelector:@selector(prepareForReuse)];
    }
    
    //////////////////////////////////////////////////

    view.frame = CGRectZero;
    
    NSString *identifier = NSStringFromClass([view class]);

    
    if (![self.reuseableViews objectForKey:identifier])
    {
        [self.reuseableViews setObject:[NSMutableSet set] forKey:identifier];
    }
    
    [[self.reuseableViews objectForKey:identifier] addObject:view];
    
    [view removeFromSuperview];
}


//================================================================================
//
//================================================================================
- (void)removeAndAddCellsIfNecessary
{
    do
    {
        if([self.collectionViewDataSource respondsToSelector:@selector(collectionView:numberOfRowsInSection:)]==NO ||
           [self.collectionViewDataSource respondsToSelector:@selector(numberOfSectionInPPCollectionView:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        static NSInteger bufferViewFactor = 8;
        static NSInteger topIndex = 0;
        static NSInteger bottomIndex = 0;
        
        //////////////////////////////////////////////////
        
        @autoreleasepool
        {
            NSUInteger totalSections = [self.collectionViewDataSource numberOfSectionInPPCollectionView:self];
            
            
            for(NSUInteger section=0; section<totalSections; section++)
            {
                NSUInteger totalRowsInSection = [self.collectionViewDataSource collectionView:self numberOfRowsInSection:section];
                
                // Find out what rows are visible
                CGRect visibleRect = CGRectMake(self.contentOffset.x, self.contentOffset.y, self.width, self.height);
                visibleRect = CGRectInset(visibleRect, 0, -1.0 * self.offsetThreshold);
          
                // Remove all rows that are not inside the visible rect
                [self.visibleViews enumerateKeysAndObjectsUsingBlock:^(id key, id obj, BOOL *stop)
                 {
                     UIView *view = (UIView *)obj;
                     CGRect viewRect = view.frame;
                     
                     //////////////////////////////////////////////////

                     if (!CGRectIntersectsRect(visibleRect, viewRect))
                     {
                         [self enqueueReusableView:view];
                         [self.viewKeysToRemove addObject:key];
                     }
                 }];
                
                //////////////////////////////////////////////////
                
                [self.visibleViews removeObjectsForKeys:self.viewKeysToRemove];
                [self.viewKeysToRemove removeAllObjects];
                
                //////////////////////////////////////////////////
                
                if ([self.visibleViews count] == 0)
                {
                    topIndex = 0;
                    bottomIndex = totalRowsInSection;
                }
                else
                {
                    NSArray *sortedKeys = [[self.visibleViews allKeys] sortedArrayUsingComparator:^(id obj1, id obj2)
                                           {
                                               if ([obj1 integerValue] < [obj2 integerValue])
                                               {
                                                   return (NSComparisonResult)NSOrderedAscending;
                                               }
                                               else if ([obj1 integerValue] > [obj2 integerValue])
                                               {
                                                   return (NSComparisonResult)NSOrderedDescending;
                                               }
                                               else
                                               {
                                                   return (NSComparisonResult)NSOrderedSame;
                                               }
                                           }];
                    topIndex = [[sortedKeys objectAtIndex:0] integerValue];
                    bottomIndex = [[sortedKeys lastObject] integerValue];
                    
                    topIndex = MAX(0, topIndex - (bufferViewFactor * self.numColumns));
                    bottomIndex = MIN(totalRowsInSection, bottomIndex + (bufferViewFactor * self.numColumns));
                }
                
                //////////////////////////////////////////////////
                
                //section header height
                
                if([self.collectionViewDataSource respondsToSelector:@selector(collectionView:titleForHeaderInSection:)]==YES)
                {
                    NSString *title = [self.collectionViewDataSource collectionView:self
                                                            titleForHeaderInSection:section];
                    
                    if(title!=nil && title.length>0)
                    {
                        id key = [NSString stringWithFormat:@"s-%td-%d",section,0];
                        
                        CGRect labelRect = CGRectFromString([self.indexToRectMap objectForKey:key]);
                        
                        //////////////////////////////////////////////////
                        
                        if(![self.visibleViews objectForKey:key] && CGRectIntersectsRect(visibleRect, labelRect))
                        {
                            AlbumContentSectionHeaderView *albumContentSectionHeaderView = (AlbumContentSectionHeaderView *)[self dequeueReusableViewForClass:[AlbumContentSectionHeaderView class]];
                            
                            if(albumContentSectionHeaderView==nil)
                            {
                                albumContentSectionHeaderView = [[[AlbumContentSectionHeaderView alloc] init] autorelease];
                            }
    
                            if(albumContentSectionHeaderView!=nil)
                            {
                                albumContentSectionHeaderView.frame = labelRect;
                         
                                albumContentSectionHeaderView.timeLabel.text = [@"  " stringByAppendingString:title];
                                
                                //////////////////////////////////////////////////

                                if(self.headerTitleColor!=nil)
                                {
                                    albumContentSectionHeaderView.timeLabel.textColor = self.headerTitleColor;
                                }
                                
                                //////////////////////////////////////////////////

                                if(self.headerViewBackgroundColor!=nil)
                                {
                                    albumContentSectionHeaderView.backgroundColor = self.headerViewBackgroundColor;
                                }
                                
                                //////////////////////////////////////////////////

                                [self addSubview:albumContentSectionHeaderView];
                                
                                [self.visibleViews setObject:albumContentSectionHeaderView forKey:key];
                                
                                //////////////////////////////////////////////////
                                
                                [albumContentSectionHeaderView removeConstraints:albumContentSectionHeaderView.constraints];
                                [albumContentSectionHeaderView setNeedsUpdateConstraints];
                            }
                        }
                    }
                }
                
                //////////////////////////////////////////////////
                
                for(NSInteger row=topIndex; row<totalRowsInSection; row++)
                {
                    // Add views
                    NSString *key = [NSString stringWithFormat:@"%zd-%zd",section,row];
                    CGRect rect   = CGRectFromString([self.indexToRectMap objectForKey:key]);
                    
                    // If view is within visible rect and is not already shown
                    if (![self.visibleViews objectForKey:key] && CGRectIntersectsRect(visibleRect, rect))
                    {
                        [self.collectionViewDataSource collectionView:self cellWidth:rect.size.width atIndex:row];
                        
                        //////////////////////////////////////////////////
                        
                        // Only add views if not visible
                        AlbumCollectionViewCell *newCell = [self.collectionViewDataSource collectionView:self cellForRowAtIndexPath:[NSIndexPath indexPathForRow:row inSection:section]];
                        
                        newCell.delegate = self;
                        
                        newCell.frame = CGRectFromString([self.indexToRectMap objectForKey:key]);
                        
                        [self addSubview:newCell];
                        
                        // Setup gesture recognizer
                        if ([newCell.gestureRecognizers count]==0)
                        {
//                            PPCollectionViewLongGestureRecognizer *longGestureRecognizer = [[PPCollectionViewLongGestureRecognizer alloc] initWithTarget:self action:@selector(didSelectView:)];
//                            
//                            if(longGestureRecognizer!=nil)
//                            {
//                                longGestureRecognizer.minimumPressDuration = 0.001;
//                                longGestureRecognizer.delegate = self;
//                                [newCell addGestureRecognizer:longGestureRecognizer];
//                            }
//                            
//                            newCell.userInteractionEnabled = YES;
//                            
//                            [longGestureRecognizer release];
                        }
                        
                        [self.visibleViews setObject:newCell forKey:key];
                    }
                }
            }
        }
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Gesture Recognizer

//================================================================================
//
//================================================================================
- (void)didSelectView:(UITapGestureRecognizer *)gestureRecognizer
{
    NSString *rectString   = NSStringFromCGRect(gestureRecognizer.view.frame);
    NSArray  *matchingKeys = [self.indexToRectMap allKeysForObject:rectString];
    NSString *key          = [matchingKeys lastObject];
    
    if(key.length>=3 &&
        [gestureRecognizer.view isMemberOfClass:[[self.visibleViews objectForKey:key] class]])
    {
        BOOL showClickCelll = YES;
        
        do
        {
            NSInteger row     = [[[key componentsSeparatedByString:@"-"] lastObject] intValue];
            NSInteger section = [[[key componentsSeparatedByString:@"-"] firstObject] intValue];
            
            AlbumCollectionViewCell *targetCell = (AlbumCollectionViewCell *)gestureRecognizer.view;
            
            if(targetCell==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self.collectionViewDelegate respondsToSelector:@selector(collectionView:shouldAddSelectedCell:atIndexPath:)]==YES)
            {
                showClickCelll = [self.collectionViewDelegate collectionView:self
                                                       shouldAddSelectedCell:targetCell
                                                                 atIndexPath:[NSIndexPath indexPathForRow:row inSection:section]];
            }
            
            if(showClickCelll==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if([self.collectionViewDelegate respondsToSelector:@selector(collectionView:didSelectCell:atIndexPath:)]==YES)
            {
                if(gestureRecognizer.state==UIGestureRecognizerStateBegan)
                {
                    if(self.editMode==YES)
                    {
                        if(targetCell.selectImageView.status==PPSelectStatus_Select)
                        {
                            [targetCell.selectImageView setStatus:PPSelectStatus_SelectHighlighted];
                        }
                        else
                        {
                            [targetCell.selectImageView setStatus:PPSelectStatus_UnSelectHighlighted];
                        }
                    }
                }
                else if(gestureRecognizer.state==UIGestureRecognizerStateEnded)
                {
                    if(self.editMode==YES)
                    {
                        if(targetCell.selectImageView.status==PPSelectStatus_SelectHighlighted)
                        {
                            [targetCell.selectImageView setStatus:PPSelectStatus_Unselect];
                        }
                        else
                        {
                            [targetCell.selectImageView setStatus:PPSelectStatus_Select];
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    [self.collectionViewDelegate collectionView:self didSelectCell:targetCell atIndexPath:[NSIndexPath indexPathForRow:row inSection:section]];
                }
            }
        }
        while (0);
    }
}


//================================================================================
//
//================================================================================
- (BOOL)gestureRecognizer:(UIGestureRecognizer *)gestureRecognizer shouldReceiveTouch:(UITouch *)touch
{
    if (![gestureRecognizer isMemberOfClass:[PPCollectionViewLongGestureRecognizer class]])
    {
        return YES;
    }
    
    NSString *rectString   = NSStringFromCGRect(gestureRecognizer.view.frame);
    NSArray  *matchingKeys = [self.indexToRectMap allKeysForObject:rectString];
    NSString *key          = [matchingKeys lastObject];
    
    if ([touch.view isMemberOfClass:[[self.visibleViews objectForKey:key] class]])
    {
        return YES;
    }
    else
    {
        return NO;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - AlbumCollectionViewCellDelegate

//================================================================================
//
//================================================================================
- (BOOL)albumViewCollectionCellShouldTouch:(AlbumCollectionViewCell *)albumCollectionViewCell
{
    BOOL showClickCelll = NO;
    
    NSString *rectString   = NSStringFromCGRect(albumCollectionViewCell.frame);
    NSArray  *matchingKeys = [self.indexToRectMap allKeysForObject:rectString];
    NSString *key          = [matchingKeys lastObject];
    
    NSInteger row     = [[[key componentsSeparatedByString:@"-"] lastObject] intValue];
    NSInteger section = [[[key componentsSeparatedByString:@"-"] firstObject] intValue];
    
    if([self.collectionViewDelegate respondsToSelector:@selector(collectionView:shouldAddSelectedCell:atIndexPath:)]==YES)
    {
        showClickCelll = [self.collectionViewDelegate collectionView:self
                                               shouldAddSelectedCell:albumCollectionViewCell
                                                         atIndexPath:[NSIndexPath indexPathForRow:row inSection:section]];
    }
    
    return showClickCelll;
}


//================================================================================
//
//================================================================================
- (void)albumViewCollectionCellTouchBegin:(AlbumCollectionViewCell *)albumCollectionViewCell
{
    if(self.editMode==YES)
    {
        if(albumCollectionViewCell.selectImageView.status==PPSelectStatus_Select)
        {
            [albumCollectionViewCell.selectImageView setStatus:PPSelectStatus_SelectHighlighted];
        }
        else
        {
            [albumCollectionViewCell.selectImageView setStatus:PPSelectStatus_UnSelectHighlighted];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)albumViewCollectionCellTouchEnd:(AlbumCollectionViewCell *)albumCollectionViewCell
{
    NSString *rectString   = NSStringFromCGRect(albumCollectionViewCell.frame);
    NSArray  *matchingKeys = [self.indexToRectMap allKeysForObject:rectString];
    NSString *key          = [matchingKeys lastObject];
    
    NSInteger row     = [[[key componentsSeparatedByString:@"-"] lastObject] intValue];
    NSInteger section = [[[key componentsSeparatedByString:@"-"] firstObject] intValue];

    
    if(self.editMode==YES)
    {
        if(albumCollectionViewCell.selectImageView.status==PPSelectStatus_SelectHighlighted)
        {
            [albumCollectionViewCell.selectImageView setStatus:PPSelectStatus_Unselect];
        }
        else
        {
            [albumCollectionViewCell.selectImageView setStatus:PPSelectStatus_Select];
        }
    }
    
    //////////////////////////////////////////////////
    
    [self.collectionViewDelegate collectionView:self
                                  didSelectCell:albumCollectionViewCell
                                    atIndexPath:[NSIndexPath indexPathForRow:row inSection:section]];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Accessory Method

//================================================================================
//
//================================================================================
- (void)setEditMode:(BOOL)editMode
{
    editMode_ = editMode;
    
    
    if(editMode_==NO)
    {
        [self resetAllVisibieView];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (void)reloadData
{
    [self relayoutViews];
}


//================================================================================
//
//================================================================================
- (void)resetAllVisibieView
{
}


//================================================================================
//
//================================================================================
- (UIView *)dequeueReusableViewForClass:(Class)viewClass
{
    NSString *identifier = NSStringFromClass(viewClass);
    
    UIView *view = nil;
    
    if ([self.reuseableViews objectForKey:identifier])
    {
        view = [[[self.reuseableViews objectForKey:identifier] anyObject]retain];

        if (view)
        {
            // Found a reusable view, remove it from the set
            [[self.reuseableViews objectForKey:identifier] removeObject:view];
        }
    }
    
    return [view autorelease];
}

@end
